#ifndef _TimeSliceRPlex_h
#define _TimeSliceRPlex_h 1

#include "floatVec.h"
#include "PixPlex.h"

// Plexes are made out of TimeSliceIChunks

#include "Environ.h"

enum ETSliceInfo { kTSI_DataType, kTSI_CopyType };
enum ETS_DataType { kTS_Value, kTS_Array, kTS_NULL };
enum ETS_CopyType { kTS_Deep, kTS_Shallow };

/*
*************************************************************************
*									*
* TimeSlice								*
*									*
*************************************************************************
*/

class TimeSlice {

protected:

  float fTimeStamp;
  byte fInfo[4];

  static float fGarbage;
  static float fdGarbage;

public:
  inline TimeSlice() : fTimeStamp(0.0) 
    { fInfo[kTSI_DataType] = kTS_NULL; } 

  inline TimeSlice( float time ) : fTimeStamp(time) 
    { fInfo[kTSI_DataType] = kTS_NULL; } 

  virtual inline TimeSlice& operator= ( const TimeSlice& ts ) { fTimeStamp = ts.fTimeStamp; memcpy(fInfo,ts.fInfo,4); return *this; } 
  
  virtual inline float& Value() { 
    if( fInfo[kTSI_DataType] != kTS_Value ) { gPrintErr( "Bad Data Assess in TSlice" ); }
    return fdGarbage; 
  }
    
  virtual inline float operator() () const { 
    if( fInfo[kTSI_DataType] != kTS_Value ) { gPrintErr( "Bad Data Assess in TSlice" ); }
    return 0.0; 
  }

  virtual inline float& Value(int index) { 
    if( fInfo[kTSI_DataType] != kTS_Array && index > 0 ) { gPrintErr( "Bad Data Assess in TSlice" ); }
    return fGarbage;
  }

  virtual inline float operator() (int index) const { 
    if( fInfo[kTSI_DataType] != kTS_Array ) { gPrintErr( "Bad Data Assess in TSlice" ); }
    return 0.0;
  }
 
  virtual inline float* DataStart() { return NULL; }  
  virtual inline const float* Data() const { return NULL; }

  inline float Time() const { return fTimeStamp; }

  virtual inline void Free() = 0;
  inline void invalidate() { Free(); }
};


class TTemporalTimeSlice : public TimeSlice {

  float fP;

public:
  inline TTemporalTimeSlice() : TimeSlice()  { fInfo[kTSI_DataType] = kTS_NULL; } 

  inline TTemporalTimeSlice( float time, float value ) : TimeSlice(time)    { fInfo[kTSI_DataType] = kTS_Value; fP = value; } 

  inline void SetData( float time, float value ) { 
		fInfo[kTSI_DataType] = kTS_Value; fP = value; fTimeStamp = time; 
	} 
 
  inline TimeSlice& operator= ( const TimeSlice& ts ) { fP = ((const TTemporalTimeSlice&)ts).fP; return TimeSlice::operator=(ts); } 
  
  inline float& Value() {   return fP;  }
  
  inline float operator() () const { 
    return fP; 
  }

  inline void Free() { ; }
};

class TSpatialTimeSlice : public TimeSlice {

  float* fP;
  PixStack* fPixStack;

public:
  inline TSpatialTimeSlice() : TimeSlice()
    { fInfo[kTSI_DataType] = kTS_Array; fP = NULL; } 

  inline TSpatialTimeSlice( float time, float *valp, ETS_CopyType ctype, PixStack* ps=NULL) : TimeSlice(time) 
    { fInfo[kTSI_DataType] = kTS_Array; fInfo[kTSI_CopyType] = ctype; fP = valp; fPixStack = ps; }

  inline void SetData( float time, float *valp, ETS_CopyType ctype, PixStack* ps=NULL)
    { fInfo[kTSI_DataType] = kTS_Array; fInfo[kTSI_CopyType] = ctype; fP = valp; fPixStack = ps; fTimeStamp = time; }

  inline TimeSlice& operator= ( const TSpatialTimeSlice& ts ) {
		 const TSpatialTimeSlice& sts = (const TSpatialTimeSlice&) ts; 
		fP = sts.fP; fPixStack = sts.fPixStack; return TimeSlice::operator=(ts); 
	} 
  
  inline float& Value(int index) { 
    if( fP ) return fP[index]; 
    else { gPrintErr( "Attempt to Access Freed Data in TimeSlice. " ); return fGarbage; } 
  }
  inline float operator() (int index) const { 
    if( fP ) return fP[index]; 
    else { gPrintErr( "Attempt to Access Freed Data in TimeSlice. " ); return 0.0; } 
  }

  inline float* DataStart() { return fP; }
  inline const float* Data() const { return fP; }

  inline void Free() { 
		if( fP && fInfo[kTSI_CopyType] == kTS_Deep ) {
			if(fPixStack) { fPixStack->push(fP); }
			else { delete [] fP; } 
			fP = NULL; 
		}
	}
};

/*
*************************************************************************
*									*
* TSpatialTimeSliceIChunk     						*
*									*
*************************************************************************
*/


// Plexes are made out of TSpatialTimeSliceIChunks

#include <stddef.h>

class TSpatialTimeSliceIChunk
{

protected:      

  TSpatialTimeSlice*           data;           // data, from client

  int            base;           // lowest possible index
  int            low;            // lowest valid index
  int            fence;          // highest valid index + 1
  int            top;            // highest possible index + 1

  TSpatialTimeSliceIChunk*     nxt;            // circular links
  TSpatialTimeSliceIChunk*     prv;

public:

// constructors

                 TSpatialTimeSliceIChunk(TSpatialTimeSlice*     d,       // ptr to array of elements
                        int      base_idx, // initial indices
                        int      low_idx,  
                        int      fence_idx,
                        int      top_idx);

  virtual       ~TSpatialTimeSliceIChunk();

// status reports

  int            size() const;    // number of slots

  inline virtual int    empty() const ;
  inline virtual int    full() const ; 

  int            can_grow_high () const ;  // there is space to add data
  int            can_grow_low () const;        
  
  int            base_index() const;   // lowest possible index;
  int            low_index() const;    // lowest actual index;
  inline virtual int    first_index() const;  // lowest valid index or fence if none
  inline virtual int    last_index() const;   // highest valid index or low-1 if none
  int            fence_index() const;  // highest actual index + 1
  int            top_index() const;    // highest possible index + 1

// indexing conversion

  int            possible_index(int i) const; // i between base and top
  int            actual_index(int i) const;   // i between low and fence
  inline virtual int    valid_index(int i) const;    // i not deleted (mainly for mchunks)

  int            possible_pointer(const TSpatialTimeSlice* p) const; // same for ptr
  int            actual_pointer(const TSpatialTimeSlice* p) const; 
  inline virtual int    valid_pointer(const TSpatialTimeSlice* p) const; 

  TSpatialTimeSlice*           pointer_to(int i) const ;   // pointer to data indexed by i
                                      // caution: i is not checked for validity
  int            index_of(const TSpatialTimeSlice* p) const; // index of data pointed to by p
                                      // caution: p is not checked for validity

  inline virtual int    succ(int idx) const;     // next valid index or fence if none
  inline virtual int    pred(int idx) const;     // previous index or low - 1 if none

  inline virtual TSpatialTimeSlice*   first_pointer() const;   // pointer to first valid pos or 0
  inline virtual TSpatialTimeSlice*   last_pointer() const;    // pointer to first valid pos or 0
  inline virtual TSpatialTimeSlice*   succ(TSpatialTimeSlice*  p) const;     // next pointer or 0
  inline virtual TSpatialTimeSlice*   pred(TSpatialTimeSlice* p) const;     // previous pointer or 0

// modification

  inline virtual TSpatialTimeSlice*   grow_high ();      // return spot to add an element
  inline virtual TSpatialTimeSlice*   grow_low ();  

  inline virtual void   shrink_high ();    // logically delete top index
  inline virtual void   shrink_low ();     

  virtual void   clear(int lo);     // reset to empty ch with base = lo
  virtual void   cleardown(int hi); // reset to empty ch with top = hi
  void           re_index(int lo);  // re-index so lo is new low

// chunk traversal

  TSpatialTimeSliceIChunk*     next() const;
  TSpatialTimeSliceIChunk*     prev() const;

  void           link_to_prev(TSpatialTimeSliceIChunk* prev);
  void           link_to_next(TSpatialTimeSliceIChunk* next);
  void           unlink();

// state checks

  TSpatialTimeSlice*           invalidate();     // mark self as invalid; return data
                                   // for possible deletion

  virtual int    OK() const;             // representation invariant

  void   error(const char*) const;
  void   empty_error() const;
  void   full_error() const;
  void   index_error() const;
};

// TSpatialTimeSlicePlex is a partly `abstract' class: few of the virtuals
// are implemented at the Plex level, only in the subclasses

class TSpatialTimeSlicePlex
{
protected:      

  TSpatialTimeSliceIChunk*       hd;          // a chunk holding the data
  int              lo;          // lowest  index
  int              fnc;         // highest index + 1
  int              csize;       // size of the chunk

  void             invalidate();              // mark so OK() is false
  void             del_chunk(TSpatialTimeSliceIChunk*);        // delete a chunk

  TSpatialTimeSliceIChunk*       tl() const;                // last chunk;
  int              one_chunk() const;         // true if hd == tl()

public:

// constructors, etc.

                    TSpatialTimeSlicePlex();                  // no-op

  virtual           ~TSpatialTimeSlicePlex();

  
// Access functions 
  
  virtual TSpatialTimeSlice&      elem (int idx) = 0; // access by index;
  virtual TSpatialTimeSlice&      elem (Pix p) = 0;   // access by Pix;

  virtual TSpatialTimeSlice&      high_elem () = 0;      // access high element
  virtual TSpatialTimeSlice&      low_elem () = 0;       // access low element

// read-only versions for const Plexes

  virtual const TSpatialTimeSlice& operator [] (int idx) const = 0; // access by index;
  virtual const TSpatialTimeSlice& operator () (Pix p) const = 0;   // access by Pix;

  virtual const TSpatialTimeSlice& high_element () const = 0;      // access high element
  virtual const TSpatialTimeSlice& low_element () const = 0;       // access low element


// Index functions

  virtual int       valid (int idx) const = 0;      // idx is an OK index

  virtual int       low() const = 0;         // lowest index or fence if none
  virtual int       high() const = 0;        // highest index or low-1 if none

  int               ecnef() const;         // low limit index (low-1)
  int               fence() const;         // high limit index (high+1)

  virtual void      prev(int& idx) const= 0; // set idx to preceding index
                                          // caution: pred may be out of bounds

  virtual void      next(int& idx) const = 0;       // set to next index
                                          // caution: succ may be out of bounds

  virtual Pix       first() const = 0;        // Pix to low element or 0
  virtual Pix       last() const = 0;         // Pix to high element or 0
  virtual void      prev(Pix& pix) const = 0;  // preceding pix or 0
  virtual void      next(Pix& pix) const = 0;  // next pix or 0
  virtual int       owns(Pix p) const = 0;     // p is an OK Pix

// index<->Pix 

  virtual int       Pix_to_index(Pix p) const = 0;   // get index via Pix
  virtual Pix       index_to_Pix(int idx) const = 0; // Pix via index

// Growth

  virtual int       add_high(const TSpatialTimeSlice& elem) =0;// add new element at high end
                                                // return new high

  virtual int       add_low(const TSpatialTimeSlice& elem) = 0;   // add new low element,
                                                // return new low

// Shrinkage

  virtual int       del_high() = 0;           // remove the element at high end
                                          // return new high
  virtual int       del_low() = 0;        // delete low element, return new lo

                                          // caution: del_low/high
                                          // does not necessarily 
                                          // immediately call TSpatialTimeSlice::~TSpatialTimeSlice


// operations on multiple elements

  virtual void      fill(const TSpatialTimeSlice& x);          // set all elements = x
  virtual void      fill(const TSpatialTimeSlice& x, int from, int to); // fill from to to
  virtual void      clear() = 0;                // reset to zero-sized Plex
  virtual int       reset_low(int newlow); // change low index,return old
  virtual void      reverse();                   // reverse in-place
  virtual void      append(const TSpatialTimeSlicePlex& a);    // concatenate a copy
  virtual void      prepend(const TSpatialTimeSlicePlex& a);   // prepend a copy

// status

  virtual int       can_add_high() const = 0;
  virtual int       can_add_low() const = 0;
  
  int               length () const;       // number of slots

  int               empty () const;        // is the plex empty?
  virtual int       full() const = 0;      // it it full?

  int               chunk_size() const;    // report chunk size;

  virtual int       OK() const = 0;        // representation invariant

  void		    error(const char* msg) const;
  void		    index_error() const;
  void		    empty_error() const;
  void		    full_error() const;
};


// TSpatialTimeSliceIChunk ops

inline int TSpatialTimeSliceIChunk:: size() const
{
  return top - base;
}


inline int TSpatialTimeSliceIChunk:: base_index() const
{
  return base;
}

inline  int TSpatialTimeSliceIChunk:: low_index() const
{
  return low;
}

inline  int  TSpatialTimeSliceIChunk:: fence_index() const
{
  return fence;
}

inline  int  TSpatialTimeSliceIChunk:: top_index() const
{
  return top;
}

inline  TSpatialTimeSlice* TSpatialTimeSliceIChunk:: pointer_to(int i) const
{
  return &(data[i-base]);
}

inline  int  TSpatialTimeSliceIChunk:: index_of(const TSpatialTimeSlice* p) const
{
  return (p - data) / sizeof(TSpatialTimeSlice) + base;
}

inline  int  TSpatialTimeSliceIChunk:: possible_index(int i) const
{
  return i >= base && i < top;
}

inline  int  TSpatialTimeSliceIChunk:: possible_pointer(const TSpatialTimeSlice* p) const
{
  return p >= data && p < &(data[top-base]);
}

inline  int  TSpatialTimeSliceIChunk:: actual_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  TSpatialTimeSliceIChunk:: actual_pointer(const TSpatialTimeSlice* p) const
{
  return p >= data && p < &(data[fence-base]);
}

inline  int  TSpatialTimeSliceIChunk:: can_grow_high () const
{
  return fence < top;
}

inline  int  TSpatialTimeSliceIChunk:: can_grow_low () const
{
  return base < low;
}

inline  TSpatialTimeSlice* TSpatialTimeSliceIChunk:: invalidate()
{
  TSpatialTimeSlice* p = data;
  data = 0;
  return p;
}


inline TSpatialTimeSliceIChunk* TSpatialTimeSliceIChunk::prev() const
{
  return prv;
}

inline TSpatialTimeSliceIChunk* TSpatialTimeSliceIChunk::next() const
{
  return nxt;
}

inline void TSpatialTimeSliceIChunk::link_to_prev(TSpatialTimeSliceIChunk* prev)
{
  nxt = prev->nxt;
  prv = prev;
  nxt->prv = this;
  prv->nxt = this;
}

inline void TSpatialTimeSliceIChunk::link_to_next(TSpatialTimeSliceIChunk* next)
{
  prv = next->prv;
  nxt = next;
  nxt->prv = this;
  prv->nxt = this;
}

inline void TSpatialTimeSliceIChunk::unlink()
{
  TSpatialTimeSliceIChunk* n = nxt;
  TSpatialTimeSliceIChunk* p = prv;
  n->prv = p;
  p->nxt = n;
  prv = nxt = this;
}

inline  int TSpatialTimeSliceIChunk:: empty() const
{
  return low == fence;
}

inline  int  TSpatialTimeSliceIChunk:: full() const
{
  return top - base == fence - low;
}

inline int TSpatialTimeSliceIChunk:: first_index() const
{
  return (low == fence)? fence : low;
}

inline int TSpatialTimeSliceIChunk:: last_index() const
{
  return (low == fence)? low - 1 : fence - 1;
}

inline  int  TSpatialTimeSliceIChunk:: succ(int i) const
{
  return (i < low) ? low : i + 1;
}

inline  int  TSpatialTimeSliceIChunk:: pred(int i) const
{
  return (i > fence) ? (fence - 1) : i - 1;
}

inline  int  TSpatialTimeSliceIChunk:: valid_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  TSpatialTimeSliceIChunk:: valid_pointer(const TSpatialTimeSlice* p) const
{
  return p >= &(data[low - base]) && p < &(data[fence - base]);
}

inline  TSpatialTimeSlice* TSpatialTimeSliceIChunk:: grow_high ()
{
  if (!can_grow_high()) full_error();
  return &(data[fence++ - base]);
}

inline  TSpatialTimeSlice* TSpatialTimeSliceIChunk:: grow_low ()
{
  if (!can_grow_low()) full_error();
  return &(data[--low - base]);
}

inline  void TSpatialTimeSliceIChunk:: shrink_high ()
{
  if (empty()) empty_error();
  data[(fence--)-base].invalidate();
}

inline  void TSpatialTimeSliceIChunk:: shrink_low ()
{
  if (empty()) empty_error();
  data[(low++)-base].invalidate();
}

inline TSpatialTimeSlice* TSpatialTimeSliceIChunk::first_pointer() const
{
  return (low == fence)? 0 : &(data[low - base]);
}

inline TSpatialTimeSlice* TSpatialTimeSliceIChunk::last_pointer() const
{
  return (low == fence)? 0 : &(data[fence - base - 1]);
}

inline TSpatialTimeSlice* TSpatialTimeSliceIChunk::succ(TSpatialTimeSlice* p) const
{
  return ((p+1) <  &(data[low - base]) || (p+1) >= &(data[fence - base])) ? 
    0 : (p+1);
}

inline TSpatialTimeSlice* TSpatialTimeSliceIChunk::pred(TSpatialTimeSlice* p) const
{
  return ((p-1) <  &(data[low - base]) || (p-1) >= &(data[fence - base])) ? 
    0 : (p-1);
}


// generic Plex operations

inline TSpatialTimeSlicePlex::TSpatialTimeSlicePlex() {}

inline int TSpatialTimeSlicePlex::chunk_size() const
{
  return csize;
}

inline  int TSpatialTimeSlicePlex::ecnef () const
{
  return lo - 1;
}


inline  int TSpatialTimeSlicePlex::fence () const
{
  return fnc;
}

inline int TSpatialTimeSlicePlex::length () const
{
  return fnc - lo;
}

inline  int TSpatialTimeSlicePlex::empty () const
{
  return fnc == lo;
}

inline TSpatialTimeSliceIChunk* TSpatialTimeSlicePlex::tl() const
{
  return hd->prev();
}

inline int TSpatialTimeSlicePlex::one_chunk() const
{
  return hd == hd->prev();
}

/*
*************************************************************************
*									*
* TSpatialTimeSliceRPlex:      						*
*									*
*************************************************************************
*/


// minimum number of chunks to index

#ifndef MIN_NCHUNKS
#define MIN_NCHUNKS  16
#endif

class TSpatialTimeSliceRPlex: public TSpatialTimeSlicePlex
{
  int              base;        // base index of lowest chunk
  int              lch;         // index of lowest used chunk
  int              fch;         // 1 + index of highest used chunk
  int              maxch;       // max chunks in array
  TSpatialTimeSliceIChunk**      chunks;      // array of chunks
  TSpatialTimeSliceIChunk*       ch;          // cached chunk
  static floatVec fTmpBuff;

  void             make_initial_chunks(int up = 1);

  void             cache(int idx) const;
  void             cache(const TSpatialTimeSlice* p) const;
  TSpatialTimeSlice*             dopred(const TSpatialTimeSlice* p) const;
  TSpatialTimeSlice*             dosucc(const TSpatialTimeSlice* p) const;

  inline void             set_cache(const TSpatialTimeSliceIChunk* t) const; // logically, 
                                               // not physically const

public:
                   TSpatialTimeSliceRPlex();                 // set low = 0;
                                               // fence = 0;
                                               // csize = default

                   TSpatialTimeSliceRPlex(int ch_size);      // low = 0; 
                                               // fence = 0;
                                               // csize = ch_size

                   TSpatialTimeSliceRPlex(int lo,            // low = lo; 
                            int ch_size);      // fence=lo
                                               // csize = ch_size

                   TSpatialTimeSliceRPlex(int lo,            // low = lo
                            int hi,            // fence = hi+1
                            const TSpatialTimeSlice& initval,// fill with initval,
                            int ch_size = 0);  // csize= ch_size
                                               // or fence-lo if 0

                   TSpatialTimeSliceRPlex(const TSpatialTimeSliceRPlex&);

  inline                  ~TSpatialTimeSliceRPlex();

  void             operator= (const TSpatialTimeSliceRPlex&);

// virtuals

  inline TSpatialTimeSlice&             high_elem ();
  inline TSpatialTimeSlice&             low_elem ();

  inline const TSpatialTimeSlice&       high_element () const;
  inline const TSpatialTimeSlice&       low_element () const;

  inline  TSpatialTimeSlice& high_element (int offSet);
  inline TSpatialTimeSlice& low_element (int offSet); 

  inline Pix              first() const;
  inline Pix              last() const;
  inline void             prev(Pix& ptr) const;
  inline void             next(Pix& ptr) const;
  int              owns(Pix p) const;
  inline TSpatialTimeSlice&             elem (Pix p);
  inline const TSpatialTimeSlice&       operator () (Pix p) const;

  inline int              low() const; 
  inline int              high() const;
  inline int              valid(int idx) const;
  inline void             prev(int& idx) const;
  inline void             next(int& x) const;
  inline TSpatialTimeSlice&             elem (int index);
  inline const TSpatialTimeSlice&       operator [] (int index) const;
    
  inline int              Pix_to_index(Pix p) const;
  inline Pix              index_to_Pix(int idx) const;    

  inline int              can_add_high() const;
  inline int              can_add_low() const;
  inline int              full() const;

  int              add_high(const TSpatialTimeSlice& elem);
  TSpatialTimeSlice*  grow_high(int& current_index);
  int              del_high ();
  int              add_low (const TSpatialTimeSlice& elem);
  int              del_low ();
  
  void             fill(const TSpatialTimeSlice& x);
  void             fill(const TSpatialTimeSlice& x, int from, int to);
  void             clear();
  void             reverse();

  int              reset_low(int newlow);
    
  int              OK () const;

  const float*	   TimeSeriesData(int index);
  const float*	   TimeStampData();
  void             DumpToFile( FILE* oFile, int size ); 
};


inline void TSpatialTimeSliceRPlex::prev(int& idx) const
{
  --idx;
}

inline void TSpatialTimeSliceRPlex::next(int& idx) const 
{
  ++idx;
}

inline  int TSpatialTimeSliceRPlex::full ()  const
{
  return 0;
}

inline int TSpatialTimeSliceRPlex::can_add_high() const
{
  return 1;
}

inline int TSpatialTimeSliceRPlex::can_add_low() const
{
  return 1;
}

inline  int TSpatialTimeSliceRPlex::valid (int idx) const
{
  return idx >= lo && idx < fnc;
}

inline int TSpatialTimeSliceRPlex::low() const
{
  return lo;
}

inline int TSpatialTimeSliceRPlex::high() const
{
  return fnc - 1;
}

inline void TSpatialTimeSliceRPlex::set_cache(const TSpatialTimeSliceIChunk* t) const
{
  ((TSpatialTimeSliceRPlex*)(this))->ch = (TSpatialTimeSliceIChunk*)t;
}

inline void TSpatialTimeSliceRPlex::cache(int idx) const
{
  if (idx < lo || idx >= fnc) index_error();
  int index = (idx - base) / csize;
  TSpatialTimeSliceIChunk* cptr = chunks[index];
  if( cptr == 0 ) { sprintf(gMsgStr,"Error setting cache in TSpatialTimeSliceRPlex: index = %d ( idx:%d, base:%d, csize:%d )", index, idx, base, csize); gFatal(); }
  set_cache(cptr);
}  

inline  TSpatialTimeSlice& TSpatialTimeSliceRPlex::low_elem () 
{
  cache(lo); return *(ch->pointer_to(lo));
}

inline  TSpatialTimeSlice& TSpatialTimeSliceRPlex::high_elem ()
{
  cache(fnc-1); return *(ch->pointer_to(fnc - 1));
}

inline  TSpatialTimeSlice& TSpatialTimeSliceRPlex::high_element (int offSet)
{
  cache(fnc-1-offSet); return *(ch->pointer_to(fnc - 1 - offSet));
}

inline const TSpatialTimeSlice& TSpatialTimeSliceRPlex::low_element () const
{
  cache(lo); return *((const TSpatialTimeSlice*)(ch->pointer_to(lo)));
}

inline TSpatialTimeSlice& TSpatialTimeSliceRPlex::low_element (int offSet) 
{
  cache(lo+offSet); return *((TSpatialTimeSlice*)(ch->pointer_to(lo+offSet)));
}

inline const TSpatialTimeSlice& TSpatialTimeSliceRPlex::high_element () const
{
  cache(fnc-1); return *((const TSpatialTimeSlice*)(ch->pointer_to(fnc - 1)));
}

inline  int TSpatialTimeSliceRPlex::Pix_to_index(Pix px) const
{
  TSpatialTimeSlice* p = (TSpatialTimeSlice*)px;
  if (!ch->actual_pointer(p)) cache(p);
  return ch->index_of(p);
}

inline  Pix TSpatialTimeSliceRPlex::index_to_Pix(int idx) const
{
  if (!ch->actual_index(idx)) cache(idx);
  return (Pix)(ch->pointer_to(idx));
}

inline Pix TSpatialTimeSliceRPlex::first() const
{
  return Pix(hd->TSpatialTimeSliceIChunk::first_pointer());
}

inline Pix TSpatialTimeSliceRPlex::last() const
{
  return Pix(tl()->TSpatialTimeSliceIChunk::last_pointer());
}

inline void TSpatialTimeSliceRPlex::prev(Pix& p) const
{
  Pix q = Pix(ch->TSpatialTimeSliceIChunk::pred((TSpatialTimeSlice*)p));
  p = (q == 0)? Pix(dopred((TSpatialTimeSlice*)p)) : q;
}

inline void TSpatialTimeSliceRPlex::next(Pix& p) const
{
  Pix q = Pix(ch->TSpatialTimeSliceIChunk::succ((TSpatialTimeSlice*)p));
  p = (q == 0)? Pix(dosucc((TSpatialTimeSlice*)p)) : q;
}

inline TSpatialTimeSlice& TSpatialTimeSliceRPlex:: elem (Pix p)
{
  return *((TSpatialTimeSlice*)p);
}


inline TSpatialTimeSlice& TSpatialTimeSliceRPlex:: elem (int idx)
{
  cache(idx); return *(ch->pointer_to(idx));
}

inline const TSpatialTimeSlice& TSpatialTimeSliceRPlex:: operator () (Pix p) const
{
  return *((const TSpatialTimeSlice*)p);
}

inline const TSpatialTimeSlice& TSpatialTimeSliceRPlex:: operator [] (int idx) const
{
  cache(idx); return *((const TSpatialTimeSlice*)(ch->pointer_to(idx)));
}

inline TSpatialTimeSliceRPlex::~TSpatialTimeSliceRPlex()
{
  delete[] chunks;
}

/*
*************************************************************************
*									*
* TTemporalTimeSliceIChunk:      					
*									*
*************************************************************************
*/

// Plexes are made out of TTemporalTimeSliceIChunks


class TTemporalTimeSliceIChunk
{
//public: // kludge until C++ `protected' policies settled
protected:      

  TTemporalTimeSlice*           data;           // data, from client

  int            base;           // lowest possible index
  int            low;            // lowest valid index
  int            fence;          // highest valid index + 1
  int            top;            // highest possible index + 1

  TTemporalTimeSliceIChunk*     nxt;            // circular links
  TTemporalTimeSliceIChunk*     prv;

public:

// constructors

                 TTemporalTimeSliceIChunk(TTemporalTimeSlice*     d,       // ptr to array of elements
                        int      base_idx, // initial indices
                        int      low_idx,  
                        int      fence_idx,
                        int      top_idx);

  virtual       ~TTemporalTimeSliceIChunk();

// status reports

  int            size() const;    // number of slots

  inline virtual int    empty() const ;
  inline virtual int    full() const ; 

  int            can_grow_high () const ;  // there is space to add data
  int            can_grow_low () const;        
  
  int            base_index() const;   // lowest possible index;
  int            low_index() const;    // lowest actual index;
  inline virtual int    first_index() const;  // lowest valid index or fence if none
  inline virtual int    last_index() const;   // highest valid index or low-1 if none
  int            fence_index() const;  // highest actual index + 1
  int            top_index() const;    // highest possible index + 1

// indexing conversion

  int            possible_index(int i) const; // i between base and top
  int            actual_index(int i) const;   // i between low and fence
  inline virtual int    valid_index(int i) const;    // i not deleted (mainly for mchunks)

  int            possible_pointer(const TTemporalTimeSlice* p) const; // same for ptr
  int            actual_pointer(const TTemporalTimeSlice* p) const; 
  inline virtual int    valid_pointer(const TTemporalTimeSlice* p) const; 

  TTemporalTimeSlice*           pointer_to(int i) const ;   // pointer to data indexed by i
                                      // caution: i is not checked for validity
  int            index_of(const TTemporalTimeSlice* p) const; // index of data pointed to by p
                                      // caution: p is not checked for validity

  inline virtual int    succ(int idx) const;     // next valid index or fence if none
  inline virtual int    pred(int idx) const;     // previous index or low - 1 if none

  inline virtual TTemporalTimeSlice*   first_pointer() const;   // pointer to first valid pos or 0
  inline virtual TTemporalTimeSlice*   last_pointer() const;    // pointer to first valid pos or 0
  inline virtual TTemporalTimeSlice*   succ(TTemporalTimeSlice*  p) const;     // next pointer or 0
  inline virtual TTemporalTimeSlice*   pred(TTemporalTimeSlice* p) const;     // previous pointer or 0

// modification

  inline virtual TTemporalTimeSlice*   grow_high ();      // return spot to add an element
  inline virtual TTemporalTimeSlice*   grow_low ();  

  inline virtual void   shrink_high ();    // logically delete top index
  inline virtual void   shrink_low ();     

  virtual void   clear(int lo);     // reset to empty ch with base = lo
  virtual void   cleardown(int hi); // reset to empty ch with top = hi
  void           re_index(int lo);  // re-index so lo is new low

// chunk traversal

  TTemporalTimeSliceIChunk*     next() const;
  TTemporalTimeSliceIChunk*     prev() const;

  void           link_to_prev(TTemporalTimeSliceIChunk* prev);
  void           link_to_next(TTemporalTimeSliceIChunk* next);
  void           unlink();

// state checks

  TTemporalTimeSlice*           invalidate();     // mark self as invalid; return data
                                   // for possible deletion

  virtual int    OK() const;             // representation invariant

  void   error(const char*) const;
  void   empty_error() const;
  void   full_error() const;
  void   index_error() const;
};

// TTemporalTimeSlicePlex is a partly `abstract' class: few of the virtuals
// are implemented at the Plex level, only in the subclasses

class TTemporalTimeSlicePlex
{
protected:      

  TTemporalTimeSliceIChunk*       hd;          // a chunk holding the data
  int              lo;          // lowest  index
  int              fnc;         // highest index + 1
  int              csize;       // size of the chunk

  void             invalidate();              // mark so OK() is false
  void             del_chunk(TTemporalTimeSliceIChunk*);        // delete a chunk

  TTemporalTimeSliceIChunk*       tl() const;                // last chunk;
  int              one_chunk() const;         // true if hd == tl()

public:

// constructors, etc.

                    TTemporalTimeSlicePlex();                  // no-op

  virtual           ~TTemporalTimeSlicePlex();

  
// Access functions 
    
  virtual TTemporalTimeSlice&      elem (int idx) = 0; // access by index;
  virtual TTemporalTimeSlice&      elem (Pix p) = 0;   // access by Pix;

  virtual TTemporalTimeSlice&      high_elem () = 0;      // access high element
  virtual TTemporalTimeSlice&      low_elem () = 0;       // access low element

// read-only versions for const Plexes

  virtual const TTemporalTimeSlice& operator [] (int idx) const = 0; // access by index;
  virtual const TTemporalTimeSlice& operator () (Pix p) const = 0;   // access by Pix;

  virtual const TTemporalTimeSlice& high_element () const = 0;      // access high element
  virtual const TTemporalTimeSlice& low_element () const = 0;       // access low element


// Index functions

  virtual int       valid (int idx) const = 0;      // idx is an OK index

  virtual int       low() const = 0;         // lowest index or fence if none
  virtual int       high() const = 0;        // highest index or low-1 if none

  int               ecnef() const;         // low limit index (low-1)
  int               fence() const;         // high limit index (high+1)

  virtual void      prev(int& idx) const= 0; // set idx to preceding index
                                          // caution: pred may be out of bounds

  virtual void      next(int& idx) const = 0;       // set to next index
                                          // caution: succ may be out of bounds

  virtual Pix       first() const = 0;        // Pix to low element or 0
  virtual Pix       last() const = 0;         // Pix to high element or 0
  virtual void      prev(Pix& pix) const = 0;  // preceding pix or 0
  virtual void      next(Pix& pix) const = 0;  // next pix or 0
  virtual int       owns(Pix p) const = 0;     // p is an OK Pix

// index<->Pix 

  virtual int       Pix_to_index(Pix p) const = 0;   // get index via Pix
  virtual Pix       index_to_Pix(int idx) const = 0; // Pix via index

// Growth

  virtual int       add_high(const TTemporalTimeSlice& elem) =0;// add new element at high end
                                                // return new high

  virtual int       add_low(const TTemporalTimeSlice& elem) = 0;   // add new low element,
                                                // return new low

// Shrinkage

  virtual int       del_high() = 0;           // remove the element at high end
                                          // return new high
  virtual int       del_low() = 0;        // delete low element, return new lo

                                          // caution: del_low/high
                                          // does not necessarily 
                                          // immediately call TTemporalTimeSlice::~TTemporalTimeSlice


// operations on multiple elements

  virtual void      fill(const TTemporalTimeSlice& x);          // set all elements = x
  virtual void      fill(const TTemporalTimeSlice& x, int from, int to); // fill from to to
  virtual void      clear() = 0;                // reset to zero-sized Plex
  virtual int       reset_low(int newlow); // change low index,return old
  virtual void      reverse();                   // reverse in-place
  virtual void      append(const TTemporalTimeSlicePlex& a);    // concatenate a copy
  virtual void      prepend(const TTemporalTimeSlicePlex& a);   // prepend a copy

// status

  virtual int       can_add_high() const = 0;
  virtual int       can_add_low() const = 0;
  
  int               length () const;       // number of slots

  int               empty () const;        // is the plex empty?
  virtual int       full() const = 0;      // it it full?

  int               chunk_size() const;    // report chunk size;

  virtual int       OK() const = 0;        // representation invariant

  void		    error(const char* msg) const;
  void		    index_error() const;
  void		    empty_error() const;
  void		    full_error() const;
};


// TTemporalTimeSliceIChunk ops

inline int TTemporalTimeSliceIChunk:: size() const
{
  return top - base;
}


inline int TTemporalTimeSliceIChunk:: base_index() const
{
  return base;
}

inline  int TTemporalTimeSliceIChunk:: low_index() const
{
  return low;
}

inline  int  TTemporalTimeSliceIChunk:: fence_index() const
{
  return fence;
}

inline  int  TTemporalTimeSliceIChunk:: top_index() const
{
  return top;
}

inline  TTemporalTimeSlice* TTemporalTimeSliceIChunk:: pointer_to(int i) const
{
  return &(data[i-base]);
}

inline  int  TTemporalTimeSliceIChunk:: index_of(const TTemporalTimeSlice* p) const
{
  return (p - data) / sizeof(TTemporalTimeSlice) + base;
}

inline  int  TTemporalTimeSliceIChunk:: possible_index(int i) const
{
  return i >= base && i < top;
}

inline  int  TTemporalTimeSliceIChunk:: possible_pointer(const TTemporalTimeSlice* p) const
{
  return p >= data && p < &(data[top-base]);
}

inline  int  TTemporalTimeSliceIChunk:: actual_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  TTemporalTimeSliceIChunk:: actual_pointer(const TTemporalTimeSlice* p) const
{
  return p >= data && p < &(data[fence-base]);
}

inline  int  TTemporalTimeSliceIChunk:: can_grow_high () const
{
  return fence < top;
}

inline  int  TTemporalTimeSliceIChunk:: can_grow_low () const
{
  return base < low;
}

inline  TTemporalTimeSlice* TTemporalTimeSliceIChunk:: invalidate()
{
  TTemporalTimeSlice* p = data;
  data = 0;
  return p;
}


inline TTemporalTimeSliceIChunk* TTemporalTimeSliceIChunk::prev() const
{
  return prv;
}

inline TTemporalTimeSliceIChunk* TTemporalTimeSliceIChunk::next() const
{
  return nxt;
}

inline void TTemporalTimeSliceIChunk::link_to_prev(TTemporalTimeSliceIChunk* prev)
{
  nxt = prev->nxt;
  prv = prev;
  nxt->prv = this;
  prv->nxt = this;
}

inline void TTemporalTimeSliceIChunk::link_to_next(TTemporalTimeSliceIChunk* next)
{
  prv = next->prv;
  nxt = next;
  nxt->prv = this;
  prv->nxt = this;
}

inline void TTemporalTimeSliceIChunk::unlink()
{
  TTemporalTimeSliceIChunk* n = nxt;
  TTemporalTimeSliceIChunk* p = prv;
  n->prv = p;
  p->nxt = n;
  prv = nxt = this;
}

inline  int TTemporalTimeSliceIChunk:: empty() const
{
  return low == fence;
}

inline  int  TTemporalTimeSliceIChunk:: full() const
{
  return top - base == fence - low;
}

inline int TTemporalTimeSliceIChunk:: first_index() const
{
  return (low == fence)? fence : low;
}

inline int TTemporalTimeSliceIChunk:: last_index() const
{
  return (low == fence)? low - 1 : fence - 1;
}

inline  int  TTemporalTimeSliceIChunk:: succ(int i) const
{
  return (i < low) ? low : i + 1;
}

inline  int  TTemporalTimeSliceIChunk:: pred(int i) const
{
  return (i > fence) ? (fence - 1) : i - 1;
}

inline  int  TTemporalTimeSliceIChunk:: valid_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  TTemporalTimeSliceIChunk:: valid_pointer(const TTemporalTimeSlice* p) const
{
  return p >= &(data[low - base]) && p < &(data[fence - base]);
}

inline  TTemporalTimeSlice* TTemporalTimeSliceIChunk:: grow_high ()
{
  if (!can_grow_high()) full_error();
  return &(data[fence++ - base]);
}

inline  TTemporalTimeSlice* TTemporalTimeSliceIChunk:: grow_low ()
{
  if (!can_grow_low()) full_error();
  return &(data[--low - base]);
}

inline  void TTemporalTimeSliceIChunk:: shrink_high ()
{
  if (empty()) empty_error();
  --fence;
}

inline  void TTemporalTimeSliceIChunk:: shrink_low ()
{
  if (empty()) empty_error();
  ++low;
}

inline TTemporalTimeSlice* TTemporalTimeSliceIChunk::first_pointer() const
{
  return (low == fence)? 0 : &(data[low - base]);
}

inline TTemporalTimeSlice* TTemporalTimeSliceIChunk::last_pointer() const
{
  return (low == fence)? 0 : &(data[fence - base - 1]);
}

inline TTemporalTimeSlice* TTemporalTimeSliceIChunk::succ(TTemporalTimeSlice* p) const
{
  return ((p+1) <  &(data[low - base]) || (p+1) >= &(data[fence - base])) ? 
    0 : (p+1);
}

inline TTemporalTimeSlice* TTemporalTimeSliceIChunk::pred(TTemporalTimeSlice* p) const
{
  return ((p-1) <  &(data[low - base]) || (p-1) >= &(data[fence - base])) ? 
    0 : (p-1);
}


// generic Plex operations

inline TTemporalTimeSlicePlex::TTemporalTimeSlicePlex() {}

inline int TTemporalTimeSlicePlex::chunk_size() const
{
  return csize;
}

inline  int TTemporalTimeSlicePlex::ecnef () const
{
  return lo - 1;
}


inline  int TTemporalTimeSlicePlex::fence () const
{
  return fnc;
}

inline int TTemporalTimeSlicePlex::length () const
{
  return fnc - lo;
}

inline  int TTemporalTimeSlicePlex::empty () const
{
  return fnc == lo;
}

inline TTemporalTimeSliceIChunk* TTemporalTimeSlicePlex::tl() const
{
  return hd->prev();
}

inline int TTemporalTimeSlicePlex::one_chunk() const
{
  return hd == hd->prev();
}

/*
*************************************************************************
*									*
* TTemporalTimeSliceRPlex:      					
*									*
*************************************************************************
*/

class TTemporalTimeSliceRPlex: public TTemporalTimeSlicePlex
{
  int              base;        // base index of lowest chunk
  int              lch;         // index of lowest used chunk
  int              fch;         // 1 + index of highest used chunk
  int              maxch;       // max chunks in array
  TTemporalTimeSliceIChunk**      chunks;      // array of chunks
  TTemporalTimeSliceIChunk*       ch;          // cached chunk
  static floatVec fTmpBuff;
  static TTemporalTimeSlice* fGarbage;

  void             make_initial_chunks(int up = 1);

  int             cache(int idx) const;
  void             cache(const TTemporalTimeSlice* p) const;
  TTemporalTimeSlice*             dopred(const TTemporalTimeSlice* p) const;
  TTemporalTimeSlice*             dosucc(const TTemporalTimeSlice* p) const;

  inline void             set_cache(const TTemporalTimeSliceIChunk* t) const; // logically, 
                                               // not physically const

public:
                   TTemporalTimeSliceRPlex();                 // set low = 0;
                                               // fence = 0;
                                               // csize = default

                   TTemporalTimeSliceRPlex(int ch_size);      // low = 0; 
                                               // fence = 0;
                                               // csize = ch_size

                   TTemporalTimeSliceRPlex(int lo,            // low = lo; 
                            int ch_size);      // fence=lo
                                               // csize = ch_size

                   TTemporalTimeSliceRPlex(int lo,            // low = lo
                            int hi,            // fence = hi+1
                            const TTemporalTimeSlice& initval,// fill with initval,
                            int ch_size = 0);  // csize= ch_size
                                               // or fence-lo if 0

                   TTemporalTimeSliceRPlex(const TTemporalTimeSliceRPlex&);

  inline                  ~TTemporalTimeSliceRPlex();

  void             operator= (const TTemporalTimeSliceRPlex&);

// virtuals

	inline  const TTemporalTimeSlice& 					low_element (int offSet) const; 
	inline  const TTemporalTimeSlice& 					high_element (int offSet) const;

  inline TTemporalTimeSlice&       high_elem ();
  inline TTemporalTimeSlice&       low_elem ();

  inline const TTemporalTimeSlice&       high_element () const;
  inline const TTemporalTimeSlice&       low_element () const;

  inline Pix              first() const;
  inline Pix              last() const;
  inline void             prev(Pix& ptr) const;
  inline void             next(Pix& ptr) const;
  int              owns(Pix p) const;
  inline TTemporalTimeSlice&             elem (Pix p);
  inline const TTemporalTimeSlice&       operator () (Pix p) const;

  inline int              low() const; 
  inline int              high() const;
  inline int              valid(int idx) const;
  inline void             prev(int& idx) const;
  inline void             next(int& x) const;
  inline TTemporalTimeSlice&             elem (int index);
  inline const TTemporalTimeSlice&       operator [] (int index) const;
    
  inline int              Pix_to_index(Pix p) const;
  inline Pix              index_to_Pix(int idx) const;    

  inline int              can_add_high() const;
  inline int              can_add_low() const;
  inline int              full() const;

  int              add_high(const TTemporalTimeSlice& elem);
  TTemporalTimeSlice*     grow_high(int& current_index);
  int              del_high ();
  int              add_low (const TTemporalTimeSlice& elem);
  int              del_low ();

  void             fill(const TTemporalTimeSlice& x);
  void             fill(const TTemporalTimeSlice& x, int from, int to);
  void             clear();
  void             reverse();
	float            max();
	float            min();

  int              reset_low(int newlow);
    
  int              OK () const; 

  const float*	   TimeSeriesData();
  const float*	   TimeStampData();
  void             DumpToFile(FILE* oFile); 
  
	inline void Reset() { clear(); }
};

inline void TTemporalTimeSliceRPlex::prev(int& idx) const
{
  --idx;
}

inline void TTemporalTimeSliceRPlex::next(int& idx) const 
{
  ++idx;
}

inline  int TTemporalTimeSliceRPlex::full ()  const
{
  return 0;
}

inline int TTemporalTimeSliceRPlex::can_add_high() const
{
  return 1;
}

inline int TTemporalTimeSliceRPlex::can_add_low() const
{
  return 1;
}

inline  int TTemporalTimeSliceRPlex::valid (int idx) const
{
  return idx >= lo && idx < fnc;
}

inline int TTemporalTimeSliceRPlex::low() const
{
  return lo;
}

inline int TTemporalTimeSliceRPlex::high() const
{
  return fnc - 1;
}

inline void TTemporalTimeSliceRPlex::set_cache(const TTemporalTimeSliceIChunk* t) const
{
  ((TTemporalTimeSliceRPlex*)(this))->ch = (TTemporalTimeSliceIChunk*)t;
}

inline int TTemporalTimeSliceRPlex::cache(int idx) const
{
  if (idx < lo || idx >= fnc) { index_error(); return 0; }
  set_cache(chunks[(idx - base) / csize]);
  return 1;
}  

inline  const TTemporalTimeSlice& TTemporalTimeSliceRPlex::low_element ()  const 
{
  if( cache(lo) ) {  return *(ch->pointer_to(lo)); } else { return *fGarbage; }
}

inline  const TTemporalTimeSlice& TTemporalTimeSliceRPlex::high_element ()  const
{
  if( cache(fnc-1) ) {  return *(ch->pointer_to(fnc - 1)); } else { return *fGarbage; }
}

inline  TTemporalTimeSlice& TTemporalTimeSliceRPlex::low_elem()  
{
  if( cache(lo) ) {  return *(ch->pointer_to(lo)); } else { return *fGarbage; }
}

inline  TTemporalTimeSlice& TTemporalTimeSliceRPlex::high_elem ()  
{
  if( cache(fnc-1)  ) {  return *(ch->pointer_to(fnc - 1)); } else { return *fGarbage; }
}

inline const TTemporalTimeSlice& TTemporalTimeSliceRPlex::low_element (int offSet)  const
{
  if( cache(lo+offSet) ) { return *(ch->pointer_to(lo+offSet)); } else { return *fGarbage; }
}

inline const TTemporalTimeSlice& TTemporalTimeSliceRPlex::high_element (int offSet) const
{
  if( cache(fnc-1-offSet) ) {  return *(ch->pointer_to(fnc - 1 - offSet)); } else { return *fGarbage; }
}

inline  int TTemporalTimeSliceRPlex::Pix_to_index(Pix px) const
{
  TTemporalTimeSlice* p = (TTemporalTimeSlice*)px;
  if (!ch->actual_pointer(p)) cache(p);
  return ch->index_of(p);
}

inline  Pix TTemporalTimeSliceRPlex::index_to_Pix(int idx) const
{
  if (!ch->actual_index(idx)) cache(idx);
  return (Pix)(ch->pointer_to(idx));
}

inline Pix TTemporalTimeSliceRPlex::first() const
{
  return Pix(hd->TTemporalTimeSliceIChunk::first_pointer());
}

inline Pix TTemporalTimeSliceRPlex::last() const
{
  return Pix(tl()->TTemporalTimeSliceIChunk::last_pointer());
}

inline void TTemporalTimeSliceRPlex::prev(Pix& p) const
{
  Pix q = Pix(ch->TTemporalTimeSliceIChunk::pred((TTemporalTimeSlice*)p));
  p = (q == 0)? Pix(dopred((TTemporalTimeSlice*)p)) : q;
}

inline void TTemporalTimeSliceRPlex::next(Pix& p) const
{
  Pix q = Pix(ch->TTemporalTimeSliceIChunk::succ((TTemporalTimeSlice*)p));
  p = (q == 0)? Pix(dosucc((TTemporalTimeSlice*)p)) : q;
}

inline TTemporalTimeSlice& TTemporalTimeSliceRPlex:: elem (Pix p)
{
  return *((TTemporalTimeSlice*)p);
}


inline TTemporalTimeSlice& TTemporalTimeSliceRPlex:: elem (int idx)
{
  cache(idx); return *(ch->pointer_to(idx));
}

inline const TTemporalTimeSlice& TTemporalTimeSliceRPlex:: operator () (Pix p) const
{
  return *((const TTemporalTimeSlice*)p);
}

inline const TTemporalTimeSlice& TTemporalTimeSliceRPlex:: operator [] (int idx) const
{
  cache(idx); return *((const TTemporalTimeSlice*)(ch->pointer_to(idx)));
}

inline TTemporalTimeSliceRPlex::~TTemporalTimeSliceRPlex()
{
  delete[] chunks;
}

#endif
