#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include "dfit.h" 
#include "lex.h"
#include "Utility.h"
#include "Environ.h"

#define LEX_WEIGHT     (-1)
#define LEX_METHOD     (-2)
#define LEX_BOUNDS     (-3)
#define LEX_WBOUNDS    (-4)
#define LEX_CINT       (-5)
#define LEX_WCINT      (-6)
#define LEX_DBK        (-7)
#define LEX_REFERENCE  (-8)
#define LEX_FILE       (-9)
#define LEX_DATA       (-10)
#define LEX_FREQ       (-11)
#define LEX_VARIABLE   (-12)
#define LEX_INDEX      (-13)
#define LEX_READ       (-14)
#define LEX_THEIL      (-15)
#define LEX_ERRCOMP    (-16)
#define LEX_OUTPUT     (-17)
#define LEX_STEADY     (-18)
#define LEX_INCREASE   (-19)
#define LEX_DECREASE   (-20)
#define LEX_TREND      (-21)
#define LEX_PARMDEF    (-22)
#define LEX_MISSING    (-23)

class TLexDfit : public TLexS 
{
  // this understands parameter definitions like parmname=parmvalue
  // '=' must be in terminators list

  TString _parm;
  
protected:
  
  virtual int postprocess_token(int val, const char* tok, int& alt);

public:
  
  const char* parm() { return _parm; }

  TLexDfit(const char* filename, const char* terminators, TArray*
	keywords = NULL) : 
    TLexS(filename, terminators, keywords) 
  {}
  virtual ~TLexDfit() {}
};

int TLexDfit::postprocess_token(int val, const char* tok, int& alt)
{ 
    if (val == LEX_IDENTIFIER)
    {
      _parm = tok;
      char c = nextc();
      if (c  == '=')
	{
	  do 
	    {
	      _parm += c;
	      c = nextc();
	    }
	  while (strchr(terminators(), c) == NULL);
	  val = LEX_PARMDEF;
	}
      pushc(c);
    }
    return TLexS::postprocess_token(val, tok, alt);
}


static char __buf_string[1024];

class VDesc : public TObject
{
public:
  // I know, I know
  CString name;
  CString array_element;
  TFilename datafile;
  TFilename refdatafile;
  int d_indexes[24];
  int r_indexes[24];
  double* refdata;
  double* data;
  int nrefdata;
  int ndata;
  TArray refstats;    // stats collection for ref data
  TArray datstats;    // stats for simulation data
  TArray teststats;   // descriptors of selected tests
  TArray outdefs;     // output definition for each var

  int rdatacols;      // number of columns stored in rdata (row-first)
  int datacols;       // number of columns stored in data (row-first)
  
  double testweights[DFIT_MAXTESTS];

  double weight;
  double score;

  VDesc(const char* name);
  virtual ~VDesc();
};

VDesc::VDesc(const char* n) : 
  data(NULL), refdata(NULL), ndata(0), nrefdata(0), name(n)
{
  for (int i = 0; i < DFIT_MAXTESTS; i++)
    testweights[i] = 1.0;
}

VDesc::~VDesc()
{
  if (data != NULL) delete data;
  if (refdata != NULL) delete refdata;
}

int DFit::test_id(const char* test)
{

  if (strncmp(test, "BOUNDS", 6) == 0)
    return MID_BOUNDS;
  if (strncmp(test, "WBOUNDS", 7) == 0)
    return MID_WBOUNDS;
  if (strncmp(test, "FREQ", 4) == 0)
    return MID_FREQ;
  if (strncmp(test, "DBK", 3) == 0)
    return MID_DBK;
  if (strncmp(test, "CINT", 4) == 0)
    return MID_CONF;
  if (strncmp(test, "WCINT", 5) == 0)
    return MID_WCONF;
  if (strncmp(test, "THEIL", 5) == 0)
    return MID_THEIL;
  if (strncmp(test, "ERRCOMP", 7) == 0)
    return MID_ERRCOMP;
  if (strncmp(test, "STEADY", 6) == 0)
    return MID_STEADY;
  if (strncmp(test, "SHAPE", 5) == 0)
    return MID_SHAPE;
  if (strncmp(test, "INCREASE", 8) == 0)
    return MID_INCREASE;
  if (strncmp(test, "DECREASE", 8) == 0)
    return MID_DECREASE;
  if (strncmp(test, "TREND", 5) == 0)
    return MID_TREND;

  CHECKS(0, "Unknown test", test);
  return 0;
}

const char* DFit::id_test(int id)
{
  
  char* res = "";
  
  switch(id) 
    {
    case MID_BOUNDS:
      res = "BOUNDS";
      break;
    case MID_WBOUNDS:
      res = "WBOUNDS";
      break;
    case MID_FREQ:
      res = "FREQ";
      break;
    case MID_DBK:
      res = "DBK";
      break;
    case MID_CONF:
      res = "CINT";
      break;
    case MID_WCONF:
      res = "WCINT";
      break;
    case MID_THEIL:
      res = "THEIL";
      break;
    case MID_ERRCOMP:
      res = "ERRCOMP";
      break;
    case MID_STEADY:
      res = "STEADY";
      break;
    case MID_SHAPE:
      res = "SHAPE";
      break;
    case MID_INCREASE:
      res = "INCREASE";
      break;
    case MID_DECREASE:
      res = "DECREASE";
      break;
    case MID_TREND:
      res = "TREND";
      break;
    }

  return (const char*)res;
}

Bool DFit::has_test( VDesc* vdesc, int test_id ) {
  Bool rv = False;
  if( vdesc != NULL ) {
	for (int i = 0; i < vdesc->teststats.items(); i++) {
		FitTest& t =  (FitTest&)vdesc->teststats[i];
		if (t.test_id() == test_id) {
		  rv = True; break;
		}
	}
  }
  return rv;
}

float* DFit::get_refdata( VDesc* vdesc, int& dsize ) {
  dsize = vdesc->nrefdata;
  float* fdata = new float[dsize];
  for( int j=0; j<dsize; j++ ) { fdata[j] = vdesc->refdata[j]; }
  return fdata;
}

void DFit::error(const char* m) {
  gFatal(m);
}

void DFit::read_config(const char* file)
{
  _vars.destroy();
  _varnames.destroy();
  _outdefs.destroy();

  double vweight = 1.0;
  double dweights[24];
  for (int i = 0; i < 24; i++)
    dweights[i] = 1.0;
  
  TArray kw; TToken_string* tt;
  TArray cparms;
  TString od, of;

  tt = new TToken_string("WEIGHT");    tt->add(LEX_WEIGHT);    kw.add(tt);
  tt = new TToken_string("TEST");      tt->add(LEX_METHOD);    kw.add(tt);
  tt = new TToken_string("REFERENCE"); tt->add(LEX_REFERENCE); kw.add(tt);
  tt = new TToken_string("FILE");      tt->add(LEX_FILE);      kw.add(tt);
  tt = new TToken_string("DATA");      tt->add(LEX_DATA);      kw.add(tt);
  tt = new TToken_string("VARIABLE");  tt->add(LEX_VARIABLE);  kw.add(tt);
  tt = new TToken_string("INDEX");     tt->add(LEX_INDEX);     kw.add(tt);
  tt = new TToken_string("READ");      tt->add(LEX_READ);      kw.add(tt);
  tt = new TToken_string("OUTPUT");    tt->add(LEX_OUTPUT);    kw.add(tt);

  // Stupid bounds test
  tt = new TToken_string("BOUNDS");    tt->add(LEX_BOUNDS);    kw.add(tt);
  tt = new TToken_string("WBOUNDS");   tt->add(LEX_WBOUNDS);   kw.add(tt);
  // Stupid confidence interval test
  tt = new TToken_string("CINT");      tt->add(LEX_CINT);      kw.add(tt);
  tt = new TToken_string("WCINT");     tt->add(LEX_WCINT);     kw.add(tt);
  // Dent and Blackie deterministic timeseries F
  tt = new TToken_string("DBK");       tt->add(LEX_DBK);       kw.add(tt);
  // Theil inequality coefficient and error composition score
  tt = new TToken_string("THEIL");     tt->add(LEX_THEIL);     kw.add(tt);
  tt = new TToken_string("ERRCOMP");   tt->add(LEX_ERRCOMP);   kw.add(tt);
  // Steady state / increasing / decreasing / trend test
  tt = new TToken_string("STEADY");    tt->add(LEX_STEADY);    kw.add(tt);
  tt = new TToken_string("INCREASE");  tt->add(LEX_INCREASE);  kw.add(tt);
  tt = new TToken_string("DECREASE");  tt->add(LEX_DECREASE);  kw.add(tt);
  tt = new TToken_string("TREND");     tt->add(LEX_TREND);     kw.add(tt);

  // Smart frequency test
  tt = new TToken_string("FREQ");      tt->add(LEX_FREQ);      kw.add(tt);

  // Missing data option
  tt = new TToken_string("MISSING");   tt->add(LEX_MISSING);      kw.add(tt);

  TLexDfit lex(file, " \t\n\"=", &kw);

  // read file
  TString cvar("");
  TToken_string datastring(128);

  VDesc* cvo = NULL;
  float weight  = 0.0;
  float missing = 0.0;
  FitTest* cvm = NULL;
  int misdef;
  int cprm;
  int tok = lex.token();

  while (tok != LEX_ENDFILE)
    {
      switch (tok) 
	{
	case LEX_METHOD:

	  cparms.destroy();

	  int cmet; cprm = 0;
	  
	  // create appropriate test object and append to vars tests
	  switch (tok = lex.token())
	    {
	    case LEX_BOUNDS:
	      cmet = MID_BOUNDS;
	      break;
	    case LEX_WBOUNDS:
	      cmet = MID_WBOUNDS;
	      break;
	    case LEX_CINT:
	      cmet = MID_CONF;
	      break;
	    case LEX_WCINT:
	      cmet = MID_WCONF;
	      break;
	    case LEX_DBK:
	      cmet = MID_DBK;
	      break;
	    case LEX_THEIL:
	      cmet = MID_THEIL;
	      break;
	    case LEX_ERRCOMP:
	      cmet = MID_ERRCOMP;
	      break;
	    case LEX_STEADY:
	      cmet = MID_STEADY;
	      break;
	    case LEX_INCREASE:
	      cmet = MID_INCREASE;
	      break;
	    case LEX_DECREASE:
	      cmet = MID_DECREASE;
	      break;
	    case LEX_FREQ:
	      cmet = MID_FREQ;
	      break;
	    default:
	      error("method not implemented");
	      break;
	    }
	  
	  weight = dweights[cmet];
	  
	  tok = lex.token();

	  while (tok == LEX_PARMDEF)
	    {
	      cparms.add(new TToken_string(lex.parm(), '='));
	      tok = lex.token();
	    }

	  if (tok == LEX_MISSING)
	    {
	      tok = lex.token();
	      if (tok == LEX_FLOAT || tok == LEX_INTEGER)
		{
		  missing = atof(lex.token_val());
		  misdef = TRUE;
		}
	      else
		error("value expected after MISSING");
	      tok = lex.token();
	    }
	  
	  if (tok == LEX_WEIGHT)
	    {
	      tok = lex.token();
	      if (tok == LEX_FLOAT || tok == LEX_INTEGER)
		weight = atof(lex.token_val());
	      else
		error("number expected after WEIGHT");
	      tok = lex.token();
	    }
	  
	  // if variable is being defined, create method with 
	  // parameters; otherwise, just store weight as default
	  if (cvar != "")
	    {
	      switch(cmet)
		{
		case MID_BOUNDS:
		  cvm = new FitTest_bounds(1, cparms);
		  break;
		case MID_WBOUNDS:
		  cvm = new FitTest_bounds(2, cparms);
		  break;
		case MID_CONF:
		  cvm = new FitTest_bounds(3, cparms);
		  break;
		case MID_WCONF:
		  cvm = new FitTest_bounds(4, cparms);
		  break;
		case MID_DBK:
		  cvm = new FitTest_dbk(cparms);
		  break;
		case MID_THEIL:
		  cvm = new FitTest_Theil(cparms);
		  break;
		case MID_ERRCOMP:
		  cvm = new FitTest_Errcomp(cparms);
		  break;
		case MID_STEADY:
		  cvm = new FitTest_steadystate(0, cparms); 
		  break;
		case MID_INCREASE:
		  cvm = new FitTest_steadystate(1, cparms);
		  break;
		case MID_DECREASE:
		  cvm = new FitTest_steadystate(-1, cparms);
		  break;
		case MID_TREND:
		  cvm = new FitTest_steadystate(2, cparms);
		  break;
		case MID_FREQ:
		  cvm = new FitTest_Freq(cparms);
		  break;
		}
	      cvm->weight() = weight;
	      cvo->teststats.add(cvm);
	      if (misdef) 
		{
		  //cvo->missing() = missing;
		  //cvo->misdef()  = TRUE;
		  misdef = FALSE;
		}
	    }
	  else dweights[cmet] = weight;
	  break;

	case LEX_OUTPUT:
	 
	  of = "";
	  lex.expect(LEX_STRING);
	  od = lex.sval();
	  if ((tok = lex.token()) == LEX_IDENTIFIER)
	    {
	      of = lex.token_val();
	      tok = lex.token();
	    }
	  
	  if (cvar == "")
	    _outdefs.add(new OutDef(od, of));
	  else
	      cvo->outdefs.add(new OutDef(od, of));
	  break;
	  

	case LEX_REFERENCE:

	  tok = lex.token();
	  if (tok == LEX_READ)
	    // read immediate: format is IDs for columns followed by data
	    {
	      int cnt = 0;
	      while ((tok = lex.token()) == LEX_IDENTIFIER) cnt++;
	      datastring = "";
	      while (tok == LEX_FLOAT || tok == LEX_INTEGER)
		{
		  datastring.add(lex.token_val());
		  tok = lex.token();
		}
	      double* ddata = new double[datastring.items()];
	      for (int k = 0; k < datastring.items(); k++)
		ddata[k] = atof(datastring.get(k));

	      cvo->nrefdata = datastring.items();
	      cvo->refdata = ddata;
	      cvo->rdatacols = cnt == 0 ? 1 : cnt;
	    }
	  else if (tok == LEX_FILE)
	    {
	      cvo->rdatacols = 0;
	      cvo->r_indexes[0] = 0; 
	      // read datafile in
	      tok = lex.token();

	      if (tok != LEX_IDENTIFIER)
		error("filename expected after FILE");

	      cvo->refdatafile = lex.token_val();
	      tok = lex.token();

	      if (tok == LEX_INDEX)
		{
		  while ((tok = lex.token()) == LEX_INTEGER)
		    cvo->r_indexes[cvo->rdatacols++] = lex.int_val();
		}
	      if (cvo->rdatacols == 0) cvo->rdatacols = 1;
	    }
	  break;

	case LEX_DATA:

	  tok = lex.token();
	  if (tok == LEX_READ)
	    // read immediate: format is IDs for columns followed by data
	    {
	      int cnt = 0;
	      while ((tok = lex.token()) == LEX_IDENTIFIER) cnt++;
	      datastring = "";
	      while (tok == LEX_FLOAT || tok == LEX_INTEGER)
		{
		  datastring.add(lex.token_val());
		  tok = lex.token();
		}
	      double* rdata = new double[datastring.items()];
	      for (int k = 0; k < datastring.items(); k++)
		rdata[k] = atof(datastring.get(k));

	      cvo->ndata = datastring.items();
	      cvo->data = rdata;
	      cvo->datacols = cnt == 0 ? 1 : cnt;
	    }
	  else if (tok == LEX_FILE)
	    {
	      cvo->datacols = 0;
	      cvo->d_indexes[0] = 0; 
	      // read datafile in
	      tok = lex.token();

	      if (tok != LEX_IDENTIFIER)
		error("filename expected after FILE");

	      cvo->datafile = lex.token_val();

	      tok = lex.token();
	    
	      if (tok == LEX_INDEX)
		{
		  while ((tok = lex.token()) == LEX_INTEGER)
		    cvo->d_indexes[cvo->datacols++] = lex.int_val();
		}

	      if (cvo->datacols == 0) cvo->datacols = 1;

	    }
	  else if (tok == LEX_INDEX)
	    {
	      // read datafile as indexed column in file (TBI)
	      while ((tok = lex.token()) == LEX_INTEGER)
		;
	    }
	  break;

	case LEX_WEIGHT:

	  // default weight for all variables
	  tok = lex.token();
	  if (tok == LEX_FLOAT || tok == LEX_INTEGER)
	    vweight = atof(lex.token_val());
	  else
	    error("number expected after WEIGHT");
	  tok = lex.token();
	  break;

	case LEX_VARIABLE:
	  
	  lex.expect(LEX_IDENTIFIER);
	  cvar = lex.token_val();
	  _varnames.add(new TString(cvar));
	  _vars.add(cvar, cvo = new VDesc(cvar), TRUE);
	  cvo->weight = vweight;
	  
	  if ((tok = lex.token()) == LEX_WEIGHT)
	    {
	      tok = lex.token();
	      if (tok == LEX_FLOAT || tok == LEX_INTEGER)
		cvo->weight = atof(lex.token_val());
	      else
		error("number expected after WEIGHT");
	      tok = lex.token();
	    }
	  
	  break;
	default:
	  error(format("syntax error at line %d", lex.line()+1));
	  break;
	}
    }

  read_refdata();
}

void DFit::compute()
{
  // compute required tests for all variables
  double tvscore = 0.0, pvscore = 0.0;
  
  if (!_dataread) read_data();

  _vars.restart();

  for (int i = 0; i < _vars.items(); i++)
    {
      VDesc* v = (VDesc*)_vars.get();

      double tscore = 0.0, pscore = 0.0;

      if (_verbose)
	printf("Calculating %s...\n", (const char*)v->name);

      // calculate variable score
      for (int j = 0; j < v->teststats.items(); j++)
	{
	  FitTest& t =  (FitTest&)v->teststats[j];

	  if (_verbose)
	    printf("\t%s... ", id_test(t.test_id()));

	  tscore += t.weight();
	  t.compute(v->data, v->ndata);
	  pscore += t.weight()*t.score();

	  if (_verbose)
	    printf("%lf\n", t.score());
	}
      
      v->score = tscore == 0.0 ? 0.0 : (pscore/tscore);
      
      tvscore += v->weight;
      pvscore += v->weight*v->score;

      if (_verbose) printf("var score was %lf\n", v->score);
    }
  
  // calculate total score
  _score = tvscore == 0.0 ? 0.0 : (pvscore/tvscore); 
  _ok = TRUE;

  if (_verbose) printf("Total score was %lf\n", _score);

}

void DFit::report() {
  CHECK(_ok, "computations have not been done");
  int i,j;
  for (i = 0; i < _outdefs.items(); i++) {
      OutDef& od = (OutDef&)_outdefs[i];
      report_object(od.form, od.file);
  }

  for (i = 0; i < _varnames.items(); i++) {
      TString& varn = (TString&)_varnames[i];
      VDesc& v = (VDesc&)_vars[varn];

      for(j = 0; j < v.outdefs.items(); j++) {
		OutDef& od = (OutDef&)(v.outdefs[j]);
		report_object(od.form, od.file, &v);
	  }
  }
}

void DFit::report_object(const char* form, const char* f, VDesc* v)
{

  // if file name ends with +, append to the file, otherwise overwrite
  char* om = "w";
  TFilename fn(f);

  if (!fn.empty() && fn[fn.size()-1] == '+')
    {
      om = "a";
      fn.cut(fn.size() - 1);
    }

  FILE* fp = (fn.empty() ? stdout : fopen((const char*)fn, om));

  CHECKS(fp != NULL, "Can't open file:", f);
  
  char c = *form++;  char id[256];

  while (c)
    {
      if (c == '$')
	{
	  int cnt = 0; 
	  c = *form++;
	  while (isalnum(c) || c == '(' || 
		 c == ')' || c == '_')
	    {
	      id[cnt++] = c;
	      c = *form++;
	    }
	  id[cnt] = '\0';

	  // process ID: check keywords, otherwise locate test and
	  // retrieve variable value
	  if (strncmp(id, "DATE", 4) == 0)
	    {
	      time_t t = time(NULL);
	      // one day I'll understand why they put a newline 
	      // at the end of the date string
	      char bt[32];
	      strcpy(bt, ctime(&t));
	      bt[strlen(bt)-1] = '\0';
	      fprintf(fp, bt);
	    }
	  else if (strncmp(id, "ID", 2) == 0)
	    {
	      fprintf(fp, "%s", v == NULL ? (const char*)_id : 
		      (const char*)v->name);
	    }
	  else if (strncmp(id, "SCORE", 5) == 0)
	    {
	      fprintf(fp, "%g", v == NULL ? _score : v->score);
	    }
	  else if (v != NULL)
	    {
	      CString test(id), parm(""); int pp;
	      if ((pp = test.find('(')) != -1)
		{
		  CHECKS (test[test.size() - 1] != ')',
			  "syntax error: missing close parenthesis", id);
		  parm = test.sub(pp+1, test.size()-1);
		  test.cut(pp);
		}

	      // look for test
	      int t, tid = test_id(test);
	      for(t = 0; t < v->teststats.items(); t++)
		{
		  FitTest& ft = (FitTest&)(v->teststats[t]);
		  if (ft.test_id() == tid)
		    {
		      if (parm.empty())
			fprintf(fp, "%g", ft.score());
		      else 
			fprintf(fp, "%s", ft.get_variable(parm));
		      break;
		    }
		}
	      if (t == v->teststats.items())
		fprintf(fp, "(nil)");
	    }
	  else fprintf(fp, "(nil)");
	}
      else 
	{
	  fprintf(fp, "%c", c);
	  c = *form++;
	}
    }
  
  if (*f != '\0') fclose(fp); 
}

void DFit::set_data( VDesc* v, double* data, int ndata) {
  v->data  = data;
  v->ndata = ndata;
}

int DFit::configure_ref_data(const char* varname, const char* data_filename, int* cols ) {
  VDesc* cvo = var_desc( varname, False );
  if( cvo == NULL ) return 1;
  
  cvo->rdatacols = 0;
  cvo->r_indexes[0] = 0; 

  cvo->refdatafile = data_filename;
  
  if( cols != NULL ) {
	int i=0;
	while ( cols[i] >= 0 ) {
	  cvo->r_indexes[cvo->rdatacols++] = cols[i++];
	}
  }
  if (cvo->rdatacols == 0) cvo->rdatacols = 1;
}	    

double DFit::var_score(const char* varname, int method_id )
{
  CHECKS(_vars.is_key(varname), "Variable does not exist", varname);
  VDesc& v = (VDesc&)_vars[varname]; 

  if (method_id == MID_GLOBAL)
    return v.score;
  
  for (int i = 0; i < v.teststats.items(); i++)
    {
      FitTest& t =  (FitTest&)v.teststats[i];
      if (t.test_id() == method_id)
	return t.score();
    }
  
  error("method id not found for variable");
  return 0.0;
}

int DFit::register_test(const char* name, const char* test, 
			float weight, int argc, char *argv[]) 
{
  VDesc* v = var_desc(name);
  FitTest* cvm = NULL;

  TArray cparms;
  for (int i = 0; i < argc; i++) {
	if( strncmp(argv[i],"array.element",13) == 0 ) {
	  CString arg(argv[i]);
	  v->array_element = arg.after("=");
	  v->array_element.strip(" ");
	} else {
	  cparms.add(new TToken_string(argv[i], '='));
	}
  }
  
  switch(test_id(test))
    {
    case MID_BOUNDS:
      cvm = new FitTest_bounds(1, cparms);
      break;
    case MID_WBOUNDS:
      cvm = new FitTest_bounds(2, cparms);
      break;
    case MID_CONF:
      cvm = new FitTest_bounds(3, cparms);
      break;
    case MID_WCONF:
      cvm = new FitTest_bounds(4, cparms);
      break;
    case MID_DBK:
      cvm = new FitTest_dbk(cparms);
      break;
    case MID_THEIL:
      cvm = new FitTest_Theil(cparms);
      break;
    case MID_ERRCOMP:
      cvm = new FitTest_Errcomp(cparms);
      break;
    case MID_STEADY:
      cvm = new FitTest_steadystate(0, cparms); 
      break;
    case MID_INCREASE:
      cvm = new FitTest_steadystate(1, cparms);
      break;
    case MID_DECREASE:
      cvm = new FitTest_steadystate(-1, cparms);
      break;
    case MID_TREND:
      cvm = new FitTest_steadystate(2, cparms);
      break;
    case MID_FREQ:
      cvm = new FitTest_Freq(cparms);
      break;
    }
  if (cvm)
    {
      cvm->weight() = (double)weight;
      v->teststats.add(cvm);
    }
  else error("dfit: invalid test specification or memory error"); 
  return 0;
}

VDesc* DFit::var_desc(const char* name, Bool create ) {
  VDesc* cvo = NULL;
  if( _vars.is_key(name) ) { 
	cvo = (VDesc*) &_vars[name];
  } else if( create ) {  
	cvo = new VDesc(name);
	_varnames.add( new CString(name) );
	_vars.add(name, cvo, TRUE);
	gPrintScreen( format(" Adding DFit:VDesc %s",name) );
  } 
  return cvo;
}

int DFit::set_weight( const char* name, float weight )  {
  FitTest* cvm = NULL;
  VDesc* v = var_desc(name);
  v->weight = (double)weight;
  return 0;
}


void DFit::read_refdata()  {
	Env::Fatal( " DFit::read_refdata() must be recoded" );
	/*
  TToken_string datastring(4086);

  for (int i = 0; i < _varnames.items(); i++) {
      TString& varn = (TString&)_varnames[i];
      VDesc& v = (VDesc&)_vars[varn];
      
      if (v.refdatafile.empty()) continue;

      ifstream in((const char*)(v.refdatafile));
 //     if (!in.is_open()) { error( format( "can't open input file %s", v.refdatafile.chars() ) ); }
      
      // read file
      TToken_string t; int nfields = 0; datastring = "";
	  while (in.good() && !in.eof()) {
		in.getline(__buf_string, sizeof(__buf_string), '\n');
		t.tokenize(__buf_string);

		if (t.items() == 0)   continue;

		if (nfields == 0) {
		  nfields = t.items();
		} else {
		  if (t.items() != nfields)
			error("number of columns in file is not consistent");
		}

		for (int j = 0; j < v.rdatacols; j++) {
		  datastring.add(t.get(v.r_indexes[j]));
		}
	  }
      
      if (v.refdata != NULL) delete v.refdata;
      v.refdata = new double[v.nrefdata = datastring.items()];
      for (int k = 0; k < datastring.items(); k++) {
		v.refdata[k] = atof(datastring.get(k));
	  }

      // reinitialize all tests
      for (int j = 0; j < v.teststats.items(); j++) {
		FitTest& t =  (FitTest&)v.teststats[j];
		t.initialize(v.refdata, v.nrefdata);
	  }
    }
    */
}

void DFit::read_data()
{
	Env::Fatal( " DFit::read_data() must be recoded" );
	/*
  TToken_string datastring(4086);

  for (int i = 0; i < _varnames.items(); i++)
    {
      TString& varn = (TString&)_varnames[i];
      VDesc& v = (VDesc&)_vars[varn];

      if (v.datafile.empty())
	continue;

      ifstream in((const char*)(v.datafile));
//      if (!in.is_open()) { error("can't open input file"); }
      
      // read file
      TToken_string t; int nfields = 0; datastring = "";
      while (in.good() && !in.eof())
	{
	  in.getline(__buf_string, sizeof(__buf_string), '\n');
	  t.tokenize(__buf_string);

	  if (t.items() == 0) 
	    continue;

	  if (nfields == 0) 
	    nfields = t.items();
	  else {
	    if (t.items() != nfields)
	      error("number of columns in file is not consistent");
	  }
	  
	  for (int j = 0; j < v.datacols; j++)
	    datastring.add(t.get(v.d_indexes[j]));
	}
      
      if (v.data != NULL) 
	delete v.data;
      v.data = new double[v.ndata = datastring.items()];
      for (int k = 0; k < datastring.items(); k++)
	v.data[k] = atof(datastring.get(k));
    }
  _dataread = TRUE;
  */
}


double DFit::recalc_score(TToken_string& t)
{
  if (t.items() != _varnames.items())
    error("number of weights does not match number of variables");

  double tw = 0.0, ts = 0.0;

  // weights are supposed to use varnames order
  
  for (int i = 0; i < _varnames.items(); i++)
    {
      TString& varn = (TString&)_varnames[i];
      VDesc& v = (VDesc&)_vars[varn];

      double nw = (double)t.get_float(i);
      
      tw += nw;
      ts += (nw * v.score);
    }

  return tw == 0.0 ? 0.0 : (ts/tw); 
  
}
