#ifndef __TLEX_H
#define __TLEX_H

// ------------------------------------------------------------------------
// basic lexical analyzer: recognizes integers, floats, separators,
// and identifiers. Terminators (passed in constructor) are returned as
// their own integer value. Does not recognize quoted strings etc.
// but can be extended by providing an appropriate postprocess_token 
// implementation. TLexS recognizes double-quoted strings as well.
// A pointer to a TArray containing Token_strings (of the form (name|value) 
// can be passed to constructor to specify keywords. When the keyword <name>
// is found <value> is returned. The value should be negative to prevent
// conflicts with terminators (-1 to -999 can be used safely).
// Recognizes C-style (/* to */) comments. 
// -------------------------------------------------------------------------

#define LEX_UNKNOWN    -1000
#define LEX_ENDFILE    -1001
#define LEX_INTEGER    -1002
#define LEX_FLOAT      -1003
#define LEX_IDENTIFIER -1004
#define LEX_IGNORE     -1005

#include <stdio.h>

#include "object.h"
#include "Plex.h"
#include "CString.h"

class TLex : public TObject 
{
  FILE*   _lexin;
  int     _yyline;
  int     _yyval;
  int     _yyalt;
  char    _yytxt[256];
  char    _yytxt2[256];
  TArray* _kw;
  char    _trm[32];
  int     _errflag;
  int     _ival;
  float   _fval;
  char    _pushc;
  bool    _pushed;
  bool    _ignore_nonprintable;
  bool    _eof;

  int      yylex();

protected:

  void set_token_val    (const char* s) { strcpy(_yytxt, s);  }
  void set_token_val_alt(const char* s) { strcpy(_yytxt2, s); }
  void set_int_val      (int v)         { _ival = v;          }
  void set_float_val    (float v)       { _fval = v;          }

  void check_newline();
  int nextc(bool ignore_nonprint = FALSE);
  void pushc(int);
  int skip_whitespace();
  virtual int postprocess_token(int val, const char* tok, int& alt);

  const char* terminators() { return _trm; }

public:

  static bool is_integer(const char*);
  static bool is_float  (const char*);

  const char* token_val()     { return _yytxt;  }
  const char* token_val_alt() { return _yytxt2; }
  int   int_val() const       { return _ival;   }
  float float_val()  const    { return _fval;   }
  int   line() const          { return _yyline; }
  bool eof() const            { return _eof;    }
  int token();

  void expect(int t) {
    if (token() != t) error_box("syntax error at line %d", _yyline);
  }

  TLex(const char* filename, const char* terminators, TArray* keywords = NULL);
  virtual ~TLex();
};


// TLexS is like TLEX but recognizes double-quoted strings (with possible
// backslashed newlines ignored and backslashed quotes added)
// Double quote must be passed to constructor in terminator list

#define LEX_STRING     -1006

class TLexS : public TLex 
{
  TString _sval;

protected:
  
  virtual int postprocess_token(int val, const char* tok, int& alt);

public:
  
  const char* sval() const { return (const char*)_sval; }

  TLexS(const char* filename, const char* terminators, TArray*
	      keywords = NULL) : 
    TLex(filename, terminators, keywords), _sval(128) 
  {}
  virtual ~TLexS() {}
};


#endif



