#include "ExprBuilder.h"
#include "Environ.h"
#include "MML_Model.h"
#include "MML_Module.h"
#include "MML_Method.h"
#include "MML_Variable.h"
#include "MML_LUT.h"

EInfoIndex TNode::kNodeType = kO_Info0;

EInfoIndex TEquationNode::kEquationType = kO_Info1;
static int array_error_count = 0;

void TNode::StackDump( FILE* oFile ) { 
	 TNode* n = this;
	 while( n->fSuper != NULL ) {
		 n = n->fSuper;
		 ENodeType nt = n->NodeType();
		 if( nt == kCommand  || nt == kModule ) { break; }
	 }
	 n->Print(oFile);
} 

//----------------------------------------------------------------------------------------
//						 TFunctionNode
//----------------------------------------------------------------------------------------

TFunctionNode::TFunctionNode(int index ) : TNode(index) {  
	SetObjInfo(kNodeType,kFunc); 
	fFunction=NULL; 
	fMethod=NULL; 
	fArg=NULL; 
	fDMode = kDMSerial;
	fSource = kFS_SMELib;
	SetLibrary( ExprBuilder::DefaultLibrary() );
	fExecutionOrdering = 0;
}

void TFunctionNode::AddArg(TListNode* n) { 
	fArg = n; n->SetBracketType(kNoB); n->Super(this);
	if(fFunction) { 
		if( !fFunction->CheckNArgs(n->Length()) ) { gPrintErr( fFunctionName + ": Wrong number of arguments!"); }
	}
} 

void  TFunctionNode::SetFunctionName(const char* function) { 
	fFunctionName = function;

}

void  TFunctionNode::FindMethod(TCommand* c) { 
	switch(fType) {
		case kMethod: 
			fMethod = c->GetModule()->GetMethod(fFunctionName,False); 
			if(  fMethod == NULL ) {
				CString fname = fFunctionName; 
				fFunction = TFunction::GetFunction(fname,fLibrary); 
				if(  fFunction == NULL ) { gPrintErr( fFunctionName + ": Unrecognized Method/Function! "); }
				else { fType = kLibFunction; }
			}
			break;
		case kLibFunction:{ 
			CString fname = fFunctionName;  
			fFunction = TFunction::GetFunction(fname,fLibrary,fSource); 
			if(  fFunction == NULL ) {
				if( fSource == kFS_UserLib ) {
					fFunction = TFunction::New( fFunctionName );
					fFunction->SetAttributes(fLibrary, "PF", fFunctionName, kFS_UserLib, fArg->Length() ); 
					fType = kLibFunction;
				} else {
					fMethod = c->GetModule()->GetMethod(fFunctionName,False); 
					if(  fMethod == NULL ) gPrintErr( fFunctionName + ": Unrecognized Function/Method! ");
					else { fType = kMethod; }
				}
			}
			} break;
	}
}

void TFunctionNode::Print( FILE* oFile ) const { 
	fprintf(oFile," %s(",(const char*)fFunctionName); 
	fArg->Print(oFile); 
	fputc(')',oFile); 
}
	

void TFunctionNode::WriteMML(CString& s)  const { 
	if(fType==kLibFunction && fFunction) fFunction->WriteMML(s);
	else { 
		if( fType==kLibFunction && fSource==kFS_SMELib ) { s += fLibrary; s += "::"; }
		s += fFunctionName; 
	}
	s += '('; fArg->WriteMML(s); s += ')'; 
}

void TFunctionNode::WriteConfigList(CString& s, int skipArgs)  const { 
	s += fLibrary; s += ","; 
	switch( fExecutionOrdering ) {
		case  1: s += "END^"; break;
		case -1: s += "BEG^"; break;
	}
	s += fFunctionName;  s += ","; 
	const TOrderedObjectDLList& nodeList = fArg->List();
	int index = 0, first = 1;
	for( Pix p = nodeList.first(); p; nodeList.next(p) ) {
		if( index++ >= skipArgs) {
			if( first ) { first = 0; } else { s += ", "; }
			TNode& n = (TNode&) nodeList(p);
			n.WriteMML(s);
		}
	}
}
	
void TFunctionNode::WriteCCode( TCommand* c,  CString& s,  int isGlobal, int wContext  ) { 
	
	if(fType==kLibFunction) {
	 if( fFunction ) fFunction->WriteCCode(c,s,fArg,isGlobal,wContext);
	 else { 
			if( fSource==kFS_SMELib ) { s += fLibrary; s+= "::"; }
			s += fFunctionName; 
			s += '('; fArg->WriteCCode(c,s,isGlobal,wContext); s += ')'; 
		}
	}
	else if(fType==kMethod) {
		if( fMethod )  fMethod->WriteCCode(c,s,fArg,isGlobal,wContext);
		else { 
			s += fFunctionName; 
			s += '('; fArg->WriteCCode(c,s,isGlobal,wContext); s += ')'; 
		}
	}
}

TNode::EDependencyStatus TFunctionNode::hasDependency( TDerivative* d ) {
	EDependencyStatus dval = getDepValue( d->Index() );
	if( dval >= kNoDependency ) return dval;
	dval =  fArg->hasDependency( d );
	if(fType==kLibFunction) {
		if( ((fFunction == NULL) ||  !fFunction->Differentiable() ) && (dval > kUndifferentiable) ) {
			if( (fMethod != NULL) && fMethod->Differentiable() ) {;}
			else {
				if( gDebug ) { 
					sprintf( gMsgStr, "^^^ Undifferentiability found in derivative %s: Function %s undifferentiable.  Stack dump:\n", d->Name(), fFunctionName.chars() ); gPrintLog();
					StackDump( Env::LogFile() );
				}
				dval = kUndifferentiable; 
			}
		}
	} else if(fType==kMethod) {
		if( ((fMethod == NULL) ||  !fMethod->Differentiable() ) && (dval > kUndifferentiable) ) { 
			if( gDebug ) { 
				sprintf( gMsgStr, "^^^ Undifferentiability found in derivative %s: Method %s undifferentiable.  Stack dump:\n", d->Name(), fFunctionName.chars() ); gPrintLog();
				StackDump( Env::LogFile() );
			}
			dval = kUndifferentiable; 
		}
	}
	setDepValue( dval, d->Index() );
	return dval;
}

 void TFunctionNode::WriteDCode(TCommand* c, CString& s,  int isGlobal, int wContext ) {
	
	if(fType==kLibFunction) {
		 if( fFunction ) {
			 fFunction->WriteDCode(c,s,fArg,isGlobal,wContext);
		 } else if(fMethod) {
			fMethod->WriteDCode(c,s,fArg,isGlobal,wContext);
		 }
	}
	else if(fType==kMethod) {
		if( fMethod )  {
			fMethod->WriteDCode(c,s,fArg,isGlobal,wContext);
		 }
	}
}


//----------------------------------------------------------------------------------------
//						 TGraphNode
//----------------------------------------------------------------------------------------

void TGraphNode::Print( FILE* oFile )  const { 
    fprintf(oFile," Graph("); fArg->Print(oFile); fputc(')',oFile); fPointList->Print(oFile);  
}

const TOrderedObjectDLList& TGraphNode::PointList() { return fPointList->List(); }

void TGraphNode::WriteCCode(TCommand* c, CString& s,   int isGlobal, int wContext ) {
	
	if( fLUT == NULL ) return;
	if( isGlobal ) {
		s +="g";  
		s += c->Var()->GetModule()->SName(); 
		s += "->";  
  } 
	s += fLUT->SName(); 
	s += '('; fArg->WriteCCode(c,s,isGlobal,wContext); s += ')'; 
}

TNode::EDependencyStatus TGraphNode::hasDependency( TDerivative* d ) {
	EDependencyStatus dval = getDepValue( d->Index() );
	if( dval >= kNoDependency ) return dval;
	dval =  fArg->hasDependency( d );
	setDepValue( dval, d->Index() );
	if( dval > kUndifferentiable ) { fLUT->Derivative( 1 ) ; }
	return dval;
}

 void TGraphNode::WriteDCode(TCommand* c, CString& s,  int isGlobal, int wContext ) {	
	
	if( fLUT == NULL ) return;
	if( isGlobal ) {
		s +="g";  
		s += c->Var()->GetModule()->SName(); 
		s += "->";  
  }
  s += "D"; 
	s += fLUT->SName(); 
	s += '('; fArg->WriteCCode(c,s,isGlobal,wContext); s += ')'; 
}

void TGraphNode::WriteMML(CString& s) const {
	if( fLUT == NULL ) return;
	s += fName; 
	s += '('; fArg->WriteMML(s); s += ')'; 
}

TLUT* TGraphNode::GetLUT() { return fLUT; }  
void TGraphNode::SetLUT( TLUT* lut ) { fLUT = lut; fName = lut->SName(); }

//----------------------------------------------------------------------------------------
//						 TModuleNode
//----------------------------------------------------------------------------------------

void TModuleNode::Print( FILE* oFile )  const { 
    fprintf(oFile,"\n\n\nModule %s \n",fName.chars() ); 
		if( fEqnList != NULL ) { fEqnList->Print(oFile); } 
}

void TModuleNode::Setup( const char* name, TListNode* eqn_list ) { 
	fName = name, 
	fEqnList = eqn_list; 
	if( eqn_list != NULL ) { eqn_list->SetSeperatorType(kSemiColonRetS); }
} 

//----------------------------------------------------------------------------------------
//						 TPointNode
//----------------------------------------------------------------------------------------

void TPointNode::Print( FILE* oFile )  const { 
	 int first=0;
	 switch( GetObjInfo(kBracketType) ) {
		 case kCurleyB: fputc('{',oFile); break;
		 case kSquareB: fputc('[',oFile); break;
		 case kRoundB: fputc('(',oFile); break;
	 }
	 fprintf(oFile," %s ",(const char*)fPointData);
	 switch( GetObjInfo(kBracketType) ) {
		 case kCurleyB: fputc('}',oFile); break;
		 case kSquareB: fputc(']',oFile); break;
		 case kRoundB: fputc(')',oFile); break;
	 }
}

//----------------------------------------------------------------------------------------
//						 TListNode
//----------------------------------------------------------------------------------------

void add_seperator( CString& s, ESeperatorType t ){ 
	switch(t) {
		case kSpaceS: s += "  "; break;
		case kReturnS: s += "\n\t"; break;
		case kSemiColonS: s += "; "; break;
		case kSemiColonRetS: s += ";\n\t"; break; 
		case kCommaS: s += ", "; break;
	}
}

void TListNode::WriteMML(CString& s)  const {
	int first = 1;
	for( Pix p = fNodeList.first(); p; fNodeList.next(p) ) {
		if( first ) { first = 0; } else { add_seperator( s, kCommaS ); }
		TNode& n = (TNode&) fNodeList(p);
		n.WriteMML(s);
	}
}
void TListNode::WriteCCode(TCommand* c, CString& s,   int isGlobal, int wContext ) {
	
	int first = 1;
	for( Pix p = fNodeList.first(); p; fNodeList.next(p) ) {
		if( first ) { first = 0; } else { add_seperator( s, kCommaS ); }
		TNode& n = (TNode&) fNodeList(p);
		n.WriteCCode(c,s,isGlobal,wContext);
	}
}

TNode::EDependencyStatus TListNode::hasDependency( TDerivative* d ) {
	EDependencyStatus dval = getDepValue( d->Index() );
	if( dval >= kNoDependency ) return dval;
	dval = kNoDependency;
	for( Pix p = fNodeList.first(); p; fNodeList.next(p) ) {
		TNode& n = (TNode&) fNodeList(p);
		EDependencyStatus tmp =	n.hasDependency( d );
		if( tmp == kUndifferentiable ) { dval = kUndifferentiable; }
		if( ( tmp > kUndifferentiable ) && ( dval != kUndifferentiable )  ) { dval = kDifferentiable; }
	}
	setDepValue( dval, d->Index() );
	return dval;
}

 void TListNode::WriteDCode(TCommand* c, CString& s,  int isGlobal, int wContext ) {
	
	Pix p;
	int first = 1;
	for( p = fNodeList.first(); p; fNodeList.next(p) ) {
		if( first ) { first = 0; } else { add_seperator( s, kCommaS ); }
		TNode& n = (TNode&) fNodeList(p);
		n.WriteCCode(c,s,isGlobal,wContext);
	}
	for( p = fNodeList.first(); p; fNodeList.next(p) ) {
		add_seperator( s, kCommaS ); 
		TNode& n = (TNode&) fNodeList(p);
		n.WriteDCode(c,s,isGlobal,wContext);
	}
}

void TListNode::Print( FILE* oFile )  const { 
	 int first=0;
	 switch( GetObjInfo(kBracketType) ) {
		 case kCurleyB: fputs("{\t",oFile); break;
		 case kSquareB: fputc('[',oFile); break;
		 case kRoundB: fputc('(',oFile); break;
	 }
	 CString s(""); add_seperator(s,(ESeperatorType)GetObjInfo(kSeperatorType));
	 for( Pix p = fNodeList.first(); p; fNodeList.next(p) ) {
		 if(first) fputs(s.chars(),oFile); 
		 else first = 1;
		 ((TNode&)fNodeList(p)).Print(oFile);
	 }
	 switch( GetObjInfo(kBracketType) ) {
		 case kCurleyB: fputs("\t}",oFile); break;
		 case kSquareB: fputc(']',oFile); break;
		 case kRoundB: fputc(')',oFile); break;
	 }
}

//----------------------------------------------------------------------------------------
//						 TStringListNode
//----------------------------------------------------------------------------------------

void TStringListNode::WriteMML(CString& s)  const {
	int first = 1;
	int na = fStringList.NArgs();
	for( int index = 0; index < na; index++ ) {
		if( first ) { first = 0; } else { add_seperator( s, kCommaS ); }
		const CString& sl = fStringList[index];
		s += sl;
	}
}
void TStringListNode::WriteCCode(TCommand* c, CString& s,   int isGlobal, int wContext ) {
	
	int first = 1;
	int na = fStringList.NArgs();
	for( int index = 0; index < na; index++ ) {
		if( first ) { first = 0; } else { add_seperator( s, kCommaS ); }
		const CString& sl = fStringList[index];
		s += sl;
	}
}

int TStringListNode::AddString( const char* string ) {
#ifdef BISON_DEBUG
	fprintf(gLogFile,"\nAdding string %s\n",string);
#endif 
	fStringList.Add(string); 
	return fStringList.NArgs();
} 

void TStringListNode::Print( FILE* oFile )  const { 
	 int first=0;
	 switch( GetObjInfo(kBracketType) ) {
		 case kCurleyB: fputs("{\t",oFile); break;
		 case kSquareB: fputc('[',oFile); break;
		 case kRoundB: fputc('(',oFile); break;
	 }
	 CString s(""); add_seperator(s,(ESeperatorType)GetObjInfo(kSeperatorType));
	int na = fStringList.NArgs();
	for( int index = 0; index < na; index++ ) {
		 if(first) fputs(s.chars(),oFile); 
		 else first = 1;
		 const CString& sl = fStringList[index];
		 fputs(sl.chars(),oFile); 
	 }
	 switch( GetObjInfo(kBracketType) ) {
		 case kCurleyB: fputs("\t}",oFile); break;
		 case kSquareB: fputc(']',oFile); break;
		 case kRoundB: fputc(')',oFile); break;
	 }
}

//----------------------------------------------------------------------------------------
//						 TStringNode
//----------------------------------------------------------------------------------------

void TStringNode::WriteMML(CString& s)  const {
	s += "{{";
	s += fString;
	s += "}}";
}

void TStringNode::WriteCCode(TCommand* c, CString& s,   int isGlobal, int wContext ) {
	
	s += "{"; s += fString; s += "}";
}

void TStringNode::Print( FILE* oFile )  const { 
	fprintf(oFile,"{{ %s }}",fString.chars());
}

//----------------------------------------------------------------------------------------
//						 TCondExpNode
//----------------------------------------------------------------------------------------

void TCondExpNode::Print( FILE* oFile )  const { 
	if( fArg0 ) { fArg0->Print(oFile); }
	switch( fRelation ) {
		case kEq: fputs(" = ",oFile); break;
		case kNEq: fputs(" != ",oFile); break;
		case kGT: fputs(" > ",oFile); break;	
		case kLT: fputs(" < ",oFile); break;	
		case kGE: fputs(" >= ",oFile); break;	
		case kLE: fputs(" <= ",oFile); break;	
		case kAnd: fputs(" && ",oFile); break;	
		case kOr: fputs(" || ",oFile); break;	
	}
	if( fArg1 ) { fArg1->Print(oFile); }
}
 

void TCondExpNode::WriteMML(CString& s)  const {
	s+= "( ";
	if( fArg0 ) { fArg0->WriteMML(s); }
	if( ExprBuilder::GetMode() == ExprBuilder::kSMML ) {
	  switch(fRelation) {
		  case kEq:  s+= " == "; break; 
		  case kNEq: s+= " != "; break;  
		  case kLT:  s+= " &lt;  "; break; 
		  case kLE:  s+= " &lt;= "; break; 
		  case kGT:  s+= " &gt;  "; break; 
		  case kGE:  s+= " &gt;= "; break; 
		  case kOr:  s+= " || "; break; 
		  case kAnd: s+= " &amp;&amp; "; break;  		
		  case kNot: s+= " !"; break;  		
	  }	
	} else {
	  switch(fRelation) {
		  case kEq:  s+= " == "; break; 
		  case kNEq: s+= " != "; break;  
		  case kLT:  s+= " <  "; break; 
		  case kLE:  s+= " <= "; break; 
		  case kGT:  s+= " >  "; break; 
		  case kGE:  s+= " >= "; break; 
		  case kOr:  s+= " || "; break; 
		  case kAnd: s+= " && "; break;  		
		  case kNot: s+= " !"; break;  		
	  }	
	}	
	if( fArg1 ) { fArg1->WriteMML(s); } 
	s+= " )";
}

void TCondExpNode::WriteCCode( TCommand* c,  CString& s,  int isGlobal, int wContext  )  {
	
	s+= "( ";
	if( fArg0 ) { fArg0->WriteCCode(c,s,isGlobal,wContext); } 
	switch(fRelation) {
		case kEq:  s+= " == "; break; 
		case kNEq: s+= " != "; break;  
		case kLT:  s+= " <  "; break; 
		case kLE:  s+= " <= "; break; 
		case kGT:  s+= " >  "; break; 
		case kGE:  s+= " >= "; break; 
		case kOr:  s+= " || "; break; 
		case kAnd: s+= " && "; break;  		
		case kNot: s+= " !"; break;  		
	}		
	if( fArg1 ) { fArg1->WriteCCode(c,s,isGlobal,wContext); } 
	s+= " )";
}

TNode::EDependencyStatus TCondExpNode::hasDependency( TDerivative* d ) {
	EDependencyStatus dval = getDepValue( d->Index() );
	if( dval >= kNoDependency ) return dval;
	dval = kNoDependency;
	if( fArg0 && (fArg0->hasDependency(d) > kUndifferentiable ) ) {
		if( gDebug ) { 
			sprintf( gMsgStr, "^^^ Undifferentiability found in derivative %s -> arg 0 in conditional: ", d->Name() ); gPrintLog(); 
			fArg0->Print(Env::LogFile());
			sprintf( gMsgStr, "     :: Stack dump: \n" ); gPrintLog(); 
			StackDump( Env::LogFile() );
		}
		dval = kUndifferentiable; 
	} 
	else if( fArg1 && (fArg1->hasDependency(d) > kUndifferentiable)  ) { 
		if( gDebug ) { 
			sprintf( gMsgStr, "^^^ Undifferentiability found in derivative %s -> arg 1 in conditional: ", d->Name() ); gPrintLog(); 
			fArg1->Print(Env::LogFile());
			sprintf( gMsgStr, "     :: Stack dump: \n" ); gPrintLog(); 
			StackDump( Env::LogFile() );
		}
		dval = kUndifferentiable; 
	} 
	setDepValue( dval, d->Index() );
	return dval;
}

void TCondExpNode::WriteDCode(TCommand* c, CString& s,  int isGlobal, int wContext ) { WriteCCode(c,s, isGlobal, wContext ); }

//----------------------------------------------------------------------------------------
  //						 TCondNode
  //----------------------------------------------------------------------------------------
	
void TCondNode::WriteCCode(TCommand* c, CString& s,   int isGlobal, int wContext )  { 
	             s += " ( ( "; 
	fCondExpr->WriteCCode(c,s,isGlobal,wContext);            s += " ) ? ( "; 
	fThenExpr->WriteCCode(c,s,isGlobal,wContext);            s += " ) : ( ";
	fElseExpr->WriteCCode(c,s,isGlobal,wContext);            s += " ) ) ";
}

TNode::EDependencyStatus TCondNode::hasDependency( TDerivative* d ) {
	EDependencyStatus dval = getDepValue( d->Index() );
	if( dval >= kNoDependency ) return dval;
	dval = kNoDependency;
	EDependencyStatus cdep = fCondExpr->hasDependency(d);
	EDependencyStatus tdep = fThenExpr->hasDependency(d);
	EDependencyStatus edep = fElseExpr->hasDependency(d);
	if( ( cdep == kUndifferentiable ) || ( tdep == kUndifferentiable ) || ( edep == kUndifferentiable ) ) { 
		dval = kUndifferentiable; 
	}
	else if( ( tdep > kUndifferentiable ) || ( edep > kUndifferentiable ) ) { 
		dval = kDifferentiable; 
	}
	setDepValue( dval, d->Index() );
	return dval;
}

 void TCondNode::WriteDCode(TCommand* c, CString& s,  int isGlobal, int wContext )  { 
	             s += " ( ( "; 
	fCondExpr->WriteDCode(c,s,isGlobal,wContext);            s += " ) ? ( "; 
	fThenExpr->WriteDCode(c,s,isGlobal,wContext);            s += " ) : ( ";
	fElseExpr->WriteDCode(c,s,isGlobal,wContext);            s += " ) ) ";
}

//----------------------------------------------------------------------------------------
//						 TConstNode
//----------------------------------------------------------------------------------------

EInfoIndex TConstNode::kConstType = kO_Info1;
	
void TConstNode::WriteCCode(TCommand* c, CString& s,   int isGlobal, int wContext )  { 
	    
	switch( GetObjInfo(kConstType) ) {
		case kAVInt: 
			s += "((float)"; s += fValue0; s += ")"; 
			break;
		case kAVFloat: s += fValue0; /* s += fFVal; */ break;
		case kAVString: s += fValue0; break;
		case kAVString2: s += fValue1; s+= "::"; s += fValue0; break;
	}
}

TNode::EDependencyStatus TConstNode::hasDependency( TDerivative* d ) {
  return kNoDependency; 
}

 void TConstNode::WriteDCode(TCommand* c, CString& s,  int isGlobal, int wContext ) {
	WriteCCode(c,s, isGlobal, wContext );
}
	
//----------------------------------------------------------------------------------------
//						 TVarNode
//----------------------------------------------------------------------------------------
EInfoIndex TVarNode::kVarType = kO_Info1;

EDependencyMode TVarNode::SetDependencyMode() { 
	if( GetObjInfo(kVarType) == kRelative ) { fDMode = kDMRelCode; }
	else {
		TNode* n = fSuper;
		while( n != NULL ) {
			ENodeType nt = n->NodeType();
			if( nt == kFunc ) { 
				fDMode = ((TFunctionNode*)n)->DependencyMode();
				break;
			} else if ( ( nt == kEquation ) || ( nt == kAttribute ) ) { break; }
			n = n->SuperNode();
		}
	}
	return fDMode;
}

void TVarNode::WriteMML(CString& s)   const { 
	if( GetObjInfo(kVarType) == kConstant ) { if(fConstant) { fConstant->WriteMML(s); } return; }
	s+= fValue;
	if( GetObjInfo(kVarType) == kRelative )  { 
		s+="@("; s+=fRelCode; 
		if( !fBC.empty() ) { s+=":"; s+=fBC; }
		s+=')'; 
	} 
	ArgArray* ml = (ArgArray*)GetMapList(); 
	if( ml != NULL ) {
	  s+='['; 
	  for( int iarg=0; iarg < ml->NArgs(); iarg++ ) {
		CString& arg = ml->elem(iarg);
		if( iarg > 0 ) { s += ","; }
		s+=arg; 
	  }
	  s+=']'; 
	}
}

void TVarNode::WriteCCode(TCommand* c, CString& s,   int isGlobal, int wContext )  { 
	if( GetObjInfo(kVarType) == kConstant ) { 
		if(fConstant) { fConstant->WriteCCode(c,s); } 
		return; 
	}
	Variable* cvar = c->GetVar();
	CString prefix, suffix;
	static int index_history_array[4] = { -1,-1,-1,-1 };
	if( fVar == "DT"  ) { s += "gDT()"; return; }
	if( fVariable == NULL ) { sprintf(gMsgStr,"Unknown Variable %s referenced in equation.", fVar.chars() ); gFatal(); } 
	if( fVariable->GetF(FisImport) ) {
		if( fVariable->Connection() ) fVariable = fVariable->Connection();
	}
	if(  c->CodeGenMode() == 3 ) { WriteMML(s);  return; } 
	int opLevel = c->Optimizable() ? ( c->CodeGenMode() == 0 ) : 0;
	if( isGlobal && !opLevel  ) {
		prefix = fVariable->GetModule()->SName() + "::I().";
	}
	
	if( fVar == "TIME" ) { 
		if( c->CodeGenMode() == 0 ) {
			s += "time"; 		
		} else {
			suffix = ".Value()"; 
			s += prefix; 
			s += fValue; 
			if(wContext==0) { s += suffix; }	
		}
		return; 
	}
	  
	ArgArray* ml = GetMapList(); 
	if( fVariable->GetF(FisIgnored) ) { s+= fVariable->DefaultValue(0); return; } 
	int isPtr = 0;
	if(  (fDMode != kDMParFunction) && (fDMode != kDSpatialFunction ) && (fDMode != kArrayFunction ) ) {
		if( c->CodeGenMode() < 2 ) {
			if( opLevel ) {
				suffix = "_ptr))"; isPtr = 1;
			} else if( ml != NULL ) {
			  int is_dimension_name, index; 
			  suffix = ".ArrayValue(p,"; 
			  for( int iarg=0; iarg < ml->NArgs(); iarg++ ) {
				CString& arg = ml->elem(iarg);
				if( iarg > 0 ) { suffix += ","; }
				if( arg == "*" ) {
				  gPrintErr( format("Wildcard outside of Array Function: Variable %s in command %s", fVariable->Name(), c->Name() ));
				} else {
				  ArrayRec* ar = Model::I0().getArrayDeclaration( arg, index, is_dimension_name ); 
				  if( ar == NULL ) {
					if (array_error_count++ < 8 )
						gPrintErr( format("Unrecognized Array argument(%d) %s: Variable %s in Command %s",
									  index, arg.chars(), fVariable->Name(), c->Name() ));
				  } else {
					if( is_dimension_name  ) {
					  int aix = cvar->getLocalArrayIndex(index_history_array,index,iarg);
					  suffix += format( "array_index[%d]", aix ); 
//					  suffix +=  format("kA__%s",arg.chars());				
					} else {
					  suffix +=  format("kA__%s",arg.chars());				
					}
				  }
				 }
				}
				suffix += ")"; 
			} else {
				if( fDMode == kDMRelCode ) {
					(suffix = ".RelValue(p,") += fRelCode; 
					if( fBC.empty() ) {
						suffix += ",CVariable::kDefaultBC"; 
					} else {
						if( fBC == "P" ) {
							suffix += ",CVariable::kPeriodicBC"; 
						} else if( (fBC == "C") || (fBC == "S") ) {
							suffix += ",CVariable::kContinuousBC"; 
						} else if( (fBC == "T") || (fBC == "B") ) {
							suffix += ",CVariable::kTestBC"; 
						} else if( (fBC == "Z") || (fBC == "0")  ) {
							suffix += ",CVariable::kZeroBC"; 
						}
					}
					suffix += ")"; 
				} else { 
					suffix = ".Value(p)"; 
				} 
			}
		} else { 
			if( opLevel ) suffix = "_val"; 
			else if( ml != NULL ) {
			  int is_dimension_name, index; 
			  suffix = ".ArrayValue("; 
			  for( int iarg=0; iarg < ml->NArgs(); iarg++ ) {
				CString& arg = ml->elem(iarg);
				if( iarg > 0 ) { suffix += ","; }
				if( arg == "*" ) {
				  gPrintErr( format("Wildcard outside of Array Function: Variable %s in command %s", fVariable->Name(), c->Name() ));
				} else {
				  ArrayRec* ar = Model::I0().getArrayDeclaration( arg, index, is_dimension_name ); 
				  if( ar == NULL ) {
					if (array_error_count++ < 8 )
						gPrintErr( format("Unrecognized Array argument(%d) %s: Variable %s in Command %s",
									  index, arg.chars(), fVariable->Name(), c->Name() ));
				  } else {
					if( is_dimension_name  ) {
					  int aix = cvar->getLocalArrayIndex(index_history_array,index,iarg);
					  suffix += format( "array_index[%d]", aix ); 
//					  suffix +=  format("kA__%s",arg.chars());				
					} else {
					  suffix +=  format("kA__%s",arg.chars());				
					}
				  }
				 }
				}
				suffix += ")"; 
			} else {
			  suffix = ".Value()"; 
			}
		}
		if(isPtr) { prefix = "(*("; }
	}
	
	s += prefix; 
	s += fValue; 
	if(wContext==0) { s += suffix; }	
}  

TNode::EDependencyStatus TVarNode::hasDependency( TDerivative* d ) {
	EDependencyStatus dval = getDepValue( d->Index() );
	if( dval >= kNoDependency ) return dval;
	if( dval == kHasCheckedDep ) {   // break circular dependnecies
		dval = kDifferentiable;
	} else {
		dval = kNoDependency;
		if( fVariable != NULL ) {
			if( fVariable == d->fIndepVar ) dval = kDirectDep;
			else {
				Variable* v = fVariable; 
				if( fVariable->GetF(FisImport) ) {
					if( fVariable->Connection() != NULL ) { v = fVariable->Connection(); }
					else { sprintf(gMsgStr,"Unconnected Variable: %s",fVariable->Name()), gPrintErr(); }
				}
				TCommand* c = NULL;
				if( v == d->fIndepVar ) dval = kDirectDep;
				else if( (c =  v->GetDynamicCommand()) != NULL ) { 
					setDepValue( kHasCheckedDep, d->Index() );
					dval = c->GetNode()->hasDependency( d );
				} 
			}
		}
	}
	if( (dval > kUndifferentiable ) && (GetObjInfo(kVarType) == kRelative) ) { 
		if( gDebug ) { 
			sprintf( gMsgStr, "^^^ Undifferentiability found in derivative %s: Spatial referencing.  Stack dump:", d->Name() ); gPrintLog();
			StackDump( Env::LogFile() );
		}
		dval = kUndifferentiable; 
	}
	setDepValue( dval, d->Index() );
	return dval;
}

 void TVarNode::WriteDCode(TCommand* c, CString& s,  int isGlobal, int wContext ) {
	
	int index = c->Derivative()->Index();
	byte dval = getDepValue( index );
	if( dval == kNoDependency ) {
		s += " 0 ";
	} else if( dval == kDirectDep ) {
		s += " 1 ";
	} else if( dval == kDifferentiable ) {
		CString tmp = fValue;
		fValue = "S_"; (((fValue += fVar ) += "_" ) += c->Derivative()->SName() ); 
		WriteCCode(c,s, isGlobal, wContext );
		fValue = tmp;
	}
}

//----------------------------------------------------------------------------------------
  //						 TOpNode
  //----------------------------------------------------------------------------------------
	
void TOpNode::WriteCCode(TCommand* c, CString& s,   int isGlobal, int wContext )  { 
	
	if(fArg1 == NULL) { s+=fOperator; fArg0->WriteCCode(c,s,isGlobal,wContext); }
	else if( fOperator == '^' ) { s += "pow((double)"; fArg0->WriteCCode(c,s,isGlobal,wContext); s+=",(double)"; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" )"; }
	else { 
		s+="( "; fArg0->WriteCCode(c,s,isGlobal,wContext); 
		s+=fOperator; 
		fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" )"; 
	}
}

TNode::EDependencyStatus TOpNode::hasDependency( TDerivative* d ) {
	EDependencyStatus dval = getDepValue( d->Index() );
	if( dval >= kNoDependency ) return dval;
	dval = kNoDependency;
  if(  fArg0->hasDependency( d ) > kUndifferentiable ) { dval = kDifferentiable; }
  if( fArg1 ) {
		if(  fArg1->hasDependency( d ) > kUndifferentiable ) { dval = kDifferentiable; }
		if(  fArg1->hasDependency( d ) == kUndifferentiable ) { dval = kUndifferentiable; }
	}
  if(  fArg0->hasDependency( d ) == kUndifferentiable ) { dval = kUndifferentiable; }
	setDepValue( dval, d->Index() );
	return dval;
}

 void  TOpNode::WriteDCode(TCommand* c, CString& s,  int isGlobal, int wContext ) {
	
	int index = c->Derivative()->Index();
	byte d0 = fArg0->getDepValue( index );
	if(fArg1 == NULL) {
		if( d0 > 2 ) { s+=fOperator; fArg0->WriteDCode(c,s,isGlobal,wContext); }
		else s+=" 0 "; 
	} else {
		byte d1 = fArg1->getDepValue( index );
		if( ( d0 > 2 ) && ( d1 > 2 ) ) {
			switch( fOperator ) {
				case '+': case '-':
				s+="( "; fArg0->WriteDCode(c,s,isGlobal,wContext); s+=fOperator; fArg0->WriteDCode(c,s,isGlobal,wContext); s+=" )";
				break; 
				case '*': 
				s+="( "; fArg0->WriteDCode(c,s,isGlobal,wContext); s+=" * "; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" + ";
				fArg0->WriteCCode(c,s,isGlobal,wContext); s+=" * "; fArg1->WriteDCode(c,s,isGlobal,wContext); s+=" )";
				break; 
				case '/':
				s+="( "; fArg0->WriteDCode(c,s,isGlobal,wContext); s+=" / "; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" - ";
				fArg1->WriteDCode(c,s,isGlobal,wContext); s+=" * ";  fArg0->WriteCCode(c,s,isGlobal,wContext); s+=" /  pow( "; 
				fArg1->WriteCCode(c,s,isGlobal,wContext); s+=", 2 ) )";
				break; 
				case '^': 
				s += "( pow((double)"; fArg0->WriteCCode(c,s,isGlobal,wContext); s+=",(double)"; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" ) * (";
				s +=" log( "; fArg0->WriteCCode(c,s,isGlobal,wContext); s+=" ) * "; fArg1->WriteDCode(c,s,isGlobal,wContext); s+=" + ( "; 
				fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" / "; fArg0->WriteCCode(c,s,isGlobal,wContext); s+=" ) * "; 
				fArg0->WriteDCode(c,s,isGlobal,wContext); s+=" ) "; 
				break; 
				default: 
				s+=""; sprintf(gMsgStr,"Unknown operator in code: %c",fOperator); gPrintErr();
			}
		} else if ( d0 > 2 ) {
			switch( fOperator ) {
				case '+': case '-':
				s+="( "; fArg0->WriteDCode(c,s,isGlobal,wContext); s+=" )";
				break; 
				case '*': case '/':
				s+="( "; fArg0->WriteDCode(c,s,isGlobal,wContext); s+=fOperator; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" )";
				break; 
				case '^': 
				s+="( "; fArg0->WriteDCode(c,s,isGlobal,wContext); s+=" * "; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" * "; 
				fArg0->WriteCCode(c,s,isGlobal,wContext); s+=fOperator; s+=" ( "; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" - 1 ) )";
				break; 
				default: 
				s+=""; sprintf(gMsgStr,"Unknown operator in code: %c",fOperator); gPrintErr();
			}
		} else if ( d1 > 2 )  {
			switch( fOperator ) {
				case '+': case '-':
				s+="( "; s+=fOperator; fArg1->WriteDCode(c,s,isGlobal,wContext); s+=" )";
				break; 
				case '*': 
				s+="( "; fArg0->WriteCCode(c,s,isGlobal,wContext); s+=" * "; fArg1->WriteDCode(c,s,isGlobal,wContext); s+=" )";
				break; 
				case '/':
				s+="( - "; fArg0->WriteCCode(c,s,isGlobal,wContext); s+=" * "; fArg1->WriteDCode(c,s,isGlobal,wContext); s+=" / ";
				s += "pow((double)"; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=", 2 ) ) ";
				break; 
				case '^': 
				s+="(  log( "; fArg0->WriteCCode(c,s,isGlobal,wContext); s+=" ) * "; fArg1->WriteDCode(c,s,isGlobal,wContext); s+=" * "; 
				s += "pow((double)"; fArg0->WriteCCode(c,s,isGlobal,wContext); s+=" , "; fArg1->WriteCCode(c,s,isGlobal,wContext); s+=" ) ) ";
				break; 
				default: 
				s+=""; sprintf(gMsgStr,"Unknown operator in code: %c",fOperator); gPrintErr();
			}
		}
	}
}

//----------------------------------------------------------------------------------------
  //						 ExprBuilder
  //----------------------------------------------------------------------------------------

CString ExprBuilder::fReturnValue;
TNode* ExprBuilder::fCurrentNode = NULL;
CString ExprBuilder::fNullString = "!!!";
TOrderedObjectDLList ExprBuilder::fNodeList;
int ExprBuilder::fNodeIndex = 0;
CString ExprBuilder::fDefaultLibrary;
TListNode* ExprBuilder::fRootNode = NULL;
ExprBuilder::EMode ExprBuilder::fMode = ExprBuilder::kTest;
EProcessContext ExprBuilder::fProcessContext = kPCSetup;
TNode* ExprBuilder::fResultNode = NULL;


TNode* ExprBuilder::CreateFuncNode( const char* function, TListNode* arg_list ) {
  TFunctionNode* fn = new TFunctionNode( fNodeIndex++ );
  AddNode(fn);
  fn->SetFunctionName(function);
  fn->AddArg(arg_list);
  arg_list->fSuper = fn;
  return fCurrentNode = fn;
}
 
TNode* ExprBuilder::CreateCondNode( TNode* cond_expr, TNode* then_expr, TNode* else_expr )  {
  TCondNode* cn = new  TCondNode( fNodeIndex++ );  
  AddNode(cn);
  cn->AddExpr(cond_expr,then_expr,else_expr);
  cond_expr->fSuper = cn;
  then_expr->fSuper = cn;
  else_expr->fSuper = cn;
  return fCurrentNode = cn;
}
 
TNode* ExprBuilder::CreateConstNode(  EAttributeValueType type, const char* value0, const char* value1 )  {
  TConstNode* cn = new TConstNode( fNodeIndex++, type );
  AddNode(cn);
  cn->AddValue(value0,0);
  if(value1) cn->AddValue(value1,1);
  return fCurrentNode = cn;
}

TNode* ExprBuilder::CreateVarNode( const char* variable, Bool hasRCode )  {
  TVarNode* vn = new TVarNode( fNodeIndex++ );
  AddNode(vn);
  if( hasRCode ) {
		CString name(variable), parts[3], sep("@");
		if( split(name, parts, 3, sep ) != 2 ) { gFatal( name + ": RelVar parse error! " ); }
		parts[1].trim();
		if( (parts[1].firstchar() == '(') && (parts[1].lastchar() == ')') ) {
			int lp = parts[1].length()-1;
			parts[1].del(lp, 1);
			parts[1].del('(');
		}
		vn->Setup(parts[0]);		
		vn->SetRelCode(parts[1]);
  } else  vn->Setup(variable);
  return fCurrentNode = vn;
}

TNode* ExprBuilder::CreateOpNode( char op, TNode* arg0, TNode* arg1 )  {
  TOpNode* on = new  TOpNode( fNodeIndex++ );  
  AddNode(on);
  on->AddExpr(op,arg0,arg1);
  arg0->fSuper = on;
  if(arg1) arg1->fSuper = on;
  return fCurrentNode = on;
}  

TNode* ExprBuilder::CreateCondExpNode( TNode* arg0, ERelationType rt, TNode* arg1  )  {
  TCondExpNode* ce = new  TCondExpNode( fNodeIndex++ );  
  AddNode(ce);
  ce->AddExpr(rt,arg0,arg1);
  if(arg0) { arg0->fSuper = ce; }
  if(arg1) { arg1->fSuper = ce; }
  return fCurrentNode = ce;
}

  
TNode* ExprBuilder::CreateEquationNode(  TEquationNode::EEquationType type, TVarNode* var, TNode* expr  )  {
  TEquationNode* eq = new  TEquationNode( fNodeIndex++, type );  
  AddNode(eq);
  eq->AddExpr(var,expr);
  expr->fSuper = eq;
  return fCurrentNode = eq;
} 
 
TNode* ExprBuilder::CreateGraphNode(  TNode* argument, TListNode* point_list  )  {
  TGraphNode* g = new  TGraphNode( fNodeIndex++ );  
  AddNode(g);
  g->AddExpr(argument,point_list);
  argument->fSuper = g;
  point_list->fSuper = g;
  return fCurrentNode = g;
} 
 
TNode* ExprBuilder::CreateModuleNode(  const char* name, TListNode* eqn_list  )  {
  TModuleNode* m = new  TModuleNode( fNodeIndex++ );  
  AddNode(m);
  CString sname(name); sname.trim('$'); sname.upcase();
  sname.gsub("WITHINSECTOR", "_in_" );
  sname += "_module";
  m->Setup(sname,eqn_list);
  if( eqn_list != NULL ) { eqn_list->fSuper = m; }
  return fCurrentNode = m;
}  

TNode* ExprBuilder::CreatePointNode( const char* point ) {
  TPointNode* n = new  TPointNode( fNodeIndex++ );  
  AddNode(n);
  n->AddPointData(point);
  return fCurrentNode = n;
}  

TListNode* ExprBuilder::CreateListNode( TNode* first_node ) {
  TListNode* n = new  TListNode( fNodeIndex++ );  
  AddNode(n);
  n->AddNode(first_node);
  first_node->fSuper = n;
  fCurrentNode = n;
  return n;
}  

TNode* ExprBuilder::CreateStringListNode( const char* first_string ) {
  TStringListNode* n = new  TStringListNode( fNodeIndex++ );  
  AddNode(n);
  n->AddString(first_string);
  return fCurrentNode = n;
} 
 
TNode* ExprBuilder::CreateStringNode( const char* first_string ) {
  TStringNode* n = new TStringNode( fNodeIndex++ );  
  AddNode(n);
  n->SetString(first_string);
  return fCurrentNode = n;
}  

TNode* ExprBuilder::CreateRootNode( TNode* first_node ) { 
	fRootNode = CreateListNode( first_node );
	fRootNode->SetBracketType(kNoB); 
	fRootNode->SetSeperatorType(kReturnS);
	return fCurrentNode = fRootNode;
} 

TNode* ExprBuilder::CreateDeclarationNode( const char* command, TStringListNode* qualifier_list ) {
  TDeclarationNode* n = new  TDeclarationNode( fNodeIndex++ );  
  AddNode(n);
  n->SetDeclaration(command,qualifier_list);
  return fCurrentNode = n;
}

TNode* ExprBuilder::CreateCommandNode( TDeclarationNode* decl  ) { 
  TCommandNode* n = new  TCommandNode( fNodeIndex++ );  
  AddNode(n);
  n->SetDeclaration(decl);
  return fCurrentNode = n;
}

TNode* ExprBuilder::CreateAttrNode( TDeclarationNode* decl ) {  
  TAttributeNode* n = new  TAttributeNode( fNodeIndex++ );  
  AddNode(n);
  n->SetDeclaration(decl);
  return fCurrentNode = n;
}

void ExprBuilder::Print( FILE* oFile ) {
	GetRootNode();
	if( fRootNode ) fRootNode->Print( oFile );
	else gPrintErr( "No root node generated!" ); 
}

TNode* ExprBuilder::SetRootNode() {
	TNode* rn = NULL;
	if( fRootNode == NULL ) {
	  fRootNode = new  TListNode( fNodeIndex++ );  
		AddNode(fRootNode);
	}
	for( Pix p = fNodeList.first(); p; fNodeList.next(p) ) {
		TNode& n = (TNode&)fNodeList(p);
		if( n.SuperNode() == NULL && (&n != fRootNode) ) {
			TNode::ENodeType nt = n.NodeType();
			if( nt == TNode::kList || nt == TNode::kCommand || nt == TNode::kAttribute || nt == TNode::kModule ) {
				fRootNode->AddNode(&n);
				n.SetSuperNode(fRootNode);
				if( rn == NULL ) rn = &n;
				else gPrintErr( "Multiple root nodes in a single file");
			} else {
				if( nt == TNode::kVar ) {
					TVarNode* vn = (TVarNode*)&n;
					sprintf(gMsgStr,"Unconnected var Node found: %s\n",(const char*)vn->Name());
					gPrintErr();
				} else {
					sprintf(gMsgStr,"Unconnected Node found, type: %x\n",n.NodeType());
					gPrintErr();
				}
			}
		}
	}
	return rn;
}


void TDeclarationNode::SetName(TStringListNode* name) {
	if(fName = name) fName->Super(this); 
}

void TDeclarationNode::AddQualifier( const char* string ) { 
	if( fQualifiers == NULL ) {
		fQualifiers = (TStringListNode*) ExprBuilder::CreateStringListNode( string );
		fQualifiers->Super(this);
	} else {
		fQualifiers->AddString( string );
	}
}

void TEquationNode::AddExpr( TVarNode* var, TNode* expr ) { 
	fVar = var; var->Super(this); 
	fExpr = expr; expr->Super(this);
} 

void TEquationNode::AddDoc( const char* doc ) {
	fDoc += doc; 
	fDoc += '\n';
}

void TEquationNode::Print( FILE* oFile )  const { 
	switch( GetObjInfo(kEquationType) ) {
		case kUpdate: fprintf(oFile,"Update %s => ",(const char*)fVar->Name()); fExpr->Print(oFile); break; 
		case kStateVar: fprintf(oFile,"Integrate %s => ",(const char*)fVar->Name()); fExpr->Print(oFile); break; 
		case kInit: fprintf(oFile,"Init %s => ",(const char*)fVar->Name()); fExpr->Print(oFile); break; 
	}
}
int TEquationNode::SameMaps( TEquationNode* en ) {
  if( en == NULL ) return 0;
  ArgArray* mL0 = (ArgArray*) fVar->GetMapList();
  ArgArray* mL1 = (ArgArray*) en->Variable()->GetMapList();
  if( mL0 == mL1 ) return 1;
  if( (mL0 == NULL) || (mL1 == NULL) ) return 0;
  return (*mL0) == (*mL1);
}

TEquationNode& TEquationNode::operator= ( TEquationNode& en ) { 
  fVar = (TVarNode*) en.Variable();
  fExpr = (TNode*) en.Expr();
  fDoc = en.Doc();
  for( int i=0; i<4; i++ ) { fValue[i] = en.getValueString(i); }
  return *this;
}

