//----------------------------------------------------------------------------------------
//      ExprBuilder.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------
  
#ifndef __ExprBuilder__
#define __ExprBuilder__

#include "SList.h"
#include "StringPlex.h"
#include "Utilities.h"
#include "MML.h"
#include "ConfigObjects.h"
#include "MML_Function.h"

const int MAX_NAME_LEN = 256;
typedef class TListNode TListNode;
typedef class Module Module;
typedef class TMethod TMethod;
typedef class Variable Variable;
typedef class TCommand TCommand;
typedef class TDerivative TDerivative;

enum ERelationType { kEq, kNEq, kLT, kLE, kGT, kGE, kOr, kAnd, kNot };
enum EBracketType { kNoB, kCurleyB, kSquareB, kRoundB };
enum ESeperatorType { kSpaceS, kReturnS, kSemiColonS, kSemiColonRetS, kCommaS };
  	
//----------------------------------------------------------------------------------------
  //						TNode
  //----------------------------------------------------------------------------------------
class TNode : public TOrderedObject
{
	friend class ExprBuilder;
protected:
	TNode* fSuper;
	byte* fDependency;  
	
	inline int depBuffSize() { 
		if( fDependency==NULL ) return 0; 	
		int rval = fDependency[0];  (rval << 8); rval |= fDependency[1];
		return rval;   
	}
	
	inline void checkDepBuff(int size) {
		int cursize = depBuffSize();
		if(cursize == 0) {
			fDependency = new byte[16]; memset(fDependency,0,16);
			fDependency[0] = 0; 	fDependency[1] = 16;
		} else if( size >  cursize ) {
			byte* tmp = new byte[2*cursize];
			memcpy(tmp,fDependency,cursize); 	memset(tmp+cursize,0,cursize);
			tmp[0] = (cursize >> 8); 	tmp[1] = cursize;
			delete[] fDependency; fDependency = tmp;
		}
	}
		
public:
  enum ENodeType { kGraph, kPoint, kList, kStringList, kString, kFunc, kCond, kConst, kVar, kOp, kCondExpr, kEquation, kModule, kDeclaration, kCommand, kAttribute };
	enum EDependencyStatus { kNew, kInDepTree, kHasDep, kHasCheckedDep, kNoDependency, kUndifferentiable, kDifferentiable, kDirectDep };
// 	if( dval != kNew ) return dval;    EDependencyStatus
	
  static EInfoIndex kNodeType;
  TNode(int index ) : TOrderedObject(index) { fSuper = NULL; fDependency = NULL; }
  inline TNode* Super(TNode* n) { return fSuper = n; }
  virtual void Print( FILE* oFile )  const  = 0;
  virtual void StackDump( FILE* oFile );
  inline ENodeType NodeType() const { return (ENodeType)GetObjInfo(kNodeType); }
	TNode* SuperNode() { return fSuper; }
	void SetSuperNode(TNode* n) { fSuper = n; }
	virtual void WriteMML(CString& s) const { ; }
	virtual void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0 ) { ; }
	virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 ) { ; }
	virtual EDependencyStatus hasDependency( TDerivative* d ) { return kNew; }

	inline void setDepValue( EDependencyStatus value, int index ) {
		checkDepBuff(index+3);
		fDependency[index+2] = (byte) value;
	}
	inline EDependencyStatus getDepValue( int index ) {
		if( (index+=2) >= depBuffSize() ) return kNew;
		return (EDependencyStatus) fDependency[index];
	}
	
};

//----------------------------------------------------------------------------------------
//						 TStringListNode
//----------------------------------------------------------------------------------------

class TStringListNode : public TNode {
protected:
  ArgArray fStringList;
public:
	const EInfoIndex kBracketType;
	const EInfoIndex kSeperatorType;
	void SetBracketType(EBracketType t) { SetObjInfo(kBracketType,t); }
	void SetSeperatorType(ESeperatorType t) { SetObjInfo(kSeperatorType,t); }

  TStringListNode(int index ) : TNode(index), kBracketType(kO_Info1), kSeperatorType(kO_Info2) { 
		SetObjInfo(kNodeType,kStringList); SetBracketType(kCurleyB); SetObjInfo(kSeperatorType,kCommaS); }
  int AddString( const char* string );
  virtual void Print( FILE* oFile )  const ;
  inline const ArgArray& List() const { return fStringList; }
  inline int Empty() { return (fStringList.NArgs()==0); }
  inline int Length() { return fStringList.NArgs(); }
  virtual void WriteMML(CString& s)  const ;
  virtual void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  );
//	CString& Top ()	{ return fStringList.Top(); }
	const CString& Top ()	const { return fStringList.Top(); }
};


//----------------------------------------------------------------------------------------
//						 TStringNode
//----------------------------------------------------------------------------------------

class TStringNode : public TNode {
protected:
  CString fString;
public:

  TStringNode(int index ) : TNode(index) {  SetObjInfo(kNodeType,kString);  }
  CString& Add( const char* string ) {
#ifdef BISON_DEBUG
		fprintf(gLogFile,"\nAdding string %s\n",string);
#endif 
		return fString += string; 
	} 
	void SetString(const char* string) { fString = string; }
  virtual void Print( FILE* oFile )  const ;
  inline int Length() { return fString.length(); }
  virtual void WriteMML(CString& s)  const ;
  virtual void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  );
	CString& String ()	{ return fString; }
};

//----------------------------------------------------------------------------------------
//						 TListNode
//----------------------------------------------------------------------------------------

class TListNode : public TNode {
protected:
  TOrderedObjectDLList fNodeList;
public:
	const EInfoIndex kBracketType;
	const EInfoIndex kSeperatorType;
	void SetBracketType(EBracketType t) { SetObjInfo(kBracketType,t); }
	void SetSeperatorType(ESeperatorType t) { SetObjInfo(kSeperatorType,t); }

  TListNode(int index ) : TNode(index), kBracketType(kO_Info1), kSeperatorType(kO_Info2) { 		
		SetObjInfo(kNodeType,kList); SetBracketType(kCurleyB); SetObjInfo(kSeperatorType,kCommaS); }
  Pix AddNode( TNode* node ) { if(node) { node->Super(this); return fNodeList.append(*node); } } 
  virtual void Print( FILE* oFile )  const ;
  inline const TOrderedObjectDLList& List() const { return fNodeList; }
  inline int Length() { return fNodeList.length(); }
  virtual void WriteMML(CString& s)  const ;
  virtual void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  );
	virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 );
	virtual EDependencyStatus hasDependency( TDerivative* d );
  inline void Clear() { fNodeList.clear(); }
  inline void Free() { fNodeList.free(); }
};

//----------------------------------------------------------------------------------------
//						 TVarNode
//----------------------------------------------------------------------------------------

class TVarNode : public TNode {
protected:
  CString fVar;
	CString fValue;
	CString fRelCode;
	CString fBC;
	Variable* fVariable;
	TConstant* fConstant;
	TStringListNode* fMapList;
	EDependencyMode fDMode;
	
public:
  enum EVarType { kDefault, kRelative, kConstant };
  static EInfoIndex kVarType;
  TVarNode(int index ) : TNode(index)  { 
		SetObjInfo(kNodeType,kVar); 
		SetObjInfo(kVarType,kDefault); 
		fVariable=NULL; 
		fDMode = kDMSerial; 
		fConstant = NULL;
		fMapList = NULL;
	}
  inline void Setup( const char* var ) { fValue = fVar = var; } 
  const CString& Name() { return fVar; }
  inline void SetVariable( Variable* var ) { fVariable = var; } 
  inline Variable* GetVariable( ) { return fVariable; } 
  inline void SetConstant( TConstant* c) { fConstant = c; SetObjInfo(kVarType,kConstant); } 
  inline void SetRelCode( const char* code ) { fRelCode = code; SetObjInfo(kVarType,kRelative); }
  inline void SetBC( const char* code ) { fBC = code; }
  inline void AddMapList( TStringListNode* sln ) { 
	fMapList=sln; if(sln) { sln->Super(this); } 
  } 
  inline const ArgArray* GetMapList() const { return (fMapList) ? &fMapList->List() : (const ArgArray*)NULL; }
  inline ArgArray* GetMapList() { return (ArgArray*) ((fMapList) ? &fMapList->List() : NULL); }
  virtual void Print( FILE* oFile )  const  {  fprintf(oFile," %s ",(const char*)fValue); }
  const CString& VariableName() const { return fVar; }
  const CString& RelCode() const { return fRelCode; }
  void SetValue(const char* s) { fValue = s; }
  inline EDependencyMode DependencyMode() { return fDMode; }
  EDependencyMode SetDependencyMode();
  
  virtual void WriteMML(CString& s)   const;
  void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  );
  virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 );
  virtual EDependencyStatus hasDependency( TDerivative* d );
 
};
//----------------------------------------------------------------------------------------
//						 TEquationNode
//----------------------------------------------------------------------------------------

class TEquationNode : public TNode {
protected:
  TNode* fExpr;
  TVarNode* fVar;
  CString fDoc;
  CString fValue[4];
public:
  enum EEquationType { kUnknown, kUpdate, kStateVar, kInit, kRelCode };

  static EInfoIndex kEquationType;
  TEquationNode(int index, EEquationType type ) : TNode(index) { 
  SetObjInfo(kNodeType,kEquation); SetObjInfo(kEquationType,type); }
  void AddExpr( TVarNode* var, TNode* expr ) ;
  void AddDoc( const char* doc ) ;
  virtual void Print( FILE* oFile )  const ;
  int SameMaps( TEquationNode* en ) ;
  TEquationNode& operator= ( TEquationNode& en );
  inline const TVarNode* Variable() const { return fVar; }
  inline TVarNode* Variable() { return fVar; }
  inline CString& getValueString(int index) { 
	if( (index<0) || (index>3) ) { sprintf(gMsgStr,"Illegal index in getValueString: %d (variable %s)", index, fVar->Name().chars() ); gFatal(); }
	return fValue[index]; 
  }
  inline void SetValue( const CString& value, int index=0 ) { fValue[index] = value; }
  void WriteEqnCCode( TCommand* c, int writeMode, int isGlobal=0, int wContext=0  ) {
	if( fValue[writeMode].empty() ) {
	  fExpr->WriteCCode( c, fValue[writeMode], isGlobal, wContext  );  
	}
  }
  void WriteEqnDCode( TCommand* c, int writeMode  ) {
	fExpr->WriteDCode( c, fValue[writeMode]  );  
  }
  inline void WriteMML(CString& s) const { fExpr->WriteMML(s); }
  inline const CString& VarName() const { return fVar->Name(); }
  inline const CString& Doc() const { return fDoc; }
	inline const TNode* Expr() const { return fExpr; }
	inline TNode* Expr() { return fExpr; }
	EEquationType EquationType() const { return (EEquationType) GetObjInfo(kEquationType); }
  inline EDependencyStatus hasDependency( TDerivative* d ) {
	return fExpr->hasDependency( d ); 
  }

};

//----------------------------------------------------------------------------------------
  //						 TFunctionNode
  //----------------------------------------------------------------------------------------

class TFunctionNode : public TNode {
protected:
	enum EFunctionType { kMethod, kLibFunction };
	
  CString fFunctionName;
  CString fLibrary;
  TListNode* fArg;
  int fExecutionOrdering;
  TFunction* fFunction;
  TMethod* fMethod;
	EDependencyMode fDMode;
	EFunctionType fType;
	EFunctionSource fSource;
public:
  TFunctionNode(int index );
	void AddArg(TListNode* n);
  void  SetFunctionName(const char* function);
	void  FindMethod(TCommand* c);
	virtual void Print( FILE* oFile )  const ;
	const CString& FunctionName() const { return fFunctionName; }
	CString& GetFunctionName() { return fFunctionName; }
	void SetLibrary(const char* lib) { fLibrary = lib; fType = kLibFunction; }
  TListNode* ArgList() const { return fArg; }
  virtual void WriteMML(CString& s)  const ;
  void WriteConfigList(CString& s, int skipArgs = 0)  const;
	virtual void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  ) ;
	virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 );
	virtual EDependencyStatus hasDependency( TDerivative* d );
	inline EDependencyMode DependencyMode() { return fDMode; }
	inline void SetFunctionSource( EFunctionSource source ) { fSource = source; }
  inline void SetDependencyMode(EDependencyMode dm) { fDMode = dm; }
	TFunction* Function() { return fFunction; }
	EFunctionSource Source() { return fSource; }
	inline void setExecutionOrdering( int ord ) { fExecutionOrdering = ord; }
	inline int  executionOrdering() { return fExecutionOrdering; }
};

//----------------------------------------------------------------------------------------
  //						 TCondNode
  //----------------------------------------------------------------------------------------

class TCondNode : public TNode {
protected:
  TNode* fCondExpr;
  TNode* fThenExpr;
  TNode* fElseExpr;
public:
  TCondNode(int index ) : TNode(index) { SetObjInfo(kNodeType,kCond);}
  inline void AddExpr(TNode* c, TNode* t, TNode* e = NULL ) { 
		fCondExpr = c; fThenExpr = t; fElseExpr = e;
		if( c ) { c->Super(this); }
		if( t ) { t->Super(this); }
		if( e ) { e->Super(this); }
	} 
  virtual void Print( FILE* oFile )  const { 
    fputc('(',oFile); fCondExpr->Print(oFile); fprintf(oFile,") ? (");  
    fThenExpr->Print(oFile); fprintf(oFile,") : ("); fElseExpr->Print(oFile); fputc(')',oFile);
  }
	TNode* CondExpr() const { return  fCondExpr; }
  TNode* ThenExpr() const { return   fThenExpr; }
  TNode* ElseExpr() const { return   fElseExpr; }
  virtual void WriteMML(CString& s)  const {  s += " ( ( "; 
		fCondExpr->WriteMML(s);            s += " ) ? ( "; 
		fThenExpr->WriteMML(s);            s += " ) : ( ";
		fElseExpr->WriteMML(s);            s += " ) ) ";
	}
	void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  );
	virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 );
	virtual EDependencyStatus hasDependency( TDerivative* d );
};

//----------------------------------------------------------------------------------------
//						 TConstNode
//----------------------------------------------------------------------------------------

class TConstNode : public TNode {
protected:
  long fIVal;
  float fFVal;
  CString fValue0;
  CString fValue1;
  CString fTranslation;
public:
  static EInfoIndex kConstType;
  TConstNode(int index, EAttributeValueType type ) : TNode(index) { 
		SetObjInfo(kConstType,type); SetObjInfo(kNodeType,kConst); 
	}

  inline void AddValue( const char* value, int index = 0  ) {
    if(index==0) fValue0 = value; 
    else if(index==1) fValue1 = value; 
    else gPrintErr( "Illegal index in TConstNode::AddValue");
		errno=0; 
    switch( GetObjInfo(kConstType) ) {
    case kAVInt:
			fFVal = fIVal = atol(value); 
			Util::CheckErrors("constant", value, index);
			break;
    case kAVFloat: 
			fFVal = atof(value); 
			Util::CheckErrors("constant", value, index);
			break;
    }
  } 
  virtual void Print( FILE* oFile )  const  {  fprintf(oFile," %s ",(const char*)fValue0); }
	virtual void WriteMML(CString& s)  const  {  
    switch( GetObjInfo(kConstType) ) {
			case kAVInt: s += fValue0; /* s += fFVal; */ break;
			case kAVFloat: s += fValue0; /* s += fFVal; */ break;
			case kAVString: s += fValue0; break;
			case kAVString2: s += fValue1; s+= "::"; s += fValue0; break;
		}
	}
	void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  );
	virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 );
	virtual EDependencyStatus hasDependency( TDerivative* d );

  inline long IntVal() { return fIVal; }
  inline float FloatVal() { return fFVal; }
  inline const CString& StringVal(int index) { return (index==1) ? fValue1 : fValue0; }
  
  inline const CString& StringVal() { 
		switch(GetObjInfo(kConstType)) {
			case kAVString: fTranslation = fValue0; break;
			case kAVString2: fTranslation = fValue1; fTranslation+= "::"; fTranslation += fValue0; break;
			case kAVInt: fTranslation = fValue0; break;
			case kAVFloat: fTranslation = fFVal; break;
		}
		return fTranslation;
	}
 
};



//----------------------------------------------------------------------------------------
  //						 TOpNode
  //----------------------------------------------------------------------------------------

class TOpNode : public TNode {
protected:
  TNode* fArg0;
  char fOperator;
  TNode* fArg1;
public:
  TOpNode(int index ) : TNode(index) { SetObjInfo(kNodeType,kOp); }
  inline void AddExpr( char op, TNode* a0, TNode* a1 ) { 
		fArg0 = a0; fOperator = op; fArg1 = a1;
		if( a0 ) { a0->Super(this); }
		if( a1 ) { a1->Super(this); }
	}
	inline char Operator() const { return fOperator; }
	
  virtual void Print( FILE* oFile ) const  { 
    if( fArg1 == NULL ) {
      fprintf(oFile," %c",fOperator); fArg0->Print(oFile); 
    } else {
      fputc('(',oFile); fArg0->Print(oFile); fputc(fOperator,oFile); fArg1->Print(oFile); fputc(')',oFile); 
    }
  }
   
   inline TNode* Arg(int argindex) const  { 
		switch(argindex) {  
			case 0: return fArg0;
			case 1: return fArg1;
			default: return NULL;
		}
	}
	
	inline int ArgIndex( TNode* n ) { 
		if( fArg0 == n ) return 0;
		if( fArg1 == n ) return 1;
		return -1;
	}
	
	inline int IsNegArg( TNode* n ) {
		return ( (fOperator == '-') && (( fArg1 == NULL ) || ( fArg1 == n ))  );
	}
	
	virtual void WriteMML(CString& s)  const {  
		if(fArg1 == NULL) { s+=fOperator; fArg0->WriteMML(s); }
		else { s+="( "; fArg0->WriteMML(s); s+=fOperator; fArg1->WriteMML(s); s+=" )"; }
	}
	void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  );
	virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 );
	virtual EDependencyStatus hasDependency( TDerivative* d );

};
//----------------------------------------------------------------------------------------
  //						 TCondExpNode
  //----------------------------------------------------------------------------------------

class TCondExpNode : public TNode {
protected:
  TNode* fArg0;
  ERelationType fRelation;
  TNode* fArg1;
public:
  TCondExpNode(int index ) : TNode(index) { SetObjInfo(kNodeType,kCondExpr); }
  inline void AddExpr( ERelationType rt, TNode* a0, TNode* a1 ) { 
		fArg0 = a0; fRelation = rt; fArg1 = a1;
		if( a0 ) a0->Super(this); 
		if( a1 ) a1->Super(this); 
	}
  virtual void Print( FILE* oFile )  const ;
  inline TNode* Arg(int argindex) const { 
		switch(argindex) {  
			case 0: return fArg0;
			case 1: return fArg1;
			default: return NULL;
		}
	}
	virtual void WriteMML(CString& s)  const;
	virtual void WriteCCode( TCommand* c,  CString& s,  int isGlobal=0, int wContext=0  );
	virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 );
	virtual EDependencyStatus hasDependency( TDerivative* d );
};


//----------------------------------------------------------------------------------------
//						 TGraphNode
//----------------------------------------------------------------------------------------
typedef class TLUT TLUT;

class TGraphNode : public TNode {
protected:
  TNode* fArg;
  TLUT* fLUT;
  TListNode* fPointList;
  CString fName;
public:
  TGraphNode(int index ) : TNode(index) { 
		SetObjInfo(kNodeType,kGraph); 
		fLUT = NULL; 
	} 
  inline void AddExpr( TNode* arg, TListNode* point_list ) { 
		fArg = arg, fPointList = point_list;
		if( arg ) { arg->Super(this); } 
		if( point_list ) { point_list->Super(this); } 
	} 
  virtual void Print( FILE* oFile )  const ;
  void SetLUT( TLUT* lut );
  TLUT* GetLUT() ;  
	TNode* Arg() const { return fArg; }
	const TOrderedObjectDLList& PointList();
	virtual void WriteMML(CString& s)  const ;
	virtual void WriteCCode(TCommand* c, CString& s,   int isGlobal=0, int wContext=0  );
	const CString& Name() const { return fName; }
	virtual void WriteDCode(TCommand* c, CString& s,  int isGlobal=0, int wContext=0 );
	virtual EDependencyStatus hasDependency( TDerivative* d );
};

//----------------------------------------------------------------------------------------
//						 TPointNode
//----------------------------------------------------------------------------------------

class TPointNode : public TNode {
protected:
  CString fPointData;
public:
	const EInfoIndex kBracketType;
	void SetBracketType(EBracketType t) {  SetObjInfo(kBracketType,t);  }

  TPointNode(int index ) : TNode(index),  kBracketType(kO_Info1) { SetObjInfo(kNodeType,kPoint); SetBracketType(kCurleyB); } 
  inline void AddPointData( const char* pointData ) { fPointData = pointData; } 
  inline CString&  PointData() { return fPointData; }
  virtual void Print( FILE* oFile )  const ;

};

	
//----------------------------------------------------------------------------------------
//						 TModuleNode
//----------------------------------------------------------------------------------------

class TModuleNode : public TNode {
protected:
  CString fName;
  TListNode* fEqnList;
  Module* fModule;
public:
  TModuleNode(int index ) : TNode(index) { SetObjInfo(kNodeType,kModule); fModule = NULL; } 
	void SetModule( Module* m  ) { fModule = m; }
	void Setup( const char* name, TListNode* eqn_list ) ;
  virtual void Print( FILE* oFile )  const ;
  inline TListNode* EqnList() const { return fEqnList; }
	inline Module* GetModule() const { return fModule; }
  inline const CString& Name() const { return fName; }
};

//----------------------------------------------------------------------------------------
//						 TDeclarationNode
//----------------------------------------------------------------------------------------

class TDeclarationNode : public TNode {
private:
  TStringListNode* fName;
  CString fCommand;
  TStringListNode* fQualifiers;
  TStringListNode* fArrayArgs;
public:
  TDeclarationNode(int index ) : TNode(index) { SetObjInfo(kNodeType,kDeclaration); fName = NULL; fArrayArgs = NULL; fQualifiers = NULL; } 
	void SetDeclaration( const char* command, TStringListNode* qlist=NULL ) { 
		fCommand = command; 
		if(fQualifiers = qlist) fQualifiers->Super(this);
#ifdef BISON_DEBUG
		fprintf(gLogFile,"\nCreating Declaration %s(%x), Qualifiers: ",(char*)fCommand,this);
		if(fQualifiers) { fQualifiers->Print(gLogFile); } else { fprintf(gLogFile,"none\n"); }
#endif 
 
	}
	void SetName(TStringListNode* name);
  virtual void Print( FILE* oFile )  const { CString s; WriteMML(s); fputs(s.chars(),oFile); }
  inline const TStringListNode* QualifierList() const { return fQualifiers; }
  const TStringListNode* Name() const { return fName; }
  TStringListNode* Name() { return fName; }
  inline void AddArrayArg( TStringListNode* args ) { fArrayArgs = args;  if(args) { args->Super(this); } }
  inline TStringListNode* getArrayArgs() { return fArrayArgs; }
  const CString& Command() const { return fCommand; }
  CString& Command() { return fCommand; }
	virtual void WriteMML(CString& s)  const  { 
		if(fQualifiers) fQualifiers->WriteMML(s); 
		s+=fCommand; s+= " "; 
		if(fName) fName->WriteMML(s); 
		if( fArrayArgs ) {
		   s+= "["; fArrayArgs->WriteMML(s); s+= "] ";
		}
	}

	void AddQualifier( const char* string );

	inline int IsExtension() const {
		if( fQualifiers )  return ( fQualifiers->List().Contains("extends") );
		else return 0;
	}

};

//----------------------------------------------------------------------------------------
//						 TAttributeNode
//----------------------------------------------------------------------------------------

class TAttributeNode : public TNode {
protected:
	TDeclarationNode* fDeclaration;
  TNode* fValue;
public:
  TAttributeNode(int index ) : TNode(index) { SetObjInfo(kNodeType,kAttribute); fValue = NULL; fDeclaration = NULL; } 
	inline TAttributeNode* SetDeclaration(TDeclarationNode* decl) { 
		fDeclaration = decl; 
#ifdef BISON_DEBUG
		fprintf(gLogFile,"\nCreating TAttributeNode, DECL(%x)(%s)\n",fDeclaration,(const char*)fDeclaration->Command());
#endif 
		return (TAttributeNode*) ((decl) ? fDeclaration->Super(this) : NULL); 
	}
	TAttributeNode* SetValue(TNode* n) { fValue = n; return (TAttributeNode*) ( n ? fValue->Super(this) : NULL); } 
  virtual void Print( FILE* oFile )  const { CString s; WriteMML(s); fputs(s.chars(),oFile); }
	const TDeclarationNode* Declaration() const { return fDeclaration; }
	TDeclarationNode* Declaration() { return fDeclaration; }
	TNode* Value() const { return fValue; }
	virtual void WriteMML(CString& s)  const  { 
		if( fDeclaration == NULL ) gPrintErr( "Attempt to write MML for undeclared Attribute." );
		else fDeclaration->WriteMML(s); 
		s+= " = "; 
		if(fValue) fValue->WriteMML(s); 
		s+= ";\n"; 
	}
};

//----------------------------------------------------------------------------------------
//						 TCommandNode
//----------------------------------------------------------------------------------------

class TCommandNode : public TNode {
protected:
	TDeclarationNode* fDeclaration;
  TListNode* fStatementList;
public:
  TCommandNode(int index ) : TNode(index) { SetObjInfo(kNodeType,kCommand); fDeclaration = NULL; fStatementList = NULL; } 
	inline TCommandNode* SetDeclaration(TDeclarationNode* decl) { fDeclaration = decl; return (TCommandNode*) (decl ? fDeclaration->Super(this) : NULL); }
  virtual void Print( FILE* oFile )  const { CString s; WriteMML(s); fputs(s.chars(),oFile); }
  inline TCommandNode* SetStatements(TListNode* n) { fStatementList = n;  return (TCommandNode*) ( n ? fStatementList->Super(this) : NULL); }
	const TDeclarationNode* Declaration() const { return fDeclaration; }
	TDeclarationNode* Declaration() { return fDeclaration; }
	TListNode* StatementList() const { return fStatementList; }
	virtual void WriteMML(CString& s)  const  { 
		if( fDeclaration == NULL ) gPrintErr( "Attempt to write MML for undeclared Object." );
		else fDeclaration->WriteMML(s); s+= " {\n"; 
		if(fStatementList) fStatementList->WriteMML(s); 
		s+= "};\n";
	}
};


//----------------------------------------------------------------------------------------
//						 ExprBuilder
//----------------------------------------------------------------------------------------

class ExprBuilder {

public:

	enum EMode { kTest, kSTELLA, kMML, kSMML, kCCode };
	
protected:

  static CString fReturnValue;
  static TNode* fCurrentNode;
  static CString fNullString;
  static TOrderedObjectDLList fNodeList;
  static CString fDefaultLibrary;
  static int fNodeIndex;
  static Pix AddNode( TNode* node ) { return fNodeList.append(*node); }
  static TListNode* fRootNode;
  static TNode* fResultNode;
  static EMode fMode;
  static EProcessContext fProcessContext;

public:
	static inline void SetMode(EMode m) { fMode = m; }
	static inline EMode GetMode() { return fMode; }
	static inline void ClearNodes() { fNodeList.clear();  if(fRootNode) { fRootNode->Clear(); } fCurrentNode = NULL; fResultNode = NULL; }	
	static inline void FreeNodes() { fNodeList.free(); fRootNode = NULL; fCurrentNode = NULL; fResultNode = NULL; }	
	static inline int  DeleteNode( TNode* node ) {  Pix p = fNodeList.find(node); 
		if(p) { fNodeList.del(p,1,1); return 1;}  else return 0; }
  static TNode* CreateFuncNode(  const char* function,  TListNode* arg_list );
  static TNode* CreateCondNode(  TNode* cond_expr,  TNode* then_expr,  TNode* else_expr ); 
  static TNode* CreateConstNode(  EAttributeValueType type, const char* value0, const char* value1 = NULL  );  
  static TNode* CreateVarNode( const char* variable, Bool hasRCode = False  );
  static TNode* CreateOpNode( char op,  TNode* arg0,  TNode* arg1 = NULL ); 
  static TNode* CreateCondExpNode(  TNode* arg0, ERelationType rt,  TNode* arg1  = NULL );
  static TNode* CreateEquationNode( TEquationNode::EEquationType type, TVarNode* var, TNode* exp );
  static TNode* CreateGraphNode(  TNode* argument,  TListNode* point_list );
  static TNode* CreateModuleNode(  const char* name,  TListNode* eqn_list );
  static TNode* CreatePointNode( const char* point );
  static TListNode* CreateListNode( TNode* first_node );
  static TNode* CreateStringListNode( const char*  first_string );
  static TNode* CreateStringNode( const char*  first_string );
	static TNode* CreateRootNode( TNode* first_node );
	static TNode* CreateCommandNode( TDeclarationNode* decl  ); 
	static TNode* CreateDeclarationNode( const char* command, TStringListNode* qualifier_list ); 
	static TNode* CreateAttrNode( TDeclarationNode* decl );   
	static inline void Initialize() { ; }
	static inline TListNode* GetRootNode() { SetRootNode(); return fRootNode; }
	static TNode* SetRootNode();     // returns local root node ( for the latest file read )
  static void Print( FILE* oFile );
  static inline void StackDump(FILE* oFile) { if(fCurrentNode) fCurrentNode->StackDump(oFile); }
  static void RunConstructors();
  static inline CString& DefaultLibrary() { return fDefaultLibrary; }
	static inline void SetDefaultLibrary(char* lib) { fDefaultLibrary = lib; }
	static inline EProcessContext ProcessContext() { return fProcessContext; }
	static inline void SetProcessContext(EProcessContext pc) { fProcessContext = pc; }
	static CString& ReturnValue() { return fReturnValue; }
	static inline void SetResult( TNode* n ) { fResultNode = n; }
	static inline TNode* GetResult() { return fResultNode; }

};

#endif





