#if __GNUG__ >= 2
#  pragma implementation
#endif

#ifdef HAS_MPE

#include "GOFCalc.h"
#include "Utility.h"
#include "Environ.h"

#ifdef HAS_GALIB

#include <ga/ga.h>
#include <ga/GASimpleGA.h>
#include <ga/GARealGenome.h>
#include <ga/GARealGenome.C>

class ParameterGA : public GASimpleGA { 
protected:

  static GAGenome* sCurrentGenome;

  ParameterGA ( GAPopulation& pop  ) : GASimpleGA(pop) { ; }

public:
 
  static ParameterGA* getInstance( int length, int pop_size ) {
  	GARealAlleleSet parmAllele( 0, 1 );
	GARealGenome* genome = new  GARealGenome(length, parmAllele, ObjectiveFunction);
	genome->initializer(genomeInitializer);
	genome->mutator(parameterMutator);
	if( pop_size < 0 ) { pop_size = 30; }
	GAPopulation* pop = new GAPopulation(*genome,pop_size);
	GANoScaling noScaling;
	pop->scaling(noScaling);
	pop->order(GAPopulation::HIGH_IS_BEST);
//	pop->initializer(populationInitializer);
	return new ParameterGA(*pop);
  }
	
  virtual void evolve(unsigned int seed=0){
    initialize(seed); 
    while(!done()){
	  step();
	  showFitness();
	} 
    if(stats.flushFrequency() > 0) {
	  stats.flushScores();
	}
  }
  
  void showFitness() {
	const GAPopulation& pop = population();
	int gen = generation();
	const char* oStr = format(" [Generation %d] Fitness: max=%.3f, min=%.3f, ave=%.3f, dev=%.3f",
				 gen, pop.fitmax(), pop.fitmin(), pop.fitave(), pop.fitdev() );
	gPrintScreen(oStr);
	GOFCalc::log(oStr);	
  }
  
  static void genomeInitializer(GAGenome& g) {
	GARealGenome& genome = (GARealGenome&)g;
	for( int i=0; i< genome.length(); i++ ) {
	   float rval = GARandomFloat(0,1);
	   genome.gene(i, rval);
	}
  }

  static int parameterMutator(GAGenome & c, float pmut) {
	if(GAFlipCoin(pmut)) {
	  GARealGenome& child= (GARealGenome&)c;
	  register int index = GARandomInt(0,child.length()-1);
	  register int del = 1-2*GAFlipCoin(0.5);
	  float gval = child.gene(index);
	  gval = gval + (0.1)*del;
	  gval = ( gval < 0 ) ? 0 : gval;
	  gval = ( gval > 1 ) ? 1 : gval;
	  child.gene(index,gval);
	  return 1;
	}
	return 0;
  }
  
  static void populationInitializer(GAPopulation& p) { ; }
  
  static GARealGenome* currentGenome() { return (GARealGenome*) sCurrentGenome; } 

  static float ObjectiveFunction(GAGenome& g) {
	sCurrentGenome = &g;
	Model::I0().Open();
	GOFCalc::updateParametersGA();
	int rv = Model::I0().RunModel( FLT_MAX, 0 );
	GOFCalc::updateData();
	float fit = GOFCalc::getFitness();
	GOFCalc::log( format("\n GOF = %.3f \n", fit ) );
	Model::I0().Close();
	return fit;
  }
};

GAGenome* ParameterGA::sCurrentGenome = NULL;

#endif

//---- GOFCalc ---------------------------------------------------------

DFit GOFCalc::fDFit;
TNamedObjectList  GOFCalc::fVariableList;
TNamedObjectList GOFCalc::fParameterList;
TNamedObjectList GOFCalc::fConfigurationList;
ESearchType GOFCalc::fSearchType;
int* GOFCalc::fTopologyMap2[2] = { NULL, NULL };
int* GOFCalc::fTopologyMap3[3] = { NULL, NULL, NULL  };
float* GOFCalc::fResults = NULL;
float GOFCalc::fCurrentGOF = 0.0;
int GOFCalc::fNCells = -1;
int GOFCalc::fPopSize = -1;
int GOFCalc::fParameterIndex = -1;
int* GOFCalc::fNCellsProc = NULL;
int* GOFCalc::fProcOffset = NULL;
int GOFCalc::fDefaultOrder = 1;
int GOFCalc::fIter = 0;
int GOFCalc::fOrder = -1;
int GOFCalc::fMaxRefine = 0;
int GOFCalc::fRefineLevel = 0;
float GOFCalc::fParmIncrementWeight = 1.0;
float GOFCalc::fInitParmIncrementWeight = 1.0;
float GOFCalc::fMutProb = 0.5;
float GOFCalc::fCrossProb = 0.8;
float GOFCalc::fRefinePct = 0.0;
FILE* GOFCalc::fLogFile = NULL;
CString GOFCalc::fLogFileName = "GOFCalc.log";
CString GOFCalc::fDisplayMode = "none";

#ifdef HAS_GALIB
static ParameterGA* sGA = NULL;
#endif

inline static int seq(const char* a, const char* b) {
  if (a == 0) return 0;
  if (b == 0) return 0;
  int diff = 0;
  while ((diff = *a - *b++) == 0 && *a++ != 0);
  return (diff == 0);
}

#ifdef HAS_GALIB

int GOFCalc::updateParametersGA() {
  int gen = sGA->generation();
  log( format( "\n****** Updating Parameters, generation %d  ******************", gen ) );
  GARealGenome* genome = sGA->currentGenome();
  int index = 0;
  for(  Pix pv = fParameterList.first(); pv; fParameterList.next(pv) ) { 
	  Variable& aVar = (Variable&) fParameterList(pv);
	  float pval = genome->gene(index++);
	  float parm = aVar.UpdateParameterProportional(pval);
	  log( format("\n\t%20s : %12.3e (%.3f)", aVar.Name(), parm, pval ) );
  }
  log( "\n" );
  return 0;
}

int GOFCalc::initGA() {
  if( sGA == NULL ) {
	int length = fParameterList.length();
	sGA = ParameterGA::getInstance(length,fPopSize);
	sGA->pMutation(fMutProb);
	sGA->pCrossover(fCrossProb);
	char* argv[1]; CString tmp;
	int nparms = 1;
	for(  Pix pv = fConfigurationList.first(); pv; fConfigurationList.next(pv) ) { 
	  TConfiguration& aConf = (TConfiguration&) fConfigurationList(pv);
	  tmp = aConf.SName(); (tmp += "=") += aConf.SValue();
	  argv[0] = tmp.data();
	  sGA->parameters(nparms,(char**)argv);
	}
	fPopSize = sGA->populationSize();
	log( format("\n\tInitialized GA: population size = %d, pMut = %.3f, pCross = %.3f\n", fPopSize, sGA->pMutation(), sGA->pCrossover() ) );
	return 1;
  }
  return 0;
}

int GOFCalc::converge_genomes( float percent_genomes, int max_runs ) {
  if( percent_genomes <= 0.0 ) return 0;
  const GAPopulation& pop = sGA->population();
  pop.sort();
  int nElem = (int) (pop.size()*percent_genomes);
  for( int i=0; i<nElem; i++ ) {
	GARealGenome& genome = (GARealGenome&)pop.individual(i);
	int index = 0; 
	float fitness = genome.fitness();
	log( format("\n\n ++++++++++++++++++++++++++++++++++ \n\tRunning grad convergence, initial parameters:\n" ) );
	gPrintScreen("Running grad convergence --------------\n\n");
	log( format("\n\n(%d) Genome fitness = %f", i, fitness) );
	for(  Pix pv = fParameterList.first(); pv; fParameterList.next(pv) ) { 
	  Variable& aVar = (Variable&) fParameterList(pv);
	  float pval = genome.gene(index++);
	  float parm = aVar.SetParameterValue(pval);
	  log( format("\n\t%20s : %12.3e (%.3f)", aVar.Name(), parm, pval ) );
	} 
	initGradOptimizer(fitness);
	for( int iter=0; iter<max_runs; iter++ ) {
	  int start = getProcOffset(iter);		
	  int stop = start + getNCells();
	  for( int ic=start; ic<stop; ic++ ) {
		Model::I0().Open();
		updateParameters(ic);
		int rv = Model::I0().RunModel( FLT_MAX, 0 );
		updateData();
		calcGOF(ic,iter);
		Model::I0().Close();
	  }
	  if( !optimize(iter) ) break;
	}
  }
  return 1;
}

#endif

int GOFCalc::initGradOptimizer( float gof ) {
  fCurrentGOF = ( gof > 0.0 ) ? gof : 0.0;
  fRefineLevel = 0;
  fParmIncrementWeight = fInitParmIncrementWeight;
  return 0;
}

int GOFCalc::evolve( int nGen ) {
#ifdef HAS_GALIB
  initGA();
  sGA->nGenerations(nGen);
  Model::I0().Open();
  gPrintScreen("Running genetic algorithm.");
  sGA->evolve();
  const GAStatistics& stats = sGA->statistics();
  stats.write(format( "%s/%s.evol", Env::ArchivePath().chars(), fLogFileName.chars() ));
  printGenomes();
  converge_genomes( fRefinePct, 50 );	  
#endif
  return 0;
}

int GOFCalc::registerParameter( Variable& v ) {
  Pix p = fParameterList.Insert(v);
  return fParameterList.index(p);
}

int GOFCalc::configureVariableTest( Variable& v, TConfigData& cd ) {
  float weight[256];
  ArrayRec* ar = v.getArrayDeclaration(0); 
  int it, ntest = (ar == NULL) ? 1 : ar->getArgs()->NArgs();
  CString name; v.FullName(name); 
  CString* test = cd.Arg(0,CD::kRequired );
  for( it=0; it<ntest; it++ ) {
	if( cd.FloatArg(it+1, weight[it] ) == 0 ) {
	  gPrintErr( format("Must specify %d weights in mpet() command for Array variable %s",ntest,name.chars()) );
	}
  }
  CString* s = 0;
  char *argv[24];
  int argc = 0;
  while( (s = cd.Arg(argc+ntest+1)) != NULL ) {
	argv[argc++] = s->data();
  }
  if( ar == NULL ) {
	fDFit.register_test( name.chars(), test->chars(), weight[0], argc, argv );
  } else {
	for( it=0; it<ntest; it++ ) {
	  if( weight[it] > 0.0 ) {
		CString array_name(name); ((array_name += "[") += ar->getArgs()->elem(it)) += "]";
		fDFit.register_test( array_name.chars(), test->chars(), weight[it], argc, argv );
	  }
	}
  }
  fVariableList.Insert(v);
  return 0;
}

int GOFCalc::setVariableWeight( Variable& v,  float val ) {
  CString name; v.FullName(name); 
  fDFit.set_weight( name.chars(), val );
  return 0;
}

int GOFCalc::configureGOF( const CString& method_name, TConfigData& cd ) {
  if( method_name == "grad" ) {
	  fSearchType = kGrad; 
  } else if(  method_name == "evol"  ) {
#ifdef HAS_GALIB
	fSearchType = kEvol;
#else
	gFatal( "Must build SME with galib to enable evol search type"  ); 
#endif
  } else {
	gFatal( format( "Unknown GOFCalc Search Algorithm: %s, should be -> { grad, evol }",method_name.chars() ) ); 
  }
  CString logfile( format( "%s/%s.%d", Env::ArchivePath().chars(), fLogFileName.chars(), gIProc ) );
  fLogFile = fopen(logfile.chars(),"w");
  if( fLogFile == NULL ) {
	gPrintErr( format("Can't open GOFCalc logfile %s", logfile.chars()) );
  }
  int iarg = 1;
  CString* s = NULL;
  while( (s = cd.Arg(iarg++)) != NULL ) {
	 int eq_index = s->index('=');
	 if( eq_index > 0 ) {
	   CString prop(s->before(eq_index));
	   CString value(s->after(eq_index));
	   int dot_index = prop.index('.');
	   CString prefix(prop.before(dot_index));
	   if( prefix.equals("ga") ) {
		 CString suffix(prop.after(dot_index));
		 TConfiguration* c = new TConfiguration(suffix,value);
		 fConfigurationList.Insert(*c);
	   } else {
		 registerProperty(prop,value);
	   }
	 }
  }
  log( format("\nSetting GOFCalc search algorithm: %s", method_name.chars() ), True );
   return 0;
}

void GOFCalc::log(const char* msg, Bool flush ) { 	
  if( fLogFile != NULL ) {
	fprintf(fLogFile,"%s",msg);
	if( flush ) {
	  fflush(fLogFile);
	}
  }
}

int GOFCalc::registerProperty( CString& property, CString& value ) { 
  if( property == "refine.levels" ) {
	fMaxRefine = atoi(value.chars());
  } else if( property == "increment.weight" ) {
	fInitParmIncrementWeight = fParmIncrementWeight = atof(value.chars());
  } else if( property == "display.mode" ) {
	fDisplayMode = value;
  } else if( property == "grad.order" ) {
	fDefaultOrder = atoi(value.chars());
  } else if( property == "pop.size" ) {
	fPopSize = atoi(value.chars());
  } else if( property == "mut.prob" ) {
	fMutProb = atof(value.chars());
  } else if( property == "evol.seed" ) {
#ifdef HAS_GALIB
	GARandomSeed( atoi(value.chars()) );
#endif
  } else if( property == "cross.prob" ) {
	fCrossProb = atof(value.chars());
  } else if( property == "refine.pct" ) {
	fRefinePct = atof(value.chars());	
  } else {
    log( format("\nAttempt to Register Unknown Parameter: %s = %s ", property.chars(), value.chars() )); 
	return 0;
  }
  log( format("\nRegistering Parameter: %s = %s ", property.chars(), value.chars() ), True ); 
  return 1;
}


int GOFCalc::readReferenceData( Bool force ) { 
  static Bool hasData = False;
  if( !hasData || force ) {
	fDFit.read_refdata(); 
  }
  hasData = True;
  return 0;
}


int GOFCalc::registerReferenceData( Variable& v, const CString& filename, int* cols ) {
  CString name; v.FullName(name); 
  ArrayRec* ar = v.getArrayDeclaration(0); 
  if( ar == NULL ) {
	 fDFit.configure_ref_data( name.chars(), filename.chars(), cols ); 
  } else {
	int it, ntest = ar->getArgs()->NArgs();
	for( it=0; it<ntest; it++ ) {
	  if( cols[it] == -99 ) {
		gPrintErr( format("Must specify %d col indices in mper() command for Array variable %s",ntest,name.chars()) );	  
	  } else if( cols[it] >= 0 ) {
		int acols[2]; acols[0] = cols[it]; acols[1] = -1;
		CString array_name(name); ((array_name += "[") += ar->getArgs()->elem(it)) += "]";
		fDFit.configure_ref_data( array_name.chars(), filename.chars(), acols );
	  }
	}
  }
  return 0;
}

int GOFCalc::updateData() {
  for(  Pix pv = fVariableList.first(); pv; fVariableList.next(pv) ) { 
	  Variable& aVar = (Variable&) fVariableList(pv);
	  CString name; aVar.FullName(name); 
	  ArrayRec* ar = aVar.getArrayDeclaration(0); 
	  if( ar == NULL ) {
		int ds = aVar.DataSize();
		float* fd = aVar.Data();
		double* dd = new double[ds];
		for( int i=0; i<ds; i++ ) { dd[i] = fd[i]; }
		fDFit.set_data( name.chars(), dd, ds );
		if( gDebug ) { 
		  gPrintLog( format("Setting Data for %s: size = %d", name.chars(), ds ) );
		}
	  } else {
		int it, ntest = ar->getArgs()->NArgs();
		for( it=0; it<ntest; it++ ) {
		  CString array_name(name); ((array_name += "[") += ar->getArgs()->elem(it)) += "]";
		  VDesc* vd = fDFit.var_desc( array_name.chars(), False );
		  if( vd ) {
			int ds;
			const float* fd = aVar.TimeSeriesData( it, ds );
			double* dd = new double[ds];
			for( int i=0; i<ds; i++ ) { dd[i] = fd[i]; }
			fDFit.set_data( vd, dd, ds );
			if( gDebug ) { 
			  gPrintLog( format("Setting Data for %s: size = %d", array_name.chars(), ds ) );
			}
		  }
		}	  
	  }
  }
  return 0;
}


/*
int GOFCalc::getNCells() {
  if( fNCells < 0 ) {
	int np = fParameterList.length();
	switch(fSearchType) {
	  case kGrad:
		switch( fOrder ) {
		  case 0: fNCells = 1; break;  
		  case 1: fNCells = 2*np + 1;    break;
		  case 2: fNCells = 2*np*(np-1) + 2*np + 1; break;
		}
	  break;
	  case kEvol:
		fNCells = 0;
	  break;
	  default:
	  gFatal("No GOFCalc Search Algorithm defined");
	}
  }
  return fNCells;
}
*/

int GOFCalc::setOrder( int max_order ) {
  if( max_order != fOrder ) {
	log( format("\nSetting Order to %d ", max_order ), True ); 
	fOrder = max_order;
	fNCells = 0;
	if( fNCellsProc == NULL ) { 
	  fNCellsProc = new int[gNProc]; 
	  fProcOffset = new int[gNProc]; 
	}
	int np = fParameterList.length();
	for( int i=0; i<=fOrder; i++ ) {
	  fNCells += getNCellsForOrder(i,np);
	}
	fNCellsProc[0] = fNCells;
	fProcOffset[0] = 0;
#ifdef USE_MPI
	if( gNProc > 1 ) {
	  for( int i=0; i<gNProc; i++) {
		fNCellsProc[i] = fNCells/gNProc;
		fProcOffset[i] = fNCellsProc[i]*i;
		int remainder = fNCells - fNCellsProc[i]*gNProc;
		if( i < remainder ) { 
		  fNCellsProc[i]++;
		  fProcOffset[i] += i;
		} else {
		  fProcOffset[i] += remainder;
		}	
	  }  
	}
#endif
	return 1;
  }
  return 0;
}

int GOFCalc::getNCellsForOrder( int order, int np ) {
  switch( order ) {
	case 0: return 1; 
	case 1: return (2*np)/order;  
	case 2: return (4*np*(np-1))/2; 
	case 3: return (8*np*(np-1)*(np-2))/(2*3);
  }
}

int GOFCalc::getPSpaceCoordinate( int parameter_index, int dim ) {
  int ncells = 0;
  int np = fParameterList.length();
  int offset = 0;
  for( int i=0; i<=fOrder; i++ ) {
	int max_cells_order = getNCellsForOrder(i,np);
	int order_index = parameter_index-offset;
	if( order_index < max_cells_order ) {
	  return getPSpaceCoordinateForOrder(order_index,dim,i);
	}
	offset += max_cells_order;
  }
  return ncells;
}

int GOFCalc::getPSpaceCoordinateForOrder( int parameter_index, int dim, int order ) {
  switch( order ) {
	case 0: return 1;					/* return (( parameter_index/fTopologyOffset[dim] ) % 3 ); */
	case 1:	{
	  int permindex = parameter_index/2;
	  if( permindex == dim ) {
		return ( (parameter_index % 2) == 0 )  ? 0 : 2;	
	  }
	  return 1;	
	}	
	case 2:	{	
	  createTopologyOffsets2();
	  int permindex = parameter_index/4;
	  if( fTopologyMap2[0][permindex] == dim ) {
		return ( (parameter_index % 4) < 2 )  ? 0 : 2;	
	  }
	  if( fTopologyMap2[1][permindex] == dim ) {
		return ( (parameter_index % 2) == 0 )  ? 0 : 2;	
	  }
	  return 1;	
	}	
	case 3:	{	
	  createTopologyOffsets3();
	  int permindex = parameter_index/8;
	  if( fTopologyMap3[0][permindex] == dim ) {
		return ( (parameter_index % 8) < 4 )  ? 0 : 2;	
	  }
	  if( fTopologyMap3[1][permindex] == dim ) {
		return ( (parameter_index % 4) < 2 )  ? 0 : 2;	
	  }
	  if( fTopologyMap3[2][permindex] == dim ) {
		return ( (parameter_index % 2) < 1 )  ? 0 : 2;	
	  }
	  return 1;	
	}	
  }
}

void GOFCalc::createTopologyOffsets2() {
  if( fTopologyMap2[0] == NULL ) {
	int cnt0=0, cnt1 = 1;
	int np = fParameterList.length();
	int nperms = np*(np-1)/2;
	fTopologyMap2[0] = new int[nperms];
	fTopologyMap2[1] = new int[nperms];
	log( format("\n\n ^^^^^^^^ Creating Topology Map2[%d]: ^^^^^^^^^^ \n  " , nperms )); 
	for( int j=0; j<nperms; j++ ) {
	  fTopologyMap2[0][j] = cnt0;  
	  fTopologyMap2[1][j] = cnt1; 
	  log( format(" (%d: %d %d)", j, cnt0, cnt1 ) ); 
	  if( ++cnt1 == np ) {
		cnt0++;
		cnt1 = cnt0 + 1;
	  } 
	}	  
	log( format("\n ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ \n  " )); 
  }
}

void GOFCalc::createTopologyOffsets3() {
  if( fTopologyMap3[0] == NULL ) {
	int cnt0=0, cnt1 = 1, cnt2 = 2;
	int np = fParameterList.length();
	int nperms = np*(np-1)*(np-2)/(2*3);
	fTopologyMap3[0] = new int[nperms];
	fTopologyMap3[1] = new int[nperms];
	fTopologyMap3[2] = new int[nperms];
	log( format("\n\n ^^^^^^^^ Creating Topology Map3[%d]: ^^^^^^^^^^ \n  " , nperms )); 
	for( int j=0; j<nperms; j++ ) {
	  fTopologyMap3[0][j] = cnt0;  
	  fTopologyMap3[1][j] = cnt1; 
	  fTopologyMap3[2][j] = cnt2; 
	  log( format(" (%d: %d %d %d)", j, cnt0, cnt1, cnt2 ) ); 
	  if( ++cnt2 == np ) {
		cnt1++;
		cnt2 = cnt1 + 1;
	  } 
	  if( cnt2 == np ) {
		cnt0++;
		cnt1 = cnt0 + 1;
		cnt2 = cnt1 + 1;
	  } 
	}	  
	log( format("\n ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ \n  " )); 
  }
}

int GOFCalc::updateParameters( int parameter_index ) {
  log("\n****** Updating Parameters *****************************" );
  int pspace_index = 1; 
  fParameterIndex = parameter_index;
  char incr_icon;
  for(  Pix pv = fParameterList.first(); pv; fParameterList.next(pv) ) { 
	  Variable& aVar = (Variable&) fParameterList(pv);
	  float parm = aVar.UpdateParameter(parameter_index,pspace_index);
	  switch(pspace_index) {
		case 0: incr_icon = '-'; break;
		case 1: incr_icon = '*'; break;
		case 2: incr_icon = '+'; break;
	  }
	  log( format("\n\t(%d)%20s : %12.3e (%c)", parameter_index, aVar.Name(), parm, incr_icon ) );
  }
  log( "\n" );
  return 0;
}

int GOFCalc::printParameters( Bool showSensitivity ) {
  log("\n****** Current Parameters *****************************" );
  for(  Pix pv = fParameterList.first(); pv; fParameterList.next(pv) ) { 
	Variable& aVar = (Variable&) fParameterList(pv);
	float parm = aVar.UpdateParameter(0);
	float dp = 0.0;
	if( showSensitivity ) {
	  float parm_sensitivity = aVar.GetParameterSensitivity(dp);
	  log( format("\n\t%20s = %12.3e, sensitivity = %5.3f, dp = %12.3e", aVar.Name(), parm, parm_sensitivity, dp ) );
	} else {
	  log( format("\n\t%20s = %12.3e ", aVar.Name(), parm ) );
	}
   }
  log( "\n" );
  return 0;
}

int GOFCalc::printGenomes() {
  log("\n****** Current Genomes *****************************" );
#ifdef HAS_GALIB
  if( sGA == NULL ) return 0;
  const GAPopulation& pop = sGA->population();
  pop.sort();
  int nElem = pop.size()/2;
  for( int i=0; i<nElem; i++ ) {
	GARealGenome& genome = (GARealGenome&)pop.individual(i);
	log( format("\n\n(%d) Genome fitness = %f", i, genome.fitness()) );
	int index = 0;
	for(  Pix pv = fParameterList.first(); pv; fParameterList.next(pv) ) { 
	  Variable& aVar = (Variable&) fParameterList(pv);
	  float pval = genome.gene(index++);
	  float parm = aVar.UpdateParameterProportional(pval);
	  log( format("\n\t%20s = %12.3e", aVar.Name(), parm  ) );
	}
  }
#endif
  log( "\n" );
  return 1;
}

int GOFCalc::moveParameterCentroid( int parameter_index ) {
  for(  Pix pv = fParameterList.first(); pv; fParameterList.next(pv) ) { 
	  Variable& aVar = (Variable&) fParameterList(pv);
	  aVar.SetParameterValue(parameter_index);
  }
  return 0;
}

float GOFCalc::getFitness() {
  fDFit.compute();
  float score = fDFit.score();
  return score;
}

int GOFCalc::calcGOF( int parameter_index, int iteration ) {
  int nc = getNCellsTotal();
  allocateResults( nc );
  fDFit.compute();
  float score = fDFit.score();
  fResults[ parameter_index ] = score;
  log( format("\n@@@(%d) GOF Score = %f", parameter_index, score ) );
 
  if(  ( parameter_index == 0 ) && ( iteration == 0 ) ) { plotVariables( ".init", True ); }
  return 0;
}

int GOFCalc::plotVariables( const char* label, Bool doRef ) {
  if( GOFCalc::getDisplayMode() == "java" ) { 
   for(  Pix pv = fVariableList.first(); pv; fVariableList.next(pv) ) { 
	  Variable& aVar = (Variable&) fVariableList(pv);
	  CString full_name; aVar.FullName(full_name); 
	  VDesc* vdesc = fDFit.var_desc( full_name, False );
	  if( doRef && fDFit.has_test( vdesc, MID_THEIL ) ) {
		CString timeseries_name(aVar.SName() + ".ref");
		int dsize = 0;
		float* fdata = fDFit.get_refdata( vdesc, dsize );
		aVar.GraphData( timeseries_name.chars(), aVar.Name(), fdata, dsize, True ); 
	  }	  
	  CString timeseries_name(aVar.SName() + label );
	  aVar.GraphData( timeseries_name.chars() ); 
	} 
  }
  return 0;
}


int GOFCalc::refineSearch() {
  if ( fRefineLevel++ < fMaxRefine ) {
	fParmIncrementWeight = fParmIncrementWeight/2.0;
	log( format("\nRefining Search (level %d): %f ", fRefineLevel, fParmIncrementWeight )); 
	return 1;
  }
  return 0;
}

int GOFCalc::allocateResults( int size ) {
  static int rsize = 0;
  if( (fResults == NULL) || (size > rsize) ) {
	if( rsize > 0 ) { delete[] fResults; }
	fResults = new float[ size ];
	rsize = size;
  }
  return rsize;
}

int GOFCalc::plotGOF( int p0, int p1, int s0, int s1 ) {
  int index = 0;
  Variable *v0 = NULL, *v1 = NULL;
  Bool debug = True;
  GOFCalc::log("\n\n********************************************\n");
  for(  Pix pv = fParameterList.first(); pv; fParameterList.next(pv) ) { 
	  Variable& aVar = (Variable&) fParameterList(pv);
	  aVar.ResetParameter();
	  if( index == p0 ) { v0 = &aVar; }
	  if( index++ == p1 ) { v1 = &aVar; }
  }
  if( (v0==NULL) || (v1==NULL) ) {
	gPrintErr( format("Illegal arg to GOFCalc::plotGOF: %d %d",p0,p1) );
	return 1;
  }
  TParmRec& pr0 = v0->getParmRec();
  TParmRec& pr1 = v1->getParmRec();
  float pmin0 = pr0.Min_Value();
  float pmin1 = pr1.Min_Value();
  float pmax0 = pr0.Max_Value();
  float pmax1 = pr1.Max_Value();
  int step0 = ( s0 > 0 ) ? s0 : pr0.Step();
  int step1 = ( s1 > 0 ) ? s1 : pr1.Step();
  allocateResults( (step0+1)*(step1+1) );
  Model::I0().Open();
  char* plot_title = format(" *** Plot GOF:: cols: { %s, range: ( %.3e, %.3e ), step: %d }, rows: { %s, range: ( %.3e, %.3e ) step: %d } ", 
				v0->Name(), pmin0, pmax0, step0, v1->Name(), pmin1, pmax1, step1 );
  GOFCalc::log( plot_title );
  gPrintScreen( plot_title );
  for( int i1=0; i1<=step1; i1++ ) {
	for( int i0=0; i0<=step0; i0++ ) {
	  float parm0 = pmin0 + ((pmax0-pmin0)*i0)/step0;
	  float parm1 = pmin1 + ((pmax1-pmin1)*i1)/step1;
	  Model::I0().Open();
	  v0->SetValue(parm0);
	  v1->SetValue(parm1);
	  int rv = Model::I0().RunModel( FLT_MAX, 0 );
	  GOFCalc::updateData();
	  float fit = GOFCalc::getFitness();
	  int mIndex = i0 + i1*(step0+1);
	  fResults[ mIndex ] = fit;
	  if( debug ) {
		log("\n****** Current Parameters *****************************" );
		for(  Pix pv = fParameterList.first(); pv; fParameterList.next(pv) ) { 
			Variable& aVar = (Variable&) fParameterList(pv);
			float parm = aVar.Value();
			log( format("\n\t%20s = %12.3e ", aVar.Name(), parm ) );
		}
		GOFCalc::log( format("\n (%d,%d:%d) Fitness = %.3f \n", i0, i1, mIndex, fit ) );
	  } else {
		GOFCalc::log( format(" %.3f", fit ) );
	  }
	  Model::I0().Close();	  
	} 
	GOFCalc::log("\n");
	putchar('.');
  }
  GOFCalc::log("\n\n********************************************\n",True);
  if( GOFCalc::getDisplayMode() == "java" ) { 
	CString plot_name( v0->SName() ); (plot_name += ":") += v1->SName();
	v0->DisplayData( plot_name, fResults, 1.0, 0.0, step0+1, step1+1 );
  }
  return 0;
}

int GOFCalc::optimize(int iter) {
  int rv = 0;
#ifdef USE_MPI
  if( gNProc > 1 ) {
	MPI_Gatherv( fResults+fProcOffset[gIProc], fNCellsProc[gIProc], MPI_FLOAT, fResults, fNCellsProc, fProcOffset, MPI_FLOAT, 0, MPI_COMM_WORLD); 
  }
#endif
  int max_index = -1;
  float gof, max_gof = fCurrentGOF;
  log( format("\n\n\n ~~~~~~~~ GOFCalc Parameter Optimization (iter %d) ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n  " , iter )); 
  if( gIProc == 0 ) {
	log( format("\nMPE values: " )); 
	for( int i=0; i<fNCells; i++ ) {
	  if( (i % 8) == 0 ) { log( "\n\t" ); }
	  gof = fResults[ i ];
	  if( gof > max_gof ) {
		max_gof = gof;
		max_index = i;
	  }
	  log( format(" (%3d: %5.3f) ", i, gof ) ); 
	}
  }
#ifdef USE_MPI  
  MPI_Bcast(&max_index, 1, MPI_INT, 0, MPI_COMM_WORLD ); 
#endif
  if( max_index >= 0 ) {
#ifdef USE_MPI  
	MPI_Bcast(&max_gof, 1, MPI_FLOAT, 0, MPI_COMM_WORLD ); 
#endif
	fCurrentGOF = max_gof;
	log( format("\n\nGOFCalc new GOF (choosing parameter index %d): %f", max_index, fCurrentGOF )); 
	printf("\nMPE (p-grad %d, refine level %d): %f", max_index, fRefineLevel, fCurrentGOF ); 
	moveParameterCentroid(max_index);
	rv = 1;
  } else {
	log( format("\n\nGOFCalc at local max: %f, parmIncrementWeight: %f ", fCurrentGOF, fParmIncrementWeight ) ); 
	printf("\nMPE local maxima reached: %f\n", fCurrentGOF ); 
	fParameterIndex = 0;
	printParameters();
	rv = GOFCalc::refineSearch();
	if( rv == 0 ) { fIter = -1; }
  }
  log( "\n\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n\n\n  " ); 
  return rv;
}

#endif
