//----------------------------------------------------------------------------------------
// MML.cp
// Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//----------------------------------------------------------------------------------------

#include "MML.h"
#include "ExprBuilder.h"

MML_AttributeValue MMLObject::fAttrValue;

//========================================================================================
// 				CLASS TMMLObject
//========================================================================================

void MMLObject::AddQualifiers(const TDeclarationNode* dn, int isExtension ) {
	const TStringListNode* sl = dn->QualifierList();
	if( sl ) { 
		const ArgArray&  ql = sl->List();
		for( int i=0; i < ql.NArgs(); i++ ) AddModifier(ql[i],isExtension);
	} 
}

void MMLObject::remove_special_characters( char* str ) const {
	char ch;
	while( (ch = *str) != '\0' ) {
		if( ch == '\xb1' ) *str = '~'; 
		if( ch == '\xb5' ) *str = 'u'; 
		if( ch == '\x9f' ) *str = 'u'; 
		if( ch == '\xa5' ) *str = '.'; 
		str++;
	}
}

	
int MMLObject::ProcessNode( TNode* node  ) {
	TNode* n = (node) ? node : fNode;
	if( n==NULL ) return 0;
		switch( n->NodeType() ) {
			case TNode::kList: {
				const TListNode* ln = (const TListNode*)n;
				const TOrderedObjectDLList& list = ln->List();
				for( Pix p = list.first(); p; list.next(p) ) {
					TNode* n = (TNode*)&list(p); 
					ProcessNode(n);
				}
				} break;
			case TNode::kCommand: 
				return ProcessCommand((const TCommandNode*)n);
			case TNode::kAttribute: 
				return ProcessAttribute((const TAttributeNode*)n);
			default:
				return ProcessGeneral(n);
		}
		return 1;
}

int MMLObject::MissingValueError( const TNode* n, TMMLData::EStatus status) {
	fAttrValue.Clear();
	if( status == TMMLData::kRequired ) {
		sprintf(gMsgStr," Node type %x: Missing Value.", n->NodeType() ); 
		gPrintErr();
	}
	return 0;
}

int MMLObject::NodeTypeError( const TNode* n ) {
		fAttrValue.Clear();
		sprintf(gMsgStr," Unexpected Node type : %x", n->NodeType() ); 
		gPrintErr();
		return 0;
}

int  MMLObject::GetAttrValue( const TNode* n, EAttributeValueType av0, EAttributeValueType av1 ) { 
	if( n->NodeType() == TNode::kConst ) {
		TConstNode* cn = (TConstNode*)n;
		EAttributeValueType tcn = (EAttributeValueType) cn->GetObjInfo(TConstNode::kConstType);
		EAttributeValueType t0 = kAVUnknown;
		if( (av0 != kAVUnknown) && (av0 == tcn) ) { t0 = tcn; }
		else if ( (av1 != kAVUnknown) && (av1 == tcn) )  { t0 = tcn; }
		else if( av0 != kAVUnknown ) return NodeTypeError( cn ); 
		switch(t0) {
			case kAVInt: {
				fAttrValue.SetInt(cn->IntVal());
			} break;
			case kAVFloat: {
				fAttrValue.SetFloat(cn->FloatVal());
			} break;
			case kAVString: {
				fAttrValue.SetString(&cn->StringVal());
			} break;
			case kAVString2: {
				fAttrValue.SetString(&cn->StringVal());
			} break;
			case kAVUnknown: {
				fAttrValue.SetString(&cn->StringVal());
			} break;
		}
	} else if( n->NodeType() == TNode::kString ) {
		if( (av0 == kAVInt) || (av0 == kAVFloat) ) { return NodeTypeError( n ); } 
		TStringNode* sn = (TStringNode*)n;
		fAttrValue.SetString(&sn->String());
	} else { 
		return NodeTypeError( n ); 
	}
	return 1;
}

int  MMLObject::GetIndexedAttrValue( const TNode* n, int index, EAttributeValueType av, TMMLData::EStatus status ) { 
	if( n==NULL ) { return MissingValueError( n, status ); }
	if( n->NodeType() ==  TNode::kList ) {
			TListNode* ln = (TListNode*)n;
			const TOrderedObjectDLList& list = ln->List();
			int cnt = 0; TNode* in = NULL;
			for( Pix p = list.first(); p; list.next(p) ) {
				if( cnt++ == index ) { in = (TNode*)&list(p); break; }
			} 		 
			if( in == NULL ) { return MissingValueError( n, status); }
			return GetAttrValue( in,  av  );
	} else {
		return GetAttrValue( n,  av  );
	}
}

int  MMLObject::GetAttributeStringList( const TNode* n, ArgArray& sl ) {
	if( n->NodeType() ==  TNode::kList ) {
		int rv = 0;
		TListNode* ln = (TListNode*)n;
		const TOrderedObjectDLList& list = ln->List();
		for( Pix p = list.first(); p; list.next(p) ) {
			TNode* in = (TNode*)&list(p);
			if( GetAttrValue( in, kAVString ) ) {
				const CString* s = fAttrValue;
				if(s) { sl.Add( *s ); rv++; }
			}
		}
		return rv;
	} else if( GetAttrValue( n, kAVString ) ) {
		const CString* s = fAttrValue;
		if(s) sl.Add( *s );
		return 1;
	}	
	return 0;	 
}

int  MMLObject::GetAttributeFloatList( const TNode* n, floatVec& fv, int sublist_index) {
	if( n->NodeType() ==  TNode::kList ) {
		TListNode* ln = (TListNode*)n;
		const TOrderedObjectDLList& list = ln->List();
		int index = 0;
		for( Pix p = list.first(); p; list.next(p) ) {
			TNode* in = (TNode*)&list(p);
			if( sublist_index < 0 ) {
				if( GetAttrValue( in, kAVFloat ) ) {
					fv.add(index++) = fAttrValue;
				}
			} else if( in->NodeType() ==  TNode::kList ) {
				TListNode* ln1 = (TListNode*)in;
				const TOrderedObjectDLList& list1 = ln1->List();
				int index1 = 0;
				for( Pix p1 = list1.first(); p1; list1.next(p1) ) {
					TNode* in1 = (TNode*)&list1(p1);
					if(index1++ == sublist_index) {
						if( GetAttrValue( in1, kAVFloat ) ) {
							fv.add(index++) = fAttrValue;
						}				
					}	
				}
			} else { gPrintErr("No sublist found"); return 0; }
		}
		return index;
	} else if( GetAttrValue( n, kAVFloat ) ) {
		fv.add(0) = fAttrValue;
		return 1;
	}	
	return 0;	 
}



//========================================================================================
// 				CLASS TMMLData
//========================================================================================


const CString* TMMLData::Arg( int index, EStatus status ) const {
  if( index >= fCArg.NArgs() ) {
    if( status == kRequired ) {
      sprintf(gMsgStr," %s: Syntax Error in configuration, Not Enough Args", fCArg[0].chars() ); 
      gPrintErr();
    }
    return NULL;
  }
  else return ( &fCArg[index] ); 
}

const CString* TMMLData::Value( int index, EStatus status ) const {
  if( index >= fValue.NArgs() ) {
    if( status == kRequired ) {
      sprintf(gMsgStr," %s: Syntax Error in configuration, No Value Found\n", Name() ); 
      gProgramBreak(gMsgStr);
    }
    return NULL;
  }
  else return &fValue[index];
}
 
int TMMLData::IntArg( int index, int& arg, EStatus status ) {    
  if(index >= fValue.NArgs() ) {
    if( status == kRequired ) {
      sprintf(gMsgStr," %s: Syntax Error in configuration, Not Enough Values", fValue[0].chars() ); 
      gPrintErr();
      return -2;
    }
    return 0;
  }
  if( !Util::is_integer( fValue[index].chars() ) ) {
    sprintf(gMsgStr," %s: Syntax Error in configuration, arg %d( %s )", 
	      fValue[0].chars() , index, fValue[index].chars() ); 
    gPrintErr();
      return -1;
  }
  arg = atoi( fValue[index].chars() );
  return 1;
}

int TMMLData::FloatArg( int index, float& arg, EStatus status ) { 
  if(index >= fValue.NArgs() ) {
    if( status == kRequired ) {
      sprintf(gMsgStr," %s: Syntax Error in configuration, Not Enough Values\n", fValue[0].chars() ); 
      gPrintErr();
      return -2;
    }
    return 0;
  }
  if( !Util::is_float( fValue[index].chars() ) ) { 
    sprintf(gMsgStr," %s: Syntax Error in configuration, arg %d( %s )", 
	    fValue[0].chars(), index, fValue[index].chars() ); 
    gPrintErr();
    return -1;
  }
  arg = atof( fValue[index].chars() );
  return 1;
}

int TMMLData::Pack() {
  int sLoc=0, i;
  for(i=0; i<4; i++) { fPackBuf[sLoc++] = fInfo[i]; } 
  for(i=0; i<fCArg.NArgs(); i++) {
    if( sLoc+fCArg[i].length() + 1 >= kMMLMaxPackSize ) { fPackBuf[2] = i; return sLoc; }
    strcpy(fPackBuf+sLoc,(const char*)fCArg[i]);
    fPackBuf[sLoc+=fCArg[i].length()] = 0; sLoc++;
    if(gDebug>2) {
      fprintf(gLogFile,"(%d,%d)Py:%s\n",i,sLoc,(const char*)fCArg[i]); 
    }
  }  
  for(i=0; i<=fName.NArgs(); i++) {
    if( sLoc+fName[i].length() + 1 >= kMMLMaxPackSize ) { fPackBuf[2] = i; return sLoc; }
    strcpy(fPackBuf+sLoc,(const char*)fName[i]);
    fPackBuf[sLoc+=fCArg[i].length()] = 0; sLoc++;
    if(gDebug>2) {
      fprintf(gLogFile,"(%d,%d)Py:%s\n",i,sLoc,(const char*)fName[i]); 
    }
  }
  return sLoc;
}

void TMMLData::UnPack( ) {
  int sLoc=0, j=0, k=0;
  for(int i=0; i<4; i++) { fInfo[i] = fPackBuf[sLoc++]; }
  while(1) {
    if( sLoc == kMMLMaxPackSize ) { return; }
    CString& s = fCArg.elem(j);
    byte c0 = ( s(k++) = fPackBuf[sLoc++] );
    if(c0 == 0) {
      if(++j == fCArg.NArgs() ) break;
      else {
	k=0;
	if(gDebug>1) {
	  fprintf(gLogFile,"(%d,%d,%d)UP:%s\n",k,j,sLoc,(const char*)fCArg[j]);
	}
      }
    }
  }  
  while(1) {
    if( sLoc == kMMLMaxPackSize ) { return; }
    CString& s = fName.elem(j);
    byte c0 = ( s(k++) = fPackBuf[sLoc++] );
    if(c0 == 0) {
      if(j++ == fName.NArgs()) break;
      else {
				k=0;
				if(gDebug>1) {
					fprintf(gLogFile,"(%d,%d,%d)UP:%s\n",k,j,sLoc,(const char*)fName[j]);
				}
      }
    }
  }
}








