#include "MML_Event.h"
#include "MML_Module.h"
#include "MML_Variable.h"
#include "MML_Time.h"
#include "MML_Model.h"
#include "MML_LUT.h"
#include "ExprBuilder.h"
#include "assoc.h"

static int array_error_count = 0;

int IGNORE_exe_g( Variable* obj ) { 
	return (obj)->ExecuteIgnoreFunction(); 
}

int IMPORT_exe_g( Variable* obj ) { 
	(obj)->RefreshData(); return 0;
}

//----------------------------------------------------------------------------------------
//			TDependency:
//----------------------------------------------------------------------------------------

TDependency::TDependency(TConstant& c) : TNamedObject( c.Name() ), kDependencyMode(kO_Info0) { 
	fConstant = &c;  fVar = NULL;
	fFunction = 0; fName += "_Dep"; 
}
TDependency::TDependency(Variable& var,int offset) : TNamedObject( offset, var.Name() ), kDependencyMode(kO_Info0) {
	fVar = &var; fConstant = NULL;  
	fFunction = 0; fName += "_Dep"; 
}

ArgArray  TCommand::kTypeList( 5, "  ", "init", "update", "integrate", "function" );
ArgArray TCommand::kExTypeList(5, "  ", "code", "script", "pipe", "import" );

EInfoIndex TCommand::kStatus = kO_Info0;
EInfoIndex TCommand::kType = kO_Info1;
EInfoIndex TCommand::kDepIndex = kO_Info2;
EInfoIndex TCommand::kExType = kO_Info3;

TCommand::~TCommand() { 
//	fDependencyList.Free(); 
}

//----------------------------------------------------------------------------------------
//			TCommand::GetModule:
//----------------------------------------------------------------------------------------


const Module* TCommand::GetModuleC() { return fVar->GetModule(); }
Module* TCommand::GetModule() { return fVar->GetModule(); }

//----------------------------------------------------------------------------------------
//			TCommand::VarName:
//----------------------------------------------------------------------------------------

const CString& TCommand::VarName() const { 
  return fVar->SName(); 
}

int TCommand::getCommandInfoTextString( char* cmd_info, int len ) {
	int i, cnt = 0;
	cmd_info[cnt++] =  (char) ( 'a' + GetObjInfo(kType) );
	cmd_info[cnt++] =  (char) ( 'a' + GetObjInfo(kDepIndex) );
	cmd_info[cnt++] =  (char) ( 'a' + GetObjInfo(kExType) );
	cmd_info[cnt++] =  (char) ( 'a' + GetObjInfo(kStatus) );
	cmd_info[cnt++] =  (fFunction == 0) ? '0' : fFunction;
	for( i=0; i<8; i++ ) {
		cmd_info[cnt++] =  (char) ( 'a' + fCommandInfo[i] );
		if( cnt >= (len-1) ) { 
			if( cnt >= len ) { gFatal("Buffer overflow in getCommandInfoTextString" ); }
			cmd_info[cnt++] =  0; return cnt; 
		}
	}
	for( i=0; i<fCode.size(); i++ ) {
		cmd_info[cnt++] =  fCode[i];
		if( cnt >= (len-1) ) { 
			if( cnt >= len ) { gFatal("Buffer overflow in getCommandInfoTextString" ); }
			cmd_info[cnt++] =  0; return cnt; 
		}
	}
	cmd_info[cnt++] =  0;
	return cnt;
}

  //----------------------------------------------------------------------------------------
  //				TCommand::PrintDependencies:
  //----------------------------------------------------------------------------------------

void TCommand::PrintDependencies(FILE* oFile) const 
{
  if( GetObjInfo(kType) == kInit ) fprintf(oFile,"\t\t\tlist init Dependencies = ( ");
  else fprintf(oFile,"\t\t\tlist update Dependencies = ( ");
  int firstp = 0;
  for ( Pix p =  fDependencyList.first(); p; fDependencyList.next(p) ) {
    TDependency& d = (TDependency&) fDependencyList(p);
    const Variable* depVar = d.Var();	
    if( depVar ) {
			if( firstp == 1 ) fprintf(oFile,","); 
			else firstp = 1; 
			fprintf(oFile," %s ",depVar->Name());
		}
  }	
  fprintf(oFile," );\n");
}
//----------------------------------------------------------------------------------------
//				TCommand::WriteMML:
//----------------------------------------------------------------------------------------
 
void TCommand::WriteMML( FILE* oFile, EWriteMode mode )  {
  CString mod;
  if( GetObjInfo(kType) != kDerivative ) { GetModifierString(mod); }
  else mod = "derivative";
  if( mode == kDeclaration ) { 
    fprintf(oFile,"\t\t%s Command %s;\n",(const char*)mod,Name());
  } else if( mode == kDefinition ) {
    fprintf(oFile,"\t\t%s Command %s {\n",(const char*)mod,Name());
    fprintf(oFile,"\t\t\tVariable = %s;\n", fVar->Name() );
    if( GetObjInfo(kType) == kIntegrate ) {
	   char ichar = fVar->getIntegrationMethod(1);
	   if( ichar == 'C' ) { fprintf(oFile,"\t\t\tClamp = 1;\n"); }
	   else if( (ichar == 'N') || (fCommandInfo[kClamp] > 1) ) { fprintf(oFile,"\t\t\tClamp = 0;\n"); }
	   else { fprintf(oFile,"\t\t\tClamp = 1;\n" ); }       
       fprintf(oFile,"\t\t\tMethod = euler;\n");
    }
	EFluxType ft = fVar->getFluxTypeIndex();
	if( ft != kF_Unknown ) { 
	  fprintf(oFile,"\t\t\tFluxType = %c;\n",fVar->getFluxType());
	}
    if( fArrayNodes != NULL ) {
	  for( Pix p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
		TEquationNode& en = (TEquationNode&)fArrayNodes->elem(p);
		TVarNode* var = en.Variable();
		ArgArray* ml = var->GetMapList();
		if( ml != NULL ) {
		  fprintf(oFile,"\t\t\tValue[");
		  for( int i=0; i<ml->NArgs(); i++ ) {
			if( i>0 ) { fprintf(oFile,","); }
			fprintf(oFile,"%s",(const char*)ml->elem(i));
		  }
		  fprintf(oFile,"] = { %s };\n",(const char*)en.getValueString(0));
		} else {
		  fprintf(oFile,"\t\t\tValue = { %s };\n",(const char*)getValueString(0));
		}
	  }
	}
//    PrintDependencies(oFile);
    if( fTranslation.length() > 0 ) 
      fprintf(oFile,"\t\t\ttranslate = %s;\n",(const char*)fTranslation);
    fprintf(oFile,"\t\t }\n");
  }
}

void TCommand::WriteXML( FILE* oFile ) {
  CString mod;
  if( GetObjInfo(kType) != kDerivative ) { GetXMLAttributes(mod); }
  else mod = " type=\"derivative\" ";
	fprintf(oFile,"\n\n\t\t<dynamic %s >\n",(const char*)mod);
	if( !fDoc.empty() ) {
		char* doc = (char*)fDoc.chars();
		remove_special_characters( doc);
		fprintf(oFile,"\n\t\t\t<doc> <![CDATA[ %s ]]> \n\t\t\t</doc>\n",doc);
	}
	if( fArrayNodes != NULL ) {   
	  for( Pix p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
		TEquationNode& en = (TEquationNode&)fArrayNodes->elem(p);
		TVarNode* var = en.Variable();
		ArgArray* ml = (ArgArray*) var->GetMapList();
		if( ml != NULL ) {
		  fprintf(oFile,"\n\t\t\t<code elem=\"%s\" >  <![CDATA[ %s ]]> \n\t\t\t</code>\n",(const char*)ml->elem(0),(const char*)en.getValueString(1));
		} else {
		  fprintf(oFile,"\n\t\t\t<code>  <![CDATA[ %s  ]]> \n\t\t\t</code>\n",(const char*)en.getValueString(1));
		}
	  }
	}

	
	if( fTranslation.length() > 0 ) 
		fprintf(oFile,"\n\t\t\t<translate> %s </translate>\n",(const char*)fTranslation);
	fprintf(oFile,"\n\t\t</dynamic>\n");
}

//----------------------------------------------------------------------------------------
//				TCommand::Config:
//----------------------------------------------------------------------------------------

int TCommand::Config( TConfigData & cd ) { 
  const CString& cmd = cd.Cmd();
  if( fSchedule==NULL || Env::GetInfo(Env::kSimOpen) ) {
		if( cmd == "OS" || cmd == "OT" ) {
			if(fSchedule==NULL) fSchedule = new ExSchedule(SName());
			return fSchedule->Config(cd);
		}
	}
  return 0; 
}

//----------------------------------------------------------------------------------------
//				TCommand::SetTimeSchedule:
//----------------------------------------------------------------------------------------

int TCommand::SetTimeSchedule( float dt, float time ) {
	if(fSchedule==NULL) fSchedule = new ExSchedule(SName());
	return fSchedule->SetTimeSchedule(dt, time);
}
//----------------------------------------------------------------------------------------
//				TCommand::SetStepSchedule:
//----------------------------------------------------------------------------------------

int TCommand::SetStepSchedule(int dt, int step ) {
	if(fSchedule==NULL) fSchedule = new ExSchedule(SName());
	return fSchedule->SetStepSchedule( dt, step );
}

//----------------------------------------------------------------------------------------
//				TCommand::Optimizable:
//----------------------------------------------------------------------------------------
 
int TCommand::Optimizable() const {
		return ( !((GetObjInfo(kDepIndex)== kHasParallelDep) || (GetObjInfo(kDepIndex)== kHasMethodCalls) ) ); 
}
 
 
word TCommand::class_id() const {
  return CLASS_COMMAND;
}
           
//----------------------------------------------------------------------------------------
//				TCommand::ExecuteOK:
//----------------------------------------------------------------------------------------

int TCommand::ExecuteOK( byte event_type ) {
	TEvent::EType et = (TEvent::EType) event_type;
	if( GetObjInfo(kStatus) == kActive ) {
		switch(et) {
			case TEvent::kUnknown: 
			case TEvent::kInit: 
				return 1;
			case TEvent::kFirstUpdate: 
			case TEvent::kFinalUpdate: 
			case TEvent::kIntegrate:  
			case TEvent::kImport:
				fExecuteCnt++;
				if( fSchedule == NULL ) return 1;
				if( fSchedule->GetObjInfo(ExSchedule::kMode ) == ExSchedule::kTimeS ) {
					return fSchedule->UpdateOK(gTime());
				}
				if( fSchedule->GetObjInfo(ExSchedule::kMode ) == ExSchedule::kStepS ) {
					return fSchedule->UpdateOK(fExecuteCnt);
				}
				return 1;
		}
	} else return 0;
} 

//----------------------------------------------------------------------------------------
//				TCommand::ProcessNodes:
//----------------------------------------------------------------------------------------

int TCommand::ProcessNodes(TNode* n, EProcessContext pc) {
	switch(pc) {
		case kPCSetup: return ProcessSetup(n); 	
		case kPCDependencies: return ProcessDependencies(n); 
		case kPCDerivatives: { 
			if( GetObjInfo(kType) == kDerivative ) { ProcessDerivatives(n); }
		} break; 
	}
	return 0;
}	

void TCommand::AddArrayNode( TEquationNode* n )	{ 
  int debug = 1; TNode* sn = NULL;
  if( fArrayNodes == NULL ) { fArrayNodes = new TOrderedObjectDLList(); }
  else {
	for( Pix p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
	  TEquationNode& en = (TEquationNode&)fArrayNodes->elem(p);
	  if( en.SameMaps(n) ) {
		en = *n;	  
		return;
	  }
	  if( en.SuperNode() != NULL ) { sn = en.SuperNode(); }
	}
  }
  if( n->SuperNode() == NULL ) { n->SetSuperNode(sn); }
  if( debug ) {
	CString stmp;
	n->Variable()->WriteMML(stmp); 
	sprintf( gMsgStr, "\n***Adding Array Node %s (%p) to command %s: %s", stmp.chars(), n, Name(), n->getValueString(0).chars()); gPrintLog();
	gPrintLog("\nCurrent Nodes:");
	for( Pix p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
	  TEquationNode& en = (TEquationNode&)fArrayNodes->elem(p);
	  stmp.clear();
	  en.Variable()->WriteMML(stmp); 
	  sprintf( gMsgStr, "\t\t%s(%p): %s ", stmp.chars(), &en, en.getValueString(0).chars() ); gPrintLog();
	}
  }
  
  int array_order=0, is_default_eqn = 1; 
  TVarNode* v = n->Variable();
  ArgArray* ml = v->GetMapList(); 
  if( ml != NULL ) {
	int is_dimension_name, index; 
	array_order = ml->NArgs();
	for( int iarg=0; iarg<array_order; iarg++ ) {
	  CString& arg = ml->elem(iarg);
	  ArrayRec* ar = Model::I0().getArrayDeclaration( arg, index, is_dimension_name ); 
	  if( !is_dimension_name ) { is_default_eqn = 0; }
//	  int rv = fVar->addArrayDeclaration( ar, is_dimension_name, index );
//	  if( rv == 0 ) { gPrintErr( format( "Attempt to add NULL ArrayRec (%s) to Variable %s", arg.chars(), fVar->Name() ) ); }
	}
  } 
  
  if( fArrayOrder < 0 ) {
	fArrayOrder = array_order;
	if( debug ) { gPrintLog( format( "\nSetting Array Order for %s to %d",Name(), array_order) ); }
  } else if( fArrayOrder != array_order ) {
	sprintf(gMsgStr," Uneven array Orders in Command %s:%s: %d vs %d",v->Name().chars(),Name(),fArrayOrder,array_order); gPrintErr();
  }

  if( is_default_eqn ) {
	fArrayNodes->append(*n);
  } else {
	fArrayNodes->prepend(*n);
  }
}

void TCommand::SetValue( CString& value )	{ 
  if( fArrayNodes == NULL ) {
	TVarNode* var = (TVarNode*) ExprBuilder::CreateVarNode(  fVar->Name()  );
	TConstNode* cn = (TConstNode*) ExprBuilder::CreateConstNode(  kAVFloat, value );  
	TEquationNode* en = (TEquationNode*) ExprBuilder::CreateEquationNode( TEquationNode::kUpdate, var, cn  );
	en->SetValue( value ); 
	var->Super(en); 
	AddArrayNode(en);
  } else {
	for( Pix p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
	  TEquationNode& en = (TEquationNode&)fArrayNodes->elem(p);
	  for( int i=0; i<4; i++ ) {
		en.SetValue( value, i );
	  }
	}
  }
}

//----------------------------------------------------------------------------------------
//				TCommand::RunNodeProcessing:
//----------------------------------------------------------------------------------------

int TCommand::RunNodeProcessing( EProcessContext pc ) {
	if( fArrayNodes == NULL ) {
	  return 0;
	}
	Pix p;
	switch(pc) {
		case kPCDependencies: {
			TNamedObjectList& plist = fVar->GetDBParamValues();
			for( p = plist.first(); p; plist.next(p) ) {
			  TDBParamValue& dbp = (TDBParamValue&)plist(p);
			  TVarNode* var = (TVarNode*) ExprBuilder::CreateVarNode(  fVar->Name()  );
			  TConstNode* cn = (TConstNode*) ExprBuilder::CreateConstNode(  kAVFloat, dbp.Value() );  
			  TStringListNode* sln = (TStringListNode*) ExprBuilder::CreateStringListNode( dbp.Name() );
			  var->AddMapList( sln );
			  TEquationNode* en = (TEquationNode*) ExprBuilder::CreateEquationNode( TEquationNode::kUpdate, var, cn  );
			  en->SetValue( dbp.Value() ); 
			  var->Super(en); 
			  AddArrayNode(en);
			}
			for( p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
			  TEquationNode* en = (TEquationNode*)&(fArrayNodes->elem(p));
			  ProcessNodeRecur(en,pc); 
			}
		} break;
		case kPCDerivatives:  {
			for( p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
			  TEquationNode* en = (TEquationNode*)&(fArrayNodes->elem(p));
			  ProcessNodeRecur(en,pc); 
			}
		} break;
		case kPCMMLGen: 
			for( p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
			  TEquationNode& en = (TEquationNode&)fArrayNodes->elem(p);
			  ExprBuilder::SetMode(ExprBuilder::kMML);
			  CString& s0 = en.getValueString(0);
			  if( s0.empty() ) {
				en.WriteMML( s0 ); 
			  }
			  ExprBuilder::SetMode(ExprBuilder::kSMML);
			  CString& s1 = en.getValueString(1);
			  if( s1.empty() ) {
				en.WriteMML( s1 ); 
			  }
			}

		break;
		case kPCCodeGen:
			for( int i=0; i<4; i++ ) {
				fCodeGenMode = i;
				for( p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
				  TEquationNode& en = (TEquationNode&)fArrayNodes->elem(p);
				  if( GetObjInfo(kType) == kDerivative ) { 
					  en.WriteEqnDCode(this,i); 
				  } 
				  else { 
					  en.WriteEqnCCode(this,i); 
				  } 
				}
			} 
		break;			
	}	
	return 0;
}

//----------------------------------------------------------------------------------------
//				TCommand::getValueString:
//----------------------------------------------------------------------------------------

CString& TCommand::getValueString( int code_index, int array_index ) {
  if( fArrayNodes != NULL ) {
	if( code_index < 0 ) { code_index = fCodeGenMode; }
	int ac = 0; TEquationNode* en = NULL;
	for( Pix p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
	  en = (TEquationNode*) &fArrayNodes->elem(p);
	  if ( ac++ == array_index ) {
		return en->getValueString( code_index ); 
	  }
	}
	if( (array_index < 0) && (en != NULL) ) {
	  return en->getValueString( code_index ); 
	}
	sprintf(gMsgStr,"Array bounds error: getValueString in TCommand %s, index: %d ", fName.chars(), array_index  ); gPrintErr();
  }
  return (CString&) gNullString;
}

//----------------------------------------------------------------------------------------
//				TCommand::ProcessCodeGen:
//----------------------------------------------------------------------------------------

int TCommand::ProcessGraph(TGraphNode* n) { 
	static int index = 0;
	Module* m = GetVar()->GetModule();
	CString g_name("Graph");
	g_name.appendIndex(index++);
	TLUT*  lut = m->GetLUT(g_name);
	lut->ProcessGeneral((TGraphNode*)n);
	n->SetLUT(lut);
	return 1; 
}

//----------------------------------------------------------------------------------------
//				TCommand::ProcessSetup:
//----------------------------------------------------------------------------------------

int TCommand::ProcessSetup(TNode* n) {
	switch( n->NodeType() ) {
		case TNode::kFunc: {
			TFunctionNode* fn = (TFunctionNode*)n;
			} break;
		case TNode::kVar: { 
			TVarNode* vn = (TVarNode*)n;
			if( GetObjInfo(kType) != kDerivative ) {
				if( vn->GetObjInfo(TVarNode::kVarType) == TVarNode::kRelative ) {
					CString parts[3], sep(":");
					if( split(vn->RelCode(), parts, 3, sep ) > 1 ) { 
						parts[1].trim();
						vn->SetBC(parts[1]);
					}				
					const CString& rc = parts[0].trim();		
					if( rc == "N" )      vn->SetRelCode("-1,0,0");
					else if( rc == "S" ) vn->SetRelCode("1,0,0");
					else if( rc == "E" ) vn->SetRelCode("0,1,0");
					else if( rc == "W" ) vn->SetRelCode("0,-1,0");
					else if( rc == "NE" ) vn->SetRelCode("-1,1,0");
					else if( rc == "SE" ) vn->SetRelCode("1,1,0");
					else if( rc == "NW" ) vn->SetRelCode("-1,-1,0");
					else if( rc == "SW" ) vn->SetRelCode("1,-1,0");
					else if( rc == "U" ) vn->SetRelCode("0,0,-1");
					else if( rc == "D" ) vn->SetRelCode("0,0,1");
					else vn->SetRelCode(rc);
	//				else { rc.del('('); rc.del(')'); }
				}
	//			const CString& vname = vn->VariableName();
			} 
		} break;
		case TNode::kGraph: 
			if( GetObjInfo(kType) != kDerivative ) {
				return ProcessGraph((TGraphNode*)n); 
			} 
		case TNode::kConst: 
			break;	
		case TNode::kOp:
			break;			
	} 
	return 0;
}
//----------------------------------------------------------------------------------------
//				TCommand::Dependencies:
//----------------------------------------------------------------------------------------

int TCommand::ProcessDerivatives ( TNode* n) {
	if( n==NULL ) return 0;
	switch( n->NodeType() ) {
		case TNode::kFunc: {
			TFunctionNode* fn = (TFunctionNode*)n;
			} break;
		case TNode::kVar: {
			TVarNode* vn = (TVarNode*)n;
			if(fDerivative) {
				int index =  fDerivative->Index();
				if( n->getDepValue(index) == TNode::kDifferentiable ) {
					Variable* v = vn->GetVariable();
					Variable* newVar = v->CreateDerivativeVariable( fDerivative ); 
					if(newVar) {
						TDependency* d = new TDependency(*newVar,0);
						fDependencyList.append(*d);
					}
				}
			}
		}	 
	}		 
}
  
int TCommand::ProcessDependencies(TNode* n) {
	if( n==NULL || fVar->GetF(FisIgnored) ) return 0;
	Env::EAppType app =  (Env::EAppType) Env::GetInfo(Env::kAppType);
	switch( n->NodeType() ) {
		case TNode::kFunc: {
			TFunctionNode* fn = (TFunctionNode*)n;
			ProcessFunction(fn);
			} break;			
		case TNode::kEquation: { 
			TEquationNode* en = (TEquationNode*)n;
			TVarNode* vn = en->Variable();
			ArgArray* mlist = vn->GetMapList();
			int index, is_dimension_name;
			if( mlist != NULL  ) {
				if( fHasArraysInEqn < 3 ) { fHasArraysInEqn = 3; }
				int len = mlist->NArgs();
				for( int i=0; i<len; i++ ) {
					CString& map = mlist->elem(i);
					ArrayRec* ar = Model::I0().getArrayDeclaration( map, index, is_dimension_name ); 
					if( ar == NULL  ) {
					  if (array_error_count++ < 16)
						gPrintErr( format("Unrecognized Array argument %d: %s in Variable %s, Use Ar() config-cmd to declare arrays.",
										index, map.chars(), vn->VariableName().chars() ));
					} else if( !is_dimension_name ) {
					  Variable* mv = ar->getMap();
					  if( fVar->GetModule() != mv->GetModule() ) { 
						Variable* v = fVar->GetModule()->GetVariable( mv->SName(), FALSE ); 
						mv = ( v==NULL ) ?  fVar->GetModule()->CreateInputVariable( mv,"C1_") : v;	
					  }
					  AddDependency( *mv );
					}
					fVar->addArrayDeclaration( ar, i, is_dimension_name, index );
				}
			}
		} break;			
		case TNode::kVar: { 
			TVarNode* vn = (TVarNode*)n;
			const CString& vname = vn->VariableName();
			Variable* v = fVar->GetModule()->GetVariable(vname,False);
			if( v && v->GetF(FisIgnored) ) {
			  Variable* rv = NULL;
			  if( v->GetCInfo(Variable::kType) !=  Variable::kTime ) { rv = Model::I0().GetVariable(vname); }
			  if( (rv == NULL) || ( rv->GetF(FisIgnored) ) ) {
				v->SetF(FisIgnored,False,"ProcessDependencies");  // If not defined elsewhere, un-ignore local version
			  } else {
				v = NULL;        // If defined elsewhere, use that one.
			  }
			}
			Variable* iv = NULL;
			Pix p = NULL;
			if(v) { p = AddDependency( *v ); vn->SetVariable(v); }
			else {				
				if( v = Model::I0().GetVariable(vname) ) { 
					iv = fVar->GetModule()->CreateInputVariable(v,"C3_");					 
					p = AddDependency( *iv );
					vn->SetVariable(iv);
				} else {
					TConstant* c = TConstant::GetConstant(vname);
					if( c == NULL ) { 
						gPrintErr( vname + ": Unknown variable in " + fVar->GetModule()->Name() + " referenced by " + fVar->Name() ); 
						return 0;  
					} else { 
						vn->SetConstant(c);
						if( c->Dynamic() ) AddDependency( *c );
						return 1; 
					} 
				}
			}
			ArgArray* mlist = vn->GetMapList();
			int index, is_dimension_name;
			if( mlist != NULL  ) {
				if( fHasArraysInEqn < 1 ) { fHasArraysInEqn = 1; }
				int len = mlist->NArgs();
				for( int i=0; i<len; i++ ) {
					CString& map = mlist->elem(i);
					if( map != "*"  ) { 
						ArrayRec* ar = Model::I0().getArrayDeclaration( map, index, is_dimension_name ); 
						if( ar == NULL ) {
						  if (array_error_count++ < 16)
							gPrintErr( format("Unrecognized Array argument %d: %s in Variable %s, Use Ar() config-cmd to declare arrays.",
											index, map.chars(), vn->VariableName().chars() ));
						} else if( is_dimension_name && ( app == Env::kCGP ) ) {
						  if( fHasArraysInEqn < 2 ) { fHasArraysInEqn = 2; }
						  if( fHasArraysInEqn < 3 ) { 
							gPrintErr( format(" Non Array variable %s referencing Array %s[%s] in equation.", fVar->Name(), vname.chars(), map.chars() ) );
						  }
						} else {
						  Variable* mv = ar->getMap();
						  if( fVar->GetModule() != mv->GetModule() ) { 
							Variable* v = fVar->GetModule()->GetVariable( mv->SName(), FALSE ); 
							mv = ( v==NULL ) ?  fVar->GetModule()->CreateInputVariable( mv,"C2_") : v;	
						  }
						  AddDependency( *mv );
						}
					}
					if( v ) { v->SetMap(map,i); }
				}
			}
			EDependencyMode dm = vn->SetDependencyMode();
			if( dm == kDMRelCode ) { 
			  SetObjInfo(TCommand::kDepIndex, TCommand::kHasParallelDep);
			  if( p != NULL ) {
				TDependency& d = (TDependency&) fDependencyList.elem(p);
				d.SetRelCode( vn->RelCode() );
			  }
			}
			if( v && ( (dm == kDMRelCode) || (dm == kDMParFunction) ) ) { 
				v->SetF(FLinkEdges,True,"TCommand::ProcessDependencies"); 
			}
			if( v && v->GetF(FisImport) ) { iv = v; v = v->Connection(); }
			if( fVar->GetF(FisStateVar) && v && (GetObjInfo(kType) == kIntegrate) ) {
				v->SetF(FisFlux,True," ProcessDependencies");
				v->SetCInfo(Variable::kType,Variable::kFlux); 
				v->SetParent(fVar);
				TNode* sn = n->SuperNode();
				int neg_flux = 0;
				if( sn->NodeType() == TNode::kOp ) {
					TOpNode* on = (TOpNode*)sn;
					if( on->IsNegArg(n) ) neg_flux = 1;
				} else if( (sn->NodeType() == TNode::kAttribute) || ( sn->NodeType() == TNode::kEquation ) ) {;}
				else { sprintf(gMsgStr,"Syntax Error in State Variable %s Equation, unexpected node: %x",fVar->Name(),sn->NodeType()); gPrintErr(); }
				Module* m = v->GetModule();
				if( fVar->GetModule() != m ) {
				  m->CreateInputVariable(fVar,"C4_");
				}		 
				if( fVar->GetF(FisClamped) ) {
					v->SetF(FisClamped,True," ProcessDependencies");
					v->SetF(FisConstant,False," ProcessDependencies: Clamped Flux");
//					v->SetF(FHasNegFlux,True," ProcessDependencies");
				}
//				fVar->SetF(FHasNegFlux,True," ProcessDependencies"); 
				if( neg_flux ) { 
					v->SetOrigin(fVar); 
				} else { 
					v->SetDestination(fVar); 
				}
			}
		} break;
		case TNode::kGraph: 
			break;	
		case TNode::kConst: 
			break;	
		case TNode::kOp:
			const TOpNode* on = (const TOpNode*)n; 
			if( on->Operator() == '/' ) {
				TNode* n1 = on->Arg(1);
				if( n1->GetObjInfo(TNode::kNodeType) == TNode::kConst ) {
					TConstNode* cn = (TConstNode*)n1;
					cn->SetObjInfo(TConstNode::kConstType,kAVFloat); 
				}
			}
			break;			
	} 
}


//----------------------------------------------------------------------------------------
//				TCommand::ProcessFunction:
//----------------------------------------------------------------------------------------

enum EFunctionType { kNoFunction, kUtilityFunction, kSpatialFunction, kAggSpatialFunction };

int TCommand::ProcessFunction(TFunctionNode* fn) {
	fn->FindMethod(this);
	TFunction* f = fn->Function();
	if(f) {
		CString& fc = f->FunctionCode();
		fFunction = fc[0];
		if( fc[1] == 'M' && GetObjInfo(kDepIndex) == kNone ) {
			SetObjInfo(kDepIndex, kHasMethodCalls);
		}
	}
	const CString& name = fn->FunctionName();
	if( (fFunction == 'P') || (fFunction == 'X') ) {
		fn->SetDependencyMode(kDMParFunction);
		SetObjInfo(TCommand::kDepIndex, TCommand::kHasParallelDep);
	} 
	if( ( (fFunction == 'S') || (fFunction == 's') )  ) {
		fn->SetDependencyMode(kDSpatialFunction);
		if( fFunction == 'S' ) { fVar->SetF(FisSpatial,True,"ProcessFunction"); }
	} 
	if( fFunction == 'a'  ) {
		fn->SetDependencyMode(kArrayFunction);
	} 
	if( fFunction == 'R' ) {
	  if(fVar->GetF(FisSpatialRand)) fVar->SetF(FisSpatial,True,"ProcessFunction");  
	  fVar->SetF(FisConstant,False,"ProcessFunction"); 
	}
	if( fFunction == 'D' ) {
		int got_var = 0;
		TListNode* ln = fn->ArgList();
		const TOrderedObjectDLList& args  = ln->List();
		Pix p = args.first();
		if( p ) {
			Variable* v = NULL;
			Module* m = fVar->GetModule();
			TNode* n = (TNode*) &args(p);
			if( n->NodeType() == TNode::kVar ) {
				if( m ) {
					const CString& vname = ((TVarNode*)n)->VariableName();
					v = m->GetVariable( vname, FALSE ); 
					if(v) { 
					  v->SetF(FHasDelay,True,"ProcessFunction"); 
					  got_var = 1;
					} 
					else {
						if( v = Model::I0().GetVariable(vname) ) { 
							Variable* iv = m->CreateInputVariable(v,"C5_");					 
							AddDependency( *iv );
							v->SetF(FHasDelay,True,"ProcessFunction");
							iv->SetF(FHasDelay,True,"ProcessFunction");
							got_var = 1;
						} 
					}
				}
			}
		} 
		if( got_var == 0 ) { gPrintErr( fVar->SName() + ": Can't find Variable-type argument to delay function" ); }
	}
	return 0;
}


int TCommand::ProcessCommand(const TCommandNode* n) { return 0; }

TEquationNode* TCommand::getEquationNode(EType etype, TStringListNode* args) {
  if( fArrayNodes == NULL ) return NULL;
  for( Pix p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
	TEquationNode& en = (TEquationNode&)fArrayNodes->elem(p);
	TVarNode* v = en.Variable();
	ArgArray* ml = v->GetMapList();
	switch( etype ) {
	  case kInit: 
		if( en.GetObjInfo(TEquationNode::kEquationType) == TEquationNode::kInit  ) {
		  if( (args == NULL) && ( ml == NULL) ) { return &en; }
		  if( (args != NULL) && ( args->List() == *ml ) ) { return &en; }
		}
	  break;
	  case kUpdate:  
		if( en.GetObjInfo(TEquationNode::kEquationType) == TEquationNode::kUpdate  ) {
		  if( (args == NULL) && ( ml == NULL) ) { return &en; }
		  if( (args != NULL) && ( args->List() == *ml ) ) { return &en; }
		}
	  break;
	  case kIntegrate:  
		if( en.GetObjInfo(TEquationNode::kEquationType) == TEquationNode::kStateVar  ) {
		  if( (args == NULL) && ( ml == NULL) ) { return &en; }
		  if( (args != NULL) && ( args->List() == *ml ) ) { return &en; }
		}
	  break;
	  default:  
		if( en.GetObjInfo(TEquationNode::kEquationType) == TEquationNode::kUnknown  ) {
		  if( (args == NULL) && ( ml == NULL) ) { return &en; }
		  if( (args != NULL) && ( args->List() == *ml ) ) { return &en; }
		}
	  break;
	}
  }  
  return NULL;
}

int TCommand::ProcessAttribute(const TAttributeNode* an) {
	TDeclarationNode* dn = (TDeclarationNode*) an->Declaration();
	TNode* vn = an->Value();
	CString& cmd = (CString&) dn->Command();
	const ArgArray&  ql = dn->QualifierList()->List();
	cmd.downcase();
	if( gDebug > 1 ) {
	  sprintf(gMsgStr,"%s.%s.ProcessAttribute %s",fVar->Name(),Name(),cmd.chars()); gPrintLog();
	}

  if( cmd == "dependencies" ) { return 1; }

  if( (cmd == "value") || (cmd == "code") ) { 
		TStringListNode* args =  dn->getArrayArgs();
		EType etype = (EType) GetObjInfo(kType);
		TEquationNode* en = getEquationNode(etype,args);
		if( en == NULL ) {
		  TVarNode* var = (TVarNode*) ExprBuilder::CreateVarNode(  fVar->Name()  ); 
		  if( args != NULL ) { var->AddMapList( args );  } 
		  switch( etype ) {
			case kInit: en = (TEquationNode*) ExprBuilder::CreateEquationNode( TEquationNode::kInit, var, vn  );  break;
			case kUpdate: en = (TEquationNode*) ExprBuilder::CreateEquationNode( TEquationNode::kUpdate, var, vn  );  break;
			case kIntegrate: en = (TEquationNode*) ExprBuilder::CreateEquationNode( TEquationNode::kStateVar, var, vn  );  break;
			default: en = (TEquationNode*) ExprBuilder::CreateEquationNode( TEquationNode::kUnknown, var, vn  );  break;
		  }
		  var->Super(en);
		  en->Super((TNode*)dn);
		  AddArrayNode( en );
		  if( gDebug > 1 ) {
			CString tmp; vn->WriteMML(tmp); 
			sprintf(gMsgStr," -> Adding ArrayNode { %s }",tmp.chars()); gPrintLog();
		  }
		}
		fNode = en; 
		ProcessGeneral(en); 
		if( vn->NodeType() == TNode::kConst ) { SetObjInfo( kDepIndex, kParameter ); }
	}
 
  if( cmd == "translate" ) {
		if( GetAttrValue( vn, kAVString ) ) {
			const CString* s = fAttrValue;
			if(s) fTranslation = *s;
			return 1;
		}
		return 0;		
  }
  
  if( GetObjInfo(kType) == kIntegrate ) {
    if( cmd == "method" ) {
			if( GetAttrValue( vn, kAVString ) ) {
				const CString* s = fAttrValue;
				if( s ) {
					if( *s == "euler" ) { SetIntegrationInfo(kIntegrationMethod,kEuler); fVar->setIntegrationMethod(0,'1'); }
					else if( *s == "rk2" ) { SetIntegrationInfo(kIntegrationMethod,kRK2); fVar->setIntegrationMethod(0,'2'); }
					else if( *s == "rk3" ) { SetIntegrationInfo(kIntegrationMethod,kRK2); fVar->setIntegrationMethod(0,'3'); }
					else if( *s == "rk4" ) { SetIntegrationInfo(kIntegrationMethod,kRK4); fVar->setIntegrationMethod(0,'4'); }
					else gPrintErr( cmd + " = " + s->chars() + ": Unrecognized value");
					return 1;
				}
			}
    }
    else if( cmd == "clamp" ) {
 			if( GetAttrValue( vn, kAVString, kAVInt ) ) {
				CString s;
				fAttrValue.GetStringVal(s);
				char tc = s[0];
				if( tc == 't' || tc == '1' ) { SetIntegrationInfo(kClamp,1); fVar->setIntegrationMethod(1,'C'); }
				else if ( tc == 'f' || tc == '0' ) { SetIntegrationInfo(kClamp,2); fVar->setIntegrationMethod(1,'N'); }
				else gPrintErr( cmd + " = " + s.chars() + ": Unrecognized value" );
				return 1;
			}
    }   
  } else {
	 if( cmd == "fluxtype" ) {
 			if( GetAttrValue( vn, kAVString, kAVInt ) ) {
				CString s;
				fAttrValue.GetStringVal(s);
				fCommandInfo[kFluxType] = s[0];
				fVar->setFluxType(s);
				if( gDebug > 1 ) {
				  sprintf(gMsgStr,"%s.%s.setFluxtype: %s",fVar->Name(),Name(),s.chars()); gPrintLog();
				}
				return 1;
			}
	 } 
  }
  return 0;
}

//----------------------------------------------------------------------------------------
//				TCommand::ProcessNodeRecur:
//----------------------------------------------------------------------------------------

int TCommand::ProcessNodeRecur(TNode* n, EProcessContext pc) {
	if( n==NULL ) return 0;
	ProcessNodes(n,pc);
	switch( n->NodeType() ) {
		case TNode::kGraph: {
			TGraphNode* gn = (TGraphNode*)n;
			ProcessNodeRecur(gn->Arg(),pc);
			} break;
		case TNode::kPoint: {
			TPointNode* pn = (TPointNode*)n;
			} break;
		case TNode::kList: {
			TListNode* ln = (TListNode*)n;
			const TOrderedObjectDLList& list = ln->List();
			for( Pix p = list.first(); p; list.next(p) ) {
				TNode* n = (TNode*)&list(p); 
				ProcessNodeRecur(n,pc);
			}
			} break;
		case TNode::kFunc: {
			TFunctionNode* fn = (TFunctionNode*)n;
			ProcessNodeRecur(fn->ArgList(),pc);
			} break;
		case TNode::kCond: {
			TCondNode* cn = (TCondNode*)n;
			ProcessNodeRecur(cn->CondExpr(),pc);
			ProcessNodeRecur(cn->ThenExpr(),pc);
			ProcessNodeRecur(cn->ElseExpr(),pc);
			} break;
		case TNode::kConst: {
			TConstNode* cn = (TConstNode*)n;
			} break;
		case TNode::kEquation: {
			TNode* expr = ((TEquationNode*)n)->Expr();
			ProcessNodeRecur(expr,pc);
			} break;
		case TNode::kVar: { 
			TVarNode* vn = (TVarNode*)n;
			} break;
		case TNode::kCondExpr: {
			TCondExpNode* cen = (TCondExpNode*)n;
			ProcessNodeRecur(cen->Arg(0),pc);
			ProcessNodeRecur(cen->Arg(1),pc);
			} break;
		case TNode::kOp: {
			TOpNode* on = (TOpNode*)n;
			ProcessNodeRecur(on->Arg(0),pc);
			ProcessNodeRecur(on->Arg(1),pc);
			} break;
		default:
			sprintf(gMsgStr,"%s: Unexpected Node type(%x)",Name(),n->NodeType());
			gPrintErr();
			return 0;
	}
	return 1;
}
 
void TCommand::PrintOpenOptSpatialLoop( FILE* CEqnFile, int ntabs ) const {
	char tabs[10]; Pix p; 
	for( int i=0; i<ntabs; i++ ) { tabs[i] = '\t'; }
	tabs[ntabs] = 0;

	switch( Model::I0().SMP() ) {
		case 0:
			fprintf(CEqnFile,"\n%swhile( %s_ptr < %s_Eptr ) {", tabs, fVar->Name(), fVar->Name() ); 
		break;
		case 1:
			fprintf(CEqnFile,"\n%sDistributedGrid& g = %s.Grid();",tabs,fVar->Name());
			fprintf(CEqnFile,"\n%sint size = g.length();",tabs); 	
			fprintf(CEqnFile,"\n#pragma parallel local( ");	
			fprintf(CEqnFile," %s_ptr",fVar->Name());
			for ( p =  fDependencyList.first(); p; fDependencyList.next(p) ) {
				TDependency& d = (TDependency&) fDependencyList(p);
				Variable* depVar = d.GetVar();
				if( depVar ) {
					if( !depVar->GetF(FisIgnored) && !(depVar->SName() == "TIME") ) { 
						fprintf(CEqnFile,", %s_ptr", depVar->Name() );
					}
				}
			}
			fprintf(CEqnFile," ) \n{ ");	
			fprintf(CEqnFile,"\n#pragma pfor ");	
			fprintf(CEqnFile,"\n%sfor( int i=0; i < size; i++ ) {",tabs);	
			fprintf(CEqnFile,"\n%s\t%s_ptr = %s.GetDataPtr(i);", tabs, fVar->Name(), fVar->Name() );
			for ( p =  fDependencyList.first(); p; fDependencyList.next(p) ) {
				TDependency& d = (TDependency&) fDependencyList(p);
				Variable* depVar = d.GetVar();
				if( depVar ) {
					if( !depVar->GetF(FisIgnored) && !(depVar->SName() == "TIME") ) { 
						fprintf(CEqnFile,"\n%s\t%s_ptr = %s.GetDataPtr(i);", tabs, depVar->Name(), depVar->Name() );
					}
				}
			}
		break;
	}
}

void TCommand::PrintCloseOptSpatialLoop( FILE* CEqnFile, int ntabs ) const {
	char tabs[10]; 
	for( int i=0; i<ntabs; i++ ) { tabs[i] = '\t'; }
	tabs[ntabs] = 0;

	switch( Model::I0().SMP() ) {
		case 0: {
			fprintf(CEqnFile,"\n%s\t%s_ptr++;",tabs,fVar->Name());
			for ( Pix p =  fDependencyList.first(); p; fDependencyList.next(p) ) {
				TDependency& d = (TDependency&) fDependencyList(p);
				Variable* depVar = d.GetVar();
				if( depVar ) {
					if( !depVar->GetF(FisIgnored) && !(depVar->SName() == "TIME") ) { 
						fprintf(CEqnFile,"\n%s\t%s.UpdateDataPtr(%s_ptr);", tabs, depVar->Name(), depVar->Name() );
					}
				}
			}
			fprintf(CEqnFile,"\n%s}",tabs); 
		} break;
		case 1:
			fprintf(CEqnFile,"\n%s}",tabs); 
			fprintf(CEqnFile,"\n} // pragma parallel ");	
		break;
	}
}

void TCommand::WriteCCode( FILE* CHdrFile, FILE* CEqnFile) {
  if( GetObjInfo(kExType) != kCode ) return;
  fprintf(CHdrFile,"\n\tint %s_%s_exe();",fVar->Name(),Name());
  fprintf(CEqnFile,"\nint %s::%s_%s_exe() {\n\tfloat value;\n", fVar->GetModule()->Name(), fVar->Name(), Name() );
  if( fHasArraysInEqn == 3 ) {
	  fprintf(CEqnFile,"\n\tstatic int array_index[4] = { -1,-1,-1,-1 };\n\n" );
  }
  Pix p;
/*
  TAssoc_array array_vals;
  for( p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
	TEquationNode* en = (TEquationNode*)(&fArrayNodes->elem(p));
	TVarNode* v = en->Variable();
	ArgArray* ml = v->GetMapList(); 
	if( ml != NULL ) {
	  int is_dimension_name, index; 
	  for( int iarg=0; iarg < ml->NArgs(); iarg++ ) {
		CString& arg = ml->elem(iarg);
		ArrayRec* ar = Model::I0().getArrayDeclaration( arg, index, is_dimension_name ); 
		if( ar == NULL ) {
		  if (array_error_count++ < 16)
			  gPrintErr( format("Unrecognized Array argument %d: %s in Variable %s, Use Ar() config-cmd to declare arrays.",
							index, arg.chars(), v->VariableName().chars() ));
		} else if( !is_dimension_name  ) {
		  if( !array_vals.add(arg.chars(),ml) ) {
			fprintf(CEqnFile,"\n\tint kA__%s = %s.GetCategoryIndex(\"%s\");",arg.chars(),ar->getMap()->Name(),arg.chars());
		  }
		}
	   }
	 }
   }
*/
  if( GetObjInfo(kType) == kFunction ) {
		WriteSwitchCCode( CEqnFile, 0, True );
		if( fVar->GetF(FisStateVar) ) { fprintf(CEqnFile,"\n\t%s.UpdateReserve();",fVar->Name());	}
  } else {
		int isInit = (GetObjInfo(kType) == kInit);
		if( fVar->GetF(FisStateVar) ) { fprintf(CEqnFile,"\n\t%s.UpdateIntegrator(%d);",fVar->Name(),isInit);	}
		fprintf(CEqnFile,"\n\tswitch( %s.UpdateMode() ) {",fVar->Name());
		fprintf(CEqnFile,"\n\t  case CVariable::kFastSpatial: ");
/*
		if( Optimizable() ) {
			fprintf(CEqnFile,"{\n\t\tfloat* %s_ptr = %s.DataStart();",fVar->Name(), fVar->Name() ); 
			fprintf(CEqnFile,"\n\t\tconst float* %s_Eptr = %s.GridEnd();",fVar->Name(), fVar->Name() );
			if( fVar->GetF(FisStateVar) ) {
				fprintf(CEqnFile,"\n\t\tfloat dt = dT();");
				fprintf(CEqnFile,"\n\t\tfloat* %s_Rptr = %s.RDataStart();",fVar->Name(), fVar->Name() );
			}      
			else if( fVar->GetF(FisFlux) ) {
				fprintf(CEqnFile,"\n\t\tfloat dt = dT();");
				if( fVar->Origin() ) {
					if( !(fVar->Origin()->activationLayerIndex() == fVar->activationLayerIndex()) ) {
							sprintf( gMsgStr, " %s : Flux origon in module with different ActivationLayer: not allowed.", Name() );
							gFatal( );
					}
					fprintf(CEqnFile,"\n\t\tfloat* %s_Optr = ((CStateVariable*)(%s.Origin()))->RDataStart();",fVar->Name(),fVar->Name());
				}
			}
			for ( p =  fDependencyList.first(); p; fDependencyList.next(p) ) {
				TDependency& d = (TDependency&) fDependencyList(p);
				Variable* depVar = d.GetVar();
				if( depVar ) {
					if(  !depVar->GetF(FisIgnored) ) { 
						if( d.Function() ==  'I' ) fprintf(CEqnFile,"\n\t\tfloat* %s_Iptr = %s.InitValData();", depVar->Name(), depVar->Name() );
						else if( depVar->SName() == "TIME" ) fprintf(CEqnFile,"\n\t\tfloat time = %s.Value();", depVar->Name() ); 
						else fprintf(CEqnFile,"\n\t\tfloat* %s_ptr = %s.DataStart();", depVar->Name(), depVar->Name() ); 
					}
				}
			}
			if( fVar->GetF(FisFlux) ) {
				Variable* ov = fVar->Origin();
				if( ov  ) {
					fprintf(CEqnFile,"\n\t\tint clamped = %s.GetF(FisClamped);",ov->Name()); 
				} else {
					fprintf(CEqnFile,"\n\t\tint clamped = 0;"); 
				}	
			}
//			fprintf(CEqnFile,"\n\t\t%s.SetF(FEdgesLinked,False,\"UpdateSV\");",fVar->Name()); 
			switch( GetObjInfo(kType) ) {
			case kIntegrate:
					fprintf(CEqnFile,"\n\t\tswitch( %s.CurrentOrder() ) {",fVar->Name());
					fprintf(CEqnFile,"\n\t\t  case 1: {");
					fprintf(CEqnFile,"\n\t\t\tif( %s.Order() > 1 ) {",fVar->Name());
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr0 = %s.SIntegratorMemory().LastData();",fVar->Name(),fVar->Name());
					PrintOpenOptSpatialLoop(CEqnFile,3);
					WriteSwitchCCode( CEqnFile, 0 );
					fprintf(CEqnFile,"\n\t\t\t\t*%s_ptr += value*dt;", fVar->Name() );
					fprintf(CEqnFile,"\n\t\t\t\tif(%s_Rptr) { *%s_Rptr++ = *%s_ptr; }", fVar->Name(), fVar->Name(), fVar->Name() );
					fprintf(CEqnFile,"\n\t\t\t\t*%s_IMptr0++ = value;", fVar->Name());
					PrintCloseOptSpatialLoop(CEqnFile,3);
					fprintf(CEqnFile,"\n\t\t\t} else {");
					PrintOpenOptSpatialLoop(CEqnFile,3);
					WriteSwitchCCode( CEqnFile, 0 );
					fprintf(CEqnFile,"\n\t\t\t\t*%s_ptr += value*dt;", fVar->Name() );
					fprintf(CEqnFile,"\n\t\t\t\tif(%s_Rptr) { *%s_Rptr++ = *%s_ptr; }", fVar->Name(), fVar->Name(), fVar->Name() );
					PrintCloseOptSpatialLoop(CEqnFile,3);
					fprintf(CEqnFile,"\n\t\t\t};");
					fprintf(CEqnFile,"\n\t\t  } break; case 2: {");
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr0 = %s.SIntegratorMemory().LastData();",fVar->Name(),fVar->Name());
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr1 = %s.SIntegratorMemory().LastDataOffset(1);",fVar->Name(),fVar->Name());
					PrintOpenOptSpatialLoop(CEqnFile,3);
					WriteSwitchCCode( CEqnFile, 0 );
					fprintf(CEqnFile,"\n\t\t\t\t*%s_IMptr0++ = value;", fVar->Name());
					fprintf(CEqnFile,"\n\t\t\t\t*%s_ptr += ( 1.5*value - 0.5* *%s_IMptr1++ )*dt;", fVar->Name(), fVar->Name() );
					fprintf(CEqnFile,"\n\t\t\t\tif(%s_Rptr) { *%s_Rptr++ = *%s_ptr; }", fVar->Name(), fVar->Name(), fVar->Name() );
					PrintCloseOptSpatialLoop(CEqnFile,3);
					fprintf(CEqnFile,"\n\t\t  } break; case 3: {");
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr0 = %s.SIntegratorMemory().LastData();",fVar->Name(),fVar->Name());
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr1 = %s.SIntegratorMemory().LastDataOffset(1);",fVar->Name(),fVar->Name());
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr2 = %s.SIntegratorMemory().LastDataOffset(2);",fVar->Name(),fVar->Name());
					PrintOpenOptSpatialLoop(CEqnFile,3);
					WriteSwitchCCode( CEqnFile, 0 );
					fprintf(CEqnFile,"\n\t\t\t\t*%s_IMptr0++ = value;", fVar->Name());
					fprintf(CEqnFile,"\n\t\t\t\t*%s_ptr += (1.92*value - 1.33* *%s_IMptr1++ + .42* *%s_IMptr2++)*dt;", fVar->Name(), fVar->Name(), fVar->Name() );
					fprintf(CEqnFile,"\n\t\t\t\tif(%s_Rptr) { *%s_Rptr++ = *%s_ptr; }", fVar->Name(), fVar->Name(), fVar->Name() );
					PrintCloseOptSpatialLoop(CEqnFile,3);
					fprintf(CEqnFile,"\n\t\t  } break; case 4: {");
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr0 = %s.SIntegratorMemory().LastData();",fVar->Name(),fVar->Name());
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr1 = %s.SIntegratorMemory().LastDataOffset(1);",fVar->Name(),fVar->Name());
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr2 = %s.SIntegratorMemory().LastDataOffset(2);",fVar->Name(),fVar->Name());
					fprintf(CEqnFile,"\n\t\t\tfloat *%s_IMptr3 = %s.SIntegratorMemory().LastDataOffset(3);",fVar->Name(),fVar->Name());
					PrintOpenOptSpatialLoop(CEqnFile,3);
					WriteSwitchCCode( CEqnFile, 0 );
					fprintf(CEqnFile,"\n\t\t\t\t*%s_IMptr0++ = value;", fVar->Name());
					fprintf(CEqnFile,"\n\t\t\t\t*%s_ptr += (2.29*value - 2.46* *%s_IMptr1++  + 1.54* *%s_IMptr2++ - .375* *%s_IMptr3++ )*dt;", 
					  fVar->Name(), fVar->Name(), fVar->Name(), fVar->Name() );
					fprintf(CEqnFile,"\n\t\t\t\tif(%s_Rptr) { *%s_Rptr++ = *%s_ptr; }", fVar->Name(), fVar->Name(), fVar->Name() );
					PrintCloseOptSpatialLoop(CEqnFile,3);
					fprintf(CEqnFile,"\n\t\t  } break; default: { gPrintErr(\"Illegal Order in StateVar update!\"); }");
					fprintf(CEqnFile,"\n\t\t};");
					if( fVar->Connection() ) { 
					  fprintf(CEqnFile,"\n\t\tif(%s.GetF(FDualConnection)) %s.Connection()->RefreshData();",fVar->Name(),fVar->Name());					
					}
				break;
			case kInit:
					PrintOpenOptSpatialLoop(CEqnFile,2);
					WriteSwitchCCode( CEqnFile, 0 );
					fprintf(CEqnFile,"\n\t\t\tif(%s_Rptr) { *%s_Rptr = *%s_ptr = value;  %s_Rptr++; }", 
					  fVar->Name(), fVar->Name(),fVar->Name(),fVar->Name());
					fprintf(CEqnFile,"\n\t\t\telse { *%s_ptr = value; }",fVar->Name());
					PrintCloseOptSpatialLoop(CEqnFile,2);
				break;
			case kUpdate: 
					if( fVar->GetCInfo(Variable::kInitMode) == Variable::kParameter ) { 
						WriteSwitchCCode( CEqnFile, 0 );							
						fprintf(CEqnFile,"\n\t\tvalue = %s.ParameterValue(value);",fVar->Name());
						PrintOpenOptSpatialLoop(CEqnFile,2);
						fprintf(CEqnFile,"\n\t\t\t*%s_ptr = value;", fVar->Name() );
						if( fVar->GetF(FisFlux) ) {
							if( fVar->Origin() ) { fprintf(CEqnFile,"\n\t\t\tif( clamped ) { SL::Clamp(%s_Optr++,%s_ptr,dt); }",fVar->Name(),fVar->Name()); }
						}
					} else {
						PrintOpenOptSpatialLoop(CEqnFile,2);
						WriteSwitchCCode( CEqnFile, 0 );
						fprintf(CEqnFile,"\n\t\t\t*%s_ptr = value;",fVar->Name());
						if( fVar->GetF(FisFlux) ) {
							if( fVar->Origin() ) { fprintf(CEqnFile,"\n\t\t\tif( clamped ) { SL::Clamp(%s_Optr++,%s_ptr,dt); }",fVar->Name(),fVar->Name()); }
						}
					}
					PrintCloseOptSpatialLoop(CEqnFile,2);
				break;
			case kDerivative: 
					if( fVar->GetCInfo(Variable::kInitMode) == Variable::kParameter ) { 
						PrintOpenOptSpatialLoop(CEqnFile,2);
							fprintf(CEqnFile,"\n\t\t\t*%s_ptr = 0.0;",fVar->Name());
					} else {
						if( fVar->GetF(FisStateVar) ) { fprintf(CEqnFile,"\n\t\tfloat dt = dT();"); }
						PrintOpenOptSpatialLoop(CEqnFile,2);
						WriteSwitchCCode( CEqnFile, 0 );
						if( fVar->GetF(FisStateVar) ) {
							fprintf(CEqnFile,"\n\t\t\t*%s_ptr += value*dt;",fVar->Name());
						} else {
							fprintf(CEqnFile,"\n\t\t\t*%s_ptr = value;",fVar->Name());
						}
					}
					PrintCloseOptSpatialLoop(CEqnFile,2);
				break;
			};
			if( fVar->GetF(FisFlux) ) {
				fprintf(CEqnFile,"\n\t\tif( %s.GetF(FICFlux) ) %s.Cov().Set(0.0); ",fVar->Name(),fVar->Name());
			}
			fprintf(CEqnFile,"\n\t\t%s.GrabMemory();",fVar->Name());
			if( fVar->GetF(FHasDelay) ) fprintf(CEqnFile,"\n\t\t%s.UpdateDelayData();",fVar->Name());
			fprintf(CEqnFile,"\n\t\tif( %s.GetF(FLinkEdges) ) { %s.LinkEdges(); }",fVar->Name(),fVar->Name());
			fprintf(CEqnFile,"\n\t  } break;");
		}
*/
		int n_array_loops = 0;
		fprintf(CEqnFile,"\n\t  case CVariable::kSpatial: {");
		if( Function() == 'X' ) { 
		  n_array_loops = WriteSwitchCCode( CEqnFile, 1, True );
		}
		else {
//			fprintf(CEqnFile,"\n\t\t%s.SetF(FEdgesLinked,False,\"UpdateSV\");",fVar->Name()); 
			switch( GetObjInfo(kType) ) {
			case kIntegrate:
				PrintOpenSpaceLoop (CEqnFile,2);
				n_array_loops = WriteSwitchCCode( CEqnFile, 1 );
				if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( p, value, dT() );",fVar->Name());
				else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( p, value, array_index, dT() );",fVar->Name());
				for( int ia=0; ia<n_array_loops; ia++ ) { fprintf(CEqnFile,"\n\t\t  } ");   }
				PrintCloseSpaceLoop (CEqnFile,2);
				if( fVar->Connection() ) { fprintf(CEqnFile,"\n\t\tif(%s->GetF(FDualConnection)) %s->Connection()->RefreshData();",fVar->Name(),fVar->Name()); }
				break;
			case kInit:
				PrintOpenSpaceLoop (CEqnFile,2);
				n_array_loops = WriteSwitchCCode( CEqnFile, 1 );
				if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.InitValue( p, value );",fVar->Name());
				else fprintf(CEqnFile,"\n\t\t\t%s.ArrayInitValue( p, value, array_index );",fVar->Name());
				for( int ia=0; ia<n_array_loops; ia++ ) { fprintf(CEqnFile,"\n\t\t  } ");   }
				PrintCloseSpaceLoop (CEqnFile,2);
				break;
			case kUpdate:
				if( fVar->GetCInfo(Variable::kInitMode) == Variable::kParameter ) { 
					PrintOpenSpaceLoop (CEqnFile,2);
					n_array_loops = WriteSwitchCCode( CEqnFile, 2 );
					if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( p, %s.ParameterValue(value,p) );",fVar->Name(),fVar->Name());
					else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( p, %s.ParameterValue(value,p), array_index  );",fVar->Name(),fVar->Name());
					for( int ia=0; ia<n_array_loops; ia++ ) { fprintf(CEqnFile,"\n\t\t  } ");   }
					PrintCloseSpaceLoop (CEqnFile,2);
				} else {
					PrintOpenSpaceLoop (CEqnFile,2);
					n_array_loops = WriteSwitchCCode( CEqnFile, 1 );
					if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( p, value );",fVar->Name() );
					else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( p, value, array_index );",fVar->Name() );
					for( int ia=0; ia<n_array_loops; ia++ ) { fprintf(CEqnFile,"\n\t\t  } ");   }
					PrintCloseSpaceLoop (CEqnFile,2);
				}
				break;
			case kDerivative: 
				if( fVar->GetF(FisStateVar) ) { fprintf(CEqnFile,"\n\t\tfloat dt = dT();"); }
				PrintOpenSpaceLoop (CEqnFile,2);
				n_array_loops = WriteSwitchCCode( CEqnFile, 1 );
				if( fVar->GetF(FisStateVar) ) {
					if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( p, value, dt );", fVar->Name() );
					else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( p, value, array_index, dt );", fVar->Name() );
				} else {
					if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( p, value );", fVar->Name() );
					else  fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( p, value, array_index );", fVar->Name() );
				}
				for( int ia=0; ia<n_array_loops; ia++ ) { fprintf(CEqnFile,"\n\t\t  } ");   }
				PrintCloseSpaceLoop (CEqnFile,2);
				break;
			}
		}
		fprintf(CEqnFile,"\n\t\t%s.GrabMemory();",fVar->Name());
		if( fVar->GetF(FHasDelay) ) {
		  fprintf(CEqnFile,"\n\t\t%s.UpdateDelayData();",fVar->Name());
		}
		if( fVar->GetF(FisFlux) ) {
			fprintf(CEqnFile,"\n\t\tif( %s.GetF(FICFlux) ) %s.Cov().Set(0.0); ",fVar->Name(),fVar->Name());
		}
		fprintf(CEqnFile,"\n\t\tif( %s.GetF(FLinkEdges) ) { %s.LinkEdges(); }",fVar->Name(),fVar->Name());
		fprintf(CEqnFile,"\n\t  } break;\n\t  case CVariable::kTemporal: {");
		if( Function() == 'X' ) { 
		  n_array_loops = WriteSwitchCCode( CEqnFile, 2, True );
		}
		else {
			switch( GetObjInfo(kType) ) {
			case kIntegrate:
				n_array_loops = WriteSwitchCCode( CEqnFile, 2 );
				if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( value, dT() );",fVar->Name());
				else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( value, array_index, dT() );",fVar->Name());
				break;
			case kInit:
				n_array_loops = WriteSwitchCCode( CEqnFile, 2 );
				if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.InitValue( value );",fVar->Name());
				else fprintf(CEqnFile,"\n\t\t\t%s.ArrayInitValue( value, array_index );",fVar->Name());
				break;
			case kUpdate:
				if( fVar->GetCInfo(Variable::kInitMode) == Variable::kParameter ) { 
					n_array_loops = WriteSwitchCCode( CEqnFile, 2 );
					if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( %s.ParameterValue(value) );",fVar->Name(),fVar->Name());
					else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( %s.ParameterValue(value), array_index  );",fVar->Name(),fVar->Name());
				} else {
					n_array_loops = WriteSwitchCCode( CEqnFile, 2 );
					if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( value );",fVar->Name());
					else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( value, array_index );",fVar->Name());
				}
				break;
			case kDerivative: 
				if( fVar->GetF(FisStateVar) ) {
					n_array_loops = WriteSwitchCCode( CEqnFile, 2 );
					if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( value, dT() );",fVar->Name());
					else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( value, dT(), array_index );",fVar->Name());
				} else {
					n_array_loops = WriteSwitchCCode( CEqnFile, 2 );
					if( n_array_loops == 0 ) fprintf(CEqnFile,"\n\t\t\t%s.Update( value );",fVar->Name());
					else fprintf(CEqnFile,"\n\t\t\t%s.ArrayUpdate( value, array_index );",fVar->Name());
				}
				break;
			}
		}
		for( int ia=0; ia<n_array_loops; ia++ ) {
		  fprintf(CEqnFile,"\n\t\t  } ");  
		}
		fprintf(CEqnFile,"\n\t  } break;\n\t}");
  }
//  fprintf(CEqnFile,"\n\t%s.UpdateGOFComputation();",fVar->Name());
  fprintf(CEqnFile,"\n\treturn 0;\n}\n");
}


int TCommand::WriteSwitchCCode(  FILE* CEqnFile, int mode, Bool isFunction ) {
  int first_if=1, has_conds = 0, array_index[4], n_array_loops = 0;
  memset(array_index,0,4*sizeof(int));
  CString eqn_prefix( (isFunction) ? "" : " value = ");
  Pix p;
  for( p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
	TEquationNode* en = (TEquationNode*)(&fArrayNodes->elem(p));
	TVarNode* v = en->Variable();
	ArgArray* ml = v->GetMapList(); 
	if( ml != NULL ) {
	  int is_dimension_name, index; 
	  for( int iarg=0; iarg < ml->NArgs(); iarg++ ) {
		CString& arg = ml->elem(iarg);
		ArrayRec* ar = Model::I0().getArrayDeclaration( arg, index, is_dimension_name ); 
		if( ar == NULL  ) {
		  if (array_error_count++ < 16)
			gPrintErr( format("Unrecognized Array argument %d: %s in Variable %s, Use Ar() config-cmd to declare arrays.",
							index, arg.chars(), v->VariableName().chars() ) );
		} else if( array_index[iarg]==0 ) {
		  array_index[iarg]=1;
		  if( fVar->GetCInfo(Variable::kInitMode) == Variable::kDBase ) {
			switch( mode ) {
			  case 0:
				fprintf(CEqnFile,"\n\t\tarray_index[%d] = *%s_ptr;",iarg,ar->getMap()->Name());
			  break;
			  case 1:
				fprintf(CEqnFile,"\n\t\tarray_index[%d] = %s.Value(p);",iarg,ar->getMap()->Name());
			  break;
			  case 2:
				fprintf(CEqnFile,"\n\t\tarray_index[%d] = %s.Value();",iarg,ar->getMap()->Name());
			  break;
			}
		  } else {
			fprintf(CEqnFile,"\n\t\t  for(array_index[%d] = 0; array_index[%d] < kA__%s__Max; array_index[%d]++ ) {",iarg,iarg,ar->Name(),iarg);
			n_array_loops++;
		  }
		}
	   }
	 } 
   }
   for( p = fArrayNodes->first(); p; fArrayNodes->next(p) ) {
	TEquationNode* en = (TEquationNode*)(&fArrayNodes->elem(p));
	fNode = en;
	TVarNode* v = en->Variable();
	ArgArray* ml = v->GetMapList(); 
	if( ml == NULL ) {
	  fprintf(CEqnFile,"\n\t\t\t%s %s;",eqn_prefix.chars(),(const char*)en->getValueString(mode));
	  return n_array_loops;
	}
	int is_dimension_name, index, write_cond = 0; 
	for( int iarg=0; iarg<ml->NArgs(); iarg++ ) {
	  CString& arg = ml->elem(iarg);
	  ArrayRec* ar = Model::I0().getArrayDeclaration( arg, index, is_dimension_name ); 
	  if( !is_dimension_name ) {
		if( write_cond == 0 ) {
		  write_cond = 1; has_conds = 1;
		  if( first_if ) {
			fprintf(CEqnFile,"\n\t\t\tif( "); first_if = 0;
		  } else {
			fprintf(CEqnFile,"\n\t\t\telse if( ");
		  }
		} else {
		  fprintf(CEqnFile," && ");
		}
		fprintf(CEqnFile,"( array_index[%d] == kA__%s )",iarg,arg.chars());
	  } 
	}
	if( write_cond ) {
	  fprintf(CEqnFile," ) { " );
	}
	if( is_dimension_name && has_conds ) {
	  fprintf(CEqnFile,"\n\t\t\telse { ");	
	}
	
	fprintf(CEqnFile,"\n\t\t\t  %s %s;",eqn_prefix.chars(),(const char*)en->getValueString(mode));
	
	if( write_cond || (is_dimension_name && has_conds) ) {
	  fprintf(CEqnFile,"\n\t\t\t} ");
	}	
  }
  return n_array_loops;
}

void TCommand::WriteCCommandCode(FILE* CEqnFile) { 
    switch( GetObjInfo(kExType) ) {
    case kCode: { 
			switch( GetObjInfo(kType) ) {
			case kIntegrate:
				fprintf(CEqnFile,"\n\tTExecuteCommand* %s = (TExecuteCommand*) %s.GetCommand(\"%s\",TCommand::kIntegrate,TCommand::kCode);",
					Name(),fVar->Name(),Name());
				fprintf(CEqnFile,"\n\t%s->SetExecutable(&%s_%s_exe_g);",Name(),fVar->Name(),Name());
				fprintf(CEqnFile,"\n\t%s->Code() = \"%s <- %s\";",Name(),Name(),(const char*)getValueString(3,0));
				break;
			case kInit:
				fprintf(CEqnFile,"\n\tTExecuteCommand* %s = (TExecuteCommand*) %s.GetCommand(\"%s\",TCommand::kInit,TCommand::kCode);",
					Name(),fVar->Name(),Name());
				fprintf(CEqnFile,"\n\t%s->SetExecutable(&%s_%s_exe_g);",Name(),fVar->Name(),Name());
				fprintf(CEqnFile,"\n\t%s->Code() = \"%s <- %s\";",Name(),Name(),(const char*)getValueString(3,0));
				break;
			case kUpdate:
			case kDerivative:
				fprintf(CEqnFile,"\n\tTExecuteCommand* %s = (TExecuteCommand*) %s.GetCommand(\"%s\",TCommand::kUpdate,TCommand::kCode);",
					Name(),fVar->Name(),Name());
				fprintf(CEqnFile,"\n\t%s->SetExecutable(&%s_%s_exe_g);",Name(),fVar->Name(),Name());
				fprintf(CEqnFile,"\n\t%s->Code() = \"%s <- %s\";",Name(),Name(),(const char*)getValueString(3,0));
				if( fVar->Origin() != NULL ) {
					fprintf(CEqnFile,"\n\tif( %s.GetF(FisClamped) ) { %s->AddDependency(%s); }",fVar->Origin()->Name(),Name(),fVar->Origin()->Name());
				}
				break;
			case kFunction:
				fprintf(CEqnFile,"\n\tTExecuteCommand* %s = (TExecuteCommand*) %s.GetCommand(\"%s\",TCommand::kFunction,TCommand::kCode);",
					Name(),fVar->Name(),Name());
				fprintf(CEqnFile,"\n\t%s->SetExecutable(&%s_%s_exe_g);",Name(),fVar->Name(),Name());
				fprintf(CEqnFile,"\n\t%s->Code() = \"%s <- Function\";",Name(),Name());
				fprintf(CEqnFile,"\n\t%s->GetFunction() = '%c';",Name(),Function());
				break;
			}
		} break;
    case kImport: {
		fprintf(CEqnFile,"\n\tTImportCommand* %s = (TImportCommand*) %s.GetCommand(\"%s\",TCommand::kUndefined,TCommand::kImport);", 
																												Name(),fVar->Name(),Name());
		Variable* cv = fVar->Connection();
		const char* origin_module = (cv == NULL) ? "external" : cv->GetModule()->Name();
		fprintf(CEqnFile,"\n\t%s->Code() = \"Import %s from %s\";",Name(),fVar->Name(),origin_module);
      } break;
    }
    for( Pix pv = fDependencyList.first(); pv; fDependencyList.next(pv) ) {
      TDependency& d = (TDependency&) fDependencyList(pv);
      Variable* depVar = d.GetVar();
      if( depVar ) {
		CString* rc = d.RelCode();
		if(  rc != NULL ) {
			fprintf(CEqnFile,"\n\t%s->AddDependency(%s,\"%s\");",Name(),depVar->Name(),rc->chars());
		} else {
			fprintf(CEqnFile,"\n\t%s->AddDependency(%s);",Name(),depVar->Name());
		}
      }
    }
	if( fFunction == 'R' ) {
	  fprintf(CEqnFile,"\n\t%s->AddDependency(TIME);",Name());
	}
    fprintf(CEqnFile,"\n\t%s->SetObjInfo(TCommand::kDepIndex, %d);",Name(),GetObjInfo(kDepIndex));
}

//----------------------------------------------------------------------------------------
//				TCommand::AddDependency:
//----------------------------------------------------------------------------------------


Pix TCommand :: AddDependency( Variable& v, const char* relcode ) { 
	static int index=0;
	if( fVar->GetF(FisIgnored) ) {
	  sprintf(gMsgStr,"Attempt to Add Dependency %s:%s(%p) to Ignored Command %s::%s:%s-> Rejected ", 
			  v.GetModule()->Name(), v.Name(), &v, fVar->GetModule()->Name(), fVar->Name(), Name() ); 
	  gPrintScreen(); 
	  return 0;
	}
	if( v.GetCInfo(Variable::kInitMode) == Variable::kDBase ) {
		Pipe* p = v.GetPipe(Pipe::kDBaseInput);
		if(p) AddPipeDependencies(p);	
	}
	TDependency* d = new TDependency(v,index++);
	if( relcode != NULL ) {
		d->SetRelCode(relcode);
	}
	if( gDebug > 2 ) { 
	  sprintf(gMsgStr,"$$D Adding Dependency %s:%s(%p) to Command %s::%s:%s", 
			  v.GetModule()->Name(), v.Name(), &v, fVar->GetModule()->Name(), fVar->Name(), Name() ); 
	  gPrintLog(); 
	}
	return fDependencyList.Insert(*d);
}

Pix TCommand :: AddDependency( TTime t ) { 
	fVar->SetF(FisConstant,False, "AddDependency: TTime" ); 
	return 0;
}

Pix TCommand :: AddDependency( TConstant& c ) { 
	if( fVar->GetF(FisIgnored) ) return 0;
	TDependency* d = new TDependency(c);
	return fDependencyList.Insert(*d);	
}

//----------------------------------------------------------------------------------------
//				TCommand::SetBreak:
//----------------------------------------------------------------------------------------

void TCommand :: SetBreak( int breakIndex ) {
		CString msg(fVar->SName()); msg += " -> "; msg += SName();
    if( GetObjInfo(kStatus) == kInActive ) gPrintErr( SName() + ": Can't Set Break on Inactive Command" );
		if(breakIndex) { if(!fCommandInfo[kBreak]) gPrintScreen( msg += " : BreakSet" ); } 
		else if(fCommandInfo[kBreak]) gPrintScreen( msg += " : BreakClear" );				
		fCommandInfo[kBreak] = breakIndex;
} 

//----------------------------------------------------------------------------------------
//				TCommand::PrintOpenSpaceLoop:
//----------------------------------------------------------------------------------------

void TCommand :: PrintOpenSpaceLoop (FILE* CEqnFile, int ntabs ) const {
	char tabs[10]; 
	for( int i=0; i<ntabs; i++ ) { tabs[i] = '\t'; }
	tabs[ntabs] = 0;
	
	switch( Model::I0().SMP() ) {
		case 0:
			fprintf(CEqnFile,"\n%sDistributedGrid& g = %s.Grid();",tabs,fVar->Name());	
			fprintf(CEqnFile,"\n%sfor( TCell* p = (TCell*)g.first(); g.onGrid(p); g.next(p) ) {",tabs);
//			if( fHasArraysInEqn > 1  ) {
//			  fprintf(CEqnFile,"\n%s\tg.setArrayValues(array_index);",tabs);
//			}	
		break;
		case 1:
			fprintf(CEqnFile,"\n%sDistributedGrid& g = %s.Grid();",tabs,fVar->Name());
			fprintf(CEqnFile,"\n%sint size = g.length();",tabs); 	
			fprintf(CEqnFile,"\n%sTCell* p ;",tabs);	
			fprintf(CEqnFile,"\n#pragma parallel local(p) \n{ ");	
			fprintf(CEqnFile,"\n#pragma pfor ");	
			fprintf(CEqnFile,"\n%sfor( int i=0; i < size; i++ ) {",tabs);	
			fprintf(CEqnFile,"\n%s\tp = (TCell*)g.getPix(i);",tabs);	
		break;
	}
} 

//----------------------------------------------------------------------------------------
//				TCommand::PrintCloseSpaceLoop:
//----------------------------------------------------------------------------------------
  
void TCommand :: PrintCloseSpaceLoop (FILE* CEqnFile, int ntabs ) const {
	char tabs[10]; 
	for( int i=0; i<ntabs; i++ ) { tabs[i] = '\t'; }
	tabs[ntabs] = 0;

	switch( Model::I0().SMP() ) {
		case 0:
			fprintf(CEqnFile,"\n%s}",tabs); 
		break;
		case 1:
			fprintf(CEqnFile,"\n%s}",tabs); 
			fprintf(CEqnFile,"\n} // pragma parallel ");	
		break;
	}
}
//----------------------------------------------------------------------------------------
//				TCommand::Dump:
//----------------------------------------------------------------------------------------
  
void TCommand::Dump( FILE* oFile ){ WriteMML( oFile ); }

//----------------------------------------------------------------------------------------
//				TCommand::AddModifier:
//----------------------------------------------------------------------------------------

int TCommand::AddModifier( const CString& mod, Bool isExtension ) {
  if( SetType(mod,isExtension) ) { 
    EType t = (EType) GetObjInfo(kType);
    if( t == kInit || t == kUpdate || t == kIntegrate || t == kFunction ) SetObjInfo( kExType, kCode );
    return 1; 
  }
  return 0;
}
//----------------------------------------------------------------------------------------
//				TCommand::GetModifierString:
//----------------------------------------------------------------------------------------

int TCommand::GetModifierString( CString& mod ) const {
  mod += kTypeList[ GetObjInfo(kType) ];
  return 1;
}

int TCommand::GetXMLAttributes( CString& mod ) const {
		mod=" event=\"";
    mod += kTypeList[ GetObjInfo(kType) ];
    mod+="\" ";
		mod+=" type=\"code\" ";
  return 1;
}

//----------------------------------------------------------------------------------------
//				TCommand::AddPipeDependencies:
//----------------------------------------------------------------------------------------

int TCommand::AddPipeDependencies( Pipe* p  ) {
	if( p && p->GetObjInfo(Pipe::kType) == Pipe::kDBaseInput ) {
		Variable* v = ((DBaseInputPipe*)p)->GetMap(0);
		if(v) AddDependency( *v );
		v = ((DBaseInputPipe*)p)->GetMap(1);
		if(v) AddDependency( *v );
	} 
	return 0;
}

//----------------------------------------------------------------------------------------
//				TImportCommand::Execute:
//----------------------------------------------------------------------------------------

int TImportCommand::Execute( Module* m ) { 
	TCommand::Execute(m); 
	fVar->RefreshData();
	 return 1; 
}
	
  //----------------------------------------------------------------------------------------
  //				TPipeCommand::
  //----------------------------------------------------------------------------------------

int TPipeCommand::fPipeCmdIndex = 0;

TPipeCommand::~TPipeCommand() { 
	if(fPipe) delete fPipe; 
}


TPipeCommand* TPipeCommand::CreatePipeCommand( TConfigData& cd, Pipe* p, Variable* v ) {
  if( p == NULL ) return NULL;
  if( v->GetCInfo(Variable::kMode) == Variable::kInput ) v = v->Connection();
  v->RegisterPipe( *p );
  CString name(cd.Cmd()); name += "_c__"; name.appendIndex(fPipeCmdIndex++);
  TPipeCommand* pc = new TPipeCommand(name,v,p);
  Pipe::EType ptype = (Pipe::EType) p->GetObjInfo(Pipe::kType);
  if( ptype == Pipe::kSocket ) {
		pc->SetObjInfo(kStatus,kInActive);
	}
	if( ptype == Pipe::kTimeSeriesInput || ptype == Pipe::kPtTimeSeriesInput ) {
		pc->SetTimeSchedule( gDT(), gTime()-gDT() );
	}
	if( Env::GetInfo(Env::kSimOpen) ) {
		Module* m = v->GetModule(); 
		p->Initialize();
		m->PostPipe(*pc);
	}
	return pc;
}

//----------------------------------------------------------------------------------------
//				TPipeCommand::Config:
//----------------------------------------------------------------------------------------

int TPipeCommand::Config( TConfigData & cd ) { 
  const CString& cmd = cd.Cmd();
	int rv = TCommand::Config( cd );
	fPipe->Config( cd );
  return rv;
}

int TPipeCommand::Execute(Module* m) { 
		TCommand::Execute(m); 
		int rv = fPipe->Execute(); 
		return rv;
}

//----------------------------------------------------------------------------------------
//				TExecuteCommand::Execute
//----------------------------------------------------------------------------------------

int TExecuteCommand::Execute( Module* m ) { 
		TCommand::Execute(m); 
		int rv = (*fEF)(fVar); 
		return rv;
}
//----------------------------------------------------------------------------------------
//				TScriptCommand:
//----------------------------------------------------------------------------------------

TScriptCommand::TScriptCommand( const CString& name, Module* m )  
	: TCommand(name,NULL,kUndefined) { SetObjInfo(kExType,kScript); fModule = m; }
TScriptCommand::TScriptCommand( const CString& name, Variable* v )  
	: TCommand(name,v,kUndefined) { SetObjInfo(kExType,kScript); fModule = v->GetModule(); }
TScriptCommand::TScriptCommand( const TScriptCommand& c  )  : TCommand(c) { SetObjInfo(kExType,kScript); fModule = c.fModule; }
int TScriptCommand::Execute( Module* m ) {
	TCommand::Execute(m); 
		int rv = system( getValueString() ); 
	return rv; 
}

//----------------------------------------------------------------------------------------
//				TEvent::TEvent:
//----------------------------------------------------------------------------------------

ArgArray TEvent::kTypeList( 6, " ", "init", "update", "updateF", "integrate", "import" );

EInfoIndex TEvent::kType = kO_Info0;
EInfoIndex TEvent::kSpatial = kO_Info1;

TEvent::TEvent( const char* name, Module* aModule  ) : TNamedObject(name)
{ fModule = aModule; fCurrentCommand = 0; fSchedule = NULL; fEventListPosted = NULL; } 

Pix  TEvent::AddCommand(Variable* v, const CString& val ) {
  TCommand* c = new TCommand( v, val );
  c->Post();
  return fCommandList.append(*c);
}

Pix  TEvent::AddCommand( TCommand &c, ECmdPosition pos   ) {
	if( c.GetObjInfo(TCommand::kStatus) != TCommand::kActive ) return 0;
  Pix p = fCommandList.find( &c );
  if( p ) return NULL;
  if( gDebug > 1 ) { 
		if( c.Var() != NULL ) {
			sprintf(gMsgStr,"$$ Adding Command %s:%s(%p)::%s to Event %s::%s", 
				c.Var()->GetModule()->Name(), c.Var()->Name(), c.Var(), c.Name(), fModule->Name(), Name() ); 
		} else {
			sprintf(gMsgStr,"$$ Adding Command %s to Event %s::%s", c.Name(), fModule->Name(), Name() ); 
		}
    gPrintLog(); 
  }
	c.Post();
  switch(pos) {
  case kEnd: return fCommandList.append(c);
  case kBeginning: return fCommandList.prepend(c);
  }
}

TEvent&  TEvent::operator += ( TEvent& e ) {
  for( Pix p0 = e.fCommandList.first(); p0; e.fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) e.fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) AddCommand(c0);
  }
	return *this;
}

int  TEvent::FastExecute() {
  if( gDebug ) { 
    sprintf(gMsgStr," ******************** Executing Event %s:%s at time %f",
	    fModule->Name(),Name(),gTime()); 
    gPrintLog(); gPrintScreen(); 
  }
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
		if( c0.ExecuteOK( GetObjInfo(kType) ) ) { 
			if( gDebug > 1 ) { 
					Variable* cvar = c0.GetVar();
					Module* cmod = (cvar) ? cvar->GetModule() : fModule;
					CString vname = (cvar) ? cvar->Name() : "script";
					sprintf(gMsgStr,"Executing Command %s::%s:%s", cmod->Name(),vname.chars(),c0.Name()); 
					gPrintLog(); gPrintScreen(); 
			}
			c0.Execute(fModule);
		}
	}
	return 0;
}

int  TEvent::Execute( int breakIndex, int set_context ) {
  if( gDebug ) { 
    sprintf(gMsgStr," ******************** Executing Event %s:%s at time %f",
	    fModule->Name(),Name(),gTime()); 
    gPrintLog(); gPrintScreen(); 
  }
  EType etype = (EType) GetObjInfo(kType);
  for( Pix p0 = (fCurrentCommand) ? fCurrentCommand : fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
		Variable* cvar = c0.GetVar();
		if( fCurrentCommand || c0.ExecuteOK( GetObjInfo(kType) ) ) { 
      fLastExecutedCommand = p0; 
      TCommand::EExType et = (TCommand::EExType) c0.GetObjInfo(TCommand::kExType);
//      cvar->SetF(FisFinalUpdate,( GetObjInfo(kType) == kFinalUpdate ));
			Module* cmod = (cvar) ? cvar->GetModule() : fModule;
      Model::I0().SetContext( this,&c0,cvar,cmod );
      if( ( c0.Break() && (p0 != fCurrentCommand) ) || Model::I0().QueryInterupt() ) { 
				CString msg("Break on "); 
				if(cvar) msg += cvar->SName(); else msg += "script";
				msg += " : "; msg += c0.SName();
				gPrintScreen(msg); fCurrentCommand = p0; return -1; 
      }
      if( gDebug > 1 ) { 			
				CString vname = (cvar) ? cvar->Name() : "script";
				sprintf(gMsgStr,"Executing Command %s::%s:%s at time %f",
					cmod->Name(),vname.chars(),c0.Name(),gTime()); 
				gPrintLog(); gPrintScreen(); 
      }
      if( breakIndex ) { Env::debugger_stop_here(); if(breakIndex < 2) breakIndex = 0; }
      c0.Execute(fModule);
      if( (et == TCommand::kCode) || (et == TCommand::kImport) ) {
//				cvar->UpdateInternals();
				if( c0.GetObjInfo(TCommand::kType) == TCommand::kFunction ) { 
//					cvar->UpdateReserve(); 
					TNamedObjectList& dl = c0.DependencyList();
					for ( Pix p =  dl.first(); p; dl.next(p) ) {
						TDependency& d = (TDependency&) dl(p);
						Variable* depVar = d.GetVar();	
						if( depVar && depVar->GetF(FVisible)) {
							Model::I0().QueryInterface(fModule->Name(),depVar->Name());
						}
					}	
				}
				if( cvar->GetF(FVisible) || (cvar->GetF(FRegistered) &&  Model::I0().MemoryOp()) ) { 
					Model::I0().QueryInterface(); 
				}
			} // else if(et == TCommand::kPipe) { cvar->UpdateConnections(); }
      fCurrentCommand = 0;
//      Env::FlushScreen();
    }
  }
  if( etype == kIntegrate ) { Model::I0().Frame().ReLink(fModule); }
  if( etype <= kInit ) { Clear(); return 1; }
  else return Schedule();
}

int  TEvent::ExecuteNextCommand( int breakIndex, int set_context ) {
	int skipTestCmdOK = 1;
  if( fCurrentCommand == 0 ) {
    fCurrentCommand = fCommandList.first();
    skipTestCmdOK = 0;
    if( gDebug  ) { 
      sprintf(gMsgStr," ******************** Executing Event %s:%s at time %f",
	      fModule->Name(),Name(),gTime()); 
      gPrintLog(); gPrintScreen(); 
    }
  } 	
  if( fCurrentCommand != 0 ) { 
    int go = 1;
    do {
      TCommand &c0 = (TCommand&) fCommandList(fCurrentCommand);
			Variable* cvar = c0.GetVar();
			Module* cmod = (cvar) ? cvar->GetModule() : fModule;
      if( skipTestCmdOK || c0.ExecuteOK( GetObjInfo(kType) ) ) { 
				TCommand::EExType et = (TCommand::EExType) c0.GetObjInfo(TCommand::kExType);
				if( gDebug > 1 ) { 
					CString vname = (cvar) ? cvar->Name() : "script";
					sprintf(gMsgStr,"Executing Command %s::%s:%s at time %f",
						cmod->Name(),vname.chars(),c0.Name(),gTime()); 
					gPrintLog(); gPrintScreen(); 
				}
				fLastExecutedCommand = fCurrentCommand;
				Model::I0().SetContext(this,&c0,cvar,cmod);
				if( breakIndex ) Env::debugger_stop_here();
				c0.Execute(fModule); go = 0;

				if( (et == TCommand::kCode) || (et == TCommand::kImport) ) {  
//					cvar->UpdateInternals();
					if( c0.GetObjInfo(TCommand::kType) == TCommand::kFunction ) { 
//						cvar->UpdateReserve(); 
						TNamedObjectList& dl = c0.DependencyList();
						for ( Pix p =  dl.first(); p; dl.next(p) ) {
							TDependency& d = (TDependency&) dl(p);
							Variable* depVar = d.GetVar();	
							if( depVar && depVar->GetF(FVisible)) {
								Model::I0().QueryInterface(fModule->Name(),depVar->Name());
							}
						}	
					}
					if( cvar->GetF(FVisible) || (cvar->GetF(FRegistered) &&  Model::I0().MemoryOp()) ) {
						Model::I0().QueryInterface();
					} 
				}
			}

			if(  fCommandList.next(fCurrentCommand) == 0 ) {
				if( GetObjInfo(kType) <= kInit ) {
					if(fEventListPosted) Model::I0().RemoveEvent(fEventListPosted);
					return 1;
				}
				else return Schedule();
      }
    } while(go);
  } else if(fEventListPosted) Model::I0().RemoveEvent(fEventListPosted);
  return 1;
}

int  TEvent::Schedule(TEvent::ECmdPosition cp) {
	if( Model::I0().GetCInfo(Model::kOptLevel) >= 5 ) { return 0; }
	if( fSchedule == NULL ) SetSchedule();
	float next_time = fSchedule->StartTime();
	Clear();
	if( GetObjInfo(kType) != kInit ) {
#ifdef DEBUG
		if( gDebug && (fSchedule->DT() == 0.0) ) { 
			sprintf(gMsgStr,"Attempting to reschedule event %s with timstep = 0.0",Name()); 
			gPrintErr(); 
		}
#endif		
		if( fSchedule->ReSchedule( gTime(), next_time ) == 0 ) { return 0; }
	}
	SetTime( next_time );
	fEventListPosted = Model::I0().PostEvent(*this,next_time,cp);
	return 1;
}

int TEvent::ProcessAttribute(const TAttributeNode* n) { return 0; }

int TEvent::ProcessCommand(const TCommandNode* cn){
	const TDeclarationNode* dn = cn->Declaration();
	const CString& cmd = dn->Command();
	const CString& name = (dn->Name()) ? dn->Name()->Top() : gNullString;
	int isExtension = dn->IsExtension();

  if( cmd == "Command" ) {
  }
  else if( cmd == "Schedule" ) {
    if( fSchedule == NULL ) { fSchedule = new ExSchedule(name); }
		fSchedule->AddQualifiers(dn,isExtension);   
    fSchedule->ProcessGeneral(cn->StatementList());
  }
  else if( cmd == "Equations" ) {
  } else return 0;
  return 1;
}

int  TEvent::SetSchedule ( ExSchedule* s  ) {
	int force_reschedule = 1;	
  if( s == NULL ) { s = fModule->GetSchedule(); force_reschedule = 0;	}
  if( s ) {
    if( fSchedule == NULL ) { fSchedule = s; return 1; }
    if( !fSchedule->ScheduleConfigured() || force_reschedule ) { (*fSchedule) = (*s); return 1; }
  }
  return 0;
}

void TEvent::Clear() {
		if(fEventListPosted) Model::I0().RemoveEvent(fEventListPosted);
		fEventListPosted = 0;
}

void TEvent::SetCurrentContext(Module*& currentModule, Variable*& currentVariable, TCommand*& currentCommand) {
	currentCommand = (TCommand*) ( (fLastExecutedCommand) ?  &fCommandList(fLastExecutedCommand) : NULL );
	currentVariable = (currentCommand) ? currentCommand->GetVar() : (Variable*)NULL;
	currentModule = (currentVariable) ? currentVariable->GetModule() : (Module*)NULL;	
}

void TEvent::SetStatus ( TCommand::EType type, TCommand::EStatus status ) {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( (type == TCommand::kUndefined) || (type == c0.GetObjInfo(TCommand::kType)) ) {
      c0.SetObjInfo( TCommand::kStatus, status  );
    }
  }
}

void TEvent::RemoveInActiveCommands ( Bool free ) {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kInActive ) {
			c0.UnPost();
			fCommandList.del(p0);
			if( free && c0.NPost() == 0 ) c0.GetVar()->DeleteCommand( c0.SName() );
    }
  }
}

TEvent&  TEvent::Merge ( TEvent& e, Bool deepCopy, Bool allowDuplNames, Bool removeOld ) {
  for( Pix p0 = e.fCommandList.first(); p0; e.fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) e.fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) {
      TCommand* cptr = (deepCopy) ? new TCommand(c0) : &c0;
      if(allowDuplNames) fCommandList.append(*cptr);
      if(removeOld) fCommandList.del(p0);  
      else AddCommand(*cptr);
    }
  }
  return *this;
}

void TEvent::SetStatus( TCommand::EStatus  s, Variable* var ) {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( var == NULL || c0.Var() == var ) c0.SetObjInfo(TCommand::kStatus,s);
  }
}

void  TEvent::WriteMML( FILE* oFile, EWriteMode mode )  {
  CString mod;
  GetModifierString(mod);
  fprintf( oFile, "\t%s Event %s {\n",(const char*)mod,Name());
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    TCommand& c = (TCommand&) fCommandList(p);
    c.WriteMML(oFile);
  }
  fprintf( oFile, "\t}\n" );
}

void  TEvent::Dump( FILE* oFile ){
  CString mod;
  GetModifierString(mod);
  fprintf( oFile, "\n\n\t%s Event %s {\n",(const char*)mod,Name());
  PrintDependencies( oFile, NULL, False );
  fprintf( oFile, "\t}\n" );
}

Pix TEvent::FindVarByName( char* name ) const {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    const TCommand &c0 = (TCommand&) fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) { 
      if( c0.VarName() == name ) return p0;
    }
  }
  return (Pix)0;
}

Pix TEvent::FindVar( Variable* var, Pix offSet, int verbose ) const {
	Pix p0 = (offSet) ? offSet : fCommandList.first();
	if(offSet) { fCommandList.next(p0); }
  for( ; p0; fCommandList.next(p0) ) {
    const TCommand &c0 = (TCommand&) fCommandList(p0);
    const Variable* v = c0.VarC();
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) { 
      if( v == var ) return p0;
    }
  }
  return (Pix)0;
}


void TEvent::CreateDependencies( TNamedObjectList& varList ) {
	EType etype = (EType) GetObjInfo(kType);
	int deactivateStateVarDepencies = 1;
	if( (etype == kFinalUpdate) || (etype == kInit)  ) { deactivateStateVarDepencies = 0; } 
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) { 
      TNamedObjectList& depList = c0.DependencyList();
	  Variable* v0 = c0.GetVar();
      for( Pix pv = depList.first(); pv; depList.next(pv) ) {
		TDependency& d = (TDependency&) depList(pv);
		Variable* depVar = d.GetVar();
		if( depVar ) {
			int verbose = 0;
			Pix p1 = 0;
			while (1) {
				p1 =  FindVar( depVar, p1, verbose );
				if( p1 == NULL ) { break; }
				else {
					TCommand &c1 = (TCommand&) fCommandList(p1);
					if( p0 != p1  && c1.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) {
						int isIntegrationCommand = (c1.GetObjInfo(TCommand::kType) == TCommand::kIntegrate);
						int activeDependency = 1; /* (d.RelCode() == NULL); */
						if( deactivateStateVarDepencies && isIntegrationCommand ) { activeDependency = 0; }
						int index = CreateDependency(p0,p1,activeDependency);

						if( gDebug > 2 ) {
							TOrderedObjectDLListNode* n0 = (TOrderedObjectDLListNode*) p0;
							sprintf(gMsgStr,"Dep Link(%s:%s): %s(%s) (%d)-> %s(%s,%x) (%x), active: %d \n",
								v0->GetModule()->Name(), Name(), v0->Name(), c0.Name(), index, depVar->Name(), 
								c1.Name(), p1, n0->plink(index), activeDependency );
							gPrintLog();
						}
					}
				}
			}
		}
      }
    }
  }
}

void TEvent::PrintDependencies( FILE* oFile, Variable* var, int dump ) const {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    Variable* v0 = c0.GetVar();
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive && ( var==NULL || var == v0 ) ) {
      if ( dump ) c0.Dump( oFile );
      fprintf( oFile, "\n\t\t\tDep( List %s ) *************    %s:   ( ",Name(),v0->Name());
      TNamedObjectList& depList = c0.DependencyList();
      for( Pix pv = depList.first(); pv; depList.next(pv) ) {
				char tc = ' ';
				TDependency& d = (TDependency&) depList(pv);
				Variable* depVar = d.GetVar();
				if( depVar ) {
					if( depVar->GetCInfo(Variable::kMode) == Variable::kInput ) tc = '&';
					else if( FindVar( depVar ) ) tc = '*';
					fprintf( oFile, " %c%s ",tc,depVar->Name());
				}
      }
      fprintf( oFile, " )\n" );
    } 
  }
}

void TEvent::AddDependencies( TEvent& e, Bool inactivate_additions ) {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) { 
      for( Pix p1 = e.fCommandList.first(); p1; e.fCommandList.next(p1) ) {
		TCommand &c1 = (TCommand&) e.fCommandList(p1);
		if( c0 != c1 && c1.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) { 
			if( c0.getValueString(0).containsName( c1.VarName() )  ) {
				Pix p2 = AddCommand( c1 );
				CreateDependency(p0,p2);    //  p0 is dependent on p2;
				if(inactivate_additions) c1.SetObjInfo(TCommand::kStatus,TCommand::kInActive);  
			}
		}
      }
    }
  }
}

void TEvent::AddFluxDependencies() {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) { 
	  Variable* v0 = c0.Var();
	  if( v0->GetCInfo(Variable::kType) == Variable::kFlux ) {
		  Variable* sv0 = v0->Origin();
		  if( sv0 != NULL ) {
			for( Pix p1 = fCommandList.first(); p1; fCommandList.next(p1) ) {
			  TCommand &c1 = (TCommand&) fCommandList(p1);
			  if( c1.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) { 
				Variable* v1 = c1.Var();
				if( v1->GetCInfo(Variable::kType) == Variable::kFlux ) {
					Variable* sv1 = v1->Destination();
					if( sv1 == sv0 ) {
					  int link_index = CreateDependency(p0,p1,True);    //  p0 is dependent on p1;
					  if( gDebug ) {
						CString sname; sv0->FullName(sname);
						gPrintLog( format("%s:: Adding Flux Dependency(%d) for %s: %s -> %s", Name(), link_index, sname.chars(), v0->Name(), v1->Name()) );
					  }
					  int sortlen = IsSortable();
					  if( sortlen == 0 ) {											
						DeactivateDependency(p0,link_index);  // DeActivate dependency if not sortable.
						if( gDebug ) { gPrintLog( format("Deactivating Flux Dependency: %s -> %s ",v0->Name(),v1->Name()) ); }
					  } else {
						c0.AddDependency( *v1 );
						if( gDebug ) { gPrintLog( format("Adding Command dep: %s:%s -> %s", v0->Name(), c0.Name(), v1->Name() ) ); }
					  }
					}
				}
			  } 
			}
		  }
	  }
	}
  }
}																																																																																																											 // For each command c in this (event) list:
TCommand* TEvent::SplitByDependency( TEvent& ei, TEvent& ed, enum ESplitConsequence sc, int localVarsOnly ) {   // move ( copy ) command c from this list to ei
	int skipStateVars = (ei.GetObjInfo(TEvent::kType) == TEvent::kInit);
  for( Pix pi = ed.fCommandList.first(); pi; ed.fCommandList.next(pi) ) {                               // if there exists command(s) in ed that depend on c.
		TCommand &ci = (TCommand&) ed.fCommandList(pi);
    if( ci.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) {
      TNamedObjectList& depList = ci.DependencyList();
      for( Pix pv = depList.first(); pv; depList.next(pv) ) {
				TDependency& d = (TDependency&) depList(pv);
				Variable* depVar = d.GetVar();
				if( depVar  ) {
					if( (depVar->GetModule() == GetModule()) || !localVarsOnly  ) {
						Pix p0 = FindVar( depVar );
						if( gDebug > 1 ) {
							sprintf(gMsgStr,"Checking Dep(skipSV:%d): (%s:%s)->%s, %s -> %s : %x : ",
								skipStateVars, ei.Name(), ci.Name(), Name(), ci.Var()->Name(),depVar->Name(),p0);
							gPrintLog();
						}
						if( p0 ) 	{
							const TCommand &cd = (TCommand&) fCommandList(p0);
							int isIntegrationCommand = (cd.GetObjInfo(TCommand::kType) == TCommand::kIntegrate);
							if( skipStateVars && isIntegrationCommand ) {
								if( gDebug > 1 ) { sprintf(gMsgStr," >> cmd %s: skip Integrator ", cd.Name() ); gPrintLog(); 	}
							} else {
								Pix rp = MoveCmd(p0,ei,sc);
								if( rp ) { return (TCommand*) &ei.fCommandList(rp); }
								if( gDebug > 1 ) { sprintf(gMsgStr," >> cmd %s: move ", cd.Name() ); gPrintLog(); 	}
							}
						}
					}
				}
			}
    }
  }
  return (TCommand*) NULL;
}

Pix TEvent::MoveCmd( Pix p0, TEvent& ei, enum ESplitConsequence sc ) {
  Pix rp = 0;
  TCommand &c0 = (TCommand&) fCommandList(p0);
  if( rp = ei.AddCommand(c0) ) {
    if( gDebug > 1 ) {
      sprintf(gMsgStr,"Add cmd: %s -> Event %s\n",c0.Name(),ei.Name());
      gPrintLog();
    }
	} else if(gDebug) {
//		Variable* v = c0.Var();
//		Module* m = v->GetModule();
//		sprintf(gMsgStr,"Warning, Command %s (%s) already exists in List %s:%s.",c0.Name(),v->Name(),m->Name(),ei.Name());
//		gPrintErr();
	}
	switch (sc) { 
	case kDoNothing: break;
	case kDelete: 
		fCommandList.Delete(p0);
		break;
	case kDeActivate: 
		c0.SetObjInfo(TCommand::kStatus,TCommand::kInActive); 
		break;
	}
  return rp;
}

void TEvent::Split( TEvent& e, int splitIndex, enum ESplitConsequence sc ) {
startSplit:
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) {
      Variable* v0 = c0.GetVar();
      if( v0 && v0->SplitCondition( splitIndex ) ) {
				MoveCmd(p0,e,sc);
				goto startSplit;
      }
    }
  }
}

void TEvent::SplitSpatially( TEvent& e, enum ESplitConsequence sc ) {
startSplit:
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) {
      Variable* v0 = c0.GetVar();
      if( v0 && v0->GetF(FisSpatial) ) {
	MoveCmd(p0,e,sc);
	goto startSplit;
      }
    }
  }
}

int TEvent::CreateDependency( Pix p0, Pix p1, Bool mark_link ) {   //  p0 is dependent on p1;
  TOrderedObjectDLListNode* n0 = (TOrderedObjectDLListNode*) p0;
  TOrderedObjectDLListNode* n1 = (TOrderedObjectDLListNode*) p1;
  int index = n0->slink(n1);
  if( mark_link ) n0->set_info(index);
  return index;
}

int TEvent::DeactivateDependency( Pix p0, int link_index ) {   
  TOrderedObjectDLListNode* n0 = (TOrderedObjectDLListNode*) p0;
  n0->clear_info(link_index);
  return link_index;
}

void  TEvent::WriteCCode( FILE* CHdrFile, FILE* CEqnFile) {
  switch( GetObjInfo(kType) ) {
  case kInit:
    fprintf(CEqnFile,"\n\tTEvent* %s = GetEvent(\"%s\",TEvent::kInit);",Name(),Name()); break;
  case kFirstUpdate:
    fprintf(CEqnFile,"\n\tTEvent* %s = GetEvent(\"%s\",TEvent::kFirstUpdate);",Name(),Name()); break;
  case kFinalUpdate:
    fprintf(CEqnFile,"\n\tTEvent* %s = GetEvent(\"%s\",TEvent::kFinalUpdate);",Name(),Name()); break;
  case kIntegrate:
    fprintf(CEqnFile,"\n\tTEvent* %s = GetEvent(\"%s\",TEvent::kIntegrate);",Name(),Name()); break;
  case kImport:
    fprintf(CEqnFile,"\n\tTEvent* %s = GetEvent(\"%s\",TEvent::kImport);",Name(),Name()); break;
  case kUnknown:
    fprintf(CEqnFile,"\n\tTEvent* %s = GetEvent(\"%s\",TEvent::kUnknown);",Name(),Name()); break;
  }
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand& c0 = (TCommand&) fCommandList(p0);
    if( c0.GetObjInfo(TCommand::kStatus) == TCommand::kActive  ) {
			Variable* var = c0.GetVar();
      const Module* mod = var->GetModule(); 
      if( fModule == mod )
				fprintf(CEqnFile,"\n\t%s->AddCommand(*%s);          // %s ",Name(), c0.Name(), var->Name() );
      else {
				fprintf(CEqnFile,"\n\t{ TCommand* c = %s::I().GetCommand(\"%s\");", mod->Name(), c0.Name() );
				fprintf(CEqnFile,"\n\t  if(c) %s->AddCommand( *c ); else gFatal(\"Can't Find Command\"); }  // %s ", Name(), var->Name() );
      }
    }
  }
}
