//----------------------------------------------------------------------------------------
//      Module.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History
//----------------------------------------------------------------------------------------

#ifndef __MML_Module__
#define __MML_Module__
  
#include "Utilities.h"
#include "SList.h"
#include "StringPlex.h"
#include "ConfigObjects.h"
#include "CStream.h"
#include "MML.h"
#include "MML_Frame.h"
#include "MML_Event.h"

typedef class TMethod TMethod;
typedef class TLUT TLUT;
typedef class TConnectionList TConnectionList;
typedef class TConnection TConnection;
typedef class Module Module;
typedef class Model Model;
typedef class Variable Variable;
typedef class TModuleNode TModuleNode;
typedef class TNode TNode;

const int kSplitConstant = 0;
const int kSplitSpatial = 1;


//----------------------------------------------------------------------------------------
//				Module
//----------------------------------------------------------------------------------------

class Module : public TConfigObject, public MMLObject { 

public:

  enum EObjectType { kNone, kConnection, kLUT, kMethod, kEvent };
  enum EUListStatus { kULNone, kULFirstOnly, kULFinalOnly, kULBoth };

  const int kNewEvent;
  Module *fSuper;
  static int fCount;
  int fScriptIndex;

protected:

    TNamedObjectList  fModuleList;
    TNamedObjectList fVarList;
    TNamedObjectList fCommandList;
    Variable* fCurrentVariable;
    TNamedObjectList fLUTList;
    TNamedObjectList fMethodList;
    TNamedObjectList fEventList;
    TNamedObjectList fScheduleList;
    TConnectionList* fConnectionList;
    LayerConfig* fLayerConfig;
    ExSchedule fSchedule;
    TModuleNode* fModuleNode;
    Model* fModel;
    CString fComment;

    void WriteInstantiator (FILE* CHdrFile) const;
    EUListStatus CheckUpdateListInclusion(TCommand& c);

    int fActivationLayer;
    int fLinkLayer;
  
public:
  static EConfigInfo kDebug, kIgnorable;
		
  Module( CString name  );
  virtual int Config( TConfigData & cd );

  Variable* CurrentVar(char* varName) { return fCurrentVariable = GetVariable(varName); }
  Variable* CurrentVar(Variable* var=NULL) { return (var) ? fCurrentVariable = var : fCurrentVariable; }
  TNamedObjectList& MethodList() { return fMethodList; }
  TNamedObjectList& LUTList() { return fLUTList; }
  TNamedObjectList& VariableList() { return fVarList; }
  TNamedObjectList& VarList() { return fVarList; }
  TNamedObjectList& EventList() { return fEventList; }
  inline Module* Super() { return fSuper; }

  TMethod* GetMethod( const char* name, Bool create = True );
  TLUT* GetLUT( const char* name = NULL, Bool create = True );
	inline Model* GetModel() { return fModel; }
	inline void SetModel(Model* m) { fModel = m; }
  TConnection* GetConnection( const char* name = NULL, Bool create = True  );
  TConnection* FindConnection( const char* name, Bool create = True  );
  int RemoveConnection( TConnection* c );
  TConnectionList* ConnectionList() { return fConnectionList; }
  virtual TEvent* GetEvent( CString cName, TEvent::EType type = TEvent::kUnknown, Bool create = True );
  virtual ExSchedule* GetSchedule( const CString& cName, Bool create = True );
  virtual Variable* GetVariable( const CString& name, Bool create = True );
  inline TCommand* GetCommand( const char* name ) { return (TCommand*) fCommandList.GetObjectByName(name); }
  Pix AddCommand( TCommand* c );
  Bool ProcessSpatialDependencies ();
  Bool ProcessTemporalDependencies ();
  void SplitListsSpatially();
  void DeleteEmptyEvents();
  void SortLists();
  void ProcessConnections ();
  void CreateEventDependencies(void);
  virtual int InitArrays() { return 0;}


  
	int RunNodeProcessing( EProcessContext pc );
	virtual int ProcessGeneral( TNode* n);
	virtual int ProcessCommand(const TCommandNode* n);
	virtual int ProcessAttribute(const TAttributeNode* n);

  virtual Variable* CreateNewVariable( const CString& name, Variable* template_var = NULL );
  Variable* CreateInputVariable( Variable* remoteVariable, char* cname = "c"  );
  virtual Module* AddSubModule(const char* module_name, Bool create = True );
  Module* AddSubModule( Module* m );
  Pix AddVariable( Variable* v );
  
  virtual int AddModifier( const CString& mod, Bool isExtension = False );
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void WriteXML( CString& xmlPath );
  virtual void WriteConfigXML( CString& xmlPath );
  int BuildConfigXML( ArgArray& msg, int mode );
  virtual void WriteDataToConfig( CStream& outStream ); 
  virtual void Dump( FILE* oFile );
  virtual void WriteCCode( FILE* CHdrFile, FILE* CEqnFile);
  void  PostEventsByType( TEvent::EType type );
  ExSchedule* GetSchedule(Bool getParent = False);
  Pix PostPipe( TPipeCommand& c  );
  TNamedObjectList* PostCommandsFromConfig();
  void ConfigureSchedules();
  virtual int PostImport() { return 0; }
  virtual int PostFirstUpdate() { return 0; }
  virtual int PostFinalUpdate() { return 0; }
  virtual int PostIntegrate() { return 0; }
  virtual int PostInit() { return 0; }
  virtual void WriteModuleCode(FILE* CHdrFile, FILE* CEqnFile) const {;}
  inline int GetScriptIndex() { return  fScriptIndex++; }
  inline void SetModuleNode( TModuleNode* m ) { fModuleNode = m; }
	virtual inline int ConfigExternal(TConfigData& cd, Module* m) { return 0; }
	
  inline float dT() { return fSchedule.DT(); }
  
	inline void VariableListDump(FILE* oFile=NULL) {
		FILE* f = (oFile) ? oFile : gLogFile;
		fprintf(f,"\nVariables:\n");
		fVarList.Dump(f);
	}
	int Finalize();
  void CreateEventLists( TEvent* initEvent );
  int FillInitializationList( TEvent* initEvent );
  inline int activationLayerIndex() { return fActivationLayer; }
  inline int linkLayerIndex() { return fLinkLayer; }
  inline void SetLayerConfig( LayerConfig* lc ) {
	if( lc != NULL ) { 
		int index = lc->AddModule(this);
		if( gDebug ) { sprintf( gMsgStr, "Adding module %s to LayerConfig %p: index %d",Name(),lc,index); gPrintLog(); }
		fLayerConfig = lc; 
	}
  }
  inline LayerConfig* getLayerConfig() { return fLayerConfig; }
};

#endif
