#include "MML_Pipe.h"
#include "MML_Variable.h"
#include "MML_Event.h"
#include "MML_Model.h"
#include "sme_postgres.h"

//----------------------------------------------------------------------------------------
//						Pipe
//----------------------------------------------------------------------------------------

int Pipe::fPipeIndex = 0;
EInfoIndex Pipe::kType = kO_Info0;
EInfoIndex Pipe::kMode = kO_Info1;

ArgArray Pipe::kTypeList( 12, " ", "anim", "graph", "sheet", "map", "window", "ptSeries", "condense", "timeSeriesInput", "ptTimeSeriesInput", "dBaseInput", "mapInput", "socket", "display" );

Pipe::Pipe( const char* name, EType type, TConfigObject* var  ) : TNamedObject(name)  { 
  SetObjInfo(kType,type); 
  fConfigObject = var; 
  fCommand = NULL;
}

int Pipe::Config( TConfigData & cd ) { 
  return 0; 
}

Pipe* Pipe::GetPipeFromConfig( TConfigData& cd, TConfigObject* cobj, TPipeCommand*& pc  ) {
  const CString& cmd = cd.Cmd();
  Variable* v = (Variable*)cobj; 
  if( v->GetCInfo(Variable::kMode) == Variable::kInput ) { 
		v = v->Connection(); 
		gPrintErr( v->SName() + ": Pipe moved to Origin of Input."); 
	}
  Pipe* p = NULL;
  int iarg; float farg; 
  if( AnimPipe::Cmd(cmd) ) {
		if( pc = v->ProcessPipes(kAnim,Variable::kReturn) ) { p = pc->GetPipe(); }
		else {
			CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
			p = new AnimPipe(name,v);
		}
  } else  if( DataDisplayPipe::Cmd(cmd) ) {
		if( pc = v->ProcessPipes(kDataDisplay,Variable::kReturn) ) { p = pc->GetPipe(); }
		else {
			CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
			p = new DataDisplayPipe(name,v);
		}
		v->SetF(FisDisplayVar,True,"Pipe::GetPipeFromConfig");
  } else if( SocketPipe::Cmd(cmd) ) {
		if( pc = v->ProcessPipes(kSocket,Variable::kReturn) ) { p = pc->GetPipe(); }
		else {
			CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
			p = new SocketPipe(name,v);
		}
  } else if( GraphPipe::Cmd(cmd) ) {
    CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
    p = new GraphPipe(name,v);
  } else if( SheetPipe::Cmd(cmd) ) {
     CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
     p = new SheetPipe(name,v);
  } else if( MapArchivePipe::Cmd(cmd) ) {
    CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
    p = new MapArchivePipe(name,v);
  } else if( PtSeriesPipe::Cmd(cmd) ) {
    CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
    p = new PtSeriesPipe(name,v);
  } else if( CondensePipe::Cmd(cmd) ) {
    CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
    p = new CondensePipe(name,v);
//  } else if( WindowPipe::Cmd(cmd) ) {
//    CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
//    p = new WindowPipe(name,v);
  }  else if( DBaseInputPipe::Cmd(cmd)  ) {
		if( pc = v->ProcessPipes(kDBaseInput,Variable::kReturn) ) { p = pc->GetPipe(); }
		else {
			CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
			p = new DBaseInputPipe(name,v);
		}
  } else if( TimeSeriesInputPipe::Cmd(cmd) ) {
		if( pc = v->ProcessPipes(kTimeSeriesInput,Variable::kReturn) ) { p = pc->GetPipe(); }
		else {
			CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
			p = new TimeSeriesInputPipe(name,v);
		}
  } else if( PtTimeSeriesInputPipe::Cmd(cmd) ) {
		if( pc = v->ProcessPipes(kPtTimeSeriesInput,Variable::kReturn) ) { p = pc->GetPipe(); }
		else {
			CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
			p = new PtTimeSeriesInputPipe(name,v);
		}
  }  else if( MapInputPipe::Cmd(cmd) ) { 
//	 fprintf(stderr,"\n  ** dbg1: process pipe %s: cmd: %s \n", v->Name(), cmd.chars() ); fflush(stderr);
	 if( pc = v->ProcessPipes(kMapInput,Variable::kReturn) ) { p = pc->GetPipe(); }
		else {
			CString name(cmd);  name += "_p_"; name.appendIndex(fPipeIndex++);
			p = new MapInputPipe(name,v);
		}
  } else return NULL;
  p->Config(cd);
  return p;
}

//----------------------------------------------------------------------------------------
//						AnimPipe
//----------------------------------------------------------------------------------------

EInfoIndex AnimPipe::kADataType = kO_Info2;

AnimPipe::AnimPipe(const char* name, Variable* var) : Pipe(name, kAnim, (TConfigObject*)var ) {
  fScale = 1.0;
  fOffset = 0.0;
  SetObjInfo(kADataType,1);
  SetObjInfo(kMode,kDisplay); 
}

Bool AnimPipe::Cmd(const CString& cmd) { return cmd == "A"; }

int  AnimPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  int iarg=1;
  if( cmd == "A") { 
    if( cd.IntArg(0,iarg) > 0 ) {
//     SetObjInfo(kADataType,iarg);
			 SetObjInfo(kADataType,2);
    } else {
      SetObjInfo(kADataType,2);
		}
    
    if(gDebug) { 
      sprintf(gMsgStr,"\n%s Output Config: A(%d)",
			       Name(),GetObjInfo(kADataType)); 
      gPrintLog(); 
    }
    return 1;
					
  } else return Pipe::Config(cd);
}

void AnimPipe::WriteDataToConfig( CStream& outStream ) const { 
  outStream << " A( " << GetObjInfo(kADataType) << ")" ; 
}

//----------------------------------------------------------------------------------------
//						DataDisplayPipe
//----------------------------------------------------------------------------------------

EInfoIndex DataDisplayPipe::kADataType = kO_Info2;

DataDisplayPipe::DataDisplayPipe(const char* name, Variable* var) : Pipe(name, kAnim, (TConfigObject*)var ) {
  SetObjInfo(kADataType,1);
  SetObjInfo(kMode,kDisplay); 
}

Bool DataDisplayPipe::Cmd(const CString& cmd) { return cmd == "DD"; }

int  DataDisplayPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  int iarg=1;
  if( cmd == "DD") { 
      
    if(gDebug) { 
      sprintf(gMsgStr,"\n%s Output Config: DD()",  Name()); 
      gPrintLog(); 
    }
    return 1;
					
  } else return Pipe::Config(cd);
}

void DataDisplayPipe::WriteDataToConfig( CStream& outStream ) const { 
  outStream << " DD() "; 
}

//----------------------------------------------------------------------------------------
//						SocketPipe
//----------------------------------------------------------------------------------------

SocketPipe::SocketPipe( const char* name, Variable* var) :  kNBytes(kO_Info2),
	Pipe(name, kSocket, (TConfigObject*)var ) {
  fScale = 1.0;
  fOffset = 0.0;
  SetObjInfo(kNBytes,1);
  SetObjInfo(kMode,kDisplay); 
}

Bool SocketPipe::Cmd(const CString& cmd) { return cmd == "SP"; }


int  SocketPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  int iarg=1;
  if( cmd == "SP") { 
    if( cd.IntArg(0,iarg) > 0 ) {
      SetObjInfo(kNBytes,iarg);
      cd.FloatArg(1,fScale);
      cd.FloatArg(2,fOffset);
    }
    
    if(gDebug) { 
      sprintf(gMsgStr,"\n%s Output Config: SP(%d,%f,%f)",
			       Name(),GetObjInfo(kNBytes),fScale,fOffset); 
      gPrintLog(); 
    }
    return 1;
					
  } else return Pipe::Config(cd);
}

void SocketPipe::WriteDataToConfig( CStream& outStream ) const { 
  outStream << " SP(" << GetObjInfo(kNBytes) << "," << fScale << "," << fOffset << ")"; 
}

//----------------------------------------------------------------------------------------
//						TPointPipe
//----------------------------------------------------------------------------------------

PointPipe::PointPipe( const char* name, EType type,  Variable* var, Point2& point ) :  kFormat(kO_Info2),
	Pipe(name,type,(TConfigObject*)var), fPoint(point) { 
  fCell=0; fCellLayer = -1;
}

PointPipe::PointPipe( const char* name, EType type,  Variable* var ) : kFormat(kO_Info2), Pipe( name, type, (TConfigObject*)var )  { 
  fPoint = Point2(-1,-1); 
  fCell=0; fCellLayer = -1; 
}

//----------------------------------------------------------------------------------------
//						TGraphPipe
//----------------------------------------------------------------------------------------

GraphPipe::GraphPipe( const char* name, Variable* var, Point2& point ) : PointPipe( name, kGraph, var, point ) {
  fIndex = 0;
  fMag = 1.5;
  fDomain[0] = 0; 
  fDomain[1] = 365; 
  SetObjInfo(kMode,kDisplay); 
}

GraphPipe::GraphPipe( const char* name, Variable* var ) : PointPipe( name, kGraph, var ) {
  fIndex = 0;
  fMag = 1.5;
  fDomain[0] = fDomain[1] = 0;
  SetObjInfo(kMode,kDisplay); 
}

int  GraphPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd(); int index;
  int ti=0, tf=0;
  if(cmd[0] ==  'G') {
    cd.IntArg(0,fPoint.elem(0),CD::kRequired);
    cd.IntArg(1,fPoint.elem(1),CD::kRequired);
    cd.FloatArg(2,fRange[0],CD::kRequired);
    cd.FloatArg(3,fRange[1],CD::kRequired);
    cd.IntArg(4,fDomain[0]);
    cd.IntArg(5,fDomain[1]);
    int index;
    if( cd.IntArg(6,index) > 0 ) { fIndex = index; }
    if(gDebug) { sprintf(gMsgStr,"\n%s Output Config: G(). Range: (%f,%f) Domain: (%i,%i) // Point: (%d,%d)",
		        Name(),fRange[0],fRange[1],fDomain[0],fDomain[1],fPoint(0),fPoint(1)); gPrintLog(); }

    return 1;
  } else return Pipe::Config(cd);
}

void GraphPipe::WriteDataToConfig( CStream& outStream ) const { 
   outStream << " G( " << fPoint(0) << "," << fPoint(1) << ",";
   outStream << fRange[0] << "," << fRange[1] << ",";
   outStream << fDomain[0] << "," << fDomain[1] << "," << fIndex << ")"; 
}

Bool GraphPipe::Cmd(const CString& cmd) { return cmd == "G"; }

//----------------------------------------------------------------------------------------
//						SheetPipe
//----------------------------------------------------------------------------------------

int SheetPipe::fSheetCount = 0;

SheetPipe::SheetPipe(const char* name, Variable* var) : PointPipe( name, kSheet, var ) {
  fIndex = 0;
  SetObjInfo(kMode,kDisplay); 
}

int SheetPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  if( cmd == "GS" ) { 
    cd.IntArg(1,fPoint.elem(0),CD::kRequired);
    cd.IntArg(2,fPoint.elem(1),CD::kRequired);

    if( gDebug) { sprintf(gMsgStr,"\n%s Output Config: Sheet(): Point: (%d,%d)",
		        Name(),fPoint(0),fPoint(1)); gPrintDebug(); }

    return 1;
  } else return Pipe::Config(cd);
}

void SheetPipe::WriteDataToConfig( CStream& outStream ) const { 
   outStream << " GS( " << fPoint(0) <<"," << fPoint(1) << ")"; 
}

Bool SheetPipe::Cmd(const CString& cmd)  { return cmd == "GS"; }

//----------------------------------------------------------------------------------------
//						TMapArchivePipe
//----------------------------------------------------------------------------------------

EInfoIndex MapArchivePipe::kMapType = kO_Info2;
EInfoIndex MapArchivePipe::kPrecision = kO_Info3;

MapArchivePipe::MapArchivePipe(const char* name, Variable* var) : Pipe( name, kMapArchive, (TConfigObject*)var )
{
  fScale = -1.0;
  fOffset = 0.0;
  fCellsize = 0.0;
  SetObjInfo(kMode,kArchive); 
}

int MapArchivePipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  CString* s;
  int iarg; 
  if( cmd == "M" ) { 
    int precision = 1;
    if( s = cd.Arg(0) ) { 
      switch( (*s)[0] ) {
      case 'H': SetObjInfo(kMapType,kHDF); break;
      case 'h': SetObjInfo(kMapType,kHDF5); break;
      case 'S': SetObjInfo(kMapType,kSDS); break;
      case 'M': SetObjInfo(kMapType,kMap2); break;
      case 'B': SetObjInfo(kMapType,kBinary); break;
      case 'G': SetObjInfo(kMapType,kGRASS); break;
      case 'd': SetObjInfo(kMapType,kGeometry); break;
      case 'A': SetObjInfo(kMapType,kArc); break;
      case 'a': SetObjInfo(kMapType,kASCII); break;
      case 'C': SetObjInfo(kMapType,kCME); break;
      case 'i': SetObjInfo(kMapType,kIOex); break;
      case 'P': SetObjInfo(kMapType,kPPM); break;
      default:  SetObjInfo(kMapType,kUnknown); break;
      }
    }
    if( cd.IntArg(1,iarg,CD::kRequired) > 0 ) {
	  SetObjInfo(kPrecision,iarg);
	}
	EMapType mt = (EMapType) GetObjInfo(kMapType);
    if( mt == kGRASS ) {
	  SetObjInfo(kPrecision,0);
    } else if( mt == kPPM ) {
	  SetObjInfo(kPrecision,3);
    } else if( mt == kASCII ) {
	  SetObjInfo(kPrecision,0);
	}
    if( s = cd.Arg(2) ) fMapName = (*s);
    cd.FloatArg(3,fScale);
    cd.FloatArg(4,fOffset);
    cd.FloatArg(5,fCellsize);
#ifdef POSTGRESS
		if( mt == kCME ) { 
			CString filename(fMapName); filename+="_Map.sds";
			pgObjRec::AddDBaseObject( "Archive", fMapName, filename, pgObjRec::kSMEOutput ); 
		}
#endif
  } else return Pipe::Config(cd);
  return 1;
}

void MapArchivePipe::WriteDataToConfig( CStream& outStream ) const {
  char ch;
  switch( GetObjInfo(kMapType) ) {
    case kHDF: ch = 'H'; break;
    case kHDF5: ch = 'h'; break;
    case kMap2: ch = 'M'; break;
    case kGRASS: ch = 'G'; break;
    case kArc: ch = 'A'; break;
    case kASCII: ch = 'a'; break;
	case kCME: ch = 'C'; break;
	case kGeometry: ch = 'd'; break;
    case kSDS: ch = 'S'; break;
    case kBinary: ch = 'B'; break;
    case kIOex: ch = 'i'; break;
    case kPPM: ch = 'P'; break;
  } 
  if( fScale > 0.0 ) {
	outStream <<  "M(" << ch << "," << GetObjInfo(kPrecision)<< "," << fMapName.chars() << "," << fScale << "," << fOffset<< "," << fCellsize << ")"; 
  } else {
	outStream <<  "M(" << ch << "," << GetObjInfo(kPrecision)<< "," << fMapName.chars() << ")"; 
  }
}

void MapArchivePipe::WriteMML( FILE* oFile, EWriteMode mode ) {
  char ch;
  switch( GetObjInfo(kMapType) ) {
    case kHDF: ch = 'H'; break;
    case kHDF5: ch = 'h'; break;
    case kMap2: ch = 'M'; break;
    case kGRASS: ch = 'G'; break;
    case kArc: ch = 'A'; break;
    case kASCII: ch = 'a'; break;
	case kCME: ch = 'C'; break;
	case kGeometry: ch = 'd'; break;
    case kSDS: ch = 'S'; break;
    case kBinary: ch = 'B'; break;
    case kIOex: ch = 'i'; break;
    case kPPM: ch = 'P'; break;
  }   
  CString cmdName("M");
  fprintf(oFile,"\n\t\t<config name=\"%s\">", cmdName.chars() );
  fprintf(oFile,"\n\t\t\t<arg name=\"type\"> %c </arg>",ch);
  fprintf(oFile,"\n\t\t\t<arg name=\"precision\"> %d </arg>",GetObjInfo(kPrecision));
  fprintf(oFile,"\n\t\t\t<arg name=\"mapname\"> %s </arg>",fMapName.chars());
  if( fScale > 0.0 ) {
	fprintf(oFile,"\n\t\t\t<arg name=\"scale\"> %f </arg>",fScale);
	fprintf(oFile,"\n\t\t\t<arg name=\"offset\"> %f </arg>",fOffset);
	fprintf(oFile,"\n\t\t\t<arg name=\"cellsize\"> %f </arg>",fCellsize);
  }
}

Bool MapArchivePipe::Cmd(const CString& cmd) { return cmd == "M"; } 

//----------------------------------------------------------------------------------------
//						TPtSeriesPipe
//----------------------------------------------------------------------------------------

PtSeriesPipe::PtSeriesPipe(const char* name, Variable* var) : PointPipe(name, kPtSeries, var ) {
  fIndex = 0;
  SetObjInfo(kMode,kArchive); 
}

int PtSeriesPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  int iarg;
  if( cmd == "P" ) { 
    cd.IntArg(0,fPoint.elem(0));
    cd.IntArg(1,fPoint.elem(1));

    if( gDebug) { 
      sprintf(gMsgStr,"\n%s Output Config: P(%d,%d)", Name(),fPoint(0),fPoint(1)); 
      gPrintDebug();  
    }
    return 1;
  } else return Pipe::Config(cd);
}

void PtSeriesPipe::WriteDataToConfig( CStream& outStream ) const { 
   outStream << " P(" << fPoint(0) << "," << fPoint(1) << ")"; 
}

Bool PtSeriesPipe::Cmd(const CString& cmd) { return cmd == "P"; }

/*
WindowPipe::WindowPipe(const char* name, Variable* var) : PointPipe( name, kWindow, var ) {
  fIndex = 0;
  SetObjInfo(kMode,kArchive); 
}
WindowPipe::WindowPipe(const char* name, EType type, Variable* var) : PointPipe(name, type, var ) {
  fIndex = 0;
  SetObjInfo(kMode,kArchive); 
}

int WindowPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  int iarg;
  if( cmd == "W" ) { 
    cd.IntArg(0,fPoint.elem(0)); 
    cd.IntArg(1,fPoint.elem(1));
    cd.IntArg(2,fSize);
    if( cd.IntArg(3,iarg) > 0) SetObjInfo(kFormat,iarg);

    if( gDebug) { 
      sprintf(gMsgStr,"\n%s Output Config: W(%d,%d,%d,%x)",
	      Name(),fPoint(0),fPoint(1),fSize,GetObjInfo(kFormat)); 
      gPrintDebug();  
    }
    return 1;
  } else return Pipe::Config(cd);
}

void WindowPipe::WriteDataToConfig( CStream& outStream ) const { 
   outStream << ," P(%d,%d,%d,%x) ",fPoint(0),fPoint(1),fSize,GetObjInfo(kFormat)); 
}

Bool WindowPipe::Cmd(const CString& cmd) { return cmd == "W"; }
*/
//----------------------------------------------------------------------------------------
//						CondensePipe
//----------------------------------------------------------------------------------------

CondensePipe::CondensePipe( const char* name, Variable* var )  : kOperation(kO_Info3), PointPipe( name, kCondense, var ) {
  fIndex = 0; 
  SetObjInfo(kMode,kArchive); 
}

int CondensePipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd(), *s;
  int iarg;
  if( cmd == "C" ) { 
    cd.IntArg(0,fPoint.elem(0)); 
    cd.IntArg(1,fPoint.elem(1));
    cd.IntArg(2,fSize);

    if( s = cd.Arg(3) ) { 
      switch( (*s)[0] ) {
      case 'M': SetObjInfo(kOperation,kMax); break;
      case 'm': SetObjInfo(kOperation,kMin); break;
      case 'S': SetObjInfo(kOperation,kSum); break;
      case 'A': SetObjInfo(kOperation,kAve); break;
      default:  SetObjInfo(kOperation,kUnknown); break;
      }
    }

    switch ( iarg ) {
    case kMax: fCumValue = -FLT_MAX; break;
    case kMin: fCumValue = FLT_MAX; break;
    case kSum: case kAve: fCumValue = 0.0; 
    }

    if( cd.IntArg(4,iarg) > 0 ) SetObjInfo(kFormat,iarg);

    if(gDebug) { 
      sprintf(gMsgStr,"\n%s Output Config: C(%d,%d,%d,%x)",
	      Name(),fPoint(0),fPoint(1),fSize,GetObjInfo(kOperation),GetObjInfo(kFormat)); 
      gPrintDebug();  
    }
    return 1;
  } else return Pipe::Config(cd);
}

void CondensePipe::WriteDataToConfig( CStream& outStream ) const {
  char ch;
  switch( GetObjInfo(kOperation) ) {
    case kMax: ch = 'M'; break;
    case kMin: ch = 'm'; break;
    case kSum: ch = 'S'; break;
    case kAve: ch = 'A'; break;
  }
  outStream << " C(" << fPoint(0) << "," << fPoint(1) << "," << fSize << "," << ch << "," << GetObjInfo(kFormat) << ")"; 
}

Bool CondensePipe::Cmd(const CString& cmd) { return cmd == "C"; }

//----------------------------------------------------------------------------------------
//						TimeSeriesInputPipe
//----------------------------------------------------------------------------------------

TimeSeriesInputPipe::TimeSeriesInputPipe( const char* name, Variable* var ) :  kFormat(kO_Info2),
  Pipe( name, kTimeSeriesInput, (TConfigObject*)var ) { SetObjInfo(kMode,kInput); }

int TimeSeriesInputPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  CString* s;
  int iarg;
  if( cmd == "t" ) {
    fSeriesName = *cd.Arg(0); 
    if( !cd.IntArg(1,iarg) ) iarg = 0;
    SetObjInfo(kFormat,iarg);
		if( ( s = cd.Arg(2)) != NULL ) { fFormat = *s;  }
#ifdef POSTGRESS
		if( iarg == 100 ) { 
			CString filename(fSeriesName); filename+="_TS.sds";
			pgObjRec::AddDBaseObject(fFormat,fSeriesName,filename); 
		}
#endif
		if( fCommand ) fCommand->SetTimeSchedule(0.0,0.0);
    return 1;
  } else return Pipe::Config(cd);
}

void TimeSeriesInputPipe::WriteDataToConfig( CStream& outStream ) const { 
   outStream << " t(" << fSeriesName.chars() << "," << GetObjInfo(kFormat) << ")"; 
}

void TimeSeriesInputPipe::WriteMML( FILE* oFile, EWriteMode mode ) {
  CString cmdName("t");
  fprintf(oFile,"\n\t\t<config name=\"%s\">", cmdName.chars() );
  fprintf(oFile,"\n\t\t\t<arg name=\"filename\"> %s </arg>",fSeriesName.chars());
  fprintf(oFile,"\n\t\t\t<arg name=\"format\"> %d </arg>",(int)GetObjInfo(kFormat));
}

Bool TimeSeriesInputPipe::Cmd(const CString& cmd) { return cmd == "t"; }

//----------------------------------------------------------------------------------------
//						TimeSeriesInputPipe
//----------------------------------------------------------------------------------------

EInfoIndex PtTimeSeriesInputPipe::kFormat =  kO_Info2;

PtTimeSeriesInputPipe::PtTimeSeriesInputPipe( const char* name, Variable* var ) : 
	Pipe( name, kPtTimeSeriesInput, (TConfigObject*)var ) { SetObjInfo(kMode,kInput); }

int PtTimeSeriesInputPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
  int iarg; CString* s;
  if( cmd == "p" ) {
    fPtSeriesName = *cd.Arg(0); 
    if( !cd.IntArg(1,iarg) ) iarg = 0;
    SetObjInfo(kFormat,iarg);
		if( ( s = cd.Arg(2)) != NULL ) { fFormat = *s;  }
#ifdef POSTGRESS
		if( iarg == 100 )  { 
			CString filename(fPtSeriesName); filename+="_PTS.sds";
			pgObjRec::AddDBaseObject(fFormat,fPtSeriesName,filename); 
		}
#endif
    if( fCommand ) fCommand->SetTimeSchedule( 0.0, 0.0 );
    return 1;
  } else return Pipe::Config(cd);
}

void PtTimeSeriesInputPipe::WriteDataToConfig( CStream& outStream ) const { 
   outStream << " p(" << fPtSeriesName.chars() << "," << GetObjInfo(kFormat) << ")"; 
}

void PtTimeSeriesInputPipe::WriteMML( FILE* oFile, EWriteMode mode ) {
  CString cmdName("p");
  fprintf(oFile,"\n\t\t<config name=\"%s\">", cmdName.chars() );
  fprintf(oFile,"\n\t\t\t<arg name=\"filename\"> %s </arg>",fPtSeriesName.chars());
  fprintf(oFile,"\n\t\t\t<arg name=\"format\"> %d </arg>",(int)GetObjInfo(kFormat));
}

Bool PtTimeSeriesInputPipe::Cmd(const CString& cmd) { return cmd == "p"; }


//----------------------------------------------------------------------------------------
//						DBaseInputPipe
//----------------------------------------------------------------------------------------

EInfoIndex DBaseInputPipe::kOrder = kO_Info2;

DBaseInputPipe::DBaseInputPipe( const char* name, Variable* var ) :  Pipe( name, kDBaseInput, (TConfigObject*)var ) { 
		SetObjInfo(kMode,kInput);
		fMaps[0] = fMaps[1] = NULL;
		fIndex[0]=fIndex[1]=fIndex[2]=fIndex[3]=0;
}

int DBaseInputPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd(); 
  int iarg, order, i;
	int argc=0;
	if( cmd[0] == 'm' ) {
    if( cmd.length() == 1 ) order = 1;
    else order = cmd[1] - '0';
    if( order < 0 || order > 4 ) { gPrintErr("Bad format for  DBaseInput command."); return 0; }
    SetObjInfo(kOrder,order);
    if( order == 4 ) {
			CString* s = cd.Arg(argc++);
			if(s) { fFileName = *s; }  //  DBaseName
			s = cd.Arg(argc++);
			if(s) fMap.Add( *s  ); 
			s = cd.Arg(argc++);
			if(s) { fFunctionName = *s; } 
			while( 1 ) { 
				s = cd.Arg(argc++);
				if(s) fMap.Add( *s  );   //  1: TableName,  2: SectorName,  3:  ParmName
				else { break; }
			}
    } else {
			fFileName = *cd.Arg(argc++);
			for( i=0; i<order; i++) { 
				CString* s = cd.Arg(argc++);
				if(s) fMap.Add( *s  ); 
			}
			for( i=0; i<=order; i++) cd.IntArg(argc++,fIndex[i]);
		}
    return 1;
  } else if( cmd == "pmd" ) {
		SetObjInfo(kOrder,0);
		CString* s = cd.Arg(argc++);
		if(s) { fFileName = *s; }  //  DBaseName
		s = cd.Arg(argc++);
		if(s) { fFunctionName = *s; } 
		while( 1 ) { 
			s = cd.Arg(argc++);
			if(s) fMap.Add( *s  );   //  1: TableName,  2: SectorName,  3:  ParmName
			else { break; }
		}
  } else return Pipe::Config(cd);
}

void DBaseInputPipe::WriteDataToConfig( CStream& outStream ) const {
   int i, order =  GetObjInfo(kOrder);
   if( order < 0 || order > 4 ) { return; }
   outStream << " m" << order << "(" << fFileName.chars(); 
   if( order == 4 ) {
		 outStream << "," << fMap[0].chars(); 
 		 outStream << "," << fFunctionName.chars(); 
		 for( i=1; i<fMap.NArgs(); i++) { outStream << "," << fMap[i].chars(); }
  } else {    
		 for( i=0; i<GetObjInfo(kOrder); i++) outStream << "," << fMap[i].chars();
		 for( i=0; i<=GetObjInfo(kOrder); i++) outStream << "," << fIndex[i];
   }
   outStream << ")";   
}

void DBaseInputPipe::WriteMML( FILE* oFile, EWriteMode mode ) {
   int i, order =  GetObjInfo(kOrder);
   if( order < 0 || order > 4 ) { return; }
   CString cmdName("m"); cmdName.appendIndex(order);
   fprintf(oFile,"\n\t\t<config name=\"%s\">", cmdName.chars() );
   fprintf(oFile,"\n\t\t\t<arg name=\"filename\"> %s </arg>",fFileName.chars());
   if( order == 4 ) {
	 fprintf(oFile,"\n\t\t\t<arg name=\"mapname\"> %s </arg>",fMap[0].chars());
	 fprintf(oFile,"\n\t\t\t<arg name=\"function\"> %s </arg>",fFunctionName.chars());
	 for( i=1; i<fMap.NArgs(); i++) { fprintf(oFile,"\n\t\t\t<arg> %s </arg>", fMap[i].chars() ); }
   } else {
	 for( i=0; i<GetObjInfo(kOrder); i++) { fprintf(oFile,"\n\t\t\t<arg> %s </arg>", fMap[i].chars() ); }
	 for( i=0; i<=GetObjInfo(kOrder); i++) { fprintf(oFile,"\n\t\t\t<arg> %d </arg>", fIndex[i] ); }
   }
}


Bool DBaseInputPipe::Cmd(const CString& cmd) { return ( (cmd == "pmd") || ( cmd[0] == 'm' && ( isdigit(cmd[1]) ||  (cmd.length() == 1)) )); }

//----------------------------------------------------------------------------------------
//						MapInputPipe
//----------------------------------------------------------------------------------------

EInfoIndex MapInputPipe::kMapClass = kO_Info2;
EInfoIndex MapInputPipe::kConfigStatus = kO_Info3;

MapInputPipe::MapInputPipe( const char* name, Variable* var ) :  Pipe( name, kMapInput, (TConfigObject*)var ) { 
	SetObjInfo(kMode,kInput); fS[1] = 1.0; fS[0] = 0.0;  fRepeatTime = -1.0;
}

int MapInputPipe::Config( TConfigData& cd ) {
  const CString& cmd = cd.Cmd();
	CString* s = NULL;
	
  if( cmd == "d" ) {
    fMapSource = *cd.Arg(0); 
    fMapName = *cd.Arg(1);
    if( (s = cd.Arg(2)) != NULL ) { fFormat = *s;  }
#ifdef POSTGRESS
		if( fMapSource == "DB" && Model::I0().DataBase() ) { 
			CString filename(fMapName); filename+=".sds"; 
			pgObjRec::AddDBaseObject(fFormat,fMapName,filename);
		}
#endif
    SetObjInfo(kMapClass,kSurfaceMap);
    SetObjInfo(kConfigStatus,1);
    return 1;
  } else if( cmd == "c" ) {
    fMapSource = *cd.Arg(0); 
    fMapName = *cd.Arg(1); 
    if( (s = cd.Arg(2)) != NULL ) { fFormat = *s; }
#ifdef POSTGRESS
		if( fMapSource == "DB" && Model::I0().DataBase() ) { 
			CString filename(fMapName); filename+=".sds";
			pgObjRec::AddDBaseObject(fFormat,fMapName,filename); 
		}
#endif
    SetObjInfo(kMapClass,kClassMap);
		SetObjInfo(kConfigStatus,1);
//	fprintf(stderr,"\n  ** dbg: config pipe %s: %s %s \n", fConfigObject->Name(), fMapSource.chars(), fMapName.chars() ); fflush(stderr);
  } else if( (cmd == "S") && ((GetObjInfo(kConfigStatus) < 2) || Env::GetInfo(Env::kEventsDefined) ) ) {
    cd.FloatArg(0,fS[1],CD::kRequired); 
    cd.FloatArg(1,fS[0]);
		SetObjInfo(kConfigStatus,2);
		return 1;
  }
  else return Pipe::Config(cd);
}

void MapInputPipe::WriteMML( FILE* oFile, EWriteMode mode ) {
  CString cmdName("d");
  if( GetObjInfo(kMapClass) == kClassMap ) { cmdName = "c"; }
  fprintf(oFile,"\n\t\t<config name=\"%s\">", cmdName.chars() );
  fprintf(oFile,"\n\t\t\t<arg name=\"mapsource\"> %s </arg>",fMapSource.chars());
  fprintf(oFile,"\n\t\t\t<arg name=\"mapname\"> %s </arg>",fMapName.chars());
  if( !fFormat.empty() ) { fprintf(oFile,"\n\t\t\t<arg name=\"format\"> %s </arg>",fFormat.chars()); }
//  fprintf(oFile,"\n\t\t</config>");
//  fprintf(oFile,"\n\t\t<config name=\"S\">");
  fprintf(oFile,"\n\t\t\t<arg name=\"scale\"> %f </arg>",fS[1]);
  fprintf(oFile,"\n\t\t\t<arg name=\"offset\"> %f </arg>",fS[0]);
  fprintf(oFile,"\n\t\t</config>");
}

void MapInputPipe::WriteDataToConfig( CStream& outStream ) const {
  if( GetObjInfo(kMapClass) == kClassMap ) 
    if( fFormat.empty() ) { outStream << " c(" << fMapSource.chars() << "," <<  fMapName.chars() << ") S(" << (double) fS[1] << "," <<  fS[0] << ")"; }
    else { outStream << " c(" << fMapSource.chars() << "," <<  fMapName.chars() << "," <<  fFormat.chars() << ") S(" << (double) fS[1] << "," <<  fS[0] << ")"; }
  else
    if( fFormat.empty() ) { outStream << " d(" << fMapSource.chars() << "," <<  fMapName.chars() << ") S(" << (double) fS[1] << "," <<  fS[0] << ")"; }   
		else { outStream << " d(" << fMapSource.chars() << "," <<  fMapName.chars() << "," <<  fFormat.chars() << ") S(" << (double) fS[1] << "," <<  fS[0] << ")"; }
}

Bool MapInputPipe::Cmd(const CString& cmd) { return ( cmd == "c" || cmd == "d"); }
