//----------------------------------------------------------------------------------------
//      Pipe.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History
//----------------------------------------------------------------------------------------

#ifndef __MML_Pipe__
#define __MML_Pipe__
  
#include "MML.h"
#include "Config.h"
#include "Point2.h"
#include "Region2.h"
#include "CStream.h"

typedef class TConfigObject TConfigObject;
typedef class Variable Variable;
typedef class TPipeCommand TPipeCommand;

//----------------------------------------------------------------------------------------
//				Pipe
//----------------------------------------------------------------------------------------

class Pipe : public TNamedObject, public MMLObject {
protected:
  TConfigObject* fConfigObject;
  TPipeCommand* fCommand;
  static int fPipeIndex;
  static ArgArray kTypeList;  // ( "undefined", "anim", "graph", "sheet", "map", "window", "ptSeries", "condense", "timeSeriesInput", "ptTimeSeriesInput", "dBaseInput", "mapInput", "socket" );

  inline int SetType( const CString& mod, Bool isExtension = False ) {
    int index = kTypeList.Find(mod);
    if( index > 0 ) {
      if( isExtension && index != GetObjInfo(kType) ) gPrintErr(" Error, can't change Probe Type in Extension.");
      else  SetObjInfo(kType,index);
    }
    return index;
  }
 
public:
  enum EMode { kUnknown, kInput, kArchive, kDisplay };
  enum EType { kUndefined, kAnim, kGraph, kSheet, kMapArchive, kWindow, kPtSeries, kCondense, kTimeSeriesInput, kPtTimeSeriesInput, kDBaseInput, kMapInput, kSocket, kDataDisplay };
  static EInfoIndex kType;
  static EInfoIndex kMode;

  Pipe( const char* name, EType type, TConfigObject* var  );
  ~Pipe()  { ; }

  virtual int Config( TConfigData & cd );
  virtual void WriteDataToConfig( CStream& outStream ) const {;}

  void Dump(FILE* oFile) { ; }
	inline virtual void Reset() { ;  } 

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}

  virtual int GetModifierString( CString& name ) const  { return 0;}
  static Pipe* GetPipeFromConfig( TConfigData& cd, TConfigObject* cobj, TPipeCommand*& pc );

  virtual inline int Execute() { return 0;}
  virtual int Initialize() { return 0; }
  virtual int Finalize() { return 0; }
	inline void  SetPipeCommand( TPipeCommand* p ) { fCommand = p; }
	inline Variable* Var() { return (Variable*) fConfigObject; }


};

//----------------------------------------------------------------------------------------
//						AnimPipe
//----------------------------------------------------------------------------------------


class  AnimPipe : public Pipe {
protected:
  float fScale;
  float fOffset;

public:

  static EInfoIndex kADataType;

  AnimPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );

  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}

  virtual int GetModifierString( CString& name ) const  { return Pipe::GetModifierString(name); }

  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { Pipe::Reset(); } 
};

//----------------------------------------------------------------------------------------
//						DataDisplayPipe
//----------------------------------------------------------------------------------------


class  DataDisplayPipe : public Pipe {
protected:

public:

  static EInfoIndex kADataType;

  DataDisplayPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );

  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}

  virtual int GetModifierString( CString& name ) const  { return Pipe::GetModifierString(name); }

  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { Pipe::Reset(); } 
};

//----------------------------------------------------------------------------------------
//						SocketPipe
//----------------------------------------------------------------------------------------


class  SocketPipe : public Pipe {
  
protected:
  float fScale;
  float fOffset;

public:

  const EInfoIndex kNBytes;

  SocketPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );

  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}

  virtual int GetModifierString( CString& name ) const  {  return Pipe::GetModifierString(name); }

  static Bool Cmd(const CString& cmd);

	inline virtual void Reset() { Pipe::Reset(); } 
};


//----------------------------------------------------------------------------------------
//						TPointPipe
//----------------------------------------------------------------------------------------


class  PointPipe : public Pipe {

protected:
  
  Point2 fPoint;
  Pix fCell;
	int fCellLayer;
public:

  const EInfoIndex kFormat ;

  PointPipe( const char* name, EType type,  Variable* var, Point2& point );
  PointPipe( const char* name, EType type,  Variable* var );

  virtual int Config( TConfigData& cd ) { return Pipe::Config(cd); }

  virtual void WriteDataToConfig( CStream& outStream ) const {;} 

  void Dump(FILE* oFile) {;}

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  {  return Pipe::GetModifierString(name);  }
	inline virtual void Reset() { Pipe::Reset(); } 
};
//----------------------------------------------------------------------------------------
//						TGraphPipe
//----------------------------------------------------------------------------------------

class  GraphPipe : public PointPipe {
protected: 
  float fRange[2];
  int fDomain[2];
  float fMag;

public:

  GraphPipe( const char* name, Variable* var );
  GraphPipe( const char* name, Variable* var, Point2& point );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  {  return PointPipe::GetModifierString(name);  }
  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { PointPipe::Reset(); } 
};

//----------------------------------------------------------------------------------------
//						TSheetPipe
//----------------------------------------------------------------------------------------

class  SheetPipe : public PointPipe {
protected:  
  int fSheetIndex;
  static int fSheetCount;

public:

  SheetPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}

  

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  { return PointPipe::GetModifierString(name); }
  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { PointPipe::Reset(); } 
};
//----------------------------------------------------------------------------------------
//						TMapPipe
//----------------------------------------------------------------------------------------

enum EMapType { kUnDefined, kHDF, kHDF5, kIOex, kMap2, kGRASS, kGeometry, kArc, kArcSVF, kIDRISI, kERDAS, kBinary, kASCII, kSDS, kCME, kPPM };

class  MapArchivePipe : public Pipe {
protected:
  CString fMapName;
  float fScale;
  float fOffset;
  float fCellsize;

public:

  static EInfoIndex kMapType;
  static EInfoIndex kPrecision ;

  MapArchivePipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}

  

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) ;
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  { return Pipe::GetModifierString(name); }
	CString& MapName() { return fMapName; }
  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { Pipe::Reset(); } 
};

/*
class  WindowPipe : public PointPipe {

protected:

  int fSize;

public:

  WindowPipe( const char* name, Variable* var );
  WindowPipe( const char* name, EType type,  Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}

  

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  { return PointPipe::GetModifierString(name); }
  static Bool Cmd(const CString& cmd);
};
*/
//----------------------------------------------------------------------------------------
//						TPtSeriesPipe
//----------------------------------------------------------------------------------------

class  PtSeriesPipe : public PointPipe {

public:

  PtSeriesPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}

  

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  { return PointPipe::GetModifierString(name); }
  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { PointPipe::Reset(); } 
};

//----------------------------------------------------------------------------------------
//						TCondensePipe
//----------------------------------------------------------------------------------------

class  CondensePipe : public PointPipe {
protected:
  float fCumValue;
  int fSize;

public:

  enum EOperation { kUnknown, kMax, kMin, kSum, kAve };
  const EInfoIndex kOperation;

  CondensePipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}

  

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  { return PointPipe::GetModifierString(name); }
  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { PointPipe::Reset(); } 
};

//----------------------------------------------------------------------------------------
//						TimeSeriesInputPipe
//----------------------------------------------------------------------------------------

class TimeSeriesInputPipe : public Pipe {
protected:
  CString fSeriesName;
  CString fFormat;

public:

  const EInfoIndex kFormat;

  TimeSeriesInputPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}

  

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const { return Pipe::GetModifierString(name);  }
  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { Pipe::Reset(); } 
};
//----------------------------------------------------------------------------------------
//						PtTimeSeriesInputPipe
//----------------------------------------------------------------------------------------

class PtTimeSeriesInputPipe : public Pipe {
protected:
  CString fPtSeriesName;
  CString fFormat;

public:

  static EInfoIndex kFormat;

  PtTimeSeriesInputPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  { return Pipe::GetModifierString(name); }
  static Bool Cmd(const CString& cmd);
	inline virtual void Reset() { Pipe::Reset(); } 
};

//----------------------------------------------------------------------------------------
//						DBaseInputPipe
//----------------------------------------------------------------------------------------

class DBaseInputPipe : public Pipe {
protected:
  CString fFileName;
  CString fFunctionName;
  ArgArray fMap;
  int fIndex[4];
  Variable* fMaps[2];

public:

  static EInfoIndex kOrder;

  DBaseInputPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}

  

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  { return Pipe::GetModifierString(name); }
  static Bool Cmd(const CString& cmd);
  inline const CString& Map( int index ) const { return fMap[index]; }
	inline void SetMap(int i, Variable* v)  { fMaps[i] = v; }
	inline void AddMap( const CString& s ) {  fMap.Add( s  ); } 
	inline Variable* GetMap(int i)  { return fMaps[i]; }
	inline virtual void Reset() { Pipe::Reset(); } 
};

//----------------------------------------------------------------------------------------
//						MapPipe
//----------------------------------------------------------------------------------------

enum EMapClass { kUnknown, kClassMap, kSurfaceMap };

class MapInputPipe : public Pipe {
protected:
  CString fMapName;
  CString fMapSource;
  CString fFormat;
  float fS[2];
  float fRepeatTime;

public:

  static EInfoIndex kMapClass;
  static EInfoIndex kConfigStatus;

  MapInputPipe( const char* name, Variable* var );

  virtual int Config( TConfigData& cd );
  void WriteDataToConfig( CStream& outStream ) const; 

  void Dump(FILE* oFile) { ; }

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void WriteCEqns(FILE* cHdrFile, FILE* cEqnFile) {;}
  virtual int GetModifierString( CString& name ) const  {  return Pipe::GetModifierString(name); ;}
  static Bool Cmd(const CString& cmd);
	CString& Format() { return fFormat; }
	inline virtual void Reset() { Pipe::Reset(); } 
};


#endif


