//----------------------------------------------------------------------------------------
//	Schedule.h;
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.;
//	Change History:;
//----------------------------------------------------------------------------------------;
  
#ifndef __MML_Schedule__
#define __MML_Schedule__
  
#include "Globals.h"
#include "ConfigObjects.h"
#include "MML.h"

inline float smax( float x, float y ) { return ( x > y ) ? x : y; }
inline float smin( float x, float y ) { return ( x < y ) ? x : y; }

typedef class ExSchedule ExSchedule;


//----------------------------------------------------------------------------------------
//						StepExSchedule
//----------------------------------------------------------------------------------------

class StepExSchedule  {

  int Start;
  int Stop;
  int StepInc;
  int Current;

public:
  StepExSchedule() { Current=0; StepInc=1; Start = -INT_MAX; Stop = INT_MAX; }

  void Init( int dt, int start, int stop ) { StepInc = dt; Start = start; Stop = stop ; Current = Start - StepInc; } 
	void SetTimeStep( int dt, int step ) { StepInc = dt; Start = step; Stop = step+StepInc*2; Current = Start; } 

  inline int ReSchedule( int index, int& next_index ) {
    next_index = index + StepInc;
    Current = next_index;
    if( next_index >= Start && next_index <= Stop ) {
      return 1;
    } else return 0;
  }

  virtual inline int UpdateOK( int index ) { 
    if( index >= Start && index <= Stop && index >= Current + StepInc ) {
      Current = index;
      return 1;
    } else return 0;
  }

  inline int Config( TConfigData& cd ) { 
    if( cd.IntArg(0,StepInc) <= 0 ) { Start = -INT_MAX; Stop = INT_MAX; return 0; }
    if( cd.IntArg(1,Start) <= 0 ) Start = -INT_MAX;
    if( cd.IntArg(2,Stop) <= 0 ) Stop = INT_MAX;
    Current = Start - StepInc;
    return 1;
  }

  virtual inline int WriteDataToConfig( CStream& outStream ) const {
    if( Start > -INT_MAX ) outStream << " OS(" << StepInc << "," << Start << "," << Stop << ")"; 
    else outStream << " OS(" << StepInc << ")";
    return 1;
  }


  inline void WriteMML( FILE* oFile ) {
    if( Start > -INT_MAX ) fprintf(oFile,"\t\t\tStart = %d;\n", Start );
    if( Stop  < INT_MAX ) fprintf(oFile,"\t\t\tStop = %d;\n", Stop );
    fprintf(oFile,"\t\t\tStep = %d;\n", StepInc );
  }

  virtual inline StepExSchedule& operator= ( const StepExSchedule& s )  {
    Start = s.Start;
    Stop = s.Stop;
    StepInc = s.StepInc;
    Current = Start - StepInc;
    return *this; 
  }
  
  virtual inline int operator== ( const StepExSchedule& s )  const {
    return ( (Start == s.Start) && (Stop == s.Stop) && (StepInc == s.StepInc) );
  }

  inline int StartTime() { return Start; }
  inline int StopTime() { return Stop; }
  inline int DT() { return StepInc; }

	virtual int ProcessCommand(const TCommandNode* n){ return 0;}
	virtual int ProcessAttribute(const TAttributeNode* n){ return 0; }

}; 

//----------------------------------------------------------------------------------------
//						TimeExSchedule
//----------------------------------------------------------------------------------------


class TimeExSchedule {

  float Start;
  float Stop;
  float Repeat;
  float StepInc;
  float Current;
  char  Unit;

public:

  TimeExSchedule() { Start = -FLT_MAX; Stop = FLT_MAX; Repeat = -1; Current = StepInc = 0.0; Unit = 0; }

  void Init( float dt, float start, float stop ) { StepInc = dt; Start = start; Stop = stop; Current = Start - StepInc; } 
	void SetTimeStep( float dt, float time ) { StepInc = dt; Start = time; Stop = time+StepInc*2; Current = Start; } 

  inline int ReSchedule( float time, float& next_time ) {
    next_time = time + StepInc;
    Current = next_time;
    if( next_time >= Start && next_time < Stop ) {
      return 1;
    } else return 0;
  }

  virtual inline int UpdateOK( float time ) { 
    if( time >= Start && time <= Stop && time >= Current + StepInc ) {
      Current = time;
      return 1;
    } else return 0;
  }
  
  inline char getUnit() { return Unit; }

  inline int Config( TConfigData& cd ) { 
	CString* s;
    if( cd.FloatArg(0,StepInc) <= 0 ) { return 0; }
    if( cd.FloatArg(1,Start) <= 0 ) { return 1; }
    if( cd.FloatArg(2,Stop) <= 0 ) { return 2; }
    if( cd.FloatArg(3,Repeat) <= 0 ) { return 3; }
    if( s = cd.Arg(4) ) { Unit = (*s)[0]; return 4; }
    Current = Start - StepInc;
    return 3;
  }

  virtual inline int WriteDataToConfig( CStream& outStream ) const {
	if( Unit != 0 ) { outStream << " OT(" << StepInc << "," << Start << "," << Stop << "," << Repeat << "," << Unit << ")"; }
    if( Repeat >= 0 ) outStream << " OT(" << StepInc << "," << Start << "," << Stop << "," << Repeat << ")"; 
    else if( Start > -FLT_MAX ) outStream << " OT(" << StepInc << "," << Start << "," << Stop << ")"; 
    else outStream << " OT(" << StepInc << ")"; 
    return 1;
  }

  inline void WriteMML( FILE* oFile ) {
    if( Repeat >= 0.0 ) fprintf(oFile,"\t\t\tRepeat = %f;\n", Repeat );
    if( Start > -FLT_MAX ) fprintf(oFile,"\t\t\tStart = %f;\n", Start );
    if( Stop  < FLT_MAX ) fprintf(oFile,"\t\t\tStop = %f;\n", Stop );
    fprintf(oFile,"\t\t\tStep = %f;\n", StepInc );
  }

  virtual inline TimeExSchedule& operator= ( const TimeExSchedule& s )  {
    Start = s.Start;
    Stop = s.Stop;
    StepInc = s.StepInc;
    Current = Start - StepInc;
    return *this; 
  }
  
  virtual inline int operator== ( const TimeExSchedule& s )  const {
    return ( (Start == s.Start) && (Stop == s.Stop) && (StepInc == s.StepInc) ); 
  }

  inline float StartTime() { return Start; }
  inline float StopTime() { return Stop; }
  inline float RepeatTime() { return Repeat; }
  inline float DT() { return StepInc; }

  void Clamp( ExSchedule* parent, StepExSchedule& s);
  
	virtual int ProcessCommand(const TCommandNode* n){ return 0;}
	virtual int ProcessAttribute(const TAttributeNode* n){ return 0;}

};

//----------------------------------------------------------------------------------------
//					 ExSchedule
//----------------------------------------------------------------------------------------

class ExSchedule : public MMLObject, public TNamedObject {
protected:

  TimeExSchedule fTSchedule;
  StepExSchedule fSSchedule;
  ExSchedule* fParent;

public:

  enum EExSchedMode { kNone, kDefault, kStepS, kTimeS, kDynS };
  enum ETimeUnit { kUndefined, kSecond, kMinute, kHour, kDay, kWeek, kYear };
  static EInfoIndex kMode, kUnit;
  static ArgArray kTypeList;  // ( "undefined", "time", "step", "dyn" );

  ExSchedule(const CString& name , ExSchedule* parent = NULL ) : TNamedObject(name) { fParent = parent;   }

  ExSchedule( const CString&  name,  EExSchedMode mode, ExSchedule* parent = NULL ) : TNamedObject(name) { 
    SetObjInfo(kMode,mode ); fParent = parent;
  }

  ExSchedule( const CString&  name,  float dt, float start, float stop ) : TNamedObject(name)  { 
    SetObjInfo(kMode,kTimeS ); 
    fTSchedule.Init( dt, start, stop);
    fParent = NULL;  
  }

  ExSchedule( const CString&  name,  int dt, int start, int stop ) : TNamedObject(name)  { 
    SetObjInfo(kMode,kStepS ); 
    fSSchedule.Init( dt, start, stop); 
  }

  inline void SetParent(ExSchedule* parent) { 
    fParent = parent;
    if( ScheduleConfigured() ) {  fTSchedule.Clamp(parent,fSSchedule); }
  }

  inline int SetMode( const CString& mod, Bool isExtension = False ) {
    int index = kTypeList.Find(mod);
    if( index > 0 ) {
      if( isExtension && index != GetObjInfo(kMode) ) gPrintErr(" Error, can't change Varaible Type in Extension.");
      else  SetObjInfo(kMode,index);
    }
    return index;
  }
  
  inline byte ScheduleConfigured() const { 
    byte mode = GetObjInfo(kMode);
    return !( mode==kNone || mode==kDefault); 
  }

  virtual inline int WriteDataToConfig( CStream& outStream ) const {
    if( ScheduleConfigured() ) { 
      switch(GetObjInfo(kMode)) {
      case kStepS: return fSSchedule.WriteDataToConfig( outStream ); 
      case kTimeS: return fTSchedule.WriteDataToConfig( outStream ); 
      }
    }
    return 0;
  }

 virtual inline int operator== ( const ExSchedule& s )  const { 
    if( ScheduleConfigured() ) { 
      switch(GetObjInfo(kMode)) {
      case kStepS: 
				if( s.GetObjInfo(kMode) == kStepS ) 
					return fSSchedule.operator== ( s.fSSchedule ); 
				else return 0;
      case kTimeS: 
				if( s.GetObjInfo(kMode) == kTimeS ) 
					return fTSchedule.operator== ( s.fTSchedule );
				else return 0;
      }
    } else return 0;
  }
 
  virtual inline ExSchedule& operator= ( const ExSchedule& s )  { 
    if( !s.ScheduleConfigured() ) { return *this; } 
    if( !ScheduleConfigured() ) SetObjInfo( kMode, s.GetObjInfo(kMode) );
    else if ( GetObjInfo(kMode) != s.GetObjInfo(kMode) ) { return *this; } 
    fParent = s.fParent;
    fTSchedule = s.fTSchedule;
    fSSchedule = s.fSSchedule;
    if( fParent ) {  fTSchedule.Clamp(fParent,fSSchedule); }
    return *this; 
  }

  inline int ReSchedule( int index, int& next_index ) { return fSSchedule.ReSchedule( index, next_index ); }
  inline int ReSchedule( float time, float& next_time ) { return fTSchedule.ReSchedule( time, next_time ); }

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual int GetModifierString( CString& name ) const ;
  virtual void Dump( FILE* oFile );
  int AddModifier( const CString& mod, Bool isExtension = False );
  virtual void WriteCCode( FILE* CHdrFile, FILE* CEqnFile);
  void WriteCCommandCode(FILE* CEqnFile);
  virtual inline int Execute() { return 0;}
  int Config( TConfigData& cd );

  inline float StartTime() { return fTSchedule.StartTime(); }    
  inline float StopTime() { return fTSchedule.StopTime(); }        
  inline float RepeatTime() { return fTSchedule.RepeatTime(); }        
  inline float DT() { return fTSchedule.DT(); } 
	inline int UpdateOK( int index ) { return fSSchedule.UpdateOK( index ); } 
	inline int UpdateOK( float time ) { return fTSchedule.UpdateOK( time ); } 
	    
	int SetTimeSchedule( float dt, float time ) {
    SetObjInfo(kMode,kTimeS );
		fTSchedule.SetTimeStep( dt, time );
		return 0;
	}
	int SetStepSchedule(int dt, int step ){
		SetObjInfo(kMode,kStepS );
		fSSchedule.SetTimeStep( dt, step ); 
		return 0;	
	}
	virtual int ProcessCommand(const TCommandNode* n) { return 0;}
	virtual int ProcessAttribute(const TAttributeNode* n);
                       
};


#endif
