#include "MML_Variable.h"
#include "MML_Event.h"
#include "MML_Module.h"
#include "MML_Model.h"
#include "MML_Connection.h"
#include "Externals.h"
#include "GOFCalc.h"

ArgArray Variable::kTypeList( 7, " ", "aux", "flux", "state", "data", "time", "constant" );
TObjectList* Variable::fFlagList = NULL;
FlagType Variable::fCurrentFlag = 1;

 FlagType	FisClamped ;		//     is Clamped Flux or State Var
 FlagType	FisConstant ;		// 	0 = Varies Temporially,  1 = constant     
 FlagType	FisSpatial ;		// 	is a spatial Variable	 
 FlagType	FisIgnored ;		// 	0 = Varies Temporially,  1 = constant     
 FlagType	FisStateVar ;		// 	1 = State variable, 0 = Aux/Flux Var
 FlagType	FisFlux ;		//  1 = is argument to flux function, 0 = is not  
 FlagType	FisImport ;		// 	is Module Input 	  
 FlagType	FisExport ;		//  is Outgoing Flux from Clamped State Var
 FlagType	FRegistered ;		//  Reqistered with Interface
 FlagType	FVisible ;		//  Visible in Interface Display
 FlagType	FDualConnection ;		//  Two-way connection
// FlagType	FisTIME ;	
 FlagType	FisSpatialRand ;
// FlagType	FHasNegFlux ;		
 FlagType	FLinkEdges;
 FlagType	FisCrossCell ;		// 	1 = Cross Cell Variable
 FlagType	FisOptimizable ;	
 FlagType	FHasDelay ;
 FlagType	FisFinalUpdate; 	
 FlagType	FcalcGOF; 	
 FlagType	FOverrideInit; 	
 FlagType	FisDisplayVar ;		//  Display Var

float Variable::fGarbage = 0.0;
EConfigInfo Variable::kDebug = kCInfo0;
EConfigInfo Variable::kType = kCInfo1;
EConfigInfo Variable::kMode = kCInfo2;
EConfigInfo Variable::kInitMode = kCInfo3;
		
CString Variable::kGarbage("");
//----------------------------------------------------------------------------------------
//				Variable
//----------------------------------------------------------------------------------------

Variable::Variable( const char* name ) : TConfigObject ( name, kVariable ), 
    fConnectionList( CString("Connections for ") += name ), fFluxList( CString("Fluxes for ") += fName )   {
  fFlags = 0;
  if(fFlagList==NULL) {
    fFlagList = new TObjectList();
    AddFlags();
  } 
  SetCInfo(kDebug,kNoConfig);
  fDataType = kFloat;
  fConnection = NULL;
  fModule = NULL;
  SetCInfo(kInitMode, kDefault);  
  fOrigin = NULL;
  fDestination = NULL;
  fSchedule = NULL;
  fVarNode = NULL;
  fIVar = -1;
  fParent = NULL;
  SetF(FisConstant,True,"Variable (constructor)");
  fDefValue.elem(0) = "0.0"; 
  fFunctionNodeList = NULL;
  fOrder = -1;
  fIntegrationMethod = "1";
  fDynCommand = NULL;
  fActivationLayer = -1;
  fLinkLayer = -1;
  fDataType = kFloat;
  fArrayRec = NULL;
  fCoord_index = 255;
  for(int i=0; i<kMaxNVarDBMaps; i++ ) { fMapVariable[i] = NULL; }
  for(int j=0; j<kMaxNVarArrays; j++) {  _Array_Index_Map[j] = -1; _Arrays[j] = NULL; }
  fFluxInfo[kF_FluxType] = kF_Unknown;
}
 
Variable::~Variable() { 
//	fCommandList.Free(); 
}

void Variable::AddFlags() {
  AddFlag("FisFlux",FisFlux);
//  AddFlag("FisTIME",FisTIME);
  AddFlag("FisStateVar",FisStateVar);
  AddFlag("FisSpatial",FisSpatial);
  AddFlag("FisClamped",FisClamped);
  AddFlag("FisConstant",FisConstant);
  AddFlag("FisExport",FisExport);
  AddFlag("FisImport",FisImport);
  AddFlag("FisIgnored",FisIgnored);
  AddFlag("FRegistered",FRegistered);
  AddFlag("FVisible",FVisible);
	AddFlag("FDualConnection",FDualConnection);
  AddFlag("FisSpatialRand",FisSpatialRand);
//	AddFlag("FHasNegFlux",FHasNegFlux);
	AddFlag("FLinkEdges",FLinkEdges);	
  AddFlag("FisCrossCell",FisCrossCell);
  AddFlag("FisOptimizable",FisOptimizable);
  AddFlag("FHasDelay",FHasDelay);
  AddFlag("FisFinalUpdate",FisFinalUpdate);
  AddFlag("FcalcGOF",FcalcGOF);
  AddFlag("FOverrideInit",FOverrideInit); 
  AddFlag("FisDisplayVar",FisDisplayVar);
}

void Variable::SetCmdStatus ( TCommand::EType type,	TCommand::EStatus status) {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( (type == c0.GetObjInfo(TCommand::kType)) || (type == TCommand::kUndefined) ) {
      c0.SetObjInfo( TCommand::kStatus, status  );
      			if( gDebug > 1 ) { sprintf(gMsgStr,"\nSetCmdStatus: %s:%s(%x) -> %x",Name(),c0.Name(),type,status); gPrintLog(); }
    }
  }
}

void Variable::SetCmdStatus ( TCommand::EExType etype, TCommand::EStatus status) {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( etype == c0.GetObjInfo(TCommand::kExType) ) {
      c0.SetObjInfo( TCommand::kStatus, status  );
			if( gDebug > 1 ) { sprintf(gMsgStr,"\nSetCmdStatusEx: %s:%s(%x) -> %x",Name(),c0.Name(),etype,status); gPrintLog(); }
    }
  }
}

void Variable::setFluxTypeIndex( EFluxType ft ) { 
  fFluxInfo[kF_FluxType] = ft; 
  if( gDebug > 1 ) {
	gPrintLog( format("%s:%s.setFluxTypeIndex: %d",fModule->Name(), Name(), ft ) ); 
  }
}

void Variable::setFluxType( CString& s ) { 
  s.downcase();
  if( s[0] == '2' ) { fFluxInfo[kF_FluxType] = kF_BiFlow; }
  else if( s[0] == '1' ) { fFluxInfo[kF_FluxType] = kF_UniFlow; }
  else if( s[0] == 'u' ) { fFluxInfo[kF_FluxType] = kF_Unknown; }
  else { gPrintErr( format ( "Unknown flux type in %s: %s", Name(), s.chars() )); }
  if( gDebug > 1 ) {
	gPrintLog( format("%s:%s.setFluxType: %s",fModule->Name(), Name(), s.chars() ) ); 
  }
}

EFluxType Variable::getFluxTypeIndex() const { return (EFluxType) fFluxInfo[kF_FluxType]; } 

char Variable::getFluxType() const {
	switch( fFluxInfo[kF_FluxType] ) {
	  case kF_BiFlow: return '2';
	  case kF_UniFlow: return '1';
	  case kF_Unknown: return 'u';	  
	  default:  { gPrintErr( format ( "Unknown flux type in %s: %x", Name(), fFluxInfo[kF_FluxType] )); }
	}
}

void Variable::SetBreak ( TCommand::EType type, int BreakOn ) {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( (type == TCommand::kUndefined) || (type == c0.GetObjInfo(TCommand::kType)) ) {
      c0.SetBreak( BreakOn );
    }
  }
}
void Variable::SetBreak ( TCommand::EExType type, int BreakOn ) {
  for( Pix p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( (type == TCommand::kNoEx) || (type == c0.GetObjInfo(TCommand::kExType)) ) {
      c0.SetBreak( BreakOn );
    }
  }
}
	
int  Variable::addArrayDeclaration( ArrayRec* array, int array_index, int is_dimension_name, int global_index ) {
  if( array == NULL ) {  return -1; }
  if( array_index > kMaxNVarArrays ) {
	sprintf(gMsgStr, " Too many Arrays-%d added to Variable %s:%s: %s(%d)", array_index, fModule->Name(),  Name(), array->Name(), global_index ); 
	gFatal(); 	
  }
  if( _Array_Index_Map[array_index] < 0 ) {
	_Arrays[array_index] = array;
	_Array_Index_Map[array_index] = global_index;
	if( gDebug ) {
	  sprintf(gMsgStr, " Add Array-%d to Variable %s:%s: %s(%d)", array_index, fModule->Name(),  Name(), array->Name(), global_index ); 
	  gPrintLog(); 	
	}
	return 1;
  }
  return 0;
}

int Variable::getLocalArrayIndex( int index_history_array[4], int global_index, int iarg ) {
	for( int j=0; j<4; j++ ) {
	  if( global_index == _Array_Index_Map[j] ) {
		int unique = 1;
		for( int k=0; k<iarg; k++ ) {
		   if(  index_history_array[k] == j ) { unique = 0; }
		}
		if( unique ) {
		  index_history_array[iarg] = j;
		  if( (gDebug > 1) && (iarg == 1) ) {
			gPrintLog( format( "SetArrayIndex(%s): %d %d ",Name(),index_history_array[0],index_history_array[1] ) );
		  }
		  return j;
		}
	  }
	}
	return -1;
}  

int Variable :: ProcessSpatialDependency( Variable* v, CString& value ) { 

  if( v->GetF(FisIgnored) || GetF(FisImport) ) return 0;
  char msg[256];

  int go = False;
  if( ( GetF(FisCrossCell) || v->GetF(FisSpatial) ) && !GetF(FisSpatial) && !GetF(FOverrideInit) ) { 
		sprintf(msg," ProcessSpatialDependency-0 -> %s", v->Name() );
    SetF(FisSpatial,True,msg); 
    go = True; 
  }
  if( GetF(FisSpatial)  && GetF(FisStateVar) && GetF(FisClamped)  &&  !v->GetF(FisSpatial)  ) { 
		sprintf(msg," ProcessSpatialDependency-Clamped Neg Flux -> %s", Name() );
    v->SetF(FisSpatial,True,msg); 
    go = True; 
  }
  return go;
}

ArrayRec* Variable::getArrayDeclaration( int order ) {
  if( order >= kMaxNVarArrays ) {
	gFatal( format( "Illegal index in %s.getArrayDeclaration: %d", Name(), order ) );
  }
  return _Arrays[order];	  
}

TCommand* Variable::GetCommandByType ( TCommand::EType etype ) {
	Pix p0;
	if( etype == TCommand::kUndefined ) return GetDefaultCommand ( );
  for( p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    if( etype == c0.GetObjInfo(TCommand::kType) ) return &c0;
  }
  return NULL;
}

TCommand* Variable::GetDefaultCommand ( ) {
	Pix p0;
  for( p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    TCommand::EType type = (TCommand::EType) c0.GetObjInfo(TCommand::kType);
    if( type == TCommand::kUpdate || type == TCommand::kIntegrate ) return &c0;
  }
  for( p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    TCommand::EExType etype = (TCommand::EExType) c0.GetObjInfo(TCommand::kExType);
    if( etype == TCommand::kImport ) return &c0;
  }
  for( p0 = fCommandList.first(); p0; fCommandList.next(p0) ) {
    TCommand &c0 = (TCommand&) fCommandList(p0);
    TCommand::EType type = (TCommand::EType) c0.GetObjInfo(TCommand::kType);
    if( type == TCommand::kInit ) return &c0;
  }
  return (TCommand*)NULL;
}

void Variable::ProcessDoc( CString& docstr ) {
	const int max_lines = 250;
	if( gDebug > 1 ) {
	  gPrintLog( format("%s:%s.ProcessDoc:\n%s\n",fModule->Name(), Name(), docstr.chars() ) );
	}
	CString doc_lines[max_lines], attr_val[2], sep("\n"), eq("=");
    int n_lines = split( docstr, doc_lines, max_lines, sep);
    for( int i=0; i<n_lines; i++ ) {
	  CString& doc_line = doc_lines[i];
	  int has_attr = split( doc_line, attr_val, 2, eq);
	  if( has_attr == 2 ) {
		ProcessAttribute(attr_val[0],attr_val[1]);
	  }
    }
}

void Variable::ProcessAttribute( CString& attribute, CString& value  ) {
  attribute.downcase(); 
  value.downcase();
  if( attribute.empty() ) return;
  if( gDebug ) {
	const char* attr = attribute.chars();
	const char* aval = value.chars();
	gPrintLog( format("%s:%s.ProcessAttribute: %s = %s",fModule->Name(), Name(), attr, aval ) );
  }
  if( attribute.contains("unit") ) {
	fUnit = value;
  } else if( attribute.contains("biflow") ) {
	 if( value.contains("yes") || value.contains("true") ) {
	   setFluxTypeIndex( kF_BiFlow );
	 } else if( value.contains("no") || value.contains("false") ) {
	   setFluxTypeIndex( kF_UniFlow  );
	 }
  } else if( attribute.contains("non-negative") ) {
	 if( value.contains("yes") || value.contains("true") ) {
	   SetF(FisClamped,True,"ProcessAttribute");
	   setIntegrationMethod(1,'C');
	 } else if( value.contains("no") || value.contains("false") ) {
	   SetF(FisClamped,False,"ProcessAttribute");
	   setIntegrationMethod(1,'N');
	 }
  }
}

int Variable::ProcessGeneral( TNode* n ) {
	static int cmd_count = 0;
	switch( n->NodeType() ) {
		case TNode::kFunc: {
			TFunctionNode* fn = (TFunctionNode*)n;
			int ex_order =  fn->executionOrdering();
			CString cName = "v"; 
			switch (ex_order) {
				case  1:  cName += "E"; break;
				case -1:  cName += "B"; break;
			}
			cName.appendIndex(cmd_count++);
			if( fn->Source() == kFS_SMELib ) {
			  RemoveCommandsFromList( "u" );
			}
			TCommand* c = GetCommand( cName, TCommand::kFunction, TCommand::kCode, True );
			if(c) {
				TNode* sn = fn->SuperNode();
				if( sn->NodeType() != TNode::kEquation ) {
				  TVarNode* var = (TVarNode*) ExprBuilder::CreateVarNode(  Name()  );
				  TEquationNode* en = (TEquationNode*) ExprBuilder::CreateEquationNode( TEquationNode::kUpdate, var, fn  ); 
				  var->Super(en); 
				  en->Super(sn);
				  sn = en;
				}
				c->SetNode(sn);
				c->AddArrayNode((TEquationNode*)sn);
				c->ProcessNode();
				return 1;
			} else return 0;
		} break;
		case TNode::kEquation: {
			TEquationNode* en = (TEquationNode*)n;
			TNode* expr = (TNode*)en->Expr();
			TVarNode* var = en->Variable();
			if( var->GetObjInfo(TVarNode::kVarType) == TVarNode::kRelative ) { break; }
			fEqnNodeList.append(*expr);
			TCommand* c = NULL;
			CString cName;
			switch( en->EquationType() ) {
				case TEquationNode::kUpdate: 
					SetCInfo(kType,kAux);
					c = GetCommandByType ( TCommand::kUpdate );
					if( c == NULL ) { 
						cName = "u"; cName.appendIndex(cmd_count++);
						c = GetCommand( cName, TCommand::kUpdate, TCommand::kCode, True );
						CString& ds = (CString&)en->Doc();
						if( !ds.empty() ) { 
						  c->SetDoc( ds ); 
						  SetDoc( ds ); 
						  ProcessDoc(ds);
						}
					}
					break;
				case TEquationNode::kStateVar: 
					SetCInfo(kType,kState); 
					c = GetCommandByType ( TCommand::kIntegrate );
					if( c == NULL ) { 
						cName = "I"; cName.appendIndex(cmd_count++);
						c = GetCommand( cName, TCommand::kIntegrate, TCommand::kCode, True );
						CString& ds = (CString&)en->Doc();
						if( !ds.empty() ) { 
						  c->SetDoc( ds ); 
						  SetDoc( ds ); 
						  ProcessDoc(ds);
						}
					}
					break;
				case TEquationNode::kInit: 
					c = GetCommandByType ( TCommand::kInit );
					if( c == NULL ) { 
						cName = "i"; cName.appendIndex(cmd_count++);
						c = GetCommand( cName, TCommand::kInit, TCommand::kCode, True );
						CString& ds = (CString&)en->Doc();
						if( !ds.empty() ) { 
						  c->SetDoc( ds ); 
						  ProcessDoc(ds);
						}
					}
					break;
			}
/*
			if( expr->NodeType() == TNode::kConst ) {
				TConstNode* cn = (TConstNode*)expr;
				const ArgArray* ml = var->GetMapList();
				if( ml ) {  
					TDBParamValue* pv = new TDBParamValue( (*ml)[0], cn->StringVal() );
					AddDBParamValue(*pv);
				}
			}
*/
			if(c) {
				c->SetNode(en);
				c->AddArrayNode(en);
				c->ProcessNode();
				return 1;
			} else return 0;
		} break;
	}
}

int Variable::PrintDBParamValues( FILE* file ) {
	int cnt = 0;
	CString dbp_name;
	for(  Pix p = fDBParamValueList.first(); p; fDBParamValueList.next(p) ) {
		TDBParamValue& dbp = (TDBParamValue&)fDBParamValueList(p);
		if( fMap[0].empty() ) { 
			sprintf(gMsgStr, " PrintDBParamValues: undefined map for dbp %s:%s:%s (value: %s): probably not used in any equation, omitting from parameter file.", fModule->Name(),  Name(), dbp.Name(),(const char*)dbp.Value() ); 
			gPrintErr(); 
		} else {
			if( dbp.SName().empty() ) { dbp_name = fMap[0]; }
			else { dbp_name = dbp.SName(); }
			fprintf(file,"\n%s \t%s \t%s \t%s \t%s", fModule->Name(), Name(), (const char*)fMap[0], dbp_name.chars(), (const char*)dbp.Value() );
			cnt++;
		}
	}
	return cnt;
}

int Variable::AddDBParamValue( const CString& parm, const CString& value) {	
	TDBParamValue* pv = new TDBParamValue(parm,value);
	AddDBParamValue( *pv );
	return 0;
}

void  Variable::MakeMapInputVariable( int isInt ) {
   if( GetCInfo(kType) == kState ) {  
	 gFatal( format("Can't read a map into a State variable (%s): Use a separate aux var (converter) to initialize.",Name()) );
   }
   SetCInfo(kType,kData);
   SetCInfo(kInitMode,kMap);
   SetF(FisConstant,True,"CAuxVariable");
   SetF(FisSpatial,True,"CAuxVariable");
   if( isInt ) { fDataType = kInt; }
} 

int Variable::ProcessCommand(const TCommandNode* cn) {
	static int cmd_count = 0;
	const TDeclarationNode* dn = cn->Declaration();
	const CString& cmd = dn->Command();
	const CString& name = (dn->Name()) ? dn->Name()->Top() : gNullString;
	int isExtension = dn->IsExtension();

  if( (cmd == "Command") || (cmd == "dynamic") ) {
    TCommand* currentCmd = (isExtension) ? GetCommand(name,TCommand::kUndefined,TCommand::kNoEx,False) : GetCommand(name);
    if( currentCmd == NULL ) gFatal(name + ": Can't Find Command for Extension");
	currentCmd->AddQualifiers(dn,isExtension);   
    currentCmd->SetNode(cn->StatementList());
    currentCmd->ProcessNode();
    if( currentCmd->SName().empty() ) {
		switch( currentCmd->GetObjInfo(TCommand::kType) ) {
			case TCommand::kIntegrate:  currentCmd->SName() = "I"; break;
			case TCommand::kInit:       currentCmd->SName() = "i"; break;
			case TCommand::kUpdate:     currentCmd->SName() = "u"; break;
			default: currentCmd->SName() = "c__"; break;
		}		
		currentCmd->SName().appendIndex(cmd_count++);
    }
    return 1;
  } else return 0;
}

int Variable::ProcessAttribute(const TAttributeNode* n) { return 0; }

int Variable::RunNodeProcessing( EProcessContext pc ) {
	switch(pc) {
		case kPCDependencies: { 
			 if(fFunctionNodeList) {  ProcessNode(fFunctionNodeList);  }		
			 if( GetF(FisImport) ) {
				 Variable* cv = Connection();	
				 if( cv == NULL || cv->GetF(FisIgnored) ) SetF(FisIgnored,True,"RunNodeProcessing");
				}
		} break;	
		case kPCMMLGen:  
		break;
		case kPCCodeGen: break;	
		case kPCDerivatives: 
		break;		
	}	 
	for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
		TCommand& c = (TCommand&) fCommandList(p);
		c.RunNodeProcessing(pc);
	}
	return 0;
}

int Variable::Config( TConfigData& cd) {
  const CString& cmd = cd.Cmd();
//  fprintf(stderr,"\n  ** dbg2: process config-cmd %s: %s", Name(), cmd.chars() ); fflush(stderr);
	CString* s = 0;
	int iarg;
	
//  if( (iarg = Model::I0().FrameConfig(cd,fModule)) > 0 ) {
//		Model::I0().SetDefaultLayers(fActivationLayer,fLinkLayer);
//		return iarg;
//  }
	if( GetF(FisImport) ) {
    if( cmd == "ip" ) {
      if( cd.HasArg(0) ) fDefValue.elem(0) = *cd.Arg(0);    
      if( cd.HasArg(1) ) fDefValue.elem(1) = *cd.Arg(1);    
      if( cd.HasArg(2) ) fDefValue.elem(2) = *cd.Arg(2);    
      if( cd.HasArg(3) ) fDefValue.elem(3) = *cd.Arg(3); 
    } else return 0;
    return 1;
  }	
  float farg;
  TPipeCommand* pc = NULL;
  Pipe* pipe = Pipe::GetPipeFromConfig( cd, this, pc ); 
  if( pipe != NULL ) {
	if(pc==NULL) {
	  pc = TPipeCommand::CreatePipeCommand( cd, pipe, this );
	  AddCommand( pc );
	}
	pipe->SetPipeCommand(pc);    
  } else if( cmd == "db" ) {
    if( cd.IntArg(0, iarg, CD::kRequired) > 0 ) SetCInfo(kDebug,iarg);	
  }  else if ( cmd == "OS" || cmd == "OT" ) {
		int cnt=0;
		sprintf(gMsgStr,"\n%s: Schedule Config(%s): ",Name(),cmd.chars());
		gPrintLog();
		for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
			TCommand& c = (TCommand&) fCommandList(p);
			if( c.GetObjInfo(TCommand::kExType) == TCommand::kPipe ) {
				if( c.Config( cd ) ) cnt++;
				sprintf(gMsgStr," (%s : %d) ",c.Name(),cnt);
				gPrintLog();
			}
		}
		if( cnt == 0 ) {
			gPrintLog(" (Variable Schedule Config) ");
			if( Env::GetInfo(Env::kEventsDefined) ) {
				for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
				TCommand& c = (TCommand&) fCommandList(p);
					if( c.Config( cd ) ) cnt++;
					sprintf(gMsgStr," (%s : %d) ",c.Name(),cnt);
					gPrintLog();
				}
			}
			if(fSchedule==NULL) fSchedule = new ExSchedule(SName());
			return fSchedule->Config(cd);				
		} 
	} else if ( cmd == "S" ) {
		sprintf(gMsgStr,"\n%s: Scale Config(%s): ",Name(),cmd.chars());
		gPrintLog();
		for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
			TCommand& c = (TCommand&) fCommandList(p);
			if( c.GetObjInfo(TCommand::kExType) == TCommand::kPipe ) {
				Pipe* p = c.GetPipe(); 
				p->Config( cd );
				sprintf(gMsgStr," (%s:%s) ",c.Name(),p->Name());
				gPrintLog();
			}
		}
	} else if( cmd == "mpet" ) {
	  cd.addToString(fGOFCmds); 
#ifdef HAS_MPE
	  GOFCalc::configureVariableTest(*this,cd);
	  SetF(FcalcGOF,True,"Config:mpet");
#else
	  sprintf( gMsgStr, "Must build with MPE enabled in order to access MPE tools: %s configured with mpe() config-cmd", Name() );
	  gFatal();
#endif	
	} else if( cmd == "mpew" ) {	 
	  cd.addToString(fGOFCmds); 
#ifdef HAS_MPE
	  if( cd.FloatArg(0,farg) ) {
		GOFCalc::setVariableWeight(*this,farg);
	  }

#else
	  sprintf( gMsgStr, "Must build with MPE enabled in order to access MPE tools: %s configured with mpe() config-cmd", Name() );
	  gFatal();
#endif	
	}  else if( cmd == "mper" ) {	
	  cd.addToString(fGOFCmds); 
#ifdef HAS_MPE
	  CString filename( Env::DataPath().chars() ); filename += "/";
	  int col_index[256];
	  if( (s = cd.Arg(0)) != NULL ) {		
		filename += *s;
		int n_index = 0;
		while( cd.IntArg( n_index+1, col_index[n_index] ) > 0 ) { if( ++n_index == 256 ) break; }
		col_index[n_index] = -99;
		GOFCalc::registerReferenceData(*this,filename,col_index);
		SetF(FcalcGOF,True,"Config:mped"); 
	  }
#else
	  sprintf( gMsgStr, "Must build with MPE enabled in order to access MPE tools: %s configured with mpe() config-cmd", Name() );
	  gFatal();
#endif	
	} else if ( cmd == "cc" ) {
		if( cd.IntArg(0,iarg) ) {
		  fCoord_index = iarg;
		  SetCInfo(kType,kData);
		  SetCInfo(kInitMode,kFrameCalc);
		  SetF(FisConstant,True,"CVariable::Config");
		  SetF(FisSpatial,True,"CVariable::Config");
		  DeleteCommands( TCommand::kUndefined );
		}
  	
	} else if( cmd == "es" ) {		
			CString name("script");
			name.appendIndex( fModule->GetScriptIndex() );
			TScriptCommand* sc = new TScriptCommand(name,this);
			if( (s = cd.Arg(0)) != NULL ) sc->Script() = *s;
			fCommandList.append(*sc); 
	} else if( GetF(FisImport) && cmd == "dc" ) {
			if( !GetF(FisStateVar) ) { gPrintErr( fName + ": Dual Connections allowed for State Variables Only."); return 0; }
      SetF(FDualConnection,True,"Config");  
			return 1;   
  } else if( cmd == "B" ) { 
     fBounds.Config(cd);
     SetF( FisOptimizable, False, "Bounds Config cmd" );
		 if( GetCInfo( kDebug ) == kNoConfig ) {
			 int d = fModule->GetCInfo(Module::kDebug);
			 if(d==kNoConfig) { d = Model::I0().GetCInfo(Model::kDebug); }
			 if( (d==kNoConfig) || (d<1) ) d = 1;
			 SetCInfo( kDebug, d );
		 }
   } else if( cmd == "comment" ) {
			if( cd.HasArg(0) ) { fComment = *cd.Arg(0); }   
   } else if( cmd == "wpm" ) {
	   Model::I0().AddParameter( *this  ); 
	   fParm.setWebParm();
	   if( cd.HasArg(0) ) { fComment = *cd.Arg(0); }   
   } else if( cmd == "pm" ) {
//		 if( GetCInfo(kType) != kData ) {
			int ustep; 
			float pval[5];
			cd.FloatArg(0,pval[0],CD::kRequired );
			if( cd.FloatArg(1,pval[1]) <= 0 ) { pval[1] = pval[0]; }
			if( cd.FloatArg(2,pval[2]) <= 0 ) { pval[2] = pval[0]; }
			if( cd.IntArg(3,ustep) <= 0 ) { ustep = -1; }

//				SetF(FisSpatial,True,"CAuxVariable::Config -> spatial parameter"); 
//				SetF( FisOptimizable, False, "CAuxVariable::Config -> spatial parameter" );

			SetupParameter( pval[0], pval[1], pval[2], ustep );
			if(gDebug) { sprintf(gMsgStr,"\n%s Parm Config: pm(%f,%f,%f,%d)",(const char*)fName,pval[0],pval[1],pval[2],ustep); gPrintLog(); } 
			
				if( MakeParameterVariable() ) {   //   Override equation with parameter value.
					for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
						TCommand& c = (TCommand&) fCommandList(p);
						if( c.GetObjInfo(TCommand::kExType) == TCommand::kCode  ) {
							c.SetValue(*cd.Arg(0));
						}
					}
				} 
				if( cd.HasArg(0) ) fDefValue.elem(0) = *cd.Arg(0);    
				if( cd.HasArg(1) ) fDefValue.elem(1) = *cd.Arg(1);    
				if( cd.HasArg(2) ) fDefValue.elem(2) = *cd.Arg(2);    
				if( cd.HasArg(3) ) fDefValue.elem(3) = *cd.Arg(3);    
//		 } 
  }  else if( (cmd == "rs") || (cmd == "r")  ) {
	if( GetF(FcalcGOF) ) { return 0; }
    if( *cd.Arg(0) == "s" ) SetF(FisSpatialRand,True,"ParseConfig");
    else SetF(FisSpatialRand,False,"ParseConfig");
  } else if( cmd == "ft" ) {
	CString* s = cd.Arg(0);
	if( s ) {
	  setFluxType( *s );
	}
  } else if( cmd == "dep" ) {
    if( GetF(FisImport) ) return 0;
    if( GetF(FisStateVar) ) return 0;
		fUserDependencies.Add( *cd.Arg(0) );   
  } else if( cmd == "s" ) {
    if( GetF(FisImport) ) return 0;
    if( !GetF(FisStateVar) ) { gFatal(format("Improper State Variable config-cmd for %s",Name())); }
    CString* s = cd.Arg(0); 
    if( s ) { 
	  char c0 = (*s)[0], c1 = (*s)[1];
	  if( c0 == 'C' ) { SetF(FisClamped,True,"ParseConfig"); fIntegrationMethod(1) = 'C'; }
	  else if( c0 == 'N' ) { SetF(FisClamped,False,"ParseConfig"); fIntegrationMethod(1) = 'N'; }
	  if( isdigit(c0) ) {
		  fIntegrationMethod(0) = c0;
		  fOrder = c0 - '0';
	  }  else if( isdigit(c1) ) {
		  fIntegrationMethod(0) = c1;
		  fOrder = c1 - '0';
	  } 
	}
  } else if( cmd == "sC" ) {
    if( GetF(FisImport) || !GetF(FisStateVar) )  { 
	  gPrintErr(format("Improper State Variable config-cmd for %s",Name())); return 0; 
	}
    CString* s = cd.Arg(0); 
    if( s ) { 
	  char c0 = (*s)[0];
	  if( c0 == 'C' ) { SetF(FisClamped,True,"ParseConfig"); fIntegrationMethod(1) = 'C'; }
	  else  SetF(FisClamped,False,"ParseConfig");
	}
  } else  if( cmd == "vi" ) {   
    if( cd.IntArg(0,iarg) ) {
      fIVar = iarg;
      if( Model::I0().Ivar[fIVar] == 1 ) gProgramBreak( SName() + ": Duplicate variable index(iv).");	
      else Model::I0().Ivar[fIVar] = 1;
    }
  } else if( cmd == "i" ) {   
    SetF(FisIgnored,True,"ParseConfig");
    if( cd.IntArg(0,iarg) ) {
      fIndex = iarg;
      if( cd.NArgs() > 1) fDefValue.elem(0) = ( *cd.Arg(1) );
      else fDefValue.elem(0) = ( "0.0" );
    }
    else fIndex = 0;
  } else if ( cmd == "F" || cmd == "UF" ) {
		SetF(FisConstant,False,"CVariable:: UF Config");
		TNode* vn = ExprBuilder::CreateVarNode( Name() );  
		TListNode* ln = ExprBuilder::CreateListNode( vn ); 
		int arg = 1;
		while(1) {
		   if( cd.HasArg(++arg) ) {
				   vn = ExprBuilder::CreateVarNode( *cd.Arg(arg) ); 
				   if(ln) { ln->AddNode(vn); }
			   }
		   else break;
		} 
		TFunctionNode* fn = NULL;
		if( cd.HasArg(1) ) { 
			CString*  fn_name = cd.Arg(1);
			int fn_ordering = 0;
			if(  fn_name->index( "END^" ) == 0 ) { 
				fn_ordering = 1; 
				fn_name = new CString( fn_name->after(3) );
			}  else if(  fn_name->index( "BEG^" ) == 0 ) { 
				fn_ordering = -1; 
				fn_name = new CString( fn_name->after(3) );
			} 
			fn = (TFunctionNode*) ExprBuilder::CreateFuncNode( *fn_name, ln ); 
			fn->setExecutionOrdering(fn_ordering);
			fn->SetLibrary(*cd.Arg(0)); 
		}
		if( fn ) {
			if(fFunctionNodeList) { fFunctionNodeList->AddNode(fn); }
			else { fFunctionNodeList = ExprBuilder::CreateListNode( fn ); } 
			if( cmd == "UF" ) { 
				fn->SetFunctionSource(kFS_UserLib); 
			} else { 
				fn->SetFunctionSource(kFS_SMELib); 
			}
		}
  } else if( cmd == "oi" ) {   
      if( cd.IntArg(0,iarg) ) { 
		SetF( FisConstant, iarg, "ParseConfig"); 
		SetF( FOverrideInit, True, "ParseConfig"); 
		if( cd.IntArg(1,iarg) ) { 
		  SetF( FisSpatial, iarg, "ParseConfig");   
		}
	  }
  } else if( cmd == "EL" ) {
	int type;
	CString *ext_mod_name = cd.Arg(0);
	CString *ext_var_name = cd.Arg(1);
	CString *modality = cd.Arg(2);
    if( cd.IntArg(0,iarg) && ( ext_mod_name != NULL ) && ( ext_var_name != NULL ) ) {
	  EVarLink* link = NULL;
	  if( (link = Externals::register_variable_link( ext_mod_name->chars(), ext_var_name->chars(), this, modality )) != NULL ) {
		MakeExternalIOVariable( link->getModality() );
	  }
	}
  } else if( cmd == "AL" ) {
    if( cd.IntArg(0,iarg) ) fActivationLayer = iarg;
    if( cd.IntArg(1,iarg) ) fLinkLayer = iarg;
  } else if( cmd == "Ar" ) {
	Module* mod = GetModule();
	int mod_ignorable = ( mod->SName().index("XXXImports") == 0 );
	if( mod_ignorable ) {
	  sprintf(gMsgStr,"Array declared in ignorable module: %s", mod->Name()  ); gFatal();
	}
    CString *s0, *s1;
    ArgArray* args = new ArgArray();
	if( (s0 = cd.Arg(0)) != NULL ) {
	  int aindex=1;
	  while( (s1 = cd.Arg(aindex++)) != NULL ) { s1->upcase(); args->Add( *s1 ); }
	  fArrayRec = new ArrayRec( *s0, args, this );
	  Model::I0().addArrayDeclaration( fArrayRec  ); 
	  sprintf(gMsgStr,"Adding Array declaration %s", s0->chars() );  gPrintScreen();
    }
  } else return 0;
  return 1;
}

int Variable::SetupParameter(float init_val, float min_value, float max_val, int  ustep ) { 

	int dim = 0;
#ifdef HAS_MPE
	if( ustep > 0 ) {
	  dim = GOFCalc::registerParameter( *this );
	}
#endif

	fParm.Setup( init_val, min_value, max_val, ustep, dim );
	fValue = init_val;
	return dim;
}

void Variable::AddUserDeps() {
	for( int i=0; i<fUserDependencies.NArgs(); i++ ) {
    CString& vname = (CString&)fUserDependencies[i];
    Variable* v = fModule->GetVariable( vname, False );
    if( v == NULL ) {
			sprintf(gMsgStr,"Can't find variable %s in Module %s: ", vname.chars(), fModule->Name() );
			gPrintErr();
    } else {
			TCommand* c = GetDynamicCommand(); 
			if( c == NULL ) {
				sprintf(gMsgStr,"Can't find dynamic command for variable %s: ", Name() );
				gPrintErr();
			}	else {
				c->AddDependency( *v );
				sprintf(gMsgStr,"Adding dependency %s to command %s:%s in module %s", v->Name(), Name(), c->Name(), fModule->Name() );
				gPrintScreen();
			}
    } 
	}   
}
  
TCommand* Variable::GetDynamicCommand() {
	if( fDynCommand == NULL ) {
		int count = 0;  
		for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
			TCommand& c = (TCommand&) fCommandList(p);
			TCommand::EType type = (TCommand::EType) c.GetObjInfo(TCommand::kType);
			if( (type == TCommand::kUpdate) ||  (type == TCommand::kIntegrate) ) {
				fDynCommand = &c;
				count++;
			}
		}
//		if( count != 1 ) { 
//			sprintf(gMsgStr, " Can't find unique dynamic cmd ( found %d cmds ) for %s! ", count, Name() ); 
//			gPrintErr(); 
//		}
	}
	return fDynCommand;
}

Variable* Variable::CreateDerivativeVariable( TDerivative* d ) {
	Variable *v = this;
	int create_import = 0;
	if( GetF(FisImport) ) {
		if( fConnection != NULL ) { v = fConnection; create_import = 1; }
		else { sprintf(gMsgStr,"Unconnected Variable: %s", Name() ), gPrintErr(); }
	}
	Pix p0 = d->AddInitiator( v );
	if( d->initiator_marked(p0,TDerivative::kProcessed) == 0 ) {
		d->mark_initiator(p0,TDerivative::kProcessed);
		int index = d->Index();
		TCommand* cmd = v->GetDynamicCommand();
		if( cmd == NULL ) {
			sprintf(gMsgStr, " Can't create derivative var d-%s / d-%s: no dynamic cmds! ", Name(), d->Name() ); 
			gPrintErr(); 
			return NULL;
		}
		TNode* node = cmd->GetNode();
		TNode::EDependencyStatus dtype = node->hasDependency( d );
		if( dtype == TNode::kNoDependency ) { 
			sprintf(gMsgStr, " Variable %s does not depend on %s ", Name(), d->Name() ); 
			gPrintErr(); 
		} else if( dtype == TNode::kUndifferentiable ) { 
			d->mark_initiator(p0,TDerivative::kUndifferentiable);
			sprintf(gMsgStr, " Variable %s is not differentiable wrt %s ", Name(), d->Name() ); 
			gPrintErr(); 
		} else  { 
			d->mark_initiator(p0,TDerivative::kInDirectDep);
			CString dname("S_"); (((dname += SName() ) += "_" ) += d->SName() ); 
			sprintf(gMsgStr, " Generating derivative of %s wrt %s: %s ", Name(), d->Name(), dname.chars() ); 
			gPrintScreen();
			Variable* newVar =  v->GetModule()->GetVariable(dname);
			if( GetCInfo(kType) == kState ) { 
				newVar->SetF(FisStateVar,True,"CreateDerivativeVariable"); 
				newVar->SetCInfo(kType,kState); 
			} else {
				newVar->SetCInfo(kType,kAux);
			}
			dname(0) = 'd';
			TCommand* newCmd = newVar->GetCommand(dname);
			newCmd->SetObjInfo(TCommand::kExType,TCommand::kCode); 
			newCmd->SetObjInfo(TCommand::kType,TCommand::kDerivative);
			newCmd->SetNode(node);
			newCmd->Derivative() = d;
			TNamedObjectList& dlist = cmd->DependencyList();
			for ( Pix p =  dlist.first(); p; dlist.next(p) ) {
				TDependency& d = (TDependency&) dlist(p);
				Variable* depVar = d.GetVar();
				if( depVar &&  !depVar->GetF(FisIgnored) ) { 
					TDependency* d = new TDependency(*depVar,0);
					TNamedObjectList& dlist1 = newCmd->DependencyList();
					dlist1.append(*d);
				}
			}
			if(create_import) {
				Variable* iv = fModule->CreateInputVariable(newVar,"C6_");	
				return iv;				 	
			}
			return newVar;		
		}
	}
	return NULL;
}


void Variable::WriteDataToConfig( CStream& outStream ) const {

	const ExSchedule* ces = NULL, *es = NULL, *s = ScheduleC();
	if( s ) s->WriteDataToConfig( outStream );
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    const TCommand& c = (const TCommand&) fCommandList(p);
    if( c.GetObjInfo(TCommand::kExType) == TCommand::kPipe ) {
      const Pipe* pipe = ((TPipeCommand&)c).GetPipe();
			Pipe::EType ptype = (Pipe::EType) pipe->GetObjInfo(Pipe::kType);
			if( ptype == Pipe::kTimeSeriesInput || ptype == Pipe::kPtTimeSeriesInput ) { ; }
			else {
				es = c.ScheduleC();
				if( ces && ( (es == NULL) || !(*ces == *es) ) ) {
					ces->WriteDataToConfig( outStream );
				}
				ces = es;
			}
			pipe->WriteDataToConfig( outStream );
    }
  }
  if( es ) es->WriteDataToConfig( outStream );
  if( GetF(FisImport) && GetF(FDualConnection) ) {
      outStream << " dc() ";  
   }
   outStream << fGOFCmds; 
  if( GetF(FisImport) ) {
    outStream << " ip("; int l = fDefValue.NArgs()-1;
    for( int i=0; i < l; i++ ) outStream << fDefValue[i].chars() << "," ;
    if(l>=0) outStream <<  fDefValue[l].chars() << ")";
    else outStream << ") ";   
    return;
  }
  
	if( fArrayRec != NULL ) {
	  fArrayRec->WriteDataToConfig( outStream );
	}

  if(GetF(FisIgnored)) {
    outStream << " i(" << fIndex << ")";
    return;
  }
  //  if( GetF(FhasReport) ) outStream << ," R(1) ");

  if( fIVar > 0 ) {
    outStream << "  vi(" << fIVar << ")";
  }
  if( GetF(FisSpatialRand) ) {
      outStream << " r(s) "; 
  }
  if( GetF(FisStateVar) ) {  
	outStream << " s(" << fIntegrationMethod[0] << ")";  
    if( fIntegrationMethod[1] == 'C' ) {  outStream << " sC(" << fIntegrationMethod[1] << ")";  }
  }

  fBounds.WriteDataToConfig( outStream ); 

  if( GetCInfo(kInitMode) == kParameter ) {
    outStream << " pm("; int l = fDefValue.NArgs()-1;
    for( int i=0; i < l; i++ ) outStream << fDefValue[i].chars() << ",";
    outStream << fDefValue[l].chars() << ")";
  }
  
  if( fCoord_index < 255 ) {
    outStream << " cc(" << fCoord_index << ")";
  }
  
  if( fFunctionNodeList ) {
		const TOrderedObjectDLList& nl = fFunctionNodeList->List();
		for( Pix p = nl.first(); p; nl.next(p) ) {
			TFunctionNode& fn = (TFunctionNode&) nl(p);
			WriteFunctionToConfig(outStream,fn);		
		}
	} 
	
	for( int i=0; i<fUserDependencies.NArgs(); i++ ) {
		outStream << " dep(" << fUserDependencies[i].chars() << ")";
	}
	
	Externals::WriteDataToConfig( outStream, (Variable*)this );
	
	if( !fComment.empty() ) { outStream << " | " << fComment.chars(); }
	
	if(  GetCInfo(kType) == kFlux ) {
	  outStream << " ft(" <<   getFluxType() << ")";  
	}
}

void Variable::WriteFunctionToConfig( CStream& outStream, TFunctionNode& fn ) const {
	if( fn.Source() == kFS_SMELib ) outStream << " F( ";
	else { outStream << " UF( "; }
	CString s;
	fn.WriteConfigList(s,1);
	outStream << s << ")";	 
}

TCommand* Variable::GetCommand( const  CString&  name, TCommand::EType type, TCommand::EExType etype, Bool create )
{
  TCommand* aCommand = (TCommand*) fCommandList.GetObjectByName( name );
  if( aCommand == NULL && create ) {
	TCommand::EStatus cmd_status = TCommand::kActive;
    switch (etype) {
			case TCommand::kPipe:
				aCommand = new TPipeCommand( name, this );
				break;
			case TCommand::kImport:
				aCommand = new TImportCommand( name, this );
				break;
			case TCommand::kCode: { 
				switch (type) {
					case TCommand::kIntegrate:
					case TCommand::kInit:
					case TCommand::kUpdate:
					case TCommand::kFunction:
						aCommand = new TExecuteCommand( name, this, type );
						if( GetCInfo(kInitMode) == kExternalInput ) {
						  EVarLink* evl = Externals::get_VarLink( this );
						  if( evl != NULL ) {
							EVarLink::EModality mode = evl->getModality();  // kIn, kOut, kInOut, kInitIn, kInitOut
							if( (mode == EVarLink::kInitIn) && ( type == TCommand::kInit ) ) {
							  cmd_status = TCommand::kInActive; 
							}
							if( (mode == EVarLink::kIn) && (( type == TCommand::kUpdate ) || ( type == TCommand::kIntegrate ) ) ) {
							  cmd_status = TCommand::kInActive; 
							}
						  }
						}
						break;
					default:
						aCommand = new TCommand( name, this, type );			
				}	
			} break;
			default:
				aCommand = new TCommand( name, this );
    }
    fCommandList.Insert(*aCommand);
    fModule->AddCommand(aCommand);	
	aCommand->SetObjInfo( TCommand::kStatus, cmd_status );
  }
  return aCommand;
}

Pix Variable::AddCommand( TCommand* c )
{
  TCommand* aCommand = (TCommand*) fCommandList.GetObjectByName( c->Name() );
  if( aCommand != NULL ) { sprintf(gMsgStr," Variable %s: Multiple Commands with same Name: %s", Name(), c->Name() ); gPrintErr(); }
  return fCommandList.append(*c);
}

int Variable::DeleteCommands( TCommand::EType t )
{
	int cnt = 0;
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    TCommand& c = (TCommand&) fCommandList(p);
    if( t == TCommand::kUndefined || c.GetObjInfo(TCommand::kType) == t ) {
			cnt++;
			if( c.NPost() == 0 ) fCommandList.del(p,1,True);
			else { 
				fCommandList.del(p);
				c.SetObjInfo(TCommand::kStatus,TCommand::kInActive);
			}
		}
	}
	return cnt;
}

int Variable::RemoveCommandsFromList( char* cname ){
	int cnt = 0, slen = strlen(cname);
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    TCommand& c = (TCommand&) fCommandList(p);
    if( strncmp(c.Name(),cname,slen) == 0 ) {
			cnt++;
			fCommandList.del(p);
			c.SetObjInfo(TCommand::kStatus,TCommand::kInActive);
		}
	}
	return cnt;
}

int Variable::Finalize()
{
	int rv = 0;
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    TCommand& c = (TCommand&) fCommandList(p);
    rv += c.Finalize();
	}
	return rv;
}

int Variable::DeleteCommand( const CString& name )
{
  Pix p  = fCommandList.GetObjectByNameP( name );
  if( p != 0 ) { 
		TCommand& c = (TCommand&) fCommandList(p);
		if( c.NPost() == 0 ) fCommandList.del( p,1,True);
		else { 
			fCommandList.del(p);
			c.SetObjInfo(TCommand::kStatus,TCommand::kInActive);
		}
	}
	return 0;
}

TPipeCommand* Variable::ProcessPipes( Pipe::EType t, EProcessAction pa )
{
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    TCommand& c = (TCommand&) fCommandList(p);
    if( c.GetObjInfo(TCommand::kExType) == TCommand::kPipe ) {
      const Pipe* pipe = ((TPipeCommand&)c).GetPipe();
			if( t == Pipe::kUndefined || pipe->GetObjInfo(Pipe::kType) == t ) { 
				switch(pa) {
					case kDelete:
						if( c.NPost() == 0 ) { fCommandList.del(p,1,True); }
						else {
							fCommandList.del(p); 
							c.SetObjInfo(TCommand::kStatus,TCommand::kInActive);
						} 
						return (TPipeCommand*)&c;
					case kReturn:
						return (TPipeCommand*)&c;
				}
			}
		}
	}
	return (TPipeCommand*)NULL;
}

int Variable::AddModifier( const CString& mod, Bool isExtension ) {
  int rv = 0;
  if( SetType( mod, isExtension ) ) { rv = 1; }
  if(  mod.contains("input") ) { 
		SetCInfo(kMode,kInput); 
		SetF(FisImport,True,"AddModifier"); rv = 1; 
	}
  return rv;
}

int Variable::GetModifierString( CString& mod ) const {
    mod += kTypeList[ GetCInfo(kType) ];
    if( GetCInfo(kMode) == kInput ) mod += " input ";
    return 1;
}

int Variable::GetXMLAttributes( CString& mod ) const {
		mod=" type=\"";
    mod += kTypeList[ GetCInfo(kType) ];
    mod+="\" ";
    if( GetCInfo(kMode) == kInput ) {
			mod += " mode=\"input\" ";
		}
    return 1;
}


  //----------------------------------------------------------------------------------------
  // TVariable::WriteCCode: 
  //----------------------------------------------------------------------------------------

void Variable::WriteCCode( FILE* CHdrFile, FILE* CEqnFile) {
  int specType = 0;
		switch( GetCInfo(kType) ) {
		case kAux: 
			fprintf(CHdrFile,"\n\tCAuxVariable %s;",Name());
			fprintf(CEqnFile,"\n\t%s(\"%s\"),",Name(),Name());
			break;
		 case kTime: 
			fprintf(CHdrFile,"\n\tTTime %s;",Name());
			fprintf(CEqnFile,"\n\t%s(\"%s\"),",Name(),Name());
			break;
		case kFlux: 
			fprintf(CHdrFile,"\n\tCFluxVariable %s;",Name());
			fprintf(CEqnFile,"\n\t%s(\"%s\"),",Name(),Name());
			break;
		case kState: 
			fprintf(CHdrFile,"\n\tCStateVariable %s;",Name());
			fprintf(CEqnFile,"\n\t%s(\"%s\"),",Name(),Name());
			break;
		default:
			fprintf(CHdrFile,"\n\tCAuxVariable %s;",Name());
			fprintf(CEqnFile,"\n\t%s(\"%s\"),",Name(),Name());
	}

}

void Variable::WriteFlags(FILE* CEqnFile) const {

  if( GetF(FisSpatialRand) ) { 
    fprintf(CEqnFile,"\n\t%s.SetF(FisSpatialRand,True,\"CG\"); %s.SetF(FisSpatial,True,\"CG\"); ", Name(), Name() ); 
  }
  if( GetF(FisImport) ) { 
    fprintf(CEqnFile,"\n\t%s.SetF(FisImport,True,\"CG\");",Name()); 
  }
  if( GetF(FisExport) ) { 
    fprintf(CEqnFile,"\n\t%s.SetF(FisExport,True,\"CG\");",Name()); 
  }
//  if( GetF(FisTIME) ) { 
//    fprintf(CEqnFile,"\n\t%s.SetF(FisTIME,True,\"CG\");",Name()); 
//  }
  if( GetF(FDualConnection) ) { 
    fprintf(CEqnFile,"\n\t%s.SetF(FDualConnection,True,\"CG\");",Name()); 
  }
//  if( GetF(FHasNegFlux) ) { 
//    fprintf(CEqnFile,"\n\t%s.SetF(FHasNegFlux,True,\"CG\");",Name()); 
//  }
  if( GetF(FisCrossCell) ) { 
    fprintf(CEqnFile,"\n\t%s.SetF(FisCrossCell,True,\"CG\");",Name()); 
  }
}

void Variable::WriteCCommandCode(FILE* CEqnFile) {
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    TCommand& c = (TCommand&) fCommandList(p);
    if( c.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) {
      c.WriteCCommandCode(CEqnFile);
    }
  }
}


void  Variable::WriteMML( FILE* oFile, EWriteMode mode )  {
	if( GetCInfo(kMode) == kIgnored ) return;
  CString mod;
  GetModifierString(mod);
  if( mode == kDeclaration ) { 
    fprintf(oFile,"\t%s Variable %s;\n",(const char*)mod,Name());
  } else if( mode == kDefinition ) {
    fprintf(oFile,"\t%s Variable %s  {\n",(const char*)mod,Name());
    for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
      TCommand& c = (TCommand&) fCommandList(p);
      c.WriteMML(oFile,kDefinition);
    }
    fprintf(oFile,"\t}\n");
  }
}

void  Variable::WriteXML( FILE* oFile ) {
	if( GetCInfo(kMode) == kIgnored ) return;
	if( GetF(FisImport ) ) {
		fprintf(oFile,"\n\n\t<port type=\"input\" name=\"%s\" >\n",Name() );
		fprintf(oFile,"\n\t\t<default> %s \n", DefaultValue(0).chars() );
		fprintf(oFile,"\n\t\t</default>\n");
		fprintf(oFile,"\n\t</port>\n");
	} else {
		CString mod;
		GetXMLAttributes(mod);
		CString status = GetF(FisExport) ? "public" : "private";
		fprintf(oFile,"\n\n\t<atom name=\"%s\" id=\"%s\" status=\"%s\" %s >\n",Name(),Name(),(const char*)status,(const char*)mod);
		TNamedObjectList& clist = CommandList();
		for( Pix cp = clist.first(); cp; clist.next(cp) ) {
			TCommand& c = (TCommand&) clist(cp);
			TNamedObjectList& dl = c.DependencyList();
			for ( Pix dP =  dl.first(); dP; dl.next(dP) ) {
				TDependency& d = (TDependency&) dl(dP);
				Variable* depVar = d.GetVar();
				if( depVar ) {
					fprintf( oFile, "\n\t\t<port type=\"input\" name=\"%s\" />", depVar->Name() );
				}	
			}
			c.WriteXML(oFile);
		}
		fprintf(oFile,"\n\t</atom>\n");
	}
}

void Variable::Dump( FILE* oFile ){
  CString mod;
  GetModifierString(mod);
  fprintf(oFile,"\n\n\t%s Variable %s {\n",(const char*)mod,Name());
  fprintf(oFile,"\t\tCInfo: { Debug: %x, Type: %x, Mode: %x }\n",GetCInfo(kDebug),GetCInfo(kType),GetCInfo(kMode));
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    TCommand& c = (TCommand&) fCommandList(p);
    c.Dump(oFile);
  }
  fprintf(oFile,"\t }\n");
}

Pipe* Variable::GetPipe( Pipe::EType type ) { 
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    Pipe* pipe = ((TCommand&)fCommandList(p)).GetPipe();
    if( pipe &&  pipe->GetObjInfo(Pipe::kType) == type ) { return pipe; }
  }
  return NULL;
}
/*
Pipe* Variable::GetPipe( Pipe* pipe0 ) { 
  int foundPipe = ( pipe0 == NULL ); 
  for( Pix p = fCommandList.first(); p; fCommandList.next(p) ) {
    Pipe* pipe = ((TCommand&)fCommandList(p)).GetPipe();
    if( pipe ) {
      if( foundPipe ) return pipe;
      if( pipe == pipe0 ) { foundPipe == 1; }
    }
  }
  return NULL;
}
*/
void Variable::SetF(FlagType  mask, int value, const char* comment ) { 
  if(value) fFlags |= mask; else fFlags &= ~mask;
  if( gDebug>1 ) {
    CString s(" ");
    GetFlagNames(mask,s);
    if( fModule ) {
      sprintf(gMsgStr,"\nSet Flag(s) (%s) for %s (%s) to %x in %s, flags = %x", 
		       (const char*)s, Name(), fModule->Name(), value, comment, fFlags); 
    } else {
      sprintf(gMsgStr,"\nSet Flag(s) (%s) for %s to %x (%s) flags = %x", 
		       (const char*)s, Name(), value, comment, fFlags); 
    }
    gPrintLog();
  }
}


void Variable::AddFlag( char* name, FlagType& flag ) {
  if(fFlagList==NULL) gFatal("Error, No FlagList Allocated\n");
  if( fCurrentFlag == 0 ) { gPrintErr("Too Many Flags Declared."); return; }
  flag = fCurrentFlag;
  if( fCurrentFlag == ULONG_MAX ) fCurrentFlag = 0;
  else fCurrentFlag *= 2;
  FlagRef* fr = new FlagRef(name,flag); 
  fFlagList->append(*fr);
}

void  Variable::GetFlagNames ( FlagType  mask, CString& flagNames )
{
  if( fFlagList == NULL ) return;
  for(  Pix p = fFlagList->first(); p; fFlagList->next(p) ) { 
    FlagRef& fr = (FlagRef&) (*fFlagList)(p);
    if( fr.Flag() & mask ) { 
      flagNames += fr.Name();
    }		
  }
}

void  Variable::PrintFlags() const
{
  sprintf(gMsgStr,"\tFlags: %x (",fFlags); gPrintLog();
  for(  Pix p = fFlagList->first(); p; fFlagList->next(p) ) { 
    FlagRef& fr = (FlagRef&) (*fFlagList)(p);	
    if( fr.Flag() & fFlags ) { 
      sprintf( gMsgStr, " %s ", fr.Name()); 
      gPrintLog();
    }
  }
}

  //----------------------------------------------------------------------------------------
  //				Variable::RegisterPipe:
  //----------------------------------------------------------------------------------------


int  Variable::RegisterPipe( Pipe& p ) {

  switch( p.GetObjInfo( Pipe::kType ) ) {
  case Pipe::kMapArchive:
    break;
  case Pipe::kAnim:
    break;
  case Pipe::kSheet:
   break;
  case Pipe::kTimeSeriesInput:
    if( (GetCInfo(kType) != kFlux) && (GetCInfo(kType) != kAux) && (GetCInfo(kType) != kData) ) {
      sprintf( gMsgStr," Config Error: %s(%x) -> Only Auxiliary Variables can take Timeseries inputs.", Name(), GetCInfo(kType) );
      gFatal(); 
    }
    SetF( FisConstant,False,"RegisterPipe");
    SetCInfo(kInitMode, kTimeSeries); 
    break;
  case Pipe::kPtTimeSeriesInput:
    if( (GetCInfo(kType) != kFlux) && (GetCInfo(kType) != kAux) && (GetCInfo(kType) != kData) ) {
      sprintf( gMsgStr," Config Error: %s(%x) -> Only Auxiliary Variables can take PTS inputs.", Name(), GetCInfo(kType) );
      gFatal(); 
    };
    SetF( FisConstant,False,"RegisterPipe");
    SetF(FisSpatial,True,"RegisterPipe.1");
    SetCInfo(kInitMode, kPtTimeSeries); 
    break;  
  case Pipe::kDBaseInput: {
    if( (GetCInfo(kType) != kFlux) && (GetCInfo(kType) != kAux) && (GetCInfo(kType) != kData) ) {
      sprintf( gMsgStr," Config Error: %s(%x) -> Only Auxiliary Variables can take DBase inputs.", Name(), GetCInfo(kType) );
      gFatal(); 
    }
    DBaseInputPipe& dbp = (DBaseInputPipe&) p;
    SetF(FisSpatial,True,"RegisterPipe.2");
    SetF(FisConstant,True,"RegisterPipe");
    AddMapInputs( dbp );
    SetCInfo(kInitMode, kDBase);
    } break;
  case Pipe::kMapInput:
//    if( GetCInfo(kType) != kAux && GetCInfo(kType) != kData) {
 //     sprintf( gMsgStr," Config Error: %s(%x) -> Only Auxiliary Variables can take Map inputs.", Name(), GetCInfo(kType) );
//      gFatal(); 
//    }
    SetCInfo(kInitMode, kMap);
    if( p.GetObjInfo(MapInputPipe::kMapClass) == kClassMap ) {
      fDataType = kInt; 
    } else {
      fDataType = kFloat;
    }
    SetF(FisConstant,True,"RegisterPipe"); 
    SetF( FisSpatial,True,"RegisterPipe.3" );
    break;
  }
	return 0;
}

int Variable::MakeParameterVariable( ) {
	int rv = 1;
	if( GetCInfo(kInitMode) == kMap ) { 
	  sprintf(gMsgStr,"Can't override map declaration for %s with parameter value, pm() config-cmd ignored", Name() );  gPrintErr(); 
	  return 0; 
	}
	if( ( GetCInfo(kInitMode) != kDefault ) && ( GetCInfo(kInitMode) != kParameter ) ) { 
		sprintf(gMsgStr,"Overriding declaration (%x) for %s with parameter value", GetCInfo(kInitMode), Name() );  gPrintErr(); 
	}
	SetCInfo(kInitMode,kParameter);
	SetCInfo(kType,kConstant);
	SetF(FisConstant,True,"CAuxVariable");
	SetF(FisSpatial,False,"CAuxVariable");
	return rv;
}

  //----------------------------------------------------------------------------------------
  // Variable::MakeConnection: 
  //----------------------------------------------------------------------------------------
  
void Variable::MakeConnection( Variable* cVar, Bool dualConnection ) {
  fConnection = cVar;
  SetF(FisImport,True, "MakeConnection");
  if( cVar ) {
    cVar->fConnectionList.append(*this);
    cVar->SetF( FisExport, True, "MakeConnection" );
  }
  if( dualConnection ) { 
		SetF( FDualConnection, True, "MakeConnection" );
		cVar->fConnection = this;
	}
}

void Variable::PrintDependencies( Pix p0, int depth, TNamedObjectList& depTree ) const {
	TOrderedObjectDLListNode* n0 = (TOrderedObjectDLListNode*) p0;
	CString intro("");
	byte processed = 250;
	for(int i=0; i<depth; i++ ) { intro += "  "; }
	if( n0->test_info( processed ) ) { 
		sprintf( gMsgStr, "%s(%d) %s  **", intro.chars(), depth, Name()  ); gPrintLog(); 
	} else { 
		sprintf( gMsgStr, "%s(%d) %s ->", intro.chars(), depth, Name() ); gPrintLog();
		n0->set_info( processed );
		DLLinkIterator id(p0);
		for ( Pix p =  id.first(); p; p= id.next() ) {
			Variable* v = (Variable*)&depTree(p);
			v->PrintDependencies( p, depth+1, depTree ); 
		}
	}
}

  //----------------------------------------------------------------------------------------
  // Variable::AddMapInputs: 
  //----------------------------------------------------------------------------------------

void Variable::CheckMapInputs( ) 
{
  Variable* aVar = (Variable*) fModule->VarList().GetObjectByName(fMap[0]);
  if( aVar == NULL ) { gPrintErr( fMap[0] + ": can't find class map for MapDep Parm." ); }
	if( aVar->GetCInfo(Variable::kInitMode) != Variable::kMap  ) { 
		gPrintErr( fMap[0] + ": wrong type (need class map) for MapDep Parm." ); 
	} 
}

void Variable::AddMapInputs( DBaseInputPipe& dbp ) 
{
	static CString mapvar; mapvar = dbp.Map(0); 
  TNamedObjectList& iter = Model::I0().ModuleList(); 
  Module* aModule = fModule; 	   
  Variable* aVar = (Variable*) fModule->VarList().GetObjectByName(mapvar);
  fMap[0] = mapvar;
  if(aVar == NULL) {
	  aVar= (Variable*) Model::I0().GetVariable(mapvar);    //    GetObjectByName(mapvar);
  }  
  if(aVar == NULL) {
    CString report("Unknown mapdep in parm: "); report += mapvar;
    gProgramBreak(report);
  }
  else if(aModule != fModule) {
    aVar = fModule->CreateInputVariable(aVar,"c_MI_");
  }
  if(aVar) {
    dbp.SetMap(0,aVar);	    
    Variable* v = (fMapVariable[0]=aVar);
    if( Env::GetInfo(Env::kEventsDefined) ) { 
		if( v->GetF(FisImport) ) {  v = v->Connection(); } 
		if( v->GetCInfo(Variable::kInitMode) != Variable::kMap  ) { 
			gPrintErr( fMap[0] + ": wrong type (need class map) for MapDep Parm." ); 
		} 
	}
  }
  if ( dbp.GetObjInfo(DBaseInputPipe::kOrder) == 2 ) {
		mapvar = dbp.Map(1);
    aVar = (Variable*) aModule->VarList().GetObjectByName(mapvar);
    fMap[1] = mapvar;
    if(aVar == NULL) {
      for( Pix p = iter.first(); p; iter.next(p) ) {
				if( (aModule = (Module*) &iter(p)) != fModule) {
					aVar= (Variable*) aModule->VarList().GetObjectByName(mapvar);
					if(aVar && !aVar->GetF(FisImport)) break;
				}
      }
    }  
    if(aVar == NULL) {
      CString report("Unknown mapdep in parm: "); report += mapvar;
      gProgramBreak(report);
    }
    else if(aModule != fModule) {
      aVar = fModule->CreateInputVariable(aVar,"c_MI2_");
    }	
    if(aVar) {
      dbp.SetMap(1,aVar);
      fMapVariable[1] = aVar;
    }				    
  }
}

void Variable::Init( Module* mod ) {
  fModule = mod; 
  Model::I0().AddVariable(this);
  mod->AddVariable(this); 
}


 //----------------------------------------------------------------------------------------
//						TParmRec
//----------------------------------------------------------------------------------------       
       
TParmRec::TParmRec() {
  fUStep = 0; fInitialized = 0; fWebParm = 0; fIncrement = 0.0; 
}
     
void TParmRec::Setup(float init_value, float min_value, float max_value, int ustep, int dim  ) {
  fValue = init_value;
  fMax_Value = max_value;
  fMin_Value = min_value;
  if( ustep > 0 ) {
	fIncrement = (max_value - min_value)/ustep;
  }
  fDim = dim;
  fUStep = ustep;
  fInitialized = 1;
}

TParmRec& TParmRec::operator = ( TParmRec& p ) {
  fValue = p.fValue;
  fMax_Value = p.fMax_Value;
  fMin_Value = p.fMin_Value;
  fDim = p.fDim;
  fUStep = 0;
  return *this;
}
    
float TParmRec::Value( float f0, float f1, float f2 ) const {
  float rv = fValue;
  switch( fDim ) {
		case 0: rv += f0*(fMax_Value-fValue); break;
		case 1: rv += f1*(fMax_Value-fValue); break;
		case 2: rv += f2*(fMax_Value-fValue); break;
	}
  return rv;
}

float TParmRec::GetSensitivity(float& incr) {
#ifdef HAS_MPE
  incr = fIncrement * GOFCalc::parmIncrementWeight();
  int pindex = GOFCalc::getParameterIndex(fDim);
  float g0 = GOFCalc::getGOFSensitivity(pindex);
  float g1 = GOFCalc::getGOFSensitivity(pindex+1);
  if( g0 > g1 ) {
	float value = fValue - incr;
	if(value < fMin_Value) value = fMin_Value;
	return (g0/((fValue - value)/fValue));
  } else {
	float value = fValue + incr;
	if(value > fMax_Value) value = fMax_Value;
	return (g0/((value - fValue)/fValue));
  }
#endif
  return 0.0;
}
	
float TParmRec::GetValue( int index ) {
  int pspace_index = 1;
  float value, incr = fIncrement;
#ifdef HAS_MPE
  pspace_index = GOFCalc::getPSpaceCoordinate( index, fDim );
  incr *= GOFCalc::parmIncrementWeight();
#endif
  switch( pspace_index ) {
	case 0: 
	  value = fValue - incr; 
	break;
	case 1: 
	  value = fValue;
	break;
	case 2: 
	  value = fValue + incr;
	break;
  } 
  return ((value < fMin_Value) ? fMin_Value : ( (value > fMax_Value) ? fMax_Value : value )); 
}

float TParmRec::GetValue( float prop_value ) {
  if(  ( prop_value < 0 )  || (prop_value > 1) ) { gFatal( format("Illegal argument to TParmRec::GetValue: %f", prop_value) ); }
  return fMin_Value + (fMax_Value - fMin_Value)*prop_value; 
}


float TParmRec::GetValue( int index, int& pspace_index ) {
  float value, incr = fIncrement;
#ifdef HAS_MPE
  pspace_index = GOFCalc::getPSpaceCoordinate( index, fDim );
  incr *= GOFCalc::parmIncrementWeight();
#endif
  switch( pspace_index ) {
	case 0: 
	  value = fValue - incr; 
	break;
	case 1: 
	  value = fValue;
	break;
	case 2: 
	  value = fValue + incr;
	break;
  } 
  return ((value < fMin_Value) ? fMin_Value : ( (value > fMax_Value) ? fMax_Value : value )); 
}




