//----------------------------------------------------------------------------------------
//      Variable.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History
//----------------------------------------------------------------------------------------

#ifndef __MML_Variable__
#define __MML_Variable__
  
#include "Utilities.h"
#include "SList.h"
#include "StringPlex.h"
#include "ConfigObjects.h"
#include "CStream.h"
#include "Bounds.h"
#include "MML.h"
#include "MML_Module.h"
#include "MML_Event.h"
#include "MML_Model.h"
#include "ExprBuilder.h"
#include "Externals.h"

typedef class TGraph TGraph;
typedef class Pipe Pipe;
typedef class TEvent TEvent;
typedef class TCommand TCommand;
typedef class TConnectionList TConnectionList;
typedef class TConnection TConnection;
typedef class Module Module;
typedef class TVarNode TVarNode;
typedef class ArrayRec ArrayRec;

typedef unsigned long FlagType;

class TDBParamValue : public TNamedObject {
	CString fValue;
public:

	TDBParamValue( const CString& attr,  const CString& value ) : TNamedObject(attr) { fValue = value; }
	const CString& Value() { return fValue; } 
};

//----------------------------------------------------------------------------------------
  //						TParmRec
  //----------------------------------------------------------------------------------------
  
class TParmRec {

  float fValue;
  float fCurrent_Value;
  float fMax_Value;
  float fMin_Value;
  float fIncrement;
  int fDim;
  int fUStep;
  byte fInitialized;
  byte fWebParm;

public:

  TParmRec();
  void Setup( float init_value, float min_value, float max_value, int ustep = -1, int dim = -1  );
  float Value( float f0=0.0, float f1=0.0, float f2=0.0 ) const;
  inline float CurrentValue( int index )  { return ( fInitialized > 1 ) ? fCurrent_Value : Initialize(index); }
  float GetValue( int index );
  float GetValue( int index, int& pspace_index );
  float GetValue( float prop_value );
  inline float GetValue() { return fValue; }
  float GetSensitivity(float& incr); 
  TParmRec& operator = ( TParmRec& p );
  inline int Configured() { return( fInitialized > 0 ); }
  inline int Initialized() { return( fInitialized > 1 ); }
  inline int ReSet() { fInitialized = 0; }
  inline float SetValue( float value ) { return (fValue = value); }
  inline float SetPropValue( float prop_value ) { return (fValue = GetValue( prop_value )); }
  inline float SetValue( int index ) { return (fValue = GetValue( index )); }
  inline float Initialize(int index) { fInitialized = 2; return (fCurrent_Value = GetValue(index));  }
  inline float Max_Value() { return fMax_Value; }
  inline float Min_Value() { return fMin_Value; }
  inline float Increment() { return fIncrement; }
  inline int   Step() { return fUStep; }
  inline byte isWebParm() { return fWebParm; }
  inline void setWebParm( byte val = 1 ) { fWebParm = val; }
};
  
//----------------------------------------------------------------------------------------
//				Variable
//----------------------------------------------------------------------------------------

 extern FlagType	FisClamped ;		//     is Clamped Flux or State Var
 extern FlagType	FisConstant ;		// 	0 = Varies Temporially,  1 = constant     
 extern FlagType	FisSpatial ;		// 	is a spatial Variable	 
 extern FlagType	FisIgnored ;		// 	0 = Varies Temporially,  1 = constant     
 extern FlagType	FisStateVar ;		// 	1 = State variable, 0 = Aux/Flux Var
 extern FlagType	FisFlux ;		//  1 = is argument to flux function, 0 = is not  
 extern FlagType	FisImport ;		// 	is Module Input 	  
 extern FlagType	FisExport ;		//  is Outgoing Flux from Clamped State Var
 extern FlagType	FRegistered ;		//  Reqistered with Interface
 extern FlagType	FVisible ;		//  Visible in Interface Display
// extern FlagType	FisTIME ;		//  Time Var
 extern FlagType	FDualConnection;  //  Two-way connection
 extern FlagType	FisSpatialRand ;		//  1 = spatial Random Number Generator
// extern FlagType	FHasNegFlux ;		//  is Outgoing Flux from Clamped State Var
 extern FlagType	FLinkEdges ;		//  1 -> Link GhostRegions after each update
 extern FlagType	FisCrossCell ;		// 	1 = Cross Cell Variable
 extern FlagType	FisOptimizable ;     
 extern FlagType	FHasDelay ;
 extern FlagType	FisFinalUpdate;     
 extern FlagType	FcalcGOF;     
 extern FlagType	FOverrideInit;     
 extern FlagType	FisDisplayVar ;		//  Display Var

#define kMaxNVarDBMaps 2
#define kMaxNVarArrays 8

enum EFluxInfo { kF_FluxType, kF_OriginClamped, kF_DestinationClamped };
enum EFluxType { kF_Unknown, kF_UniFlow, kF_BiFlow };


/*
class ArrayRec : public TNamedObject {
  ArrayRec* _ar;
  int _is_dimension_name;
  int _index;
  public:
  ArrayRec( ArrayRec* ar, int is_dimension_name, int index ) : TNamedObject(ar->Name()) {
	_is_dimension_name = is_dimension_name; _index = index;
  }

  inline ArrayRec* getArrayRec() { return _ar; }
  inline int getIndex() { return _index; }
  inline int isDimensionName() { return _is_dimension_name; }
  inline ArgArray* getArgs() { return _ar->getArgs(); }  
  inline Variable* getMap() { return _ar->getMap(); }
};
*/

class Variable : public TConfigObject, public MMLObject {
public:

  enum EDataType { kFloat, kInt, kByte };
  enum EProcessAction { kDelete, kReturn };
  enum EType { kNULL, kAux, kFlux, kState, kData, kTime, kConstant };
  enum EMode { kInternal, kInput, kIgnored };
  enum EInitMode { kDefault, kMap, kTimeSeries, kPtTimeSeries, kDBase, kParameter, kExternalInput, kFrameCalc };

// Parm fields
  TParmRec fParm;
  float fValue;
  float fData[5];
  
  static EConfigInfo kDebug;
  static EConfigInfo kType;
  static EConfigInfo kMode;
  static EConfigInfo kInitMode;
  static CString kGarbage;

protected:

  Module* fModule;
  TNamedObjectList fCommandList;
  TNamedObjectList fDBParamValueList;
  TOrderedObjectDLList fEqnNodeList;
  ArrayRec* fArrayRec;
  FlagType fFlags;
  static FlagType fCurrentFlag;
  static TObjectList *fFlagList;
  EDataType		fDataType;
  Variable*       fConnection;
  TCommand* fDynCommand;
  TNamedObjectList fConnectionList;
  CString fMap[kMaxNVarDBMaps];
  Variable* fMapVariable[kMaxNVarDBMaps];
  TVarNode* fVarNode;
	ExSchedule* fSchedule;
  ArgArray         fDefValue; 		          // Parameter values;
  ArgArray         fUserDependencies; 		 // user-specified dependencies;
  CBounds          fBounds;
  int              fIVar;
  TNamedObjectList fFluxList;       					
  Variable*       fParent; 
  TListNode* fFunctionNodeList;
  CString fIntegrationMethod;
  CString fComment;
  int fOrder;
  int fActivationLayer;
  byte fCoord_index; 
  int fLinkLayer;
  static float fGarbage;
  Variable* fOrigin;
  Variable* fDestination;
  CString fGOFCmds;  
  ArrayRec* _Arrays[kMaxNVarArrays];
  int _Array_Index_Map[kMaxNVarArrays];
  byte fFluxInfo[4];
  CString fUnit;

  void WriteFunctionToConfig( CStream& outStream, TFunctionNode& fn ) const;

public:


  static ArgArray kTypeList;  // ( " ", "aux", "flux", "state" );

  inline int SetType( const CString& mod, Bool isExtension = False ) {
    int index = kTypeList.Find(mod);
    if( index > 0 ) {
      if( isExtension && index != GetCInfo(kType) ) gPrintErr(" Error, can't change Varaible Type in Extension.");
      else  SetCInfo(kType,index);
    }
    return index;
  }
  inline CString&  GetComment() { return fComment; }
  inline void setIntegrationMethod( int index, char data ) { fIntegrationMethod(index) = data; }
  inline char getIntegrationMethod( int index ) { return fIntegrationMethod[index]; }

  Variable( const char* name );
  ~Variable();

  virtual int Config( TConfigData & cd );
  virtual int ExecuteIgnoreFunction() { return 0;}
  inline CString& GetUnits() { return fUnit; }
  inline void SetFluxInfo( EFluxInfo index, byte value ) { fFluxInfo[index] = value; }
  inline byte GetFluxInfo( EFluxInfo index ) { return fFluxInfo[index]; }
  virtual inline void WriteConfigXML( FILE* oFile ) {;}
  virtual inline int BuildConfigXML( ArgArray& msg, int mode ) {;}

  void setFluxTypeIndex( EFluxType ft ) ;
  void setFluxType( CString& s ) ;
  EFluxType getFluxTypeIndex() const ;
  char getFluxType() const ;
  void FlagImport() {;}
  void FlagExport() {;}  
  inline Module* GetModule() { return fModule; }
  void FullName(CString& name, char connector = ':' ) { name = fModule->Name(); name += connector; name += fName; }
  TNamedObjectList& CommandList() { return fCommandList; }
  inline byte CoordIndex() { return fCoord_index; }

  inline Variable* Origin() { return fOrigin; }
  inline Variable* Destination() { return fDestination; }
  inline void SetOrigin(Variable* v) { 
    if( fOrigin == NULL || fOrigin == v ) fOrigin = v; 
    else {
			sprintf(gMsgStr,"%s:%s -> Origin Already Set to %s.",Name(),v->Name(),fOrigin->Name());
			gPrintErr(); 
		}
  }
  inline void SetDestination(Variable* v) { 
    if( fDestination == NULL || fDestination == v ) fDestination = v; 
    else gPrintErr( SName() + ": Destination Already Set."); 
  }
	inline virtual void UpdateReserve() {;}
	
  inline virtual void UpdateInternals() { if(fConnection && GetF(FDualConnection)) fConnection->RefreshData(); }

   inline virtual float UpdateParameter( int index  ) { return (fValue = fParm.GetValue(index)); }    
   inline virtual float ResetParameter() { return (fValue = fParm.GetValue()); }    
   inline virtual float UpdateParameter( int index, int& pspace_index  ) { return (fValue = fParm.GetValue(index,pspace_index)); }  
   inline virtual float UpdateParameterProportional( float prop_val  ) { return (fValue = fParm.GetValue(prop_val) ); }  
   inline virtual float GetParameterSensitivity(float& incr) { return fParm.GetSensitivity(incr); }  
   inline virtual float SetParameterValue( float prop_val ) { return (fValue = fParm.SetPropValue(prop_val)); }
   inline virtual float SetParameterValue( int index ) { return (fValue = fParm.SetValue(index)); }
   inline virtual TParmRec& getParmRec() { return fParm; }
  
  virtual inline int GraphData( const char* timeseries_name, const char* graph_name=NULL, float* fd = NULL, int ds = 0, Bool overwrite = False  ) {;} 
  virtual inline int DisplayData( const char* display_name, float* fd, float fmax, float fmin, int s0, int s1, int s2=1  ) {;} 

  virtual int  addArrayDeclaration( ArrayRec* array, int array_index, int is_dimension_name, int index );
  inline int ArrayIndexMap( int index ) { return _Array_Index_Map[index]; }
  virtual ArrayRec* getArrayDeclaration( int order ); 
  int getLocalArrayIndex( int index_history_array[4], int global_index, int iarg );
  virtual void WriteCCode( FILE* CHdrFile, FILE* CEqnFile);
  void WriteFlags(FILE* CEqnFile) const;
  void WriteCCommandCode(FILE* CEqnFile);
  virtual int AddModifier( const CString& mod, Bool isExtension = False );
  virtual int GetModifierString( CString& name ) const ;
	virtual int GetXMLAttributes( CString& name ) const ;
	virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void WriteXML( FILE* oFile );
  virtual inline float Value( ) { return fValue; }
  inline virtual float SetValue( float value  ) { return fValue = value; }  
  virtual int LogDependency( Variable* v ) { return 1; } 
  virtual void WriteDataToConfig( CStream& outStream ) const;
  virtual void UpdateValueExternal( const char* external_mod_name, const char* external_var_name, EVarLink::EModality modality ) {;}
  Variable*  CreateDerivativeVariable( TDerivative* d );
  TCommand* GetDynamicCommand() ;
  inline virtual int ProcessDependency( Variable* depVar, CString& value, int processIndex ) { return 0; }
  TCommand* GetCommand( const  CString&  name, TCommand::EType type = TCommand::kUndefined, TCommand::EExType etype = TCommand::kNoEx, Bool create = True );
  Pix AddCommand( TCommand* c );
  virtual void Dump( FILE* oFile );
  inline EDataType& DataType() { return fDataType; }
  virtual inline int SplitCondition( int splitIndex ) { return 0; }
  virtual int RegisterPipe( Pipe& pipe );
  void SetCmdStatus ( TCommand::EType type = TCommand::kUndefined, TCommand::EStatus status = TCommand::kInActive );
	void SetCmdStatus ( TCommand::EExType etype, TCommand::EStatus status);
  Pipe* GetPipe( Pipe::EType type );
//  Pipe* GetPipe( Pipe* p = NULL );
  virtual void Init( Module* mod );
  void SetF(FlagType  mask, int value, const char* comment);
  int ProcessSpatialDependency( Variable* v, CString& value );
  TCommand* GetDefaultCommand ( );
  TCommand* GetCommandByType ( TCommand::EType etype = TCommand::kUndefined );
  int DeleteCommands( TCommand::EType t = TCommand::kUndefined );
  int RemoveCommandsFromList( char* cname );
  int DeleteCommand( const CString& name );
  TPipeCommand* ProcessPipes( Pipe::EType t = Pipe::kUndefined, EProcessAction pa = kReturn );
  virtual int Finalize();
  void AddUserDeps();
  virtual int SetupParameter(float init_val, float min_value, float max_val, int  ustep );
  
  virtual int ProcessGeneral( TNode* n );  
	virtual int RunNodeProcessing( EProcessContext pc );
  virtual int ProcessCommand(const TCommandNode* n);
	virtual int ProcessAttribute(const TAttributeNode* n);

  void ProcessDoc( CString& docstr );
  void ProcessAttribute( CString& attribute, CString& value  );

  inline FlagType GetF(FlagType  mask) const { return ( fFlags & mask ); }
  void AddFlag( char* name, FlagType& flag);
  virtual void AddFlags();
  void  GetFlagNames ( FlagType  mask, CString& flagNames);
  void  PrintFlags() const;
  void PrintDependencies( Pix p0, int depth, TNamedObjectList& depTree ) const;
  void AddMapInputs( DBaseInputPipe& dbp );
  void CheckMapInputs( );
  void MakeConnection( Variable* cVar, Bool dualConnection = False );
  Variable* Connection( Variable* v = NULL ) { 
  	if(v) { fConnection = v; return v; } else { return fConnection; } 
  }
  virtual inline void RefreshData() {;}
  inline CString& Map(int index=0) { if( index<kMaxNVarDBMaps) { return fMap[index]; } else return kGarbage; }
  inline void SetMap( const CString& map, byte index=0 ) { if( index<kMaxNVarDBMaps) { fMap[index] = map; } }
  virtual float* Data() { return NULL; }
  virtual const float* TimeSeriesData( int array_index, int& length ) { length = 0; return (const float*)NULL; }
  virtual int DataSize() { return 0; }
  void SetBreak ( TCommand::EType type = TCommand::kUndefined, int BreakOn = True );
  void SetBreak ( TCommand::EExType type = TCommand::kNoEx, int BreakOn = True );
	inline ExSchedule* Schedule() { return fSchedule;	} 
	inline const ExSchedule* ScheduleC() const { return fSchedule;	} 
	inline void SetVariableNode( TVarNode* v ) { fVarNode = v; }
	inline const CString& DefaultValue(int index) const { return fDefValue[index]; 	}
	inline Variable* Parent() { return fParent; } 
	inline void SetParent(Variable* v) { fParent = v; } 
  inline int Order() { return fOrder; }
  inline void SetOrder(int o) { fOrder = o; }
  inline int activationLayerIndex() { return fActivationLayer; }
  inline int LinkLayer() { return fLinkLayer; }
  inline Pix AddDBParamValue( TDBParamValue& dbpv ) { return fDBParamValueList.append(dbpv); }
  inline TNamedObjectList& GetDBParamValues() { return fDBParamValueList; }
  int AddDBParamValue( const CString& parm, const CString& value);
  
  virtual inline int SetupDBParam( const CString& map  ) 	{
		if( GetCInfo(kInitMode) != kDBase ) { MakeDBaseVar(); fMap[0] = map; }
		else if( fMap[0] != map ) { 
			sprintf(gMsgStr,"Map discrepancy in DBParmdata for %s: %s vs %s", Name(), fMap[0].chars(), map.chars() );  gPrintErr(); 
		}
		return 0;
	}

  int PrintDBParamValues( FILE* file );

  virtual int MakeParameterVariable( ) ;

   virtual inline void  MakeDBaseVar() {
     SetCInfo(kType,kData);
     SetCInfo(kInitMode,kDBase);
     SetF(FisConstant,True,"CAuxVariable");
     SetF(FisSpatial,False,"CAuxVariable");
   }

   virtual inline void MakeTimeSeriesVariable() {
     SetCInfo(kType,kData);
     SetCInfo(kInitMode,kTimeSeries);
     SetF(FisConstant,False,"CAuxVariable");
     SetF(FisSpatial,False,"CAuxVariable");
   } 

   virtual inline void MakePtTimeSeriesVariable() {
     SetCInfo(kType,kData);
     SetCInfo(kInitMode,kPtTimeSeries);
     SetF(FisConstant,False,"CAuxVariable");
     SetF(FisSpatial,True,"CAuxVariable");
   }
 
   virtual void MakeMapInputVariable( int isInt );
   
   virtual inline void MakeExternalIOVariable( EVarLink::EModality mod ) {
	 SetF(FisSpatial,True,"MakeExternalIOVariable");
	 SetF(FisConstant,False,"MakeExternalIOVariable");
	 if( (mod ==  EVarLink::kInitIn) ||  (mod ==  EVarLink::kIn) ) {
	   SetCInfo(kType,kData);
	   SetCInfo(kInitMode,kExternalInput);
	 } 
   } 
};

class FlagRef : public TNamedObject {
  FlagType fFlag;
public:
  FlagRef( char* name, FlagType flag ) : TNamedObject(name) { fFlag = flag; }
  FlagType Flag() { return fFlag; }
  virtual void Dump(FILE* oFIle) const {;}


};

#endif

