//----------------------------------------------------------------------------------------
//	DistributedGrid.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#ifndef __DistributedGrid__
#define __DistributedGrid__

#include "MultiGrid.h"
#include "Ghost.h"
#include "TMap2.h"
#include "MML_Model.h"
#include <math.h>

#define pNN Point2(-1,0)
#define pNE Point2(-1,1)
#define pNW Point2(-1,-1)
#define pSS Point2(1,0)
#define pSE Point2(1,1)
#define pSW Point2(1,-1)
#define pEE Point2(0,1)
#define pWW Point2(0,-1)

const int kDGLocalGrid = 255;
const int kDGLinkedPoints = 254;
const int kDGStudyArea = 253;

typedef class FLBase FLBase;

/*
*************************************************************************
*									*
* PointSet								*
*									*
*************************************************************************
*/

const int kDG_NPointOrderings = 10;

class DistributedGrid : public TOrderedObject { 

public:
  enum EPointOrdering { kPO_Undefined, kGridOrdering, kUpLinkOrdering, kDownLinkOrdering };
  enum EGridInfo { kBaseType, kFrameType, kisGlobal, kOrderingIndex };

protected:
		byte fGridInfo[8];
    TMap2* fRegionMap;
    PixVec fRootPointVec;
		PixVec fPointOrdering[kDG_NPointOrderings];
		EPointOrdering fPointOrderingType[kDG_NPointOrderings];
    Pix fCurrentPix;
		ByteBuffer fCommBuff;
		int fLinkedPointIndex;
		int fGridLinkLayer;
		
		TLayer* fGridLayer;
		ActivationLayer* fActivationLayer;
		MultiGrid* fMultiGrid;
		int fActivationLayerIndex;

	int CalcPointOrdering( EPointOrdering po, const OrderedPoint* pt = NULL );	
	Pix firstLink(Pix p, EPointOrdering po, int& lindex, int link_layer= -1 );	
	Pix nextLink(Pix p, EPointOrdering po, int& lindex, int link_layer= -1 );

  inline Pix TranslateByLinks( Pix p0, int dr, int dc, int dv=0, int link_layer = -1 ) const {
		if( link_layer < 0 ) link_layer = fGridLinkLayer;
    if( dr > 0 ) {
		  p0 = fGridLayer->translateByLinks( (TCell*)p0, SS-NE, dr, link_layer ) ;
    } else if ( dr < 0 ) {
		  p0 = fGridLayer->translateByLinks( (TCell*)p0, NN-NE, -dr, link_layer ) ;
    }
    if( dc > 0 ) {
		  p0 = fGridLayer->translateByLinks( (TCell*)p0, EE-NE, dc, link_layer ) ;
    } else if ( dc < 0 ) {
		  p0 = fGridLayer->translateByLinks( (TCell*)p0, WW-NE, -dc, link_layer ) ;
    }
    if( dv > 0 ) {
		  p0 = fGridLayer->translateByLinks( (TCell*)p0, DD-NE, dv, link_layer ) ;
    } else if( dv < 0 ) {
		  p0 = fGridLayer->translateByLinks( (TCell*)p0, UU-NE, -dv, link_layer ) ;
    }
    return p0;
  }
	
	   
public:

	static int fDebug;
  enum EBaseType { kNull, kDirect, kMap };
  enum EFrameType { kUndefined, kGrid, kNetwork, kTree, kCollection, kScalar  };
  enum ELinks { kLink0, kLink1, kLink2, kLink3 };

  DistributedGrid( MultiGrid* grid, TMap2* regionMap, int activationLevel, int gridLayer, int linkLayer ) 
		: fMultiGrid( grid )
	{ 		
		fActivationLayerIndex = activationLevel;
		fActivationLayer = fMultiGrid->activationLayer(fActivationLayerIndex);
		fRegionMap = regionMap;
		memset( fPointOrderingType, 0, kDG_NPointOrderings*sizeof(EPointOrdering) );
		fPointOrderingType[0] = kGridOrdering;
		fCurrentPix = 0;
		fGridLayer = fMultiGrid->getCellLayer(gridLayer);
		fGridLinkLayer = linkLayer;
	}
	
	inline int getLinkLayer() { return fGridLinkLayer; }
	inline int getGridLayer() { return fGridLayer->Index(); }
	
	
  inline Bool operator==( DistributedGrid& aGrid ) {  return ( fActivationLayerIndex == aGrid.fActivationLayerIndex ); }

  inline Bool Same ( int activationLevel ) {  return ( fActivationLayerIndex == activationLevel ); }
  inline int activationLayerIndex () {  return fActivationLayerIndex; }
  inline MultiGrid* getMultiGrid() { return fMultiGrid; }
	inline void SetGridLinkLayer( int layer ) {  fGridLinkLayer = layer; }
	inline int row( TCell* c ) { return fGridLayer->getCellLocCoord( TLayer::kRows, c); }
	inline int col( TCell* c ) { return fGridLayer->getCellLocCoord( TLayer::kRows, c); }
	inline byte GetInfo(EGridInfo index) { return fGridInfo[index]; }
  inline void SetInfo(EGridInfo index, byte val) { fGridInfo[index] = val; }
  
  inline Region2& Region(int iproc=gIProc, int layer_index=-1 ) const { 
		TLayer* l = fMultiGrid->getCellLayer( layer_index );
		TPartition2* p = l->partition();
		return *(p->region(iproc));
  } 
  inline void GRegion(Region2& r, int iproc=gIProc, int layer_index=-1 ) const { 
		r = Region(iproc,layer_index);
		r.grow(fActivationLayer->maxGhost()); 
  } 

  inline void GlobalRegion( Region2& r, int layer_index=-1  ) const {
		TLayer* l = fMultiGrid->getCellLayer( layer_index );
		l->globalRegion(r);
  } 

  int GetOwner(  TLayer* l, TCell* c ) {  return l->partition()->getowner( l, c ); }  
  int GetOwner(  Point2& p, int cell_layer_index = -1 ) {  
		TLayer* l = fMultiGrid->getCellLayer(cell_layer_index); 
		return l->partition()->getowner( p(0), p(1) ); 
	}
	
  inline int InteriorSize() { return fActivationLayer->nActiveCells(); }
  inline int TotalSize() { return fActivationLayer->nCells(); }
	inline ByteGrid* RegionMap() { int cell_layer_index; return fMultiGrid->GetPartition(cell_layer_index)->cellDistribution(); }

  inline void CheckGhostSize(int gSize) { if(gSize>fActivationLayer->maxGhost()) { gFatal("GhostSize Too Large");  } }
  	
	inline Pix FirstLinkedPix(Pix p0) {
		return firstLink(p0, fPointOrderingType[fGridInfo[kOrderingIndex]], fLinkedPointIndex);	
	}
	
	inline Pix NextLinkedPix(Pix p0) {
		return nextLink(p0, fPointOrderingType[fGridInfo[kOrderingIndex]], fLinkedPointIndex);	
	}
	
	inline Pix NeighborPix( Pix p0, Grid_Direction idir ) {
		int rr, rc; GMR(rr,rc,idir);
		return fGridLayer->translateNEWS( (TCell*)p0, rr, rc );
	}
	
	inline Pix LinkedPix( Pix rp, int linkIndex=0, int dist=1, int link_layer = -1 ) {
		if( link_layer < 0 ) link_layer = fGridLinkLayer;
		return fGridLayer->translateByLinks( (TCell*)rp, linkIndex, dist, link_layer);
	}
	
  inline byte GetPointInfo ( Pix p, EInfoIndex index ) { return (GetPoint(p)).GetObjInfo(index); }
 
	inline TCell* first() {  return fGridLayer->firstCell(); }
	inline void next(Pix& p) { fGridLayer->nextCell((TCell*&)p); }
	inline void next(TCell*& c) { fGridLayer->nextCell(c);; }
	
  inline const OrderedPoint& GetPoint (Pix idx) const {
		if( idx == 0 ) { gFatal( "NULL Pix in GetPoint()" ); } 
		return *((OrderedPoint*)idx); 
	}

	PixVec&  RootPointVec() { return fRootPointVec; }
		

  inline int onGrid(const OrderedPoint& pt ) const { return  (pt.GetObjInfo(TCell::kGhostIndex) == 0); }

  inline const OrderedPoint* onGrid( Pix idx ) {
		if( idx == 0 ) return NULL;
    if( ((TCell*)idx)->GetObjInfo(TCell::kGhostIndex) == 0 ) return (OrderedPoint*)idx;
    else return (OrderedPoint*) NULL;
  }  
	
	inline Pix TranslateByGridCoord( Pix p0, int rr, int rc, int rv=0 ) const {
		return (rv==0) ? fMultiGrid->translateNEWS( (TCell*)p0, rr, rc ) : fMultiGrid->translateNEWS( (TCell*)p0, rr, rc, rv );
	}
	
	inline Pix getCell( unsigned int ir, unsigned int ic ) {
	  return fGridLayer->getCell( ir, ic );
	}
	                                            
	inline Pix Translate( Pix p0, int dr, int dc, int dv=0 ) const {
		if( fGridLinkLayer > 0 ) { 
			 return TranslateByLinks( p0, dr, dc, dv, fGridLinkLayer ); 
		} else { 
			 return (dv==0) ? fMultiGrid->translateNEWS( (TCell*)p0, dr, dc ) : fMultiGrid->translateNEWS( (TCell*)p0, dr, dc, dv );
		}
	}
	 	
  inline Pix TranslateByLinkIndex( Pix p0, int lIndex=0, int nLinks=1, int link_layer= -1 ) const {
		if( link_layer < 0 ) link_layer = fGridLinkLayer;
    return fGridLayer->translateByLinks( (TCell*)p0, lIndex, nLinks, link_layer ) ;
  }

	inline int SetPointOrdering(EPointOrdering po, const OrderedPoint* pt = NULL, int ordering_index = -1 ) {
		if( ordering_index > 0 )  {
			if( ordering_index <= kDG_NPointOrderings ) {
				fGridInfo[kOrderingIndex] = ordering_index;
			} else { 
				sprintf(gMsgStr,"Error, only %d Point orderings allowed (tried %d).",kOrderingIndex,ordering_index);
				gPrintErr(); return -1; 
			}
		} else if( ordering_index == -1 ) { 
			fGridInfo[kOrderingIndex] = fGridInfo[kOrderingIndex]+1; 
		} else if( ordering_index == 0 ) { 
			gPrintErr("Point Ordering number 0 is reserved (default grid ordering)."); return -1; 
		}	
		ordering_index = fGridInfo[kOrderingIndex];
		if( fPointOrderingType[ordering_index] == kPO_Undefined ) { 	
			fPointOrderingType[ ordering_index ] = po;
			return CalcPointOrdering(po,pt);
		} else if( fPointOrderingType[ordering_index] != po ) {
			gPrintErr("Can't change existing Point Ordering, must use different index.");
			return -1;
		} else {
			return ordering_index;
		}
	}

	inline void SetPointOrdering(int ordering_index = 0) {
		if( ordering_index <= kDG_NPointOrderings ) fGridInfo[kOrderingIndex] = ordering_index;
		else { 
			sprintf(gMsgStr,"Error, only %d Point orderings allowed (tried %d).",kOrderingIndex,ordering_index);
			gPrintErr();
		}
		if( fPointOrderingType[ordering_index] == kPO_Undefined ) { gPrintErr("Setting Undefined Point Ordering"); }
	}
	
  
};
    
#endif
