#ifndef _included_Ghost_h
#define _included_Ghost_h

/****************************************************************
* Ghost.h							*
*								*
* declare functions useful for manipulating ghost regions	*
*								*
*****************************************************************/

#include "IntGrid.h"
#include "Grid.h"

/*
*************************************************************************
*									*
* GhostIterator 							*
*									*
*************************************************************************
*/

class GhostIterator {
   int state;
   Region2 R;
   Point2 ghost;
public:
   GhostIterator(const Region2& Reg, const Point2& G): 
	state(0), R(Reg), ghost(G) {;}

   Region2 operator()() const;

   int GhostIterator::operator !() const;
   void operator ++();
   operator void *();
};

/*
*************************************************************************
*									*
* GhostCellCounter							*
*									*
*************************************************************************
*/

class GhostCounter {

  int fLeg;
  int fRow[2];
  int fCol[2];
  int fInc[2];

public:

  GhostCounter(int iGhost, const Region2* grid, int& ir, int& ic) { 
    fLeg=0;
    fInc[0] = grid->increment(0);
    fInc[1] = grid->increment(1);
    fRow[0] = grid->lower(0) - (iGhost)*fInc[0];
    fRow[1] = grid->upper(0) + (iGhost)*fInc[0];
    fCol[0] = grid->lower(1) - (iGhost)*fInc[1];
    fCol[1] = grid->upper(1) + (iGhost)*fInc[1];
    ir = fRow[0];
    ic = fCol[0]; 
  }

  inline int GhostCounter::Count( int& ir, int& ic ) {

    switch(fLeg) {
    case 0:
      ic+=fInc[1];
      if( ic >= fCol[1] ) { fLeg++; ir = fRow[1]; ic = fCol[0]; }
      break;
    case 1:
      ic+=fInc[1];
      if( ic >= fCol[1] ) { fLeg++; ir = fRow[0]; ic = fCol[0]; } 
      break;
    case 2:
      ir+=fInc[0];
      if( ir >= fRow[1]-1 ) { fLeg++; ir = fRow[0]; ic = fCol[1]; }
      break;
    case 3:
      ir+=fInc[0];
      if( ir >= fRow[1]-1 ) { fLeg++; ir = fRow[0]; ic = fCol[0]; }
      break;
    case 4:
      fLeg=0;
      return 0;
    }
    return 1;
  }
};

/*
*************************************************************************
*									*
* GhostIndexSet								*
*									*
*************************************************************************
*/


class GhostIndexSet {

  IntArray fLayerOffset;
  IntArray fBOffset;
  IntArray fBlockLen;
  int fCurrentOffset;
  int fCurrentGhost;
  EG_LinkMode fMode;
  int fMaxGhost;
  Bool fFinalized;

#ifdef USE_MPI
  MPI_Datatype* fGhostCellLayout;
#endif

  void SetupMap( int ighost );

public:
 
  GhostIndexSet( int nLayers, EG_LinkMode mode );
  inline int* BlockOffset() { return fBOffset.ptr(0); }
  inline int* BlockLength() { return fBlockLen.ptr(0); }
  inline int Length( int ghostSize ) { return fLayerOffset(ghostSize); }

#ifdef USE_MPI
  MPI_Datatype GetGhostIndexMap( int ighost ) ;
#endif

	void DumpData( int ighost, const float* data );
  int AddEntry(int offset, int ighost);
  void Finalize( );
};

/*
*************************************************************************
*									*
* GhostCellMap								*
*									*
*************************************************************************
*/

class GhostCellMap {

  GhostIndexSet fSend;
  GhostIndexSet fRecv;

public:

  GhostCellMap(int nLayers): fSend(nLayers,kL_Output), fRecv(nLayers,kL_Input) {;} 

  int AddEntry(int offset, int ighost, EG_LinkMode mode) {    
    switch(mode) {
    case kL_Input: return fRecv.AddEntry( offset, ighost ); break;
    case kL_Output: return fSend.AddEntry( offset, ighost ); break;
    }
  }
  
#ifdef USE_MPI
  MPI_Datatype GetGhostIndexMap( int ighost, EG_LinkMode mode ) {
    MPI_Datatype rv; 
    switch(mode) {
    case kL_Input: rv = fRecv.GetGhostIndexMap( ighost ); break;
    case kL_Output: rv = fSend.GetGhostIndexMap( ighost ); break;
    }
    return rv;
  }
  
  void DumpData( int ighost, const float* data, EG_LinkMode mode ) {
    switch(mode) {
			case kL_Input: fRecv.DumpData( ighost, data ); break;
			case kL_Output: fSend.DumpData( ighost, data ); break;
    }
  }
#endif
};

#endif
