//----------------------------------------------------------------------------------------
//	HDFMgr.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------

#ifndef __HDFMgr__
#define __HDFMgr__  

#include "Globals.h"
#include "PathString.h"
#include "XGraph.h"

extern "C" {
#ifdef HAS_HDF
#include "mfhdf.h"
#else
#include "sme_types.h"
#endif
}

typedef class SDSObject SDSObject;
typedef class TMap TMap;

#include <stdio.h>
#include <ctype.h>

const int kAttrStrLen = 100;
const int  kMaxRefs = 256;

//----------------------------------------------------------------------------------------
//						HDFMgr
//----------------------------------------------------------------------------------------

class HDFM {

public:

  enum EVDataType { kUnknown, kVData, kVGroup, kSData  };
  enum EVDataClass { kFUndefined, kFArray, kFMap, kFDataSet, kFSeries, kFPointSeries, kFDBaseParm, kFDBase2Parm };
  enum ERefType { kFName, kFClass };
  enum EDirInfo { kStatus, kNArgs, kType, kMode, kClass, kIOMode, kFirstOut, kHLevel , kMag};
  enum EIOType { kFNone, kFInput, kFOutput, kFReadArchive };

private:

  //	Create ids and keys for dsets: Arrays, Maps, Series, Data.;
  static int32 		fTopRef; 
  static int32 		fArraysRef; 
  static int32 		fMapsRef; 
  static int32 		fSeriesRef; 
  static int32 		fDataRef;         

  static int32   	fCurrentRef;
  static int32   	fParentRef;
  static int32          fVGID_Current;
      
  static int32    	fFile;
  static int32         fSD_id;
  static CPathString   	fPath;

  static char fTmp0[kAttrStrLen];
  static char fTmp1[kAttrStrLen];
  static char fTmp2[kAttrStrLen];
	
  static int32 fNTagRefs;
  static SDSObject* fSDSObject;
  static byte fInfo[16];
  static int32 fRef;
  static int32 fSDSRef;
  static ArgArray fArg;
  static TMap* fCurrentMap;

  enum EVDataRef { kID, kRef };

  static int32 GetVG( int32 vgid, const char* name, EVDataRef ref ) ;
  static int32 GetVG( int32 vgid, const char* name, int32* vgidp ) ;
  static int32 GetChild( int32 vgParentID, const char* name, const char* className, EVDataType tag, EVDataRef rtype   );
  static int32 CreateDSetSubgroup( const char* dsetName, const char* subgroupName, const char* className );
  static inline int32 GetNextRef( int32 vgRef ) {
#ifdef HAS_HDF 
		return (fFile) ? Vgetid((HFILEID)fFile,vgRef) : FAIL; 
#else
		return 0;
#endif
	} 
  static inline EVDataType Type() { return (EVDataType) fInfo[kType]; }
  static EVDataType GetRefAndType( int32 vgID, int igroup );
  static EVDataType GetType( int32 tag );
	   
public:

  static int32 Open ( EIOType ioType ); 
  static int32 Open(const CPathString& filePath, const char* mode);
  static void Close();
  static int GetArgs();
  static int PushDir( int index );
  static int PopDir( int toTop=0 );
  static inline int TopDir() { return PopDir(1); }
//  static void ClearAnim();

  static int32 GetSDS( EVDataClass type,   const char* dsetName,  const char* variableName );
  static int AddSDS( int32 sds_id, EVDataClass type, const char* variableName );
  static int ReadSDSToMap(TMap& map, const char* dSetName, const char* varName, int index ); 
  
  static inline void SetInfo( EDirInfo index, byte val ) { fInfo[index] = val; }
  static inline byte GetInfo( EDirInfo index ) { return fInfo[index]; }
  static inline void SetSDS(int32 ref);
  static inline TMap* GetMap() { return fCurrentMap; }

  static inline const CString& Arg(int index) { return fArg[index]; }
  static inline int NStrArgs() { return fArg.NArgs(); }

  static inline int PrintArgs( FILE* oFile ) {  
    if ( fArg.length() == 0 ) {
      fprintf(oFile,"\nThis group has no subgroups\n");
      return 0;
    }
    else { fArg.Print( oFile ); return 1; } 
  }  

  static inline byte NArgs() { return fInfo[kNArgs]; }
  static inline byte HStatus() { return fInfo[kStatus]; }
  static inline byte Mode() { return fInfo[kMode]; }
  static inline EVDataClass Class() { return (EVDataClass) fInfo[kClass]; }
  static EVDataClass Class(int32 vgRef);

  static inline byte SetMode( byte mode ) { return fInfo[kMode] = mode; }
  static inline SDSObject* SDS() { return fSDSObject; }
  static int32  SDid() { return fSD_id; }

  static inline void SDSExport( char* pathName );
  static inline void Finalize();
  static inline const char* DSetName();
  static inline const char* VariableName();

  static int32 SDRead( int index = 0 ); 
  static int NData();

  static void TextDump( const CString& path, int index=0, int nData=1 );
#ifdef HAS_X
  static int ViewData(TSheet* aSheet, int index=0, int nData=1);
  static int GraphData(TGraph* aGraph, int index=0, int nData=1); 
  static int AnimateData(XAnim* aAnim, int index=0, int nData=1); 
#endif
  static int DumpMap(CString& path, int index=0, int nData=1);
  static int DumpHDF(CString& path, int index=0, int nData=1);


  static int ImportSeries( char* name, char* pathName, char* fileName, int format );
  static int ImportPtSeries( char* name, char* pathName, char* fileName, int format );
  static int ImportMap( char* name, char* pathName, char* fileName, int format );
};


#endif
