#ifndef __MultiCoverage__
#define __MultiCoverage__

#include "MultiGrid.h"
#include "DistributedGrid.h"
#include "MemoryMgr.h"
#include "TMap.h"

/*
*************************************************************************
*									*
* MultiCoverage     					        *
*									*
*************************************************************************
*/

typedef class TMap TMap;

enum EReduceOp { kR_Max, kR_Min, kR_Sum, kR_Ave };
enum ECoverageDataType { kIntensive, kExtensive };
enum  EScaleType { kDefaultScaling, kDirectScaling, kAutoScaling, kRainbowScaling };


#define PSCserial 		1
#define PSCfullGrid 	2
#define PSCsubGrid 		3

class MultiCoverage: public MemObject, public floatVec {

	static int kLinkEdgesTagBase;
	
  int fNReq;
  byte fInfo[8];
  float fGarbage;
  static MemoryManager* fMemMgr;
  int fNCells;
  int fTag;
  static int fCoverageCount;

#ifdef USE_MPI
  byte fDumpTransferData;
  static MPI_Comm fComm;
  MPI_Request* fReq;
  MPI_Status* fStatus;
#endif

  void Compress( byte* data, int bsize );
	void MergeData( TMap& map, void* src, Region2& r, int nbytes );
  void initialize_fields() ;

protected:

  MultiGrid* fMultiGrid;
  int fActivationIndex;
  ActivationLayer* fActivationLayer;
  ECoverageDataType fCoverageDataType;
  PixXPlex fGridPlex;
  int fGhostSize;


public:

  MultiCoverage();
  MultiCoverage( MultiGrid* mgrid, int activationLayer, int ghostSize=-1, int memoryIndex = MemObject::kPermanentMemory );
  MultiCoverage(  MultiCoverage& cov ) {  initialize_fields(); Setup( cov.fMultiGrid, cov.fActivationIndex, cov.fGhostSize, cov.fMemoryIndex ); }

  int Setup( MultiGrid* mgrid, int activationLayer, int ghostSize=-1, int memoryIndex = MemObject::kPermanentMemory  );
  int Setup( MultiCoverage& cov, int ghostSize = -1, int memoryIndex  = MemObject::kPermanentMemory );
  int Setup( DistributedGrid* pset, int gSize, int memoryIndex );
  inline int Allocated() { return ( fMultiGrid == NULL ) ? 0 : 1; }
  inline void CoverageDataType( ECoverageDataType cdt ) {  fCoverageDataType = cdt; }
  inline int isIntensive() { return (fCoverageDataType == kIntensive); }
    
  inline ActivationLayer* activationLayer() { 
		if( !Allocated() ) return NULL;	
		if( fActivationLayer == NULL ) { 
			fActivationLayer	= fMultiGrid->activationLayer(fActivationIndex); 
			fNCells = fActivationLayer->nCells(); 
		}
		return fActivationLayer;
	}
	inline int activationIndex() { return fActivationIndex; }
   
  inline TPartition2* GetPartition(int& cell_layer_index ) {
		if( !Allocated() ) return NULL;		
		return fMultiGrid->GetPartition(cell_layer_index);
	}
		
	inline ByteGrid* RegionMap( int& cell_layer_index ) { return GetPartition(cell_layer_index)->cellDistribution(); }
	
	inline void Set(float value) { 
		float* start = (float*)data(); float* end = start + fNCells; 
		while( start < end ) { *start++ = value; } 
	}

  inline const float* DataEnd() { return  data() + fNCells; }
  inline const float* GridEnd() { return  data() + fActivationLayer->nActiveCells(); }

	inline void ShallowCopy(FloatStore& v){ shallow_copy(v); }

  int LinkEdges(const char* name, int dumpTransfer=0);
  
  DistributedGrid* Grid( int gridLayer, int linkLayer );
  inline MultiGrid* multiGrid() { return fMultiGrid; }


  int StartLinkEdges();
  int CompleteLinkEdges(const char* name);
  void Initialize() { reinitialize(); }

	inline void GrabMemory() { ((FloatStore*)fOrigin)->SetCurrentOwner(this); }
	inline int  CheckMemory() { return ((FloatStore*)fOrigin)->CheckCurrentOwner(this); }

  inline MemoryManager* Manager() { return fMemMgr; }

  inline const float* Data() { return data(); }
  inline int Size() { return fNCells; }

  float& Value( unsigned int row, unsigned int col, int layer_index = -1 );

  inline int CellIndex( TCell* cell ) {
	if( cell == NULL ) return -1; 
    return cell->memoryIndex( fActivationIndex );
  }
  	
  inline float& Value( TCell* cell ) {
		if( cell == NULL ) return fErrorVal; 
    int32 index = cell->memoryIndex( fActivationIndex );
    if( index < 0 ) return fErrorVal;
#ifdef DEBUG
    if( index >= fNCells ) { sprintf(gMsgStr,"Illegal Index in MultiCoverage: %d, size = %d ",index,fNCells);  gFatal(); } 
    return elem(index);
#else
    return elem(index);
#endif
  }

inline float& operator () (int32 index) {
  return elem(index);
}

inline float& operator () (TCell* cell) {
	if( cell == NULL ) return fErrorVal; 
	int32 index = cell->memoryIndex( fActivationIndex );
  return elem(index);
}

  inline float& Value(int32 index) {
#ifdef DEBUG
    if( index<0 || index>=fNCells ) { sprintf(gMsgStr,"Illegal Index in MultiCoverage: %d, size = %d ",index,fNCells);  gFatal();  } 
#endif
    return elem (index);
  }    

  inline int32 SetValue(int32 index, float value) {
#ifdef DEBUG
    if(index<0 || index >=fNCells ) { sprintf(gMsgStr,"Illegal Index in MultiCoverage: %d, size = %d ",index,fNCells);  gFatal(); } 
#endif
    elem(index) = value;
    return index;
  } 
 
  inline int32 SetValue( TCell* cell, float value ) {
    int32 index = cell->memoryIndex( fActivationIndex );
    if(index<0) return index;
#ifdef DEBUG
    if(index >= fNCells) { sprintf(gMsgStr,"Illegal Index in MultiCoverage: %d, size = %d ",index,fNCells);  gFatal();  }
#endif
    elem(index) = value;
    return index;
  }

  inline int32 AddValue(int32 index, float value) {
#ifdef DEBUG
    if(index<0 || index >=fNCells ) { sprintf(gMsgStr,"Illegal Index in MultiCoverage: %d, size = %d ",index,fNCells);  gFatal(); } 
#endif
    elem(index) += value;
    return index;
  } 
 
  inline int32 AddValue( TCell* cell, float value ) {
    int32 index = cell->memoryIndex( fActivationIndex );
    if(index<0) return index;
#ifdef DEBUG
    if(index >= fNCells) { 
	  sprintf(gMsgStr,"Illegal Index in MultiCoverage: %d, size = %d ",index,fNCells);  
	  gPrintErr();  return -1; 
    }
#endif
    elem(index) += value;
    return index;
  }
  
#ifdef USE_MPI
	inline void dumpTransferData(EG_LinkMode mode) {
		for( int iproc=0; iproc<gNProc; iproc++ ) {
			if(mode == kL_Output) { sprintf(gMsgStr,"\ndumpTransferData: Send to proc %d",iproc); gPrintLog(); }
			else { sprintf(gMsgStr,"\ndumpTransferData: Receive from proc %d",iproc); gPrintLog(); }
			fActivationLayer->dumpTransferData( iproc, fGhostSize, data(), mode );
		}
	}
#endif

  MultiCoverage& operator = (MultiCoverage& b);
  MultiCoverage& operator += (MultiCoverage& b);
  float Reduce ( EReduceOp op, int rootProc=0, CellPlex* cp = NULL, int ignoreINF = 0 );
  float Reduce ( EReduceOp op, int rootProc, MultiCoverage& mask, int ignoreINF = 0  );
  float* MultiReduce ( EReduceOp op, int rootProc, MultiCoverage& mask, int ignoreINF = 0 );
  void SpreadValues( float* values, MultiCoverage& mask );
	int LocalGridToMap( TMap& map, unsigned long background, EScaleType st=kDefaultScaling, int layer_index=-1, int targetProc=0 );
	void CondenseToMap( TMap& map, unsigned long background, EScaleType st=kDefaultScaling, int layer_index=-1, int targetProc=0 );
	int  CopyToMap( TMap& map, Bool swap=FALSE, EScaleType st=kDefaultScaling, int layer_index=-1, int targetProc=0 );
	inline void CopyData( const float* fdata , int fsize ) { memcpy( (void*) data(), (void*) fdata,  fsize*sizeof(float) ); } 
	
	MultiCoverage&  MultiCoverage::CopyFromMap( TMap& map, int layer_index=-1, int srcProc = -1, const char* name = NULL  );
	
  inline Region2& Region(int iproc=gIProc, int layer_index=-1 ) const { 
		TLayer* l = fMultiGrid->getCellLayer( layer_index );
		TPartition2* p = l->partition();
		return *(p->region(iproc));
  } 

  inline void GlobalRegion( Region2& r, int layer_index=-1  ) const {
		TLayer* l = fMultiGrid->getCellLayer( layer_index );
		l->globalRegion(r);
  } 

  inline float ValueRel( TCell* cell, int dr, int dc ) const {
		cell = fMultiGrid->translateNEWS(cell,dr,dc);
    int32 index = cell->memoryIndex( fActivationIndex );
    if( index < 0 ) return 0.0;
#ifdef DEBUG
    if( index >= fNCells ) { sprintf(gMsgStr,"Illegal Index in MultiCoverage: %d, size = %d ",index,fNCells);  gFatal(); } 
#endif
    return operator[] (index);
  }

	void SetDataValues( float* data, int layOut, int size );


/*
  inline Pix MapPix( Pix p, DistributedGrid& g ) { return fPSet->MapPix(p,g); }
	inline Pix MapPixFast( Pix p, DistributedGrid& g ) { return fPSet->MapPixFast(p,g); }

  inline const OrderedPoint* GetPoint( Point2& point ) const { 
    PointRef pr( point(0), point(1), Region() );
    return fPSet->GetPoint(pr);
  }

  inline Pix GetPix ( Point2& point ) const { 
    const OrderedPoint* pt = GetPoint(point); 
    return (pt) ? fPSet->GetPix(*pt) : (Pix)0; 
  }

  virtual void Dump( FILE* oFile ){ fprintf(oFile,"MultiCoverage Dump:\n"); TextDump(oFile); }


  MultiCoverage& operator=( ByteGrid& map );

  inline DistributedGrid* PointSet() const {  return fPSet; }

  int CopyData( const float* fdata, int nelem ) { 
    if( nelem != fNObjects ) { gPrintErr("Bad Data size MultiCoverage::CopyData"); return 0; } 
    deep_copy(fdata,nelem);
    return nelem;
  }


  void Condense( byte*& cData, int targetProc, int bsize, EDataLayout layout, float s = 1.0, float o = 0.0 );

  float AggregateValue( const OrderedPoint& point, int scale );


  
  inline float Value( Pix p ) const {
    const int index = fPSet->Index( p );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in MultiCoverage");  } 
#endif
    return operator[] (index);
  }

  
  inline float ValueRel( const OrderedPoint& point, int dr, int dc ) const {
    Pix p0 = fPSet->GetPix(point);
    return  ((p0) ? ValueRel( p0, dr, dc ) : 0.0);
  }

  inline float& operator() ( Pix p0, Point2 ptrel ) {
    Pix rp = fPSet->TranslateByLinks( p0, ptrel );    
    if(rp == 0) { 
#ifdef DEBUG
      if(p0 == 0) sprintf(gMsgStr,"Null Pix in ValueRel\n"); 
      if(gDebug>2){
	const OrderedPoint& point = fPSet->GetPoint(p0);
	sprintf(gMsgStr,">>Bad Point Access: (%d,%d)->(%d,%d)\n",point(0),point(1),ptrel(0),ptrel(1)); 
	gPrintLog(); 
      }
#endif
      return fGarbage = 0;
    }
    const int index = fPSet->Index( rp );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in MultiCoverage");  }
#endif
    return elem(index);
  }

  inline float& operator() ( Pix p0, int lIndex, int nLinks ) {
    Pix rp = fPSet->TranslateByLinkIndex( p0, lIndex, nLinks );    
    if(rp == 0) { 
#ifdef DEBUG
      if(p0 == 0) sprintf(gMsgStr,"Null Pix in ValueRel\n"); 
      if(gDebug>2){
	const OrderedPoint& point = fPSet->GetPoint(p0);
	sprintf(gMsgStr,">>Bad Point Access: (%d,%d)->( L:%d, nL:%d )\n",
		point(0),point(1),lIndex,nLinks); 
	gPrintLog(); 
      }
#endif
      return fGarbage = 0;
    }
    const int index = fPSet->Index( rp );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in MultiCoverage");  }
#endif
    return elem(index);
  }

  inline float ValueRel( Pix p0, int dr, int dc ) const {
    Pix rp = fPSet->TranslateByLinks( p0, dr, dc );
    if(rp == 0) { 
#ifdef DEBUG
      if(p0 == 0) sprintf(gMsgStr,"Null Pix in ValueRel\n"); 
      if(gDebug>2){
				const OrderedPoint& point = fPSet->GetPoint(p0);
				sprintf(gMsgStr,">>Bad Point Access: (%d,%d)->(%d,%d)\n",point(0),point(1),dr,dc); 
				gPrintLog(); 
      }
#endif
      return 0.0;
    } 
    const int index = fPSet->Index( rp );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in MultiCoverage"); } 
#endif
    return  operator[] (index);
  }
 

  inline float& operator() ( const OrderedPoint& point ) {
    int index = fPSet->Index( point );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in MultiCoverage");  }
#endif
    return elem(index);
  }

  inline int SetValue( Pix p, float value ) {
    const int index = fPSet->Index( p );
    elem(index) = value;
#ifdef DEBUG
		if( ( gDebug > 1 ) && DistributedGrid::fDebug ) { 
			sprintf(gMsgStr,"\nSet Point (%d) -> (%f)",index,value);
			gPrintLog(); 
		}
#endif
    return index;
  }
  

  inline float& operator()( Pix p ) {
    const int index = fPSet->index( p );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in MultiCoverage");  }
#endif
    return elem(index);
  }


  void SubSet(byte index, float value);
  float RemoteValue( Pix p, int recvProc=0 );    // p should be non-zero on only one proc!
  void CopyToMap(  TMap2& map, Bool swap = FALSE, Bool directScale = FALSE );           // Convert to map;
//  void CopyToMapRescaled( TMap2& m, float min, float max, 
//    float umin, float umax, Bool rescale = FALSE);

	
*/
};

#endif

