#include "TMap.h"
#include "Globals.h"
#include "HDFMgr.h"
#include "MML_Pipe.h"
#include "MML_Time.h"

#ifdef HAS_HDF5
extern "C" {
#include "hdf5.h"
}
#else
extern "C" {
#include "sme_types.h"
}
#endif

#ifdef HAS_NETCDF
#include "netcdf.hh"
#endif

const int MAXDATALNLEN = 10000;

extern "C" {
#include "GRASS.h"
}

EInfoIndex TMap::kMapType = kO_Info0;
EInfoIndex TMap::kMapClass = kO_Info1;
EInfoIndex TMap::kRegionSet = kO_Info3;
EInfoIndex TMap::kDataType = kO_Info2;

int  TMap::Read( LayerConfig& lc, int nbytes ) {
  const char* format = ( lc.Format() == "" ) ? ((const char*)NULL) :  lc.Format().chars();
  return Read( lc.MapSource(), lc.MapName(), 1, &lc, format );
}

int  TMap::Read( const char* mapSrc, const char* fileName, int nbytes, LayerConfig* lc, const char* format ) {
  fMapName = fileName;
  switch ( mapSrc[0] ) {
  case 'H':
    SetObjInfo(kMapType,kHDF);
    fMapName = fileName;
    fSD_id = HDFM::Open(HDFM::kFInput);
    return SDRead();
  case 'M': {
    SetObjInfo(kMapType,kMap2);
    CPathString pathName(Env::DataPath());  
    pathName.Add(fileName);
    if( (format == NULL) || (strcmp(format,"default") == 0) ) {
	  return ReadM2File(pathName); 
	} else {
	  CPathString templateName(Env::DataPath());  
	  templateName.Add(format);
	  return ReadM2File2( pathName, templateName ); 
	}
  }
  case 'd': {
    SetObjInfo(kMapType,kGeometry);
    CPathString pathName(Env::DataPath());  
    pathName.Add(fileName);
    return readDepthFile(pathName.Path(),lc); }
  case 'P': {
    SetObjInfo(kMapType,kMap2);
    CPathString pathName(Env::DataPath());  
    pathName.Add(fileName);
    return ReadPPMFile(pathName); }
  case 'G': 
    SetObjInfo(kMapType,kGRASS);
    return ReadGrassMap(fileName,nbytes);
  case 'A': 
    SetObjInfo(kMapType,kArc);
    if( format == NULL ) {
	  return ReadArcMap(fileName);
	} else {
	  CPathString pathName(Env::DataPath());  
	  pathName.Add(format);
	  return ReadArcMap2( fileName, pathName ); 
	}
  case 'V': 
    SetObjInfo(kMapType,kArcSVF);
    return ReadArcInfoSVF(fileName);
  case 'I': 
    SetObjInfo(kMapType,kIDRISI);
    return ReadIDRISIMap(fileName);
  case 'E': 
    SetObjInfo(kMapType,kERDAS);
    return ReadERDASMap(fileName);
  case 'D':
    SetObjInfo(kMapType, kCME);
		CPathString path(Env::DataPath());
		path.Add("cache.in");  
		path.Add(fileName);  
    return ReadCmeMap(path.Path(".sds"), nbytes);
  }
  sprintf(gMsgStr,"Unknown map type %s in TMap::Read: %s",mapSrc,fileName);
  gPrintErr();  
}

int TMap::readDepthFile( const CString& pathName, LayerConfig* lc  ) {
	if( lc == NULL ) { 
	  sprintf(gMsgStr,"Undefined LayerConfig in TMap::readDepthFile: %s", pathName.chars()); gFatal(); 
	} else {
	  sprintf(gMsgStr,"Reading GeometryFile(rows=%d,cols=%d,%d): %s", lc->Dim(0), lc->Dim(1), lc->Dim(2), pathName.chars() ); gPrintScreen(); 
	}
	int Nx = lc->Dim(1); // cols
	int Ny = lc->Dim(0); // rows
	int Nz = lc->Dim(2); 
	int tt, ix, iy, jx, jy, jz, index, debug = 0;
	FILE *tfile;
	int ncols = Nx*Ny;
	float* h = new float[ncols*Nz];
	byte *in = new byte[2*Nx];
	float *debug_depth = new float[Ny];
	byte *inptr, t1, t2;
	float* depth_array = lc->GetDepthArray();

	if (gIProc == 0) {
		tfile = fopen (pathName.chars(), "r");
	}
	if ((tfile == 0) && (gIProc == 0)) {
		gFatal ("fopen on geometry failed.\n");
	}

	if( debug ) {
		fprintf(Env::LogFile(),"\nDepth Data dump, jz = %d", Nz - 1 );
	}
	for (jy = 0; jy < Ny; jy++) { debug_depth[jy] = 0.0; }
	
	for (jz = 0; jz < Nz; jz++ ) {
		for (jy = 0; jy < Ny; jy++) {
			if ( gIProc == 0 ) {
			  fread (in, sizeof (char), 2 * Nx, tfile);
			}
#ifdef USE_MPI
			MPI_Bcast (in, 2 * Nx, MPI_CHAR, 0, MPI_COMM_WORLD);
#endif
			inptr = in;
			for (jx = 0; jx < Nx; jx++) {
				t1 = *inptr++;
				t2 = *inptr++;
				tt = t1 * 256 + t2;
				index = jz*ncols + jy*Nx + jx;
				h[ index ] = /* 0.01 * */ (float) tt; /* orf NOTE... multiplied by 100 in readcb.c! this gets us around
													grid resolutions less than 1 meter while still using JA format */
				if( (debug>1) && (jz == 0) ) {
					fprintf(Env::LogFile(),"\n%4i (%3i, %3i, %3i): %f", index, jx,jy,jz, h[index] );
				}
				if( jx == (Nx-1) ) {
				  debug_depth[jy] += (float) tt;
				}
			}
		}
	}
	for (jy = 0; jy < Ny; jy++) { 
	  fprintf(Env::LogFile()," %5.1f ",debug_depth[jy]);
	}

	if (gIProc == 0) {
	  fclose (tfile);
	}
	
	ByteGrid::ReAlloc( 0, 0, Ny-1, Nx-1, 1 );  // row_start_coord, col_start_coord, row_end_coord, col_end_coord, nbytes 
	
	float depth, thresh = 1.0e-3;
	TMap* depthMap = new TMap( (const Region2*)this, (EDataType) kFloats );
	lc->setDepthMap(depthMap);

	if( debug) { 
	  sprintf(gMsgStr,"\nDepth Data: (Nx,Ny) = (%d,%d) ",Nx,Ny); gPrintLog();
	}
	for (jy = 0; jy < Ny; jy++) {
		if( debug) { 
		  sprintf(gMsgStr,"row %3i: ",jy); gPrintLog();
		}
		for (jx = 0; jx < Nx; jx++) {
			index = jy*Nx + jx;
			inptr = fData + index;
			*inptr = 0;
			if( debug > 1 ) { 
			  sprintf(gMsgStr,"\nDepth array for (%d, %d), index = %d ",jx,jy,index); gPrintLog();
			}
			float total_depth = 0.0;
			for (jz = 0; jz < Nz; jz++) {
			  depth = h[ index + jz*ncols ];
			  total_depth += depth;
			  if( (debug>1) && (jx==70) ) {  sprintf(gMsgStr," %f : %f ",depth,total_depth); gPrintLog(); }
			  if( depth > thresh ) {
				if( depth > depth_array[jz] ) { depth_array[jz] = depth; }
				if( *inptr == 255 ) { gFatal("Memory Overflow in readGeometry"); }
				*inptr += 1;
			  }
			}
			depthMap->SetValue( jy, jx, total_depth );
			if( debug) { 
			  fprintf(Env::LogFile()," %5.1f",total_depth);
			}
		}
	}	
	delete[] h;
	delete[] in;
	return 0;
}


int TMap::ReadCategories( const char* mapSrc, const char* fileName ) {
  switch ( mapSrc[0] ) {
  case 'M': {
    SetObjInfo(kMapType,kMap2);
    CPathString pathName(Env::DataPath());  
    pathName.Add(fileName);
    return ReadM2Categories(pathName); }
	default:
		gPrintErr("Can't read categories from this map type.");
  }
}

int TMap::Write( int index, Bool lastWrite ) {
/*
	int bi = Env::BatchIndex();
	if( bi >= 0 ) {
		if( bi > 0 ) { fDSetName.truncate(3); }
		fDSetName.appendIndex(bi,3);
	}
*/
  float time = gTime();
  CString basename( fDSetName.contains('-') ? fDSetName.before('-') : fDSetName );
  fDSetName = basename; fDSetName  += "-";  fDSetName += time;
  EMapType mt = (EMapType) GetObjInfo(kMapType);
  fWIndex = index;
  switch ( mt  ) {
  case kHDF: {
    CPathString pathName(Env::ArchivePath()); 
    pathName.Add("Animation");
    return DumpHDF(pathName,index,1,NULL); }
  case kHDF5: {
    CPathString pathName(Env::ArchivePath()); 
    pathName.Add("Animation");
    return DumpHDF5(pathName,fDSetName.chars()); }
  case kIOex:
    fSD_id = HDFM::Open(HDFM::kFOutput);
    return SDWrite();
  case kSDS:
    return SDSWrite(index);
  case kCME:
    return WriteCmeMap( fDSetName, index, gTime(), lastWrite );
  case kMap2: case kBinary: case kPPM: {
    CPathString pathName(Env::ArchivePath());
    pathName.Add("Maps");
    return DumpMap(pathName, index, 1, NULL); }
  case kGRASS: {
    CString name(fDSetName); 
    if( index >= 0 ) {
	  name += "."; 
	  name.appendIndex(index); 
	}
    return WriteGrassMap(name);
  } case kArc: {
    CPathString mapName(Env::ArchivePath());
    mapName.Add("Maps");  
    CString mapFileName( downcase(fDSetName) );
    mapFileName += "arc"; 
    if( index >= 0 ) {
	  mapFileName += "."; 
	  mapFileName.appendIndex(index); 
	}
    mapName.Add(mapFileName);
	return WriteArcMap(mapName.Path());
  } case kASCII: {
	CPathString mapName(Env::ArchivePath());
    CString mapFileName( fDSetName );
	mapName.Add("Maps");  
    mapFileName += "ASCII";
    if( index >= 0 ) {
	  mapFileName += "."; 
	  mapFileName.appendIndex(index); 
	}
    mapName.Add(mapFileName);
	return WriteASCIIMap(mapName.Path());
  }
  }
}


void TMap::ReadTextData( FILE* file, const char* name ) {
	int idata, tst, nelem = NElements(), offset;
	SetDataType(kInts);
	Allocate();
	int ir, ic;
	for( int i=0; i< nelem; i++ ) {
		Util::skip_white(file);
		tst = fscanf(file,"%d",&idata);
		if( tst < 1 ) { sprintf(gMsgStr,"Format error in Map data: %s", name); gFatal(); }
		fIData[i] = idata;
	}
}

void TMap::ReadTextData2( FILE* file, const char* name, int row_min, int row_max, int col_min, int col_max, int rows, int cols ) {
	int idata, tst, ir, ic, index=0;
	SetDataType(kInts);
	Allocate();
	for (int ir = 0; ir < rows; ir++) {
	  for (int ic = 0; ic < cols; ic++) {
		Util::skip_white(file);
		tst = fscanf(file,"%d",&idata);
		if( tst < 1 ) { sprintf(gMsgStr,"Format error in Map data: %s", name); gFatal(); }
		  if( (ir >= row_min) && (ir <= row_max) && (ic >= col_min) && (ic <= col_max)) {
			fIData[index++] = idata;
		  } 
	  }	
	}
}


void TMap::BroadcastData( int srcProc, char* info, int mergeMode ) {
#ifdef USE_MPI
  EDataType dtype = (EDataType) GetObjInfo(kDataType);
  if( (dtype != kFloats) && (dtype != kInts)  ) {
	TMap2::BroadcastData( srcProc, info, mergeMode );
  } else { 
	int infosize = (info) ? strlen(info)+1 : 0;
	char info_in[kMapInfoSize]; info_in[0] = 0;
	infosize = ( infosize < kMapInfoSize ) ? infosize : kMapInfoSize;
	int memLen=nelem();
	TMsgBuff msgBuff(1000,fComm);
	if( gIProc == srcProc ) {
	  msgBuff.PackInt(infosize);
	  msgBuff.PackInt(lower(0));
	  msgBuff.PackInt(lower(1));
	  msgBuff.PackInt(upper(0));
	  msgBuff.PackInt(upper(1));
	  if( infosize > 0 ) msgBuff.PackString(info);
	  if( memLen > 0 ) { 
		switch(dtype) {
		  case kFloats: msgBuff.PackFloat(fFData,memLen); break;
		  case kInts:   msgBuff.PackInt(fIData,memLen);   break;
		}
	  }
	}
	msgBuff.BCast(srcProc);			
	if( gIProc != srcProc ) {
	  infosize = msgBuff.UnPackInt();
	  int l0 = msgBuff.UnPackInt();
	  int l1 = msgBuff.UnPackInt();
	  int u0 = msgBuff.UnPackInt();
	  int u1 = msgBuff.UnPackInt();
	  if( ((fFData==NULL) && (dtype == kFloats)) || ((fIData==NULL) && (dtype == kInts)) ) {
		  setlower( l0, l1 );
		  setupper( u0, u1 );  
		  Allocate();
	  }
	  if( infosize > 0 ) { msgBuff.UnPackString(info_in,infosize); }
	  int memLen=nelem();
	  if( memLen > 0 ) { 
		switch(dtype) {
		  case kFloats: msgBuff.UnPackFloat(fFData,memLen); break;    // overWrite
		  case kInts:   msgBuff.UnPackInt(fIData,memLen);   break;
		}		    
	  }
	}
  }
#endif
}

/*
void  TMap::CookieCutCov( MultiCoverage& cov, int layer_index ) {
  int local_debug = 0;
  SetDataType(kFloats);
  Allocate();
  ByteGrid::ReAlloc(1);
  float nullValue = FLT_MAX;
  Set(nullValue);
	ActivationLayer* al = cov.activationLayer();
	TLayer* l = cov.multiGrid()->getCellLayer( layer_index );
	TPartition2* p = l->partition();
	int32 ncells = al->nActiveCells();
	float value; unsigned long ival;
	for( TCell* c = l->firstCell(); c;  l->nextCell(c) ) {
		Point2& p = l->getCellLocPoint( c );
		if( inside(p) ) {
			value = al->mapValue( c, cov.activationIndex(), (float*)cov.Data(), kIntensive, nullValue );
			byte ig = c->GetObjInfo( TCell::kGhostIndex );
			SetValue( p(0), p(1), value );
			ByteGrid::SetValue( p(0), p(1), ig );
			if( local_debug > 1 ) { 
				int32 cellindex = c->Index();
				int32 memindex = c->memoryIndex( al->Index() );
				sprintf(gMsgStr,"cell (%d:%d), setting value (%d,%d) -> %f",cellindex,memindex,p(0),p(1),value); gPrintLog(); 
			}
		}
	}
}
*/

int TMap::ReadCmeMap(const char* name, int nbytes) {
#ifdef HAS_HDF

  // read from SDS file with global replication index
  int32 sd_id  = SDstart(name, DFACC_RDONLY);
  if( sd_id == FAIL ) { sprintf( gMsgStr, "Can't open SDS file %s", name ); gPrintErr(); return FAIL; }
  int32 sds_id = SDselect(sd_id, 0);
  int32 rank, type, attrs, dims[4];
  char nm[256];

  // read dimension info
  SDgetinfo(sds_id, nm, &rank, dims, &type, &attrs);

  // read time dimension scale
  int32 dim_id = SDgetdimid(sds_id, 0);
  float* times = new float[dims[0]];  /* TBC might not be needed */
  SDgetdimscale(dim_id, times);
  void* data = NULL;

	switch(type) {
		case DFNT_INT8:
			ByteGrid::ReAlloc( 0, 0, dims[3]-1, dims[2]-1, 1 );
			data = fData;
		break;
		case DFNT_INT16:
			ByteGrid::ReAlloc( 0, 0, dims[3]-1, dims[2]-1, 2 );
			data = fData;
		break;
		case DFNT_INT32:
			ByteGrid::ReAlloc( 0, 0, dims[3]-1, dims[2]-1, 4 );
			data = fData;
		break;
		case DFNT_FLOAT:
			SetFields( dims[3]-1, dims[2]-1, 0 );
			fFData = new float[ nelem() ];
			data = fFData;
		break;
		default:
			sprintf(gMsgStr,"Illegal SDS type for %s",name); gPrintErr();
			return FAIL;
	} 
  // read the specified replication
  // TBC use time frame 0 for now
  int32 edges[4];
  int32 start[4];

  edges[0] = 1;
  edges[1] = 1;
  edges[2] = dims[2];
  edges[3] = dims[3];

  start[0] = 0; /* TBC */
  start[1] = Env::ReplicationIndex();
  start[2] = 0;
  start[3] = 0;
  
  // TBI set extent, nbytes
  
  int32 ret = SDreaddata(sds_id, start, NULL, edges, data);

  int    btop;                    // if TRUE, <0,0> in maps is at
                                   // top left; else bot left
	int32   boolval;
  float  xscale, yscale;         // scale and offset for coordinate
  float  xofs, yofs;             // translation (only on maps)
  float    umin;                 // min value to return
  float    umax;                 // max value to return
  float cat_min, cat_max;          // min & max categories.
  
	int32 ai = SDfindattr(sd_id, "coord_btop");
	SDreadattr(sd_id, ai, &btop);
	ai = SDfindattr(sd_id, "coord_xscale");
	SDreadattr(sd_id, ai, &xscale);
	ai = SDfindattr(sd_id, "coord_yscale");
	SDreadattr(sd_id, ai, &yscale);
	ai = SDfindattr(sd_id, "coord_xoffset");
	SDreadattr(sd_id, ai, &xofs);
	ai = SDfindattr(sd_id, "coord_yoffset");
	SDreadattr(sd_id, ai, &yofs);
	ai = SDfindattr(sd_id, "scale_min");
	SDreadattr(sd_id, ai, &umin);
	ai = SDfindattr(sd_id, "scale_max");
	SDreadattr(sd_id, ai, &umax);
	ai = SDfindattr(sd_id, "n_cats");
	SDreadattr(sd_id, ai, &boolval);
	ai = SDfindattr(sd_id, "data_min");
	SDreadattr(sd_id, ai, &cat_min);
	ai = SDfindattr(sd_id, "data_max");
	SDreadattr(sd_id, ai, &cat_max);
	
	if (boolval > 0) {          // has categories, so it's integer
	  char cat_id[64];
	  for (int i = (int)cat_min; i <= (int)cat_max; i++) {
	      sprintf(cat_id,"%d", i);
	      if ((ai = SDfindattr(sd_id, cat_id)) != FAIL) 	{
					SDreadattr(sd_id, ai, nm);
					fCatMap[i] = (Pix) new CString(nm);
				}
		}
	}

	fS[0] = xscale;
	fS[1] = yscale;
  
  // TBI read category names
  
  delete times; /* TBC should go away if above alloc goes away */
      
  SDendaccess(sds_id);
  SDend(sd_id);

#endif
  return 1;
}

int TMap::WriteCmeMap( const char* name, int time_index, float time_value, Bool last_write ) {
	int32 istat = -1;
#ifdef HAS_HDF

  static floatVec times;
  int32 sd_id;
  
  CPathString path(Env::DataPath());
  path.Add("cache"); path.Add("out"); path.Add(name); 
  sd_id  = SDstart( path.Path( ".sds" ), DFACC_RDWR );

  // read from SDS file with global replication index
  int32 ndatasets, nglobal_attr, sds_id;
  if( SDfileinfo( sd_id, &ndatasets, &nglobal_attr ) == FAIL ) { gPrintErr("Error reading SDS FIle"); exit(-1); }
	if( ndatasets > 0 )  { sds_id = SDselect(sd_id, 0); }
	else {
		int32 rank, type, attrs, dims[4], dType;
		char nm[256];

		dims[0] = SD_UNLIMITED;
		dims[1] = 1;
		dims[2] = extents(0);
		dims[3] = extents(1);
		switch(NBytes()) {
		case 1: dType = DFNT_INT8; break;
		case 2: dType = DFNT_INT16; break;
		case 4: dType = DFNT_INT32; break;
		default: printf("ERROR, Illegal data size in SDSWrite: %d\n",NBytes()); return -1;
		}
		fSDS_id = SDcreate(sds_id,nm,dType,4,dims);
		SDsetattr(sds_id,(char*)"scale",DFNT_FLOAT32,2,fS);
		int32 R[4] = { lower(0), lower(1), upper(0), upper(1) };
		SDsetattr( sds_id, (char*)"region", DFNT_INT32, 4, R );
		int nbytes = NBytes();
		SDsetattr( sds_id, (char*)"nbytes", DFNT_INT32, 1, &nbytes );
	}
  int32 start[3], edges[3];
  edges[0] = 1;
  edges[1] = 1;
  edges[2] = extents(0);
  edges[3] = extents(1);

  start[0] = time_index;
	start[1] = 0;
  start[2] = 0;
  start[3] = 0;
  
	istat = SDwritedata(sds_id, start, NULL, edges, (void*)Data());
  if(istat != 0) printf("\nerror writing SDS file: %s\n",name);
  
  times.add(time_index) = time_value;
  if(last_write) { 
		int32 dim_id = SDgetdimid(sds_id, 0);
		SDsetdimscale(dim_id, time_index+1, DFNT_FLOAT32, (void*) times.data() ); 
	}
  
  /* read attributes when/if I know where to put them
  int32 ai = SDfindattr(sd_id, "coord_btop");
  SDreadattr(sd_id, ai, &_btop);
  ai = SDfindattr(sd_id, "coord_xscale");
  SDreadattr(sd_id, ai, &_xscale);
  ai = SDfindattr(sd_id, "coord_yscale");
  SDreadattr(sd_id, ai, &_yscale);
  ai = SDfindattr(sd_id, "coord_xoffset");
  SDreadattr(sd_id, ai, &_xofs);
  ai = SDfindattr(sd_id, "coord_yoffset");
  SDreadattr(sd_id, ai, &_yofs);
  ai = SDfindattr(sd_id, "scale_min");
  SDreadattr(sd_id, ai, &_umin);
  ai = SDfindattr(sd_id, "scale_max");
  SDreadattr(sd_id, ai, &_umax);
  ai = SDfindattr(sd_id, "type");
  SDreadattr(sd_id, ai, &_type);
  ai = SDfindattr(sd_id, "name");
  SDreadattr(sd_id, ai, nm);
  */
  
  // TBI read category names
        
  SDendaccess(sds_id);
  SDend(sd_id);

#endif
  return istat;
}

/*
int TMap::ReadGrassMap(const char* name, int nbytes) {

#ifdef HAS_GRASS_4_2

  if( gIProc == 0 || gGlobalFileSystem ) { 

		Cell_head ch;
		grass_get_region(&ch);
		if( GetObjInfo(kRegionSet) ) {
			if( (*this) != ch ) { gPrintErr( "Can't change Map region."); return 0; }
		} else {
			if( SetGeoRegion( ch ) ) { SetObjInfo(kRegionSet,True); }
		}
		switch( GetObjInfo(kMapClass) ) {
		case kSurfaceMap: {
			SetDataType(kFloats);
			if(fFDataG==NULL) { fFData = new float [ ch.rows * ch.cols ]; }
			float NullVal = FLT_MAX;
			grass_read_float_raster_map ( (char*) name, fFDataG, &NullVal );
			if(nbytes>0) {
				SetDataType(kBytes);
				ByteGrid::ReAlloc( 0, 0, ch.rows, ch.cols, nbytes );
				CopyDataFrom( fFDataG, nbytes, fS[1], fS[0], *this);
			}
			} break;
		case kClassMap: {
			SetDataType(kInts);
			if( fIDataG == NULL ) { fIData = new int [ ch.rows * ch.cols ]; }
			int NullVal = INT_MAX;
			grass_read_int_raster_map ( (char*)name, fIDataG, &NullVal );
			if(nbytes>0) {
				SetDataType(kBytes);
				ByteGrid::ReAlloc( 0, 0, ch.rows, ch.cols, nbytes );
				CopyDataFrom( fIDataG, nbytes, *this);
			}
			} break;
		case kUnknown:
			gPrintErr("Unknown Map Type.");
			return 0;
		}
	}
  if( !gGlobalFileSystem ) BroadcastData(0);
  
#endif
  return 1;
}


int TMap::WriteGrassMap(const char* name) {
  if( gIProc != 0 ) return 0; 
#ifdef HAS_GRASS_4_2
  if( GetObjInfo(kRegionSet) == 0 ) {
    grass_set_region_rows_and_cols (extents(0), extents(1));
  } else {
    SetObjInfo(kRegionSet,1);
  }
  switch( GetObjInfo(kMapClass) ) {
  case kSurfaceMap: {
    float NullVal = FLT_MAX;
    grass_write_float_raster_map ( (char*)name, fFDataG, &NullVal );
    } break;
  case kClassMap: {
    int NullVal = INT_MAX;
    grass_write_int_raster_map ( (char*)name, fIDataG, &NullVal );
    } break;
  case kUnknown:
    gPrintErr("Unknown Map Type.");
    return 0;
  }
#endif
  return 1;
}
*/
int TMap::ReadGrassMap(const char* name, int nbytes) {

	int rv = 0;
#ifdef HAS_GRASS_4_2
  if( gIProc == 0 || gGlobalFileSystem ) { 

                Cell_head ch;
                grass_get_region(&ch);
                if( GetObjInfo(kRegionSet) ) {
                        if( (*this) != ch ) { gPrintErr( "Can't change Map region."); return 0; }
                } else {
                        if( SetGeoRegion( ch ) ) { SetObjInfo(kRegionSet,True); }
                }
                switch( GetObjInfo(kMapClass) ) {
                case kSurfaceMap: {
												SetDataType(kFloats);
                         if(fFData==NULL) { fFData = new float [ ch.rows * ch.cols ]; }
                        float NullVal = FLT_MAX;
                        if( grass_read_float_raster_map ( (char*)name, fFData, &NullVal ) < 0 ) { rv = -1; }
                        if(nbytes>0) {
																Region2 r(0, 0, ch.rows, ch.cols);
                                ReAlloc( r );
                                CopyDataFrom( fFData, nbytes, fS[1], fS[0], *this);
                        } 
								} break;
                case kClassMap: {
												SetDataType(kInts);
                        if( fIData == NULL ) { fIData = new int [ ch.rows * ch.cols ]; }
                        int NullVal = INT_MAX;
                        if( grass_read_int_raster_map ( (char*)name, fIData, &NullVal ) < 0 ) { rv = -1; }
                        if(nbytes>0) {
																Region2 r(0, 0, ch.rows, ch.cols);
                                ReAlloc( r );
                                CopyDataFrom( fIData, nbytes, *this);
                        } 
								} break;
                case kUnknown:
                        gPrintErr("Unknown Map Type.");
                        return 0;
                }
        }
  if( !gGlobalFileSystem ) BroadcastData(0);
  
#endif
  return rv;
}


int TMap::WriteGrassMap(const char* name) {
  if( gIProc != 0 ) return 0; 
  static int debug = 0;
#ifdef HAS_GRASS_4_2
	if( debug == 0 ) {
		if( GetObjInfo(kRegionSet) == 0 ) {
			grass_set_region_rows_and_cols (extents(0), extents(1));
		} else {
			SetObjInfo(kRegionSet,1);
		}
		switch( GetObjInfo(kMapClass) ) {
		case kSurfaceMap: {
			float NullVal = FLT_MAX;
			grass_write_float_raster_map ( (char*)name, fFData, &NullVal );
			} break;
		case kClassMap: {
			int NullVal = INT_MAX;
			grass_write_int_raster_map ( (char*)name, fIData, &NullVal );
			} break;
		case kUnknown:
			gPrintErr("Unknown Map Type.");
			return 0;
		}
	}
#endif
	if( debug ) {
		int rows = extents(0), cols = extents(1);	
		sprintf( gMsgStr, "\n\n Write GRASS Map(%f): %s \n", gTime(), name ); gPrintLog();
		for (int i=20; i< 25; i++ ) {
			int k = 17 + i*cols;
			sprintf( gMsgStr, " %.3g %.3g %.3g %.3g %.3g ", fFData[ k++ ], fFData[ k++ ], fFData[ k++ ], fFData[ k++ ], fFData[ k++ ] ); gPrintLog();
		}
	}
  return 1;
}

/*
int TMap::ReadGrassMap(const char* name, int nbytes) {

#ifdef HAS_GRASS_4_2

  if( gIProc == 0 || gGlobalFileSystem ) { 

		Cell_head ch;
		grass_get_region(&ch);
		if( GetObjInfo(kRegionSet) ) {
			if( (*this) != ch ) { gPrintErr( "Can't change Map region."); return 0; }
		} else {
			if( SetGeoRegion( ch ) ) { SetObjInfo(kRegionSet,True); }
		}
		switch( GetObjInfo(kMapClass) ) {
		case kSurfaceMap: {
			if(fFData==NULL) { fFData = new float [ ch.rows * ch.cols ]; }
			float NullVal = FLT_MAX;
			grass_read_float_raster_map ( name, fFData, &NullVal );
				if(nbytes>0) {
					ByteGrid::ReAlloc( 0, 0, ch.rows, ch.cols, nbytes );
					CopyDataFrom( fFData, nbytes, fS[1], fS[0], *this);
				}
			} break;
		case kClassMap: {
			if( fIData == NULL ) { fIData = new int [ ch.rows * ch.cols ]; }
			int NullVal = INT_MAX;
			grass_read_int_raster_map ( name, fIData, &NullVal );
			if(nbytes>0) {
					ByteGrid::ReAlloc( 0, 0, ch.rows, ch.cols, nbytes );
					CopyDataFrom( fIData, nbytes, *this);
				}
			} break;
		case kUnknown:
			gPrintErr("Unknown Map Type.");
			return 0;
		}
	}
  if( !gGlobalFileSystem ) BroadcastData(0);
  
#endif
  return 1;
}


int TMap::WriteGrassMap(const char* name) {
  if( gIProc != 0 ) return 0; 
#ifdef HAS_GRASS_4_2
  if( GetObjInfo(kRegionSet) == 0 ) {
    grass_set_region_rows_and_cols (extents(0), extents(1));
  } else {
    SetObjInfo(kRegionSet,1);
  }
  switch( GetObjInfo(kMapClass) ) {
  case kSurfaceMap: {
    float NullVal = FLT_MAX;
    grass_write_float_raster_map ( name, fFData, &NullVal );
    } break;
  case kClassMap: {
    int NullVal = INT_MAX;
    grass_write_int_raster_map ( name, fIData, &NullVal );
    } break;
  case kUnknown:
    gPrintErr("Unknown Map Type.");
    return 0;
  }
#endif
  return 1;
}
*/
int TMap::WriteArcMap(const char* filename) {
	FILE	 *fp;
	int nodata = -2;
  if( gIProc != 0 ) return 0; 
	/* **	Open files */
	
	if (!(fp = fopen (filename, "w"))) {
		sprintf ( gMsgStr, "Can't open %s.\n", filename); gPrintErr();
		return (1);
	}
	/* 	read and write headers for in/output files  */
			
        (void) fprintf (fp, "ncols   %d\n", extents(1) );
        (void) fprintf (fp, "nrows   %d\n", extents(0) );
        (void) fprintf (fp, "xllcorner   %f\n", fC[0](k1) );
        (void) fprintf (fp, "yllcorner   %f\n", fC[0](k0) );
        (void) fprintf (fp, "cellsize    %f\n", fCellsize );
        (void) fprintf (fp, "nodata_value  %d\n", nodata );

	/*   Write array into ascii file 	*/

	for (int ir = 0; ir < extents(0); ir++ ) {
		for (int ic = 0; ic < extents(1); ic++ ) {
			long value = IntValue(ir,ic);
			if( value == -999 ) { fprintf ( fp, "%d ", nodata ); }
			else { fprintf ( fp, "%d ", value ); }
		}
		fprintf(fp,"\n");
	}
	(void) fclose(fp);
	return (0);
}

int TMap::WriteASCIIMap(const char* filename) {
	FILE	 *fp;
	if( gIProc != 0 ) return 0; 
	/* **	Open files */
	
	if (!(fp = fopen (filename, "w"))) {
		sprintf ( gMsgStr, "Can't open %s.\n", filename); gPrintErr();
		return (1);
	}
	/* 	read and write headers for in/output files  */
			
        (void) fprintf (fp, "ncols   %d\n", extents(1) );
        (void) fprintf (fp, "nrows   %d\n", extents(0) );

	/*   Write array into ascii file 	*/

	for (int ir = 0; ir < extents(0); ir++ ) {
		for (int ic = 0; ic < extents(1); ic++ ) {
			float value = FltValue(ir,ic);
			fprintf ( fp, "%f ", value ); 
		}
		fprintf(fp,"\n");
	}
	(void) fclose(fp);
	return (0);
}

int TMap::ReadNetCDFMap( const char* filename, const char* varname ) {
  SetDataType(kFloats);
	
#ifdef HAS_NETCDF
		
  if( gIProc == 0 || gGlobalFileSystem ) { 

	  NcFile nc (filename, NcFile::ReadOnly);

	  // Check if the file was opened
	  if (! nc.is_valid()) {
		gPrintErr( format("Can't read netCDF file %s\n",filename));
		return 0;
	  }
	  
	  NcVar* var = nc.get_var( varname );
	   
	  // Check if the var is defined in file
	  if ( var == NULL ) {
		gPrintErr( format("Can't find var %s in netCDF file %s\n",varname,filename));
		return 0;
	  }

	  NcDim* dim0 = nc.get_dim( 0 );
	  
	  // Check if the dim is defined
	  if ( dim0 == NULL ) {
		gPrintErr( format("Undefined dimension 0 for var %s in netCDF file %s\n",varname,filename));
		return 0;
	  }
	  
	  NcDim* dim1 = nc.get_dim( 1 );
	  
	  // Check if the dim is defined
	  if ( dim1 == NULL ) {
		gPrintErr( format("Undefined dimension 1 for var %s in netCDF file %s\n",varname,filename));
		return 0;
	  }

	  int ncols = (int) dim0->size();
	  int nrows = (int) dim1->size();
	  SetFields( nrows, ncols, 0 );
	  Allocate();
	  
	  NcBool got_data = var->get( fFData, ncols, nrows );
	  
	  if ( !got_data ) {
		gPrintErr( format("Can't get data for var %s in netCDF file %s\n",varname,filename));
		return 0;
	  }
  }
  if( !gGlobalFileSystem ) BroadcastData(0);
  return 1;	
#endif
  return 0;
}

int TMap::WriteNetCDFMap( const char* filename, const char* varname ) {
	
#ifdef HAS_NETCDF
		
  if( gIProc == 0 || gGlobalFileSystem ) { 

	  NcFile nc (filename, NcFile::Replace);

	  // Check if the file was opened
	  if (! nc.is_valid()) {
		gPrintErr( format("Can't open netCDF file %s\n",filename));
		return 0;
	  }
	  
	  // Create dimensions
	  NcDim* cols = nc.add_dim("cols", extents(1) );
	  NcDim* rows = nc.add_dim("rows", extents(0) );

	  // Create variable and their attributes
	  NcVar* P = nc.add_var(varname, ncFloat, cols, rows );
	   
	  NcBool put_data = P->put( fFData, P->edges() );
	  
	  if ( !put_data ) {
		gPrintErr( format("Can't put data for var %s in netCDF file %s\n",varname,filename));
		return 0;
	  } 
  }	
  return 1;
#endif
  return 0;
}

inline int get_line( char* buff, int max_len, FILE* fp ) {
  int ch, ml = max_len-1; char* tbuff = buff;
  while ( (ch = fgetc(fp)) !=  EOF ) {
	if( (ch == '\0') || (ch == '\n') || (ch == '\r') || (ch == '\f') ) {
	  break;
	} else {
	  if( (tbuff-buff) >= ml ) break;
	  *tbuff++ = (char) ch;
	} 
  }
  *tbuff++ = '\0';
  return tbuff-buff;
}

inline void skip_white( FILE* fp ) {
  int ch;
  while ( (ch = fgetc(fp)) !=  EOF ) {
	if( !isspace(ch) ) {
	  ungetc(ch,fp);
	  break;
	} 
  }
}

int TMap::ReadArcMap(const char* filename) {
	FILE	 *fp;
	char	buf[MAXDATALNLEN], *start_point, *end_point;
	int 	nodata, nr, nc;
	float dum1, dum2, dum3;
	
	SetDataType(kFloats);

	/*
		**	Open files
		*/
		
	if( gIProc == 0 || gGlobalFileSystem ) { 
		if (!(fp = fopen (filename, "r"))) {
			(void) printf ("Can't open %s.\n", filename);
			return (1);
		}

		/* 	read and write headers for in/output files  */
		(void) fscanf (fp, "%s %d", buf, &nc);
		(void) fscanf (fp, "%s %d", buf, &nr);
		(void) fscanf (fp, "%s %f", buf, &dum1);
		(void) fscanf (fp, "%s %f", buf, &dum2);
		(void) fscanf (fp, "%s %f", buf, &dum3);
		(void) fscanf (fp, "%s %d", buf, &nodata);
		(void) skip_white( fp );

		SetFields( nr, nc, 0 );
		Allocate();

		/*   Read ascii into array and initialize output array to NODATA  */
		for (int ir = 0; ir < nr; ir++) {
			/* reads a line into 'buf' and gets its length */
			int slen = get_line (buf, MAXDATALNLEN, fp);
			end_point = buf;
			/* advance across columns of buf */
			for (int ic = 0; ic < nc; ic++) {
				/* set start_point to end_point, which is buf */
				start_point = end_point;
				float val = (float)strtod (start_point, &end_point);
				SetValue( ir, ic, val );
			}
		}
		(void) fclose(fp);
	}
	if( !gGlobalFileSystem ) BroadcastData(0);
 	return (0);
}



int TMap::ReadArcMap2( const char* filename, const char* templatename ) {
	FILE	 *fp;
	char	buf[MAXDATALNLEN], *start_point, *end_point;
	int 	nodata, nr, nc;
	float dum1, dum2, dum3;
	
	SetDataType(kFloats);

	/*
		**	Open files
		*/
		
	if( gIProc == 0 || gGlobalFileSystem ) { 
		if (!(fp = fopen (templatename, "r"))) {
			(void) printf ("Can't open %s.\n", templatename);
			return (1);
		}

		/* 	read and write headers for in/output files  */
		(void) fscanf (fp, "%s %d", buf, &nc);
		(void) fscanf (fp, "%s %d", buf, &nr);
		(void) fscanf (fp, "%s %f", buf, &dum1);
		(void) fscanf (fp, "%s %f", buf, &dum2);
		(void) fscanf (fp, "%s %f", buf, &dum3);
		(void) fscanf (fp, "%s %d", buf, &nodata);
		(void) skip_white( fp );
		
		byte* dbuff = new byte[nr*nc];
		int top=0, bottom=nr, left=nc, right=0;
		
		/*   Read ascii into array and initialize output array to NODATA  */
		for (int ir = 0; ir < nr; ir++) {
			/* reads a line into 'buf' and gets its length */
			int slen = get_line (buf, MAXDATALNLEN, fp);
			end_point = buf;
			/* advance across columns of buf */
			for (int ic = 0; ic < nc; ic++) {
				/* set start_point to end_point, which is buf */
				start_point = end_point;
				float val = (float)strtod (start_point, &end_point);
				if( val > 0 ) {
				  if( ir < bottom ) bottom = ir;
				  if( ir > top ) top = ir;
				  if( ic < left ) left = ir;
				  if( ic > right ) right = ir;
				}
				dbuff[ ir*nc + ic ] = ( val > 0 ) ? 1 : 0;
			}
		}
		(void) fclose(fp);

		SetFields( top-bottom+1, right-left+1, 0 );
		Allocate();
//		printf ("Allocating %d float elements for map %s: subgrid(t:%d,b:%d,l:%d,r:%d) \n", nelem(), filename, top, bottom, left, right );
		
		if (!(fp = fopen (filename, "r"))) {
			(void) printf ("Can't open %s.\n", filename);
			return (1);
		}

		/* 	read and write headers for in/output files  */
		(void) fscanf (fp, "%s %d", buf, &nc);
		(void) fscanf (fp, "%s %d", buf, &nr);
		(void) fscanf (fp, "%s %f", buf, &dum1);
		(void) fscanf (fp, "%s %f", buf, &dum2);
		(void) fscanf (fp, "%s %f", buf, &dum3);
		(void) fscanf (fp, "%s %d", buf, &nodata);
		(void) skip_white( fp );

		/*   Read ascii into array and initialize output array to NODATA  */
		for (int ir = 0; ir < nr; ir++) {
			/* reads a line into 'buf' and gets its length */
			int slen = get_line (buf, MAXDATALNLEN, fp);
			end_point = buf;
			/* advance across columns of buf */
			for (int ic = 0; ic < nc; ic++) {
				/* set start_point to end_point, which is buf */
				start_point = end_point;
				float val = (float)strtod (start_point, &end_point);
				if( (ir >= bottom) && (ir <= top) && (ic >= left) && (ic <= right) ) {
				  if( dbuff[ ir*nc + ic ] > 0 ) {
					SetValue( ir - bottom, ic - left, val );
				  } else {
					SetValue( ir - bottom, ic - left, (float) 0.0 );				  
				  }
				}
			}
		}
		(void) fclose(fp);
		delete[] dbuff;
	}
	if( !gGlobalFileSystem ) BroadcastData(0);
 	return (0);
}


int32 TMap::SDWrite( const char* info, const char* units, const char* format ) 
{
	int32 istat = -1;
#ifdef HAS_HDF
  if( gIProc != 0 ) return 0; 
	fSD_id = HDFM::Open(HDFM::kFOutput);
  if( fSDS_id == 0 ) {
    int32 dims[3],dType;
    dims[0] = SD_UNLIMITED;
    dims[1] = extents(0);
    dims[2] = extents(1);
    switch(NBytes()) {
    case 1: dType = DFNT_INT8; break;
    case 2: dType = DFNT_INT16; break;
    case 4: dType = DFNT_INT32; break;
    default: printf("ERROR, Illegal data size in SDSWrite: %d\n",NBytes()); return -1;
    }
    fSDS_id = SDcreate(fSD_id,(char*)DSetName(),dType,3,dims);
    if( fSDS_id == FAIL ) { 
      sprintf(gMsgStr,"Can't Create SDS object %s, SD_id = %d",DSetName(),fSD_id); 
      gPrintErr(); return 0; 
    } else { sprintf(gMsgStr,"Created SDS Object %s, SDS_id = %d",DSetName(),fSD_id); gPrintScreen(); }
    SDsetdatastrs(fSDS_id,(char*)DSetName(),(char*)units,(char*)format,NULL);
    SDsetattr(fSDS_id,(char*)"scale",DFNT_FLOAT32,2,fS);
    SDsetattr(fSDS_id,(char*)"info",DFNT_CHAR8,sizeof(info),(void*)info);
    int R[4] = { lower(0), lower(1), upper(0), upper(1) };
    SDsetattr( fSDS_id, (char*)"region", DFNT_INT32, 4, R );
    int nbytes = NBytes();
    SDsetattr( fSDS_id, (char*)"nbytes", DFNT_INT32, 1, &nbytes );
    HDFM::AddSDS(fSDS_id,HDFM::kFMap,VariableName());
  }

  int32 start[3], edges[3];
  edges[0] = 1;
  edges[1] = extents(0);
  edges[2] = extents(1);

  int32 size = fWIndex;
  if( size < 0 ) {
    int32 dim_id = SDgetdimid( fSDS_id, 0 );
    SDdiminfo( dim_id, NULL, &size, NULL, NULL);
  }
  start[0] = size;
  start[1] = 0;
  start[2] = 0;
  istat = SDwritedata(fSDS_id,start,NULL,edges,Data());

  if(istat != 0) { sprintf(gMsgStr,"Error writing SDS file: %s, SDS_id = %d",DSetName(),fSDS_id); gPrintErr(); }
#endif
  return istat;
}


int32 TMap::SDSWrite( int index ) 
{
  if( gIProc != 0 ) return 0; 
  int32 istat = -1;
#ifdef HAS_HDF
	if( fSDS_id == 0 ) {
    CPathString fileP(Env::ArchivePath());
    fileP.Add("Animation");  fileP.Add(fDSetName); 
    fSD_id = SDstart( fileP.Path(".sds"), DFACC_CREATE );

    int32 dims[3],dType;
    dims[0] = SD_UNLIMITED;
    dims[1] = extents(0);
    dims[2] = extents(1);
    switch(NBytes()) {
    case 1: dType = DFNT_INT8; break;
    case 2: dType = DFNT_INT16; break;
    case 4: dType = DFNT_INT32; break;
    default: printf("ERROR, Illegal data size in SDSWrite: %d\n",NBytes()); return -1;
    }
    fSDS_id = SDcreate(fSD_id,fDSetName.data(),dType,3,dims);
    SDsetattr(fSDS_id,(char*)"scale",DFNT_FLOAT32,2,fS);
    int32 R[4] = { lower(0), lower(1), upper(0), upper(1) };
    SDsetattr( fSDS_id, (char*)"region", DFNT_INT32, 4, R );
    int nbytes = NBytes();
    SDsetattr( fSDS_id, (char*)"nbytes", DFNT_INT32, 1, &nbytes );
  }

  int32 start[3], edges[3];
  edges[0] = 1;
  edges[1] = extents(0);
  edges[2] = extents(1);

  start[0] = index;
  start[1] = 0;
  start[2] = 0;
	istat = SDwritedata(fSDS_id,start,NULL,edges,Data());

  if(istat != 0) printf("\nerror writing SDS file: %s\n",DSetName());
#endif
  return istat;
}

int32 TMap::ExportSetAttributes( int32 sds_id  ) {
#ifdef HAS_HDF
  char info[kAttrStrLen];
  ReadAttr("info",info);
  SDsetattr(sds_id,(char*)"info",DFNT_CHAR8,sizeof(info),info);
  int R[4];
  int32 rindex = SDfindattr(fSDS_id,"region");
  SDreadattr(fSDS_id,rindex,R);
  SDsetattr( sds_id, (char*)"region", DFNT_INT32, 4, R );
  rindex = SDfindattr(fSDS_id,"nbytes");
  SDreadattr(fSDS_id,rindex,&fNBytes);
  SDsetattr( sds_id, (char*)"nbytes", DFNT_INT32, 1, &fNBytes );
  rindex = SDfindattr(fSDS_id,"scale");
  SDreadattr(fSDS_id,rindex,fS);
  SDsetattr(sds_id,(char*)"scale",DFNT_FLOAT32,2,fS);
#endif
	return 1;
}

void TMap::Dump( FILE* oFile, float background, char* label ) const
{
  int rows = extents(0);
  int cols = extents(1);
  int do_ghost = ( ( fData != NULL ) && (GetObjInfo(kDataType) != kBytes) ); 
  float value;

  fprintf(oFile,"\n\n~~~~~~~~~~~~~~~Map Dump: %s~~~~~~~~~~~~~~~~~~\n",(const char*)label);
	for(int ir = lower(0); ir<= upper(0); ir+=increment(0) ) {
		for(int ic = lower(1); ic<= upper(1); ic+=increment(1) ) {
      value = Value(ir,ic);
      if( do_ghost ) { 
				switch( IntValue( ir,ic ) ) {
					case 0: fputc(' ',oFile); break;
					case 1: fputc('*',oFile); break;
					case 2: fputc('@',oFile); break;
					default: fputc('&',oFile); break;
				}
			}
			if( value == background ) { fprintf(oFile,"-------- ");  }
			else { fprintf(oFile,"%8f ",value); }
		}
		fprintf(oFile,"\n");
  }
}

int32 TMap::SDRead( int index, int32 ref, char* info, char* units, char* format   ) {
	int32 istat = 0;
#ifdef HAS_HDF
  if( gIProc == 0 || gGlobalFileSystem ) { 
		fSD_id = HDFM::Open(HDFM::kFInput);
		if( fSDS_id == 0 ) {
			int32 sds_index; 
			if( ref < 0 ) sds_index = HDFM::GetSDS( HDFM::kFMap, DSetName(), VariableName() );
			else sds_index = SDreftoindex(fSD_id,ref); 
			fSDS_id = SDselect(fSD_id,sds_index);
			if( fSDS_id == FAIL ) { 
				sprintf(gMsgStr," Can't find SDS file %s : %s",VariableName(),DSetName()); 
				gPrintErr(); return FAIL;
			}
			int32 R[4];
			int32 rindex = SDfindattr(fSDS_id,"region");
			SDreadattr(fSDS_id,rindex,R);
			rindex = SDfindattr(fSDS_id,"nbytes");
			SDreadattr(fSDS_id,rindex,&fNBytes);
			rindex = SDfindattr(fSDS_id,"scale");
			SDreadattr(fSDS_id,rindex,fS);
			ByteGrid::ReAlloc(R[0],R[1],R[2],R[3],1);
			char name[kAttrStrLen];
			SDgetdatastrs(fSDS_id, name, units, format, NULL, kAttrStrLen );
			DSetName(name);
		} else SDgetdatastrs(fSDS_id, NULL, units, format, NULL, kAttrStrLen );
		ReadAttr("info",info);

		int32 start[3], edges[3];

		edges[0] = 1;
		start[0] = index;
		edges[1] = extents(0);
		edges[2] = extents(1);
		start[1] = 0;
		start[2] = 0;
		istat = SDreaddata(fSDS_id,start,NULL,edges,Data()); 
	}
  if( !gGlobalFileSystem ) BroadcastData(0);
#endif
  return istat;
}
  

int32 TMap::SDFinalize() {
#ifdef HAS_HDF
  EMapType mt = (EMapType)GetObjInfo(kMapType);
  switch ( mt  ) {
  case kIOex:
    if( SDSObject::SDFinalize() ) { return 1; }
    break;
  case kSDS:
    SDendaccess(fSDS_id);
    SDend(fSD_id);
    return 1;
  }
#endif
  return 0;
} 

int TMap::TextDump( CPathString& fileP, int index, int nMaps, const char* info, const char* units, const char* format ) { 
  FILE* outFile;
  CString fileN(fVariableName);
  ((fileN += "_") += fDSetName) += ".TEXT";
  fileP.Add(fileN);
  if(index==0) outFile = fopen(fileP,"w");
  else outFile = fopen(fileP,"a");
  if(outFile==NULL) { sprintf(gMsgStr,"Can't open file: %s\n",fileP.chars()); gPrintErr();  return 0; }
  if(info) fprintf(outFile,"Info = %s\n",info);
  if(units) fprintf(outFile,"Units = %s\n",units);
  if(format) fprintf(outFile,"Format = %s\n",format);
  Dump( outFile ); 
  fclose( outFile ); 
}

int TMap::DumpMap( CPathString& path, int index, int nMaps, const char* info )
{
  EMapType mt = (EMapType)GetObjInfo(kMapType);
  switch ( mt  ) {
  case kMap2:
    return WriteM2File(path,fDSetName,index,info);
  case kPPM:
    return WritePPMFile(path,fDSetName,index,info);
  case kBinary: 
    return WriteBinary(path,fDSetName,index);
  }
}

unsigned short* TMap::getCompressedData( float& fmax, float& fmin) {
  fmax = -FLT_MAX; fmin = FLT_MAX;
  int datasize = extents(0)*extents(1), sdata_ceiling = 256*256-2;
  unsigned short* sdata = new unsigned short[datasize], *sdata_ptr = sdata;
  switch( GetObjInfo(kDataType) ) {
	  case kFloats: { 
		float nullValue = FLT_MAX;
		float *data = fFData, *data_end = fFData + datasize;
		do {
		  if( *data != nullValue ) {
			if( fmax < *data ) fmax = *data;
			if( fmin > *data ) fmin = *data;
		  }
		} while( data++ < data_end );
		data = fFData;
		float s = sdata_ceiling/(fmax-fmin); 
		do {
		  if( *data == nullValue ) { *sdata_ptr++ = 0; }
		  else { *sdata_ptr++ = 1 + (*data - fmin)*s; }
		} while( data++ < data_end );
	  } break;
	  case kInts: {
		int* data = f                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     