#include <stdlib.h>
#include <stdio.h>
#include <dfit.H>  
#include <lex.H>
#include <utility.H>
#include <ctype.h>
#include <time.h>

#define LEX_WEIGHT     (-1)
#define LEX_METHOD     (-2)
#define LEX_BOUNDS     (-3)
#define LEX_WBOUNDS    (-4)
#define LEX_CINT       (-5)
#define LEX_WCINT      (-6)
#define LEX_DBK        (-7)
#define LEX_REFERENCE  (-8)
#define LEX_FILE       (-9)
#define LEX_DATA       (-10)
#define LEX_FREQ       (-11)
#define LEX_VARIABLE   (-12)
#define LEX_INDEX      (-13)
#define LEX_READ       (-14)
#define LEX_THEIL      (-15)
#define LEX_ERRCOMP    (-16)
#define LEX_OUTPUT     (-17)
#define LEX_STEADY     (-18)
#define LEX_INCREASE   (-19)
#define LEX_DECREASE   (-20)
#define LEX_TREND      (-21)

#define LEX_PARMDEF    (-22)

class TLexDfit : public TLexS 
{
  // this understands parameter definitions like parmname=parmvalue
  // '=' must be in terminators list

  TString _parm;
  
protected:
  
  virtual int postprocess_token(int val, const char* tok, int& alt);

public:
  
  const char* parm() { return _parm; }

  TLexDfit(const char* filename, const char* terminators, TArray*
	keywords = NULL) : 
    TLexS(filename, terminators, keywords) 
  {}
  virtual ~TLexDfit() {}
};

int TLexDfit::postprocess_token(int val, const char* tok, int& alt)
{ 
    if (val == LEX_IDENTIFIER)
    {
      _parm = tok;
      char c = nextc();
      if (c  == '=')
	{
	  do 
	    {
	      _parm << c;
	      c = nextc();
	    }
	  while (strchr(terminators(), c) == NULL);
	  val = LEX_PARMDEF;
	}
      pushc(c);
    }
    return TLexS::postprocess_token(val, tok, alt);
}


static char __buf_string[1024];

class VDesc : public TObject
{
public:
  // I know, I know
  TString name;
  double* refdata;
  double* data;
  int nrefdata;
  int ndata;
  TArray refstats;    // stats collection for ref data
  TArray datstats;    // stats for simulation data
  TArray teststats;   // descriptors of selected tests
  TArray outdefs;     // output definition for each var

  int rdatacols;      // number of columns stored in rdata (row-first)
  int datacols;       // number of columns stored in data (row-first)
  
  double testweights[DFIT_MAXTESTS];

  double weight;
  double score;

  VDesc(const char* name);
  virtual ~VDesc();
};

VDesc::VDesc(const char* n) : 
  data(NULL), refdata(NULL), ndata(0), nrefdata(0), name(n)
{
  for (int i = 0; i < DFIT_MAXTESTS; i++)
    testweights[i] = 1.0;
}

VDesc::~VDesc()
{
  if (data != NULL) delete data;
  if (refdata != NULL) delete refdata;
}

int DFit::test_id(const char* test)
{

  if (strncmp(test, "BOUNDS", 6) == 0)
    return MID_BOUNDS;
  if (strncmp(test, "WBOUNDS", 7) == 0)
    return MID_WBOUNDS;
  if (strncmp(test, "FREQ", 4) == 0)
    return MID_FREQ;
  if (strncmp(test, "DBK", 3) == 0)
    return MID_DBK;
  if (strncmp(test, "CINT", 4) == 0)
    return MID_CONF;
  if (strncmp(test, "WCINT", 5) == 0)
    return MID_WCONF;
  if (strncmp(test, "THEIL", 5) == 0)
    return MID_THEIL;
  if (strncmp(test, "ERRCOMP", 7) == 0)
    return MID_ERRCOMP;
  if (strncmp(test, "STEADY", 6) == 0)
    return MID_STEADY;
  if (strncmp(test, "SHAPE", 5) == 0)
    return MID_SHAPE;
  if (strncmp(test, "INCREASE", 8) == 0)
    return MID_INCREASE;
  if (strncmp(test, "DECREASE", 8) == 0)
    return MID_DECREASE;
  if (strncmp(test, "TREND", 5) == 0)
    return MID_TREND;

  CHECKS(0, "Unknown test", test);
  return 0;
}

void DFit::error(const char* m)
{
  printf("%s\n", m);
  exit(0);
}

void DFit::read_config(const char* file)
{
  double vweight = 1.0;
  double dweights[24];
  for (int i = 0; i < 24; i++)
    dweights[i] = 1.0;
  
  TArray kw; TToken_string* tt;
  TArray cparms;
  TString od, of;

  tt = new TToken_string("WEIGHT");    tt->add(LEX_WEIGHT);    kw.add(tt);
  tt = new TToken_string("TEST");      tt->add(LEX_METHOD);    kw.add(tt);
  tt = new TToken_string("REFERENCE"); tt->add(LEX_REFERENCE); kw.add(tt);
  tt = new TToken_string("FILE");      tt->add(LEX_FILE);      kw.add(tt);
  tt = new TToken_string("DATA");      tt->add(LEX_DATA);      kw.add(tt);
  tt = new TToken_string("VARIABLE");  tt->add(LEX_VARIABLE);  kw.add(tt);
  tt = new TToken_string("INDEX");     tt->add(LEX_INDEX);     kw.add(tt);
  tt = new TToken_string("READ");      tt->add(LEX_READ);      kw.add(tt);
  tt = new TToken_string("OUTPUT");    tt->add(LEX_OUTPUT);    kw.add(tt);

  // Stupid bounds test
  tt = new TToken_string("BOUNDS");    tt->add(LEX_BOUNDS);    kw.add(tt);
  tt = new TToken_string("WBOUNDS");   tt->add(LEX_WBOUNDS);   kw.add(tt);
  // Stupid confidence interval test
  tt = new TToken_string("CINT");      tt->add(LEX_CINT);      kw.add(tt);
  tt = new TToken_string("WCINT");     tt->add(LEX_WCINT);     kw.add(tt);
  // Dent and Blackie deterministic timeseries F
  tt = new TToken_string("DBK");       tt->add(LEX_DBK);       kw.add(tt);
  // Theil inequality coefficient and error composition score
  tt = new TToken_string("THEIL");     tt->add(LEX_THEIL);     kw.add(tt);
  tt = new TToken_string("ERRCOMP");   tt->add(LEX_ERRCOMP);   kw.add(tt);
  // Steady state / increasing / decreasing / trend test
  tt = new TToken_string("STEADY");    tt->add(LEX_STEADY);    kw.add(tt);
  tt = new TToken_string("INCREASE");  tt->add(LEX_INCREASE);  kw.add(tt);
  tt = new TToken_string("DECREASE");  tt->add(LEX_DECREASE);  kw.add(tt);
  tt = new TToken_string("TREND");     tt->add(LEX_TREND);     kw.add(tt);

  // Smart frequency test
  tt = new TToken_string("FREQ");      tt->add(LEX_FREQ);      kw.add(tt);

  TLexDfit lex(file, " \t\n\"=", &kw);

  // read file
  TString cvar("");
  TToken_string datastring;

  VDesc* cvo = NULL;
  float weight = 0.0;
  FitTest* cvm = NULL;

  int cprm = 0;
  int tok = lex.token();

  while (tok != LEX_ENDFILE)
    {
      switch (tok) 
	{
	case LEX_METHOD:

	  cparms.destroy();

	  int cmet; cprm = 0;
	  
	  // create appropriate test object and append to vars tests
	  switch (tok = lex.token())
	    {
	    case LEX_BOUNDS:
	      cmet = MID_BOUNDS;
	      break;
	    case LEX_WBOUNDS:
	      cmet = MID_WBOUNDS;
	      break;
	    case LEX_CINT:
	      cmet = MID_CONF;
	      break;
	    case LEX_WCINT:
	      cmet = MID_WCONF;
	      break;
	    case LEX_DBK:
	      cmet = MID_DBK;
	      break;
	    case LEX_THEIL:
	      cmet = MID_THEIL;
	      break;
	    case LEX_ERRCOMP:
	      cmet = MID_ERRCOMP;
	      break;
	    case LEX_STEADY:
	      cmet = MID_STEADY;
	      break;
	    case LEX_INCREASE:
	      cmet = MID_INCREASE;
	      break;
	    case LEX_DECREASE:
	      cmet = MID_DECREASE;
	      break;
	    case LEX_FREQ:
	      cmet = MID_FREQ;
	      break;
	    default:
	      error("method not implemented");
	      break;
	    }
	  
	  weight = dweights[cmet];
	  
	  tok = lex.token();

	  while (tok == LEX_PARMDEF)
	    {
	      cparms.add(new TToken_string(lex.parm(), '='));
	      tok = lex.token();
	    }

	  if (tok == LEX_WEIGHT)
	    {
	      tok = lex.token();
	      if (tok == LEX_FLOAT || tok == LEX_INTEGER)
		weight = atof(lex.token_val());
	      else
		error("number expected after WEIGHT");
	      tok = lex.token();
	    }
	  
	  // if variable is being defined, create method with 
	  // parameters; otherwise, just store weight as default
	  if (cvar != "")
	    {
	      switch(cmet)
		{
		case MID_BOUNDS:
		  cvm = new FitTest_bounds(1, cparms);
		  break;
		case MID_WBOUNDS:
		  cvm = new FitTest_bounds(2, cparms);
		  break;
		case MID_CONF:
		  cvm = new FitTest_bounds(3, cparms);
		  break;
		case MID_WCONF:
		  cvm = new FitTest_bounds(4, cparms);
		  break;
		case MID_DBK:
		  cvm = new FitTest_dbk(cparms);
		  break;
		case MID_THEIL:
		  cvm = new FitTest_Theil(cparms);
		  break;
		case MID_ERRCOMP:
		  cvm = new FitTest_Errcomp(cparms);
		  break;
		case MID_STEADY:
		  cvm = new FitTest_steadystate(0, cparms); 
		  break;
		case MID_INCREASE:
		  cvm = new FitTest_steadystate(1, cparms);
		  break;
		case MID_DECREASE:
		  cvm = new FitTest_steadystate(-1, cparms);
		  break;
		case MID_TREND:
		  cvm = new FitTest_steadystate(2, cparms);
		  break;
		case MID_FREQ:
		  cvm = new FitTest_Freq(cparms);
		  break;
		}
	      cvm->weight() = weight;
	      cvo->teststats.add(cvm);
	    }
	  else dweights[cmet] = weight;
	  break;

	case LEX_OUTPUT:
	 
	  of = "";
	  lex.expect(LEX_STRING);
	  od = lex.sval();
	  if ((tok = lex.token()) == LEX_IDENTIFIER)
	    {
	      of = lex.token_val();
	      tok = lex.token();
	    }
	  
	  if (cvar == "")
	    _outdefs.add(new OutDef(od, of));
	  else
	      cvo->outdefs.add(new OutDef(od, of));
	  break;
	  

	case LEX_REFERENCE:

	  tok = lex.token();
	  if (tok == LEX_READ)
	    // read immediate: format is IDs for columns followed by data
	    {
	      int cnt = 0;
	      while ((tok = lex.token()) == LEX_IDENTIFIER) cnt++;
	      datastring = "";
	      while (tok == LEX_FLOAT || tok == LEX_INTEGER)
		{
		  datastring.add(lex.token_val());
		  tok = lex.token();
		}
	      double* ddata = new double[datastring.items()];
	      for (int k = 0; k < datastring.items(); k++)
		ddata[k] = atof(datastring.get(k));

	      cvo->nrefdata = datastring.items();
	      cvo->refdata = ddata;
	      cvo->rdatacols = cnt == 0 ? 1 : cnt;
	    }
	  else if (tok == LEX_FILE)
	    {
	      int nind = 0;
	      int indexes[24]; indexes[0] = 0; 
	      // read datafile in
	      tok = lex.token();

	      if (tok != LEX_IDENTIFIER)
		error("filename expected after FILE");

	      ifstream in(lex.token_val());
	      if (!in.is_open())
		error("can't open input file");

	      tok = lex.token();
	    
	      if (tok == LEX_INDEX)
		{
		  while ((tok = lex.token()) == LEX_INTEGER)
		    indexes[nind++] = lex.int_val();
		}

	      if (nind == 0) nind = 1;
	      
	      // read file
	      TToken_string t; int nfields = 0; datastring = "";
	      while (in.good() && !in.eof())
		{
		  in.getline(__buf_string, sizeof(__buf_string), '\n');
		  t.tokenize(__buf_string);
		  if (t.items() == 0) 
		    continue;
		  if (nfields == 0) nfields = t.items();
		  else {
		    if (t.items() != nfields)
		      error("number of columns in file is not consistent");
		  }
		  
		  for (int i = 0; i < nind; i++)
		    datastring.add(t.get(indexes[i]));
		}
	      
	      double* ddata = new double[datastring.items()];
	      for (int k = 0; k < datastring.items(); k++)
		ddata[k] = atof(datastring.get(k));

	      cvo->nrefdata = datastring.items();
	      cvo->refdata = ddata;
	      cvo->rdatacols = nind;	      
	    }
	  break;
	case LEX_DATA:

	  tok = lex.token();
	  if (tok == LEX_READ)
	    // read immediate: format is IDs for columns followed by data
	    {
	      int cnt = 0;
	      while ((tok = lex.token()) == LEX_IDENTIFIER) cnt++;
	      datastring = "";
	      while (tok == LEX_FLOAT || tok == LEX_INTEGER)
		{
		  datastring.add(lex.token_val());
		  tok = lex.token();
		}
	      double* rdata = new double[datastring.items()];
	      for (int k = 0; k < datastring.items(); k++)
		rdata[k] = atof(datastring.get(k));

	      cvo->ndata = datastring.items();
	      cvo->data = rdata;
	      cvo->datacols = cnt == 0 ? 1 : cnt;
	    }
	  else if (tok == LEX_FILE)
	    {
	      int nind = 0;
	      int indexes[24]; indexes[0] = 0; 
	      // read datafile in
	      tok = lex.token();

	      if (tok != LEX_IDENTIFIER)
		error("filename expected after FILE");

	      ifstream in(lex.token_val());
	      if (!in.is_open())
		error("can't open input file");

	      tok = lex.token();
	    
	      if (tok == LEX_INDEX)
		{
		  while ((tok = lex.token()) == LEX_INTEGER)
		    indexes[nind++] = lex.int_val();
		}

	      if (nind == 0) nind = 1;

	      // read file
	      TToken_string t; int nfields = 0; datastring = "";
	      while (in.good() && !in.eof())
		{
		  in.getline(__buf_string, sizeof(__buf_string)-1, '\n');
		  t.tokenize(__buf_string);
		  if (t.items() == 0) 
		    continue;
		  if (nfields == 0) nfields = t.items();
		  else {
		    if (t.items() != nfields)
		      error("number of columns in file is not consistent");
		  }
		  
		  for (int i = 0; i < nind; i++)
		    datastring.add(t.get(indexes[i]));
		}
	      
	      double* ddata = new double[datastring.items()];
	      for (int k = 0; k < datastring.items(); k++)
		ddata[k] = atof(datastring.get(k));

	      cvo->ndata = datastring.items();
	      cvo->data = ddata;
	      cvo->datacols = nind;
	    }
	  else if (tok == LEX_INDEX)
	    {
	      // read datafile as indexed column in file (TBI)
	      while ((tok = lex.token()) == LEX_INTEGER)
		;
	    }
	  break;
	case LEX_WEIGHT:
	  // default weight for all variables
	  tok = lex.token();
	  if (tok == LEX_FLOAT || tok == LEX_INTEGER)
	    vweight = atof(lex.token_val());
	  else
	    error("number expected after WEIGHT");
	  tok = lex.token();
	  break;
	case LEX_VARIABLE:
	  
	  lex.expect(LEX_IDENTIFIER);
	  cvar = lex.token_val();
	  _varnames.add(new TString(cvar));
	  _vars.add(cvar, cvo = new VDesc(cvar), TRUE);
	  cvo->weight = vweight;
	  
	  if ((tok = lex.token()) == LEX_WEIGHT)
	    {
	      tok = lex.token();
	      if (tok == LEX_FLOAT || tok == LEX_INTEGER)
		cvo->weight = atof(lex.token_val());
	      else
		error("number expected after WEIGHT");
	      tok = lex.token();
	    }
	  
	  break;
	default:
	  error(format("syntax error at line %d", lex.line()+1));
	  break;
	}
    }
  
  // loop over variables and initialize all tests
  _vars.restart();
  for (int i = 0; i < _vars.items(); i++)
    {
      VDesc* v = (VDesc*)_vars.get();

      for (int j = 0; j < v->teststats.items(); j++)
	{
	  FitTest& t =  (FitTest&)v->teststats[j];
	  t.initialize(v->refdata, v->nrefdata);
	}
    }
}

void DFit::compute()
{
  // compute required tests for all variables
  double tvscore = 0.0, pvscore = 0.0;
  
  _vars.restart();

  for (int i = 0; i < _vars.items(); i++)
    {
      VDesc* v = (VDesc*)_vars.get();

      double tscore = 0.0, pscore = 0.0;

      // calculate variable score
      for (int j = 0; j < v->teststats.items(); j++)
	{
	  FitTest& t =  (FitTest&)v->teststats[j];
	  tscore += t.weight();
	  t.compute(v->data, v->ndata);
	  pscore += t.weight()*t.score();
	}
      
      v->score = tscore == 0.0 ? 0.0 : pscore/tscore;
      
      tvscore += v->weight;
      pvscore += v->weight*v->score;
    }
  
  // calculate total score
  _score = tvscore == 0.0 ? 0.0 : pvscore/tvscore; 
  _ok = TRUE;
}

void DFit::report()
{
  CHECK(_ok, "computations have not been done");

  for (int i = 0; i < _outdefs.items(); i++)
    {
      OutDef& od = (OutDef&)_outdefs[i];
      report_object(od.form, od.file);
    }

  for (int i = 0; i < _varnames.items(); i++)
    {
      TString& varn = (TString&)_varnames[i];
      VDesc& v = (VDesc&)_vars[varn];

      for(int j = 0; j < v.outdefs.items(); j++)
	{
	  OutDef& od = (OutDef&)(v.outdefs[j]);
	  report_object(od.form, od.file, &v);
	}
    }
}

void DFit::report_object(const char* form, const char* f, VDesc* v)
{

  // if file name ends with +, append to the file, otherwise overwrite
  char* om = "w";
  TFilename fn(f);

  if (!fn.empty() && fn[fn.len()-1] == '+')
    {
      om = "a";
      fn.cut(fn.len() - 1);
    }

  FILE* fp = (fn.empty() ? stdout : fopen((const char*)fn, om));

  CHECKS(fp != NULL, "Can't open file:", f);
  
  char c = *form++;  char id[256];

  while (c)
    {
      if (c == '$')
	{
	  int cnt = 0; 
	  c = *form++;
	  while (isalnum(c) || c == '(' || 
		 c == ')' || c == '_')
	    {
	      id[cnt++] = c;
	      c = *form++;
	    }
	  id[cnt] = '\0';

	  // process ID: check keywords, otherwise locate test and
	  // retrieve variable value
	  if (strncmp(id, "DATE", 4) == 0)
	    {
	      time_t t = time(NULL);
	      // one day I'll understand why they put a newline 
	      // at the end of the date string
	      char bt[32];
	      strcpy(bt, ctime(&t));
	      bt[strlen(bt)-1] = '\0';
	      fprintf(fp, bt);
	    }
	  else if (strncmp(id, "ID", 2) == 0)
	    {
	      fprintf(fp, "%s", v == NULL ? (const char*)_id : 
		      (const char*)v->name);
	    }
	  else if (strncmp(id, "SCORE", 5) == 0)
	    {
	      fprintf(fp, "%g", v == NULL ? _score : v->score);
	    }
	  else if (v != NULL)
	    {
	      TString test(id), parm(""); int pp;
	      if ((pp = test.find('(')) != -1)
		{
		  CHECKS (test[test.len() - 1] != ')',
			  "syntax error: missing close parenthesis", id);
		  parm = test.sub(pp+1, test.len()-1);
		  test.cut(pp);
		}

	      // look for test
	      int t, tid = test_id(test);
	      for(t = 0; t < v->teststats.items(); t++)
		{
		  FitTest& ft = (FitTest&)(v->teststats[t]);
		  if (ft.test_id() == tid)
		    {
		      if (parm.empty())
			fprintf(fp, "%g", ft.score());
		      else 
			fprintf(fp, "%s", ft.get_variable(parm));
		      break;
		    }
		}
	      if (t == v->teststats.items())
		fprintf(fp, "(nil)");
	    }
	  else fprintf(fp, "(nil)");
	}
      else 
	{
	  fprintf(fp, "%c", c);
	  c = *form++;
	}
    }
  
  if (*f != '\0') fclose(fp); 
}


void DFit::set_data(const char* var, double* data, int ndata)
{
  CHECKS(_vars.is_key(var), "Variable does not exist", var);
  VDesc& v = (VDesc&)_vars[var]; 
  v.data  = data;
  v.ndata = ndata;
}

double DFit::var_score(const char* varname, int method_id = MID_GLOBAL)
{
  CHECKS(_vars.is_key(varname), "Variable does not exist", var);
  VDesc& v = (VDesc&)_vars[varname]; 

  if (method_id == MID_GLOBAL)
    return v.score;
  
  for (int i = 0; i < v.teststats.items(); i++)
    {
      FitTest& t =  (FitTest&)v.teststats[i];
      if (t.test_id() == method_id)
	return t.score();
    }
  
  error("method id not found for variable");
  return 0.0;
}


#ifdef USE_TCL

#define VAR_SCORE

int tcl_dfit::write(const char* filename)
{
  return 1;
}

int tcl_dfit::read (const char* filename)
{
  return 1;
}

void tcl_dfit::do_method(methodID ID, int argc, char** argv,
				int optc, char** optv)
{
  TFeatured_object::do_method(ID, argc, argv, optc, optv);
}
  

tcl_dfit::tcl_dfit(int ac, char** av, int oc, char** ov, char* name) :
  TFeatured_object(ac, av, oc, ov, name)
{
}

tcl_dfit::~tcl_dfit()
{
}

#endif
