#include <strings.H>
extern "C" {
// CEPHES library of probability distributions
#include <cprob.h>
}
#include <dfitm.H>
#include <math.h>
#include <utility.H>
#include <malloc.h>

#define PI  3.141592653589793
#define PI2 6.28318530717959
#define ABSI(x) ((x<0)?(-1*(x)):(x))

// stats functions
double dmean(double* a, int n);
double dmin(double* a, int n);
double dmax(double* a, int n);
double dest_variance(double* a, int n);
double dest_sd(double* a, int n);

// fft stuff
void real_ft(double* x, int n, int inv);
void fft(double* x, int n, int inv);
void periodogram(double* x, int n, double* p);

int  peakdetect (double* s, int n, int maxpeaks, 
		 int* peakx, double* peaky, 
		 double nosig_threshold  = 0.1,
		 double nosig_peakheight = 0.5,
		 double nosig_sharpness  = 0.1);


// implementasiun (no, it's not English)
double dmean (double *array, int size)
{
  double sum;
  int i;
  
  sum = 0;
  for (i=0; i<size; i++)
    sum += array[i];
  
  return (sum / size);
} 

double dest_variance (double *array, int size)
{
  double sum=0;
  double mean = dmean(array, size);

  for (int i=0; i<size; i++)
    sum += ((array[i] - mean) * (array[i] - mean));

  return sum / (size -1);
}

double dmax (double *array, int size)
{
  double max = array[0];
  
  for (int i=0; i < size; i++)
    if (array[i] > max) max = array[i];

  return max;
}

double dmin (double *array, int size)
{
  double min = array[0];
  
  for (int i=0; i < size; i++)
    if (array[i] <min) min = array[i];

  return min;
}

double dest_sd (double *array, int size)
{
  return sqrt(dest_variance(array, size));
}

void periodogram(double* x, int n, double* p)
{
  int i, j, m ;
  double *w, den ;
  
  m = n/2 ;

  w = (double *) malloc((unsigned) (n+1)*sizeof(double));
  memcpy(w, x, n*sizeof(double)) ;

  real_ft(w,n,1);
  
  p[0] = 0.0 ;
  p[m] = w[1]*w[1] ;
  den = p[m];
  for (i=1;i<m;i++) {
    j = 2*i;
    p[i] = w[j]*w[j]+w[j+1]*w[j+1] ;
    den += p[i] ;
  }
  den = (double)(2.0*n*PI) ;
  
  for (j=0;j<=m;j++)
    p[j] /= den;  
}

void fft(double* x, int nc, int inv)
{
  int i, j, k, n ;
  int i1, j1, m, minc, chk ;
  double tmp, c1=0.0, s1=0.0, c, s, theta ;
  double xr, xi ;

/* bit reversal */

  n = nc/2 ;
  chk = 0 ;
  j = n ;
  for (i=1;i<(nc-1);i++) {
    chk = 1-chk ;
    if (j > i) {
      j1 = 2*j ;
      i1 = 2*i ;
      tmp = x[j1] ;       /* swap real components */
      x[j1] = x[i1] ;
      x[i1] = tmp ;
      tmp = x[j1+1] ;     /* swap imag components */
      x[j1+1] = x[i1+1] ;
      x[i1+1] = tmp ;
    }
    m = n ;
    if(chk) {
      j -= m ;
      m /= 2 ;
      while(j >= m && m > 1) {
	j -= m ;
	m /= 2 ;
      }
    }
    j += m ;
  }
            /* transform calculation: initialized as d(j) = x(j) */
  n = 2*nc ;
  m = 2 ;
  while(n > m) {
    minc = 2*m ;
    c = 1.0 ;
    s = 0.0 ;
    for(k=0;k<m;k+=2) {
      for(i=k;i<n;i+=minc) {
	j=i+m;
	i1 = i+1 ;
	j1 = j+1 ;
	xr = c*x[j] - s*x[j1] ;
	xi = c*x[j1] + s*x[j] ;
	x[j] = x[i] - xr ;
	x[j1] = x[i1] - xi ;
	x[i] += xr;
	x[i1] += xi;
      }
      tmp = c ;
      c += (c*c1 - s*s1) ;
      s += (s*c1 + tmp*s1) ;
    }
    m=minc ;
    if(n > m) {
      theta = -PI2/m ;
      theta *= inv ;         /* inverse transform if inv = -1 */
      tmp = sin(theta/2.0) ;
      c1 = -2.0*tmp*tmp ;
      s1 = sin(theta) ;
    }
  }
}

void real_ft(double* x, int n, int inv)
{
  int i, k, m=n, mk ;
  double c1, s1, c, s, tmp, theta ;
  double ar, ai, br, bi, dinv ;
  
  m /= 2 ;
  theta = -PI / (double)m ;
  dinv = -0.5 ;
  
  if (inv != -1) {
    fft(x,m,1) ;
  } 
  else {
    dinv = 0.5;
    theta = -theta ;
  }
  
  tmp = sin(theta/2.0);
  c1 = -2.0*tmp*tmp ;
  s1= sin(theta) ;
  c = 1.0 + c1 ;
  s = s1 ;
  for (i=1;i<=m/2;i++) {
    k = 2*i ;
    mk = 2*m-k ; 
    
    ar = (x[k] + x[mk])/2.0 ;
    ai = (x[k+1] - x[mk+1])/2.0 ;
    
    br = -dinv*(x[k+1] + x[mk+1]) ;
    bi = dinv*(x[k] - x[mk]) ;
    
    x[k] = ar + c*br - s*bi ;
    x[k+1] = ai + c*bi + s*br ;
    
    x[mk] = ar - c*br + s*bi;
    x[mk+1] = -ai + c*bi + s*br;
    
    tmp=c ;
    c += (c*c1-s*s1) ;
    s += (s*c1 + tmp*s1) ;
  }
  
  tmp = x[0] ;
  x[0] += x[1] ;
  if (inv != -1) {
    x[1] = tmp - x[1] ;
  } 
  else {
    x[0] /= 2.0 ;
    x[1] = (tmp-x[1])/2.0 ;
    fft(x,m,-1);
  }
}

int peakdetect (double* s, int n, int maxpeaks,
		int* xpeak, double* ypeak, 
		double nosig_threshold,
		double nosig_peakheight,
		double nosig_sharpness)
{
  // records i <= maxpeaks peaks in data; puts x coord in xpeak[i]
  // and value in ypeak[i]. Number of all peaks is returned (even
  // if greater than maxpeaks). Peaks must start and end at y <=
  // nosig_threshold, be high at least nosig_peakheight, and 
  // be wide no more than ndata* nosig_sharpness.

  // It's for periodograms, so the first point is assumed to be 0. Data
  // are expected to range 0 to 1 for default values to be meaningful.

  int plen = 0, curp = 0, np = 0; double curh = 0.0;
  for (int i = 1; i < n; i++)
    {
      if (s[i] >= nosig_threshold)
	{
	  plen++;
	  if (s[i] > curh) 
	    {
	      curh = s[i]; 
	      curp = i;
	    }
	}
      else 
	{
	  if (curh > nosig_peakheight) 
	    {
	      if ((double)plen <= ((double)n*nosig_sharpness))
		{
		  if (np < maxpeaks)
		    {
		      xpeak[np] = curp;
		      ypeak[np] = curh;
		    }
		  np++;
		}
	    }
	  curh = 0.0;
	  curp = 0;
	  plen = 0;
	}
    }
  return np;
}


// methods

void FitTest::define_variable(const char* v, const char* val)
{
  _out_vars.add(v, new TString(val), TRUE);
}

const char* FitTest::get_variable(const char* v)
{
  if (!_out_vars.is_key(v))
    return "(nil)";
  return (const char*)(TString&)_out_vars[v];
}

void FitTest::define_parameters(TArray& pa) 
{ 
  for (int i = 0; i < pa.items(); i++)
    {
      TToken_string& t = (TToken_string&)pa[i];
      TString* tt = new TString(t.get(1));
      _parms.add(t.get(0), tt); 
    }
}


int FitTest_bounds::test_id() 
{
  switch(_method) {
    case 1: return MID_BOUNDS;  break;
    case 2: return MID_WBOUNDS; break;
    case 3: return MID_CONF;    break;
    case 4: return MID_WCONF;   break;
  }
  return 0;
}

void FitTest_bounds::initialize(double* data, int ndata)
{
  define_variable("WEIGHT", format("%g", weight()));
  
  // compute mean, sd, min, max of data;
  if (_method < 3)
    {
      _min = dmin(data, ndata);
      _max = dmax(data, ndata);
    }
  else
    {
      
      double level = 0.95;
      if (is_parameter("level"))
	level = dparm("level");

      double t = stdtri(ndata-1, level);
      
      _std  = dest_sd(data, ndata);
      _mean = dmean(data, ndata);
      _min  = _mean - t*_std;
      _max  = _mean + t*_std;

      define_variable("MIN",  format("%g", _min));
      define_variable("MAX",  format("%g", _max));
      define_variable("MEAN", format("%g", _mean));
      define_variable("SD",   format("%g", _std));
      define_variable("T",    format("%g", t));
    }
}

void FitTest_bounds::compute(double* data, int ndata)
{
  double contr, csum = 0.0;

  double k = 1.0;
  if (is_parameter("k"))
    k = dparm("k");

  for (int i = 0; i < ndata; i++)
    {
      contr = 1.0;
      if (data[i] < _min || data[i] > _max)
	{
	  contr = 0.0;
	  if (_method == 2 || _method == 4)
	    {
	      // compute weight according to distance from nearest bound
	      contr = (data[i] < _min) ? _min - data[i] : data[i] - _max;
	      contr /= k * (_max - _min);
	      if (contr < 0.0) contr = -contr;
	      contr = 1.0 - contr;
	    }
	}
      csum += contr;
    }
  
  define_variable("MAX", format("%g", _max));
  define_variable("MIN", format("%g", _min));
  define_variable("ACTUAL", format("%g", csum/(double)ndata));

  score(csum < 0.0 ? 0.0 : csum / (double)ndata);
  ok() = TRUE;
}

void FitTest_bounds::print_on(ostream& o) const
{
  // o << "\tmin = " << _min << " max = " << _max;
  switch (_method)
    {
    case 1:
      o << " O=" << score();
      break;
    case 2:
      o << " OW=" << score();
      break;
    case 3:
      o << " OS= " << score();
      break;
    case 4:
      o << " OSW=" << score();
      break;
    }
}

void FitTest_dbk::initialize(double* data, int ndata)
{
  _refdata  = data;
  _nrefdata = ndata;
}

void FitTest_dbk::compute(double* data, int ndata)
{
  // only use the minimum number of data; assume they're correctly
  // matching each other
  int n = ndata;
  
  if (ndata != _nrefdata)
    {
      cerr << "warning: ndata and nrefdata differ\n";
      n = ndata > _nrefdata? _nrefdata : ndata;
    }

  double* r = _refdata, *d = data;

  // normalize if required: this becomes a "shape" test
  if (is_parameter("normalize") && iparm("normalize") > 0)
    {
      r = new double[n];
      d = new double[n];
      
      double rmn = dmin(_refdata, _nrefdata);
      double rmx = dmax(_refdata, _nrefdata);
      double dmn = dmin(data, ndata);
      double dmx = dmax(data, ndata);
      
      for (int i = 0; i < n; i++)
	{
	  d[i] = (data[i] - dmn)/(dmx - dmn); 
	  r[i] = (_refdata[i] - rmn)/(rmx - rmn); 
	}
    }
  
  double sx = 0.0, sy = 0.0, sx2 = 0.0,  sy2 = 0.0, 
    sd = 0.0, sxy = 0.0;
  
  for (int i = 0; i < n; i++)
    {
      sx  += d[i];
      sy  += r[i];
      sx2 += d[i]*d[i];
      sy2 += r[i]*r[i];
      sxy += d[i]*r[i];
      sd  += (d[i]-r[i])*(d[i]-r[i]);
    }
  
  _mx  = dmean(d, n);
  _my  = dmean(r, n);

  double dn = (double)n;

  // regress refdata vs testdata 
  double b = (sxy - ((sx * sy)/dn))/(sx2 - ((sx*sx)/dn));
  double a = _my - b*_mx;

  // compute residual mean squared error
  double s = 0;
  for (int i = 0; i < n; i++)
    {
      double yhat = _my + (b * (d[i] - _mx));
      s += (r[i] - yhat) * (r[i] - yhat);
    }
  s /= (dn - 2.0);

  // compute R-squared
  double sdyh2 = 0.0, sdy2 = 0.0;
  for (int i = 0; i < n; i++)
    {
      double yh = a + b*d[i];
      sdyh2 = (yh - _my)*(yh - _my);
      sdy2 = (r[i] - _my)*(r[i] - _my);
    }
  double r2 = sdyh2/sdy2;


  // compute dbk F statistic
  double db = b - 1.0;
  double f = ((dn - 2.0) * ((dn*a*a) + ((dn+dn)*_mx*a*db) + (sx2*db*db))) / 
    ((dn+dn)*s*s);

  // publish variables
  define_variable("SLOPE",     format("%g", b));
  define_variable("INTERCEPT", format("%g", a));
  define_variable("RMSE",      format("%g", s));
  define_variable("F",         format("%g", f));
  define_variable("RSQUARED",  format("%g", r2));

  if (sd == 0.0) // data are identical
    {
      // fisher would give a NaN in this case
      score(1.0);
      return;
    }

  // compute probability
  double p = fdtrc(2, n-2, f);

  // score is probability of slope = 1.0 and intercept = 0.0
  score(p < 0.001 ? 0.0 : p);

  if (is_parameter("weight_r2") && iparm("weight_r2") == 1)
    weight() = weight() * r2;

  if (d != data)
    {
      delete d;
      delete r;
    }

  define_variable("WEIGHT",  format("%g", weight()));
}

void FitTest_dbk::print_on(ostream& o) const
{
  o << " DBK=" << score();
}

void FitTest_Theil::initialize(double* data, int ndata)
{
  define_variable("WEIGHT",  format("%g",weight()));

  _refdata  = data;
  _nrefdata = ndata;
}

void FitTest_Theil::compute(double* data, int ndata)
{
  // only use the minimum number of data; assume they're correctly
  // matching each other
  int n = ndata;
  
  if (ndata != _nrefdata)
    {
      cerr << "warning: ndata and nrefdata differ\n";
      n = ndata > _nrefdata? _nrefdata : ndata;
    }

  double* r = _refdata, *d = data;

  // normalize if required: this becomes a "shape" test
  if (is_parameter("normalize") && iparm("normalize") > 0)
    {
      r = new double[n];
      d = new double[n];

      double rmn = dmin(_refdata, _nrefdata);
      double rmx = dmax(_refdata, _nrefdata);
      double dmn = dmin(data, ndata);
      double dmx = dmax(data, ndata);

      for (int i = 0; i < n; i++)
	{
	  d[i] = (data[i] - dmn)/(dmx - dmn); 
	  r[i] = (_refdata[i] - rmn)/(rmx - rmn); 
	}
    }

  double sx = 0, sy = 0, sx2 = 0, sy2 = 0, sd =0, sxy = 0;
 
  for (int i = 0; i < n; i++)
    {
      sx  += d[i];
      sy  += r[i];
      sx2 += d[i]*d[i];
      sy2 += r[i]*r[i];
      sxy += d[i]*r[i];
      sd  += (d[i]-r[i])*(d[i]-r[i]);
    }
  
  _uval =  sqrt(sd/(double)n)/(sqrt(sx2/(double)n)+sqrt(sy2/(double)n));

  define_variable("U", format("%g", _uval));

  if (d != data)
    {
      delete d;
      delete r;
    }

  score(1.0 - _uval);
}

void FitTest_Theil::print_on(ostream& o) const
{
  o << " THL=" << score();
}

void FitTest_Errcomp::initialize(double* data, int ndata)
{
  
  define_variable("WEIGHT", format("%g", weight()));
  _refdata  = data;
  _nrefdata = ndata;
}

void FitTest_Errcomp::compute(double* data, int ndata)
{
  // only use the minimum number of data; assume they're correctly
  // matching each other
  int n = ndata;
  
  if (ndata != _nrefdata)
    {
      cerr << "warning: ndata and nrefdata differ\n";
      n = ndata > _nrefdata? _nrefdata : ndata;
    }

  if (is_parameter("bias")) 
    _wmean = dparm("bias");

  if (is_parameter("variance")) 
    _wvar = dparm("variance");

  if (is_parameter("random"))
    _wrand = dparm("random");

  // normalize weights and check defaults
  double wt = _wmean + _wvar + _wrand;
  if (wt == 0) _wrand = 1.0;
  else if (wt != 1.0) 
    {
      _wmean /= wt;
      _wvar /= wt;
      _wrand /= wt;
    }

  double sx = 0, sy = 0, sx2 = 0, sy2 = 0, sd =0, sxy = 0;
  
  for (int i = 0; i < n; i++)
    {
      sx  += data[i];
      sy  += _refdata[i];
      sx2 += data[i]*data[i];
      sy2 += _refdata[i]*_refdata[i];
      sxy += data[i]*_refdata[i];
      sd  += (data[i]-_refdata[i])*(data[i]-_refdata[i]);
    }
  
  _mx  = dmean(data, n);
  _my  = dmean(_refdata, n);
  _sdx = dest_sd(data, n);
  _sdy = dest_sd(_refdata, n);

  _corr = sxy / sqrt(sx2*sy2);
  
  // compute components of error
  _meanp = ((_my - _mx)*(_my - _mx));
  _varp  = (_sdx - (_corr * _sdy)); _varp *= _varp;
  _randp = ((1.0 - (_corr*_corr))*(_sdx*_sdx));
  
  double ds = _meanp + _varp + _randp;
  _meanp /= ds; 
  _varp  /= ds;
  _randp /= ds;

  define_variable("PEARSON",  format("%g", _corr));
  define_variable("BIAS",     format("%g", _meanp));
  define_variable("VARIANCE", format("%g", _varp));
  define_variable("RANDOM",   format("%g", _randp));

  // compute score for error components if desired
  // _sm will be higher when error partitioning is less critical
  _sm = 0.0;

  if (_wmean < _meanp) _sm += _meanp - _wmean;
  if (_wvar  < _varp)  _sm += _varp - _wvar;
  if (_wrand < _randp) _sm += _randp - _wrand;
  
  score(1.0 - _sm);
}

void FitTest_Errcomp::print_on(ostream& o) const
{
  o << " ERC=" << score();
}

void FitTest_steadystate::initialize(double* data, int ndata)
{
}

int FitTest_steadystate::test_id() 
{
  switch(_steady) {
    case 0:  return MID_STEADY;   break;
    case -1: return MID_DECREASE; break;
    case 1:  return MID_INCREASE; break;
    case 2:  return MID_TREND;    break;
  }
  return 0;
}

void FitTest_steadystate::compute(double* data, int ndata)
{    
  if (is_parameter("start"))
    _t1 = iparm("start");

  if (is_parameter("end"))
    _t2 = iparm("end");

  if ((_t1 + _t2) == 0) 
    _t2 = ndata - 1;

  int n = _t2 - _t1 + 1;

  double d[n+2];
  double r[n+2];
  int c = 0;

  for (int i = _t1; i <= _t2; i++, c++)
    {
      d[c] = data[i];
      r[c] = (double)c;
    }
  
  double sx = 0, sy = 0, sx2 = 0, sy2 = 0, sxy = 0, 
    sxyd = 0.0, sx2d = 0.0, sy2d = 0.0;
  
  double mx  = dmean(r, c);
  double my  = dmean(d, c);

  for (int i = 0; i < c; i++)
    {
      sx   += r[i];
      sy   += d[i];
      sx2  += r[i]*r[i];
      sy2  += d[i]*d[i];
      sxy  += r[i]*d[i];
      sxyd += (r[i] - mx) * (d[i] - my);
      sx2d += (r[i] - mx) * (r[i] - mx);
      sy2d += (d[i] - my) * (d[i] - my);
    }
  

  // regress refdata vs testdata 
  double b = sxyd/sx2d;
  double a = my - b*mx;

  double scor = 0.0;

  // compute standard error of b
  double sy2exp = (sxyd*sxyd)/sx2d;
  double sb = sqrt(((sy2d - sy2exp)/((double)c - 2))/sx2d);
  double t =  stdtri(c - 2, 0.95);

  // compute R-squared
  double sdyh2 = 0.0, sdy2 = 0.0;
  for (int i = 0; i < c; i++)
    {
      double yh = a + b*r[i];
      sdyh2 = (yh - my)*(yh - my);
      sdy2 = (d[i] - my)*(d[i] - my);
    }
  double r2 = sdyh2/sdy2;

  // confidence interval of b
  _dmin = b - sb*t;
  _dmax = b + sb*t;

  define_variable("SLOPE",          format("%g", b));
  define_variable("INTERCEPT",      format("%g", a));
  define_variable("SLOPE_STDERR",   format("%g", sb));
  define_variable("SLOPE_CMIN",     format("%g", _dmin));
  define_variable("SLOPE_CMAX",     format("%g", _dmax));
  define_variable("RSQUARED",       format("%g", r2));

  double trend = 0.0;
  
  switch (_steady)
    {
    case 0: // check for steady state
      if ((_dmin < 0.0 && _dmax > 0.0) || (_dmin > 0.0 && _dmax < 0.0))
	scor = 1.0;
      break;
    case 1: // check for increasing
      if (_dmin > 0.0 && _dmax > 0.0) 
	scor = 1.0;
      break;
    case -1: // check for decreasing
      if (_dmin < 0.0 && _dmax < 0.0) 
	scor = 1.0;
      break;
    case 2: // check for specific trend
      if (is_parameter("slope"))
	trend = dparm("slope");
      if (_dmin >= trend && _dmax <= trend) 
	scor = 1.0;
      break;
    }

  if (is_parameter("weight_r2") && iparm("weight_r2") == 1)
    weight() = weight() * r2;

  define_variable("WEIGHT",  format("%g", weight()));

  score(scor);
}

void FitTest_steadystate::print_on(ostream& out) const 
{
  out << " SST(" << _steady << ")=" << score();
}


void FitTest_Freq::initialize(double* refdata, int nrdata) 
{
  int difference = is_parameter("difference") ? iparm("difference") : 0;

  // if periods are not given, analyze ref signal for freq components
  if (!is_parameter("period"))
    {
      double* rd = refdata;

      double mdata  = dmean(rd, nrdata);
      double vdata  = dest_variance(rd, nrdata);
      
      if (difference) nrdata --;
      int nsc=1; while(nsc < nrdata) nsc *= 2;
      
      double* rdata = new double[nsc + 1];
      memset(rdata, 0, (nsc+1)*sizeof(double));
      
      for (int i = 0; i < nrdata; i++)
	{
	  if (difference)
	    rdata[i] = rd[i+1] - rd[i];
	  else
	    rdata[i] = rd[i] - mdata;
	}
      
      // rdata now contains reference signal: compute periodogram
      _rperiod  = new double[nsc/2 + 2];
      _nrperiod = nsc/2 + 1;
      periodogram(rdata, nsc, _rperiod);
      
      delete rdata;
      
      for (int i = 0; i < _nrperiod; i++)
	_rperiod[i] /= vdata;
      
      // normalize periodogram
      double min, max; min = max = _rperiod[0];
      for (int i = 0; i < _nrperiod; i++)
	{
	  if (min > _rperiod[i]) min = _rperiod[i];
	  if (max < _rperiod[i]) max = _rperiod[i];
	}
      
      TString t;
      for (int i = 0; i < _nrperiod; i++)
	{
	  _rperiod[i] = (_rperiod[i] - min)/(max - min);
	  t << format("%g\n", _rperiod[i]);
	}

      define_variable("RDATA_PERIODOGRAM", t);

    }
}

void FitTest_Freq::compute(double* data, int ndata)
{

  // relevant parameters
  int difference = is_parameter("difference") ? iparm("difference") : 0;
  double ntr = is_parameter("peak_threshold") ? dparm("peak_threshold") : 0.1;
  double pkh = is_parameter("peak_height") ? dparm("peak_height") : 0.5;
  double pkw = is_parameter("peak_width") ? dparm("peak_width") : 0.1;
  int mpk    = is_parameter("max_peaks") ? iparm("max_peaks") : 4;

  double mdata  = dmean(data, ndata);
  double vdata  = dest_variance(data, ndata);
  
  if (difference) ndata --;

  int nsc=1 ;  while(nsc < ndata)  nsc *= 2 ;
  double* rdata = new double[nsc + 1];
  memset(rdata, 0, (nsc+1)*sizeof(double));
  
  for (int i = 0; i < ndata; i++)
    {
      if (difference)
	rdata[i] = data[i+1] - data[i];
      else
	rdata[i] = data[i] - mdata;
    }

  _dperiod  = new double[nsc/2 + 2];
  _ndperiod = nsc/2 + 1;

  periodogram(rdata, nsc, _dperiod);

  delete rdata;

  for (int i = 0; i < _ndperiod; i++)
      _dperiod[i] /= vdata;

  // normalize periodogram
  double min, max; min = max = _dperiod[0];
  for (int i = 0; i < _ndperiod; i++)
   {
     if (min > _dperiod[i]) min = _dperiod[i];
     if (max < _dperiod[i]) max = _dperiod[i];
   }
  TString t;
  for (int i = 0; i < _ndperiod; i++)
    {
      _dperiod[i] = (_dperiod[i] - min)/(max - min);
      t << format("%g\n", _dperiod[i]);
    }

  define_variable("DATA_PERIODOGRAM", t);
  // find peaks in both periodograms and 
  // compare them
  
  double scor = 0.0;
  
  double rpk[12], dpk[12];
  int    rpx[12], dpx[12];
  int    rnp, dnp;

  if (is_parameter("acceptable_drift"))
    _acceptable_drift = iparm("acceptable_drift");

  if (is_parameter("period"))
    {
      TToken_string sp(sparm("period"), '_');
      rnp = sp.items() > mpk ? mpk : sp.items();
      for (int i = 0; i < rnp; i++)
	{
	  rpx[i] = sp.get_int(i);
	  rpk[i] = 1.0;
	}
    } 
  else rnp = peakdetect(_rperiod, _nrperiod, mpk, rpx, rpk, ntr, pkh, pkw);

  dnp = peakdetect(_dperiod, _ndperiod, mpk, dpx, dpk, ntr, pkh, pkw);

  t = "";
  for (int i = 0; i < rnp; i++)
    t << format("%d %g\n", rpx[i], rpk[i]);
  define_variable("RDATA_PERIODS", t);

  t = "";
  for (int i = 0; i < dnp; i++)
    t << format("%d %g\n", dpx[i], dpk[i]);
  define_variable("DATA_PERIODS", t);
  
  // if no frequencies are present the test weight drops to 0
  if (rnp == 0 && dnp == 0)
    weight() = 0.0;
  else 
    {

      // score = sum((1 - distance)*strength)/((sum(strength)) 
      // where: distance =
      // (x(ref peak) - x(nearest peak in data))/(acceptable freq drift + 1);
      // if there is no peak within acceptable frequency drift
      // (default 1 time unit), distance is 1
      
      double n_scurzot = 0;
      double d_scurzot = 0;
      
      for (int i = 0; i < rnp; i++)
	{
	  // find nearest peak in data lying within accepted drift
	  int nrp = -1; int nrdist = _ndperiod;
	  for (int j = 0; j < dnp; j++)
	    {
	      int dist = ABSI(rpx[i] - dpx[j]);
	      if (dist <= _acceptable_drift && dist < nrdist)
	      {
		nrp = dpx[i];
		nrdist = dist;
	      }
	    }
	  if (nrp != -1 && rpx[i] <= (ndata/2))
	    {
	      n_scurzot += rpk[i] * 
		(1 - ((double)nrdist / (double)(_acceptable_drift + 1)));
	    }
	  d_scurzot += rpk[i];
	}
      scor = n_scurzot/d_scurzot;
    }

  define_variable("WEIGHT",  format("%g",weight()));
  score(scor);
}

void FitTest_Freq::print_on(ostream& out) const 
{
  out << " FRQ=" << score();
}
