// TABLE is a matrix with arbitrary type and replications
// API access to data is <row, col>, storage is column-first

#include "table.H"

#ifdef HAVE_HDF
#include <mfhdf.h>
#endif

extern "C" {
#include <ctype.h>
};

#include <fstream.h>
#include <utility.H>
#include <array.H>
#include <strings.H>

byte*    TTable::_buf;
unsigned TTable::_bufsz = 4096;

byte* TTable::buffer() 
{
  if (_buf == NULL)
    _buf = new byte[_bufsz];
  return _buf;
}

int TTable::sds_type() 
{
  int ret = 0;
#ifdef HAVE_HDF
  switch (_dtype) {
  case TABLE_BYTE:
    ret = DFNT_INT8;
    break;
  case TABLE_SHORT:
    ret = DFNT_INT16;
    break;
  case TABLE_LONG:
    ret = DFNT_INT32;
    break;
  case TABLE_FLOAT:
    ret = DFNT_FLOAT32;
    break;
  }
#endif
  return ret;
}

ttype TTable::table_type(int sds) 
{
  ttype ret = TABLE_BYTE;
#ifdef HAVE_HDF
  switch (sds) {
  case DFNT_INT8:
    ret = TABLE_BYTE;
    break;
  case DFNT_INT16:
    ret = TABLE_SHORT;
    break;
  case DFNT_INT32:
    ret = TABLE_LONG;
    break;
  case DFNT_FLOAT32:
    ret = TABLE_FLOAT;
    break;
  }
#endif
  return ret;
}

// -------------------------------------------------------------------
// RLDecoder: will eventually move to a lib with brother RLEncoder (in
// SSComm.cc
// -----------------------------------------------------------------------
// run-length decoding algorithm

class RLDecoder
{
  ttype _bsize;
  byte*    _d;
  long     _ofs;

public:

  long decode(byte* b, long max = 0l);

  RLDecoder(byte* d, ttype bsize) : _d(d), _bsize(bsize), _ofs(0l)
  { ; }
  ~RLDecoder() { ; }
};

void TTable::copy(TTable& t)
{
  _dtype = t.type();
  _size[0] = t.size(0);
  _size[1] = t.size(1);
  _replications = t.replications();
  _time = t.time();
  alloc();

  for (int i = 0; i < _replications; i++)
    memcpy((char*)_data[i], t.data(i), _bsize);
}


long RLDecoder::decode(byte* b, long max)
{
  // chunk size is in first bytes (unsigned)
  unsigned cs;
  memcpy(&cs, b, sizeof(unsigned));
  cs -= sizeof(unsigned);
  byte* base = b + sizeof(unsigned); 

  CHECK(_bsize == TABLE_BYTE, "RL decoding of > 8-bit is not implemented");

  // only works for _bsize == 1
  for (unsigned i = 0; i < cs; )
  {
    byte n = base[i++]; 
    byte val = base[i++];
    if (max > 0l)
      if ((_ofs + (long)n) > max)
	{
	  // TBC --- error: it happens (but does not hurt apparently)
	  // printf("W: too many bytes (%ld -> %ld; i = %u, cs = %u)\n",
	  // _ofs, _ofs + n, i, cs);
	  n -= (byte)(_ofs + n - max);
	}
    memset(_d + (unsigned)_ofs, val, (int)n);
    _ofs += (long)n;
  }
  return _ofs;
}


ttype TTable::format2type(char f) 
{
  ttype ret;
  switch (f) {
  case 'F':
  case 'f':
    ret = TABLE_FLOAT;
    break;
  case 'B':
  case 'b':
    ret = TABLE_BYTE;
    break;
  case 'S':
  case 's':
    ret = TABLE_SHORT;
    break;
  case 'L':
  case 'l':
    ret = TABLE_LONG;
    break;
  }
  return ret;
}

unsigned TTable::type2size(ttype t)
{
  unsigned ret;
  switch (t) {
  case TABLE_BYTE:
    ret = sizeof(byte);
    break;
  case TABLE_SHORT:
    ret = sizeof(short);
    break;
  case TABLE_LONG:
    ret = sizeof(long);
    break;
  case TABLE_FLOAT:
    ret = sizeof(float);
    break;
  }
  return ret;
}

void TTable::destroy()
{
  if (_data != NULL)
    {
      for(int i = 0; i < _replications; i++)
	delete _data[i];
      delete _data;
    }
}

// allocates <replications> x*y objects of specified type 
// and returns buffer or NULL
void* TTable::alloc()
{
  destroy();

  //  printf ("alloc called\n");

  _bsize = (long)_size[0] * (long)_size[1] * (long)type2size(_dtype); 

  if (_bsize == 0l) return NULL;
  _data = new void*[_replications];

  CHECK (_data != NULL, "TTable::alloc: malloc failed (001)");

  for(int i = 0; i < _replications; i++)
    {
      _data[i] = new byte[_bsize];
      CHECK (_data != NULL, "TTable::alloc: malloc failed (001)");
    }

  // printf ("%x: allocated %d tables for type %d (bytes %d), size %d x %d: bytes %ld\n",
  //  _data, _replications, (int)_dtype, type2size(_dtype), _size[0], _size[1], _bsize);

  return _data;
}

// reallocs if necessary to accommodate new specs
void* TTable::realloc(ttype t, int x, int y, float time)
{
  // printf ("realloc called with %d %d\n", x, y);
  if (time != -1.0)
    _time = time;

  if (t == _dtype && x == _size[0] && y == _size[1])
    return _data;
  _dtype = t;
  _size[0] = x; 
  _size[1] = y;
  return alloc();
}

// returns float value of data object with conversion
float TTable::Fval (int x, int y, avgmethod avg, int rep)
{   
  //  printf("reading index %ld\n", (long)(x*_size[0]+y));
  
  float vals[_replications]; float ret = 0.0; float sum = 0;  
  if (_data == NULL) 
    return 0.0;
  for (int i = 0; i < _replications; i++)
    {
      switch(_dtype) {
      case TABLE_BYTE:
	vals[i] = (float)(((byte*)_data[i])[x*_size[0]+y]);
	break;
      case TABLE_SHORT:
	vals[i] = (float)(((short*)_data[i])[x*_size[0]+y]);
	break;
      case TABLE_LONG:
	vals[i] = (float)(((long*)_data[i])[x*_size[0]+y]);
	break;
      case TABLE_FLOAT:
	vals[i] = (((float*)_data[i])[x*_size[0]+y]);
	break;
      }
    }
  
  switch (avg) {
  case TABLE_RAWFIRST:
    ret = vals[rep];
    break;
  case TABLE_AVERAGE:
    
  case TABLE_STD:
    break;
  }
  return ret;
} 

// value with proper type
byte TTable::bval(int x, int y, int rep)
{
  CHECK(_dtype == TABLE_BYTE, "TTable: byte asked on non-byte table");
  if (_data == NULL) 
    return (byte)0;
  return ((byte*)_data[rep])[x*_size[0]+y];
}

short TTable::ival(int x, int y, int rep)
{
  CHECK(_dtype == TABLE_SHORT, "TTable: short asked on non-short table");
  if (_data == NULL) 
    return 0;
  return ((short*)_data[rep])[x*_size[0]+y];
}

long TTable::lval(int x, int y, int rep)
{
  CHECK(_dtype == TABLE_LONG, "TTable: long asked on non-long table");
  if (_data == NULL) 
    return 0l;
  return ((long*)_data[rep])[x*_size[0]+y];
}

float TTable::fval(int x, int y, int rep)
{
  CHECK(_dtype == TABLE_FLOAT, "TTable: float asked on non-float table");
  if (_data == NULL) 
    return 0.0;
  return ((float*)_data[rep])[x*_size[0]+y];
}

void TTable::set(int x, int y, int rep, float value)
{
  //  printf("setting index %ld in rep %d\n", (long)(x*_size[0]+y), rep);

  switch (_dtype) {
  case TABLE_BYTE:
    ((byte*)_data[rep])[x*_size[0]+y] = (byte)value;
    break;
  case TABLE_SHORT:
    ((short*)_data[rep])[x*_size[0]+y] = (short)value;
    break;
  case TABLE_LONG:
    ((long*)_data[rep])[x*_size[0]+y] = (long)value;
    break;
  case TABLE_FLOAT:
    ((float*)_data[rep])[x*_size[0]+y] = value;
    break;
  }
}

void TTable::set(int x, int y, int rep, long value)
{
  //  printf("setting index %ld in rep %d\n", (long)(x*_size[0]+y), rep);

  switch (_dtype) {
  case TABLE_BYTE:
    ((byte*)_data[rep])[x*_size[0]+y] = (byte)value;
    break;
  case TABLE_SHORT:
    ((short*)_data[rep])[x*_size[0]+y] = (short)value;
    break;
  case TABLE_LONG:
    ((long*)_data[rep])[x*_size[0]+y] = value;
    break;
  case TABLE_FLOAT:
    ((float*)_data[rep])[x*_size[0]+y] = (float)value;
    break;
  }
}

void TTable::bset(int x, int y, int rep, byte value)
{
  // printf("bsetting index %ld in rep %d\n", (long)((x*_size[0])+y), rep);
  ((byte*)_data[rep])[(x*_size[0])+y] = value;
}

void TTable::iset(int x, int y, int rep, short value)
{
  // printf("isetting index %ld in rep %d\n", (long)((x*_size[0])+y), rep);
  ((short*)_data[rep])[(x*_size[0])+y] = value;
}

void TTable::lset(int x, int y, int rep, long value)
{
  // printf("lsetting index %ld in rep %d\n", (long)((x*_size[0])+y), rep);
  ((long*)_data[rep])[(x*_size[0])+y] = value;
}

void TTable::fset(int x, int y, int rep, float value)
{
  // printf("fsetting index %ld (%d, %d) in rep %d\n", (long)((x*_size[0])+y), x, y, rep);
  ((float*)_data[rep])[(x*_size[0])+y] = value;
}

// read replication from socket/filedesc with optional RLE compression
int TTable::read (FILE* in, int rep)
{
  
  CHECK(rep < _replications, "TTable::read: non-existent replication");

  unsigned long nb; unsigned dx, dy; float min, max; char format;

  // read header: format, nbytes, x, y, min, max \n
  fscanf(in, "%c %ld %u %u %g %g", &format, &nb, &dx, &dy, &min, &max);
  fgetc(in);

  // printf("header: %c %ld %u %u %g %g\n", format, nb, dx, dy, min, max);

  if (dx == 0 || dy == 0) return 0;

  // dx = number of COLUMNS
  // dy = number of ROWS

  bool compress = isupper(format);

  // adjust size if required
  realloc(format2type(format), dx, dy);
  _min = min; _max = max;
    
  if (compress) 
    {
      RLDecoder dec((byte*)_data[rep], format2type(format));

      for ( ; !feof(in); )
	{
	  unsigned cs;
	  // read chunk size
	  fscanf(in, "%u", &cs);
	  // eat the newline left in buffer before switching to binary mode
	  fgetc(in);
	  
	  // printf("%u \n", cs);
	  
	  if (cs == 0)
	    break;
	  // read data into buffer 
	  int z = fread(buffer(), type2size(_dtype), (int)cs, in);
	  
	  // TBC should be z == cs; it isn't at times (why?)
	  // CHECK (z == cs, "TTable::read: wrong RLE decoding");
	  
	  // uncompress into vector
	  long l = dec.decode(buffer(), nb);
	  
	  // printf("decoded %ld bytes\n", l);
	} 
      fflush(in);
    } 
  else 
    {
      int b; long l; float f; 
      // read values in ASCII one by one
      for (int i = 0; i < dy; i++)
	for (int j = 0; j < dx; j++)
	  {
	    // /printf ("reading <%d %d>... ", i ,j);
	    switch (_dtype) 
	      {
	      case TABLE_BYTE:
		fscanf(in, "%d", &b);
		fgetc(in);
		// printf ("%d\n", b);
		bset(i, j, rep, (byte)b);
		break;
	      case TABLE_SHORT:
		fscanf(in, "%d", &b);
		fgetc(in);
		// printf ("%d\n", b);
		iset(i, j, rep, b);
		break;
	      case TABLE_LONG:
		fscanf(in, "%ld", &l);
		fgetc(in);
		// printf ("%ld\n", l);
		lset(i, j, rep, l);
		break;
	      case TABLE_FLOAT:
		fscanf(in, "%g", &f);
		fgetc(in);
		//printf ("%g\n", f);
		fset(i, j, rep, f);
		break;
	      }
	  }
    }
  return 1;
}

const void* TTable::data(int rep) const  
{ 
  return _data == NULL ? NULL : _data[rep];
}

void TTable::compute_bounds(float& min, float& max) 
{
  for (int i = 0; i < _replications; i++)
    for (int r = 0; r < size(0); r++) 
      for (int c = 0; c < size(1); c++)
	{
	  if (min > Fval(r, c, TABLE_RAWFIRST, i)) 
	    min = Fval(r, c, TABLE_RAWFIRST, i);
	  if (max < Fval(r, c, TABLE_RAWFIRST, i)) 
	    max = Fval(r, c, TABLE_RAWFIRST, i);
	}
} 

// read replication from text or HDF file 
int TTable::read (const char* f, bool hdf, int replication)
{
  TArray a;
  
  ifstream in(f);
  if (!in.is_open())
    fatal_box("can't open input file %s", f);
  
  TToken_string t; int nfields = 0;
  while (in.good() && !in.eof())
    {
      in.getline(__tmp_string, sizeof(__tmp_string), '\n');
      t.tokenize(__tmp_string);
      if (t.items() == 0) 
	continue;
      if (nfields == 0) nfields = t.items();
      else {
	if (t.items() != nfields)
	  fatal_box("number of columns in file %s is not consistent", f);
      }
      
      a.add(new TToken_string(t));
    }
  
  realloc(TABLE_FLOAT, nfields, a.items());
  
  for (int k = 0; k < a.items(); k++)
    {
      TToken_string& t = (TToken_string&)a[k];
      for (int q = 0; q < nfields; q++)
	fset(q, k, replication, t.get_float(q));
    }

  return 1;
}

// write replication to text or HDF file 
int TTable::write(const char* f, bool hdf, int replication)
{
  if (hdf) 
    {
#ifdef HAVE_HDF
      
      // 4 dims with time at 1 for total compatibility with timestamped
      // series 
      int32 dims[4];
      dims[0] = 1;
      dims[1] = replications();
      dims[2] = size(0);
      dims[3] = size(1);

      int sd_id  = SDstart(f, DFACC_WRITE);
      int sds_id = SDcreate(sd_id, NULL, sds_type(), 4, dims);

      write_sds_data(sd_id, 0);

      SDendaccess(sds_id);
      SDend(sd_id);
#endif      
    }
  else dump(f);
  return 1;
}

void TTable::flip_vertical() 
{
  // flip data vertically
  int bs = size(1)*type2size(_dtype);
  byte* row = new byte[bs];
  for (int i = 0; i < _replications; i++)
    {
      byte* data = (byte*)_data[i];

      // swap rows 
      for (int r = 0; r < (size(0)/2); r++) 
	{
	  byte* afrom = data + (r*size(1));
	  byte* ato   = data + ((size(0)-r-1)*size(1));
	  memcpy(row, ato, bs);
	  memcpy(ato, afrom, bs);
	  memcpy(afrom, row, bs);
	}
    }  
}

// appends slab to HDF file along time dimension: passed the sd_id of
// existing SDS array and time dim index. Type and dimensions are
// assumed correct.
int TTable::write_sds_data(int sds_id, int idx, bool reversed)
{

#ifdef HAVE_HDF
  // dimensions are X Y R T=idx
  // if reversed rows become columns

  int32 edges[4];
  int32 start[4];

  edges[0] = 1;
  edges[1] = 1;
  if (reversed) {
    edges[2] = size(1);
    edges[3] = size(0);
  } else {
    edges[2] = size(0);
    edges[3] = size(1);
  }
  start[0] = idx; 
  start[2] = 0;
  start[3] = 0;
  
  for (int i = 0; i < replications(); i++)
    {
      start[1] = i;
      int32 ret = SDwritedata(sds_id, start, NULL, edges, (void*)_data[i]);

      /***************************************************************
      else
	{
	  void* data = malloc(size(0)*size(1)*type2size(_dtype));
	  for (int j = 0; j < size(0); j++) 
	    for (int k = 0; k < size(1); k++)
	      {
		switch (_dtype) 
		  {
		  case TABLE_BYTE:
		    ((byte*)data)[j+k*size(1)] = bval(j,k,i);
		    break;
		  case TABLE_SHORT:
		    ((short*)data)[j+k*size(1)] = ival(j,k,i);
		    break;
		  case TABLE_LONG:
		    ((long*)data)[j+k*size(1)] = lval(j,k,i);
		    break;
		  case TABLE_FLOAT:
		    ((float*)data)[j+k*size(1)] = fval(j,k,i);
		    break;
		  }
	      }
	  int32 ret = SDwritedata(sds_id, start, NULL, edges, (void*)_data[i]);
	  free(data);
	}
	************************************************************/
    }
#endif
  return 1;
}

// read data from HDF file along time dimension: passed the sd_id of
// existing SDS array, time dim index and dimensions in dataset. 
// Type and dimensions are assumed correct for table allocation
int TTable::read_sds_data(int sds_id, int idx)
{

#ifdef HAVE_HDF
  int32 edges[4];
  int32 start[4];

  edges[0] = 1;
  edges[1] = 1;
  edges[2] = size(0);
  edges[3] = size(1);

  start[0] = idx; 
  start[2] = 0;
  start[3] = 0;
  
  for (int i = 0; i < replications(); i++)
    {
      start[1] = i;
      int32 ret = SDreaddata(sds_id, start, NULL, edges, (void*)_data[i]);
    }
  
#endif
  return 1;
}

// dump to text (create file or append)
int TTable::dump(const char* f, avgmethod avg, bool append)
{
  return 1;
}

// can create with size or without
TTable::TTable(ttype type, int x, int y, float time, int replications) : 
  _dtype(type), _data(NULL), _replications(replications), _time(time),
  _min(0.0), _max(0.0)
{
  _size[0] = x;
  _size[1] = y;
  alloc();
}

TTable::TTable(TTable& t) 
{
  copy(t);
}

TTable::~TTable()
{
  // delete all replications
  destroy();
}






