// table: timestamped 2d table object with variable type and possible
// replication
// can read itself from/to HDF files and from network in ASCII and
// RL encoded format

#ifndef __TABLE_H
#define __TABLE_H

#ifndef __OBJECT_H
#include <object.H>
#endif

#ifndef __CLASSES_H
#include <classes.H>
#endif

#include <stdio.h>

// allowed types and sizes
enum ttype {
  TABLE_BYTE = 0,
  TABLE_SHORT = 1,
  TABLE_LONG = 2,
  TABLE_FLOAT = 3
};

// how to deal with replicates
enum avgmethod {
  TABLE_RAWFIRST = 0, // value of first replication
  TABLE_AVERAGE  = 1, // average 
  TABLE_STD      = 2  // standard deviation
};   

class TTable : public TObject
{
  void**       _data;       // data buffer
  long         _bsize;        // byte size of data buffer
  unsigned int _size[2];      // size of each dimension
  float        _time;         // timestamp
  int          _replications; // n. of replications
  ttype        _dtype;        // type of data
  float        _min, _max;    // min, max in data (if read from FILE*)

  static byte*       _buf;             // byte buffer for read/write      
  static unsigned    _bufsz;           // buffer size
  
  static byte* buffer();
  static unsigned& bufsz() { return _bufsz; }

  void destroy();
  void* alloc();
  ttype format2type(char f); 
  void copy(TTable& t);

public:

  static unsigned type2size(ttype t); 

  const float time() const         { return _time; }
  const unsigned size(int d) const { return _size[d]; }
  const ttype type() const         { return _dtype; }
  const int replications() const   { return _replications; }
  const void* data(int rep) const;
  const float min() const          { return _min; }
  const float max() const          { return _max; }
  const unsigned rows() const      { return _size[1]; }
  const unsigned columns() const   { return _size[0]; }

  // flip the data vertically
  void flip_vertical();

  // HDF interface
  int  sds_type();
  static ttype table_type(int sds_type);
  int write_sds_data(int, int, bool reversed = FALSE);
  int read_sds_data(int, int);
  
  void  time(float t) { _time = t; }

  // re-allocates <replications> x*y objects of specified type 
  // and returns buffer or NULL; does not do unnecessary allocs
  void* realloc(ttype t, int x, int y, float time = -1.0);

  // returns float value of data object with conversion and averaging
  float  Fval(int x, int y, avgmethod a = TABLE_RAWFIRST, int rep = 0); 
  float  operator()(int x, int y, avgmethod a = TABLE_RAWFIRST, int rep = 0) 
  { return Fval(x, y, a, rep); }
  TTable& operator=(TTable& t) 
  { copy(t); return *this; }

  // returns value of specified replication with proper type
  byte   bval(int x, int y, int rep = 0);
  short  ival(int x, int y, int rep = 0);
  long   lval(int x, int y, int rep = 0);
  float  fval(int x, int y, int rep = 0);

  // sets value (convert float to proper type)
  void set(int x, int y, int rep, float value);
  // sets value (convert long to proper type)
  void set(int x, int y, int rep, long value);
  // set typed value
  inline void bset(int x, int y, int rep, byte value);
  inline void iset(int x, int y, int rep, short value);
  inline void lset(int x, int y, int rep, long value);
  inline void fset(int x, int y, int rep, float value);

  // compute min/max on all rep data
  void compute_bounds(float& min, float& max);

  // read replication from socket/filedesc with optional RLE compression
  int read (FILE* s, int replication = 0); 
  // read replication from text or HDF file 
  int read (const char* f, bool hdf = FALSE, int replication = 0);      
  // write replication to text or HDF file 
  int write(const char* f, bool hdf = FALSE, int replication = 0);
  // dump to text file (create file or append)
  int dump(const char* f, avgmethod a = TABLE_RAWFIRST, bool append = FALSE);
  
  // can create with size or without
  TTable(ttype type = TABLE_BYTE, int x = 0, int y = 0, float time = 0.0,
	 int replications = 1);
  TTable(TTable& t);
  virtual ~TTable();
};


#endif




