#ifdef USE_TCL

#include "tclobj.H"
extern "C" {
#include <tk.h>
extern Tcl_Interp* Interp;
extern int Tcl_GetCommandInfo(Tcl_Interp*, char*, Tcl_CmdInfo*);
};

#include <stdarg.h>
#include <utility.H>
#include <stdlib.h>
#include <array.H>
#include <ctype.h>

#define MAXNOPT 60
#define MAXLOPT 1024

// this is the max length of the result - might need a smarter 
// func with realloc - not easy unless we give up using vsprintf
static char _mtmp_[65535];

class tmp_obj : public TObject
{
  TclObject* _to;
  public:
    tmp_obj(TclObject* o) : _to(o) {;}
    virtual ~tmp_obj() { _to->destroy(); delete _to; }    
};

TArray& temp_objs()
{
  static TArray* _tobjs = NULL;
  if (_tobjs == NULL)
  {
    _tobjs = new TArray(32);
    // needed since TclObject::destroy() calls DeleteTclObj
    // which deletes the object associated with the command
    _tobjs->autodestroy(FALSE);
  }
  return *_tobjs;
}

void DeleteTempTclObjects()
{
  for (int i = 0; i < temp_objs().items(); i++)
    {
      TclObject& o = (TclObject&)(temp_objs()[i]);
      o.destroy();
      temp_objs().add(NULL, i);
    }
}

// Utility func to separate command line into argv and optv returns
// name of called method and makes it argv[0] options are introduced
// by '-': if they have arguments, args must be given as
// '-option=value' Arguments starting with - can be passed prefixed by
// '#'. A lowercase letter must follow - to be recognized as an
// option; passing - alone disables option checking from there on

const char* set_cl(int& argc, char** argv, int& ac, char*** av, \
		   int& oc, char*** ov)
{
  ac = oc = 0;
  bool opt_allowed = TRUE;
  
  //  TString src(80); 
  //TString arg(80); 
  //TToken_string res(80);

  TArray o(10), a(10);

  if (argc <= 1) return NULL;

  for (int i = 2; i < argc; i++)
    {
      char* s; bool optflag = FALSE;
      
      if (*argv[i] == '-' && opt_allowed)
	{
	  if (*(argv[i]+1) == '\0') // disable option checking
	    {
	      opt_allowed = FALSE;
	      continue;
	    }
	  else 
	    optflag = islower(*(argv[i]+1));
	}

      
      if (optflag) s = argv[i]+1;
      else         s = (*argv[i] == '#' && *(argv[i]+1) > '\0') ? 
	               (argv[i]+1) : argv[i];

      TString arg(s), src(12); TToken_string res(64);

      // if option ends by ':' the next arg is the option value
      if (optflag && arg[arg.len()-1] == ':')
	{
	  src = arg.sub(0,arg.len()-1);
	  res.add(src);
	  res.add(i == (argc - 1) ? "" : argv[++i]);
	}
      else res = (const char*)arg;
      
      if (optflag)
	o.add(new TToken_string(res));
      else
	a.add(new TToken_string(res));
    }
  
  // allocate new arguments
  ac = a.items(); oc = o.items();
  (*av) = ac > 0 ? new char*[ac] : (char**)NULL;
  (*ov) = oc > 0 ? new char*[oc] : (char**)NULL;

  for (int aa = 0; aa < ac; aa++) 
    {
      TToken_string& t = (TToken_string&)a[aa];
      (*av)[aa] = new char[t.len()+1];
      strcpy((*av)[aa], (const char*)t);
    }
  for (int oo = 0; oo < oc; oo++) 
    {
      TToken_string& t = (TToken_string&)o[oo];
      (*ov)[oo] = new char[t.len()+1];
      strcpy((*ov)[oo], (const char*)t);
    }

  return argv[1]; // object name or method 
}

void free_cl(int ac, char** av, int oc, char** ov)
{
  if (ac > 0)
    {
      for (int i = 0; i < ac; i++)
	delete av[i];
      delete av;
    }
  if (oc > 0)
    {
      for (int j = 0; j < oc; j++)
	delete ov[j];
      delete ov;
    }
}

void DeleteTclObject(TclObject* data)
{
  /* delete data; */ 
  data->mark_for_deletion(); 
}

// command procedure used for TclObject methods
int TclObjectCmdProc(ClientData data, Tcl_Interp* interp,
		     int argc, char* argv[])
{
  int ret = TCL_OK;

  TclObject* sp = (TclObject*)data;

  int ac, oc; char **av, **ov;

  // parse argv and options; put result in new arrays
  const char* method = set_cl(argc, argv, ac, &av, oc, &ov);

  // call method: class sets result
  bool destroyed = sp->call_method(method, ac, av, oc, ov);  

  // free options
  free_cl(ac, av, oc, ov);

  // set result
  if (!destroyed)
    {
      Tcl_SetResult(interp, (char*)(const char*)(sp->result()), TCL_VOLATILE);
      ret = sp->is_error() ?  TCL_ERROR :  TCL_OK;
    }
  else
    Tcl_SetResult(interp, "", TCL_STATIC);

  if (sp->expired())
    delete sp;

  return ret;
}


// command procedure called for each object reference
int CreateTclObject(ClientData d, Tcl_Interp* interp,
		    int argc, char* argv[])
     // called by commands scanning TclObjectTable
{
  int ac, oc; char **av, **ov;
  const char* name = set_cl(argc, argv, ac, &av, oc, &ov);
  
  // retrieve class description
  TclClassDesc* tcd = (TclClassDesc*)d;
  
  // create object passing args to constructor
  // object installs by itself
  TclObject* to = (tcd->_mkfunc)(ac, av, oc, ov, (char*)name);
  
  if (to != NULL)
    {
      
      // remember pointer to class name to invoke TCL methods
      to->set_class_name((char*)tcd->_name);
      
      if (to->is_error())
	strcpy(interp->result, to->result());
      else
	{
	  // execute TCL initialization code if present; ignore error
	  TString t(128); t =  tcd->_name; t << "::"; t <<  tcd->_name;
	  
	  Tcl_CmdInfo tci;
	  if (Tcl_GetCommandInfo(interp, (char*)(const char*)t, &tci) != 0) 
	    // let TCL code set the result: must return $this normally
	    to->set_error(to->invoke_tcl_method(t, ac, av));
	  else
	Tcl_SetResult(interp, (char*)(const char*)(to->result()), 
		      TCL_VOLATILE);
	}
    }
  // free command line
  free_cl(ac, av, oc, ov);

  if (to == NULL) return TCL_ERROR;
  return to->is_error() ? TCL_ERROR : TCL_OK;
}



// TclObject methods
TclObject* TclObject::name2obj(const char* name)
{
  Tcl_CmdInfo cmi;
  if (Tcl_GetCommandInfo(Interp, (char*)name, &cmi))
    return (TclObject*)(cmi.clientData);
  else return NULL;
}

void TclObject::error(const char* fmt, ...)
{
  va_list pars;
  va_start(pars, fmt);
  vsprintf(_mtmp_, fmt, pars);
  va_end(pars);
  _result = _mtmp_;
  _error = TRUE;
}

methodID TclObject::add_method(const char* name, methodID id)
{
  if (id == -1) id = _meth_cnt++;
  TString  m(name); 
  if (UseIncrTcl) m.insert("_",0);
  TString* v = new TString(5); 
  (*v) << id;
  if (_methods.add(m,v))
    error("redefinition of method %s", name);
  return id;
}

methodID TclObject::method_id(const char* name)
{
  if (_methods.is_key(name))
    {
      TString& o = (TString&)_methods[name];
      return atoi((const char *)(TString&)o);
    }
  return -1;
}

int TclObject::set_result(const char*  fmt, ...)
{
  if (fmt == NULL) reset_result();
  else 
    {
      va_list pars;
      va_start(pars, fmt);
      vsprintf(_mtmp_, fmt, pars);
      va_end(pars);
      _result = _mtmp_;
    }
}

int TclObject::add_result(const char* tk)
{
  _result.add(tk);
}

int TclObject::add_result(double d)
{
  _result.add(format("%g",d));
}

int TclObject::add_result(int n)
{
  _result.add(n);
}

int TclObject::add_result(long l)
{
  _result.add(l);
}

void TclObject::destroy()
{
  // execute TCL destructor code if present
  TString t; t = _class_name; t << "::~" << _class_name;

  Tcl_CmdInfo tci;
  if (Tcl_GetCommandInfo(Interp, (char*)(const char*)t, &tci) != 0) 
    invoke_tcl_method(t, 0, NULL);

  //  printf ("object %s destroyed\n", (const char*)_name);

  Tcl_DeleteCommand(Interp, (char*)(const char*)_name);
}

int InstallItclClass(const char* cl, ...)
{
  TString s(512); s = "";

  s << "itcl_class " << cl << " {" << '\n' 
    << "protected x" << '\n'
      << "constructor args { set x [_" << cl << " new $args] \n}\n" << '\n'
	<< "destructor {$x _destroy \n}\n" << '\n'
	  << "method name {} { return $x \n}\n" << '\n';
  // add all methods as "method <m> args { return [#cmd <m> $args]}\n"
  
  va_list pars;
  va_start(pars, cl);
  const char* arg;
  while ((arg = va_arg(pars, const char*)) != NULL)
    {
      if (strncmp(arg,"inherit",7) == 0)
	{
	  s << arg << '\n';
	}
      else
	{
	  s << "method " << arg << " args {\n\t return [$x _" 
	    << arg << " $args] \n}" << '\n';
	}
    }
  va_end(pars);
  
  s <<  "}" << '\n';

  if (Tcl_Eval(Interp, (char*)(const char*)s) != TCL_OK)
    {
      cerr << "Class initialization failed!\n";
      exit(255);
    }
}


int TclObject::execute(const char* code)
{
  reset_result(); Tcl_ResetResult(Interp);
  int ret = Tcl_GlobalEval(Interp, (char*)code);
  set_result(Interp->result);
  return ret;
}

const char* TclObject::execute(const char* code, int& ret)
{
  reset_result(); Tcl_ResetResult(Interp);
  ret = Tcl_GlobalEval(Interp, (char*)code);
  return (const char*)(Interp->result);
}

const char* TclObject::expr(const char* code, int& ret)
{
  reset_result(); Tcl_ResetResult(Interp);
  ret = Tcl_ExprString(Interp, (char*)code);
  return (const char*)(Interp->result);
}

int TclObject::execute_with_ret(const char* code)
{
  // execute code and return (integer) result
  Tcl_ResetResult(Interp);
  if (execute(code) == TCL_OK)
    return atoi(Interp->result);
  else return TCL_ERROR;
}

int TclObject::invoke_tcl_method(const char* tclproc, int argc, char** argv,
				 bool ign_result)
{
  TString t(128);
  t = tclproc; t << " " << name();
  for (int i = 0; i < argc; i++)
    t << " {" << argv[i] << "}";
  int r;
  TString tl(Interp->result);

  execute(t,r);

  if (ign_result && r != TCL_ERROR)
      strcpy(Interp->result,(const char*)tl);
  else 
      set_result("%s", Interp->result);

  return r;
}

const char* TclObject::get_global_var(const char* var)
{
  return (const char*)Tcl_GetVar(Interp, (char*)var, TCL_GLOBAL_ONLY);
}

void TclObject::set_global_var(const char* var, const char* val)
{
  Tcl_SetVar(Interp, (char*)var, (char*)val, TCL_GLOBAL_ONLY);
}

void TclObject::invalid_method (const char* meth, int argc, char** argv, 
				int optc, char** optv) 
{
  error("invalid method %s called", meth);
}

bool TclObject::call_method (const char* meth, int argc, char** argv, 
			     int optc, char** optv) 
{  
  methodID id = -1;
  reset_result();

  // process built_in methods
  if (meth == NULL)
    {
      do_method(NULL_METHOD, 0, NULL, 0, NULL);
    }
  else if (strcmp(meth, UseIncrTcl ? "_destroy" : "delete") == 0)
    {
      destroy();
      return TRUE;
    }
  // user can redefine all methods except new and delete
  else if ((id = method_id(meth)) != -1)  // call method added by 'add_method'
    {
      do_method(id, argc, argv, optc, optv);
    }
  // built-in methods bound to virtuals (if not redefined)
  else if (strcmp(meth, "class") == 0)    // return class name
    {
      set_result("%s", _class_name);
    }
  else 
    {
      // method not found: 
      // invoke TCL proc  '<classname>::<methodname> <objname> args...'
      TString t; t = format("%s::%s", _class_name, meth);
      
      Tcl_CmdInfo tci;
      if (Tcl_GetCommandInfo(Interp, (char*)(const char*)t, &tci) != 0) 
	_error = invoke_tcl_method(t, argc, argv);
      else invalid_method(meth, argc, argv, optc, optv);
    }
  return FALSE;
}

TclObject::TclObject(int argc, char** argv , 
		     int optc, char** optv, char* name) :
		     _result(32, ' '), _error(FALSE), _meth_cnt(0),
		     _name(32), _expired(FALSE)
{
  bool nw = FALSE;
  if (name == NULL || strcmp(name,"new") == 0 || strcmp(name,"_tmp_") == 0)
    {
      _name = gensym();
      nw = TRUE;
      // if it's _tmp_ it's scheduled for deletion in _tobjs array
      // deleted by user with DeleteTempTclObjects()
      if (strcmp(name, "_tmp_") == 0)
        temp_objs().add(new tmp_obj(this));
    }
  else _name = name;

  // registers itself into interpreter
  Tcl_CreateCommand(Interp, (char*)(const char*)_name, TclObjectCmdProc, 
		    this, DeleteTclObject);

  // return command name
  set_result((const char*)_name);
}

TclObject::~TclObject()
{ 
  //  printf ("object %s deleted\n", (const char*)_name);
}

#endif
