/* ELM raster flux functions

   This set of modules contains the cell-cell surface&ground water flux functions,
   the integration of surface, unsat, and saturated water in vertical dimension,
   and the acquisition and use of dynamic boundary conditions in external cells.
   The hydrologic and solute budgets for basins/IRs are calculated for the fluxes.
*/

/***
The major functions in the sequence in which they are found in this file:
    Flux_SWater (alternating direction explicit (ADE) calling function - surface water)
    Flux_SWcells (calculate potential and actual fluxes among cells - surface water)
    Flux_SWstuff (flux solutes associated with water fluxes - surface water)
    Flux_GWater (alternating direction explicit (ADE) calling function - ground water)
    Flux_GWcells (calculate potential and actual fluxes among cells - ground water;
       also includes surface-ground water integration, solute fluxes)
    ReadBCond (read data on dynamic stage in domain boundary cells-INCOMPLETE)
    
***/
   

/* double precision for all constituent solute vars */

/* general NOTES on revisions to this source file:
        april 00 VERSION 2.1 - output available for application for water quality performance
        july 02 VERSIOIN 2.1a - first widely-available public release of code -
              started adding dynamic stage boundary conditions - INCOMPLETE, not functional
              added some more code documentation, etc.

*/

#include "Fluxes.h" 

float step_Cell; /* = time step/CELL_SIZE  */
float baseDatum;

FILE *F_extStage;

FILE *garb;

int BCnumCells; /* number of external boundary condition cells with dynamic data */
char BCsrc_data[20]; /* name of model (or historical) data source for external bound conds */

/*************************************************************************************************/
/* Surface water fluxing routine
   This is the alternating direction function that first fluxes water in the E/W direction and then in the N/S 
   direction. It sweeps from left to right, then goes back from right to left, then goes from    
   top to bottom and finally from bottom to top. This alternates every hyd_iter. 

   Surface water model boundary exchanges may only occur across cells designated with designated attribute in 
   the boundary condition map file.
 */

/* Note that surface water variable is actually updated in the Flux_SWstuff function */

#if NeedFunctionPrototypes
void Flux_SWater 
(int it, float *SURFACE_WAT,float *SED_ELEV,float *HYD_MANNINGS_N, double *STUF1, double *STUF2, double *STUF3, float HDmax)
#else 
void Flux_SWater ( it, SURFACE_WAT, SED_ELEV, HYD_MANNINGS_N, STUF1, STUF2, STUF3, HDmax)
int it;
float *SURFACE_WAT, *SED_ELEV, *HYD_MANNINGS_N, HDmax;
double  *STUF1, *STUF2, *STUF3;
#endif

{ int ix, iy;
 float FF;
 step_Cell = sq_celWid * sfstep/CELL_SIZE; /* constant used in surface water flux equations */


 
/* check the donor and recipients cells for a) on-map, b) the cell attribute that allows sfwater
   boundary flow from the system and c) the attribute that indicates levee presence:
   the levee attribute of 1 (bitwise) allows flow to east;
   attribute of 2 (bitwise) allows flow to south (levee atts calc'd by WatMgmt.c) */

     /* as always, x is row, y is column! */
 for(ix=1; ix<=s0; ix++) 
 {
     if (it%2) {  /* alternate loop directions every other hyd_iter (it) */
         for(iy=1; iy<=s1; iy++)  /* loop from west to east */
         {
             
             if( ( ON_MAP[T(ix,iy)] && ON_MAP[T(ix,iy+1)] && (int)(ON_MAP[T(ix,iy)]-1) & 1  ) || 
                 BCondFlow[T(ix,iy+1)] == 3 ||  BCondFlow[T(ix,iy)] == 3  )
             {
                 FF = Flux_SWcells(ix,iy,ix,iy+1,SURFACE_WAT,SED_ELEV,HYD_MANNINGS_N,HDmax); 
				/* FF units = m */
                 if (FF != 0.0) Flux_SWstuff ( ix,iy,ix,iy+1,FF,SURFACE_WAT,STUF1,STUF2,STUF3);
             }
         } 
     }
     

     else { 
     for(iy=s1; iy>=1; iy--)   /* loop from east to west */
         if( ( ON_MAP[T(ix,iy)] && ON_MAP[T(ix,iy-1)] && (int)(ON_MAP[T(ix,iy-1)]-1) & 1 ) || 
              BCondFlow[T(ix,iy-1)] == 3 || BCondFlow[T(ix,iy)] == 3  )
         {
             FF = Flux_SWcells(ix,iy-1,ix,iy,SURFACE_WAT,SED_ELEV,HYD_MANNINGS_N,HDmax); 
				/* FF units = m */
             if (FF != 0.0) Flux_SWstuff ( ix,iy-1,ix,iy,FF,SURFACE_WAT,STUF1,STUF2,STUF3);
         }
         } 
 }
	
 for(iy=1; iy<=s1; iy++) 
 {
     if (it%2) {  /* alternate loop directions every other hyd_iter (it) */
     for(ix=1; ix<=s0; ix++)  /* loop from north to south */
         if( ( ON_MAP[T(ix,iy)] && ON_MAP[T(ix+1,iy)]  && (int)(ON_MAP[T(ix,iy)]-1) & 2 ) ||
              BCondFlow[T(ix+1,iy)] == 3 || BCondFlow[T(ix,iy)] == 3  )
         { 
                 FF = Flux_SWcells(ix,iy,ix+1,iy,SURFACE_WAT,SED_ELEV,HYD_MANNINGS_N,HDmax); 
 				/* FF units = m */
             if (FF != 0.0) Flux_SWstuff ( ix,iy,ix+1,iy,FF,SURFACE_WAT,STUF1,STUF2,STUF3);
         }
         } 
     else { 
     for(ix=s0; ix>=1; ix--)  /* loop from south to north */
         if( ( ON_MAP[T(ix,iy)] && ON_MAP[T(ix-1,iy)] && (int)(ON_MAP[T(ix-1,iy)]-1) & 2 ) ||
              BCondFlow[T(ix-1,iy)] == 3 || BCondFlow[T(ix,iy)] == 3  )
         {
             
             FF = Flux_SWcells(ix-1,iy,ix,iy,SURFACE_WAT,SED_ELEV,HYD_MANNINGS_N,HDmax); 
				/* FF units = m */
             if (FF != 0.0) Flux_SWstuff ( ix-1,iy,ix,iy,FF,SURFACE_WAT,STUF1,STUF2,STUF3);
         }
     }
     }
 

}

/************************************************************************************************/
/* Surface water flux eqns.
   Application of Manning's eqn to calculate flux between two adjacent cells (i0,i1) and (j0,j1)
   Returns height flux.  Flux is positive if flow is from i to j.
   Checks for available volume, and that flow cannot make the head in recepient cell higher
   than in the donor one. */


#if NeedFunctionPrototypes
float 
Flux_SWcells(int i0,int i1,int j0,int j1, float *SWater,float *Elevation,float *MC, float HDmax)
#else 
float Flux_SWcells(i0, i1, j0, j1, SWater, Elevation, MC, HDmax)
float *SWater, *Elevation, *MC, HDmax;
int i0, i1, j0, j1;
#endif

{	
	float fr, dh, adh, M, Hi, Hj, H, F = 0.;
	int cellLoci = T(i0,i1);
	int cellLocj = T(j0,j1);

        int j = 5;
        
        
	M = (MC[cellLoci] + MC[T(j0,j1)])/2.;



/*	If an on-map cell is marked 3, we are at a model boundary allowing surface water exchange */		
    if  ( !ON_MAP[cellLoci] && BCondFlow[cellLocj] == 3 ) { 
            /* Hi = Elevation[cellLocj] - 0.05 + SWater[cellLoci];  alternating within hyd_iter needed to keep track of whatever (temporary) depth present */
             Hi = Elevation[cellLocj] + Max(SWater[cellLocj]-0.05,0.0) ;  /* the off-map cell given head 5 cm less than donor */

            /* new dynamic boundary condition stage */
/*  Hi = BCdata[j].arrayStage[((int)(TIME))] ; */
        
        M = MC[cellLocj];       /* the mannings n is not avg, but the value of onmap boundary cell */
    }
	else Hi = SWater[cellLoci] + Elevation[cellLoci];

    if ( BCondFlow[cellLoci] == 3 && !ON_MAP[cellLocj] ) { /*dec98*/
            /* Hj = Elevation[cellLoci] - 0.05 + SWater[cellLocj];   alternating within hyd_iter needed to keep track of whatever (temporary) depth present */
        Hj = Elevation[cellLoci] + Max(SWater[cellLoci]-0.05,0.0) ;  /* the off-map cell given head 5 cm less than donor  */
        M = MC[cellLoci];      /* the mannings n is not avg, but the value of onmap boundary cell */
	}
	else Hj = SWater[cellLocj] + Elevation[cellLocj];

	dh = Hi - Hj;		/* dh is "from --> to" */
	adh = ABS (dh);
		
	if (dh > 0) 
	{
	   	if (SWater[cellLoci] < DetentZ) return 0.0; 
		F = (M != 0) ? 
			/* (sq_celWid * pow(adh,alpha2) * base_flux_rate / M  * pow(SWater[cellLoci],alpha1)*sfstep/CELL_SIZE) : */ 
			(pow(adh,alpha2) * base_flux_rate / M  * pow(SWater[cellLoci],alpha1)*step_Cell) : 
			(0.0);
                    /* ensure adequate volume avail */
		F =  ( F > ramp(SWater[cellLoci] - DetentZ) ) ? (ramp(SWater[cellLoci] - DetentZ)) : (F);
		 /* check to ensure no flip/flops associated with depth */
                if ( ( Hi - F ) < ( Hj + F ) )	F = Min ( dh/2.0, ramp(SWater[cellLoci] - DetentZ) );
	}
	else
	{	
	   	if (SWater[cellLocj] < DetentZ) return 0.0; 
		/* F is negative in this case */
		F = (M != 0) ? 
			/* ( - sq_celWid * pow(adh,alpha2) * base_flux_rate / M  * pow(SWater[cellLocj],alpha1)*sfstep/CELL_SIZE) : */ 
			( - pow(adh,alpha2) * base_flux_rate / M  * pow(SWater[cellLocj],alpha1)*step_Cell) : 
			(0.0);
                    /* ensure adequate volume avail */
		F =  ( -F > ramp(SWater[cellLocj] - DetentZ) ) ? (-ramp(SWater[cellLocj] - DetentZ)) : (F);
		 /* check to ensure no flip/flops associated with depth */		
		if ( ( Hi - F ) > ( Hj + F ) )  F = - Min ( adh/2.0, ramp(SWater[cellLocj] - DetentZ) );
	}
	
	return (F);	/* returns height flux */
}

/*************************************************************************************************/
/* Flux of surface water solutes (nutrients, salinity, etc), and update surface water variable, budget calcs.
   These are units of solute mass being fluxed from i0,i1 to j0,j1 
   Flux & SURFACE_WAT are in units of height (m); 
   for mass balance checks, the order of STUF1-3 is salt, nitrogen, phosphorus */

#if NeedFunctionPrototypes
void Flux_SWstuff 
(int i0,int i1,int j0,int j1, float Flux, float *SURFACE_WAT, double *STUF1, double *STUF2, double *STUF3)
#else 
void Flux_SWstuff ( i0,i1,j0,j1,Flux,SURFACE_WAT,STUF1,STUF2,STUF3 )
int i0,i1,j0,j1;
float Flux,*SURFACE_WAT;
double *STUF1,*STUF2,*STUF3;
#endif

{
    float m1=0.0, m2=0.0, m3=0.0;
    int  ii, flo_chek, cel_i, cel_j;
    struct basnDef *basins;

    cel_i = T(i0,i1);
    cel_j = T(j0,j1);
    
 
 if (Flux >0.0) {
     m1 = (SURFACE_WAT[cel_i]>0.0) ? (STUF1[cel_i]*Flux/SURFACE_WAT[cel_i]) : (0.0);
     m2 = (SURFACE_WAT[cel_i]>0.0) ? (STUF2[cel_i]*Flux/SURFACE_WAT[cel_i]) : (0.0);
     m3 = (SURFACE_WAT[cel_i]>0.0) ? (STUF3[cel_i]*Flux/SURFACE_WAT[cel_i]) : (0.0);
 }
 else  {
     m1 = (SURFACE_WAT[cel_j]>0.0) ? (STUF1[cel_j]*Flux/SURFACE_WAT[cel_j]) : (0.0);
     m2 = (SURFACE_WAT[cel_j]>0.0) ? (STUF2[cel_j]*Flux/SURFACE_WAT[cel_j]) : (0.0);
     m3 = (SURFACE_WAT[cel_j]>0.0) ? (STUF3[cel_j]*Flux/SURFACE_WAT[cel_j]) : (0.0);
 }
	
 STUF1[cel_j] += m1;  /* add the masses of solutes */
 STUF2[cel_j] += m2;
 STUF3[cel_j] += m3;
 STUF1[cel_i] -= m1;
 STUF2[cel_i] -= m2;
 STUF3[cel_i] -= m3;
 SURFACE_WAT[cel_j] += Flux; /* now update the surfwater depths */
 SURFACE_WAT[cel_i] -= Flux;

 if (debug > 2) {
     if (STUF3[cel_j] < -MinCheck) {
         sprintf(msgStr,"Day %6.1f: capacityERR - neg surface water P (%f kg) in cell (%d,%d) after SWfluxes!", 
                 TIME, STUF3[cel_j], j0,j1 ); 
         WriteMsg( msgStr,True );  }
     if (STUF3[cel_i] < -MinCheck) {
         sprintf(msgStr,"Day %6.1f: capacityERR - neg surface water P (%f kg) in cell (%d,%d) after SWfluxes!", 
                 TIME, STUF3[cel_i], i0,i1 ); 
         WriteMsg( msgStr,True );  }
     if (STUF1[cel_j] < -MinCheck) {
         sprintf(msgStr,"Day %6.1f: capacityERR - neg surface water S (%f kg) in cell (%d,%d) after SWfluxes!", 
                 TIME, STUF1[cel_j], j0,j1 ); 
         WriteMsg( msgStr,True );  }
     if (STUF1[cel_i] < -MinCheck) {
         sprintf(msgStr,"Day %6.1f: capacityERR - neg surface water S (%f kg) in cell (%d,%d) after SWfluxes!", 
                 TIME, STUF1[cel_i], i0,i1 ); 
         WriteMsg( msgStr,True );  }
     if (SURFACE_WAT[cel_j] < -MinCheck) {
         sprintf(msgStr,"Day %6.1f: capacityERR - negative surface water (%f m) in cell (%d,%d)!", 
                 TIME, SURFACE_WAT[cel_j], j0,j1 ) ; 
         WriteMsg( msgStr,True ); }

     if (SURFACE_WAT[cel_i] < -MinCheck) {
         sprintf(msgStr,"Day %6.1f: capacityERR - negative surface water (%f m) in cell (%d,%d)!", 
                 TIME, SURFACE_WAT[cel_i], i0,i1 ) ; 
         WriteMsg( msgStr,True );  }
 }
        

/* mass balance sums */
 if (basn[cel_i] != basn[cel_j])  { 

	/* first do the normal case where all cells are on-map */
     if ( ON_MAP[cel_j] && ON_MAP[cel_i] ) { 
         if ( Flux > 0  ) { /* positive fluxes out of basn[cel_i] */
             if (basn_list[basn[cel_i]]->family !=  
                 basn_list[basn[cel_j]]->family ) {        /* if the flow is not within the family... */
                 if  ( !basn_list[basn[cel_i]]->parent  ) { /* and if the donor or recipient is a child... */
                     /* then find out about the flow for the family's sake */
                     VOL_OUT_OVL[basn_list[basn[cel_i]]->family] += Flux*CELL_SIZE;
                     P_OUT_OVL[basn_list[basn[cel_i]]->family] += m3;
                     S_OUT_OVL[basn_list[basn[cel_i]]->family] += m1;
                 }
                 if ( !basn_list[basn[cel_j]]->parent ) {                 
                     /* then find out about the flow for the family's sake */
                     VOL_IN_OVL[basn_list[basn[cel_j]]->family] += Flux*CELL_SIZE;
                     P_IN_OVL[basn_list[basn[cel_j]]->family] += m3;
                     S_IN_OVL[basn_list[basn[cel_j]]->family] += m1;
                 }
                     /* now sum the parents' flows */
                 VOL_OUT_OVL[basn[cel_i]] += Flux*CELL_SIZE; 
                 P_OUT_OVL[basn[cel_i]] += m3; 
                 S_OUT_OVL[basn[cel_i]] += m1; 
                 VOL_IN_OVL[basn[cel_j]]  += Flux*CELL_SIZE; 
                 P_IN_OVL[basn[cel_j]]  += m3; 
                 S_IN_OVL[basn[cel_j]]  += m1; 
                 
             }
             else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                 if  ( !basn_list[basn[cel_i]]->parent  ) { 
                     VOL_OUT_OVL[basn[cel_i]] += Flux*CELL_SIZE; 
                     P_OUT_OVL[basn[cel_i]] += m3; 
                     S_OUT_OVL[basn[cel_i]] += m1; 
                 }
                 if ( !basn_list[basn[cel_j]]->parent ) {                 
                     VOL_IN_OVL[basn[cel_j]]  += Flux*CELL_SIZE; 
                     P_IN_OVL[basn[cel_j]]  += m3; 
                     S_IN_OVL[basn[cel_j]]  += m1; 
                 }
             }
                    	
             if (debug > 1 && WatMgmt) { /* check for basin misconfiguration (allowable basin-basin flows) */
                 basins = basn_list[basn[cel_i]];
                 flo_chek = 0;
                 for (ii=0; ii<basins->numFLok; ii++) { if (basn[cel_j] == basins->FLok[ii] ) flo_chek = 1; }
                 if (!flo_chek) {
                     sprintf(msgStr,"Day %5.3f: ERROR - no (pos) SW flow from cell (%d,%d) of basin %d into cell (%d,%d) of basin %d!", 
                             TIME, i0,i1,basn[cel_i], j0,j1, basn[cel_j]); 
                     WriteMsg( msgStr,True );  }
             }
             

         }
         else { /* negative fluxes out of basn[cel_j] */
             if (basn_list[basn[cel_i]]->family !=  
                 basn_list[basn[cel_j]]->family ) {        /* if the flow is not within the family... */
                 if  ( !basn_list[basn[cel_j]]->parent  ) { /* and if the donor or recipient is a child... */
                     /* then find out about the flow for the family's sake */
                     VOL_OUT_OVL[basn_list[basn[cel_j]]->family] -= Flux*CELL_SIZE;
                     P_OUT_OVL[basn_list[basn[cel_j]]->family] -= m3;
                     S_OUT_OVL[basn_list[basn[cel_j]]->family] -= m1;
                 }
                 if ( !basn_list[basn[cel_i]]->parent ) {                 
                     /* then find out about the flow for the family's sake */
                     VOL_IN_OVL[basn_list[basn[cel_i]]->family] -= Flux*CELL_SIZE;
                     P_IN_OVL[basn_list[basn[cel_i]]->family] -= m3;
                     S_IN_OVL[basn_list[basn[cel_i]]->family] -= m1;
                 }
                     /* now sum the parents' flows */
                 VOL_OUT_OVL[basn[cel_j]] -= Flux*CELL_SIZE; 
                 P_OUT_OVL[basn[cel_j]] -= m3; 
                 S_OUT_OVL[basn[cel_j]] -= m1; 
                 VOL_IN_OVL[basn[cel_i]]  -= Flux*CELL_SIZE; 
                 P_IN_OVL[basn[cel_i]]  -= m3; 
                 S_IN_OVL[basn[cel_i]]  -= m1; 
                 
             }
             else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                 if  ( !basn_list[basn[cel_j]]->parent  ) { 
                     VOL_OUT_OVL[basn[cel_j]] -= Flux*CELL_SIZE; 
                     P_OUT_OVL[basn[cel_j]] -= m3; 
                     S_OUT_OVL[basn[cel_j]] -= m1; 
                 }
                 if ( !basn_list[basn[cel_i]]->parent ) {                 
                     VOL_IN_OVL[basn[cel_i]]  -= Flux*CELL_SIZE; 
                     P_IN_OVL[basn[cel_i]]  -= m3; 
                     S_IN_OVL[basn[cel_i]]  -= m1; 
                 }
             }


             if (debug > 1 && WatMgmt) { /* check for basin misconfiguration (allowable basin-basin flows) */
                 basins = basn_list[basn[cel_j]];
                 flo_chek = 0;
                 for (ii=0; ii<basins->numFLok; ii++) { if (basn[cel_i] == basins->FLok[ii] ) flo_chek = 1; }
                 if (!flo_chek) {
                     sprintf(msgStr,"Day %5.3f: ERROR - no (neg) SW flow from cell (%d,%d) of basin %d into cell (%d,%d) of basin %d!", 
                             TIME, i0,i1, basn[cel_j], j0,j1, basn[cel_i]); 
                     WriteMsg( msgStr,True );  }
             }
                 
             
         }
     }
     else  if ( !ON_MAP[cel_j]) /* so now the j,j cell is off-map, recipient if pos flow */
         if (Flux > 0) { 
             if ( !basn_list[basn[cel_i]]->parent ) { /* child's play */
                 VOL_OUT_OVL[basn_list[basn[cel_i]]->family] += Flux*CELL_SIZE;
                 P_OUT_OVL[basn_list[basn[cel_i]]->family] += m3;
                 S_OUT_OVL[basn_list[basn[cel_i]]->family] += m1;
             }
             /* parents' play */
             VOL_OUT_OVL[basn[cel_i]] += Flux*CELL_SIZE; 
             VOL_OUT_OVL[0]+= Flux*CELL_SIZE;
             P_OUT_OVL[basn[cel_i]] += m3; 
             P_OUT_OVL[0]+= m3;
             S_OUT_OVL[basn[cel_i]] += m1; 
             S_OUT_OVL[0]+= m1;
         }
         else { /* negative flows */
             if ( !basn_list[basn[cel_i]]->parent ) {/* child's play */
                 VOL_IN_OVL[basn_list[basn[cel_i]]->family] -= Flux*CELL_SIZE;
                 P_IN_OVL[basn_list[basn[cel_i]]->family] -= m3;
                 S_IN_OVL[basn_list[basn[cel_i]]->family] -= m1;
             }
             /* parents' play */
             VOL_IN_OVL[basn[cel_i]]  -= Flux*CELL_SIZE;
             VOL_IN_OVL[0]               -= Flux*CELL_SIZE;
             P_IN_OVL[basn[cel_i]]  -= m3;
             P_IN_OVL[0]               -= m3;
             S_IN_OVL[basn[cel_i]]  -= m1;
             S_IN_OVL[0]               -= m1;
         }
     else  if ( !ON_MAP[cel_i]) /* so now the i,i cell is off-map, donor if pos flow */
         if (Flux > 0) { 
             if ( !basn_list[basn[cel_j]]->parent ) {/* child's play */
                 VOL_IN_OVL[basn_list[basn[cel_j]]->family] += Flux*CELL_SIZE;
                 P_IN_OVL[basn_list[basn[cel_j]]->family] += m3;
                 S_IN_OVL[basn_list[basn[cel_j]]->family] += m1;
             }
             /* parents' play */
             VOL_IN_OVL[basn[cel_j]] += Flux*CELL_SIZE;
             VOL_IN_OVL[0]              += Flux*CELL_SIZE;
             P_IN_OVL[basn[cel_j]] += m3;
             P_IN_OVL[0]              += m3;
             S_IN_OVL[basn[cel_j]] += m1;
             S_IN_OVL[0]              += m1;
         }
         else { /*negative flows */
             if ( !basn_list[basn[cel_j]]->parent ) {/* child's play */
                 VOL_OUT_OVL[basn_list[basn[cel_j]]->family] -= Flux*CELL_SIZE;
                 P_OUT_OVL[basn_list[basn[cel_j]]->family] -= m3;
                 S_OUT_OVL[basn_list[basn[cel_j]]->family] -= m1;
             }
             /* parents' play */
             VOL_OUT_OVL[basn[cel_j]]  -= Flux*CELL_SIZE;
             VOL_OUT_OVL[0]               -= Flux*CELL_SIZE;
             P_OUT_OVL[basn[cel_j]]  -= m3;
             P_OUT_OVL[0]               -= m3;
             S_OUT_OVL[basn[cel_j]]  -= m1;
             S_OUT_OVL[0]               -= m1;
         }
 }
	       	
 return;
	
}

/*************************************************************************************************/
/* Groundwater fluxing routine
   This is the alternating direction function that first fluxes water in the E/W direction and then in the N/S 
   direction. It sweeps from left to right, then goes back from right to left, then goes from    
   top to bottom and finally from bottom to top. This alternates every hyd_iter.
   Water is fluxed with mass conservation, allowing outflow from (no inflow to) the model system.
   Order of constituents is salt, N, and P*/

#if NeedFunctionPrototypes
void Flux_GWater (int it, float *SatWat, float *Unsat, float *SfWat,
                  float *DOMdepth, float *rate, float *poros, float *sp_yield, float *elev,
                  double *gwSTUF1, double *gwSTUF2, double *gwSTUF3,
                  double *swSTUF1, double *swSTUF2, double *swSTUF3)
#else 
void Flux_GWater ( it, SatWat, Unsat, SfWat, DOMdepth, rate, poros, sp_yield, elev,
                   gwSTUF1, gwSTUF2, gwSTUF3, swSTUF1, swSTUF2, swSTUF3)
int it;
float *SatWat, *Unsat, *SfWat, *DOMdepth, *rate, *poros, *sp_yield, *elev;
double*gwSTUF1, *gwSTUF2, *gwSTUF3, *swSTUF1, *swSTUF2, *swSTUF3 ; 
#endif

{ int ix, iy, cell; /* ix is row, iy is col! */
/* we only check the donor cell for on-map, allowing losses from the system */
 for(ix=1; ix<=s0; ix++) 
 {
     if (it%2) {  /* alternate loop directions every other hyd_iter (it) */
     for(iy=1; iy<=s1; iy++)  /* loop from west to east */
             /* allow boundary flow if donor cell is marked (5 or 10) */
             if (( ON_MAP[T(ix,iy)] && ON_MAP[T(ix,iy+1)]) || 
                ((BCondFlow[T(ix,iy+1)]+1) % 5 == 0) || ((BCondFlow[T(ix,iy)]+1) % 5 == 0) ) 
         {
             Flux_GWcells(ix,iy,ix,iy+1,SatWat, Unsat, SfWat, rate, poros, sp_yield, elev,
                          gwSTUF1, gwSTUF2, gwSTUF3, swSTUF1, swSTUF2, swSTUF3); 
         }
     } 
             
     else { 
     for(iy=s1; iy>=1; iy--)   /* loop from east to west */
             /* allow boundary flow if donor cell is marked (5 or 10) */
         if (( ON_MAP[T(ix,iy)] && ON_MAP[T(ix,iy-1)]) || 
                ((BCondFlow[T(ix,iy-1)]+1) % 5 == 0) || ((BCondFlow[T(ix,iy)]+1) % 5 == 0) ) 
         {
         
             Flux_GWcells(ix,iy,ix,iy-1,SatWat, Unsat, SfWat, rate, poros, sp_yield, elev,
                          gwSTUF1, gwSTUF2, gwSTUF3, swSTUF1, swSTUF2, swSTUF3); 
         }
     } 
 }
	
 for(iy=1; iy<=s1; iy++) 
 {
     if (it%2) {  /* alternate loop directions every other hyd_iter (it) */
     for(ix=1; ix<=s0; ix++)  /* loop from north to south */
             /* allow boundary flow if donor cell is marked (5 or 10) */
         if (( ON_MAP[T(ix,iy)] && ON_MAP[T(ix+1,iy)]) || 
                ((BCondFlow[T(ix+1,iy)]+1) % 5 == 0) || ((BCondFlow[T(ix,iy)]+1) % 5 == 0) ) 
         {
         
             Flux_GWcells(ix,iy,ix+1,iy,SatWat, Unsat, SfWat, rate, poros, sp_yield, elev,
                          gwSTUF1, gwSTUF2, gwSTUF3, swSTUF1, swSTUF2, swSTUF3); 
         }
         } 
     else { 
     for(ix=s0; ix>=1; ix--)  /* loop from south to north */
             /* allow boundary flow if donor cell is marked (5 or 10) */
         if (( ON_MAP[T(ix,iy)] && ON_MAP[T(ix-1,iy)]) || 
                ((BCondFlow[T(ix-1,iy)]+1) % 5 == 0) || ((BCondFlow[T(ix,iy)]+1) % 5 == 0) ) 
         {
         
             Flux_GWcells(ix,iy,ix-1,iy,SatWat, Unsat, SfWat, rate, poros, sp_yield, elev,
                          gwSTUF1, gwSTUF2, gwSTUF3, swSTUF1, swSTUF2, swSTUF3); 
         }
     } 
 }

}

/************************************************************************************************/
/* Ground water flux eqns, incl. integration of groundwater with surface water.
   Application of Darcy's eqn to calculate flux between two adjacent cells (i0,i1) and (j0,j1)
   Flux is positive if flow is from i to j.
   Checks for available volume, and that flow cannot make the head in recepient cell higher
   than in the donor one.
   Integrates the vertical exchange among surface, unsaturated, and saturated water;
   updates all hydro and solute state variables.  
   Calcs the budget sums for water and solutes.  */

#if NeedFunctionPrototypes
void Flux_GWcells( int i0, int i1, int j0, int j1, float *SatWat, float *Unsat, float *SfWat, 
                    float *rate, float *poros, float *sp_yield, float *elev,
                    double *gwSTUF1, double *gwSTUF2, double *gwSTUF3,
                    double *swSTUF1, double *swSTUF2, double *swSTUF3)
#else 
void Flux_GWcells( i0, i1, j0, j1, SatWat, Unsat, SfWat, rate, poros, sp_yield, elev,
                    gwSTUF1, gwSTUF2, gwSTUF3, swSTUF1, swSTUF2, swSTUF3)
float *SatWat, *Unsat, *SfWat, *rate, *poros, *sp_yield, *elev;
double *gwSTUF1,*gwSTUF2, *gwSTUF3, *swSTUF1, *swSTUF2, *swSTUF3; 
int i0, i1, j0, j1;
#endif
{	
    int cell_don, cell_rec, i, j, sign, equilib, revers=1;
    float GW_head_i,GW_head_j,tot_head_i,tot_head_j;
    float dh, H_pot_don, H_pot_rec, AvgRate;
    
    float Flux, fluxTOunsat_don, fluxHoriz; /* vertical and horiz components of the calculated total groundwater Flux */
    float UnsatZ_don, UnsatCap_don, UnsatPot_don, Sat_vs_unsat; /* unsaturated zone capacities, potentials, in the donor cell */
    float UnsatZ_rec, UnsatCap_rec, UnsatPot_rec; /* unsaturated zone capacities, potentials, in the recipient cell */
    float sfTOsat_don, unsatTOsat_don, unsatTOsat_rec, satTOsf_rec, horizTOsat_rec; /* fluxes for vertical integration */
    
    float sedwat_don, sedwat_rec;
    double sed_stuf1fl_rec, sed_stuf2fl_rec, sed_stuf3fl_rec; /* vertical flux of solutes */
    double sf_stuf1fl_don, sf_stuf2fl_don, sf_stuf3fl_don; /* vertical flux of solutes */
    double sed_stuf1fl_horz, sed_stuf2fl_horz, sed_stuf3fl_horz; /* horiz flux of solutes */

    
    i = T(i0,i1); 
    j = T(j0,j1);
    
    if (ON_MAP[i] ) {
        GW_head_i = SatWat[i] / poros[HAB[i]];
        tot_head_i = GW_head_i + SfWat[i];
    }
    
    if (ON_MAP[j] ) {
        GW_head_j = SatWat[j] / poros[HAB[j]];
        tot_head_j = GW_head_j + SfWat[j];
    }
    

    if (!ON_MAP[i] ) { /* for an external i0,i1 cell */
        poros[HAB[i]] = poros[HAB[j]]; /* other variables = donor cell or zero */
        sp_yield[HAB[i]] = sp_yield[HAB[j]]; 
        elev[i] = elev[j];
        rate[i] = rate[j];
        Unsat[i] = 0.0; /* no water in any potential unsat zone */
        /* estimate of the external sat water height: if the donor
         head is above a threshold value, then
         allow grroundwater flux out of system.  If the donor cell is
         pretty dry, then no flux */
            
            /* outflow of groundwater from boundary cell*/
        if (BCondFlow[j] == 4) { 
            if (tot_head_j > (elev[j] + 0.2 )) /* 0.2 was too low?, no appears too high (not enough water loss)*/
            {
                SatWat[i] = SatWat[j] ;    /* was sat[j] - 0.0 in all runs up to ELM2 southseep; then was sat-0.1 thru c27up */
                SfWat[i] = Max(SfWat[j] - 0.3, 0.0); /* was = 0.35 earlier; then was sf-0.1 thru c27up*/
            }
        else
            {
                SatWat[i] = SatWat[j] - 0.05; /* was SatWat[j]-0.0 in all runs up to ELM2 southseep; then was sat-0.05 thru c27up, then 0.1 thru ELM2 hydCond*/
                SfWat[i] = 0.0; /* Max(SfWat[j] - 0.05, 0.0) */; /* was SfWat[j]-0.0 in all runs up to ELM2 southseep, then -0.05 thru c27up */
            }
        }
            /* inflow of groundwater to boundary cell */
        if (BCondFlow[j] == 9) { /* this is intended for static BCond for CALM, the Hillsboro Can */
            /* give the groundwater cells outside the model a conc of constituents */
            gwSTUF1[i] = SatWat[j]*0.0e1; /* m * m^2 * kg/m^3 = kg */
            gwSTUF2[i] = SatWat[j]*0.0e-5;
            gwSTUF3[i] = SatWat[j]*1.0e-5;

            if (tot_head_j > (elev[j]  )) 
            {
                SatWat[i] = elev[j]*poros[HAB[j]];
                SfWat[i] = SfWat[j] + 0.05; /* change from 0.1, oct27*/ /* lower gwat inflow when internal cell wet */
            }
        else
            {
                SatWat[i] = elev[j]*poros[HAB[j]];
                SfWat[i] = SfWat[j] + 0.1; /* change from 0.2, oct27*/ /* higher gwat inflow when internal cell dry */
            }
        }
            
    GW_head_i = SatWat[i] / poros[HAB[i]];
    tot_head_i = GW_head_i + SfWat[i];
    }

    if (!ON_MAP[j] ) { /* for an external j0,j1 cell */
        poros[HAB[j]] = poros[HAB[i]]; /* other variables = donor cell or zero */
        sp_yield[HAB[j]] = sp_yield[HAB[i]]; 
        elev[j] = elev[i];
        rate[j] = rate[i];
        Unsat[j] = 0.0; /* no water in any potential unsat zone */
        /* estimate of the external sat water height: if the donor
         head is above a threshold value, then
         allow grroundwater flux out of system.  If the donor cell is
         pretty dry, then no flux */
            
            /* outflow of groundwater from boundary cell*/
        if (BCondFlow[i] == 4) { 
            if (tot_head_i > (elev[i] + 0.2 )) /* 0.2 was too low?, no appears too high (not enough water loss)*/
            {
                SatWat[j] = SatWat[i];    /* was sat[i] - 0.1 */
                SfWat[j] = Max(SfWat[i] - 0.3, 0.0); /* was = 0.35 */
            }
        else
            {
                SatWat[j] = SatWat[i] - 0.05;
                SfWat[j] = 0.0;
            }
        }
            /* inflow of groundwater to boundary cell */
        if (BCondFlow[i] == 9) { /* this is intended for static BCond for CALM, the Hillsboro Can */
            /* give the groundwater cells outside the model a conc of constituents */
            gwSTUF1[j] = SatWat[i]*0.0e1; /* m * m^2 * kg/m^3 = kg */
            gwSTUF2[j] = SatWat[i]*0.0e-5;
            gwSTUF3[j] = SatWat[i]*1.0e-5;

            if (tot_head_i > (elev[i]  )) 
            {
                SatWat[j] = elev[i]*poros[HAB[i]];
                SfWat[j] = SfWat[i] + 0.05; /* change from 0.1, oct27*/ /* lower gwat inflow when internal cell wet */
            }
        else
            {
                SatWat[j] = elev[i]*poros[HAB[i]];
                SfWat[j] = SfWat[i] + 0.1; /* change from 0.2, oct27*/ /* higher gwat inflow when internal cell dry */
            }
        }
            
    GW_head_j = SatWat[j] / poros[HAB[j]];
    tot_head_j = GW_head_j + SfWat[j];
    }

    AvgRate = ( rate[i] + rate[j] )/2.0; /* average hyd conductivity of the two cells */
    
/* hydraulic head difference, positive flux from i to j */
    dh = tot_head_i - tot_head_j; 

 
        /* determine donor and recipient cells based on head difference,
           and provide the sign of the flux;
           The surface water detention depth also used here as threshold
           head difference for fluxing (currently global, 1 cm)*/
    if (dh > DetentZ) 
    {
    	cell_don=i;
    	cell_rec=j;
        sign = 1.0; 
    }
    else if (dh < -DetentZ) 
    {	
    	cell_don=j;
    	cell_rec=i;
        sign = -1.0;
    }
    else 
        return; /* no flux (or surface-groundwater integration) under minimal head diffs */
    

/* Potential horizontal flux eqn (Darcy's eqn simplified to square cells).
   This is the maximum height (m) of water vol to flux under fully saturated condition.
   (alpha3 is the third "flux_parm" read from the Driver.parm file (a calib parm, normally 1.0)).
   Note that the Min check is not important under current implementations, as SatWat includes water
   down to the base datum, which is currently 6 meters below sea level (actually, 1929 NGVD) */
    Flux = Min(Abs(dh) * alpha3 * AvgRate * SatWat[cell_don] / CELL_SIZE * gwstep , SatWat[cell_don]);
 
/**** this do-while routine (1) integrates the surface, saturated, and unsaturated water,
  and (2) checks to ensure the heads do not reverse in a time step due to large fluxes.
  If heads do reverse, the total Flux is decremented until there is no reversal */
/****/
    do {
        /* The total potential flux is apportioned to (1) the horizontal component
           that fluxes to an adjacent cell and (2) the vertical component that
           remains in the donor cell after the horizontal outflow from a donor cell.
           Thus, an unsaturated zone is created, or increased in size, with loss of
           saturated water from the donor cell; this lateral gravitational flow leaves behind
           the field capacity moisture in an unsat zone. (If donor-cell surface water is present,
           it potentially will replace the unsaturated soil capacity within the same time
           step in this routine).*/
        fluxTOunsat_don = Flux * (poros[HAB[cell_don]] - sp_yield[HAB[cell_don]])  ;
        fluxHoriz = Flux - fluxTOunsat_don;

/**** given the total potential groundwater Flux, get the donor cell's
      new **post-flux** capacities */
        UnsatZ_don  =   elev[cell_don] - (SatWat[cell_don]-Flux) / poros[HAB[cell_don]] ; /* unsat zone depth */
        if (debug>2 && (UnsatZ_don < -0.001 ) ) {
            sprintf(msgStr,"Day %6.1f: capacityERR - neg unsat depth (%f m) in donor cell (%d,%d) in GWfluxes!", 
                    TIME, UnsatZ_don, i0,i1 ); WriteMsg( msgStr,True );}
            
        UnsatCap_don =  UnsatZ_don * poros[HAB[cell_don]]; /* maximum pore space capacity (UnsatCap)
                                                              in the depth (UnsatZ) of new unsaturated zone */
        UnsatPot_don  = UnsatCap_don - (Unsat[cell_don]+fluxTOunsat_don); /* (height of) the volume of pore space
                                                                             (soil "removed") that is unoccupied in the unsat zone */
            /* determining the pathway of flow (to sat vs. unsat) of surface water depending on depth
               of an unsat zone relative to the surface water.  With a relatively deep unsat zone, this
               downflow tends to zero (infiltration occurs within the vertical hydrology module of UnitMod.c) */ 
        Sat_vs_unsat  = 1/Exp(100.0*Max((SfWat[cell_don]-UnsatZ_don),0.0)); 

    /* sf-unsat-sat fluxes */
        if (SfWat[cell_don] > 0.0) { /* surface water downflow is assumed to be as fast
                                        as horizontal groundwater outflows */
            sfTOsat_don  = 
                ( (1.0-Sat_vs_unsat)*UnsatPot_don>SfWat[cell_don] ) ? 
                ( SfWat[cell_don] ) : 
                ( (1.0-Sat_vs_unsat)*UnsatPot_don); 
           /* with downflow of surface water into an unsat zone, the proportion of that height
              that is made into saturated storage is allocated to the sat storage variable */
            if (sfTOsat_don >=  UnsatPot_don) {
                    sfTOsat_don = UnsatPot_don ; /* downflow constrained to the avail soil capacity */
                    unsatTOsat_don = Unsat[cell_don]; /* allocate unsat to sat in this case */
                }
            else { 
                unsatTOsat_don = (UnsatZ_don > 0.0) ?
                    ( (sfTOsat_don/poros[HAB[cell_don]] ) / UnsatZ_don * Unsat[cell_don] ) :
                    (0.0); /*  allocate to saturated storage whatever proportion of
                               unsat zone that is now saturated by sfwat downflow */
            }
        }
        else { /* w/o surface water, these vertical fluxes are zero */
            sfTOsat_don = unsatTOsat_don = 0.0;
        }
/**** potential new head in donor cell will be ... */
        H_pot_don = (SatWat[cell_don] - fluxTOunsat_don - fluxHoriz + sfTOsat_don + unsatTOsat_don )
            / poros[HAB[cell_don]] +(SfWat[cell_don] - sfTOsat_don) ;
                

/**** recipient cell's **pre-flux** capacities */
        UnsatZ_rec  =   elev[cell_rec] - SatWat[cell_rec] / poros[HAB[cell_rec]] ; /* unsat zone depth */
        if (debug>2 && (UnsatZ_rec < -0.001 ) ) {
            sprintf(msgStr,"Day %6.1f: capacityERR - neg unsat depth (%f m) in recipient cell (%d,%d) in GWfluxes!", 
                    TIME, UnsatZ_rec, i0,i1 ); WriteMsg( msgStr,True ); }
            
        UnsatCap_rec =  UnsatZ_rec * poros[HAB[cell_rec]]; /* maximum pore space capacity (UnsatCap)
                                                              in the depth (UnsatZ) of new unsaturated zone */
        UnsatPot_rec  = UnsatCap_rec - Unsat[cell_rec]; /* (height of) the volume of pore space (soil "removed")
                                                           that is unoccupied in the unsat zone */
     /* sf-unsat-sat fluxes */
        horizTOsat_rec = fluxHoriz; /* lateral inflow to soil into saturated storage */
        satTOsf_rec = Max(fluxHoriz - UnsatPot_rec, 0.0); /* upwelling beyond soil capacity */
        unsatTOsat_rec = (UnsatZ_rec > 0.0) ? /* incorporation of unsat moisture into sat storage with
                                                 rising water table due to horiz inflow */
            ( ((horizTOsat_rec-satTOsf_rec)/poros[HAB[cell_rec]] ) / UnsatZ_rec * Unsat[cell_rec] ) :
            (0.0);
/**** potential new head in recipient cell will be ... */
        H_pot_rec = (SatWat[cell_rec] + horizTOsat_rec + unsatTOsat_rec - satTOsf_rec)
            / poros[HAB[cell_rec]] + (SfWat[cell_rec] + satTOsf_rec) ;

/**** check for a head reversal - if so, reduce flux to achieve equilibrium (donor >= recip) */
        if ( (Flux != 0.0) && ((H_pot_rec - H_pot_don) > MinCheck) ) {
            revers++;
            Flux *= 0.90;
            equilib = 0; 
               /* if the unsat water storage causes an unfixable (very rare(/never?)) head reversal due to the
                   sfwat and gwater integration alone, set the Flux to zero, then allow the vertical
                   sf/ground integration to be done with no horiz flux and be done with it */
            if (revers>4) {
                    /*tmp debug level=0 */
                if (debug>0) { sprintf(msgStr,"Day %6.1f: FYI - head reversal (%f m) due to surf/ground integration, associated with cell (%d,%d).  Total flux was to be %f. Fixed with 0 flux. ", 
                                       TIME, (H_pot_don - H_pot_rec),  i0,i1,Flux*sign ); WriteMsg( msgStr,True );}
                Flux =  0.0;
                }
        }
        else {
            equilib = 1;
        }
        
    } while  (!equilib); /* end of routine for integrating groundwater-surface water and preventing head reversals */
        
/**********
  finished calc'ing the water fluxes
  *************/
/* calc the flux of the solutes between sed/soil across cells in horiz dir,
 but don't update the state vars until the conc's for vertical flows have been calc'd */
    sedwat_don = SatWat[cell_don] + Unsat[cell_don];
    sed_stuf1fl_horz = (sedwat_don > 0.0) ? (gwSTUF1[cell_don] / sedwat_don*fluxHoriz) : (0.0);
    sed_stuf2fl_horz = (sedwat_don > 0.0) ? (gwSTUF2[cell_don] / sedwat_don*fluxHoriz) : (0.0);
    sed_stuf3fl_horz = (sedwat_don > 0.0) ? (gwSTUF3[cell_don] / sedwat_don*fluxHoriz) : (0.0);

/* pass along the solutes between sed/soil and surface water in vertical dir;
   for "simplicity", this does not account for the phosphorus active-zone conc. gradient,
   but assumes the homogeneity of conc within the entire soil column */
    if (sfTOsat_don > 0.0) {
        sf_stuf1fl_don = (SfWat[cell_don] > 0.0) ? (swSTUF1[cell_don] / SfWat[cell_don]*sfTOsat_don) : (0.0);
        sf_stuf2fl_don = (SfWat[cell_don] > 0.0) ? (swSTUF2[cell_don] / SfWat[cell_don]*sfTOsat_don) : (0.0);
        sf_stuf3fl_don = (SfWat[cell_don] > 0.0) ? (swSTUF3[cell_don] / SfWat[cell_don]*sfTOsat_don) : (0.0);
        gwSTUF1[cell_don] += sf_stuf1fl_don;
        gwSTUF2[cell_don] += sf_stuf2fl_don;
        gwSTUF3[cell_don] += sf_stuf3fl_don;
        swSTUF1[cell_don] -= sf_stuf1fl_don;
        swSTUF2[cell_don] -= sf_stuf2fl_don;
        swSTUF3[cell_don] -= sf_stuf3fl_don;
        
    }
    if (satTOsf_rec > 0.0) {
        sedwat_rec = SatWat[cell_rec] + Unsat[cell_rec];
        sed_stuf1fl_rec = (sedwat_rec > 0.0) ? (gwSTUF1[cell_rec] / sedwat_rec*satTOsf_rec) : (0.0);
        sed_stuf2fl_rec = (sedwat_rec > 0.0) ? (gwSTUF2[cell_rec] / sedwat_rec*satTOsf_rec) : (0.0);
        sed_stuf3fl_rec = (sedwat_rec > 0.0) ? (gwSTUF3[cell_rec] / sedwat_rec*satTOsf_rec) : (0.0);
        gwSTUF1[cell_rec] -= sed_stuf1fl_rec;
        gwSTUF2[cell_rec] -= sed_stuf2fl_rec;
        gwSTUF3[cell_rec] -= sed_stuf3fl_rec;
        swSTUF1[cell_rec] += sed_stuf1fl_rec;
        swSTUF2[cell_rec] += sed_stuf2fl_rec;
        swSTUF3[cell_rec] += sed_stuf3fl_rec;
    }
    
/* update the groundwater solutes due to horiz flows */
    gwSTUF1[cell_don] -= sed_stuf1fl_horz;
    gwSTUF2[cell_don] -= sed_stuf2fl_horz;
    gwSTUF3[cell_don] -= sed_stuf3fl_horz;
    gwSTUF1[cell_rec] += sed_stuf1fl_horz;
    gwSTUF2[cell_rec] += sed_stuf2fl_horz;
    gwSTUF3[cell_rec] += sed_stuf3fl_horz;
        

/* update the hydro state variables */
    SfWat[cell_don]  += (-sfTOsat_don);
    Unsat[cell_don]  += ( fluxTOunsat_don - unsatTOsat_don) ;
    SatWat[cell_don] += (sfTOsat_don + unsatTOsat_don - fluxTOunsat_don - fluxHoriz);
    SfWat[cell_rec]  += ( satTOsf_rec); 
    Unsat[cell_rec]  += (-unsatTOsat_rec);
    SatWat[cell_rec] += (horizTOsat_rec + unsatTOsat_rec - satTOsf_rec); /* (horizTOsat_rec + satTOsf_rec) = fluxHoriz */
        
/*******BUDGET
**************/
/*  sum the flow of groundwater and solutes (nutrients, salinity, etc) among basins, with budget calcs.*/
    if ( basn[i] != basn[j] ) { 

        fluxHoriz = sign*fluxHoriz*CELL_SIZE; /* signed water flux volume (m^3) */
        sed_stuf1fl_horz = sign*sed_stuf1fl_horz; /* signed solute flux (kg) */
        sed_stuf3fl_horz = sign*sed_stuf3fl_horz; /* signed solute flux (kg) */

	/* first do the normal case where all cells are on-map */
        if ( ON_MAP[j] && ON_MAP[i] ) { 
            if ( fluxHoriz > 0  ) { /* fluxes out of basn[i] */
             if (basn_list[basn[i]]->family !=  
                 basn_list[basn[j]]->family ) {        /* if the flow is not within the family... */
                 if  ( !basn_list[basn[i]]->parent  ) { /* and if the donor or recipient is a child... */
                     /* then find out about the flow for the family's sake */
                     VOL_OUT_GW[basn_list[basn[i]]->family] += fluxHoriz;
                     P_OUT_GW[basn_list[basn[i]]->family] += sed_stuf3fl_horz;
                     S_OUT_GW[basn_list[basn[i]]->family] += sed_stuf1fl_horz;
                 }
                 if ( !basn_list[basn[j]]->parent ) {                 
                     /* then find out about the flow for the family's sake */
                     VOL_IN_GW[basn_list[basn[j]]->family] += fluxHoriz;
                     P_IN_GW[basn_list[basn[j]]->family] += sed_stuf3fl_horz;
                     S_IN_GW[basn_list[basn[j]]->family] += sed_stuf1fl_horz;
                 }
                     /* now sum the parents' flows */
                 VOL_OUT_GW[basn[i]] += fluxHoriz;
                 VOL_IN_GW[basn[j]]  += fluxHoriz;
                 P_OUT_GW[basn[i]] += sed_stuf3fl_horz;
                 P_IN_GW[basn[j]]  += sed_stuf3fl_horz;
                 S_OUT_GW[basn[i]] += sed_stuf1fl_horz;
                 S_IN_GW[basn[j]]  += sed_stuf1fl_horz;
             }
             
                    	
             else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                 if  ( !basn_list[basn[i]]->parent  ) { 
                     VOL_OUT_GW[basn[i]] += fluxHoriz; 
                     P_OUT_GW[basn[i]] += sed_stuf3fl_horz; 
                     S_OUT_GW[basn[i]] += sed_stuf1fl_horz; 
                 }
                 if ( !basn_list[basn[j]]->parent ) {                 
                     VOL_IN_GW[basn[j]]  += fluxHoriz; 
                     P_IN_GW[basn[j]]  += sed_stuf3fl_horz; 
                     S_IN_GW[basn[j]]  += sed_stuf1fl_horz; 
                 }
             }

            }
            else { /* negative fluxes out of basn[j] */
             if (basn_list[basn[i]]->family !=  
                 basn_list[basn[j]]->family ) {        /* if the flow is not within the family... */
                 if  ( !basn_list[basn[j]]->parent  ) { /* and if the donor or recipient is a child... */
                     /* then find out about the flow for the family's sake */
                     VOL_OUT_GW[basn_list[basn[j]]->family] -= fluxHoriz;
                     P_OUT_GW[basn_list[basn[j]]->family] -= sed_stuf3fl_horz;
                     S_OUT_GW[basn_list[basn[j]]->family] -= sed_stuf1fl_horz;
                 }
                 if ( !basn_list[basn[i]]->parent ) {                 
                     /* then find out about the flow for the family's sake */
                     VOL_IN_GW[basn_list[basn[i]]->family] -= fluxHoriz;
                     P_IN_GW[basn_list[basn[i]]->family] -= sed_stuf3fl_horz;
                     S_IN_GW[basn_list[basn[i]]->family] -= sed_stuf1fl_horz;
                 }
                     /* now sum the parents' flows */
                VOL_OUT_GW[basn[j]] -= fluxHoriz;
                VOL_IN_GW[basn[i]]  -= fluxHoriz;
                P_OUT_GW[basn[j]] -= sed_stuf3fl_horz;
                P_IN_GW[basn[i]]  -= sed_stuf3fl_horz;
                S_OUT_GW[basn[j]] -= sed_stuf1fl_horz;
                S_IN_GW[basn[i]]  -= sed_stuf1fl_horz;

            }
             else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                 if  ( !basn_list[basn[j]]->parent  ) { 
                     VOL_OUT_GW[basn[j]] -= fluxHoriz; 
                     P_OUT_GW[basn[j]] -= sed_stuf3fl_horz; 
                     S_OUT_GW[basn[j]] -= sed_stuf1fl_horz; 
                 }
                 if ( !basn_list[basn[i]]->parent ) {                 
                     VOL_IN_GW[basn[i]]  -= fluxHoriz; 
                     P_IN_GW[basn[i]]  -= sed_stuf3fl_horz; 
                     S_IN_GW[basn[i]]  -= sed_stuf1fl_horz; 
                 }
             }
            }
            

        }
        else  if ( !ON_MAP[j]) {/* so now the j,j cell is off-map, recipient if pos flow */
            if (fluxHoriz > 0) { 
             if ( !basn_list[basn[i]]->parent ) { /* child's play */
                 VOL_OUT_GW[basn_list[basn[i]]->family] += fluxHoriz;
                 P_OUT_GW[basn_list[basn[i]]->family] += sed_stuf3fl_horz;
                 S_OUT_GW[basn_list[basn[i]]->family] += sed_stuf1fl_horz;
             }
             /* parents' play */
                VOL_OUT_GW[basn[i]] += fluxHoriz;
                VOL_OUT_GW[0]              += fluxHoriz;
                P_OUT_GW[basn[i]] += sed_stuf3fl_horz;
                P_OUT_GW[0]              += sed_stuf3fl_horz;
                S_OUT_GW[basn[i]] += sed_stuf1fl_horz;
                S_OUT_GW[0]              += sed_stuf1fl_horz;
            }
            else {/* negative flows */
             if ( !basn_list[basn[i]]->parent ) {/* child's play */
                 VOL_IN_GW[basn_list[basn[i]]->family] -= fluxHoriz;
                 P_IN_GW[basn_list[basn[i]]->family] -= sed_stuf3fl_horz;
                 S_IN_GW[basn_list[basn[i]]->family] -= sed_stuf1fl_horz;
             }
             /* parents' play */
                VOL_IN_GW[basn[i]]  -= fluxHoriz;
                VOL_IN_GW[0]               -= fluxHoriz;
                P_IN_GW[basn[i]]  -= sed_stuf3fl_horz;
                P_IN_GW[0]               -= sed_stuf3fl_horz;
                S_IN_GW[basn[i]]  -= sed_stuf1fl_horz;
                S_IN_GW[0]               -= sed_stuf1fl_horz;
            }
        }
        
        else  if ( !ON_MAP[i]) { /* so now the i,i cell is off-map, donor if pos flow */
            if (fluxHoriz > 0) { 
             if ( !basn_list[basn[j]]->parent ) {/* child's play */
                 VOL_IN_GW[basn_list[basn[j]]->family] += fluxHoriz;
                 P_IN_GW[basn_list[basn[j]]->family] += sed_stuf3fl_horz;
                 S_IN_GW[basn_list[basn[j]]->family] += sed_stuf1fl_horz;
             }
             /* parents' play */
                VOL_IN_GW[basn[j]] += fluxHoriz;
                VOL_IN_GW[0]              += fluxHoriz;
                P_IN_GW[basn[j]] += sed_stuf3fl_horz;
                P_IN_GW[0]              += sed_stuf3fl_horz;
                S_IN_GW[basn[j]] += sed_stuf1fl_horz;
                S_IN_GW[0]              += sed_stuf1fl_horz;
            }
            else {/*negative flows */
             if ( !basn_list[basn[j]]->parent ) {/* child's play */
                 VOL_OUT_GW[basn_list[basn[j]]->family] -= fluxHoriz;
                 P_OUT_GW[basn_list[basn[j]]->family] -= sed_stuf3fl_horz;
                 S_OUT_GW[basn_list[basn[j]]->family] -= sed_stuf1fl_horz;
             }
             /* parents' play */
                VOL_OUT_GW[basn[j]]  -= fluxHoriz;
                VOL_OUT_GW[0]               -= fluxHoriz;
                P_OUT_GW[basn[j]]  -= sed_stuf3fl_horz;
                P_OUT_GW[0]               -= sed_stuf3fl_horz;
                S_OUT_GW[basn[j]]  -= sed_stuf1fl_horz;
                S_OUT_GW[0]               -= sed_stuf1fl_horz;
            }
    }
}


    return ; 
        
}

  
/***************************************************************************************/
/* Read in daily stage data for cells external to ON_MAP model cells.
   The source may be output from SFWMM, ELM, or other sources (e.g., historical).
*/

#if NeedFunctionPrototypes
void ReadBCond  (float baseDatum )
#else 
void ReadBCond  ( baseDatum )
float baseDatum;
#endif

{
    struct BCmodel *BCmap;
    
    int src_row, src_col;
    int ix, iy;
    
    int j = 0, i = 0;
    int lincnt; /* lincnt is input file record # starting at the point where the model reads values */
    double Jdate_read;
    char lline[2001], *line;
    int yyyy, mm, dd;
    char varNam[20], scenario[20];
    

/* temp */ if ( (garb = fopen("/vol/hy_ext/SME/garbout", "w") ) == NULL) { printf("exit now"); exit(1); } 
 
        /* load the structure that defines the model grid cells' relationship to the data source grid cells */
   for(ix=1; ix<=s0; ix++) /* loop from north to south */
        for(iy=1; iy<=s1; iy++)  /* loop from west to east */
            if ( BCondFlow[T(ix,iy)] >0 ) {
                if (!ON_MAP[T(ix,iy+1)] ) {
                    BCondFlow[T(ix,iy+1)] = 5;
                }
                
                    /* if (!ON_MAP[T(ix,iy-1)] ) {
                    BCondFlow[T(ix,iy-1)] = 5;
                    } */
                
                if (!ON_MAP[T(ix+1,iy)] ) {
                    BCondFlow[T(ix+1,iy)] = 5;
                }
                
                    /*   if (!ON_MAP[T(ix-1,iy)] ) {
                    BCondFlow[T(ix-1,iy)] = 5;
                    } */
                i++;
                
            }
                
/*                  if ( (BCcells[T(ix,iy)].mod_row = (struct BCgrid *) malloc( (size_t) sizeof( struct BCgrid ))) == NULL )  */
/*                  {  */
/*                      printf( "No memory for model boundary condition structure at %d, %d\n ", ix,iy ) ;  */
/*                      exit( -2 ) ;  */
/*                  }  */
/*                 BCcells[i].mod_row = ix; */
/*                 BCcells[i].mod_col = iy; */
/*                 BCcells[i].mod_ID = T(ix,iy); */
/*                 BCcells[i].src_row = 5; */
/*                 BCcells[i].src_col = 5; */
               /*  BCcells[T(ix,iy)]->src_ID = j++; */
/*                 i++; */
                
            
 
                 

     
/* **** */
/* read external boundary condition stage time series data */
    usrErr0 ( "\tExternal stage heights time series...");


    
    if ( SMP )
    { if(H_OPSYS==UNIX) 
        sprintf( mapFileName, "%s/%s/Data/BoundCond_stage", ModelPath, ProjName );
    else sprintf( mapFileName, "%s%s:Data:BoundCond_stage", ModelPath, ProjName );
    }
    else  sprintf( mapFileName,"BoundCond_stage" );

/* Open file with external stage data */
    if ( ( F_extStage = fopen( mapFileName, "r" ) ) ==  NULL )
    {
        printf( "Can't open external stage data input file! " ) ;
        exit(-1) ;
    }


    fgets(lline, 2000, F_extStage); /* first header line with the restudy alternative & number of cells to read */
    line=lline;
    sscanf(line, "%s",&scenario);
    if (strcmp(scenario,SimAlt) != 0) {
        sprintf(msgStr, "The scenario (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!", scenario, mapFileName, SimAlt); usrErr(msgStr);
        exit(-1);
    }
    line = Scip( line, '=' ); /* data formatted with this preceding the source (model or observed) data that is being read */
    *line = sscanf ( line, "%s", &BCsrc_data );
     
    line = Scip( line, '=' ); /* data formatted with this preceding the # of cells to process in the source data */
    *line = sscanf ( line, "%d", &BCnumCells );

    fgets(lline, 2000, F_extStage); /* read 2'nd header line with column names */
    line=lline;
    line = Scip( line, TAB );
    line = Scip( line, TAB );

/* now we're past the three date columns, ready to read the variable names */
    for ( j = 0; j < BCnumCells; j++ ) { /* loop through all the data columns */
        line = Scip( line, TAB );
        sscanf ( line,"%d,%d\t",&src_row,&src_col); /* read the cell row,col IDs of the source data in the header */

        BCdata[j].arrayStage = (float*) nalloc( (Jdate_end-Jdate_init+3)*sizeof(float), "timeseries temp" );

        BCdata[j].src_row = src_row;
        BCdata[j].src_col = src_col;

            /* determine which model grid cells are included in the current source grid cell */
        if (strcmp(BCsrc_data, "SFWMM") == 0) {
                /* grid data from sfwmm output */
            j=j;
            
        }
        else if (strcmp(BCsrc_data, "ELM") == 0) {
                /* grid data from elm output, or other observations on elm grid */
            printf ("\n\nERROR - not set up to use %s as source of the grid data for boundary conditions.\n", BCsrc_data);
            exit(-1);
        }
        else {
            printf ("\n\nERROR - do not recognize the source (%s) of the grid data for boundary conditions.\n", BCsrc_data);
            exit(-1);
        }
        
                
        

    }

    lincnt = 0; /*  input file record # starting at the point where the model reads values */

    while ( fgets(lline, 2000, F_extStage) != NULL && !feof( F_extStage ) )
    {
        line = lline;
        sscanf( line, "%d %d %d",&yyyy,&mm,&dd); /* read the date of the current record */
         
            /* julian day, returns a double (args = mon, day, yr, hrs,min,secs) */
        Jdate_read = julday( mm,dd,yyyy,0,0,0.0 ); 
        if ((lincnt == 0) && (Jdate_read > Jdate_init) ) {
            printf (" \n***Error\nExternal BC stage (%s) init date > simulation start date\n", mapFileName); exit (-1);
        }
        if ((Jdate_read >= Jdate_init) && (Jdate_read <= Jdate_end)) {
       /*     fprintf(garb,"%4d\t%2d\t%2d",yyyy,mm,dd); temp */
            line = Scip( line, TAB );
            line = Scip( line, TAB );
            for ( j = 0; j < BCnumCells; j++ ) { /* loop through all the control structure columns in the input file */
                line = Scip( line, TAB );
                sscanf( line, "%f", &BCdata[j].arrayStage[((int)(lincnt))] );
                    /* data read in ft NGVD '27, convert to m NGVD '27 plus the ELM base datum below NGVD*/
                BCdata[j].arrayStage[((int)(lincnt))] = BCdata[j].arrayStage[((int)(lincnt))] * 0.3048 + baseDatum; 
              /*  fprintf(garb,"\t%7.2f",BCdata[j].arrayStage[((int)(lincnt))]); temp */
            }
              
            lincnt++; /* increment the number of daily data records */
        /*    fprintf(garb,"\n"); temp */ 
             
        }

    } /* end of reading external stage data records */
/*    fclose(garb); temp */   

    if (lincnt == 0 ) {
        printf (" \n***Error\nExternal stage time series data (%s) has date problem (0 records read)\n", mapFileName); exit (-1);
    }
    else if (lincnt != (Jdate_end-Jdate_init+1) ) {
        printf (" \n***Error\nExternal stage time series data (%s) has problem (%d records read, %d expected)\n", mapFileName, lincnt,(int)(Jdate_end-Jdate_init+1)); exit (-1);
    }
    sprintf (msgStr, "done, found %d records.",lincnt); usrErr(msgStr);
     
    fclose (F_extStage);
    return;
     
 
}  /* end of reading external boundary condition stage time series data (if needed) */

/***************************************************************************************/
