/* Unit model, vertical solutions, for the Everglades Landscape Model (ELM),
                  the Conservation Area Landscape Model (CALM),
                  and other implementations as needed.

   version 2.1a, July 2002
*/

/*###############
To change between ELM and CALM or other scales: 
	no code changes here or elsewhere -
        just provide different data:
        1) input maps (diff scale/loc),
        2) rainfall (currently diff functions and files used for data),
        3) meteorological (# of CALM files just subset of ELM ),
        4) CanalData.* (CALM files just subset of ELM),
        5) HabSwData (CALM file just subset of ELM), and 
        6) Driver.parm files (diff paths, model name, etc).
###############*/

/* 
This source file contains the vertical solutions to ELM dynamics, calls spatial (horiz) flows,
and calls the succession routine.
Major functions/sections located in this file in the order of: 
  1) declarations, incl. data in misc graph functions, 
  2) vertical dynamic functions/modules (execution sequence determined by user at runtime)
     cell_dyn1 (global (non-spatial) calcs, interpolaters, habswitch call)
     cell_dyn2 (algae/periphyton (2 communities)
     cell_dyn3 (consumers - **unused**)
     cell_dyn4 (deposited organic matter and deposited organic phosphorus)
     cell_dyn5 (nitrogen in surface & pore water - **unused**)
     cell_dyn6 (fire - **unused**)
     horizFlow (calling function for horizontal fluxes)
     cell_dyn7 (vertical hydrology)
     cell_dyn8 (macrophytes)
     cell_dyn9 (phosphorus in surface & pore water and sorbed to soil)
     cell_dyn13 (phosphorus settling rate module - "replicate" Everglades Water Quality Model")
     cell_dyn10 (salt or conservative tracer)
     cell_dyn11 (standing dead detritus)
     cell_dyn12 (floc)
  3) call_cell_dyn (calling function for cell_dynxx's)
  4) init_vars (initialization routines)
  5) gen_output (output calling routine)
  6) ReadParms (read global parameters)
  7) alloc_memory (array memory allocation calling routines)
  8) misc graph functions read/interpolate (graphX)

  The sequence of execution of these functions is determined by calling sequence in Generic_Driver.c, using
data input by the user at runtime.

There are several other source files with dynamic calculations:
	BudgStats.c	budget and other summary stats (whole-map and basin-specific)
	WatMgmt.c	canal<->cell, water control structure flows
	Fluxes.c	cell<->cell flows
	Success.c	habitat switching 
	Fire.c	        fire propogation (not used now)

There are >17 source files total in the project, depending on the platform being used 

Detail of output determined by "debug" parm in Driver.parm file:
	debug >0 prints mass balance info to water, P, and S files 
	debug >1 print struct flows & concs, and canal depths & concs to separate files 
	debug >2 long console output 
	debug >3 prints some cell vert&horiz flux data (and indiv reach info in old style) to Driver1.out 
	debug >4 prints to another debugging file for special purposes 
*/

	
/* general NOTES on revisions to this source file:
        april 00 VERSION 2.1 - output available for application for water quality performance
        july 02 VERSIOIN 2.1a - first widely-available public release of code -
              misc modifications, such as revising method and types of
              parameters that are read in (reduced number of parms in "habitat-specific" dbase);
              added some more code documentation, etc.

*/

#include "UnitMod.h"

/*******
 a whole bunch of global arrays!!!!!
******/

float *ALG_INCID_LIGHT, *ALG_LIGHT_CF, *ALG_WAT_CF, *ALG_LIGHT_EXTINCT, *ALG_SAT, *ALG_TEMP_CF,
    *ALG_REFUGE, *ALG_AVAIL, *TP_algResp;
float *NC_ALG, *NC_ALG_GPP, *NC_ALG_RESP, *NC_ALG_CONS, *NC_ALG_MORT, *NC_ALG_AVAIL_MORT,
    *NC_ALG_MORT_POT, *NC_ALG_NUT_CF, *NC_ALG_PR1, *NC_ALG_PR2, *NC_ALG_PROD_CF,
    *NC_ALG_RESP_POT, *NC_ALG_NPP;
float *C_ALG, *C_ALG_GPP, *C_ALG_RESP, *C_ALG_CONS, *C_ALG_MORT, *C_ALG_AVAIL_MORT,
    *C_ALG_MORT_POT, *C_ALG_NUT_CF, *C_ALG_PR1, *C_ALG_PR2, *C_ALG_PROD_CF,
    *C_ALG_RESP_POT, *C_ALG_NPP;
float *NC_ALG_P, *C_ALG_P, *NC_ALG_GPP_P, *C_ALG_GPP_P, *NC_ALG_MORT_P, *C_ALG_MORT_P,
    *NC_ALG_PCrep, *C_ALG_PCrep; /*dec99 added vars for variable stoich */
double *NC_ALG_PC, *C_ALG_PC;

float *CONS_INGEST, *CONS_BIOM, *CONS_EGEST, *CONS_MORT_BIOM, *CONS_RESPIRATION, *CONS_INGEST_ALGAE,
    *CONS_INGEST_DEPOM, *CONS_INGEST_NPHBIO,*CONS_INGEST_PHBIO, *CONS_INGEST_POT, *CONS_INGEST_FLOC,
    *CONS_INGEST_STDETR, *CONS_SPEC_ING;

double *DEPOS_ORG_MAT;
float *OM_TOT_C_AVAIL, *DOM_FR_CONS, *DOM_FR_FLOC, *DOM_FR_MAC, *DOM_FR_STDET,
    *DOM_DECOMP, *DOM_CONSUMED, *DOM_FIRE, *DOM_Z, *DOM_CONS_POT,
    *DOM_C, *DOM_DECOMP_POT, *DOM_MOIST_CF, *DOM_PR1,*DOM_PR2, *DOM_QUALITY_CF,
    *DOM_SED_AEROB_Z, *DOM_SED_ANAEROB_Z, *DOM_TEMP_CF, *SED_ELEV, *SED_INACT_Z, *ELEVATION,
    *SED_ELEV_IC, *OM_ABV_SED;
float *DOP_FLOC, *DOP_nphBio, *DOM_fr_nphBio, *Floc_fr_phBio, *DOP_STDET,
    *DOP_CF, *DOP_DECOMP;
double *DOM_P_OM, *DOP; /*oct99 added TPtoSOIL, TPtoVOL, and DIM */
float  *BulkD, *DOM_BD, *Inorg_Z, *DIM, *TPtoSOIL, *TPtoVOL; /*  soil bulk dens (kg/m3), organic bulk dens (kg/m3), iorganic depth (m), inorg mass (kg), soil TP (mg/kg and ug/cm3) */

double *DIN_SED_WT, *DIN_SF_WT  ;
float *DIN_SEDWT_MINER, *DIN_DNFLOW, *DIN_SEDWT_UPTAKE, *DIN_SEDWT_DENIT,
    *DIN_UPFLOW, *DIN_FR_RAIN, *DIN_SFWT_MINER, *DIN_SFWT_UPTAKE,
    *DIN_DNFLOW_POT, *DIN_SEDWT_CONCACT, *DIN_SEDWT_CONCTOT, *DIN_SEDWT_DENIT_POT,
    *DIN_SEDWT_PR1, *DIN_SEDWT_PR2, *DIN_SEDWT_UPTAKE_POT, *DIN_SED_CONC_INT,
    *DIN_SFWT_CONC, *DIN_SFWT_PR1, *DIN_SFWT_UPT_POT, *DIN_UPFLOW_POT;

float *FIRE_DIRECT, *FIRE_INIT, *FIRE_LOSS, *FIRE_TOT_DIST, *FIRE_DIST_IN, *FIRE_DIST_DUMP,
    *FIRE_FUEL_BULK_DENS, *FIRE_FUEL_LOADING, *FIRE_LIGHT_HIT, *FIRE_MOIST_EFFECT,
    *FIRE_SPECIFIC_RATE, *FIRE_SPREAD_RATE, *FIRE_WIND_EFFECT;

float *HYD_PRECIP_DAY, *SAT_WATER, *UNSAT_TO_SAT_FL, *SF_WT_TO_SAT_DOWNFLOW, *SAT_WT_TRANSP,
    *SAT_TO_UNSAT_FL, *SAT_SURF_UP_FL, *SAT_WT_RECHG, *SURFACE_WAT, *SF_WT_FROM_RAIN,
    *SF_WT_EVAP, *SF_WT_INFILTRATION, *UNSAT_WATER, *UNSAT_TRANSP, *HYD_CANOPY_TRANSP,
    *HYD_DOM_ACTWAT_PRES, *HYD_DOM_ACTWAT_VOL, *HYD_EVAP_CALC, *HYD_MANNINGS_N, *HYD_MAX_SED_WAT,
    *HYD_RCCONDUCT, *HYD_SAT_POT_TRANS, *HYD_SED_WAT_VOL, *HYD_TOT_POT_TRANSP, *HydTotHd,
    *HYD_TRANSP, *HYD_UNSAT_POT_TRANS, *HYD_WATER_AVAIL, *SAT_ADD_FM_RISE, *SAT_TO_UNSAT_POT,
    *SAT_VS_UNSAT, *SAT_WT_HEAD, *SAT_WT_POT, *SAT_WT_PR1, *SAT_WT_PR2, *SAT_WT_PR3,
    *SFWAT_PR1, *SFWT_VOL, *SF_WT_HEAD, *SF_WT_POT_INF, *UNSAT_AVAIL, *UNSAT_CAP,
    *UNSAT_CAP_CHANGE, *UNSAT_DEPTH, *UNSAT_MOIST_PRP, *UNSAT_PERC, *UNSAT_PR1,
    *UNSAT_WT_POT, *UNSAT_HYD_COND_CF;
float *wmm_rain;   /* sfwmm rainfall, put into elm grid */
float HDmax; /* UNUSED, max cell-cell head difference */
float *CLOUDY, *SOLRADGRD, *SOLRAD274,*H2O_TEMP,*AIR_TEMP, *HUMIDITY, *DewPt, *WIND_SPEED,
    *WIND_SPEED_CANOPY, *PAN_CT, *PAN_CH, *PAN_CW; /*sep98 modifs to meteorology */

float *MAC_NOPH_BIOMAS, *NPHBIO_TRANSLOC, *NPHBIO_MORT, *PHBIO_TRANSLOC, *NPHBIO_FIRE, *NPHBIO_CONSUMED,
    *MAC_PH_BIOMAS, *PHBIO_NPP, *PHBIO_CONSUMED, *PHBIO_FIRE, *PHBIO_MORT, *MAC_HEIGHT, *MAC_LAI,
    *MAC_LIGHT_CF, *MAC_MAX_BIO, *MAC_NUT_CF, *MAC_PROD_CF, *MAC_REL_BIOM, *MAC_SALT_CF, *MAC_TEMP_CF,
    *MAC_TOT_BIOM, *MAC_WATER_CF, *NPHBIO_AVAIL, *NPHBIO_MORT_POT, *NPHBIO_PR1, *NPHBIO_PR2,
    *NPHBIO_PR3, *NPHBIO_REFUGE, *NPHBIO_SAT, *NPHBIO_TRANSLOC_POT, *PHBIO_AVAIL, *PHBIO_GROW_MAX,
    *PHBIO_MORT_POT, *PHBIO_PR1, *PHBIO_PR2, *PHBIO_PR3, *PHBIO_REFUGE, *PHBIO_SAT, *PHBIO_TRANSLOC_POT,
    *MAC_WATER_AVAIL_CF, *LAI_eff;
float *mac_nph_PC_rep, *mac_ph_PC_rep; /*oct99*/
double *phbio_npp_P, *phbio_npp_OM,*phbio_fire_P, *phbio_fire_OM, *phbio_mort_P,
    *phbio_mort_OM,*phbio_transl_P, *phbio_transl_OM, *nphbio_transl_P, *nphbio_transl_OM, *nphbio_fire_P,
    *nphbio_fire_OM, *nphbio_mort_P,*nphbio_mort_OM, *mac_nph_P, *mac_nph_PC, *mac_nph_OM, *mac_nph_CtoOM,
    *mac_ph_P,*mac_ph_PC, *mac_ph_OM, *mac_ph_CtoOM;

double *TP_SED_WT, *TP_SF_WT, *TP_SORB, *TP_SFWT_CONC, *TP_SORBCONC, *TP_SEDWT_CONCACT,
    *TP_SED_WT_AZ, *TP_SED_CONC, *TP_Act_to_Tot;
float *TP_DNFLOW, *TP_SED_MINER, *TP_SEDWT_UPTAKE, *TP_UPFLOW, *TP_SORBTION, 
    *TP_SFWT_MINER, *TP_FR_RAIN,*TP_SFWT_UPTAK, *TP_DNFLOW_POT, *TP_K, 
    *TP_SEDWT_CONCACTMG, *TP_SEDWT_PR1, *TP_SEDWT_PR2, *TP_SEDWT_UPT_POT,
    *TP_SFWT_CONC_MG, *TP_SFWT_PR1, *TP_SFWT_UPT_POT, 
    *TP_SORB_POT, *TP_UPFLOW_MG, *TP_UPFLOW_POT, *TP_settl, *TP_Act_to_TotRep;

double *SALT_SED_WT, *SALT_SURF_WT, *SAL_SED_WT, *SAL_SF_WT_mb;
float *SAL_MEAN,*SALT_SFWAT_DOWNFL, *SALT_SED_TO_SF_FLOW,
    *SALT_SFWAT_DOWNFL_POT, *SAL_SF_WT;

float *STAND_DETRITUS, *STDET_PROD, *STDET_BURN, *STDET_CONSUMED, *STDET_TO_DOM, *STDET_AVAIL,
    *STDET_BURN_POT, *STDET_PR1, *STDET_PR2, *STDET_REFUG, *STDET_SAT, *STDET_TO_DOM_POT, *WIND_STORM;
float *stdet_prod_P, *stdet_prod_OM, *stdet_burn_P, *stdet_burn_OM, *stdet_toDOM_P, *stdet_toDOM_OM,
    *stdet_P, *stdet_PC, *stdet_OM, *stdet_CtoOM;

float *FLOC, *FLOC_FR_CONSUM, *FLOC_FR_ALGAE, *FLOC_CONSUMED, *FLOC_DECOMP, *FLOC_DEPO,
    *FLOC_CONS_POT, *FLOC_C_AVAIL, *FLOC_DECOMP_POT, *FLOC_Z,
    *FLOC_DECOMP_QUAL_CF, *FLOC_DEPO_POT, *FLOC_PR1, *FLOC_PR2, *FLOC_PR3,
    *FlocP_FR_ALGAE, *FlocP_PhBio, *FlocP_DECOMP, *FlocP_DEPO, *FlocP_OMrep, *FLOC_Z, *FlocP;
double *FlocP_OM;



/* global parameters and habitat-specific parameters (latter arrays of size HAB) */
float *ALG_MAX,
    ALG_IC_MULT, ALG_REF_MULT, ALG_SHADE_FACTOR, AlgComp, ALG_C_TO_OM, ALG_KS_N,
    ALG_LIGHT_SAT, ALG_NC, ALG_PC, algMortDepth, ALG_RC_MORT, ALG_RC_MORT_DRY,
    ALG_RC_PROD, ALG_RC_RESP, ALG_TEMP_OPT, NC_ALG_KS_P, C_ALG_KS_P, C_ALG_threshTP,
    alg_R_accel, alg_light_ext_coef, alg_uptake_coef, alg_alkP_min;
float CONS_IC_MULT, CONS_TEMP_CF, CONS_ASSIM, CONS_C_TO_OM, CONS_MAX, CONS_NC, CONS_PC,
    CONS_RC_INGEST, CONS_RC_MORT, CONS_RC_RESP, CONS_T_OPT;
float *DOM_MAXDEPTH, *DOM_NPHBIO_PROP, *DOM_PHBIO_PROP, *DOM_AEROBTHIN,
    DOM_C_OM_OPT, DOM_DECOMPRED, DOM_DECOMP_TOPT, DOM_DECOMP_POPT,
    DOM_RCDECOMP, sorbToTP, DOM_decomp_coef;
float *DIN_CONC_GRAD,
    DIN_DIFFDEPTH, DIN_IN_RAIN, DIN_DIFFCOEF, DIN_ICSEDWAT, DIN_ICSFWAT,
    DIN_K_OF_NH4, DIN_RCDENIT;
float FIRE_HEAT_FOR_IGNIT, FIRE_LIGHT_THRESH, FIRE_PROP_THRESH, 
    *FIRE_FUEL_HEAT, *FIRE_RX_VELOC;
float ALTIT, AMPL, DATUM_DISTANCE, DAYJUL, DAYLENGTH, LATDEG, LATRAD,
    SOLALPHA, SOLALTCORR, SOLBETA, SOLCOSDEC, SOLDEC, SOLDEC1, SOLELEV_SINE, SOLOMEGA, SOLRADATMOS,
    SOLRISSET_HA, SOLRISSET_HA1, WIND_DIR;
float *HYD_POROSITY, *HYD_RCINFILT, *HYD_SPEC_YIELD,
    HYD_RCRECHG, HYD_ICUNSATMOIST, HYD_PAN_K, HYD_IC_SFWAT,
    HYD_IC_UNSATZ, DetentZ, MinCheck;
float *MAC_CANOPDECOUP, *MAC_KSN, *MAC_KSP,
    *MAC_LIGHTSAT,*MAC_MAXCANOPCOND, *MAC_MAXHT, *MAC_MAXLAI, *MAC_MAXROUGH,
    *MAC_MINROUGH, *MAC_SALIN_THRESH, *MAC_SPR_GROW, *MAC_TEMPOPT, *MAC_WAT_TOLER,
    *NPHBIO_ABVBEL, *NPHBIO_CTOOM, *NPHBIO_MAX, *NPHBIO_NC, *NPHBIO_PC,
    *NPHBIO_ROOTDEPTH, *PHBIO_CTOOM, *PHBIO_MAX, *PHBIO_NC, *PHBIO_PC,
    *PHBIO_RCMORT, *PHBIO_RCNPP, *Mac_transl_rc, mann_height_coef,
    MAC_IC_MULT, MAC_LITTERSEAS, MAC_REFUG_MULT, mac_uptake_coef;
float *TP_CONC_GRAD,
    TP_DIFFDEPTH, TP_IN_RAIN, PO4toTP, settlVel, PO4toTPint, TPpart_thresh, TP_DIFFCOEF,
    TP_ICSEDWAT, TP_ICSFWAT, TP_K_INTER, TP_K_SLOPE;
float *SALT_ICSEDWAT, *SALT_ICSFWAT;
float *STDET_ABVBEL, *STDET_CTOOM, *STDET_MAX, *STDET_NC, *STDET_PC, *STDET_SHRED, *STDET_WIND,
    STDET_IC_MULT, STDET_REF_MULT, StDetLos;
float *FLOC_CONSPROP, *FLOC_CTOOM, *FLOC_IC, *FLOC_NC, *FLOC_PC,
    Floc_BD, Floc_rcSoil, TP_P_OM, FlocMax;
/* for implementing cell_dyn13 only (Everglades Water Quality Model TP settling model equivalent) */
/* the WQMsettlVel is a spatial map array of parameters; TP_settlDays is a dynamic var */
float WQMthresh, *WQMsettlVel, *TP_settlDays; 

/* other */
float *IC_MULT_MAP; 
int *basn, *BCondFlow; /* hydro basins map and boundary conditions flow constraint map */
unsigned char *HAB, *ON_MAP;
int kMaxNHab = 255;
float *delaymap01, delay01(); /* not used, but leave for potential use later??? */

/* points list for meteorological data */
PTSeriesList *pTSList6;
PTSeriesList *pTSList5;
PTSeriesList *pTSList4;
PTSeriesList *pTSList3;
/* points list for CALM (not ELM) rainfall data */
PTSeriesList *pTSList2;

float Timestep0 = 1.0, TMod0 = 0; int NPtTS0 = 0, TIndex0 = 0;
float Timestep1 = 1.0, TMod1 = 0; int NPtTS1 = 0, TIndex1 = 0;
float Timestep2 = 1.0, TMod2 = 0; int NPtTS2 = 0, TIndex2 = 0;
float Timestep3 = 1.0, TMod3 = 0; int NPtTS3 = 0, TIndex3 = 0;
float Timestep4 = 1.0, TMod4 = 0; int NPtTS4 = 0, TIndex4 = 0;
float Timestep5 = 1.0, TMod5 = 0; int NPtTS5 = 0, TIndex5 = 0;
float Timestep6 = 1.0, TMod6 = 0; int NPtTS6 = 0, TIndex6 = 0;



#if NeedFunctionPrototypes
float graph5(unsigned char y, float x);
#else
float graph5();
#endif

float g5[13][2] = { {0.0000000,176.0000000},{30.3999996,178.0000000},{60.7999992,184.0000000},{91.1999969,189.0000000},
		{122.0000000,194.0000000},{152.0000000,194.0000000},{182.0000000,187.0000000},{213.0000000,187.0000000},
		{243.0000000,187.0000000},{274.0000000,187.0000000},{304.0000000,185.0000000},{335.0000000,185.0000000},
		{365.0000000,185.0000000} };

#if NeedFunctionPrototypes
float graph7(unsigned char y, float x);
#else
float graph7();
#endif

float g7[11][2] = { {0.0000000,0.0000000},{0.1000000,0.0200000},{0.2000000,0.0400000},{0.3000000,0.0700000},
		{0.4000000,0.1250000},{0.5000000,0.2150000},{0.6000000,0.3450000},{0.7000000,0.5750000},
		{0.8000000,0.8550000},{0.9000000,0.9850000},{1.0000000,1.0000000} };

#if NeedFunctionPrototypes
float graph8(unsigned char y, float x);
#else
float graph8();
#endif

float g8[11][2] = { {0.0000000,0.0050000},{0.1000000,0.0100000},{0.2000000,0.0400000},{0.3000000,0.1000000},
		{0.4000000,0.2800000},{0.5000000,0.7150000},{0.6000000,0.8650000},{0.7000000,0.9350000},
		{0.8000000,0.9750000},{0.9000000,0.9950000},{1.0000000,1.0000000} };



float delay00(Map,delay,init)
float delay;
float init;
float *Map;
{
	if(TIME<delay) return init;
	else return Map[(int)((TIME-delay)/DT)];
}



float delay01(Map,dt,ix,iy)
float Map;
int ix, iy; float dt;
{
	float rv;
	int i, step;
	static int first_call = 0, ixi=0, iyi=0;

	step = (int)(dt);
	if( ixi == 0 && iyi == 0 ) { ixi=ix; iyi=iy; }
	if( ix == ixi && iy == iyi ) first_call++;
	if( first_call <= step ) rv = Map;
	else rv = delaymap01[T(ix-1,iy-1)];
	delaymap01[T(ix-1,iy-1)] = Map;
	return(rv);
}


/*###############
The following modules are the dynamic calcs for the model.
They are called in the order determined by the call_cell_dyn function.
################*/


/*******
global dynamics, interpolation-generators, and habitat switching

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn1(step)
int step;
{
    int ix, iy, cellLoc; 
    float fTemp;

/* a choice of rainfall routines */
    if (ELMrun) { /* ELM (CALM can also) currently uses maps generated by HSM for the SFWMM/NSM */
        if (Mod(TIME,1.0) < 0.001)   
            if ( rain_data_wmm(wmm_rain) == NULL)  /*remap sfwmm rain array to elm's */ 
            { sprintf(msgStr,"Problem with rainfall data, time = %f.\n",TIME); WriteMsg(msgStr,1); exit(-1); }    
    }
    else { /* default to the spatial interpolation routine (used in CALM) */
        fTemp = FMOD(TIME,Timestep2*NPtTS2);
	PTSL_CreatePointMap(pTSList2,HYD_PRECIP_DAY,'f',(int)(fTemp/Timestep2),1.0);
    }
            
	
    DAYJUL = ( Mod(TIME,365.0) >0.0 ) ? ( Mod(TIME,365.0) ) : ( 365.0);
    DAYLENGTH = AMPL*Sin((DAYJUL-79.0)*0.01721)+12.0;
/*  interpolate daily data on avg air temp, avg wind spd, cloudiness, and dew point temp */ 
/* fctns in Driver_Utilities.c and Serial.c  */	
    fTemp = FMOD(TIME,Timestep3*NPtTS3); /* this value of #records (NPtTS) is same for all variables in the file, of course */
     PTSL_CreatePointMap(pTSList3,AIR_TEMP,'f',(int)(fTemp/Timestep3),1.0); 
     PTSL_CreatePointMap(pTSList4,DewPt,'f',(int)(fTemp/Timestep4),1.0);  
     PTSL_CreatePointMap(pTSList5,WIND_SPEED,'f',(int)(fTemp/Timestep5),1.0);  
     PTSL_CreatePointMap(pTSList6,CLOUDY,'f',(int)(fTemp/Timestep6),1.0);  
                
    WIND_DIR =  graph5(0x0,DAYJUL);

    SOLDEC1 = 0.39785*Sin(4.868961+0.017203*DAYJUL   
                          +0.033446*Sin(6.224111+0.017202*DAYJUL));
/*     SOLCOSDEC = Sqrt(1.0-pow(SOLDEC1,2.0)); */
    SOLCOSDEC = sqrt(1.0-SOLDEC1*SOLDEC1);
    SOLELEV_SINE = Sin(LATRAD)*SOLDEC1+Cos(LATRAD)*SOLCOSDEC;
    SOLALTCORR = (1.0-Exp(-0.014*(ALTIT-274.0)/(SOLELEV_SINE*274.0)));
    /* SOLDEC = Arctan(SOLDEC1/Sqrt(1.0-pow(SOLDEC1,2.0))); */
    SOLDEC = Arctan(SOLDEC1/sqrt(1.0-SOLDEC1*SOLDEC1));
    SOLRISSET_HA1 = -Tan(LATRAD)*Tan(SOLDEC);
/*     SOLRISSET_HA = ( (SOLRISSET_HA1==0.0) ) ? ( PI*0.5 ) : (   ( (SOLRISSET_HA1<0.0) ) ?  */
/*                                                                ( PI+Arctan(Sqrt(1.0-pow(SOLRISSET_HA1,2.0))/SOLRISSET_HA1)  ) :  */
/*                                                                (    Arctan(Sqrt(1.0-pow(SOLRISSET_HA1,2.0))/SOLRISSET_HA1))); */
    SOLRISSET_HA = ( (SOLRISSET_HA1==0.0) ) ? ( PI*0.5 ) : (   ( (SOLRISSET_HA1<0.0) ) ? 
                                                               ( PI+Arctan(sqrt(1.0-SOLRISSET_HA1*SOLRISSET_HA1)/SOLRISSET_HA1)  ) : 
                                                               (    Arctan(sqrt(1.0-SOLRISSET_HA1*SOLRISSET_HA1)/SOLRISSET_HA1)));
    SOLRADATMOS = 458.37*2.0*(1.0+0.033*Cos(360.0/365.0*PI/180.0*DAYJUL))
        * ( Cos(LATRAD)*Cos(SOLDEC)*Sin(SOLRISSET_HA) 
            + SOLRISSET_HA*180.0/(57.296*PI)*Sin(LATRAD)*Sin(SOLDEC));
            
  
        /* daily habitat switching */
    if ( (TIME - (int)TIME) < dt/2 && HabSwitch ) /* HabSwitch flag to invoke switching */
        for(ix=1; ix<=s0; ix++)  
            for(iy=1; iy<=s1; iy++)  
                if(ON_MAP[cellLoc= T(ix,iy)]) { /* TPtoSoil == kg/kg */
                    HAB [cellLoc] = F_Hab_SwA (ix, iy, SURFACE_WAT, TPtoSOIL, FIRE_DIRECT, HAB); 
                } 
}



/*******
  algae/periphyton - dynamics of C and P of two communities of periphyton
  
  NC_ALG[cellLoc] == carbon mass of the NonCalcareous (more appropriately, the eutrophic, or non-native) periphyton community (gC/m^2)
  C_ALG[cellLoc] == carbon mass of the Calcareous (more appropriately, the oligotrophic, or native) periphyton community (gC/m^2)

  NC_ALG_P[cellLoc] == phosphorus mass of NonCalcareous (more appropriately, the eutrophic, or non-native) periphyton community (gP/m^2)
  C_ALG_P[cellLoc] == phosphorus mass of Calcareous (more appropriately, the oligotrophic, or native) periphyton community (gP/m^2)
  
Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
  ******/
void cell_dyn2(step)
 int step;
 {
 int ix, iy, x, y, cellLoc; 
 float reduc, min_litTemp,I_ISat, Z_extinct; /* photoperiod not used Ph_per = DAYLENGTH/24.0 */
 float PO4Pconc, PO4P, C_ALG_thresh_CF, mortPot;
 
  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  {

/* light, water, temperature controls apply to both calc and non-calc */
		 ALG_LIGHT_EXTINCT[cellLoc]  = /* 0.1*SOM_CONC[cellLoc] + */ alg_light_ext_coef; /* dec99, changed light extinction coef,
                                                fixed parm now, with no suspended organic matter */
                     /* algal self-shading implicit in density-dependent constraint function later */
		 ALG_INCID_LIGHT[cellLoc]  = SOLRADGRD[cellLoc]*Exp(-MAC_LAI[cellLoc]*ALG_SHADE_FACTOR);
                 Z_extinct = SURFACE_WAT[cellLoc]*ALG_LIGHT_EXTINCT[cellLoc];
		 I_ISat = ALG_INCID_LIGHT[cellLoc]/ALG_LIGHT_SAT;
                     /* averaged over whole water column (based on Steele '65) */
                 ALG_LIGHT_CF[cellLoc]  = ( Z_extinct > 0.0 ) ? 
                     ( 2.718/Z_extinct * (Exp(-I_ISat * Exp(-Z_extinct)) - Exp(-I_ISat)) ) :
                     (I_ISat*Exp(1.0-I_ISat));
                     /* low-water growth constraint ready for something better based on data */
                 ALG_WAT_CF[cellLoc]  = ( SURFACE_WAT[cellLoc]>0.0 ) ? ( 1.0 ) : ( 0.0);
                 ALG_TEMP_CF[cellLoc]  = tempCF(0, 0.20, ALG_TEMP_OPT, 5.0, 40.0, H2O_TEMP[cellLoc]);

 		 min_litTemp = Min(ALG_LIGHT_CF[cellLoc],ALG_TEMP_CF[cellLoc]);

/* the 2 communities have same form of growth response to avail phosphorus;
   avail PO4 is calc'd from TP, and is always at least 10% of TP conc */
/* using regression for predicting PO4 from TP */
		 PO4Pconc = Max(TP_SFWT_CONC_MG[cellLoc]*PO4toTP + PO4toTPint,
                                0.10 * TP_SFWT_CONC_MG[cellLoc]); /* mg/L */
                 
                 NC_ALG_NUT_CF[cellLoc]  =
                     Exp(-alg_uptake_coef * Max(NC_ALG_KS_P-PO4Pconc, 0.0)/NC_ALG_KS_P) ; /* mg/L */
                 C_ALG_NUT_CF[cellLoc]  =
                     Exp(-alg_uptake_coef * Max(C_ALG_KS_P-PO4Pconc, 0.0)/C_ALG_KS_P) ; /* mg/L */
/*                  NC_ALG_NUT_CF[cellLoc]  =  PO4Pconc/(PO4Pconc+NC_ALG_KS_P); */
/*                  C_ALG_NUT_CF[cellLoc]  = PO4Pconc/(PO4Pconc+C_ALG_KS_P);  */

                     /* the form of the control function assumes that at very low
                        P conc, the microbial assemblage scavenges P, allowing periphyton a minimum nutrient availability */
                 NC_ALG_PROD_CF[cellLoc]  = Min(min_litTemp,ALG_WAT_CF[cellLoc])*Max(NC_ALG_NUT_CF[cellLoc], alg_alkP_min);
 		 C_ALG_PROD_CF[cellLoc]   = Min(min_litTemp,ALG_WAT_CF[cellLoc])*Max(C_ALG_NUT_CF[cellLoc], alg_alkP_min);

		 NC_ALG_AVAIL_MORT[cellLoc]  = Max(NC_ALG[cellLoc]-ALG_REFUGE[cellLoc],0);
		 C_ALG_AVAIL_MORT[cellLoc]  = Max(C_ALG[cellLoc]-ALG_REFUGE[cellLoc],0);

                 NC_ALG_RESP_POT[cellLoc]  = 
		 		( UNSAT_DEPTH[cellLoc]>algMortDepth ) ? 
		 		( 0.0) :
		 		( ALG_RC_RESP*ALG_TEMP_CF[cellLoc]*NC_ALG_AVAIL_MORT[cellLoc] ); 
		 C_ALG_RESP_POT[cellLoc]  = 
		 		( UNSAT_DEPTH[cellLoc]>algMortDepth ) ? 
		 		( 0.0) :
		 		( ALG_RC_RESP*ALG_TEMP_CF[cellLoc] *C_ALG_AVAIL_MORT[cellLoc] ); 

		 NC_ALG_RESP[cellLoc]  =  
		 		( NC_ALG_RESP_POT[cellLoc]*DT>NC_ALG_AVAIL_MORT[cellLoc] ) ? 
		 		( NC_ALG_AVAIL_MORT[cellLoc]/DT ) : 
		 		( NC_ALG_RESP_POT[cellLoc]);
		 C_ALG_RESP[cellLoc]  =  
		 		( C_ALG_RESP_POT[cellLoc]*DT>C_ALG_AVAIL_MORT[cellLoc] ) ? 
		 		( C_ALG_AVAIL_MORT[cellLoc]/DT ) : 
		 		( C_ALG_RESP_POT[cellLoc]);

		 NC_ALG_PR1[cellLoc]  = NC_ALG_RESP[cellLoc];
		 C_ALG_PR1[cellLoc]  = C_ALG_RESP[cellLoc];
                 

                     /* this is the threshold control function that increases
                        calcareous periph mortality (loss of calc. sheath?) as P increases */
                     /* moved the threshold response from respiration to mortality, as the respiration
                       increase allowed the P:C to increase to great excess */
                 C_ALG_thresh_CF = Min(exp(alg_R_accel*Max( TP_SFWT_CONC_MG[cellLoc]-C_ALG_threshTP,0.0)/C_ALG_threshTP), 100.0);

		 NC_ALG_MORT_POT[cellLoc]  = 
		 		( UNSAT_DEPTH[cellLoc]>algMortDepth ) ? 
		 		( NC_ALG_AVAIL_MORT[cellLoc]*ALG_RC_MORT_DRY ) : 
		 		( NC_ALG_AVAIL_MORT[cellLoc]*ALG_RC_MORT);
		 C_ALG_MORT_POT[cellLoc]  = 
		 		( UNSAT_DEPTH[cellLoc]>algMortDepth ) ? 
		 		( C_ALG_AVAIL_MORT[cellLoc]*ALG_RC_MORT_DRY ) : 
		 		( C_ALG_thresh_CF * C_ALG_AVAIL_MORT[cellLoc]*ALG_RC_MORT);

		 NC_ALG_MORT[cellLoc]  = 
		 		( (NC_ALG_MORT_POT[cellLoc]+NC_ALG_PR1[cellLoc])*DT>NC_ALG_AVAIL_MORT[cellLoc] ) ? 
		 		( (NC_ALG_AVAIL_MORT[cellLoc]-NC_ALG_PR1[cellLoc]*DT)/DT ) : 
		 		( NC_ALG_MORT_POT[cellLoc]);
		 C_ALG_MORT[cellLoc]  = 
		 		( (C_ALG_MORT_POT[cellLoc]+C_ALG_PR1[cellLoc])*DT>C_ALG_AVAIL_MORT[cellLoc] ) ? 
		 		( (C_ALG_AVAIL_MORT[cellLoc]-C_ALG_PR1[cellLoc]*DT)/DT ) : 
		 		( C_ALG_MORT_POT[cellLoc]);
                 
/* gross production of the 2 communities (gC/m2/d, NOT kgC/m2/d) */
                     /* density constraint contains both noncalc and calc, competition effect accentuated by calc algae */

                 NC_ALG_GPP[cellLoc]  =  NC_ALG_PROD_CF[cellLoc]*ALG_RC_PROD*NC_ALG[cellLoc]       
         		*Max( (1.0-(AlgComp*C_ALG[cellLoc]+NC_ALG[cellLoc])/ALG_MAX[HAB[cellLoc]]),0.0);
		 C_ALG_GPP[cellLoc]  =  C_ALG_PROD_CF[cellLoc]*ALG_RC_PROD*C_ALG[cellLoc] 
		 	*Max( (1.0-(        C_ALG[cellLoc]+NC_ALG[cellLoc])/ALG_MAX[HAB[cellLoc]]),0.0);
                 
                     
/* P uptake is dependent on available P and relative to a maximum P:C ratio for the tissue
   (g C/m^2/d * P:Cmax * dimless * dimless = gP/m2/d (NOT kg) )*/
                 NC_ALG_GPP_P[cellLoc] = NC_ALG_GPP[cellLoc] *ALG_PC * NC_ALG_NUT_CF[cellLoc]
                     * Max(1.0-NC_ALG_PC[cellLoc]/ALG_PC, 0.0); 
                 C_ALG_GPP_P[cellLoc] = C_ALG_GPP[cellLoc] * ALG_PC * C_ALG_NUT_CF[cellLoc]
                     * Max(1.0-C_ALG_PC[cellLoc]/ALG_PC, 0.0); 
                 
/* check for available P mass (the nutCF does not) */
                 PO4P = Min(PO4Pconc * SFWT_VOL[cellLoc],  1000.0*TP_SF_WT[cellLoc]); /*g P available (from conc. in mg/l == g/m3) */
                 reduc = ( (NC_ALG_GPP_P[cellLoc]+C_ALG_GPP_P[cellLoc]) > 0) ? 
                     (PO4P / ( (NC_ALG_GPP_P[cellLoc]+C_ALG_GPP_P[cellLoc])*CELL_SIZE*DT) ) :
                     (1.0);
		/* can have high conc, but low mass of P avail, in presence of high peri biomass and high demand */
		/* reduce the production proportionally if excess demand is found */
                 if (reduc < 1.0) {
                     NC_ALG_GPP[cellLoc] *= reduc;   
                     NC_ALG_GPP_P[cellLoc] *= reduc;   
                     C_ALG_GPP[cellLoc] *= reduc; 
                     C_ALG_GPP_P[cellLoc] *= reduc; 
                 }

/* state variables calc'd (gC/m2, NOT kgC/m2) */
		 NC_ALG[cellLoc] =  NC_ALG[cellLoc] 
		 		+ (NC_ALG_GPP[cellLoc]
                                - NC_ALG_RESP[cellLoc] - NC_ALG_MORT[cellLoc]) * DT;

		 C_ALG[cellLoc] =  C_ALG[cellLoc] 
		 		+ (C_ALG_GPP[cellLoc] 
		 		- C_ALG_RESP[cellLoc]  - C_ALG_MORT[cellLoc]) * DT;

/* carbon NPP not currently used elsewhere, only for output */
		 NC_ALG_NPP[cellLoc]  = NC_ALG_GPP[cellLoc]-NC_ALG_RESP[cellLoc]; 
		 C_ALG_NPP[cellLoc]  = C_ALG_GPP[cellLoc]-C_ALG_RESP[cellLoc];                 

                 
/*  now calc the P  associated with the C fluxes (GPP_P already calc'd) */
                 mortPot = (double) NC_ALG_MORT[cellLoc] * NC_ALG_PC[cellLoc];
                 NC_ALG_MORT_P[cellLoc] = (mortPot*DT>NC_ALG_P[cellLoc]) ?
                     (NC_ALG_P[cellLoc]/DT) :
                     (mortPot);
                 mortPot = (double) C_ALG_MORT[cellLoc] * C_ALG_PC[cellLoc];
                 C_ALG_MORT_P[cellLoc] = (mortPot*DT>C_ALG_P[cellLoc]) ?
                     (C_ALG_P[cellLoc]/DT) :
                     (mortPot);
                 

/* state variables calc'd (gP/m2, NOT kgP/m2) */
		 NC_ALG_P[cellLoc] =  NC_ALG_P[cellLoc] 
		 		+ (NC_ALG_GPP_P[cellLoc] - NC_ALG_MORT_P[cellLoc]) * DT;

		 C_ALG_P[cellLoc] =  C_ALG_P[cellLoc] 
		 		+ (C_ALG_GPP_P[cellLoc] - C_ALG_MORT_P[cellLoc]) * DT;

                 NC_ALG_PC[cellLoc] = (NC_ALG[cellLoc]>0.0) ?
                     (NC_ALG_P[cellLoc]/ NC_ALG[cellLoc]) :
                     ( ALG_PC * 0.03); /* default to 3% of  max P:C */
                 C_ALG_PC[cellLoc] = (C_ALG[cellLoc]>0.0) ?
                     (C_ALG_P[cellLoc]/ C_ALG[cellLoc]) :
                     ( ALG_PC * 0.03 ); /* default to 3% of max P:C */
                 NC_ALG_PCrep[cellLoc] = (float)NC_ALG_PC[cellLoc] * 1.0e6; /* new variable in float instead of double, for output compatibility */
                 C_ALG_PCrep[cellLoc] = (float)C_ALG_PC[cellLoc] * 1.0e6; /* new variable in float instead of double, for output compatibility */

                 if (debug > 0 && NC_ALG[cellLoc] < -MinCheck) { sprintf(msgStr,"Day %6.1f: ERROR - neg NC_ALG C biomass (%f m) in cell (%d,%d)!", 
                              TIME, NC_ALG[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
                 if (debug > 0 && C_ALG[cellLoc] < -MinCheck)  { sprintf(msgStr,"Day %6.1f: ERROR - neg C_ALG C biomass (%f m) in cell (%d,%d)!", 
                              TIME, C_ALG[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
                 if (debug > 0 && NC_ALG_P[cellLoc] < -MinCheck) { sprintf(msgStr,"Day %6.1f: ERROR - neg NC_ALG_P P biomass (%f m) in cell (%d,%d)!", 
                              TIME, NC_ALG_P[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
                 if (debug > 0 && C_ALG_P[cellLoc] < -MinCheck)  { sprintf(msgStr,"Day %6.1f: ERROR - neg C_ALG_P P biomass (%f m) in cell (%d,%d)!", 
                              TIME, C_ALG_P[cellLoc], ix,iy ); WriteMsg( msgStr,True );}

                     
		 TP_SFWT_UPTAK[cellLoc]  = (NC_ALG_GPP_P[cellLoc]+C_ALG_GPP_P[cellLoc])
                     *0.001*CELL_SIZE; /* gP/m2 => kg P */
/* recalc P in surface water state variable (kg P) */
		 TP_SF_WT[cellLoc] = TP_SF_WT[cellLoc] - TP_SFWT_UPTAK[cellLoc] * DT;
		 TP_SFWT_CONC[cellLoc]  = 
		 		( SFWT_VOL[cellLoc] > 0.0 ) ? 
		 		( TP_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
		 		( 0.0); /* used in P fluxes for mass balance */
		 TP_SFWT_CONC_MG[cellLoc]  = 
		 		( SURFACE_WAT[cellLoc] > DetentZ ) ? 
		 		(TP_SFWT_CONC[cellLoc]*1000.0) : 
		 		(0.0); /* used for reporting and other modules to evaluate P conc when water is present */

          }
	}
  }
}


/*******
  consumer dynamics
  WE DO NOT CALL THIS ROUTINE, AND NO CONSUMER EQNS ARE IMPLEMENTED IN OTHER FUNCTIONS

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn3(step)
 int step;
 { 
int ix, iy, x, y, cellLoc; 
/*  (units = gC/m2 (NOT kgC/m2)) */
  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  {
                 CONS_TEMP_CF = tempCF(0, 0.20, CONS_T_OPT, 5.0, 40.0, AIR_TEMP[cellLoc]);

/* ingestion potential */
		 CONS_INGEST_POT[cellLoc]  = CONS_TEMP_CF*CONS_BIOM[cellLoc] 
		 		*CONS_RC_INGEST*(1.0-CONS_BIOM[cellLoc]/CONS_MAX);
		 OM_TOT_C_AVAIL[cellLoc]  = DOM_C[cellLoc]+FLOC_C_AVAIL[cellLoc]
		 		+STDET_AVAIL[cellLoc]+ALG_AVAIL[cellLoc]+PHBIO_AVAIL[cellLoc]+NPHBIO_AVAIL[cellLoc];
/* ingestion constrained by (equally) available resources */
		 CONS_INGEST[cellLoc]  =  
		 		( CONS_INGEST_POT[cellLoc]*DT>OM_TOT_C_AVAIL[cellLoc]*1000.0 ) ? 
		 		( OM_TOT_C_AVAIL[cellLoc]*1000.0/DT ) : 
		 		( CONS_INGEST_POT[cellLoc]);
		 CONS_EGEST[cellLoc]  =  CONS_INGEST[cellLoc]*(1.0-CONS_ASSIM);
		 CONS_MORT_BIOM[cellLoc]  = CONS_RC_MORT*CONS_BIOM[cellLoc];
		 CONS_RESPIRATION[cellLoc]  = CONS_BIOM[cellLoc]*CONS_TEMP_CF*CONS_RC_RESP;
/* calc state var (gC/m2) */
		 CONS_BIOM[cellLoc] =  CONS_BIOM[cellLoc] 
		 		+ (CONS_INGEST[cellLoc] 
		 		- CONS_EGEST[cellLoc] - CONS_MORT_BIOM[cellLoc] - CONS_RESPIRATION[cellLoc]) * DT;
/* ingestion potentials of diff organic matter sources */
		 CONS_SPEC_ING[cellLoc]  = CONS_INGEST[cellLoc]*0.001/OM_TOT_C_AVAIL[cellLoc];
		 CONS_INGEST_ALGAE[cellLoc]  = ALG_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_DEPOM[cellLoc]  = DOM_C[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_NPHBIO[cellLoc]  = NPHBIO_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_PHBIO[cellLoc]  = PHBIO_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_FLOC[cellLoc]  = FLOC_C_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_STDETR[cellLoc]  = STDET_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
	  }
	}
  }
}



/*******
  Organic Soil: Deposited Organic Matter and Deposited Organic Phosphorus dynamics
  
     DEPOS_ORG_MAT[cellLoc] == mass of Deposited Organic Matter (DOM/AOM) of soil zone, w/o floc layer (kg OM/m^2)

     DOP[cellLoc] ==  mass of Deposited Organic Phosphorus of soil zone, w/o floc layer, w/o P sorbed (kg P/m^2)
  
Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
  ******/
void cell_dyn4(step)
 int step;
 {
int ix, iy, x, y, cellLoc;
 float TPsoil, TP_sedMin, TP_sfMin;
  
 

  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  {


/* inputs of organic matter (kg OM/m2)*/
/*  non-photo goes either to stand det or deposited OM,
    photo goes either to stand det or floc */
              DOM_fr_nphBio[cellLoc] = DOM_NPHBIO_PROP[HAB[cellLoc]]*nphbio_mort_OM[cellLoc];
              DOM_FR_STDET[cellLoc]  = stdet_toDOM_OM[cellLoc];
              DOM_FR_FLOC[cellLoc]  =  FLOC_DEPO[cellLoc] ; 

/* losses of organic matter (kg OM/m2) */
/* fire is zero */
		 DOM_FIRE[cellLoc]  = 
		 		( DEPOS_ORG_MAT[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*DT>DEPOS_ORG_MAT[cellLoc] ) ? 
		 		( DEPOS_ORG_MAT[cellLoc]/DT ) : 
		 		( DEPOS_ORG_MAT[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]);
		 DOM_PR1[cellLoc]  = DOM_FIRE[cellLoc];

/* 		 DOM_QUALITY_CF[cellLoc]  = Max( TP_SEDWT_CONCACTMG[cellLoc] */
/* 		 		/(TP_SEDWT_CONCACTMG[cellLoc] + DOM_DECOMP_POPT), 0.01);  */
                     /* modified jan01 to constrain to 1.0 */
                 DOM_QUALITY_CF[cellLoc]  =
                  Min(Exp(-DOM_decomp_coef * Max(DOM_DECOMP_POPT-TP_SEDWT_CONCACTMG[cellLoc], 0.0)
                      /DOM_DECOMP_POPT),1.0) ; /* mg/L */
                 DOM_TEMP_CF[cellLoc] = tempCF(0, 0.20, DOM_DECOMP_TOPT, 5.0, 40.0, H2O_TEMP[cellLoc]);
/*oct99*/
/* this moisture control function was calc'd earlier associated with the floc layer */
/* 		 DOM_MOIST_CF[cellLoc]  =  pow(Max(UNSAT_MOIST_PRP[cellLoc],0.0),0.75) ; */
/*                  DOM_MOIST_CF[cellLoc]  =  (UNSAT_DEPTH[cellLoc]>DOM_AEROBTHIN[HAB[cellLoc]]) ? */
/*                      ( Max(UNSAT_MOIST_PRP[cellLoc],0.0) ) : */
/*                      ( 1.0 ); */
/*oct99*/
		 DOM_SED_AEROB_Z[cellLoc]  = Min(Max(UNSAT_DEPTH[cellLoc],DOM_AEROBTHIN[HAB[cellLoc]]),DOM_MAXDEPTH[HAB[cellLoc]]);
		 DOM_SED_ANAEROB_Z[cellLoc]  = DOM_MAXDEPTH[HAB[cellLoc]]-DOM_SED_AEROB_Z[cellLoc];

                     /* beta3 is the calib parm read from Driver.parm */
                 DOM_DECOMP_POT[cellLoc]  = (double) beta3*DOM_RCDECOMP*DOM_QUALITY_CF[cellLoc]*DOM_TEMP_CF[cellLoc]*DEPOS_ORG_MAT[cellLoc]
		 		*(Min(DOM_SED_AEROB_Z[cellLoc]/DOM_MAXDEPTH[HAB[cellLoc]],1.0)*DOM_MOIST_CF[cellLoc]
		 		+DOM_DECOMPRED*Min(DOM_SED_ANAEROB_Z[cellLoc]/DOM_MAXDEPTH[HAB[cellLoc]],1.0) );
		 DOM_DECOMP[cellLoc]  =  
		 		( (DOM_DECOMP_POT[cellLoc]+DOM_PR1[cellLoc])*DT>DEPOS_ORG_MAT[cellLoc] ) ? 
		 		( (DEPOS_ORG_MAT[cellLoc]-DOM_PR1[cellLoc]*DT)/DT ) : 
		 		( DOM_DECOMP_POT[cellLoc]);
/* calc state var (kg OM/m2) */
		 DEPOS_ORG_MAT[cellLoc] =  DEPOS_ORG_MAT[cellLoc] + 
		 		( DOM_fr_nphBio[cellLoc] + DOM_FR_STDET[cellLoc] + DOM_FR_FLOC[cellLoc] 
		 		- DOM_DECOMP[cellLoc] - DOM_FIRE[cellLoc]) * DT;

/* soil elevation */
                 DOM_Z[cellLoc] = (double) DEPOS_ORG_MAT[cellLoc] / DOM_BD[cellLoc] ; /* (m) organic depth  */
		 SED_ELEV[cellLoc]  = DOM_Z[cellLoc]+Inorg_Z[cellLoc]+SED_INACT_Z[cellLoc];  /* total elevation (m) */

/* P DOM stoich (kg P /m2) */
                 DOP_nphBio[cellLoc] = DOM_NPHBIO_PROP[HAB[cellLoc]]*nphbio_mort_P[cellLoc];    
                 DOP_STDET[cellLoc] = stdet_toDOM_P[cellLoc] ; 
                 DOP_FLOC[cellLoc] = FlocP_DEPO[cellLoc]; 

                 DOP_DECOMP[cellLoc] = (double) DOM_DECOMP[cellLoc] * DOM_P_OM[cellLoc]; 
/* not accounting for fire and consumer effects (neither operating now) */

/* calc state var of total P in soil (NOT including dissolved in pore water or sorbed) (kgP/m2) */
		 DOP[cellLoc] =  DOP[cellLoc]  
		 		+ ( DOP_nphBio[cellLoc] + DOP_STDET[cellLoc] + DOP_FLOC[cellLoc] 
		 		- DOP_DECOMP[cellLoc]) * DT; /* kgP/m2 */

/* now the P ratio */
                 DOM_P_OM[cellLoc] = (DEPOS_ORG_MAT[cellLoc]>0.0) ? ( DOP[cellLoc] / DEPOS_ORG_MAT[cellLoc]) : (0.0); /* kgP/kgOM */
                 TPsoil = DOP[cellLoc]*CELL_SIZE + TP_SORB[cellLoc]; /* kg TP in soil */
                 TPtoSOIL[cellLoc] = ((DEPOS_ORG_MAT[cellLoc]*CELL_SIZE + DIM[cellLoc])>0.0) ?
                     (  TPsoil / (DEPOS_ORG_MAT[cellLoc]*CELL_SIZE + DIM[cellLoc]) ) : (0.0); /* kgP/kgsoil */
                 TPtoVOL[cellLoc] =  (CELL_SIZE * DOM_Z[cellLoc]>0.0) ?
                     (TPsoil / (CELL_SIZE * DOM_Z[cellLoc]) ) :
                  (0.0); /* kgP/m3 soil */

/* now the P gain in sed water with decomp
 a small proportion goes into surface water P (below) */
                 TP_sedMin  =  (1.0 - DOM_AEROBTHIN[HAB[cellLoc]]  / DOM_MAXDEPTH[HAB[cellLoc]] )
                     * DOP_DECOMP[cellLoc] * CELL_SIZE;
                 
/* calc P in sed water state variables (kg P) */
                 TP_SED_WT[cellLoc] =  TP_SED_WT[cellLoc] + TP_sedMin * DT;
                     /* this is the active zone, where uptake, sorption, and mineralization take place */
                 TP_SED_WT_AZ[cellLoc] =  TP_SED_WT_AZ[cellLoc] + TP_sedMin * DT;

		 TP_SED_CONC[cellLoc] = (HYD_SED_WAT_VOL[cellLoc]>0.0) ?
                     (TP_SED_WT[cellLoc] / HYD_SED_WAT_VOL[cellLoc]) :
                  (0.0);
                 TP_SEDWT_CONCACT[cellLoc] = ( HYD_DOM_ACTWAT_PRES[cellLoc] > 0.0) ?
                     ( TP_SED_WT_AZ[cellLoc]/HYD_DOM_ACTWAT_VOL[cellLoc] ) :
                  (TP_SED_CONC[cellLoc]); /* g/L */
                 TP_SEDWT_CONCACTMG[cellLoc]  = TP_SEDWT_CONCACT[cellLoc]*1000.0; /* mg/L */
              
/* now store the ratio of the conc in the active zone relative to total, prior to horiz fluxes
*****code in transition **** */              
                 TP_Act_to_Tot[cellLoc] = 1.0 / TP_CONC_GRAD[HAB[cellLoc]];
/*                  TP_Act_to_Tot[cellLoc] = (TP_SED_CONC[cellLoc] > 0.0  ) ? */
/*                      ( TP_SEDWT_CONCACT[cellLoc] / TP_SED_CONC[cellLoc] ) :  */
/*                   (0.0);  */
                 TP_Act_to_TotRep[cellLoc] =  (float) TP_Act_to_Tot[cellLoc];
                 
/* now the P gain in surface water with decomp in the very thin upper layer of the soil */
                     /* if there is no surface water present, assume that this
                        relative contribution will be an additional sorbed component that
                        is introduced to surface water column immediately upon hydration
                        with surface water */
                 TP_sfMin  =  DOM_AEROBTHIN[HAB[cellLoc]] / DOM_MAXDEPTH[HAB[cellLoc]]
                     * DOP_DECOMP[cellLoc] * CELL_SIZE;
                 
/* calc P in surface water state variable (kg P) */
                 TP_SF_WT[cellLoc] = TP_SF_WT[cellLoc] + TP_sfMin * DT;
                 TP_SFWT_CONC[cellLoc]  = 
                     ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                     ( TP_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                     ( 0.0); /* used in P fluxes for mass balance */
                 TP_SFWT_CONC_MG[cellLoc]  = 
                     ( SURFACE_WAT[cellLoc] > DetentZ ) ? 
                     (TP_SFWT_CONC[cellLoc]*1000.0) : 
                     (0.0); /* used for reporting and other modules to evaluate P conc when water is present */

	  }
	}
  }
}


/*******
  nitrogen dynamics 
  WE DO NOT CALL THIS ROUTINE, AND NO NITROGEN EQNS ARE IMPLEMENTED IN OTHER FUNCTIONS
  (this N module is outdated and needs fix'in)

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn5(step)
 int step;
 { 
int ix, iy, x, y, cellLoc; float posSum, negSum, fTemp, rFactor;

  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  {
		 if(debug>3) { x = dbgPt.x-lcl_start[0]+1; y = dbgPt.y-lcl_start[1]+1; }
/* 		 DIN_SEDWT_MINER[cellLoc]  =  DOM_DECOMP[cellLoc]*DOM_NC*DOM_CTOOM*CELL_SIZE; */
		 DIN_SFWT_CONC[cellLoc]  = ( SURFACE_WAT[cellLoc] > DetentZ ) ? ( DIN_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : ( 0.0);
		 DIN_SED_CONC_INT[cellLoc]  = HYD_DOM_ACTWAT_VOL[cellLoc]+(HYD_SED_WAT_VOL[cellLoc]-HYD_DOM_ACTWAT_VOL[cellLoc])*DIN_CONC_GRAD[HAB[cellLoc]];
		 DIN_SEDWT_CONCACT[cellLoc]  = ( DIN_SED_CONC_INT[cellLoc]>0.0 ) ? ( DIN_SED_WT[cellLoc]/DIN_SED_CONC_INT[cellLoc] ) : ( 0.0);
		 DIN_DNFLOW_POT[cellLoc]  = (SF_WT_INFILTRATION[cellLoc]+SF_WT_TO_SAT_DOWNFLOW[cellLoc])*CELL_SIZE*DIN_SFWT_CONC[cellLoc] +Max((DIN_SFWT_CONC[cellLoc]-DIN_SEDWT_CONCACT[cellLoc])*DIN_DIFFCOEF*8.64*DIN_DIFFDEPTH,0.0);
		 DIN_DNFLOW[cellLoc]  =  ( DIN_DNFLOW_POT[cellLoc]*DT>DIN_SF_WT[cellLoc] ) ? ( DIN_SF_WT[cellLoc]/DT ) : ( DIN_DNFLOW_POT[cellLoc]);
		 DIN_UPFLOW_POT[cellLoc]  = SAT_SURF_UP_FL[cellLoc]*CELL_SIZE*DIN_SEDWT_CONCACT[cellLoc] +Max((DIN_SEDWT_CONCACT[cellLoc]-DIN_SFWT_CONC[cellLoc])*DIN_DIFFCOEF*8.64*DIN_DIFFDEPTH,0.0);
		 DIN_SEDWT_CONCTOT[cellLoc]  = ( HYD_SED_WAT_VOL[cellLoc]>0.0 ) ? ( DIN_SED_WT[cellLoc]/HYD_SED_WAT_VOL[cellLoc] ) : ( 0.0);
		 DIN_SEDWT_DENIT_POT[cellLoc]  = DOM_SED_ANAEROB_Z[cellLoc]*CELL_SIZE*DIN_SEDWT_CONCTOT[cellLoc] *DIN_RCDENIT*pow(1.04,(AIR_TEMP[cellLoc]-20.0));
		 DIN_SEDWT_DENIT[cellLoc]  = ( DIN_SEDWT_DENIT_POT[cellLoc]*DT>DIN_SED_WT[cellLoc] ) ? ( DIN_SED_WT[cellLoc]/DT ) : (  DIN_SEDWT_DENIT_POT[cellLoc]);
		 DIN_SEDWT_PR1[cellLoc]  = DIN_SEDWT_DENIT[cellLoc];
		 DIN_UPFLOW[cellLoc]  = ( (DIN_UPFLOW_POT[cellLoc]+DIN_SEDWT_PR1[cellLoc])*DT>DIN_SED_WT[cellLoc] ) ? ( (DIN_SED_WT[cellLoc]-DIN_SEDWT_PR1[cellLoc]*DT)/DT ) : ( DIN_UPFLOW_POT[cellLoc]);
		 DIN_SEDWT_PR2[cellLoc]  = DIN_UPFLOW[cellLoc]+DIN_SEDWT_PR1[cellLoc];
		 DIN_SEDWT_UPTAKE_POT[cellLoc]  = PHBIO_NPP[cellLoc]*PHBIO_NC[HAB[cellLoc]]*DIN_K_OF_NH4*CELL_SIZE;
		 DIN_SEDWT_UPTAKE[cellLoc]  =  ( (DIN_SEDWT_PR2[cellLoc]+DIN_SEDWT_UPTAKE_POT[cellLoc])*DT>DIN_SED_WT[cellLoc] ) ? ( (DIN_SED_WT[cellLoc]-DIN_SEDWT_PR2[cellLoc]*DT)/DT ) : ( DIN_SEDWT_UPTAKE_POT[cellLoc]);
		 posSum = negSum = 0;
		 DIN_SED_WT[cellLoc] =  DIN_SED_WT[cellLoc] + (DIN_SEDWT_MINER[cellLoc] + DIN_DNFLOW[cellLoc] +  0.0  - DIN_SEDWT_UPTAKE[cellLoc] -  0.0  - DIN_SEDWT_DENIT[cellLoc] - DIN_UPFLOW[cellLoc]) * DT;
		 DIN_SED_WT[cellLoc] = (DIN_SED_WT[cellLoc] > 0) ? DIN_SED_WT[cellLoc] : 0.0;
				 if(debug>3) if(step==0) if( ix==x && iy==y ) { sprintf(msgStr,"\nDIN_SED_WT fluxes = ( DIN_SED_WT[cellLoc] , DIN_SEDWT_MINER[cellLoc] , DIN_DNFLOW[cellLoc] ,  0.0  , DIN_SEDWT_UPTAKE[cellLoc] ,  0.0  , DIN_SEDWT_DENIT[cellLoc] , DIN_UPFLOW[cellLoc])\n"); WriteMsg(msgStr,ALLPROCS); }
		 if(debug>3) if( ix==x && iy==y ) { sprintf(msgStr,"\nDIN_SED_WT fluxes = (%f,%f,%f,%f,%f,%f,%f,%f,)", DIN_SED_WT[cellLoc] , DIN_SEDWT_MINER[cellLoc] , DIN_DNFLOW[cellLoc] ,  0.0  , DIN_SEDWT_UPTAKE[cellLoc] ,  0.0  , DIN_SEDWT_DENIT[cellLoc] , DIN_UPFLOW[cellLoc]); WriteMsg(msgStr,ALLPROCS); }
		 DIN_FR_RAIN[cellLoc]  =  SF_WT_FROM_RAIN[cellLoc]*CELL_SIZE*DIN_IN_RAIN*0.001;
		 DIN_SFWT_MINER[cellLoc]  = FLOC_DECOMP[cellLoc]*FLOC_NC[HAB[cellLoc]]*FLOC_CTOOM[HAB[cellLoc]];
		 DIN_SFWT_UPT_POT[cellLoc]  = NC_ALG_NPP[cellLoc]*0.001*ALG_NC*CELL_SIZE;
		 DIN_SFWT_PR1[cellLoc]  = DIN_DNFLOW[cellLoc];
		 DIN_SFWT_UPTAKE[cellLoc]  = ( (DIN_SFWT_UPT_POT[cellLoc]+DIN_SFWT_PR1[cellLoc])*DT>DIN_SF_WT[cellLoc] ) ? ( (DIN_SF_WT[cellLoc]-DIN_SFWT_PR1[cellLoc]*DT)/DT ) : ( DIN_SFWT_UPT_POT[cellLoc]);
		 posSum = negSum = 0;
		 DIN_SF_WT[cellLoc] = DIN_SF_WT[cellLoc] + (DIN_FR_RAIN[cellLoc] + DIN_SFWT_MINER[cellLoc] + DIN_UPFLOW[cellLoc] +  0.0  - DIN_DNFLOW[cellLoc] - DIN_SFWT_UPTAKE[cellLoc] -  0.0 ) * DT;
		 DIN_SF_WT[cellLoc] = (DIN_SF_WT[cellLoc] > 0) ? DIN_SF_WT[cellLoc] : 0.0;
				 if(debug>3) if(step==0) if( ix==x && iy==y ) { sprintf(msgStr,"\nDIN_SF_WT fluxes = (DIN_SF_WT[cellLoc] , DIN_FR_RAIN[cellLoc] , DIN_SFWT_MINER[cellLoc] , DIN_UPFLOW[cellLoc] ,  0.0  , DIN_DNFLOW[cellLoc] , DIN_SFWT_UPTAKE[cellLoc] ,  0.0 )\n"); WriteMsg(msgStr,ALLPROCS); }
		 if(debug>3) if( ix==x && iy==y ) { sprintf(msgStr,"\nDIN_SF_WT fluxes = (%f,%f,%f,%f,%f,%f,%f,%f,)",DIN_SF_WT[cellLoc] , DIN_FR_RAIN[cellLoc] , DIN_SFWT_MINER[cellLoc] , DIN_UPFLOW[cellLoc] ,  0.0  , DIN_DNFLOW[cellLoc] , DIN_SFWT_UPTAKE[cellLoc] ,  0.0 ); WriteMsg(msgStr,ALLPROCS); }
	  }
	}
  }
}


/*******
  fire dynamics 
  WE DO NOT CALL THIS ROUTINE, AND FIRE LOSSES ARE THUS ZERO

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn6(step)
 int step;
 {
    int ix, iy, cellLoc, onCells=0;
    float waterLevel;
 
        FIRE_LIGHT_THRESH = 0.02*Cos(DAYJUL/365.0*2.0*PI)+0.98;
        onCells = 0;
        for(ix=1; ix<=s0; ix++)
          for(iy=1; iy<=s1; iy++)
                if(ON_MAP[cellLoc= T(ix,iy)])
                {
        			FIRE_WIND_EFFECT[cellLoc] = Exp(Max(0.1*WIND_SPEED[cellLoc],0.0));
        printf("(1) SURFACE_WAT %f \n", SURFACE_WAT[cellLoc]);
                   waterLevel = SURFACE_WAT[cellLoc];
                   onCells++;
                   FIRE_SPECIFIC_RATE[cellLoc] = 0.0;
                }
        waterLevel /= onCells;
	/*        elfm(step, FIRE_SPECIFIC_RATE, waterLevel);*/

 }


/*******
  horizontal fluxes of water and dissolved/suspended constituents (Phosphorus, Nitrogen (not running), Salt)
   among cells and in water management network.
    
  ******/
void horizFlow(step)
int step;
{
    int it;

    for ( it = 0; it < hyd_iter; it++ )
    {
        sprintf(msgStr,"\b\b%2d",it);  usrErr0(msgStr);
       
/* horizontal raster-vector canal fluxes and water management functions
   Water Management switch set at runtime in Driver.parm
   this routine also integrates surface, unsaturated, and saturated exchanges
   It is only called ever other (odd) hyd_iter iteration */
        if (WatMgmt /* && (it%2) */  ) Run_Canal_Network(SURFACE_WAT,SED_ELEV,HYD_MANNINGS_N,SAT_WATER,HYD_POROSITY,
                                                         HYD_RCCONDUCT, DIN_SF_WT,TP_SF_WT,SALT_SURF_WT,DIN_SED_WT,TP_SED_WT,SALT_SED_WT,
                                                         UNSAT_WATER,HYD_SPEC_YIELD ); /*tmp*/

/* Function for horiz fluxing of surface water, no exchange with sat/unsat water */
/* if the order of the solute is changed, be sure to change the mass bal info in FluxSWstuff fcn */
        Flux_SWater (it,SURFACE_WAT,SED_ELEV,HYD_MANNINGS_N,SALT_SURF_WT,DIN_SF_WT,TP_SF_WT,HDmax);       
	
/* Function for horiz fluxing of ground water and its vertical itegration with unsat and surface water 
   It is only called ever other (odd) hyd_iter iteration, and passes the realized # gwat iterations to the function.  
   If the order of the solutes is changed, be sure to change the mass bal info in FluxGWstuff fcn */
        if ( it%2 ) Flux_GWater ((it-1)/2, SAT_WATER, UNSAT_WATER, SURFACE_WAT, DOM_MAXDEPTH, HYD_RCCONDUCT, HYD_POROSITY,
                                 HYD_SPEC_YIELD, SED_ELEV,
                                 SALT_SED_WT, DIN_SED_WT, TP_SED_WT, SALT_SURF_WT, DIN_SF_WT, TP_SF_WT);  

    }  

}



/*******
  Hydrology (vertical fluxes only)

  SURFACE_WAT[cellLoc] == water storage above the land surface elevation (meters)
  UNSAT_WATER[cellLoc] == water storage in the pore spaces of the unsaturated zone (meters)
  SAT_WATER[cellLoc] == water storage in the pore spaces of the saturated zone (meters)
  
  (the horizontal solution, horizFlow function, makes other vert calcs to
  integrate the horiz flows within the three vertical zones of water storage)

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn7(step)
 int step;
 {
    int ix, iy, x, y, cellLoc; 
    float VAP_PRESS_SAT, VapPres_DPT, VAP_SAT_DEFICIT, SatWat_Root_CF, field_cap; /* modif vars,  sept98, dec98 */
    float hum2, mann_height, N_density, f_LAI_eff;
    
/* the horizontal raster and vector fluxes are always called before this cell_dyn */
/* now the vertical hydro calcs */
    for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {
            if(ON_MAP[cellLoc= T(ix,iy)])  {


/*  note that rainfall during a time step is added to surface water storage and available */
/* for runoff (horizFlow) before the calc of infiltration & ET associated with that new input */
/* (infiltration/ET etc will be of avail water the next time step after a rainfall event and horiz flows) */

                  
                  /* data arrays were created in cell_dyn1 */
                SF_WT_FROM_RAIN[cellLoc]  =   (ELMrun) ?
                    (wmm_rain[cellLoc] *0.0001) :      /* HSM:NSM/SFWMM input data, convert here from tenths of mm to meters */
                    (HYD_PRECIP_DAY[cellLoc]*0.0254);  /* interpolated data, convert here from inches to meters */

/******** meteorological calcs: temperature, cloudiness, humidity (&dew point temp), and wind speed spatially distributed */
      /* read in, interpolate (cell_dyn1) daily data on avg air temp, avg wind spd, cloudiness, and dew point temp */
              VapPres_DPT =   610.78*Exp(17.269*DewPt[cellLoc]/(DewPt[cellLoc]+237.3));
              VAP_PRESS_SAT = 610.78*Exp(17.269*AIR_TEMP[cellLoc]/(AIR_TEMP[cellLoc]+237.3));
              HUMIDITY[cellLoc] =  Min( VapPres_DPT / VAP_PRESS_SAT, 1.0); 
              VAP_SAT_DEFICIT = 1.0 - HUMIDITY[cellLoc];
              WIND_SPEED_CANOPY[cellLoc] = WIND_SPEED[cellLoc]*0.666;
/*               PAN_CH[cellLoc] = 1.035+0.240*pow((HUMIDITY[cellLoc]/0.60),2.0)-0.275*pow((HUMIDITY[cellLoc]/0.60),3.0 ) ; */
/*               PAN_CT[cellLoc] = 0.463+0.425*(AIR_TEMP[cellLoc]/20.0)+0.112*pow((AIR_TEMP[cellLoc]/20.0),2.0); */
/*               PAN_CW[cellLoc] =  0.672+0.406*(WIND_SPEED_CANOPY[cellLoc]/6.7)-0.078*pow((WIND_SPEED_CANOPY[cellLoc]/6.7),2.0 ) ; */
              hum2 = HUMIDITY[cellLoc]*HUMIDITY[cellLoc];
              PAN_CH[cellLoc] = 1.035+0.240*hum2/0.36-0.275*hum2*HUMIDITY[cellLoc]/0.216 ;
              PAN_CT[cellLoc] = 0.463+0.425*(AIR_TEMP[cellLoc]/20.0)+0.112*AIR_TEMP[cellLoc]*AIR_TEMP[cellLoc]/400.0;
              PAN_CW[cellLoc] =  0.672+0.406*(WIND_SPEED_CANOPY[cellLoc]/6.7)-0.078*WIND_SPEED_CANOPY[cellLoc]*WIND_SPEED_CANOPY[cellLoc]/44.89 ;
              SOLRAD274[cellLoc] = SOLRADATMOS*(SOLBETA-SOLOMEGA*
                                                (( CLOUDY[cellLoc]>0.0 ) ? 
                                                 ( CLOUDY[cellLoc] ) : ( 0.0)))-SOLALPHA;
              SOLRADGRD[cellLoc] = SOLRAD274[cellLoc]+((SOLRADATMOS+1.0)-SOLRAD274[cellLoc])*SOLALTCORR;
              H2O_TEMP[cellLoc] = AIR_TEMP[cellLoc];
              
/******** determine new unsat potentials */
              SAT_WT_HEAD[cellLoc]  = SAT_WATER[cellLoc]/HYD_POROSITY[HAB[cellLoc]];
              UNSAT_DEPTH[cellLoc]  = SED_ELEV[cellLoc]-SAT_WT_HEAD[cellLoc];
              UNSAT_CAP[cellLoc]  =  UNSAT_DEPTH[cellLoc]*HYD_POROSITY[HAB[cellLoc]];
              if (debug>2) { 
                  if ( (UNSAT_DEPTH[cellLoc] < -0.001) ) {
                      sprintf(msgStr,"Day %6.1f: capacityERR - neg unsat depth (%f m) in cell (%d,%d) before cell_dyn7! ", 
                              TIME, UNSAT_DEPTH[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
                  if ((UNSAT_WATER[cellLoc] - UNSAT_CAP[cellLoc])> 0.001) {
                      sprintf(msgStr,"Day %6.1f: capacityERR - UNSAT_WATER (%f m) exceeds UNSAT_CAP (%f m) in cell (%d,%d) before cell_dyn7!", 
                              TIME, UNSAT_WATER[cellLoc],UNSAT_CAP[cellLoc], ix,iy ); WriteMsg( msgStr,True );  }
              }
          
              UNSAT_MOIST_PRP[cellLoc]  = 
                  ( UNSAT_CAP[cellLoc]>0.0 ) ? 
                  ( Min(UNSAT_WATER[cellLoc]/UNSAT_CAP[cellLoc],1.0) ) : 
                  ( 1.0); 
                  /* determining the pathway of flow of surface water depending on depth
                     of an unsat zone relative to the surface water  */ 
              SAT_VS_UNSAT[cellLoc]  = 1/Exp(100.0*Max((SURFACE_WAT[cellLoc]-UNSAT_DEPTH[cellLoc]),0.0)); 
              UNSAT_HYD_COND_CF[cellLoc]  = graph7(0x0,UNSAT_MOIST_PRP[cellLoc] ); 
                     /* field capacity = porosity - specific yield; spec yield== proportion of total soil vol
                            that represents water that can be moved by gravity */
              field_cap = (HYD_POROSITY[HAB[cellLoc]]-HYD_SPEC_YIELD[HAB[cellLoc]]);
                  /* unsat_avail is proportion of water in pore space available for gravitational flow (above field capacity) */
                  /* e.g., when moisture prop in pore space <= field_cap/pore_space, no percolation */
                  /* using old moisture proportion (hasn't changed unless unsat zone was replaced by sat water) */
              UNSAT_AVAIL[cellLoc]  = Max(UNSAT_MOIST_PRP[cellLoc]
                                          -(field_cap)/HYD_POROSITY[HAB[cellLoc]],0.0);
              UNSAT_WT_POT[cellLoc]  = Max(UNSAT_CAP[cellLoc]-UNSAT_WATER[cellLoc],0.0);

/******** now determine the potential total transpiration and evaporation  */
              HYD_CANOPY_TRANSP[cellLoc]  = 0.75*DAYLENGTH 
                  * (MAC_MAXCANOPCOND[HAB[cellLoc]]*3600.0*0.018*0.001)* VAP_SAT_DEFICIT/1E5;
                  /* beta2 is a global calibration parameter (near 1.0, read from Driver.parm) for evaporation */
              HYD_EVAP_CALC[cellLoc]  = 0.482*SOLRADGRD[cellLoc]/585.0
                  *PAN_CW[cellLoc]*PAN_CT[cellLoc]*PAN_CH[cellLoc]*0.01*HYD_PAN_K*beta2;
		/*  effective LAI accounts for leaf area submerged */
              LAI_eff[cellLoc] =  
                  (MAC_HEIGHT[cellLoc]>0.0) ? 
                  (Max(1.0 - SURFACE_WAT[cellLoc]/MAC_HEIGHT[cellLoc], 0.0)*MAC_LAI[cellLoc]) : 
                  (0.0)  ;
         /* total (canopy-driven and evap-driven) potential transpiration without water limitation; 
         	had 0.33 * LAI_eff (aug31=0.5) */
              f_LAI_eff = exp(-LAI_eff[cellLoc]); /* ranges from 1.0 at LAI_eff=0, to 0.22 at LAI_eff=2.0, using 0.75 parm */
              
              HYD_TOT_POT_TRANSP[cellLoc]  = (HYD_CANOPY_TRANSP[cellLoc]*(1.0-MAC_CANOPDECOUP[HAB[cellLoc]]) 
                                              + (HYD_EVAP_CALC[cellLoc]*MAC_CANOPDECOUP[HAB[cellLoc]]) ) 
                  *(1.0-f_LAI_eff); /*jan2000 had 0.66; dec98 had 0.66 & CALM's newSeepLocs run had 0.66,
                                           changed back to 0.33 april99 (calm paper had 0.2?,
                                           prior to calm paper, believe it was 0.33)*/

		 /* 0-1 control function of saturated water available to roots - 
		 	capillary draw when roots don't quite reach down to water table */
              SatWat_Root_CF =  Exp(-10.0* Max(UNSAT_DEPTH[cellLoc]-NPHBIO_ROOTDEPTH[HAB[cellLoc]],0.0) ); /* was exp(-10.0*...)*/
		 /* 0-1 proportion representing available water for determining plant growth stress: 
		 	no stress to macrophyte root system if unsat depth < root depth, otherwise 
		 	function of moisture proportion within root zone */
              HYD_WATER_AVAIL[cellLoc]  = (UNSAT_DEPTH[cellLoc] > NPHBIO_ROOTDEPTH[HAB[cellLoc]] ) ? 
                  ( Max(UNSAT_MOIST_PRP[cellLoc], SatWat_Root_CF) ) :
                  ( 1.0 ); /* dec98 */
              MAC_WATER_AVAIL_CF[cellLoc]  = graph8(0x0,HYD_WATER_AVAIL[cellLoc]); 

/******** next calc the potential and actual flows */
/* unsatLoss(1) unsat to sat percolation */
	/*unsat to sat flow here only includes percolation (rising water table accomodated in update after horiz fluxes) */ 
              UNSAT_PERC[cellLoc]  = Min(HYD_RCINFILT[HAB[cellLoc]]*UNSAT_HYD_COND_CF[cellLoc],UNSAT_AVAIL[cellLoc]*UNSAT_WATER[cellLoc]);
              UNSAT_TO_SAT_FL[cellLoc]  =  
                  ( (UNSAT_PERC[cellLoc])*DT > UNSAT_WATER[cellLoc] ) ? 
                  ( UNSAT_WATER[cellLoc]/DT ) : 
                  ( UNSAT_PERC[cellLoc]);
              UNSAT_PR1[cellLoc]  = UNSAT_TO_SAT_FL[cellLoc];
/* unsatLoss(2) unsat to atmosph transpiration */
              HYD_UNSAT_POT_TRANS[cellLoc]  = (UNSAT_DEPTH[cellLoc] > NPHBIO_ROOTDEPTH[HAB[cellLoc]] ) ?
                  ( HYD_TOT_POT_TRANSP[cellLoc]*MAC_WATER_AVAIL_CF[cellLoc] ) :
                  (0.0); /* no unsat transp if roots are in saturated zone, dec98 */
              UNSAT_TRANSP[cellLoc]  = 
                  ( (HYD_UNSAT_POT_TRANS[cellLoc]+UNSAT_PR1[cellLoc])*DT>UNSAT_WATER[cellLoc] ) ? 
                  ( (UNSAT_WATER[cellLoc]-UNSAT_PR1[cellLoc]*DT)/DT ) : 
                  ( HYD_UNSAT_POT_TRANS[cellLoc]);

/* satLoss(1) sat to deep aquifer recharge **RATE IS ALWAYS ZERO RIGHT NOW *****/
              SAT_WT_RECHG[cellLoc]  = 
                  ( HYD_RCRECHG*HYD_SPEC_YIELD[HAB[cellLoc]]/HYD_POROSITY[HAB[cellLoc]]*DT>SAT_WATER[cellLoc] ) ? 
                  ( SAT_WATER[cellLoc]/DT ) : 
                  ( HYD_RCRECHG*HYD_SPEC_YIELD[HAB[cellLoc]]/HYD_POROSITY[HAB[cellLoc]]); /* !!!! THIS RATE IS ALWAYS ZERO */
                 
/* sat to surf upflow  when gwater exceeds soil capacity due to lateral inflows
   accomodated in gwFluxes */

/* satLoss(2) sat to unsat with lowering water table due to recharge loss ONLY (horiz outflows accomodated in gwFluxes)
   (leaves field capacity amount in unsat zone)*/
              SAT_TO_UNSAT_FL[cellLoc]  =  
                  ( SAT_WT_RECHG[cellLoc]*field_cap/HYD_POROSITY[HAB[cellLoc]]*DT > SAT_WATER[cellLoc] ) ? 
                  ( (SAT_WATER[cellLoc])/DT ) : 
                  ( SAT_WT_RECHG[cellLoc]*field_cap/HYD_POROSITY[HAB[cellLoc]]) ;
              SAT_WT_PR1[cellLoc]  = SAT_TO_UNSAT_FL[cellLoc];
/* satLoss(3) sat to atmosph */
              HYD_SAT_POT_TRANS[cellLoc]  = HYD_TOT_POT_TRANSP[cellLoc]*SatWat_Root_CF; /*dec98*/
              SAT_WT_TRANSP[cellLoc]  =  
                  ( (HYD_SAT_POT_TRANS[cellLoc]+SAT_WT_PR1[cellLoc])*DT > SAT_WATER[cellLoc] ) ? 
                  ( (SAT_WATER[cellLoc]-SAT_WT_PR1[cellLoc]*DT)/DT ) : 
                  ( HYD_SAT_POT_TRANS[cellLoc]);

/* sfwatLoss(1) sf to sat */
                     /* downflow to saturated assumed to occur in situations with small
                        unsat layer overlain by significant surface water (SAT_VS_UNSAT),
                        with immediate hydraulic connectivity betweent the two storages */
              SF_WT_TO_SAT_DOWNFLOW[cellLoc]  = 
                  ( (1.0-SAT_VS_UNSAT[cellLoc])*UNSAT_WT_POT[cellLoc]*DT>SURFACE_WAT[cellLoc] ) ? 
                  ( SURFACE_WAT[cellLoc]/DT ) : 
                  ( (1.0-SAT_VS_UNSAT[cellLoc])*UNSAT_WT_POT[cellLoc]); 
/* sfwatLoss(2) sf to unsat infiltration (sat_vs_unsat splits these losses to groundwater, but downflow to sat is given priority) */
              SF_WT_POT_INF[cellLoc]  = 
                  ( (SAT_VS_UNSAT[cellLoc]*HYD_RCINFILT[HAB[cellLoc]]+SF_WT_TO_SAT_DOWNFLOW[cellLoc])*DT>SURFACE_WAT[cellLoc] ) ? 
                  ( (SURFACE_WAT[cellLoc]-SF_WT_TO_SAT_DOWNFLOW[cellLoc]*DT)/DT ) : 
                  ( SAT_VS_UNSAT[cellLoc]*HYD_RCINFILT[HAB[cellLoc]]);
              SF_WT_INFILTRATION[cellLoc]  = 
                  ( SF_WT_POT_INF[cellLoc]*DT > (UNSAT_WT_POT[cellLoc]-SF_WT_TO_SAT_DOWNFLOW[cellLoc]*DT) ) ? 
                  ((UNSAT_WT_POT[cellLoc]-SF_WT_TO_SAT_DOWNFLOW[cellLoc]*DT)/DT ) : 
                  ( SF_WT_POT_INF[cellLoc]);
              SFWAT_PR1[cellLoc]  = SF_WT_INFILTRATION[cellLoc]+SF_WT_TO_SAT_DOWNFLOW[cellLoc];
/* sfwatLoss(3) sf to atmosph */
              SF_WT_EVAP[cellLoc]  =  /* jan 2000 had 0.2; reduce evap based on effective lai shading, had 0.2 (calm paper?)
                                       CALM's newSeepLocs run had 0.5, changed back to 0.2  */
                  ( (f_LAI_eff*HYD_EVAP_CALC[cellLoc]+SFWAT_PR1[cellLoc] )*DT>SURFACE_WAT[cellLoc] ) ? 
                  ( (SURFACE_WAT[cellLoc]-SFWAT_PR1[cellLoc]*DT)/DT ) : 
                  ( f_LAI_eff*HYD_EVAP_CALC[cellLoc]); 


/******** then update the state variable storages */

/* unsat loss priority:  percolation,  transpiration */
/* calc unsaturated storage state var (m) */
              UNSAT_WATER[cellLoc] = UNSAT_WATER[cellLoc] 
                  + (SAT_TO_UNSAT_FL[cellLoc] + SF_WT_INFILTRATION[cellLoc] 
                     - UNSAT_TO_SAT_FL[cellLoc] - UNSAT_TRANSP[cellLoc]) * DT;

/* sat loss priority:  recharge to deep aquifer, re-assign to unsat with lowered water table, transpiration */
/* calc saturated storage state var (m) */
              SAT_WATER[cellLoc] =  SAT_WATER[cellLoc] 
                  + (UNSAT_TO_SAT_FL[cellLoc] + SF_WT_TO_SAT_DOWNFLOW[cellLoc] 
                     - SAT_WT_TRANSP[cellLoc] - SAT_TO_UNSAT_FL[cellLoc] - SAT_WT_RECHG[cellLoc]) * DT;

/* sfwat loss priority: downflow to replace groundwater loss, infiltration to unsat, evaporation */
/* calc surface storage state var (m) */
              SURFACE_WAT[cellLoc] = SURFACE_WAT[cellLoc] 
                  + (SF_WT_FROM_RAIN[cellLoc]  
                     - SF_WT_EVAP[cellLoc] - SF_WT_INFILTRATION[cellLoc] - SF_WT_TO_SAT_DOWNFLOW[cellLoc]) * DT;

/******** lastly, update of head calcs, unsat capacity, moisture proportion, etc.
 in order to calc water in diff storages for solute concentrations */
              SAT_WT_HEAD[cellLoc]  = SAT_WATER[cellLoc]/HYD_POROSITY[HAB[cellLoc]];
              UNSAT_DEPTH[cellLoc]  = Max(SED_ELEV[cellLoc]-SAT_WT_HEAD[cellLoc],0.0);
              UNSAT_CAP[cellLoc]  =  UNSAT_DEPTH[cellLoc]*HYD_POROSITY[HAB[cellLoc]];
              if (debug>2) {
                  if (UNSAT_DEPTH[cellLoc] < -0.001) {
                      sprintf(msgStr,"Day %6.1f: capacityERR - neg unsat depth (%f m) in cell (%d,%d) after cell_dyn7!", 
                              TIME, UNSAT_DEPTH[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
                  if ((UNSAT_WATER[cellLoc] - UNSAT_CAP[cellLoc])> 0.001) {
                      sprintf(msgStr,"Day %6.1f: capacityERR - UNSAT_WATER (%f m) exceeds UNSAT_CAP (%f m) in cell (%d,%d) after cell_dyn7!", 
                              TIME, UNSAT_WATER[cellLoc],UNSAT_CAP[cellLoc], ix,iy ); WriteMsg( msgStr,True );  }
              }
              UNSAT_MOIST_PRP[cellLoc]  = 
                  ( UNSAT_CAP[cellLoc]>0.0 ) ? 
                  ( Min(UNSAT_WATER[cellLoc]/UNSAT_CAP[cellLoc],1.0) ) : 
                  ( 1.0); 
              HYD_DOM_ACTWAT_VOL[cellLoc]  = ( Min(DOM_MAXDEPTH[HAB[cellLoc]],UNSAT_DEPTH[cellLoc])*UNSAT_MOIST_PRP[cellLoc] +
                                               Max(DOM_MAXDEPTH[HAB[cellLoc]]-UNSAT_DEPTH[cellLoc], 0.0)*HYD_POROSITY[HAB[cellLoc]] )
                  *CELL_SIZE; 
                  /* flag for presence of small amount of water storage in the active zone must be present */ 
              HYD_DOM_ACTWAT_PRES[cellLoc]  = 
                  ( HYD_DOM_ACTWAT_VOL[cellLoc] > CELL_SIZE*0.01 ) ? 
                  ( 1.0 ) : ( 0.0); 
              HYD_SED_WAT_VOL[cellLoc]  = (SAT_WATER[cellLoc]+UNSAT_WATER[cellLoc])*CELL_SIZE;
              SFWT_VOL[cellLoc]  = SURFACE_WAT[cellLoc]*CELL_SIZE;

              HydTotHd[cellLoc]  = SAT_WT_HEAD[cellLoc]+SURFACE_WAT[cellLoc]; /* only used for reporting purposes */

                  /* at the square of xx% of the macrophyte's height, the manning's n
                     calc will indicate the macrophyte *starting* to bend over,
                     starting to offer increasingly less resistance */
              mann_height = (mann_height_coef*MAC_HEIGHT[cellLoc])*(mann_height_coef*MAC_HEIGHT[cellLoc]); /*oct99 */
              N_density = Max(MAC_MAXROUGH[HAB[cellLoc]] * MAC_REL_BIOM[cellLoc], MAC_MINROUGH[HAB[cellLoc]] );
                  /* manning's n for overland (horiz) flow */
              HYD_MANNINGS_N[cellLoc]  = Max(-Abs((N_density-MAC_MINROUGH[HAB[cellLoc]])
                                                  *(pow(2.0,(1.0-SURFACE_WAT[cellLoc]/mann_height))-1.0) ) 
                                             + N_density,MAC_MINROUGH[HAB[cellLoc]]);

                  /* sum of transpiration for output only, name change dec98*/
              HYD_TRANSP[cellLoc]  = UNSAT_TRANSP[cellLoc]+SAT_WT_TRANSP[cellLoc];

              if (debug > 2) {
                  if (SAT_WT_HEAD[cellLoc] - 0.001 > SED_ELEV[cellLoc] ) {
                      sprintf(msgStr,"Day %6.1f: capacityERR - SAT_WT_HEAD exceeds elev in cell (%d,%d) by %f.", 
                              TIME, ix,iy,(SAT_WT_HEAD[cellLoc] - SED_ELEV[cellLoc]) ); 
                      WriteMsg( msgStr,True );}
                  if (SURFACE_WAT[cellLoc] > 0.2 && UNSAT_DEPTH[cellLoc] > 0.1 ) {
                      sprintf(msgStr,"Day: %6.1f.  Warning - large sfwat depth (%5.2f m) in presence of unsat= %5.2f m, %4.2f %% moist, in cell (%d,%d).", 
                              TIME, SURFACE_WAT[cellLoc], UNSAT_DEPTH[cellLoc],UNSAT_MOIST_PRP[cellLoc], ix,iy ); 
                      WriteMsg( msgStr,True ); }
                  if (UNSAT_WATER[cellLoc] < -0.001) {
                      sprintf(msgStr,"Day %6.1f: capacityERR - neg UNSAT_WATER (%f m) in cell (%d,%d) after cell_dyn7!", 
                              TIME, UNSAT_WATER[cellLoc], ix,iy ); 
                      WriteMsg( msgStr,True );  }
                  if (SAT_WATER[cellLoc] < -MinCheck) {
                      sprintf(msgStr,"Day %6.1f: capacityERR - neg SAT_WATER (%f m) in cell (%d,%d) after cell_dyn7!", 
                              TIME, SAT_WATER[cellLoc], ix,iy ); 
                      WriteMsg( msgStr,True );  }
                  if (SURFACE_WAT[cellLoc] < -MinCheck) {
                      sprintf(msgStr,"Day %6.1f: capacityERR - neg SURFACE_WAT (%f m) in cell (%d,%d) after cell_dyn7!", 
                              TIME, SURFACE_WAT[cellLoc], ix,iy ); 
                      WriteMsg( msgStr,True );  }
              }
            }
	}
    }
 }


/*******
Macrophytes - dynamics (growth, mortality, etc) of macrophyte community/habitat
   MAC_NOPH_BIOMAS[cellLoc] == carbon mass of live Non-Photosynthetic tissues of macrophytes (kgC/m^2)
   MAC_PH_BIOMAS[cellLoc] == carbon mass of live Photosynthetic tissues of macrophytes (kgC/m^2)

   mac_nph_P[cellLoc] == phosphorus mass of live Non-Photosynthetic tissues of macrophytes (kgP/m^2)
   mac_ph_P[cellLoc] == phosphorus mass of live Photosynthetic tissues of macrophytes (kgP/m^2)

   mac_nph_OM[cellLoc] == organic matter mass of live Non-Photosynthetic tissues of macrophytes (kgOM/m^2)
   mac_ph_OM[cellLoc] == organic matter mass of live Photosynthetic tissues of macrophytes (kgOM/m^2)

spatial (cell) location defines habitat ( arrayOf[HAB[cellLoc]] );
habitat type can switch when global dynamic function (cell_dyn1) calls succession function (F_Hab_SwA())

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn8(step)
 int step;
 {
int ix, iy, x, y, cellLoc; 
float reduc, NPP_P, min_litTemp, nphbio_ddep, phbio_ddep, MAC_PHtoNPH, MAC_PHtoNPH_Init;

  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  { 
              
/* various control functions for production calc */
		 MAC_LIGHT_CF[cellLoc]  = SOLRADGRD[cellLoc]/MAC_LIGHTSAT[HAB[cellLoc]]
		 		*Exp(1.0-SOLRADGRD[cellLoc]/MAC_LIGHTSAT[HAB[cellLoc]]);
		 MAC_TEMP_CF[cellLoc]  = tempCF(0, 0.20, MAC_TEMPOPT[HAB[cellLoc]], 5.0, 40.0, AIR_TEMP[cellLoc]);
		 MAC_WATER_CF[cellLoc]  = Min(MAC_WATER_AVAIL_CF[cellLoc], 
		 		Max(1.0-Max((SURFACE_WAT[cellLoc]-MAC_WAT_TOLER[HAB[cellLoc]])
		 		/MAC_WAT_TOLER[HAB[cellLoc]],0.0),0.0));
         /* earlier version included effect of po4 sfwat  */
                 MAC_NUT_CF[cellLoc]  =
                     Exp(-mac_uptake_coef * Max(MAC_KSP[HAB[cellLoc]]-TP_SEDWT_CONCACTMG[cellLoc], 0.0)
                         /MAC_KSP[HAB[cellLoc]]) ; /* mg/L */
/* 		 MAC_NUT_CF[cellLoc]  =  Max( TP_SEDWT_CONCACTMG[cellLoc] */
/*                                               /( TP_SEDWT_CONCACTMG[cellLoc]+MAC_KSP[HAB[cellLoc]]), 0.01) ; */
		 MAC_SALT_CF[cellLoc]  = 1.0; /* tmp, removed this ( MAC_SALIN_THRESH[HAB[cellLoc]]>0.0 ) ? (  Max( 1.0-Max(SAL_MEAN[cellLoc]-MAC_SALIN_THRESH[HAB[cellLoc]],0.0)/MAC_SALIN_THRESH[HAB[cellLoc]],0.0) ) : ( Max(1.0-SAL_MEAN[cellLoc],0.0)); */
                 min_litTemp = Min(MAC_LIGHT_CF[cellLoc], MAC_TEMP_CF[cellLoc]);
                 MAC_PROD_CF[cellLoc]  = Min(min_litTemp,MAC_WATER_CF[cellLoc])
		 		 *MAC_NUT_CF[cellLoc] *MAC_SALT_CF[cellLoc];
/* net primary production, kg C/m2/d */
		 PHBIO_NPP[cellLoc]  = PHBIO_RCNPP[HAB[cellLoc]]*MAC_PROD_CF[cellLoc]*MAC_PH_BIOMAS[cellLoc]
		 		* (1.0-MAC_TOT_BIOM[cellLoc]/MAC_MAX_BIO[cellLoc]);
/* P uptake is dependent on available P and relative to a maximum P:C ratio for the tissue (kg C/m^2/d * P:Cmax * dimless = kgP/m2/d ) */
                 NPP_P = PHBIO_NPP[cellLoc]  * PHBIO_PC[HAB[cellLoc]]  * Max(MAC_NUT_CF[cellLoc]*2.0,1.0)
                     * Max(1.0-mac_ph_PC[cellLoc]/PHBIO_PC[HAB[cellLoc]], 0.0);
/* check for available P mass that will be taken up from sed water in active zone (nutCF does not) */
                 reduc = (NPP_P > 0.0) ? 
                     (TP_SED_WT_AZ[cellLoc] / ( NPP_P*CELL_SIZE*DT) ) : /* oct99 within-plant variable stoichiometry */
                     (1.0);
		/* reduce the production proportionally if excess demand is found */
                /* can have high conc, but low mass of P avail, in presence of high plant biomass and high demand */
                 if (reduc < 1.0) {
                     PHBIO_NPP[cellLoc] *= reduc;
                     NPP_P *= reduc;
                 }
                 
/* losses from photobio */
		 phbio_ddep = Max(1.0-Max( (PHBIO_SAT[cellLoc]-MAC_PH_BIOMAS[cellLoc]) 
                                           /(PHBIO_SAT[cellLoc]-PHBIO_REFUGE[cellLoc]),0.0),0.0);
                 PHBIO_AVAIL[cellLoc]  = MAC_PH_BIOMAS[cellLoc]*phbio_ddep;
		 PHBIO_FIRE[cellLoc]  = 
		 		( PHBIO_AVAIL[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*DT>PHBIO_AVAIL[cellLoc] ) ? 
		 		( PHBIO_AVAIL[cellLoc]/DT ) : 
		 		( PHBIO_AVAIL[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]);
		 PHBIO_PR1[cellLoc]  = PHBIO_FIRE[cellLoc];

                 MAC_PHtoNPH_Init = PHBIO_MAX[HAB[cellLoc]] / NPHBIO_MAX[HAB[cellLoc]] ; /*if habitat type changes */
                 MAC_PHtoNPH = (MAC_NOPH_BIOMAS[cellLoc]>0.0) ? ( MAC_PH_BIOMAS[cellLoc] / MAC_NOPH_BIOMAS[cellLoc]) : (MAC_PHtoNPH_Init);
                 
                 NPHBIO_TRANSLOC_POT[cellLoc]  = (MAC_PHtoNPH>MAC_PHtoNPH_Init) ?
                     (exp(Mac_transl_rc[HAB[cellLoc]] *(MAC_PHtoNPH-MAC_PHtoNPH_Init)) - 1.0) :
                     (0.0); 
		 NPHBIO_TRANSLOC[cellLoc]  =  
		 		( (NPHBIO_TRANSLOC_POT[cellLoc]+PHBIO_PR1[cellLoc])*DT >PHBIO_AVAIL[cellLoc] ) ? 
		 		( (PHBIO_AVAIL[cellLoc]-PHBIO_PR1[cellLoc]*DT)/DT ) : 
		 		( NPHBIO_TRANSLOC_POT[cellLoc]);


/* losses from non-photobio  */
		 nphbio_ddep = Max(1.0-Max((NPHBIO_SAT[cellLoc]-MAC_NOPH_BIOMAS[cellLoc])
                                          /(NPHBIO_SAT[cellLoc]-NPHBIO_REFUGE[cellLoc]),0.0),0.0);
		 NPHBIO_AVAIL[cellLoc]  = MAC_NOPH_BIOMAS[cellLoc]*nphbio_ddep; 
		 NPHBIO_FIRE[cellLoc]  = 
		 		( NPHBIO_AVAIL[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*NPHBIO_ABVBEL[HAB[cellLoc]]*DT >NPHBIO_AVAIL[cellLoc] ) ? 
		 		( NPHBIO_AVAIL[cellLoc]/DT ) : 
		 		( NPHBIO_AVAIL[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*NPHBIO_ABVBEL[HAB[cellLoc]]);
		 NPHBIO_PR1[cellLoc]  = NPHBIO_FIRE[cellLoc];

                 PHBIO_TRANSLOC_POT[cellLoc]  = (MAC_PHtoNPH<MAC_PHtoNPH_Init) ?
                     (exp(Mac_transl_rc[HAB[cellLoc]] *(MAC_PHtoNPH_Init-MAC_PHtoNPH)) - 1.0) :
                     (0.0); 
		 PHBIO_TRANSLOC[cellLoc]  =  
		 		( (PHBIO_TRANSLOC_POT[cellLoc]+NPHBIO_PR1[cellLoc])*DT >NPHBIO_AVAIL[cellLoc] ) ? 
		 		( (NPHBIO_AVAIL[cellLoc]-NPHBIO_PR1[cellLoc]*DT)/DT ) : 
		 		( PHBIO_TRANSLOC_POT[cellLoc]);
		 NPHBIO_PR2[cellLoc]  = NPHBIO_PR1[cellLoc]+PHBIO_TRANSLOC[cellLoc];

		 NPHBIO_PR3[cellLoc]  = NPHBIO_PR2[cellLoc];
		 NPHBIO_MORT_POT[cellLoc]  = NPHBIO_AVAIL[cellLoc]*PHBIO_RCMORT[HAB[cellLoc]]
                     * (1.0 + Max(1.0-MAC_PH_BIOMAS[cellLoc]/PHBIO_MAX[HAB[cellLoc]],0.0) )/2.0; /*oct99 decreased mort w/ increasing photobio */
		 NPHBIO_MORT[cellLoc]  =
                                ( (NPHBIO_PR3[cellLoc]+NPHBIO_MORT_POT[cellLoc])*DT>NPHBIO_AVAIL[cellLoc] ) ? 
		 		( (NPHBIO_AVAIL[cellLoc]-NPHBIO_PR3[cellLoc]*DT)/DT ) : 
		 		( NPHBIO_MORT_POT[cellLoc]);
                 

/* calc nonphotosynthetic biomass state var (kgC/m2) */
		 MAC_NOPH_BIOMAS[cellLoc] =  MAC_NOPH_BIOMAS[cellLoc] 
                     + (NPHBIO_TRANSLOC[cellLoc] - NPHBIO_MORT[cellLoc] 
                        - PHBIO_TRANSLOC[cellLoc] - NPHBIO_FIRE[cellLoc] ) * DT;
                 
/* further loss calcs for photobio */
		 PHBIO_PR2[cellLoc]  = NPHBIO_TRANSLOC[cellLoc]+PHBIO_PR1[cellLoc];
		 PHBIO_MORT_POT[cellLoc]  = PHBIO_RCMORT[HAB[cellLoc]] *PHBIO_AVAIL[cellLoc] 
		 		*(1.0 + (1.0-MAC_WATER_AVAIL_CF[cellLoc]) )/2.0;
		 PHBIO_MORT[cellLoc]  =
                                ( (PHBIO_MORT_POT[cellLoc]+PHBIO_PR2[cellLoc])*DT>PHBIO_AVAIL[cellLoc] ) ? 
		 		( (PHBIO_AVAIL[cellLoc]-PHBIO_PR2[cellLoc]*DT)/DT ) : 
		 		( PHBIO_MORT_POT[cellLoc]);

/* calc nonphotosynthetic biomass state var (kgC/m2) */
		 MAC_PH_BIOMAS[cellLoc] = MAC_PH_BIOMAS[cellLoc] 
		 		+ (PHBIO_TRANSLOC[cellLoc] + PHBIO_NPP[cellLoc] 
		 		 - PHBIO_FIRE[cellLoc] - PHBIO_MORT[cellLoc] - NPHBIO_TRANSLOC[cellLoc]) * DT;

/* book-keeping calcs used in other modules */
		 MAC_TOT_BIOM[cellLoc]  = MAC_PH_BIOMAS[cellLoc]+MAC_NOPH_BIOMAS[cellLoc];
		 MAC_REL_BIOM[cellLoc]  = 
		 		( MAC_TOT_BIOM[cellLoc] > 0.0 ) ? 
		 		MAC_TOT_BIOM[cellLoc]/MAC_MAX_BIO[cellLoc] : 
		 		0.0001;
		 MAC_HEIGHT[cellLoc]  = pow(MAC_REL_BIOM[cellLoc],0.33)*MAC_MAXHT[HAB[cellLoc]];
		 MAC_LAI[cellLoc]  = MAC_REL_BIOM[cellLoc]*MAC_MAXLAI[HAB[cellLoc]];
/* note that an "effective" LAI that accounts for water depth is calc'd in hydro module */

/*  now calc the P and organic matter associated with the C fluxes */
                /*  phbio_npp_P[cellLoc] = PHBIO_NPP[cellLoc] * PHBIO_PC[HAB[cellLoc]];  */    /* habitat-specfic stoichiometry */
                 phbio_npp_P[cellLoc] = NPP_P;     /* oct99 within-plant variable stoichiometry */
                 phbio_npp_OM[cellLoc] = PHBIO_NPP[cellLoc] / PHBIO_CTOOM[HAB[cellLoc]]; /* habitat-specfic stoichiometry */

                 phbio_fire_P[cellLoc] = PHBIO_FIRE[cellLoc] * mac_ph_PC[cellLoc];
                 phbio_fire_OM[cellLoc] = PHBIO_FIRE[cellLoc] / mac_ph_CtoOM[cellLoc];

                 phbio_mort_P[cellLoc] = PHBIO_MORT[cellLoc] * mac_ph_PC[cellLoc];
                 phbio_mort_OM[cellLoc] = PHBIO_MORT[cellLoc] / mac_ph_CtoOM[cellLoc];

                 phbio_transl_P[cellLoc] = PHBIO_TRANSLOC[cellLoc] * mac_nph_PC[cellLoc];
                 phbio_transl_OM[cellLoc] = PHBIO_TRANSLOC[cellLoc] / mac_nph_CtoOM[cellLoc];

                 nphbio_transl_P[cellLoc] = NPHBIO_TRANSLOC[cellLoc] * mac_ph_PC[cellLoc];
                 nphbio_transl_OM[cellLoc] = NPHBIO_TRANSLOC[cellLoc] / mac_ph_CtoOM[cellLoc];

                 nphbio_fire_P[cellLoc] = NPHBIO_FIRE[cellLoc] * mac_nph_PC[cellLoc];
                 nphbio_fire_OM[cellLoc] = NPHBIO_FIRE[cellLoc] / mac_nph_CtoOM[cellLoc];
                 
                 nphbio_mort_P[cellLoc] = NPHBIO_MORT[cellLoc] * mac_nph_PC[cellLoc];
                 nphbio_mort_OM[cellLoc] = NPHBIO_MORT[cellLoc] / mac_nph_CtoOM[cellLoc];


/* state vars: now calc the P and OM associated with those C state vars */
                 mac_nph_P[cellLoc] = mac_nph_P[cellLoc]
                     + (nphbio_transl_P[cellLoc] - nphbio_mort_P[cellLoc]
                        - phbio_transl_P[cellLoc] - nphbio_fire_P[cellLoc] ) * DT;
                 mac_nph_PC[cellLoc] = ( (MAC_NOPH_BIOMAS[cellLoc] > 0.0) && (mac_nph_P[cellLoc] > 0.0)) ?
                     (mac_nph_P[cellLoc] / MAC_NOPH_BIOMAS[cellLoc]) : /*dec99 these second mass checks not needed now (were probably not good to start with) */
                     0.3 * NPHBIO_PC[HAB[cellLoc]]; /* default to 0.3 of max for habitat */
                 mac_nph_PC_rep[cellLoc] = (float)mac_nph_PC[cellLoc] * 1.0e6; /* new variable in float instead of double, for output compatibility */
                 

                 mac_nph_OM[cellLoc] = mac_nph_OM[cellLoc]
                     + (nphbio_transl_OM[cellLoc] - nphbio_mort_OM[cellLoc]
                        - phbio_transl_OM[cellLoc] - nphbio_fire_OM[cellLoc] ) * DT;
                 mac_nph_CtoOM[cellLoc] = ( (mac_nph_OM[cellLoc] > 0.0) && (MAC_NOPH_BIOMAS[cellLoc]>0.0)) ?
                     (MAC_NOPH_BIOMAS[cellLoc] / mac_nph_OM[cellLoc]) :
                     NPHBIO_CTOOM[HAB[cellLoc]];

                 mac_ph_P[cellLoc] = mac_ph_P[cellLoc]
                     + (phbio_transl_P[cellLoc] + phbio_npp_P[cellLoc] - phbio_mort_P[cellLoc]
                        - nphbio_transl_P[cellLoc] - phbio_fire_P[cellLoc] ) * DT;
                 mac_ph_PC[cellLoc] = ( (MAC_PH_BIOMAS[cellLoc] > 0.0) && (mac_ph_P[cellLoc]>0.0)) ?
                     (mac_ph_P[cellLoc] / MAC_PH_BIOMAS[cellLoc]) :
                     0.3 * PHBIO_PC[HAB[cellLoc]]; /* default to 0.3 of max for habitat */
                 mac_ph_PC_rep[cellLoc] = (float)mac_ph_PC[cellLoc] * 1.0e6; /* new variable in float instead of double, for output compatibility */

                 mac_ph_OM[cellLoc] = mac_ph_OM[cellLoc]
                     + (phbio_transl_OM[cellLoc] + phbio_npp_OM[cellLoc] - phbio_mort_OM[cellLoc]
                        - nphbio_transl_OM[cellLoc] - phbio_fire_OM[cellLoc] ) * DT;
                 mac_ph_CtoOM[cellLoc] = ( (mac_ph_OM[cellLoc] > 0.0) && (MAC_PH_BIOMAS[cellLoc]>0.0)) ?
                     (MAC_PH_BIOMAS[cellLoc] / mac_ph_OM[cellLoc]) :
                     PHBIO_CTOOM[HAB[cellLoc]];

                 if (debug > 0 && MAC_NOPH_BIOMAS[cellLoc] < -MinCheck) { sprintf(msgStr,"Day %6.1f: ERROR - neg NPhoto C biomass (%f m) in cell (%d,%d)!", 
                              TIME, MAC_NOPH_BIOMAS[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
                 if (debug > 0 && MAC_PH_BIOMAS[cellLoc] < -MinCheck)  { sprintf(msgStr,"Day %6.1f: ERROR - neg Photo C biomass (%f m) in cell (%d,%d)!", 
                              TIME, MAC_PH_BIOMAS[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
                 if (debug > 0 && mac_nph_P[cellLoc] < -MinCheck) { sprintf(msgStr,"Day %6.1f: ERROR - neg NPhoto P biomass (%f m) in cell (%d,%d)!", 
                              TIME, mac_nph_P[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
                 if (debug > 0 && mac_ph_P[cellLoc] < -MinCheck)  { sprintf(msgStr,"Day %6.1f: ERROR - neg Photo P biomass (%f m) in cell (%d,%d)!", 
                              TIME, mac_ph_P[cellLoc], ix,iy ); WriteMsg( msgStr,True );}


/******** phosphorus removed from water here *************/
		 TP_SEDWT_UPTAKE[cellLoc]  = phbio_npp_P[cellLoc]*CELL_SIZE; 
/* recalc P in sed water state variable (kg P) */
		 TP_SED_WT[cellLoc] =  TP_SED_WT[cellLoc] - (TP_SEDWT_UPTAKE[cellLoc]) * DT;
		 TP_SED_CONC[cellLoc] = (HYD_SED_WAT_VOL[cellLoc]>0.0) ?
                     (TP_SED_WT[cellLoc] / HYD_SED_WAT_VOL[cellLoc]) :
                  (0.0);
                 
                      /* this is the active zone, where uptake, sorption, and mineralization take place */
                TP_SED_WT_AZ[cellLoc] =  TP_SED_WT_AZ[cellLoc] - (TP_SEDWT_UPTAKE[cellLoc]) * DT;
                 TP_SEDWT_CONCACT[cellLoc] = ( HYD_DOM_ACTWAT_PRES[cellLoc] > 0.0) ?
                     ( TP_SED_WT_AZ[cellLoc]/HYD_DOM_ACTWAT_VOL[cellLoc] ) :
                     (TP_SED_CONC[cellLoc]); /* g/L */
                 TP_SEDWT_CONCACTMG[cellLoc]  = TP_SEDWT_CONCACT[cellLoc]*1000.0; /* mg/L */
              
          }
	}
  }
}


/*******
Phosphorus dynamics in water and sorbed to soil

TP_SF_WT[cellLoc] == mass of P in surface water (kg P)
TP_SED_WT[cellLoc] == mass of P in sediment/soil (pore) water (kg P)
TP_SORB[cellLoc] == mass of P sorbed to soil (kg P)

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn9(step)
 int step;
 {
 int ix, iy, x, y, cellLoc; 
 float TP_SF_pr1, TPpartic, TPsettlRat, TP_settl_pot;
 double PO4Pconc, PO4P, nonPO4Pconc;
 
  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  {
/* calc concentrations after horiz fluxes */
              TP_SFWT_CONC[cellLoc]  = 
                  ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                  ( TP_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                  ( 0.0); /* used in P fluxes for mass balance */
             /*oct99 using regression for predicting PO4 from TP */
              PO4Pconc =  Max( TP_SFWT_CONC[cellLoc]*PO4toTP + 0.001*PO4toTPint,0.0);  /* g/l  (note that intercept may be <0)*/
      /* after spatial (horiz) fluxes, recalc the active zone P mass based on old active/total ratio */
              TP_SED_CONC[cellLoc] = (HYD_SED_WAT_VOL[cellLoc]>0.0) ?
                  (TP_SED_WT[cellLoc] / HYD_SED_WAT_VOL[cellLoc]) :
                  (0.0);
            /* this is the active zone, where uptake, sorption, and mineralization take place */
             TP_SED_WT_AZ[cellLoc] = TP_SED_CONC[cellLoc] * TP_Act_to_Tot[cellLoc] * HYD_DOM_ACTWAT_VOL[cellLoc];
              TP_SEDWT_CONCACT[cellLoc] =(HYD_DOM_ACTWAT_PRES[cellLoc] > 0.0) ?
                  ( TP_SED_WT_AZ[cellLoc]/HYD_DOM_ACTWAT_VOL[cellLoc] ) :
                  ( TP_SED_CONC[cellLoc]);

/* inputs to surf  P (kg P) */
              TP_FR_RAIN[cellLoc]  = SF_WT_FROM_RAIN[cellLoc]*CELL_SIZE*TP_IN_RAIN*0.001;

/* calc various loss and/or vertical exchange potentials (kg P) */
              
              TP_UPFLOW_POT[cellLoc]  = SAT_SURF_UP_FL[cellLoc]*CELL_SIZE*TP_SEDWT_CONCACT[cellLoc] 
                  + Max((TP_SEDWT_CONCACT[cellLoc]-PO4Pconc)
                        *TP_DIFFCOEF*8.64/TP_DIFFDEPTH*CELL_SIZE,0.0); /* was multiplied by depth dec99 */
              TP_UPFLOW[cellLoc]  = 
                  ( (TP_UPFLOW_POT[cellLoc])*DT>TP_SED_WT_AZ[cellLoc] ) ? 
                  ( (TP_SED_WT_AZ[cellLoc])/DT ) : 
                  ( TP_UPFLOW_POT[cellLoc]);
              TP_SEDWT_PR1[cellLoc]  = TP_UPFLOW[cellLoc];

              TP_K[cellLoc]  = Max(TP_K_SLOPE*TP_SORBCONC[cellLoc]+TP_K_INTER,0.0);
/*fix rate oct99*/
              TP_SORB_POT[cellLoc]  = 
                  ( HYD_DOM_ACTWAT_PRES[cellLoc]>0.0 ) ? 
                  ( (double) 0.001 
                    *(TP_K[cellLoc]*(pow(Max(TP_SEDWT_CONCACT[cellLoc],0.0),0.8) )
                      *0.001*(DEPOS_ORG_MAT[cellLoc]*CELL_SIZE+DIM[cellLoc])-TP_SORB[cellLoc] ) ) : 
                  ( 0.0);
              if (TP_SORB_POT[cellLoc]>0.0) {
                  TP_SORBTION[cellLoc]  =  
                      ( (TP_SORB_POT[cellLoc]+TP_SEDWT_PR1[cellLoc])*DT>TP_SED_WT_AZ[cellLoc] ) ? 
                      ( (TP_SED_WT_AZ[cellLoc]-TP_SEDWT_PR1[cellLoc]*DT)/DT ) : 
                      ( TP_SORB_POT[cellLoc]);
              }
              else { /* neg sorption, loss from sorb variable*/
                  TP_SORBTION[cellLoc]  =  
                      ( (-TP_SORB_POT[cellLoc])*DT>TP_SORB[cellLoc] ) ? 
                      ( (-TP_SORB[cellLoc])/DT ) : 
                      ( TP_SORB_POT[cellLoc]);
              }
              

              TP_DNFLOW_POT[cellLoc]  = (SF_WT_INFILTRATION[cellLoc]+SF_WT_TO_SAT_DOWNFLOW[cellLoc])
                  *CELL_SIZE*TP_SFWT_CONC[cellLoc]  /* oct99 was the state var, NOT conc!!!, dec99 now replaced with TPconc */ 
                  + Max((PO4Pconc-TP_SEDWT_CONCACT[cellLoc])
                        *TP_DIFFCOEF*8.64/TP_DIFFDEPTH*CELL_SIZE,0.0) ;/* was multiplied by depth dec99 */
              TP_DNFLOW[cellLoc]  =  
                  ( ( TP_DNFLOW_POT[cellLoc])*DT > TP_SF_WT[cellLoc] ) ? 
                  ( ( TP_SF_WT[cellLoc])/DT ) : 
                  ( TP_DNFLOW_POT[cellLoc]);
/* calc P in sed water state variables (kg P) */
              TP_SED_WT[cellLoc] =  TP_SED_WT[cellLoc] +
                  ( TP_DNFLOW[cellLoc] - TP_UPFLOW[cellLoc] - TP_SORBTION[cellLoc]) * DT;
             /* this is the active zone, where uptake, sorption, and mineralization take place */
              TP_SED_WT_AZ[cellLoc] =  TP_SED_WT_AZ[cellLoc] +
                  (TP_DNFLOW[cellLoc] - TP_UPFLOW[cellLoc] - TP_SORBTION[cellLoc]) * DT;

/* calc P in surface water state variable (kg P) */
              TP_SF_WT[cellLoc] = TP_SF_WT[cellLoc] + 
                  (TP_UPFLOW[cellLoc] + TP_FR_RAIN[cellLoc] 
                   - TP_DNFLOW[cellLoc]) * DT;

/* calc P sorbed-to-soil state variable (kg P) */
              TP_SORB[cellLoc] = TP_SORB[cellLoc] + (TP_SORBTION[cellLoc]) * DT;

              TP_SED_CONC[cellLoc] = (HYD_SED_WAT_VOL[cellLoc]>0.0) ?
                  (TP_SED_WT[cellLoc] / HYD_SED_WAT_VOL[cellLoc]) :
                  (0.0); /* g/L */
              TP_SEDWT_CONCACT[cellLoc] = ( HYD_DOM_ACTWAT_PRES[cellLoc] > 0.0) ?
                  ( TP_SED_WT_AZ[cellLoc]/HYD_DOM_ACTWAT_VOL[cellLoc] ) :
                  (TP_SED_CONC[cellLoc]); /* g/L */
              TP_SEDWT_CONCACTMG[cellLoc]  = TP_SEDWT_CONCACT[cellLoc]*1000.0;

              TP_SORBCONC[cellLoc] = ((DEPOS_ORG_MAT[cellLoc]*CELL_SIZE + DIM[cellLoc])>0.0) ?
                  ( (double) TP_SORB[cellLoc]*1000.0 / (DEPOS_ORG_MAT[cellLoc]*CELL_SIZE + DIM[cellLoc]) ) :
                  (0.0); /* gP/kgsoil */

              

              TP_SFWT_CONC[cellLoc]  = 
                  ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                  ( TP_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                  ( 0.0); /* g/L used in P fluxes for mass balance */
              TP_SFWT_CONC_MG[cellLoc]  = 
                  ( SURFACE_WAT[cellLoc] > DetentZ ) ? 
                  (TP_SFWT_CONC[cellLoc]*1000.0) : 
                  (0.0); /* used for reporting and other modules to evaluate P conc when water is present */

/* the following is an empirical method to calculate settling of particulate P out of the water column
   into the sediments.  It uses the fixed ratio of PO4 to TP, but allows for a decreasing proportion of
   TP to be in (large) particulate form as TP concentration drops below a chosen threshold - the sum of
   the TP is considered to be dissolved plus large particulate plus small particulate (that cannot settle out) */
                  /* mass (kg) of particulate fraction of TP, available for settling to sediments */
                  /* using regression for predicting PO4 from TP */
              PO4Pconc =  Max(TP_SFWT_CONC_MG[cellLoc]*PO4toTP + PO4toTPint,0.0);  /* mg/l (note that intercept may be <0)*/
              nonPO4Pconc = Max(TP_SFWT_CONC_MG[cellLoc]-PO4Pconc,0.0); /* non-PO4 conc, mg/l */
              TPpartic = nonPO4Pconc * (1.0-exp(-nonPO4Pconc/TPpart_thresh)) *0.001 * SFWT_VOL[cellLoc] ; /* kg particulate P */


                  /* settling rate, 1/d */
              TPsettlRat = ( SURFACE_WAT[cellLoc] > DetentZ ) ?
                  (settlVel/SURFACE_WAT[cellLoc]) :
                  0.0;
              
                  /* potential settling of particulate TP (kg/d) */
              TP_settl_pot = TPsettlRat * TPpartic;
              TP_settl[cellLoc]  =  
                  ( ( TP_settl_pot)*DT > TPpartic ) ? 
                  ( (TPpartic)/DT ) : 
                  ( TP_settl_pot);
/* calc P in surface water state variable (kg P) */
              TP_SF_WT[cellLoc] = TP_SF_WT[cellLoc] - TP_settl[cellLoc] * DT;

/* various book-keeping calcs used in other modules */
/* conc surf and sed wat = kgP/m3==gP/L, another var calc for mg/L */
              TP_SFWT_CONC[cellLoc]  = 
                  ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                  ( TP_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                  ( 0.0); /* used in P fluxes for mass balance */
              TP_SFWT_CONC_MG[cellLoc]  = 
                  ( SURFACE_WAT[cellLoc] > DetentZ ) ? 
                  (TP_SFWT_CONC[cellLoc]*1000.0) : 
                  (0.0); /* used for reporting and other modules to evaluate P conc when water is present */
              
              if (debug > 0 && TP_SF_WT[cellLoc] < -MinCheck) { sprintf(msgStr,"Day %6.1f: ERROR - neg TP_SF_WT (%f m) in cell (%d,%d)!", 
                                                                    TIME, TP_SF_WT[cellLoc], ix,iy ); WriteMsg( msgStr,True ); usrErr( msgStr ); }
              if (debug > 0 && TP_SED_WT_AZ[cellLoc] < -MinCheck)  { sprintf(msgStr,"Day %6.1f: ERROR - neg TP_SED_WT_AZ (%f m) in cell (%d,%d)!", 
                                                                      TIME, TP_SED_WT_AZ[cellLoc], ix,iy ); WriteMsg( msgStr,True ); usrErr( msgStr ); }

	  }
	}
  }
}

/*******
Everglades Water Quality Model net settling rate algorithm for TP loss
this should be run WITHOUT running ecological modules:
THIS SHOULD ONLY BE RUN WITH HYDROLOGY (& tracer/salt) - DO NOT RUN BIO/CHEM MODULES
phosphorus input from rain and loss from settling out of water column

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn13(step)
 int step;
 {
 int ix, iy, x, y, cellLoc; 
 float TPsettlRat, TP_settl_pot;
 
  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) { 

	  if(ON_MAP[cellLoc= T(ix,iy)])  {
/* concentration of P in surface water used in P fluxes for mass transfer (kgP/m3==gP/L) */
              TP_SFWT_CONC[cellLoc]  = 
                  ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                  ( TP_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                  ( 0.0); 
/* rainfall inputs to surface water P (m * m^2 * g/m^3 * kg/g == kg P) */
              TP_FR_RAIN[cellLoc]  = SF_WT_FROM_RAIN[cellLoc]
                  *CELL_SIZE*TP_IN_RAIN*0.001;              
/* TP settling rate calculation (m/d) = 0 if water depth (m) less than depth threshold parm */
              if (SURFACE_WAT[cellLoc] > WQMthresh ) {
                  TPsettlRat = WQMsettlVel[cellLoc];
                  TP_settlDays[cellLoc]++;
              }
              else
                  TPsettlRat = 0.0;
              
/* before we had to put in the day accumulator*/
/*               TPsettlRat = ( SURFACE_WAT[cellLoc] > WQMthresh ) ?  */
/*                   (WQMsettlVel[cellLoc]) : 0.0; */
/* potential settling of particulate TP (m/d * m^2 * kg/m^3 = kg/d) */
              TP_settl_pot = TPsettlRat * CELL_SIZE * TP_SFWT_CONC[cellLoc];

/*  like EWQM, shows mass bal error when ->   TP_settl[cellLoc]  =   TP_settl_pot; */
              
/* constrain settling to be no more than kg P available in water column */
              TP_settl[cellLoc]  =   
                  ( ( TP_settl_pot)*DT > TP_SF_WT[cellLoc] ) ?  
                  ( (TP_SF_WT[cellLoc])/DT ) :  
                  ( TP_settl_pot); 
/* calc P in surface water state variable (kg P) */
              TP_SF_WT[cellLoc] = TP_SF_WT[cellLoc] +
                  ( TP_FR_RAIN[cellLoc] - TP_settl[cellLoc] ) * DT;

/* this was in EWQM:              if (TP_SF_WT[cellLoc]<0.0) TP_SF_WT[cellLoc]=0.0; */
              
/* concentration of P in surface water used in P fluxes for mass transfer (kgP/m3==gP/L) */
              TP_SFWT_CONC[cellLoc]  = 
                  ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                  ( TP_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                  ( 0.0); 
/* concentration used for reporting (e.g., in maps) when water is present. 
   The hydraulic detention depth parm also used for all
   concentration reporting thresholds (currently 1 cm)
*/
              TP_SFWT_CONC_MG[cellLoc]  = 
                  ( SURFACE_WAT[cellLoc] > DetentZ ) ?  
                  (TP_SFWT_CONC[cellLoc]*1000.0) : 
                  (0.0); 
	  }
	}
  }
}



/*******
salt dynamics in water

SALT_SURF_WT[cellLoc] == mass of salt dissolved in surface water (kg salt)
SALT_SED_WT[cellLoc] == mass of salt dissolved in sediment/soil (pore) water (kg salt)

this module can be used as a conservative tracer if salinity is not important to objectives;
replicate this module so salinity and an independent, conservative tracer, are desired

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn10(step)
 int step;
 {
int ix, iy, x, y, cellLoc;
 double SALT_SED_TO_SF_FLOW_pot;
 

  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  {
		 SAL_SF_WT_mb[cellLoc]  = 
                     ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                     ( SALT_SURF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                     ( 0.0); /* used in salt fluxes for mass balance */
		 SAL_SED_WT[cellLoc]  = 
                     ( HYD_SED_WAT_VOL[cellLoc]>0.0 ) ? 
                     ( SALT_SED_WT[cellLoc]/HYD_SED_WAT_VOL[cellLoc] ) : 
                     ( 0.0);

                 SALT_SFWAT_DOWNFL_POT[cellLoc]  = (SF_WT_INFILTRATION[cellLoc] + SF_WT_TO_SAT_DOWNFLOW[cellLoc])
                     * CELL_SIZE*SAL_SF_WT_mb[cellLoc]
                     + Max((SAL_SF_WT_mb[cellLoc]-SAL_SED_WT[cellLoc])
                           * TP_DIFFCOEF*8.64/TP_DIFFDEPTH*CELL_SIZE,0.0)  ; /*diffusion parms same as P */
		 SALT_SFWAT_DOWNFL[cellLoc]  =  
                     ( SALT_SFWAT_DOWNFL_POT[cellLoc]*DT>SALT_SURF_WT[cellLoc] ) ? 
                     ( SALT_SURF_WT[cellLoc]/DT ) : 
                     ( SALT_SFWAT_DOWNFL_POT[cellLoc]);

                 SALT_SED_TO_SF_FLOW_pot  =  
                    SAT_SURF_UP_FL[cellLoc]*CELL_SIZE*SAL_SED_WT[cellLoc]
                     + Max((SAL_SED_WT[cellLoc]-SAL_SF_WT_mb[cellLoc])
                            *TP_DIFFCOEF*8.64/TP_DIFFDEPTH*CELL_SIZE,0.0)  ; /*diffusion parms same as P */
                 SALT_SED_TO_SF_FLOW[cellLoc]  =  
                     ( SALT_SED_TO_SF_FLOW_pot*DT>SALT_SED_WT[cellLoc] ) ? 
                     ( SALT_SED_WT[cellLoc]/DT ) : 
                     ( SALT_SED_TO_SF_FLOW_pot );
/* calc state vars (kg salt) */
		 SALT_SED_WT[cellLoc] =  SALT_SED_WT[cellLoc]  
                     + (SALT_SFWAT_DOWNFL[cellLoc] - SALT_SED_TO_SF_FLOW[cellLoc]) * DT;

		 SALT_SURF_WT[cellLoc] = SALT_SURF_WT[cellLoc] 
                     + (SALT_SED_TO_SF_FLOW[cellLoc] - SALT_SFWAT_DOWNFL[cellLoc] ) * DT;

/* book-keeping concentration calcs, (kg/m3==g/L==ppt) used in other modules */
		 SAL_SF_WT_mb[cellLoc]  = 
                     ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                     ( SALT_SURF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                     ( 0.0); /* used in salt fluxes for mass balance */
		 SAL_SF_WT[cellLoc]  = 
                     ( SURFACE_WAT[cellLoc] > DetentZ ) ? 
                     ( SAL_SF_WT_mb[cellLoc]  ) : 
                     ( 0.0); /* used for reporting and other modules to evaluate salinity when water is present */
		 SAL_SED_WT[cellLoc]  = 
                     ( HYD_SED_WAT_VOL[cellLoc]>0.0 ) ? 
                     ( SALT_SED_WT[cellLoc]/HYD_SED_WAT_VOL[cellLoc] ) : 
                     ( 0.0);

		 SAL_MEAN[cellLoc]  = (SAL_SED_WT[cellLoc]+SAL_SF_WT[cellLoc])/2.0;
	  }
	}
  }
                  
}



/*******
Standing detritus dynamics

STAND_DETRITUS[cellLoc] == carbon mass of standing dead detritus of macrophytes (kg C/m^2)

stdet_P[cellLoc] == phosphorus mass of standing dead detritus of macrophytes (kg P/m^2)

stdet_OM[cellLoc] == organic matter mass of standing dead detritus of macrophytes (kg OM/m^2)

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn11(step)
 int step;
 {
int ix, iy, x, y, cellLoc; 

  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  {
/* calc inputs (kgC/m2) */
		 STDET_PROD[cellLoc]  =   (1.0-DOM_PHBIO_PROP[HAB[cellLoc]])*PHBIO_MORT[cellLoc]
                     + (1.0-DOM_NPHBIO_PROP[HAB[cellLoc]]) *NPHBIO_MORT[cellLoc];
/* calc loss potentials (kgC/m2) */
/* 		 STDET_BURN_POT[cellLoc]  = STAND_DETRITUS[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*STDET_ABVBEL[HAB[cellLoc]]; */
/* 		 STDET_BURN[cellLoc]  =   */
/*                      ( STDET_BURN_POT[cellLoc]*DT>STAND_DETRITUS[cellLoc] ) ?  */
/*                      ( STAND_DETRITUS[cellLoc]/DT ) :  */
/*                      ( STDET_BURN_POT[cellLoc]); */
/* 		 STDET_PR1[cellLoc]  = STDET_BURN[cellLoc]; */

		 STDET_TO_DOM_POT[cellLoc]  = STAND_DETRITUS[cellLoc]*StDetLos 
                     * Max(1.0-Max(WIND_STORM[cellLoc]-WIND_SPEED_CANOPY[cellLoc],0.0)
                           /(WIND_STORM[cellLoc]-STDET_WIND[HAB[cellLoc]]),0.0);
		 STDET_TO_DOM[cellLoc]  = 
                     ( (/* STDET_PR1[cellLoc]+ */STDET_TO_DOM_POT[cellLoc])*DT>STAND_DETRITUS[cellLoc] ) ? 
                     ( (STAND_DETRITUS[cellLoc]/* -STDET_PR1[cellLoc]*DT */)/DT ) : 
                     ( STDET_TO_DOM_POT[cellLoc]);

/* calc state var (kgC/m2) */
		 STAND_DETRITUS[cellLoc] =  STAND_DETRITUS[cellLoc] 
                     + (STDET_PROD[cellLoc] /* - STDET_BURN[cellLoc]  */- STDET_TO_DOM[cellLoc]) * DT;
                 
/* calc avail to fire and consumers */
		 STDET_AVAIL[cellLoc]  = STAND_DETRITUS[cellLoc]
		 		* Max(1.0-Max((STDET_SAT[cellLoc]-STAND_DETRITUS[cellLoc])/ (STDET_SAT[cellLoc]-STDET_REFUG[cellLoc]),0.0),0.0);

/*  now calc the P and organic matter associated with those C fluxes */
                 stdet_prod_P[cellLoc] = (1.0-DOM_PHBIO_PROP[HAB[cellLoc]])*phbio_mort_P[cellLoc]
		 		+ (1.0-DOM_NPHBIO_PROP[HAB[cellLoc]]) *nphbio_mort_P[cellLoc];
                 stdet_prod_OM[cellLoc] = (1.0-DOM_PHBIO_PROP[HAB[cellLoc]]) * phbio_mort_OM[cellLoc]
		 		+ (1.0-DOM_NPHBIO_PROP[HAB[cellLoc]]) * nphbio_mort_OM[cellLoc];

/*                  stdet_burn_P[cellLoc] = STDET_BURN[cellLoc] * stdet_PC[cellLoc]; */
/*                  stdet_burn_OM[cellLoc] = STDET_BURN[cellLoc] / stdet_CtoOM[cellLoc]; */

                 stdet_toDOM_P[cellLoc] = STDET_TO_DOM[cellLoc] * stdet_PC[cellLoc];
                 stdet_toDOM_OM[cellLoc] = STDET_TO_DOM[cellLoc] / stdet_CtoOM[cellLoc];

/*  now calc the P and organic matter state vars associated with the C state var */
                 stdet_P[cellLoc] = stdet_P[cellLoc]
                     + (stdet_prod_P[cellLoc] /* - stdet_burn_P[cellLoc] */ - stdet_toDOM_P[cellLoc] ) * DT;
                 stdet_PC[cellLoc] = ( (STAND_DETRITUS[cellLoc] > 0.0) && (stdet_P[cellLoc]>0.0))?
                     (stdet_P[cellLoc] / STAND_DETRITUS[cellLoc]) :
                     STDET_PC[HAB[cellLoc]];

                 stdet_OM[cellLoc] = stdet_OM[cellLoc]
                     + (stdet_prod_OM[cellLoc] /* - stdet_burn_OM[cellLoc] */ - stdet_toDOM_OM[cellLoc] ) * DT;
                 stdet_CtoOM[cellLoc] = ((stdet_OM[cellLoc] > 0.0) && (STAND_DETRITUS[cellLoc]>0.0) )?
                     (STAND_DETRITUS[cellLoc] / stdet_OM[cellLoc]) :
                     STDET_CTOOM[HAB[cellLoc]];
	  }
	}
  }
}



/*******
FLOCculent organic matter dynamics
   FLOC[cellLoc] == mass of organic flocculent material at the interface between soil and surface water (kg OM/m^2)
   FlocP[cellLoc] == mass of phosphorus in the flocculent material at the interface between soil and surface water (kg P/m^2)

this routine originally was Suspended Organic Matter; was dramatically modified to instead
represent the organic matter in the flocculent sediment layer

Parameter definitions:
      global parameters in GlobalParms.xls Excel sheet (text export=GlobalParms)
      habitat-specific parameters in HabParms.fmp FileMakerPro database (text export=HabParms)
******/
void cell_dyn12(step)
 int step;
 {
 int ix, iy, x, y, cellLoc; 
 float FlocP_DECOMP_pot, FlocP_DEPO_pot, FlocP_settl, Floc_settl;
 
  for(ix=1; ix<=s0; ix++) {
	for(iy=1; iy<=s1; iy++) {

	  if(ON_MAP[cellLoc= T(ix,iy)])  {
/* inputs (kg OM / m2)  */
                  /* all periphyton mortality goes to floc */
              FLOC_FR_ALGAE[cellLoc]  = (double) (C_ALG_MORT[cellLoc]+NC_ALG_MORT[cellLoc])
                  /ALG_C_TO_OM*0.001 ; 
/*  non-photo goes either to stand det or deposited OM,
    photo goes either to stand det or floc */
              Floc_fr_phBio[cellLoc]  = DOM_PHBIO_PROP[HAB[cellLoc]]*phbio_mort_OM[cellLoc];

             /* all settling from water column goes to floc */
              FlocP_settl = TP_settl[cellLoc] / CELL_SIZE; /* kg P/m2; */
                  /* the particulate P settling is assumed a fixed Redfield P:OM ratio */
              Floc_settl =   FlocP_settl / TP_P_OM; 
                 
          
/* outputs (kg OM / m2) */
              FLOC_DECOMP_QUAL_CF[cellLoc]  = /* use the avg conc of sed and sf water here */
                  Exp(-DOM_decomp_coef * Max(DOM_DECOMP_POPT-
                                 (TP_SFWT_CONC_MG[cellLoc]+TP_SEDWT_CONCACTMG[cellLoc])/2.0, 0.0)
                      /DOM_DECOMP_POPT) ; /* mg/L */
/*               FLOC_DECOMP_QUAL_CF[cellLoc]  = Max(TP_SFWT_CONC_MG[cellLoc] */
/*          		/(DOM_DECOMP_POPT+TP_SFWT_CONC_MG[cellLoc]), 0.01) ;  */
/* decomp in surface water has higher rate than in soil,
 assuming this stock is of much higer substrate quality than the total soil layer */
/* beta3 is the calib parm read from Driver.parm */
              DOM_MOIST_CF[cellLoc]  =  (UNSAT_DEPTH[cellLoc]>DOM_AEROBTHIN[HAB[cellLoc]]) ?
                     ( Max(UNSAT_MOIST_PRP[cellLoc],0.0) ) :
                     ( 1.0 );
              FLOC_DECOMP_POT[cellLoc]  = beta3 * 10.0*DOM_RCDECOMP*FLOC[cellLoc]
                   *DOM_TEMP_CF[cellLoc] *FLOC_DECOMP_QUAL_CF[cellLoc] * DOM_MOIST_CF[cellLoc];
              FLOC_DECOMP[cellLoc]  = 
                  ( (FLOC_DECOMP_POT[cellLoc])*DT>FLOC[cellLoc] ) ? 
                  ( (FLOC[cellLoc])/DT ) : 
                  ( FLOC_DECOMP_POT[cellLoc]);
              FLOC_PR1[cellLoc]  = FLOC_DECOMP[cellLoc];

/* the incorporation of the floc layer into the "true" soil layer occurs
   at a rate dependent on the floc depth under flooded conditions, then constant rate under dry conditions */
              FLOC_DEPO_POT[cellLoc]  = 
                  ( SURFACE_WAT[cellLoc] > DetentZ ) ? 
                  ( FLOC_Z[cellLoc]/FlocMax * FLOC[cellLoc]*Floc_rcSoil ) : 
                  ( FLOC[cellLoc]*Floc_rcSoil);
              FLOC_DEPO[cellLoc]  = 
                  ( (FLOC_DEPO_POT[cellLoc]+FLOC_PR1[cellLoc])*DT>FLOC[cellLoc] ) ? 
                  ( (FLOC[cellLoc]-FLOC_PR1[cellLoc]*DT)/DT ) : 
                  ( FLOC_DEPO_POT[cellLoc]); 
/* calc the state var (kg OM / m2) */
              FLOC[cellLoc] =  FLOC[cellLoc] 
                  + ( Floc_settl + Floc_fr_phBio[cellLoc] + FLOC_FR_ALGAE[cellLoc]
                      - FLOC_DECOMP[cellLoc] - FLOC_DEPO[cellLoc] ) * DT;

/* the depth of floc is dependent on a fixed floc bulk density */
              FLOC_Z[cellLoc] = (double) FLOC[cellLoc] / Floc_BD ;
                 

/* book-keeping calcs used in other modules */
             /*  FLOC_C_AVAIL[cellLoc]  = FLOC[cellLoc]/CELL_SIZE*FLOC_CTOOM[HAB[cellLoc]]; */

/* Floc phosphorus (kg P / m2)  */
              FlocP_FR_ALGAE[cellLoc]  = (double) (NC_ALG_MORT_P[cellLoc]
                                          + C_ALG_MORT_P[cellLoc])*0.001; /* kg P/m2 */
              FlocP_PhBio[cellLoc] = DOM_PHBIO_PROP[HAB[cellLoc]]*phbio_mort_P[cellLoc] ;    

              FlocP_DECOMP_pot =  FLOC_DECOMP[cellLoc] * FlocP_OM[cellLoc];
              FlocP_DECOMP[cellLoc]  = 
                  ( (FlocP_DECOMP_pot)*DT>FlocP[cellLoc] ) ? 
                  ( (FlocP[cellLoc])/DT ) : 
                  ( FlocP_DECOMP_pot); 
              FlocP_DEPO_pot =  FLOC_DEPO[cellLoc] * FlocP_OM[cellLoc];
              FlocP_DEPO[cellLoc]  = 
                  ( (FlocP_DEPO_pot+FlocP_DECOMP[cellLoc])*DT>FlocP[cellLoc] ) ? 
                  ( (FlocP[cellLoc]-FlocP_DECOMP[cellLoc]*DT)/DT ) : 
                  ( FlocP_DEPO_pot); 
              
/* calc the state var (kg P / m2) */
              FlocP[cellLoc] =  FlocP[cellLoc] 
                  + ( FlocP_settl + FlocP_PhBio[cellLoc] + FlocP_FR_ALGAE[cellLoc]
                      - FlocP_DECOMP[cellLoc] - FlocP_DEPO[cellLoc] ) * DT;

              FlocP_OM[cellLoc] = ( FLOC[cellLoc]>0.0) ? (FlocP[cellLoc]/FLOC[cellLoc]) : (0.0); /* kgP/kgOM */
              FlocP_OMrep[cellLoc] = (float) FlocP_OM[cellLoc] * 1.0e6; /* mgP/kgOM */
              
              if (debug > 0 && FLOC[cellLoc] < -MinCheck) { sprintf(msgStr,"Day %6.1f: ERROR - neg FLOC OM biomass (%f m) in cell (%d,%d)!", 
                                                                    TIME, FLOC[cellLoc], ix,iy ); WriteMsg( msgStr,True );}
              if (debug > 0 && FlocP[cellLoc] < -MinCheck)  { sprintf(msgStr,"Day %6.1f: ERROR - neg FLOC P biomass (%f m) in cell (%d,%d)!", 
                                                                      TIME, FlocP[cellLoc], ix,iy ); WriteMsg( msgStr,True );}

/* now the P gain in sediment pore water with decomp - 90% goes to porewater, 10% to sfwat */
              TP_SED_MINER[cellLoc]  =  0.90 * FlocP_DECOMP[cellLoc] * CELL_SIZE ; 
/* calc P in sed water state variables (kg P) */
              TP_SED_WT[cellLoc] =  TP_SED_WT[cellLoc] + 
                  (TP_SED_MINER[cellLoc]) * DT;
            /* this is the active zone, where uptake, sorption, and mineralization take place */
              TP_SED_WT_AZ[cellLoc] =  TP_SED_WT_AZ[cellLoc] + 
                  (TP_SED_MINER[cellLoc]) * DT;

              TP_SED_CONC[cellLoc] = (HYD_SED_WAT_VOL[cellLoc]>0.0) ?
                  (TP_SED_WT[cellLoc] / HYD_SED_WAT_VOL[cellLoc]) :
                  (0.0);
               TP_SEDWT_CONCACT[cellLoc] = ( HYD_DOM_ACTWAT_PRES[cellLoc] > 0.0) ?
                  ( TP_SED_WT_AZ[cellLoc]/HYD_DOM_ACTWAT_VOL[cellLoc] ) :
                  (TP_SED_CONC[cellLoc]);
              TP_SEDWT_CONCACTMG[cellLoc]  = TP_SEDWT_CONCACT[cellLoc]*1000.0;
              
              
/* now the P gain in surface water with decomp - 90% goes to porewater, 10% to sfwat */
              TP_SFWT_MINER[cellLoc]  = 0.10*FlocP_DECOMP[cellLoc] * CELL_SIZE ;  
/* calc P in surface water state variable (kg P) */
              TP_SF_WT[cellLoc] = TP_SF_WT[cellLoc] + 
                  (TP_SFWT_MINER[cellLoc]) * DT;
              TP_SFWT_CONC[cellLoc]  = 
                  ( SFWT_VOL[cellLoc] > 0.0 ) ? 
                  ( TP_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : 
                  ( 0.0); /* used in P fluxes for mass balance */
              TP_SFWT_CONC_MG[cellLoc]  = 
                  ( SURFACE_WAT[cellLoc] > DetentZ ) ? 
                  (TP_SFWT_CONC[cellLoc]*1000.0) : 
                  (0.0); /* used for reporting and other modules to evaluate P conc when water is present */
	  }
	}
  }
}

/*******
calling function fof the cell_dyns
******/
int call_cell_dyn(sector, step)
 int sector, step;
 {
	int rv=0;

/*###############
This function calls the cell_dyn dynamic modules as defined in the Driver.parm data file.
(also in that file: WatMgmt set to 1 for running canals etc., 0 to turn off)
Normal order for calling ELM modules(Sectors): 1 0 7 10 9 2 8 11 4 12 99.

S#99	mass balance, budget, avg, hydroperiod, etc calculations
S#0		hydro: cell-cell horiz (&canals if WaterManagement is on)
S#1		global forcings
S#2		algae/periphyton
S#3		consumers
S#4		DOM/DOP
S#5		nitrogen
S#6		fire
S#7		hydro: vertical
S#8		macrophytes
S#9		phosphorus
S#10	salt
S#11	standDetritus
S#12	FLOC
###############*/
	switch(sector) {

		case 99: { stats(step); rv=1; } break;
		case 0: { horizFlow(step); rv=1; } break;
		case 1: { cell_dyn1(step); rv=1; } break;
		case 2: { cell_dyn2(step); rv=1; } break;
		case 3: { cell_dyn3(step); rv=1; } break;
		case 4: { cell_dyn4(step); rv=1; } break;
		case 5: { cell_dyn5(step); rv=1; } break;
		case 6: { cell_dyn6(step); rv=1; } break;
		case 7: { cell_dyn7(step); rv=1; } break;
		case 8: { cell_dyn8(step); rv=1; } break;
		case 9: { cell_dyn9(step); rv=1; } break;
		case 10: { cell_dyn10(step); rv=1; } break;
		case 11: { cell_dyn11(step); rv=1; } break;
		case 12: { cell_dyn12(step); rv=1; } break;
		case 13: { cell_dyn13(step); rv=1; } break;
            default: if(Lprocnum==1) printf("Warning, undefined sector number:%d\n",sector);
	}
	return rv;
}


/*******
model initialization
******/
void init_vars(first_run)
int first_run;
 {
int ix,iy, cellLoc;
char date_read[20],yyyy[5],mm[3], dd[3], lline[201], *line;
 char filename[30];
float VAP_PRESS_SAT, VapPres_DPT, VAP_SAT_DEFICIT; /* modif vars,  sept98 */
float tmp; /* used in checking nutrient availability for MichMent kinetics */
float min_litTemp; /* used to determine min of temper, light cf's for alg and macs */
float I_ISat, Z_extinct, PO4Pconc, PO4P;
 float MAC_PHtoNPH, MAC_PHtoNPH_Init;
 
 
 if(first_run) {
        usrErr0("\tMaps... ");
	if (PosAn<2) { /* in position analysis re-init (PosAn>1), don't execute this */
	read_map_file("ModArea",ON_MAP,'c',4.0,0.0);            /* defines model area */
	read_map_file("BoundCond",BCondFlow,'d',1.0,0.0);       /* boundary condition flow restrictions */

            /* ONLY when running the EWQM settling rate version of ELM */
        /* basin-based settling rates (tenths mm/d->m/d) */
        if (ESP) read_map_file("basinSetVel",WQMsettlVel,'f',0.0001,0.0);       

 	read_map_file("icMult",IC_MULT_MAP,'f',0.001,0.0);      /* multiplier map for initial conditions (dimless, 0.1->1.0) */
	read_map_file("basins",basn,'d',1.0,0.0);               /* water basins map  */
	read_map_file("HydrCond",HYD_RCCONDUCT,'f',1.0,0.0);    /* hydraulic conductivity (m/d) (did have 0.1 scaling) */
        }
	read_map_file("Elevation",ELEVATION,'f',0.01,0.0);      /* elevation (in cm => m) */
	read_map_file("HAB",HAB,'c',1.0,0.0);                   /* habitat (veg classifications) */
	read_map_file("icSfWt",SURFACE_WAT,'f',0.01,0.0);       /* initial ponded surface water (in cm => m) */
	read_map_file("icUnsat",UNSAT_DEPTH,'f',0.01,0.0);      /* initial depth of unsaturated zone (in cm => m) */
	read_map_file("soilTP",TPtoSOIL,'f',0.000001,0.0);  /* soil TP map (in mgP/kgSoil  => kgP/kgSoil) */
	read_map_file("soilBD",BulkD,'f',1.0,0.0);         /* soil bulk dens map (mgSoil/cm3 == kgSoil/m3)  */
	read_map_file("soil_orgBD",DOM_BD,'f',1.0,0.0);    /* organic soil bulk dens map (mgOM/cm3 == kgOM/m3)  */
        usrErr("done.");

	if (PosAn<2) { /* in position analysis re-init (PosAn>1), don't execute this */
            alloc_mem_stats (); /* allocate memory for budget & stats arrays (need to have read the basin map first) */
            basinInit(); /* set up the basins' indicator regions and text<->numeric assignments */
        }
        
        if(HabSwitch) F_Hab_Init ( "HabSwData" );
        for(ix=0; ix<=s0+1; ix++) {
	  for(iy=0; iy<=s1+1; iy++) {
              cellLoc = T(ix,iy);
              SED_ELEV[cellLoc] =  ELEVATION[cellLoc]+DATUM_DISTANCE; 
          }
        }

	if (PosAn<2) { /* in position analysis re-init (PosAn>1), don't execute this */
            if(WatMgmt) Canal_Network_Init(DATUM_DISTANCE,SED_ELEV); /* WatMgmt.c - initialize the canal network */
            /* new dynamic boundary condition stage */
                /* ReadBCond (DATUM_DISTANCE);*/ /* Fluxes.c - read the stages in cells external to the model for bound. cond. flow data */
            
        }
        else {
                /* this will only re-initialize the canal depths, concs */
            CanalReInit();
        }
        
        
	if (PosAn<2) { /* in position analysis re-init (PosAn>1), don't do these budget files and parms */
            if (debug>0) budg_files();
        
        

        usrErr0("\tHabitat-specific parms... ");
/* reading parameters that vary with habitat */

/* the get_DBase_parm function expects the appropriate parameter to be
   associated with it's module and parm ID numbers
   with the form: ("dbaseNam",module#,module_parmID#,ignore#) - the ID numbers must
   be associated with the proper parameter, as the parm name is not read from the file.
   These ID numbers are ONLY used in this parameter read function, and are NOT used
   in any data structures or elsewhere in the ELM code.
*/
	HYD_RCINFILT = get_DBase_parm("HabParms",7,1,0);
	HYD_SPEC_YIELD = get_DBase_parm("HabParms",7,2,0); 
	HYD_POROSITY = get_DBase_parm("HabParms",7,3,0);
        
	ALG_MAX = get_DBase_parm("HabParms",2,1,0);

 	DOM_MAXDEPTH = get_DBase_parm("HabParms",4,1,0); 
	DOM_NPHBIO_PROP = get_DBase_parm("HabParms",4,2,0);
	DOM_PHBIO_PROP = get_DBase_parm("HabParms",4,3,0);
	DOM_AEROBTHIN = get_DBase_parm("HabParms",4,4,0);

	PHBIO_MAX = get_DBase_parm("HabParms",8,1,0);
	NPHBIO_MAX = get_DBase_parm("HabParms",8,2,0);
	NPHBIO_ABVBEL = get_DBase_parm("HabParms",8,3,0);
	MAC_MAXHT = get_DBase_parm("HabParms",8,4,0);
	NPHBIO_ROOTDEPTH = get_DBase_parm("HabParms",8,5,0);
	MAC_MAXROUGH = get_DBase_parm("HabParms",8,6,0);
	MAC_MINROUGH = get_DBase_parm("HabParms",8,7,0);
	MAC_MAXLAI = get_DBase_parm("HabParms",8,8,0);
	MAC_MAXCANOPCOND = get_DBase_parm("HabParms",8,9,0);
	MAC_CANOPDECOUP = get_DBase_parm("HabParms",8,10,0);
	MAC_TEMPOPT = get_DBase_parm("HabParms",8,11,0);
	MAC_LIGHTSAT = get_DBase_parm("HabParms",8,12,0);
	MAC_KSP = get_DBase_parm("HabParms",8,13,0);
	PHBIO_RCNPP = get_DBase_parm("HabParms",8,14,0);
	MAC_SPR_GROW = get_DBase_parm("HabParms",8,15,0);
	PHBIO_RCMORT = get_DBase_parm("HabParms",8,16,0);
	MAC_WAT_TOLER = get_DBase_parm("HabParms",8,17,0);
	MAC_SALIN_THRESH = get_DBase_parm("HabParms",8,18,0);
	PHBIO_CTOOM = get_DBase_parm("HabParms",8,19,0);
	NPHBIO_CTOOM = get_DBase_parm("HabParms",8,20,0);
	PHBIO_PC = get_DBase_parm("HabParms",8,21,0);
	NPHBIO_PC = get_DBase_parm("HabParms",8,22,0);
	PHBIO_NC = get_DBase_parm("HabParms",8,23,0);
	NPHBIO_NC = get_DBase_parm("HabParms",8,24,0);
	MAC_KSN = get_DBase_parm("HabParms",8,25,0);
        Mac_transl_rc = get_DBase_parm("HabParms",8,26,0);

	FLOC_CONSPROP = get_DBase_parm("HabParms",12,1,0);
	FLOC_IC = get_DBase_parm("HabParms",12,2,0);
	FLOC_CTOOM = get_DBase_parm("HabParms",12,3,0);
	FLOC_PC = get_DBase_parm("HabParms",12,4,0);
	FLOC_NC = get_DBase_parm("HabParms",12,5,0);

	STDET_MAX = get_DBase_parm("HabParms",11,1,0);
	STDET_ABVBEL = get_DBase_parm("HabParms",11,2,0); 
	STDET_CTOOM = get_DBase_parm("HabParms",11,3,0);
	STDET_PC = get_DBase_parm("HabParms",11,4,0);
	STDET_NC = get_DBase_parm("HabParms",11,5,0);
	STDET_SHRED = get_DBase_parm("HabParms",11,6,0);
	STDET_WIND = get_DBase_parm("HabParms",11,7,0);

        SALT_ICSEDWAT = get_DBase_parm("HabParms",10,1,0);
	SALT_ICSFWAT = get_DBase_parm("HabParms",10,2,0);

	TP_CONC_GRAD = get_DBase_parm("HabParms",9,1,0);

	DIN_CONC_GRAD = get_DBase_parm("HabParms",5,1,0);

	FIRE_RX_VELOC = get_DBase_parm("HabParms",6,1,0);
	FIRE_FUEL_HEAT = get_DBase_parm("HabParms",6,2,0);
/* end of reading habitat-specific parms */
        usrErr("done.");


        if (!(ELMrun) ) { /* rainfall for CALM (or potentially other) models, use interpolated time series */
            usrErr0 ("\tRainfall spatial time series... ");
		pTSList2 = PTSL_Init(50, 2);
		PTSL_ReadLists(pTSList2,"precip",TIndex2++,&Timestep2,&NPtTS2, 1); /* rainfall in col 1 */
                usrErr("done.");
        }
        
        
        usrErr0 ("\tMeteorological spatial time series... ");
            /* final arg in PTSL_ReadLists is the column number of of the data (not incl. the 3 date columns) */
        pTSList3 = PTSL_Init(50, 2);
        PTSL_ReadLists(pTSList3,"meteor",TIndex3++,&Timestep3,&NPtTS3, 1); /* air temperature data in col 1 */
        pTSList4 = PTSL_Init(50, 2);
        PTSL_ReadLists(pTSList4,"meteor",TIndex4++,&Timestep4,&NPtTS4, 2);  /* dew point temperature data in col 2 */
        pTSList5 = PTSL_Init(50, 2);
        PTSL_ReadLists(pTSList5,"meteor",TIndex5++,&Timestep5,&NPtTS5, 3);  /* wind speed data in col 3 */
        pTSList6 = PTSL_Init(50, 2);
        PTSL_ReadLists(pTSList6,"meteor",TIndex6++,&Timestep6,&NPtTS6, 4);  /* cloud cover data in col 4 */
        usrErr("done.");
        } /* end of braces for initialization that is NOT done in PosAn re-init */
        
        return; /* end of spatial data read */
    }

        TIME = 0;
	DAYJUL = ( Mod(TIME,365.0) >0.0 ) ? ( Mod(TIME,365.0) ) : ( 365.0);
	FIRE_LIGHT_THRESH = 0.02*Cos(DAYJUL/365.0*2.0*PI)+0.98;
	LATRAD = (Int(LATDEG)+(LATDEG-Int(LATDEG))*5.0/3.0)*PI/180.0;
	AMPL = Exp(7.42+0.045*LATRAD*180.0/PI)/3600.0;
	DAYLENGTH = AMPL*Sin((DAYJUL-79.0)*0.01721)+12.0;
	SOLALPHA = 32.9835-64.884*(1.0-1.3614*Cos(LATRAD));
	SOLDEC1 = 0.39785*Sin(4.868961+0.017203*DAYJUL   +0.033446*Sin(6.224111+0.017202*DAYJUL));
	SOLCOSDEC = Sqrt(1.0-pow(SOLDEC1,2.0));
	SOLELEV_SINE = Sin(LATRAD)*SOLDEC1+Cos(LATRAD)*SOLCOSDEC;
	SOLALTCORR = (1.0-Exp(-0.014*(ALTIT-274.0)/(SOLELEV_SINE*274.0)));
	SOLBETA = 0.715-0.3183*(1.0-1.3614*Cos(LATRAD));
	SOLDEC = Arctan(SOLDEC1/Sqrt(1.0-pow(SOLDEC1,2.0)));
	SOLRISSET_HA1 = -Tan(LATRAD)*Tan(SOLDEC);
	SOLRISSET_HA = ( (SOLRISSET_HA1==0.0) ) ? ( PI*0.5 ) : (   ( (SOLRISSET_HA1<0.0) ) ? ( PI+Arctan(Sqrt(1.0-pow(SOLRISSET_HA1,2.0))/SOLRISSET_HA1)  ) : (      Arctan(Sqrt(1.0-pow(SOLRISSET_HA1,2.0))/SOLRISSET_HA1)));
	SOLRADATMOS = 458.37*2.0*(1.0+0.033*Cos(360.0/365.0*PI/180.0*DAYJUL))* (Cos(LATRAD)*Cos(SOLDEC)*Sin(SOLRISSET_HA) + SOLRISSET_HA*180.0/(57.296*PI)*Sin(LATRAD)*Sin(SOLDEC));
	WIND_DIR =  graph5(0x0,DAYJUL);
        
        for(ix=0; ix<=s0+1; ix++) {
	  for(iy=0; iy<=s1+1; iy++) { cellLoc = T(ix,iy);
/* sep98 temperature, cloudiness, humidity (&dew point temp), and wind speed spatial now */
/* read in, interpolate (cell_dyn1) daily data on avg air temp, avg wind spd, cloudiness, and dew point temp */
/* rainfall read from sfwmm data, remapped to ELM resolution */
                 DewPt[cellLoc] = AIR_TEMP[cellLoc] = 25.0; WIND_SPEED[cellLoc] = CLOUDY[cellLoc] = wmm_rain[cellLoc] = HYD_PRECIP_DAY[cellLoc] = SF_WT_FROM_RAIN[cellLoc]  = 0.0;
    	 
                 VapPres_DPT =   610.78*Exp(17.269*DewPt[cellLoc]/(DewPt[cellLoc]+237.3));
                 VAP_PRESS_SAT = 610.78*Exp(17.269*AIR_TEMP[cellLoc]/(AIR_TEMP[cellLoc]+237.3));
                 HUMIDITY[cellLoc] =  Min( VapPres_DPT / VAP_PRESS_SAT, 1.0); 
                 VAP_SAT_DEFICIT = 1.0 - HUMIDITY[cellLoc];
		 WIND_SPEED_CANOPY[cellLoc] = WIND_SPEED[cellLoc]*0.666;
		 PAN_CH[cellLoc] = 1.035+0.240*pow((HUMIDITY[cellLoc]/0.60),2.0)-0.275*pow((HUMIDITY[cellLoc]/0.60),3.0 ) ;
                 PAN_CT[cellLoc] = 0.463+0.425*(AIR_TEMP[cellLoc]/20.0)+0.112*pow((AIR_TEMP[cellLoc]/20.0),2.0);
		 PAN_CW[cellLoc] =  0.672+0.406*(WIND_SPEED_CANOPY[cellLoc]/6.7)-0.078*pow((WIND_SPEED_CANOPY[cellLoc]/6.7),2.0 ) ;

                 SOLRAD274[cellLoc] = SOLRADATMOS*(SOLBETA-SOLOMEGA*(( CLOUDY[cellLoc]>0.0 ) ? 
                                               ( CLOUDY[cellLoc] ) : ( 0.0)))-SOLALPHA;
                 SOLRADGRD[cellLoc] = SOLRAD274[cellLoc]+((SOLRADATMOS+1.0)-SOLRAD274[cellLoc])*SOLALTCORR;
                 H2O_TEMP[cellLoc] = AIR_TEMP[cellLoc];
                 CONS_TEMP_CF = Exp(0.20*(AIR_TEMP[cellLoc]-CONS_T_OPT))*pow(((40.0-AIR_TEMP[cellLoc])/(40.0-CONS_T_OPT)),(0.20*(40.0-CONS_T_OPT)));
         
		 ALG_REFUGE[cellLoc] = ALG_MAX[HAB[cellLoc]]*ALG_REF_MULT;
		 ALG_SAT[cellLoc] = ALG_MAX[HAB[cellLoc]]*0.9;

		 SED_INACT_Z[cellLoc] = SED_ELEV[cellLoc]-DOM_MAXDEPTH[HAB[cellLoc]];

                 MAC_MAX_BIO[cellLoc] = NPHBIO_MAX[HAB[cellLoc]]+PHBIO_MAX[HAB[cellLoc]];
		 NPHBIO_REFUGE[cellLoc] = NPHBIO_MAX[HAB[cellLoc]]*MAC_REFUG_MULT;
		 NPHBIO_SAT[cellLoc] = NPHBIO_MAX[HAB[cellLoc]]*0.9;
		 PHBIO_REFUGE[cellLoc] = PHBIO_MAX[HAB[cellLoc]]*MAC_REFUG_MULT;
		 PHBIO_SAT[cellLoc] = PHBIO_MAX[HAB[cellLoc]]*0.9;
                 STDET_REFUG[cellLoc] = STDET_MAX[HAB[cellLoc]]*STDET_REF_MULT;
		 STDET_SAT[cellLoc] = STDET_MAX[HAB[cellLoc]]*0.9;
		 WIND_STORM[cellLoc] = STDET_WIND[HAB[cellLoc]]+10.0;
	}
  }
    for(ix=1; ix<s0+1; ix++) {
	  for(iy=1; iy<s1+1; iy++) { cellLoc = T(ix,iy);
/*initialization of major state variables */

              /* hydro */
/* map */        UNSAT_DEPTH[cellLoc] = Max(UNSAT_DEPTH[cellLoc] + HYD_IC_UNSATZ,0.0); /* m */
		 SAT_WT_HEAD[cellLoc] = SED_ELEV[cellLoc]- UNSAT_DEPTH[cellLoc]; /* m */
		 SAT_WATER[cellLoc] = SAT_WT_HEAD[cellLoc] * HYD_POROSITY[HAB[cellLoc]]; /* m */

		 UNSAT_WATER[cellLoc] = HYD_POROSITY[HAB[cellLoc]]*UNSAT_DEPTH[cellLoc] *HYD_ICUNSATMOIST; /* m */
		 UNSAT_CAP[cellLoc] = UNSAT_DEPTH[cellLoc]*HYD_POROSITY[HAB[cellLoc]]; /* m */
                 UNSAT_MOIST_PRP[cellLoc]  =  /* dimless proportion, 0-1 */
                     ( UNSAT_CAP[cellLoc]>0.0 ) ? 
                     ( Min(UNSAT_WATER[cellLoc]/UNSAT_CAP[cellLoc],1.0) ) : 
                     ( 1.0); 

/* map */        SURFACE_WAT[cellLoc] =  Max(SURFACE_WAT[cellLoc] + HYD_IC_SFWAT, 0.0); /* m */
		 SFWT_VOL[cellLoc] = SURFACE_WAT[cellLoc]*CELL_SIZE;

		 HYD_DOM_ACTWAT_VOL[cellLoc] = Max(DOM_MAXDEPTH[HAB[cellLoc]]-UNSAT_DEPTH[cellLoc]*
                                                   (1.0-UNSAT_MOIST_PRP[cellLoc]),0.0)*HYD_POROSITY[HAB[cellLoc]]*CELL_SIZE;
		 HYD_SED_WAT_VOL[cellLoc] = (SAT_WATER[cellLoc]+UNSAT_WATER[cellLoc])*CELL_SIZE;
                 
              /* soil */
                 DIM[cellLoc] = (BulkD[cellLoc] - DOM_BD[cellLoc]) * DOM_MAXDEPTH[HAB[cellLoc]] * CELL_SIZE; /* kg inorganic/m2 */
                 Inorg_Z[cellLoc] = (1.0 - ( DOM_BD[cellLoc] / BulkD[cellLoc] )) * DOM_MAXDEPTH[HAB[cellLoc]]; /*  fixed inorganic depth (m) */
/* map */        DOM_Z[cellLoc] = DOM_MAXDEPTH[HAB[cellLoc]] - Inorg_Z[cellLoc]; /* m */

                 DEPOS_ORG_MAT[cellLoc] = DOM_BD[cellLoc]*DOM_Z[cellLoc]; /* (mgOM/cm3 ==> kgOM/m3) * m = kgOM/m2 */

		 DOM_SED_AEROB_Z[cellLoc] = Min(UNSAT_DEPTH[cellLoc]+DOM_AEROBTHIN[HAB[cellLoc]],DOM_MAXDEPTH[HAB[cellLoc]]); /* m */
		 DOM_SED_ANAEROB_Z[cellLoc] = DOM_MAXDEPTH[HAB[cellLoc]]-DOM_SED_AEROB_Z[cellLoc]; /* m */

/* map */        DOP[cellLoc] =  (1.0-sorbToTP)*TPtoSOIL[cellLoc] * BulkD[cellLoc] * DOM_MAXDEPTH[HAB[cellLoc]] ; /* kgP/kg_soil * kg_soil/m3 * m == kgP/m2 */

              /* floc layer of soil */
/* map */        FLOC[cellLoc] = FLOC_IC[HAB[cellLoc]]; /* kg OM/m2  */
/* map */        FlocP[cellLoc] = FLOC[cellLoc]*FLOC_PC[HAB[cellLoc]]*FLOC_CTOOM[HAB[cellLoc]]; /* kg P /m2 */
                 FLOC_Z[cellLoc] = (double) FLOC[cellLoc] / Floc_BD ; /* m */
		 FLOC_C_AVAIL[cellLoc] = FLOC[cellLoc]*FLOC_CTOOM[HAB[cellLoc]]; /* kg C/m2  */

              /* phosphorus */
/* map */        TP_SFWT_CONC[cellLoc]  = TP_ICSFWAT*0.001 * IC_MULT_MAP[cellLoc] * 10.0; /* mg/L * g/mg * dimless * multiplier => g/L */
		 TP_SF_WT[cellLoc] =TP_SFWT_CONC[cellLoc] * SFWT_VOL[cellLoc]; /* kg P */
		 TP_SFWT_CONC_MG[cellLoc] = TP_SFWT_CONC[cellLoc]*1000.0; /* mg/L */
                      /* using regression for predicting PO4 from TP */
		 PO4Pconc = Max(TP_SFWT_CONC_MG[cellLoc]*PO4toTP + PO4toTPint, 0.10 * TP_SFWT_CONC_MG[cellLoc]); /* mg/L */
                 PO4P = PO4Pconc * SFWT_VOL[cellLoc] /1000.0; /*kg P available (from conc. in mg/l) */

                     /* ic_mult_map varies from 0.1 to 1 */
/* map */        TP_SED_CONC[cellLoc] = TP_ICSEDWAT*0.001 * IC_MULT_MAP[cellLoc] * 10.0; /* mg/L * g/mg * dimless * multiplier => g/L */
                 TP_SED_WT[cellLoc] = TP_SED_CONC[cellLoc] * HYD_SED_WAT_VOL[cellLoc]; /* kg P */
                     /* this is the active zone, where uptake, sorption, and mineralization take place */
/* map */        TP_Act_to_Tot[cellLoc] = 1.0 / TP_CONC_GRAD[HAB[cellLoc]]; /* ratio of TP conc in active zone relative to total */
                 TP_SED_WT_AZ[cellLoc] = TP_SED_CONC[cellLoc] * TP_Act_to_Tot[cellLoc] * HYD_DOM_ACTWAT_VOL[cellLoc]; /* kg P */
                 TP_SEDWT_CONCACT[cellLoc] =(HYD_DOM_ACTWAT_VOL[cellLoc] > 0.0) ?
                     ( TP_SED_WT_AZ[cellLoc]/HYD_DOM_ACTWAT_VOL[cellLoc] ) :
                     ( TP_SED_CONC[cellLoc]); /* g/L */
		 TP_SEDWT_CONCACTMG[cellLoc] = TP_SEDWT_CONCACT[cellLoc]*1000.0; /* mg/L */

/* map */        TP_SORB[cellLoc] = sorbToTP*TPtoSOIL[cellLoc]
                     * BulkD[cellLoc] * DOM_MAXDEPTH[HAB[cellLoc]] * CELL_SIZE; /* dimless * kgP/kg_soil * kg_soil/m3 * m * m^2 == kgP */

              /* nitrogen */
		 DIN_SF_WT[cellLoc] = SFWT_VOL[cellLoc]*DIN_ICSFWAT*0.001;
		 DIN_SFWT_CONC[cellLoc] = ( SURFACE_WAT[cellLoc] > DetentZ ) ? ( DIN_SF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : ( 0.0);

                 DIN_SED_WT[cellLoc] = ((HYD_SED_WAT_VOL[cellLoc]-HYD_DOM_ACTWAT_VOL[cellLoc])*DIN_CONC_GRAD[HAB[cellLoc]]+HYD_DOM_ACTWAT_VOL[cellLoc])*DIN_ICSEDWAT*0.001;
		 DIN_SED_CONC_INT[cellLoc] = HYD_DOM_ACTWAT_VOL[cellLoc]+(HYD_SED_WAT_VOL[cellLoc]-HYD_DOM_ACTWAT_VOL[cellLoc])*DIN_CONC_GRAD[HAB[cellLoc]];
		 DIN_SEDWT_CONCACT[cellLoc] = ( DIN_SED_CONC_INT[cellLoc]>0.0 ) ? ( DIN_SED_WT[cellLoc]/DIN_SED_CONC_INT[cellLoc] ) : ( 0.0);

              /* salt */
		 SALT_SED_WT[cellLoc] = HYD_SED_WAT_VOL[cellLoc]*SALT_ICSEDWAT[HAB[cellLoc]];
		 SAL_SED_WT[cellLoc] = ( HYD_SED_WAT_VOL[cellLoc]>0.0 ) ? ( SALT_SED_WT[cellLoc]/HYD_SED_WAT_VOL[cellLoc] ) : ( 0.0);
		 SALT_SURF_WT[cellLoc] = SFWT_VOL[cellLoc]*SALT_ICSFWAT[HAB[cellLoc]];
		 SAL_SF_WT[cellLoc] = ( SURFACE_WAT[cellLoc] > DetentZ ) ? ( SALT_SURF_WT[cellLoc]/SFWT_VOL[cellLoc] ) : ( 0.0);
		 SAL_MEAN[cellLoc] = (SAL_SED_WT[cellLoc]+SAL_SF_WT[cellLoc])/2.0;

              /* periphyton */
/* map */        NC_ALG[cellLoc] = ALG_MAX[HAB[cellLoc]]*ALG_IC_MULT*pow(IC_MULT_MAP[cellLoc],0.3) * 0.1; /* reduce ncalc by 0.1; g C/m2 */
/* map */        C_ALG[cellLoc]  = ALG_MAX[HAB[cellLoc]]*ALG_IC_MULT*(1.0 - pow(IC_MULT_MAP[cellLoc],0.3) ); /* g C/m2 */
                 /* ic PC of periph in oligotrophic area is 3% of max P:C, varies across space via (0.1->1.0) map */
/* map */        NC_ALG_PC[cellLoc] = ALG_PC * 0.3 * IC_MULT_MAP[cellLoc]; /* gP/ gC */
/* map */        C_ALG_PC[cellLoc]  = ALG_PC * 0.3 * IC_MULT_MAP[cellLoc]; /* gP/ gC */

                 NC_ALG_P[cellLoc] = NC_ALG[cellLoc]*NC_ALG_PC[cellLoc];   /* g P/m2 */
                 C_ALG_P[cellLoc] = C_ALG[cellLoc]*C_ALG_PC[cellLoc];  /* g P/m2 */  

              /* macrophytes */
/* map */        MAC_PH_BIOMAS[cellLoc] = PHBIO_MAX[HAB[cellLoc]]*MAC_IC_MULT *pow(IC_MULT_MAP[cellLoc],0.3); /* kg C/m2 */
                   /*  now calc the P and OM associated with that C */
/* map */        mac_ph_PC[cellLoc] = PHBIO_PC[HAB[cellLoc]] * 0.6 * pow(IC_MULT_MAP[cellLoc],0.3); 
                 mac_ph_P[cellLoc] = MAC_PH_BIOMAS[cellLoc] * mac_ph_PC[cellLoc]; /* kg P/m2 */
                 mac_ph_OM[cellLoc] = MAC_PH_BIOMAS[cellLoc] / PHBIO_CTOOM[HAB[cellLoc]];
                 mac_ph_CtoOM[cellLoc] = PHBIO_CTOOM[HAB[cellLoc]];
		 PHBIO_AVAIL[cellLoc] = MAC_PH_BIOMAS[cellLoc]*Max(1.0-Max((PHBIO_SAT[cellLoc]-MAC_PH_BIOMAS[cellLoc]) /(PHBIO_SAT[cellLoc]-PHBIO_REFUGE[cellLoc]),0.0),0.0);

/* map */        MAC_NOPH_BIOMAS[cellLoc] = NPHBIO_MAX[HAB[cellLoc]]*MAC_IC_MULT *pow(IC_MULT_MAP[cellLoc],0.3); /* kg C/m2 */
                   /*  now calc the P and OM associated with that C */
/* map */        mac_nph_PC[cellLoc] = NPHBIO_PC[HAB[cellLoc]]  * 0.6 * pow(IC_MULT_MAP[cellLoc],0.3); 
                 mac_nph_P[cellLoc] = MAC_NOPH_BIOMAS[cellLoc] * mac_nph_PC[cellLoc];  /* kg P/m2 */
                 mac_nph_OM[cellLoc] = MAC_NOPH_BIOMAS[cellLoc] / NPHBIO_CTOOM[HAB[cellLoc]];
                 mac_nph_CtoOM[cellLoc] = NPHBIO_CTOOM[HAB[cellLoc]];
		 NPHBIO_AVAIL[cellLoc] = MAC_NOPH_BIOMAS[cellLoc]*Max(1.0-Max((NPHBIO_SAT[cellLoc]-MAC_NOPH_BIOMAS[cellLoc])/(NPHBIO_SAT[cellLoc]-NPHBIO_REFUGE[cellLoc]),0.0),0.0);

		 MAC_TOT_BIOM[cellLoc] = MAC_PH_BIOMAS[cellLoc]+MAC_NOPH_BIOMAS[cellLoc];
		 MAC_REL_BIOM[cellLoc] = ( MAC_TOT_BIOM[cellLoc] > 0 ) ? MAC_TOT_BIOM[cellLoc]/MAC_MAX_BIO[cellLoc] : 0.0;
		 MAC_LAI[cellLoc] = MAC_REL_BIOM[cellLoc]*MAC_MAXLAI[HAB[cellLoc]];
		 MAC_HEIGHT[cellLoc] = MAC_REL_BIOM[cellLoc]*MAC_MAXHT[HAB[cellLoc]];
                 LAI_eff[cellLoc] =  (MAC_HEIGHT[cellLoc]>0.0) ? (Max(1.0 - SURFACE_WAT[cellLoc]/MAC_HEIGHT[cellLoc], 0.0)*MAC_LAI[cellLoc]) : (0.0)  ;                 

              /* standing detritus */
/* map */        STAND_DETRITUS[cellLoc] = STDET_MAX[HAB[cellLoc]]*STDET_IC_MULT *pow(IC_MULT_MAP[cellLoc],0.3);
                   /*  now calc the P associated with that C */
                 stdet_P[cellLoc] = STAND_DETRITUS[cellLoc]*STDET_PC[HAB[cellLoc]];
/* map */        stdet_PC[cellLoc] = STDET_PC[HAB[cellLoc]];
                 stdet_OM[cellLoc] = STAND_DETRITUS[cellLoc] / STDET_CTOOM[HAB[cellLoc]];
                 stdet_CtoOM[cellLoc] = STDET_CTOOM[HAB[cellLoc]];
		 STDET_AVAIL[cellLoc] = STAND_DETRITUS[cellLoc]* Max(1.0-Max((STDET_SAT[cellLoc]-STAND_DETRITUS[cellLoc])/ (STDET_SAT[cellLoc]-STDET_REFUG[cellLoc]),0.0),0.0);

              /* consumers */
		 CONS_BIOM[cellLoc] = CONS_MAX*CONS_IC_MULT;

/* end of initialization of major state variables */
                 
/* calculations used if particular modules are turned off
 ***************************
 NOTE: THIS section is not fully updated for properly
 turning off each module. In particular, the budget will
 not properly reflect actual dynamics.
 Noted for version 2.5, January 1, 2000
 ***************************
*/
		 NC_ALG_MORT_POT[cellLoc] = ( UNSAT_DEPTH[cellLoc]>0.05 ) ? ( NC_ALG[cellLoc]*ALG_RC_MORT_DRY ) : ( NC_ALG[cellLoc]*ALG_RC_MORT);
                     /* calcareous periphyton */
		 C_ALG_MORT_POT[cellLoc] = ( UNSAT_DEPTH[cellLoc]>0.05 ) ? ( C_ALG[cellLoc]*ALG_RC_MORT_DRY ) : ( C_ALG[cellLoc]*ALG_RC_MORT);
                 ALG_TEMP_CF[cellLoc]  = tempCF(0, 0.20, ALG_TEMP_OPT, 5.0, 40.0, H2O_TEMP[cellLoc]);
		 NC_ALG_RESP_POT[cellLoc]  = 
		 		( UNSAT_DEPTH[cellLoc]<0.05 ) ? 
		 		( ALG_RC_RESP*ALG_TEMP_CF[cellLoc]*NC_ALG[cellLoc] ) : 
		 		( 0.0);
		 NC_ALG_RESP[cellLoc]  =  
		 		( NC_ALG_RESP_POT[cellLoc]*DT>NC_ALG[cellLoc] ) ? 
		 		( NC_ALG[cellLoc]/DT ) : 
		 		( NC_ALG_RESP_POT[cellLoc]);
                     /* calcareous periphyton */
		 C_ALG_RESP_POT[cellLoc]  = 
		 		( UNSAT_DEPTH[cellLoc]<0.05 ) ? 
		 		( ALG_RC_RESP*ALG_TEMP_CF[cellLoc]*C_ALG[cellLoc] ) : 
		 		( 0.0);
		 C_ALG_RESP[cellLoc]  =  
		 		( C_ALG_RESP_POT[cellLoc]*DT>C_ALG[cellLoc] ) ? 
		 		( C_ALG[cellLoc]/DT ) : 
		 		( C_ALG_RESP_POT[cellLoc]);
		 NC_ALG_PR1[cellLoc] = NC_ALG_RESP[cellLoc];
                     /* calcareous periphyton */
		 C_ALG_PR1[cellLoc]  = C_ALG_RESP[cellLoc];
                     /* ALG_AVAIL[cellLoc] = 0.001*NC_ALG[cellLoc]*Max (1.0-Max((ALG_SAT[cellLoc]-NC_ALG[cellLoc])/(ALG_SAT[cellLoc]-ALG_REFUGE[cellLoc]),0.0),0.0); */
                     /* total of calc and noncalc algae */
                 ALG_AVAIL[cellLoc]  = 0.001*(C_ALG[cellLoc]+NC_ALG[cellLoc])*Max (1.0-Max((ALG_SAT[cellLoc]- (C_ALG[cellLoc]+NC_ALG[cellLoc]) )/(ALG_SAT[cellLoc]-ALG_REFUGE[cellLoc]),0.0),0.0);

		 CONS_INGEST_POT[cellLoc] = CONS_TEMP_CF*CONS_BIOM[cellLoc] *CONS_RC_INGEST*(1.0-CONS_BIOM[cellLoc]/CONS_MAX);
		 OM_TOT_C_AVAIL[cellLoc] = DOM_C[cellLoc]+FLOC_C_AVAIL[cellLoc]+STDET_AVAIL[cellLoc]+ALG_AVAIL[cellLoc]+PHBIO_AVAIL[cellLoc]+NPHBIO_AVAIL[cellLoc];
		 CONS_INGEST[cellLoc] =  ( CONS_INGEST_POT[cellLoc]*DT>OM_TOT_C_AVAIL[cellLoc]*1000.0 ) ? ( OM_TOT_C_AVAIL[cellLoc]*1000.0/DT ) : ( CONS_INGEST_POT[cellLoc]);
		 CONS_SPEC_ING[cellLoc] = CONS_INGEST[cellLoc]*0.001/OM_TOT_C_AVAIL[cellLoc];
		 CONS_INGEST_ALGAE[cellLoc] = ALG_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
		 NC_ALG_CONS[cellLoc] = ( (NC_ALG_PR1[cellLoc]+CONS_INGEST_ALGAE[cellLoc])*DT>NC_ALG[cellLoc] ) ? ( (NC_ALG[cellLoc]-NC_ALG_PR1[cellLoc]*DT)/DT ) : ( CONS_INGEST_ALGAE[cellLoc]);
                     /* calcareous periphyton */
		 C_ALG_CONS[cellLoc]  = ( (C_ALG_PR1[cellLoc]+CONS_INGEST_ALGAE[cellLoc]/2.0)*DT>C_ALG[cellLoc] ) ? ( (C_ALG[cellLoc]-C_ALG_PR1[cellLoc]*DT)/DT ) : ( CONS_INGEST_ALGAE[cellLoc]/2.0);
		 NC_ALG_PR2[cellLoc] = NC_ALG_PR1[cellLoc]+NC_ALG_CONS[cellLoc];
                     /* calcareous periphyton */
		 C_ALG_PR2[cellLoc]  = C_ALG_PR1[cellLoc]+C_ALG_CONS[cellLoc];
		 NC_ALG_AVAIL_MORT[cellLoc] = NC_ALG[cellLoc]-ALG_REFUGE[cellLoc];
		 NC_ALG_MORT[cellLoc] = ( (NC_ALG_MORT_POT[cellLoc]+NC_ALG_PR2[cellLoc])*DT>NC_ALG_AVAIL_MORT[cellLoc] ) ? ( (NC_ALG_AVAIL_MORT[cellLoc]-NC_ALG_PR2[cellLoc]*DT)/DT ) : ( NC_ALG_MORT_POT[cellLoc]);
                     /* calcareous periphyton */
		 C_ALG_AVAIL_MORT[cellLoc]  = C_ALG[cellLoc]-ALG_REFUGE[cellLoc];
		 C_ALG_MORT[cellLoc]  = ( (C_ALG_MORT_POT[cellLoc]+C_ALG_PR2[cellLoc])*DT>C_ALG_AVAIL_MORT[cellLoc] ) ? ( (C_ALG_AVAIL_MORT[cellLoc]-C_ALG_PR2[cellLoc]*DT)/DT ) : ( C_ALG_MORT_POT[cellLoc]);

/* light, water, temperature controls apply to calc and non-calc periphyton */
		 ALG_LIGHT_EXTINCT[cellLoc]  = /* 0.1*SOM_CONC[cellLoc] + */ 0.01; /* light extinction coef */
                     /* algal self-shading implicit in density-dependent constraint function later */
		 ALG_INCID_LIGHT[cellLoc]  = SOLRADGRD[cellLoc]*Exp(-MAC_LAI[cellLoc]*ALG_SHADE_FACTOR);
                 Z_extinct = SURFACE_WAT[cellLoc]*ALG_LIGHT_EXTINCT[cellLoc];
		 I_ISat = ALG_INCID_LIGHT[cellLoc]/ALG_LIGHT_SAT;
                     /*  averaged over whole water column (based on Steele '65) */
                 ALG_LIGHT_CF[cellLoc]  = ( Z_extinct > 0.0 ) ? 
                     ( 2.718/Z_extinct * (Exp(-I_ISat * Exp(-Z_extinct)) - Exp(-I_ISat)) ) :
                     (I_ISat*Exp(1.0-I_ISat));
                     /*  low-water growth constraint ready for something better based on data */
                 ALG_WAT_CF[cellLoc]  = ( SURFACE_WAT[cellLoc]>0.0 ) ? ( 1.0 ) : ( 0.0);
/* the 2 communities have same growth response to avail phosphorus - avail P is roughly calc'd from TP */
                 NC_ALG_NUT_CF[cellLoc]  =  PO4Pconc/(PO4Pconc+NC_ALG_KS_P) ;
                 C_ALG_NUT_CF[cellLoc]  = PO4Pconc/(PO4Pconc+C_ALG_KS_P); 
 		 min_litTemp = Min(ALG_LIGHT_CF[cellLoc],ALG_TEMP_CF[cellLoc]);
                 NC_ALG_PROD_CF[cellLoc]  = Min(min_litTemp,ALG_WAT_CF[cellLoc])*NC_ALG_NUT_CF[cellLoc];
 		 C_ALG_PROD_CF[cellLoc]   = Min(min_litTemp,ALG_WAT_CF[cellLoc])*C_ALG_NUT_CF[cellLoc];
/* gross production of the 2 communities (gC/m2, NOT kgC/m2) */
                     /* density constraint contains both noncalc and calc, competition effect accentuated by calc algae */
                     /* used to increase calc growth by factor of 10 */
                 NC_ALG_GPP[cellLoc]  =  NC_ALG_PROD_CF[cellLoc]*ALG_RC_PROD*NC_ALG[cellLoc]       
         		*Max( (1.0-(AlgComp*C_ALG[cellLoc]+NC_ALG[cellLoc])/ALG_MAX[HAB[cellLoc]]),0.0);
		 C_ALG_GPP[cellLoc]  =  C_ALG_PROD_CF[cellLoc]*ALG_RC_PROD*C_ALG[cellLoc] 
		 	*Max( (1.0-(    C_ALG[cellLoc]+NC_ALG[cellLoc])/ALG_MAX[HAB[cellLoc]]),0.0);
/* check for available P mass (the MichMent kinetics nutCF do not) */
                 tmp = ( (NC_ALG_GPP[cellLoc]+C_ALG_GPP[cellLoc]) > 0) ? /*sept21*/
		 		PO4P / ( (NC_ALG_GPP[cellLoc]+C_ALG_GPP[cellLoc]) 
		 		* 0.001*ALG_PC*CELL_SIZE*DT) :
		 		1.0;
                 if (tmp < 1.0) {
                     NC_ALG_GPP[cellLoc] *= tmp;   
                     C_ALG_GPP[cellLoc] *= tmp; 
		/* can have high conc, but low mass of P avail, in presence of high peri biomass and high demand */
		/* reduce the production proportionally if excess demand is found */
		   }
/* total of calc and noncalc algae available and their total NPP */
		 NC_ALG_NPP[cellLoc]  = NC_ALG_GPP[cellLoc]-NC_ALG_RESP[cellLoc]; 
		 C_ALG_NPP[cellLoc]  = C_ALG_GPP[cellLoc]-C_ALG_RESP[cellLoc]; 

		 CONS_EGEST[cellLoc] =  CONS_INGEST[cellLoc]*(1.0-CONS_ASSIM);
		 CONS_MORT_BIOM[cellLoc] = CONS_RC_MORT*CONS_BIOM[cellLoc];
		 CONS_INGEST_DEPOM[cellLoc] = DOM_C[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_NPHBIO[cellLoc] = NPHBIO_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_PHBIO[cellLoc] = PHBIO_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_FLOC[cellLoc] = FLOC_C_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];
		 CONS_INGEST_STDETR[cellLoc] = STDET_AVAIL[cellLoc]*CONS_SPEC_ING[cellLoc];


		 DOM_QUALITY_CF[cellLoc]  = (Max(TP_SFWT_CONC_MG[cellLoc],TP_SEDWT_CONCACTMG[cellLoc]))
		 		/DOM_DECOMP_POPT; /* mg/L */
		 DOM_TEMP_CF[cellLoc] = Exp(0.20*(H2O_TEMP[cellLoc]-DOM_DECOMP_TOPT))*pow(((40.0-H2O_TEMP[cellLoc])/(40.0-DOM_DECOMP_TOPT)),(0.20*(40.0-DOM_DECOMP_TOPT)));
		 DOM_MOIST_CF[cellLoc] = pow(Max(UNSAT_MOIST_PRP[cellLoc],0.0),0.75);
		 DOM_DECOMP_POT[cellLoc] = DOM_RCDECOMP*DOM_QUALITY_CF[cellLoc]*DOM_TEMP_CF[cellLoc]*DEPOS_ORG_MAT[cellLoc]*(Min(DOM_SED_AEROB_Z[cellLoc]/DOM_MAXDEPTH[HAB[cellLoc]],1.0)*DOM_MOIST_CF[cellLoc]+DOM_DECOMPRED*Min(DOM_SED_ANAEROB_Z[cellLoc]/DOM_MAXDEPTH[HAB[cellLoc]],1.0));
/* 		 DOM_CONS_POT[cellLoc] = CONS_INGEST_DEPOM[cellLoc]/DOM_CTOOM; */

		 FIRE_DIRECT[cellLoc] = 0;
		 OM_ABV_SED[cellLoc] = STAND_DETRITUS[cellLoc]/STDET_CTOOM[HAB[cellLoc]]*STDET_ABVBEL[HAB[cellLoc]] +MAC_NOPH_BIOMAS[cellLoc]/NPHBIO_CTOOM[HAB[cellLoc]]*NPHBIO_ABVBEL[HAB[cellLoc]] +MAC_PH_BIOMAS[cellLoc]/PHBIO_CTOOM[HAB[cellLoc]];
		 FIRE_FUEL_LOADING[cellLoc] = OM_ABV_SED[cellLoc]*1000.0;

		 FIRE_MOIST_EFFECT[cellLoc] = 5.0*SURFACE_WAT[cellLoc]/MAC_HEIGHT[cellLoc] + 2.0*SF_WT_FROM_RAIN[cellLoc] + UNSAT_MOIST_PRP[cellLoc];
		 FIRE_FUEL_BULK_DENS[cellLoc] = FIRE_FUEL_LOADING[cellLoc]/MAC_HEIGHT[cellLoc];
		 FIRE_SPREAD_RATE[cellLoc] = ( ((FIRE_DIRECT[cellLoc]==1.0)  || ((FIRE_DIRECT[cellLoc]==2.0) && ((WIND_DIR<90.0) || (WIND_DIR>270.0)))  || ((FIRE_DIRECT[cellLoc]==3.0) && (WIND_DIR>180.0)) || ((FIRE_DIRECT[cellLoc]==4.0) && ((WIND_DIR>90.0) && (WIND_DIR<270.0))) || ((FIRE_DIRECT[cellLoc]==5.0) && (WIND_DIR<180.0)) || ((FIRE_DIRECT[cellLoc]==6.0) && (WIND_DIR<90.0)) || ((FIRE_DIRECT[cellLoc]==7.0) && (WIND_DIR>270.0)) || ((FIRE_DIRECT[cellLoc]==8.0) && ((WIND_DIR<270.0) && (WIND_DIR>180.0))) || ((FIRE_DIRECT[cellLoc]==9.0) && ((WIND_DIR>90.0) && (WIND_DIR<180.0)))) ) ? ( (FIRE_WIND_EFFECT[cellLoc]*FIRE_FUEL_HEAT[HAB[cellLoc]] * FIRE_FUEL_LOADING[cellLoc] * FIRE_RX_VELOC[HAB[cellLoc]] * 1.0/Exp(FIRE_MOIST_EFFECT[cellLoc]) ) / (FIRE_FUEL_BULK_DENS[cellLoc]*FIRE_HEAT_FOR_IGNIT)   ) : ( 0.0);
		 FIRE_SPECIFIC_RATE[cellLoc] = 0.0; 

		 DOM_FIRE[cellLoc] = ( DEPOS_ORG_MAT[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*DT>DEPOS_ORG_MAT[cellLoc] ) ? ( DEPOS_ORG_MAT[cellLoc]/DT ) : ( DEPOS_ORG_MAT[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]);
		 DOM_PR1[cellLoc] = DOM_FIRE[cellLoc];
		 DOM_CONSUMED[cellLoc] = ( (DOM_CONS_POT[cellLoc]+DOM_PR1[cellLoc])*DT>DEPOS_ORG_MAT[cellLoc] ) ? ( (DEPOS_ORG_MAT[cellLoc]-DOM_PR1[cellLoc]*DT)/DT ) : ( DOM_CONS_POT[cellLoc]);
		 DOM_PR2[cellLoc] = DOM_CONSUMED[cellLoc]+DOM_PR1[cellLoc];
		 DOM_DECOMP[cellLoc] =  ( (DOM_DECOMP_POT[cellLoc]+DOM_PR2[cellLoc])*DT>DEPOS_ORG_MAT[cellLoc] ) ? ( (DEPOS_ORG_MAT[cellLoc]-DOM_PR2[cellLoc]*DT)/DT ) : ( DOM_DECOMP_POT[cellLoc]);
/* added for P DOM stoich */
                 DOP_DECOMP[cellLoc] = DOM_DECOMP[cellLoc] * DOM_P_OM[cellLoc]; 

		 FIRE_TOT_DIST[cellLoc] = 0;

		 SAT_VS_UNSAT[cellLoc] = 1/Exp(100.0*Max((SURFACE_WAT[cellLoc]-UNSAT_DEPTH[cellLoc]),0.0));
		 HYD_MAX_SED_WAT[cellLoc] = HYD_POROSITY[HAB[cellLoc]]*SED_ELEV[cellLoc];
		 SAT_WT_POT[cellLoc] = Max((HYD_MAX_SED_WAT[cellLoc]-SAT_WATER[cellLoc]),0.0);
		 SF_WT_TO_SAT_DOWNFLOW[cellLoc] = ( (1.0-SAT_VS_UNSAT[cellLoc])*SAT_WT_POT[cellLoc]*DT>SURFACE_WAT[cellLoc] ) ? ( SURFACE_WAT[cellLoc]/DT ) : ( (1.0-SAT_VS_UNSAT[cellLoc])*SAT_WT_POT[cellLoc]);
		 UNSAT_CAP_CHANGE[cellLoc] = UNSAT_CAP[cellLoc]-delay01(UNSAT_CAP[cellLoc],1.0,ix,iy);
		 SAT_TO_UNSAT_POT[cellLoc] = ( UNSAT_CAP_CHANGE[cellLoc] >0.0 ) ? ( UNSAT_CAP_CHANGE[cellLoc]*(HYD_POROSITY[HAB[cellLoc]]-HYD_SPEC_YIELD[HAB[cellLoc]]) ) : ( 0.0);
		 SAT_WT_RECHG[cellLoc] = ( HYD_RCRECHG*DT>SAT_WATER[cellLoc] ) ? ( SAT_WATER[cellLoc]/DT ) : ( HYD_RCRECHG);
		 SAT_WT_PR1[cellLoc] = SAT_WT_RECHG[cellLoc];
		 SAT_TO_UNSAT_FL[cellLoc] =  ( (SAT_TO_UNSAT_POT[cellLoc]+SAT_WT_PR1[cellLoc])*DT > SAT_WATER[cellLoc] ) ? ( (SAT_WATER[cellLoc]-SAT_WT_PR1[cellLoc]*DT)/DT ) : ( SAT_TO_UNSAT_POT[cellLoc]);
		 SAT_WT_PR2[cellLoc] = SAT_TO_UNSAT_FL[cellLoc]+SAT_WT_PR1[cellLoc];
		 SAT_SURF_UP_FL[cellLoc] = Max((SAT_WATER[cellLoc]-HYD_MAX_SED_WAT[cellLoc]-SAT_WT_PR2[cellLoc]*DT)/DT, 0.0);
		 SF_WT_POT_INF[cellLoc] = ( (SURFACE_WAT[cellLoc]<HYD_RCINFILT[HAB[cellLoc]]*DT) ) ? ( SURFACE_WAT[cellLoc]/DT ) : ( HYD_RCINFILT[HAB[cellLoc]]);
		 UNSAT_WT_POT[cellLoc] = Max(UNSAT_CAP[cellLoc]-UNSAT_WATER[cellLoc],0.0);
		 SF_WT_INFILTRATION[cellLoc] = ( SF_WT_POT_INF[cellLoc]*SAT_VS_UNSAT[cellLoc]*DT>UNSAT_WT_POT[cellLoc]-SAT_WT_PR2[cellLoc]*DT ) ? ( (UNSAT_WT_POT[cellLoc]-SAT_WT_PR2[cellLoc]*DT)/DT ) : ( SF_WT_POT_INF[cellLoc]*SAT_VS_UNSAT[cellLoc]);
		 HYD_DOM_ACTWAT_PRES[cellLoc] = ( HYD_DOM_ACTWAT_VOL[cellLoc] > 0.01 ) ? ( 1.0 ) : ( 0.0);
		 HYD_WATER_AVAIL[cellLoc] = Min(1.0, UNSAT_MOIST_PRP[cellLoc]+Exp(-10.0*Max(UNSAT_DEPTH[cellLoc]-NPHBIO_ROOTDEPTH[HAB[cellLoc]],0.0)));

		 NPHBIO_FIRE[cellLoc] = ( NPHBIO_AVAIL[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*NPHBIO_ABVBEL[HAB[cellLoc]]*DT >NPHBIO_AVAIL[cellLoc] ) ? ( NPHBIO_AVAIL[cellLoc]/DT ) : ( NPHBIO_AVAIL[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*NPHBIO_ABVBEL[HAB[cellLoc]]);
		 NPHBIO_PR1[cellLoc] = NPHBIO_FIRE[cellLoc];
/* now remove the P and OM associated with that C */
                 nphbio_fire_P[cellLoc] = NPHBIO_FIRE[cellLoc] * mac_nph_PC[cellLoc];
                 nphbio_fire_OM[cellLoc] = NPHBIO_FIRE[cellLoc] / mac_nph_CtoOM[cellLoc];
		 PHBIO_GROW_MAX[cellLoc] = PHBIO_RCNPP[HAB[cellLoc]]*MAC_PH_BIOMAS[cellLoc]* (1.0-MAC_PH_BIOMAS[cellLoc]/PHBIO_MAX[HAB[cellLoc]]);
		 MAC_LIGHT_CF[cellLoc] = SOLRADGRD[cellLoc]/MAC_LIGHTSAT[HAB[cellLoc]]*Exp(1.0-SOLRADGRD[cellLoc]/MAC_LIGHTSAT[HAB[cellLoc]]);
		 MAC_TEMP_CF[cellLoc] = Exp(0.20*(AIR_TEMP[cellLoc]-MAC_TEMPOPT[HAB[cellLoc]])) *pow(((40.0-AIR_TEMP[cellLoc])/(40.0-MAC_TEMPOPT[HAB[cellLoc]])),(0.20*(40.0-MAC_TEMPOPT[HAB[cellLoc]])));
		 MAC_WATER_AVAIL_CF[cellLoc] = graph8(0x0,HYD_WATER_AVAIL[cellLoc]);
		 MAC_WATER_CF[cellLoc] = Min(MAC_WATER_AVAIL_CF[cellLoc], Max(1.0-Max((SURFACE_WAT[cellLoc]-MAC_WAT_TOLER[HAB[cellLoc]])/MAC_WAT_TOLER[HAB[cellLoc]],0.0),0.0));
		 MAC_NUT_CF[cellLoc] = Min( DIN_SEDWT_CONCACT[cellLoc]/(DIN_SEDWT_CONCACT[cellLoc]+MAC_KSN[HAB[cellLoc]]*0.001), TP_SEDWT_CONCACT[cellLoc]/(TP_SEDWT_CONCACT[cellLoc]+MAC_KSP[HAB[cellLoc]]*0.001) );

		 MAC_SALT_CF[cellLoc] = ( MAC_SALIN_THRESH[HAB[cellLoc]]>0.0 ) ? (  Max( 1.0-Max(SAL_MEAN[cellLoc]-MAC_SALIN_THRESH[HAB[cellLoc]],0.0)/MAC_SALIN_THRESH[HAB[cellLoc]],0.0) ) : ( Max(1.0-SAL_MEAN[cellLoc],0.0));
                 min_litTemp = Min(MAC_LIGHT_CF[cellLoc], MAC_TEMP_CF[cellLoc]);
                 MAC_PROD_CF[cellLoc]  = Min(min_litTemp,MAC_WATER_CF[cellLoc])
		 		*MAC_NUT_CF[cellLoc]*MAC_SALT_CF[cellLoc];
		 PHBIO_NPP[cellLoc] = PHBIO_RCNPP[HAB[cellLoc]]*MAC_PROD_CF[cellLoc]*MAC_PH_BIOMAS[cellLoc]* (1.0-MAC_TOT_BIOM[cellLoc]/MAC_MAX_BIO[cellLoc]);
/* check for available P mass that will be taken up (MichMent kinetics in nutCF does not) */
		 tmp = (PHBIO_NPP[cellLoc] > 0) ? /*sept21*/
		 		TP_SED_WT[cellLoc] / ( PHBIO_NPP[cellLoc] * NPHBIO_PC[HAB[cellLoc]]*CELL_SIZE*DT) :
		 		1.0;
                 if (tmp < 1.0) PHBIO_NPP[cellLoc] *= tmp;   
		/* reduce the production proportionally if excess demand is found */
		/* can have high conc, but low mass of P avail, in presence of high plant biomass and high demand */
/* now add the P and OM associated with that C */
                 phbio_npp_P[cellLoc] = PHBIO_NPP[cellLoc] * PHBIO_PC[HAB[cellLoc]];     /* habitat-specfic stoichiometry */
                 phbio_npp_OM[cellLoc] = PHBIO_NPP[cellLoc] / PHBIO_CTOOM[HAB[cellLoc]]; /* habitat-specfic stoichiometry */

/*oct99 added init ("target") ph/nph ratio and new transloc algorithm */
                 MAC_PHtoNPH_Init = PHBIO_MAX[HAB[cellLoc]] / NPHBIO_MAX[HAB[cellLoc]] ;
                 MAC_PHtoNPH = MAC_PH_BIOMAS[cellLoc] / MAC_NOPH_BIOMAS[cellLoc];

                 PHBIO_TRANSLOC_POT[cellLoc]  = (MAC_PHtoNPH<MAC_PHtoNPH_Init) ?
                     (exp(Mac_transl_rc[HAB[cellLoc]] *(MAC_PHtoNPH_Init-MAC_PHtoNPH)) - 1.0) :
                     (0.0); 

/*                  PHBIO_TRANSLOC_POT[cellLoc] = ( MAC_SPR_GROW[HAB[cellLoc]]==(1.0+Int(DAYJUL/30.4)) ) ? ( 0.5*Max(PHBIO_GROW_MAX[cellLoc]-PHBIO_NPP[cellLoc],0.0) ) : ( 0.0); */
		 PHBIO_TRANSLOC[cellLoc] =  ( (PHBIO_TRANSLOC_POT[cellLoc]+NPHBIO_PR1[cellLoc])*DT >NPHBIO_AVAIL[cellLoc] ) ? ( (NPHBIO_AVAIL[cellLoc]-NPHBIO_PR1[cellLoc]*DT)/DT ) : ( PHBIO_TRANSLOC_POT[cellLoc]);
		 NPHBIO_PR2[cellLoc] = NPHBIO_PR1[cellLoc]+PHBIO_TRANSLOC[cellLoc];
/*  now remove the P and OM associated with that C */
                 phbio_transl_P[cellLoc] = PHBIO_TRANSLOC[cellLoc] * mac_nph_PC[cellLoc];
                 phbio_transl_OM[cellLoc] = PHBIO_TRANSLOC[cellLoc] / mac_nph_CtoOM[cellLoc];
		 NPHBIO_CONSUMED[cellLoc] = ( (NPHBIO_PR2[cellLoc]+CONS_INGEST_NPHBIO[cellLoc])*DT>NPHBIO_AVAIL[cellLoc] ) ? ( (NPHBIO_AVAIL[cellLoc]-NPHBIO_PR2[cellLoc]*DT)/DT ) : ( CONS_INGEST_NPHBIO[cellLoc]);
		 NPHBIO_PR3[cellLoc] = NPHBIO_PR2[cellLoc]+NPHBIO_CONSUMED[cellLoc];
		 NPHBIO_MORT_POT[cellLoc] = NPHBIO_AVAIL[cellLoc]*PHBIO_RCMORT[HAB[cellLoc]]*(1.0-MAC_PH_BIOMAS[cellLoc]/PHBIO_MAX[HAB[cellLoc]]);
		 NPHBIO_MORT[cellLoc] =  ( (NPHBIO_PR3[cellLoc]+NPHBIO_MORT_POT[cellLoc])*DT>NPHBIO_AVAIL[cellLoc] ) ? ( (NPHBIO_AVAIL[cellLoc]-NPHBIO_PR3[cellLoc]*DT)/DT ) : ( NPHBIO_MORT_POT[cellLoc]);
		 PHBIO_MORT_POT[cellLoc] = PHBIO_RCMORT[HAB[cellLoc]] *PHBIO_AVAIL[cellLoc] *(1.0-MAC_WATER_AVAIL_CF[cellLoc]);
/* now remove the P and OM associated with that C */
                 nphbio_mort_P[cellLoc] = NPHBIO_MORT[cellLoc] * mac_nph_PC[cellLoc];
                 nphbio_mort_OM[cellLoc] = NPHBIO_MORT[cellLoc] / mac_nph_CtoOM[cellLoc];

		 NPHBIO_TRANSLOC_POT[cellLoc] = (MAC_PHtoNPH>MAC_PHtoNPH_Init) ?
                     (exp(Mac_transl_rc[HAB[cellLoc]] *(MAC_PHtoNPH-MAC_PHtoNPH_Init)) - 1.0) :
                     (0.0); 
		 PHBIO_FIRE[cellLoc] = ( PHBIO_AVAIL[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*DT>PHBIO_AVAIL[cellLoc] ) ? ( PHBIO_AVAIL[cellLoc]/DT ) : ( PHBIO_AVAIL[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]);
		 PHBIO_PR1[cellLoc] = PHBIO_FIRE[cellLoc];
/* now remove the P and OM associated with that C */
                 phbio_fire_P[cellLoc] = PHBIO_FIRE[cellLoc] * mac_ph_PC[cellLoc];
                 phbio_fire_OM[cellLoc] = PHBIO_FIRE[cellLoc] / mac_ph_CtoOM[cellLoc];
		 NPHBIO_TRANSLOC[cellLoc] = ( (NPHBIO_TRANSLOC_POT[cellLoc]+PHBIO_PR1[cellLoc])*DT >MAC_PH_BIOMAS[cellLoc] ) ? ( (MAC_PH_BIOMAS[cellLoc]-PHBIO_PR1[cellLoc]*DT)/DT ) : ( NPHBIO_TRANSLOC_POT[cellLoc]);
/*  now remove the P and OM associated with that C */
                 nphbio_transl_P[cellLoc] = NPHBIO_TRANSLOC[cellLoc] * mac_ph_PC[cellLoc];
                 nphbio_transl_OM[cellLoc] = NPHBIO_TRANSLOC[cellLoc] / mac_ph_CtoOM[cellLoc];
		 PHBIO_PR2[cellLoc] = NPHBIO_TRANSLOC[cellLoc]+PHBIO_PR1[cellLoc];
		 PHBIO_CONSUMED[cellLoc] =  ( (CONS_INGEST_PHBIO[cellLoc]+PHBIO_PR2[cellLoc])*DT > PHBIO_AVAIL[cellLoc] ) ? ( (PHBIO_AVAIL[cellLoc]-PHBIO_PR2[cellLoc]*DT)/DT ) : ( CONS_INGEST_PHBIO[cellLoc]);
		 PHBIO_PR3[cellLoc] = PHBIO_CONSUMED[cellLoc]+PHBIO_PR2[cellLoc];
		 PHBIO_MORT[cellLoc] = ( (PHBIO_MORT_POT[cellLoc]+PHBIO_PR3[cellLoc])*DT>PHBIO_AVAIL[cellLoc] ) ? ( (PHBIO_AVAIL[cellLoc]-PHBIO_PR3[cellLoc]*DT)/DT ) : ( PHBIO_MORT_POT[cellLoc]);
/*  now remove the P associated with that C */
                 phbio_mort_P[cellLoc] = PHBIO_MORT[cellLoc] * mac_ph_PC[cellLoc];
                 phbio_mort_OM[cellLoc] = PHBIO_MORT[cellLoc] / mac_ph_CtoOM[cellLoc];

                 stdet_prod_P[cellLoc] = (1.0-DOM_PHBIO_PROP[HAB[cellLoc]])*phbio_mort_P[cellLoc]
		 		+ (1.0-DOM_NPHBIO_PROP[HAB[cellLoc]]) *nphbio_mort_P[cellLoc];
                 stdet_prod_OM[cellLoc] = (1.0-DOM_PHBIO_PROP[HAB[cellLoc]]) * phbio_mort_OM[cellLoc]
		 		+ (1.0-DOM_NPHBIO_PROP[HAB[cellLoc]]) * nphbio_mort_OM[cellLoc];
		 STDET_BURN_POT[cellLoc] = STAND_DETRITUS[cellLoc]*FIRE_SPECIFIC_RATE[cellLoc]*STDET_ABVBEL[HAB[cellLoc]];
		 STDET_BURN[cellLoc] =  ( STDET_BURN_POT[cellLoc]*DT>STAND_DETRITUS[cellLoc] ) ? ( STAND_DETRITUS[cellLoc]/DT ) : ( STDET_BURN_POT[cellLoc]);
/*  now remove the P associated with that C */
                 stdet_burn_P[cellLoc] = STDET_BURN[cellLoc] * stdet_PC[cellLoc];
                 stdet_burn_OM[cellLoc] = STDET_BURN[cellLoc] / stdet_CtoOM[cellLoc];
		 STDET_PR1[cellLoc] = STDET_BURN[cellLoc];
		 STDET_CONSUMED[cellLoc] = ( (CONS_INGEST_STDETR[cellLoc]+STDET_PR1[cellLoc])*DT>STAND_DETRITUS[cellLoc] ) ? ( (STAND_DETRITUS[cellLoc]-STDET_PR1[cellLoc]*DT)/DT ) : ( CONS_INGEST_STDETR[cellLoc]);
		 STDET_PR2[cellLoc] = STDET_CONSUMED[cellLoc]+STDET_PR1[cellLoc];
		 STDET_TO_DOM_POT[cellLoc] = CONS_INGEST_STDETR[cellLoc]*STDET_SHRED[HAB[cellLoc]]+ STAND_DETRITUS[cellLoc]*0.1* Max(1.0-Max(WIND_STORM[cellLoc]-WIND_SPEED_CANOPY[cellLoc],0.0)/(WIND_STORM[cellLoc]-STDET_WIND[HAB[cellLoc]]),0.0);
		 STDET_TO_DOM[cellLoc] = ( (STDET_PR2[cellLoc]+STDET_TO_DOM_POT[cellLoc])*DT>STAND_DETRITUS[cellLoc] ) ? ( (STAND_DETRITUS[cellLoc]-STDET_PR2[cellLoc]*DT)/DT ) : ( STDET_TO_DOM_POT[cellLoc]);
/*  now remove the P associated with that C */
                 stdet_toDOM_P[cellLoc] = STDET_TO_DOM[cellLoc] * stdet_PC[cellLoc];
                 stdet_toDOM_OM[cellLoc] = STDET_TO_DOM[cellLoc] / stdet_CtoOM[cellLoc];

		 FLOC_DECOMP_QUAL_CF[cellLoc] = Min(TP_SFWT_CONC_MG[cellLoc]/DOM_DECOMP_POPT,1.0) *Min(DOM_C_OM_OPT/FLOC_CTOOM[HAB[cellLoc]],1.0);
		 FLOC_DECOMP_POT[cellLoc] = DOM_RCDECOMP*FLOC[cellLoc]*DOM_TEMP_CF[cellLoc] *FLOC_DECOMP_QUAL_CF[cellLoc];
		 FLOC_CONS_POT[cellLoc] = CONS_INGEST_FLOC[cellLoc]/FLOC_CTOOM[HAB[cellLoc]]*CELL_SIZE;
		 FLOC_CONSUMED[cellLoc] =  ( FLOC_CONS_POT[cellLoc]*DT>FLOC[cellLoc] ) ? ( FLOC[cellLoc]/DT ) : ( FLOC_CONS_POT[cellLoc]);
		 FLOC_PR1[cellLoc] = FLOC_CONSUMED[cellLoc];
		 FLOC_DECOMP[cellLoc] = ( (FLOC_DECOMP_POT[cellLoc]+FLOC_PR1[cellLoc])*DT>FLOC[cellLoc] ) ? ( (FLOC[cellLoc]-FLOC_PR1[cellLoc]*DT)/DT ) : ( FLOC_DECOMP_POT[cellLoc]);
		 FLOC_DEPO_POT[cellLoc] = (  SURFACE_WAT[cellLoc] > DetentZ ) ? ( FLOC[cellLoc]*0.05 ) : ( FLOC[cellLoc]/DT);
		 FLOC_PR2[cellLoc] = FLOC_DECOMP[cellLoc]+FLOC_PR1[cellLoc];
		 FLOC_PR3[cellLoc] = FLOC_PR2[cellLoc];
		 FLOC_DEPO[cellLoc] = ( (FLOC_DEPO_POT[cellLoc]+FLOC_PR3[cellLoc])*DT>FLOC[cellLoc] ) ? ( (FLOC[cellLoc]-FLOC_PR3[cellLoc]*DT)/DT ) : ( FLOC_DEPO_POT[cellLoc]);
 
		 HYD_MANNINGS_N[cellLoc]  = Max(-Abs((MAC_MAXROUGH[HAB[cellLoc]]-MAC_MINROUGH[HAB[cellLoc]])*  (pow(2.0,(1.0-SURFACE_WAT[cellLoc]/MAC_HEIGHT[cellLoc]))-1.0)) + MAC_MAXROUGH[HAB[cellLoc]],MAC_MINROUGH[HAB[cellLoc]]);
	}
  }
}

/*******
generate output
******/

void gen_output(int step, ViewParm *view)
{
    static int iw[30000];
    int        oIndex, ix, iy, cellLoc;
    ViewParm   *vp;


    static outVar_struct tgen[] = {
	    { &ALG_INCID_LIGHT, "ALG_INCID_LIGHT", 'f' },
	    { &ALG_LIGHT_EXTINCT, "ALG_LIGHT_EXTINCT", 'f' },
	    { &ALG_LIGHT_CF, "ALG_LIGHT_CF", 'f' },
	    { &ALG_WAT_CF, "ALG_WAT_CF", 'f' },
	    { &ALG_TEMP_CF, "ALG_TEMP_CF", 'f' },
	    { &ALG_AVAIL, "ALG_AVAIL", 'f' },
	    { &ALG_REFUGE, "ALG_REFUGE", 'f' },
	    { &ALG_SAT, "ALG_SAT", 'f' },
	    { &NC_ALG_AVAIL_MORT, "NC_ALG_AVAIL_MORT", 'f' },
	    { &NC_ALG_CONS, "NC_ALG_CONS", 'f' },
	    { &NC_ALG_GPP, "NC_ALG_GPP", 'f' },
	    { &NC_ALG_MORT, "NC_ALG_MORT", 'f' },
	    { &NC_ALG_MORT_POT, "NC_ALG_MORT_POT", 'f' },
	    { &NC_ALG_NPP, "NC_ALG_NPP", 'f' },
	    { &NC_ALG_NUT_CF, "NC_ALG_NUT_CF", 'f' },
	    { &NC_ALG_PROD_CF, "NC_ALG_PROD_CF", 'f' },
	    { &NC_ALG_RESP, "NC_ALG_RESP", 'f' },
	    { &NC_ALG_RESP_POT, "NC_ALG_RESP_POT", 'f' },
	    { &NC_ALG, "NC_ALG", 'f' },
	    { &C_ALG_AVAIL_MORT, "C_ALG_AVAIL_MORT", 'f' },
	    { &C_ALG_CONS, "C_ALG_CONS", 'f' },
	    { &C_ALG_GPP, "C_ALG_GPP", 'f' },
	    { &C_ALG_MORT, "C_ALG_MORT", 'f' },
	    { &C_ALG_MORT_POT, "C_ALG_MORT_POT", 'f' },
	    { &C_ALG_NPP, "C_ALG_NPP", 'f' },
	    { &C_ALG_NUT_CF, "C_ALG_NUT_CF", 'f' },
	    { &C_ALG_PROD_CF, "C_ALG_PROD_CF", 'f' },
	    { &C_ALG_RESP, "C_ALG_RESP", 'f' },
	    { &C_ALG_RESP_POT, "C_ALG_RESP_POT", 'f' },
	    { &C_ALG, "C_ALG", 'f' },
	    { &CONS_BIOM, "CONS_BIOM", 'f' },
	    { &CONS_EGEST, "CONS_EGEST", 'f' },
	    { &CONS_INGEST, "CONS_INGEST", 'f' },
	    { &CONS_INGEST_ALGAE, "CONS_INGEST_ALGAE", 'f' },
	    { &CONS_INGEST_DEPOM, "CONS_INGEST_DEPOM", 'f' },
	    { &CONS_INGEST_NPHBIO, "CONS_INGEST_NPHBIO", 'f' },
	    { &CONS_INGEST_PHBIO, "CONS_INGEST_PHBIO", 'f' },
	    { &CONS_INGEST_POT, "CONS_INGEST_POT", 'f' },
	    { &CONS_INGEST_FLOC, "CONS_INGEST_FLOC", 'f' },
	    { &CONS_INGEST_STDETR, "CONS_INGEST_STDETR", 'f' },
	    { &CONS_MORT_BIOM, "CONS_MORT_BIOM", 'f' },
	    { &CONS_RESPIRATION, "CONS_RESPIRATION", 'f' },
	    { &CONS_SPEC_ING, "CONS_SPEC_ING", 'f' },
	    { &DEPOS_ORG_MAT, "DEPOS_ORG_MAT", 'f' },
	    { &DIN_DNFLOW, "DIN_DNFLOW", 'f' },
	    { &DIN_DNFLOW_POT, "DIN_DNFLOW_POT", 'f' },
	    { &DIN_FR_RAIN, "DIN_FR_RAIN", 'f' },
	    { &DIN_SED_CONC_INT, "DIN_SED_CONC_INT", 'f' },
	    { &DIN_SED_WT, "DIN_SED_WT", 'f' },
	    { &DIN_SEDWT_CONCACT, "DIN_SEDWT_CONCACT", 'f' },
	    { &DIN_SEDWT_CONCTOT, "DIN_SEDWT_CONCTOT", 'f' },
	    { &DIN_SEDWT_DENIT, "DIN_SEDWT_DENIT", 'f' },
	    { &DIN_SEDWT_DENIT_POT, "DIN_SEDWT_DENIT_POT", 'f' },
	    { &DIN_SEDWT_MINER, "DIN_SEDWT_MINER", 'f' },
	    { &DIN_SEDWT_UPTAKE, "DIN_SEDWT_UPTAKE", 'f' },
	    { &DIN_SEDWT_UPTAKE_POT, "DIN_SEDWT_UPTAKE_POT", 'f' },
	    { &DIN_SF_WT, "DIN_SF_WT", 'f' },
	    { &DIN_SFWT_CONC, "DIN_SFWT_CONC", 'f' },
	    { &DIN_SFWT_MINER, "DIN_SFWT_MINER", 'f' },
	    { &DIN_SFWT_UPT_POT, "DIN_SFWT_UPT_POT", 'f' },
	    { &DIN_SFWT_UPTAKE, "DIN_SFWT_UPTAKE", 'f' },
	    { &DIN_UPFLOW, "DIN_UPFLOW", 'f' },
	    { &DIN_UPFLOW_POT, "DIN_UPFLOW_POT", 'f' },
	    { &DOM_Z, "DOM_Z", 'f' },
	    { &DOM_C, "DOM_C", 'f' },
	    { &DOM_CONS_POT, "DOM_CONS_POT", 'f' },
	    { &DOM_CONSUMED, "DOM_CONSUMED", 'f' },
	    { &DOM_DECOMP, "DOM_DECOMP", 'f' },
	    { &DOM_DECOMP_POT, "DOM_DECOMP_POT", 'f' },
	    { &DOM_FIRE, "DOM_FIRE", 'f' },
	    { &DOM_FR_CONS, "DOM_FR_CONS", 'f' },
	    { &DOM_FR_MAC, "DOM_FR_MAC", 'f' },
	    { &DOM_FR_FLOC, "DOM_FR_FLOC", 'f' },
	    { &DOM_FR_STDET, "DOM_FR_STDET", 'f' },
	    { &DOM_MOIST_CF, "DOM_MOIST_CF", 'f' },
	    { &DOM_QUALITY_CF, "DOM_QUALITY_CF", 'f' },
	    { &DOM_SED_AEROB_Z, "DOM_SED_AEROB_Z", 'f' },
	    { &DOM_SED_ANAEROB_Z, "DOM_SED_ANAEROB_Z", 'f' },
	    { &DOM_TEMP_CF, "DOM_TEMP_CF", 'f' },
	    { &ELEVATION, "ELEVATION", 'f' },
	    { &FIRE_DIRECT, "FIRE_DIRECT", 'f' },
	    { &FIRE_DIST_DUMP, "FIRE_DIST_DUMP", 'f' },
	    { &FIRE_DIST_IN, "FIRE_DIST_IN", 'f' },
	    { &FIRE_FUEL_BULK_DENS, "FIRE_FUEL_BULK_DENS", 'f' },
	    { &FIRE_FUEL_LOADING, "FIRE_FUEL_LOADING", 'f' },
	    { &FIRE_INIT, "FIRE_INIT", 'f' },
	    { &FIRE_LIGHT_HIT, "FIRE_LIGHT_HIT", 'f' },
	    { &FIRE_LOSS, "FIRE_LOSS", 'f' },
	    { &FIRE_MOIST_EFFECT, "FIRE_MOIST_EFFECT", 'f' },
	    { &FIRE_SPECIFIC_RATE, "FIRE_SPECIFIC_RATE", 'f' },
	    { &FIRE_SPREAD_RATE, "FIRE_SPREAD_RATE", 'f' },
	    { &FIRE_TOT_DIST, "FIRE_TOT_DIST", 'f' },
	    { &HAB, "HAB", 'c' },
	    { &HYD_CANOPY_TRANSP, "HYD_CANOPY_TRANSP", 'f' },
	    { &HYD_DOM_ACTWAT_PRES, "HYD_DOM_ACTWAT_PRES", 'f' },
	    { &HYD_DOM_ACTWAT_VOL, "HYD_DOM_ACTWAT_VOL", 'f' },
	    { &HYD_EVAP_CALC, "HYD_EVAP_CALC", 'f' },
	    { &HYD_MANNINGS_N, "HYD_MANNINGS_N", 'f' },
	    { &HYD_MAX_SED_WAT, "HYD_MAX_SED_WAT", 'f' },
	    { &HYD_PRECIP_DAY, "HYD_PRECIP_DAY", 'f' },
	    { &HYD_RCCONDUCT, "HYD_RCCONDUCT", 'f' },
	    { &HYD_SAT_POT_TRANS, "HYD_SAT_POT_TRANS", 'f' },
	    { &HYD_SED_WAT_VOL, "HYD_SED_WAT_VOL", 'f' },
	    { &HYD_TOT_POT_TRANSP, "HYD_TOT_POT_TRANSP", 'f' },
	    { &HydTotHd, "HydTotHd", 'f' },
	    { &HYD_TRANSP, "HYD_TRANSP", 'f' },
	    { &HYD_UNSAT_POT_TRANS, "HYD_UNSAT_POT_TRANS", 'f' },
	    { &HYD_WATER_AVAIL, "HYD_WATER_AVAIL", 'f' },
	    { &MAC_HEIGHT, "MAC_HEIGHT", 'f' },
	    { &MAC_LAI, "MAC_LAI", 'f' },
	    { &MAC_LIGHT_CF, "MAC_LIGHT_CF", 'f' },
	    { &MAC_MAX_BIO, "MAC_MAX_BIO", 'f' },
	    { &MAC_NOPH_BIOMAS, "MAC_NOPH_BIOMAS", 'f' },
	    { &MAC_NUT_CF, "MAC_NUT_CF", 'f' },
	    { &MAC_PH_BIOMAS, "MAC_PH_BIOMAS", 'f' },
	    { &MAC_PROD_CF, "MAC_PROD_CF", 'f' },
	    { &MAC_REL_BIOM, "MAC_REL_BIOM", 'f' },
	    { &MAC_SALT_CF, "MAC_SALT_CF", 'f' },
	    { &MAC_TEMP_CF, "MAC_TEMP_CF", 'f' },
	    { &MAC_TOT_BIOM, "MAC_TOT_BIOM", 'f' },
	    { &MAC_WATER_CF, "MAC_WATER_CF", 'f' },
	    { &MAC_WATER_AVAIL_CF, "MAC_WATER_AVAIL_CF", 'f' },
	    { &NPHBIO_AVAIL, "NPHBIO_AVAIL", 'f' },
	    { &NPHBIO_CONSUMED, "NPHBIO_CONSUMED", 'f' },
	    { &NPHBIO_FIRE, "NPHBIO_FIRE", 'f' },
	    { &NPHBIO_MORT, "NPHBIO_MORT", 'f' },
	    { &NPHBIO_MORT_POT, "NPHBIO_MORT_POT", 'f' },
	    { &NPHBIO_REFUGE, "NPHBIO_REFUGE", 'f' },
	    { &NPHBIO_SAT, "NPHBIO_SAT", 'f' },
	    { &NPHBIO_TRANSLOC, "NPHBIO_TRANSLOC", 'f' },
	    { &NPHBIO_TRANSLOC_POT, "NPHBIO_TRANSLOC_POT", 'f' },
	    { &OM_ABV_SED, "OM_ABV_SED", 'f' },
	    { &OM_TOT_C_AVAIL, "OM_TOT_C_AVAIL", 'f' },
	    { &ON_MAP, "ON_MAP", 'c' },
	    { &PHBIO_AVAIL, "PHBIO_AVAIL", 'f' },
	    { &PHBIO_CONSUMED, "PHBIO_CONSUMED", 'f' },
	    { &PHBIO_FIRE, "PHBIO_FIRE", 'f' },
	    { &PHBIO_GROW_MAX, "PHBIO_GROW_MAX", 'f' },
	    { &PHBIO_MORT, "PHBIO_MORT", 'f' },
	    { &PHBIO_MORT_POT, "PHBIO_MORT_POT", 'f' },
	    { &PHBIO_NPP, "PHBIO_NPP", 'f' },
	    { &PHBIO_REFUGE, "PHBIO_REFUGE", 'f' },
	    { &PHBIO_SAT, "PHBIO_SAT", 'f' },
	    { &PHBIO_TRANSLOC, "PHBIO_TRANSLOC", 'f' },
	    { &PHBIO_TRANSLOC_POT, "PHBIO_TRANSLOC_POT", 'f' },
	    { &mac_nph_PC_rep, "mac_nph_PC_rep", 'f' },
	    { &mac_ph_PC_rep, "mac_ph_PC_rep", 'f' },
	    { &TP_DNFLOW, "TP_DNFLOW", 'f' },
	    { &TP_DNFLOW_POT, "TP_DNFLOW_POT", 'f' },
	    { &TP_FR_RAIN, "TP_FR_RAIN", 'f' },
	    { &TP_K, "TP_K", 'f' },
	    { &TP_SED_CONC, "TP_SED_CONC", 'f' },
	    { &TP_SED_MINER, "TP_SED_MINER", 'f' },
	    { &TP_SED_WT, "TP_SED_WT", 'f' },
	    { &TP_SEDWT_CONCACT, "TP_SEDWT_CONCACT", 'f' },
	    { &TP_SEDWT_CONCACTMG, "TP_SEDWT_CONCACTMG", 'f' },
	    { &TP_Act_to_TotRep, "TP_Act_to_TotRep", 'f' },
	    { &TP_SED_WT_AZ, "TP_SED_WT_AZ", 'f' },
	    { &TP_SEDWT_UPT_POT, "TP_SEDWT_UPT_POT", 'f' },
	    { &TP_SEDWT_UPTAKE, "TP_SEDWT_UPTAKE", 'f' },
	    { &TP_SF_WT, "TP_SF_WT", 'f' },
	    { &TP_SFWT_CONC, "TP_SFWT_CONC", 'f' },
	    { &TP_SFWT_CONC_MG, "TP_SFWT_CONC_MG", 'f' },
	    { &TP_SFWT_MINER, "TP_SFWT_MINER", 'f' },
	    { &TP_SFWT_UPT_POT, "TP_SFWT_UPT_POT", 'f' },
	    { &TP_SFWT_UPTAK, "TP_SFWT_UPTAK", 'f' },
	    { &TP_algResp, "TP_algResp", 'f' },
	    { &TP_settl, "TP_settl", 'f' },
	    { &TP_settlDays, "TP_settlDays", 'f' },
	    { &TP_SORB_POT, "TP_SORB_POT", 'f' },
	    { &TP_SORBCONC, "TP_SORBCONC", 'f' },
	    { &TP_SORBTION, "TP_SORBTION", 'f' },
	    { &TP_SORB, "TP_SORB", 'f' },
	    { &TP_UPFLOW, "TP_UPFLOW", 'f' },
	    { &TP_UPFLOW_MG, "TP_UPFLOW_MG", 'f' },
	    { &TP_UPFLOW_POT, "TP_UPFLOW_POT", 'f' },
	    { &SAL_MEAN, "SAL_MEAN", 'f' },
	    { &SAL_SED_WT, "SAL_SED_WT", 'f' },
	    { &SAL_SF_WT, "SAL_SF_WT", 'f' },
	    { &SALT_SED_TO_SF_FLOW, "SALT_SED_TO_SF_FLOW", 'f' },
	    { &SALT_SED_WT, "SALT_SED_WT", 'f' },
	    { &SALT_SFWAT_DOWNFL, "SALT_SFWAT_DOWNFL", 'f' },
	    { &SALT_SFWAT_DOWNFL_POT, "SALT_SFWAT_DOWNFL_POT", 'f' },
	    { &SALT_SURF_WT, "SALT_SURF_WT", 'f' },
	    { &SAT_ADD_FM_RISE, "SAT_ADD_FM_RISE", 'f' },
	    { &SAT_SURF_UP_FL, "SAT_SURF_UP_FL", 'f' },
	    { &SAT_TO_UNSAT_FL, "SAT_TO_UNSAT_FL", 'f' },
	    { &SAT_TO_UNSAT_POT, "SAT_TO_UNSAT_POT", 'f' },
	    { &SAT_VS_UNSAT, "SAT_VS_UNSAT", 'f' },
	    { &SAT_WATER, "SAT_WATER", 'f' },
	    { &SAT_WT_HEAD, "SAT_WT_HEAD", 'f' },
	    { &SAT_WT_POT, "SAT_WT_POT", 'f' },
	    { &SAT_WT_RECHG, "SAT_WT_RECHG", 'f' },
	    { &SAT_WT_TRANSP, "SAT_WT_TRANSP", 'f' },
	    { &SED_ELEV, "SED_ELEV", 'f' },
	    { &SED_ELEV_IC, "SED_ELEV_IC", 'f' },
	    { &SED_INACT_Z, "SED_INACT_Z", 'f' },
	    { &SF_WT_EVAP, "SF_WT_EVAP", 'f' },
	    { &SF_WT_FROM_RAIN, "SF_WT_FROM_RAIN", 'f' },
	    { &SF_WT_HEAD, "SF_WT_HEAD", 'f' },
	    { &SF_WT_INFILTRATION, "SF_WT_INFILTRATION", 'f' },
	    { &SF_WT_POT_INF, "SF_WT_POT_INF", 'f' },
	    { &SF_WT_TO_SAT_DOWNFLOW, "SF_WT_TO_SAT_DOWNFLOW", 'f' },
	    { &SFWT_VOL, "SFWT_VOL", 'f' },
	    { &FLOC_C_AVAIL, "FLOC_C_AVAIL", 'f' },
	    { &FLOC_CONS_POT, "FLOC_CONS_POT", 'f' },
	    { &FLOC_CONSUMED, "FLOC_CONSUMED", 'f' },
	    { &FLOC_DECOMP, "FLOC_DECOMP", 'f' },
	    { &FLOC_DECOMP_POT, "FLOC_DECOMP_POT", 'f' },
	    { &FLOC_DECOMP_QUAL_CF, "FLOC_DECOMP_QUAL_CF", 'f' },
	    { &FLOC_DEPO, "FLOC_DEPO", 'f' },
	    { &FLOC_DEPO_POT, "FLOC_DEPO_POT", 'f' },
	    { &FLOC_FR_ALGAE, "FLOC_FR_ALGAE", 'f' },
	    { &FLOC_FR_CONSUM, "FLOC_FR_CONSUM", 'f' },
	    { &FLOC_Z, "FLOC_Z", 'f' },
	    { &FlocP_OMrep, "FlocP_OMrep", 'f' },
	    { &STAND_DETRITUS, "STAND_DETRITUS", 'f' },
	    { &STDET_AVAIL, "STDET_AVAIL", 'f' },
	    { &STDET_BURN, "STDET_BURN", 'f' },
	    { &STDET_BURN_POT, "STDET_BURN_POT", 'f' },
	    { &STDET_CONSUMED, "STDET_CONSUMED", 'f' },
	    { &STDET_PROD, "STDET_PROD", 'f' },
	    { &STDET_REFUG, "STDET_REFUG", 'f' },
	    { &STDET_SAT, "STDET_SAT", 'f' },
	    { &STDET_TO_DOM, "STDET_TO_DOM", 'f' },
	    { &STDET_TO_DOM_POT, "STDET_TO_DOM_POT", 'f' },
	    { &stdet_PC, "stdet_PC", 'f' },
	    { &SURFACE_WAT, "SURFACE_WAT", 'f' },
	    { &FLOC, "FLOC", 'f' },
	    { &UNSAT_AVAIL, "UNSAT_AVAIL", 'f' },
	    { &UNSAT_CAP, "UNSAT_CAP", 'f' },
	    { &UNSAT_CAP_CHANGE, "UNSAT_CAP_CHANGE", 'f' },
	    { &UNSAT_DEPTH, "UNSAT_DEPTH", 'f' },
	    { &UNSAT_HYD_COND_CF, "UNSAT_HYD_COND_CF", 'f' },
	    { &UNSAT_MOIST_PRP, "UNSAT_MOIST_PRP", 'f' },
	    { &UNSAT_PERC, "UNSAT_PERC", 'f' },
	    { &UNSAT_TO_SAT_FL, "UNSAT_TO_SAT_FL", 'f' },
	    { &UNSAT_TRANSP, "UNSAT_TRANSP", 'f' },
	    { &UNSAT_WATER, "UNSAT_WATER", 'f' },
	    { &UNSAT_WT_POT, "UNSAT_WT_POT", 'f' },
	    { &WIND_STORM, "WIND_STORM", 'f' },	
	    { &AIR_TEMP, "AIR_TEMP", 'f' },
	    { &CLOUDY, "CLOUDY", 'f' },
	    { &FIRE_WIND_EFFECT, "FIRE_WIND_EFFECT", 'f' },
	    { &H2O_TEMP, "H2O_TEMP", 'f' },
	    { &HUMIDITY, "HUMIDITY", 'f' },
	    { &PAN_CH, "PAN_CH", 'f' },
	    { &PAN_CT, "PAN_CT", 'f' },
	    { &PAN_CW, "PAN_CW", 'f' },
	    { &SOLRAD274, "SOLRAD274", 'f' },
	    { &SOLRADGRD, "SOLRADGRD", 'f' },
	    { &DewPt, "DewPt", 'f' },
	    { &WIND_SPEED, "WIND_SPEED", 'f' },
	    { &WIND_SPEED_CANOPY, "WIND_SPEED_CANOPY", 'f' },
	    { &LAI_eff, "LAI_eff", 'f' },
	    { &DOP_DECOMP, "DOP_DECOMP", 'f' },
	    { &DOP, "DOP", 'f' },
	    { &DOM_P_OM, "DOM_P_OM", 'f' },
	    { &SfWatAvg, "SfWatAvg", 'f' },
	    { &HydPerAnn, "HydPerAnn", 'f' },
	    { &TotHeadAvg, "TotHeadAvg", 'f' },    
	    { &UnsatZavg, "UnsatZavg", 'f' },
	    { &UnsatMoistAvg, "UnsatMoistAvg", 'f' },    
	    { &RainAvg, "RainAvg", 'f' },
	    { &ETAvg, "ETAvg", 'f' },    
	    { &EvapAvg, "EvapAvg", 'f' },    
	    { &TranspAvg, "TranspAvg", 'f' },    
	    { &TPSfWatAvg, "TPSfWatAvg", 'f' },
	    { &TPSfUptAvg, "TPSfUptAvg", 'f' },
	    { &TPSfMinAvg, "TPSfMinAvg", 'f' },    
	    { &TP_settlAvg, "TP_settlAvg", 'f' },    
	    { &TPSedWatAvg, "TPSedWatAvg", 'f' },
	    { &TPSedUptAvg, "TPSedUptAvg", 'f' },
	    { &TPSedMinAvg, "TPSedMinAvg", 'f' },
	    { &TPSorbAvg, "TPSorbAvg", 'f' },
	    { &TPtoSOILAvg, "TPtoSOILAvg", 'f' },
	    { &TPtoVOLAvg, "TPtoVOLAvg", 'f' },
	    { &SaltSfAvg, "SaltSfAvg", 'f' },    
	    { &SaltSedAvg, "SaltSedAvg", 'f' },    
	    { &Floc_fr_phBioAvg, "Floc_fr_phBioAvg", 'f' },    
	    { &DOM_fr_stdetAvg, "DOM_fr_stdetAvg", 'f' },    
	    { &PeriAvg, "PeriAvg", 'f' },
	    { &PeriLiteCFAvg, "PeriLiteCFAvg", 'f' },
	    { &NC_PeriAvg, "NC_PeriAvg", 'f' },
	    { &NC_Peri_nppAvg, "NC_Peri_nppAvg", 'f' },
	    { &NC_Peri_mortAvg, "NC_Peri_mortAvg", 'f' },
	    { &NC_PeriRespAvg, "NC_PeriRespAvg", 'f' },
	    { &NC_PeriNutCFAvg, "NC_PeriNutCFAvg", 'f' },
	    { &NC_Peri_PCAvg, "NC_Peri_PCAvg", 'f' },
	    { &C_PeriAvg, "C_PeriAvg", 'f' },
	    { &C_Peri_nppAvg, "C_Peri_nppAvg", 'f' },
	    { &C_Peri_mortAvg, "C_Peri_mortAvg", 'f' },
	    { &C_PeriRespAvg, "C_PeriRespAvg", 'f' },
	    { &C_PeriNutCFAvg, "C_PeriNutCFAvg", 'f' },
	    { &C_Peri_PCAvg, "C_Peri_PCAvg", 'f' },
	    { &Mac_totBioAvg, "Mac_totBioAvg", 'f' },    
	    { &Mac_nphBioAvg, "Mac_nphBioAvg", 'f' },    
	    { &Mac_phBioAvg, "Mac_phBioAvg", 'f' },    
	    { &Mac_nppAvg, "Mac_nppAvg", 'f' },    
	    { &Mac_nphMortAvg, "Mac_nphMortAvg", 'f' },    
	    { &Mac_phMortAvg, "Mac_phMortAvg", 'f' },    
	    { &mac_nph_PCAvg, "mac_nph_PCAvg", 'f' },
	    { &mac_ph_PCAvg, "mac_ph_PCAvg", 'f' },
	    { &LAI_effAvg, "LAI_effAvg", 'f' },    
	    { &Manning_nAvg, "Manning_nAvg", 'f' },
	    { &MacNutCfAvg, "MacNutCfAvg", 'f' },
	    { &MacWatCfAvg, "MacWatCfAvg", 'f' },
	    { &Std_detrAvg, "Std_detrAvg", 'f' },

	    { NULL, NULL, '\0' },
	};

    outVar_struct *ptable;
    int  i;

    for (i = 0, ptable = tgen; ptable->pfvar != NULL; ptable++, i++)
    {
	vp = view + i;

	if (vp->step > 0)
                
            if (strcmp(ptable->pname,"HydPerAnn")!=0) {
                
                if  (step % vp->step == 0) {                
                    oIndex = iw[i]++;
                    write_output(oIndex, vp, *(ptable->pfvar), ptable->pname, ptable->ctype, step);
                }
            }
        
            else
                if (FMOD(DAYJUL, 273.0) ==0) { /* hydroperiod is printed at a special-case time */
                    oIndex = iw[i]++;
                    write_output(oIndex, vp, *(ptable->pfvar), ptable->pname, ptable->ctype, step);
                }
        
     
    }

        /* after printing, zero the arrays holding averages or hydroperiods */
    if (FMOD(TIME, avgIntvl) ==0) {
        zero_avg();
    }

    if (FMOD(DAYJUL, 273.0) ==0) {
        zero_hydper();
    }


} /* end of output routine */


/*******
read output parameters (readOutlist(xxx) inside read_output_parms)
was called here, but now is in DriverUtilities.c.  If the number of
variables in the output list is modified (above and in Model.outlist),
you need to modify that number in the xxx argument of readOutlist(xxx)
******/


/*******
read global parameters (that do not vary with habitat)
******/
int ReadParms( parmFile)
FILE *parmFile;
 {
     int test = 1, format=0; char tst; int numParm = 0;
     

 	getChar(parmFile,NULL,&tst); if(tst == POUND) getInt(parmFile,NULL,&format); else ungetc(tst,parmFile); 
	if(test) {
            
/* Geography, hydrology */

		scan_forward(parmFile,"SOLOMEGA=");
		if ( (test = fscanf(parmFile,"%f",&SOLOMEGA) ) <= 0 ) return test;
		fprintf(outfile,"SOLOMEGA= %f\n",SOLOMEGA);
                numParm++;

		scan_forward(parmFile,"ALTIT=");
		if ( (test = fscanf(parmFile,"%f",&ALTIT) ) <= 0 ) return test;
		fprintf(outfile,"ALTIT= %f\n",ALTIT);
                numParm++;

		scan_forward(parmFile,"LATDEG=");
		if ( (test = fscanf(parmFile,"%f",&LATDEG) ) <= 0 ) return test;
		fprintf(outfile,"LATDEG= %f\n",LATDEG);
                numParm++;

		scan_forward(parmFile,"DATUM_DISTANCE=");
		if ( (test = fscanf(parmFile,"%f",&DATUM_DISTANCE) ) <= 0 ) return test;
		fprintf(outfile,"DATUM_DISTANCE= %f\n",DATUM_DISTANCE);
                numParm++;

		scan_forward(parmFile,"HYD_IC_SFWAT=");
		if ( (test = fscanf(parmFile,"%f",&HYD_IC_SFWAT) ) <= 0 ) return test;
		fprintf(outfile,"HYD_IC_SFWAT= %f\n",HYD_IC_SFWAT);
                numParm++;

		scan_forward(parmFile,"HYD_IC_UNSATZ=");
		if ( (test = fscanf(parmFile,"%f",&HYD_IC_UNSATZ) ) <= 0 ) return test;
		fprintf(outfile,"HYD_IC_UNSATZ= %f\n",HYD_IC_UNSATZ);
                numParm++;

		scan_forward(parmFile,"DetentZ=");
		if ( (test = fscanf(parmFile,"%f",&DetentZ) ) <= 0 ) return test;
		fprintf(outfile,"DetentZ= %f\n",DetentZ);
                numParm++;

		scan_forward(parmFile,"MinCheck=");
		if ( (test = fscanf(parmFile,"%f",&MinCheck) ) <= 0 ) return test;
		fprintf(outfile,"MinCheck= %f\n",MinCheck);
                numParm++;

		scan_forward(parmFile,"HYD_ICUNSATMOIST=");
		if ( (test = fscanf(parmFile,"%f",&HYD_ICUNSATMOIST) ) <= 0 ) return test;
		fprintf(outfile,"HYD_ICUNSATMOIST= %f\n",HYD_ICUNSATMOIST);
                numParm++;

		scan_forward(parmFile,"HYD_RCRECHG=");
		if ( (test = fscanf(parmFile,"%f",&HYD_RCRECHG) ) <= 0 ) return test;
		fprintf(outfile,"HYD_RCRECHG= %f\n",HYD_RCRECHG);
                numParm++;

		scan_forward(parmFile,"HYD_PAN_K=");
		if ( (test = fscanf(parmFile,"%f",&HYD_PAN_K) ) <= 0 ) return test;
		fprintf(outfile,"HYD_PAN_K= %f\n",HYD_PAN_K);
                numParm++;
                
 /* Periphyton/Algae */

                scan_forward(parmFile,"ALG_IC_MULT=");
                if ( (test = fscanf(parmFile,"%f",&ALG_IC_MULT) ) <= 0 ) return test;
		fprintf(outfile,"ALG_IC_MULT= %f\n",ALG_IC_MULT);
                numParm++;
                
                scan_forward(parmFile,"ALG_REF_MULT=");
		if ( (test = fscanf(parmFile,"%f",&ALG_REF_MULT) ) <= 0 ) return test;
		fprintf(outfile,"ALG_REF_MULT= %f\n",ALG_REF_MULT);
                numParm++;

                scan_forward(parmFile,"ALG_SHADE_FACTOR=");
		if ( (test = fscanf(parmFile,"%f",&ALG_SHADE_FACTOR) ) <= 0 ) return test;
		fprintf(outfile,"ALG_SHADE_FACTOR= %f\n",ALG_SHADE_FACTOR);
                numParm++;

		scan_forward(parmFile,"alg_light_ext_coef=");
		if ( (test = fscanf(parmFile,"%f",&alg_light_ext_coef) ) <= 0 ) return test;
		fprintf(outfile,"alg_light_ext_coef= %f\n",alg_light_ext_coef);
                numParm++;

		scan_forward(parmFile,"ALG_LIGHT_SAT=");
		if ( (test = fscanf(parmFile,"%f",&ALG_LIGHT_SAT) ) <= 0 ) return test;
		fprintf(outfile,"ALG_LIGHT_SAT= %f\n",ALG_LIGHT_SAT);
                numParm++;

		scan_forward(parmFile,"algMortDepth=");
		if ( (test = fscanf(parmFile,"%f",&algMortDepth) ) <= 0 ) return test;
		fprintf(outfile,"algMortDepth= %f\n",algMortDepth);
                numParm++;

		scan_forward(parmFile,"ALG_RC_MORT_DRY=");
		if ( (test = fscanf(parmFile,"%f",&ALG_RC_MORT_DRY) ) <= 0 ) return test;
		fprintf(outfile,"ALG_RC_MORT_DRY= %f\n",ALG_RC_MORT_DRY);
                numParm++;

		scan_forward(parmFile,"ALG_RC_MORT=");
		if ( (test = fscanf(parmFile,"%f",&ALG_RC_MORT) ) <= 0 ) return test;
		fprintf(outfile,"ALG_RC_MORT= %f\n",ALG_RC_MORT);
                numParm++;

		scan_forward(parmFile,"ALG_RC_PROD=");
		if ( (test = fscanf(parmFile,"%f",&ALG_RC_PROD) ) <= 0 ) return test;
		fprintf(outfile,"ALG_RC_PROD= %f\n",ALG_RC_PROD);
                numParm++;

		scan_forward(parmFile,"ALG_RC_RESP=");
		if ( (test = fscanf(parmFile,"%f",&ALG_RC_RESP) ) <= 0 ) return test;
		fprintf(outfile,"ALG_RC_RESP= %f\n",ALG_RC_RESP);
                numParm++;

		scan_forward(parmFile,"alg_R_accel=");
		if ( (test = fscanf(parmFile,"%f",&alg_R_accel) ) <= 0 ) return test;
		fprintf(outfile,"alg_R_accel= %f\n",alg_R_accel);
                numParm++;

		scan_forward(parmFile,"AlgComp=");
		if ( (test = fscanf(parmFile,"%f",&AlgComp) ) <= 0 ) return test;
		fprintf(outfile,"AlgComp= %f\n",AlgComp);
                numParm++;

		scan_forward(parmFile,"NC_ALG_KS_P=");
		if ( (test = fscanf(parmFile,"%f",&NC_ALG_KS_P) ) <= 0 ) return test;
		fprintf(outfile,"NC_ALG_KS_P= %f\n",NC_ALG_KS_P);
                numParm++;

		scan_forward(parmFile,"C_ALG_KS_P=");
		if ( (test = fscanf(parmFile,"%f",&C_ALG_KS_P) ) <= 0 ) return test;
		fprintf(outfile,"C_ALG_KS_P= %f\n",C_ALG_KS_P);
                numParm++;
              /*  ALG_KS_N  - not running Nit; need parm for both periph types */

		scan_forward(parmFile,"C_ALG_threshTP=");
		if ( (test = fscanf(parmFile,"%f",&C_ALG_threshTP) ) <= 0 ) return test;
		fprintf(outfile,"C_ALG_threshTP= %f\n",C_ALG_threshTP);
                numParm++;

		scan_forward(parmFile,"alg_uptake_coef=");
		if ( (test = fscanf(parmFile,"%f",&alg_uptake_coef) ) <= 0 ) return test;
		fprintf(outfile,"alg_uptake_coef= %f\n",alg_uptake_coef);
                numParm++;

		scan_forward(parmFile,"alg_alkP_min=");
		if ( (test = fscanf(parmFile,"%f",&alg_alkP_min) ) <= 0 ) return test;
		fprintf(outfile,"alg_alkP_min= %f\n",alg_alkP_min);
                numParm++;
                
		scan_forward(parmFile,"ALG_TEMP_OPT=");
		if ( (test = fscanf(parmFile,"%f",&ALG_TEMP_OPT) ) <= 0 ) return test;
		fprintf(outfile,"ALG_TEMP_OPT= %f\n",ALG_TEMP_OPT);
                numParm++;
                
		scan_forward(parmFile,"ALG_C_TO_OM=");
		if ( (test = fscanf(parmFile,"%f",&ALG_C_TO_OM) ) <= 0 ) return test;
		fprintf(outfile,"ALG_C_TO_OM= %f\n",ALG_C_TO_OM);
                numParm++;
                
		scan_forward(parmFile,"ALG_NC=");
		if ( (test = fscanf(parmFile,"%f",&ALG_NC) ) <= 0 ) return test;
		fprintf(outfile,"ALG_NC= %f\n",ALG_NC);
                numParm++;
                
		scan_forward(parmFile,"ALG_PC=");
		if ( (test = fscanf(parmFile,"%f",&ALG_PC) ) <= 0 ) return test;
		fprintf(outfile,"ALG_PC= %f\n",ALG_PC);
                numParm++;

/*Soil */
        
                scan_forward(parmFile,"DOM_RCDECOMP=");
		if ( (test = fscanf(parmFile,"%f",&DOM_RCDECOMP) ) <= 0 ) return test;
		fprintf(outfile,"DOM_RCDECOMP= %f\n",DOM_RCDECOMP);
                numParm++;

                scan_forward(parmFile,"DOM_DECOMPRED=");
		if ( (test = fscanf(parmFile,"%f",&DOM_DECOMPRED) ) <= 0 ) return test;
		fprintf(outfile,"DOM_DECOMPRED= %f\n",DOM_DECOMPRED);
                numParm++;

                scan_forward(parmFile,"DOM_DECOMP_POPT=");
		if ( (test = fscanf(parmFile,"%f",&DOM_DECOMP_POPT) ) <= 0 ) return test;
		fprintf(outfile,"DOM_DECOMP_POPT= %f\n",DOM_DECOMP_POPT);
                numParm++;

                scan_forward(parmFile,"DOM_C_OM_OPT=");
		if ( (test = fscanf(parmFile,"%f",&DOM_C_OM_OPT) ) <= 0 ) return test;
		fprintf(outfile,"DOM_C_OM_OPT= %f\n",DOM_C_OM_OPT);
                numParm++;

                scan_forward(parmFile,"DOM_DECOMP_TOPT=");
		if ( (test = fscanf(parmFile,"%f",&DOM_DECOMP_TOPT) ) <= 0 ) return test;
		fprintf(outfile,"DOM_DECOMP_TOPT= %f\n",DOM_DECOMP_TOPT);
                numParm++;

                scan_forward(parmFile,"DOM_decomp_coef=");
		if ( (test = fscanf(parmFile,"%f",&DOM_decomp_coef) ) <= 0 ) return test;
		fprintf(outfile,"DOM_decomp_coef= %f\n",DOM_decomp_coef);
                numParm++;

		scan_forward(parmFile,"sorbToTP=");
		if ( (test = fscanf(parmFile,"%f",&sorbToTP) ) <= 0 ) return test;
		fprintf(outfile,"sorbToTP= %f\n",sorbToTP);
                numParm++;

/* Macrophytes */

                scan_forward(parmFile,"MAC_IC_MULT=");
		if ( (test = fscanf(parmFile,"%f",&MAC_IC_MULT) ) <= 0 ) return test;
		fprintf(outfile,"MAC_IC_MULT= %f\n",MAC_IC_MULT);
                numParm++;

/* MAC_LITTERSEAS not used */
		scan_forward(parmFile,"MAC_LITTERSEAS=");
		if ( (test = fscanf(parmFile,"%f",&MAC_LITTERSEAS) ) <= 0 ) return test;
		fprintf(outfile,"MAC_LITTERSEAS= %f\n",MAC_LITTERSEAS);
                numParm++;

		scan_forward(parmFile,"MAC_REFUG_MULT=");
		if ( (test = fscanf(parmFile,"%f",&MAC_REFUG_MULT) ) <= 0 ) return test;
		fprintf(outfile,"MAC_REFUG_MULT= %f\n",MAC_REFUG_MULT);
                numParm++;

		scan_forward(parmFile,"mann_height_coef=");
		if ( (test = fscanf(parmFile,"%f",&mann_height_coef) ) <= 0 ) return test;
		fprintf(outfile,"mann_height_coef= %f\n",mann_height_coef);
                numParm++;

		scan_forward(parmFile,"mac_uptake_coef=");
		if ( (test = fscanf(parmFile,"%f",&mac_uptake_coef) ) <= 0 ) return test;
		fprintf(outfile,"mac_uptake_coef= %f\n",mac_uptake_coef);
                numParm++;

/* Floc */
                
		scan_forward(parmFile,"Floc_BD=");
		if ( (test = fscanf(parmFile,"%f",&Floc_BD) ) <= 0 ) return test;
		fprintf(outfile,"Floc_BD= %f\n",Floc_BD);
                numParm++;
                
		scan_forward(parmFile,"FlocMax=");
		if ( (test = fscanf(parmFile,"%f",&FlocMax) ) <= 0 ) return test;
		fprintf(outfile,"FlocMax= %f\n",FlocMax);
                numParm++;
                
		scan_forward(parmFile,"TP_P_OM=");
		if ( (test = fscanf(parmFile,"%f",&TP_P_OM) ) <= 0 ) return test;
		fprintf(outfile,"TP_P_OM= %f\n",TP_P_OM);
                numParm++;

		scan_forward(parmFile,"Floc_rcSoil=");
		if ( (test = fscanf(parmFile,"%f",&Floc_rcSoil) ) <= 0 ) return test;
		fprintf(outfile,"Floc_rcSoil= %f\n",Floc_rcSoil);
                numParm++;
                
/* Standing Detritus */
                
		scan_forward(parmFile,"STDET_IC_MULT=");
		if ( (test = fscanf(parmFile,"%f",&STDET_IC_MULT) ) <= 0 ) return test;
		fprintf(outfile,"STDET_IC_MULT= %f\n",STDET_IC_MULT);
                numParm++;

		scan_forward(parmFile,"STDET_REF_MULT=");
		if ( (test = fscanf(parmFile,"%f",&STDET_REF_MULT) ) <= 0 ) return test;
		fprintf(outfile,"STDET_REF_MULT= %f\n",STDET_REF_MULT);
                numParm++;

		scan_forward(parmFile,"StDetLos=");
		if ( (test = fscanf(parmFile,"%f",&StDetLos) ) <= 0 ) return test;
		fprintf(outfile,"StDetLos= %f\n",StDetLos);
                numParm++;
                
/* Phosphorus */

		scan_forward(parmFile,"TP_DIFFCOEF=");
		if ( (test = fscanf(parmFile,"%f",&TP_DIFFCOEF) ) <= 0 ) return test;
		fprintf(outfile,"TP_DIFFCOEF= %f\n",TP_DIFFCOEF);
                numParm++;

		scan_forward(parmFile,"TP_DIFFDEPTH=");
		if ( (test = fscanf(parmFile,"%f",&TP_DIFFDEPTH) ) <= 0 ) return test;
		fprintf(outfile,"TP_DIFFDEPTH= %f\n",TP_DIFFDEPTH);
                numParm++;

		scan_forward(parmFile,"TP_IN_RAIN=");
		if ( (test = fscanf(parmFile,"%f",&TP_IN_RAIN) ) <= 0 ) return test;
		fprintf(outfile,"TP_IN_RAIN= %f\n",TP_IN_RAIN);
                numParm++;

		scan_forward(parmFile,"TP_ICSFWAT=");
		if ( (test = fscanf(parmFile,"%f",&TP_ICSFWAT) ) <= 0 ) return test;
		fprintf(outfile,"TP_ICSFWAT= %f\n",TP_ICSFWAT);
                numParm++;

		scan_forward(parmFile,"TP_ICSEDWAT=");
		if ( (test = fscanf(parmFile,"%f",&TP_ICSEDWAT) ) <= 0 ) return test;
		fprintf(outfile,"TP_ICSEDWAT= %f\n",TP_ICSEDWAT);
                numParm++;

		scan_forward(parmFile,"TP_K_INTER=");
		if ( (test = fscanf(parmFile,"%f",&TP_K_INTER) ) <= 0 ) return test;
		fprintf(outfile,"TP_K_INTER= %f\n",TP_K_INTER);
                numParm++;

		scan_forward(parmFile,"TP_K_SLOPE=");
		if ( (test = fscanf(parmFile,"%f",&TP_K_SLOPE) ) <= 0 ) return test;
		fprintf(outfile,"TP_K_SLOPE= %f\n",TP_K_SLOPE);
                numParm++;

		scan_forward(parmFile,"PO4toTP=");
		if ( (test = fscanf(parmFile,"%f",&PO4toTP) ) <= 0 ) return test;
		fprintf(outfile,"PO4toTP= %f\n",PO4toTP);
                numParm++;

		scan_forward(parmFile,"PO4toTPint=");
		if ( (test = fscanf(parmFile,"%f",&PO4toTPint) ) <= 0 ) return test;
		fprintf(outfile,"PO4toTPint= %f\n",PO4toTPint);
                numParm++;

		scan_forward(parmFile,"TPpart_thresh=");
		if ( (test = fscanf(parmFile,"%f",&TPpart_thresh) ) <= 0 ) return test;
		fprintf(outfile,"TPpart_thresh= %f\n",TPpart_thresh);
                numParm++;

		scan_forward(parmFile,"settlVel=");
		if ( (test = fscanf(parmFile,"%f",&settlVel) ) <= 0 ) return test;
		fprintf(outfile,"settlVel= %f\n",settlVel);
                numParm++;

		scan_forward(parmFile,"WQMthresh=");
		if ( (test = fscanf(parmFile,"%f",&WQMthresh) ) <= 0 ) return test;
		fprintf(outfile,"WQMthresh= %f\n",WQMthresh);
                numParm++;

/* Nitrogen (module not run, july 02) */

                scan_forward(parmFile,"DIN_DIFFCOEF=");
		if ( (test = fscanf(parmFile,"%f",&DIN_DIFFCOEF) ) <= 0 ) return test;
		fprintf(outfile,"DIN_DIFFCOEF= %f\n",DIN_DIFFCOEF);
                numParm++;

                scan_forward(parmFile,"DIN_DIFFDEPTH=");
		if ( (test = fscanf(parmFile,"%f",&DIN_DIFFDEPTH) ) <= 0 ) return test;
		fprintf(outfile,"DIN_DIFFDEPTH= %f\n",DIN_DIFFDEPTH);
                numParm++;

		scan_forward(parmFile,"DIN_IN_RAIN=");
		if ( (test = fscanf(parmFile,"%f",&DIN_IN_RAIN) ) <= 0 ) return test;
		fprintf(outfile,"DIN_IN_RAIN= %f\n",DIN_IN_RAIN);
                numParm++;

		scan_forward(parmFile,"DIN_ICSFWAT=");
		if ( (test = fscanf(parmFile,"%f",&DIN_ICSFWAT) ) <= 0 ) return test;
		fprintf(outfile,"DIN_ICSFWAT= %f\n",DIN_ICSFWAT);
                numParm++;

		scan_forward(parmFile,"DIN_ICSEDWAT=");
		if ( (test = fscanf(parmFile,"%f",&DIN_ICSEDWAT) ) <= 0 ) return test;
		fprintf(outfile,"DIN_ICSEDWAT= %f\n",DIN_ICSEDWAT);
                numParm++;

		scan_forward(parmFile,"DIN_K_OF_NH4=");
		if ( (test = fscanf(parmFile,"%f",&DIN_K_OF_NH4) ) <= 0 ) return test;
		fprintf(outfile,"DIN_K_OF_NH4= %f\n",DIN_K_OF_NH4);
                numParm++;

		scan_forward(parmFile,"DIN_RCDENIT=");
		if ( (test = fscanf(parmFile,"%f",&DIN_RCDENIT) ) <= 0 ) return test;
		fprintf(outfile,"DIN_RCDENIT= %f\n",DIN_RCDENIT);
                numParm++;

/* Fire (module not run, july 02) */
		scan_forward(parmFile,"FIRE_HEAT_FOR_IGNIT=");
		if ( (test = fscanf(parmFile,"%f",&FIRE_HEAT_FOR_IGNIT) ) <= 0 ) return test;
		fprintf(outfile,"FIRE_HEAT_FOR_IGNIT= %f\n",FIRE_HEAT_FOR_IGNIT);
                numParm++;

		scan_forward(parmFile,"FIRE_PROP_THRESH=");
		if ( (test = fscanf(parmFile,"%f",&FIRE_PROP_THRESH) ) <= 0 ) return test;
		fprintf(outfile,"FIRE_PROP_THRESH= %f\n",FIRE_PROP_THRESH);
                numParm++;

/* Consumers (module not run, july 02) */
		scan_forward(parmFile,"CONS_IC_MULT=");
		if ( (test = fscanf(parmFile,"%f",&CONS_IC_MULT) ) <= 0 ) return test;
		fprintf(outfile,"CONS_IC_MULT= %f\n",CONS_IC_MULT);
                numParm++;

		scan_forward(parmFile,"CONS_ASSIM=");
		if ( (test = fscanf(parmFile,"%f",&CONS_ASSIM) ) <= 0 ) return test;
		fprintf(outfile,"CONS_ASSIM= %f\n",CONS_ASSIM);
                numParm++;

		scan_forward(parmFile,"CONS_C_TO_OM=");
		if ( (test = fscanf(parmFile,"%f",&CONS_C_TO_OM) ) <= 0 ) return test;
		fprintf(outfile,"CONS_C_TO_OM= %f\n",CONS_C_TO_OM);
                numParm++;

		scan_forward(parmFile,"CONS_MAX=");
		if ( (test = fscanf(parmFile,"%f",&CONS_MAX) ) <= 0 ) return test;
		fprintf(outfile,"CONS_MAX= %f\n",CONS_MAX);
                numParm++;

		scan_forward(parmFile,"CONS_NC=");
		if ( (test = fscanf(parmFile,"%f",&CONS_NC) ) <= 0 ) return test;
		fprintf(outfile,"CONS_NC= %f\n",CONS_NC);
                numParm++;

		scan_forward(parmFile,"CONS_PC=");
		if ( (test = fscanf(parmFile,"%f",&CONS_PC) ) <= 0 ) return test;
		fprintf(outfile,"CONS_PC= %f\n",CONS_PC);
                numParm++;

		scan_forward(parmFile,"CONS_RC_INGEST=");
		if ( (test = fscanf(parmFile,"%f",&CONS_RC_INGEST) ) <= 0 ) return test;
		fprintf(outfile,"CONS_RC_INGEST= %f\n",CONS_RC_INGEST);
                numParm++;

		scan_forward(parmFile,"CONS_RC_MORT=");
		if ( (test = fscanf(parmFile,"%f",&CONS_RC_MORT) ) <= 0 ) return test;
		fprintf(outfile,"CONS_RC_MORT= %f\n",CONS_RC_MORT);
                numParm++;

		scan_forward(parmFile,"CONS_RC_RESP=");
		if ( (test = fscanf(parmFile,"%f",&CONS_RC_RESP) ) <= 0 ) return test;
		fprintf(outfile,"CONS_RC_RESP= %f\n",CONS_RC_RESP);
                numParm++;

		scan_forward(parmFile,"CONS_T_OPT=");
		if ( (test = fscanf(parmFile,"%f",&CONS_T_OPT) ) <= 0 ) return test;
		fprintf(outfile,"CONS_T_OPT= %f\n",CONS_T_OPT);
                numParm++;
	}
	fprintf (stderr, "%d global parameters.\n", numParm);
        return test;
}

/*******
get the global map dimensions  
******/
void get_map_dims() {

	read_map_dims("Elevation");
}

/*******
memory allocation 
******/

void alloc_memory() {
	ON_MAP = (unsigned char *) nalloc(sizeof(unsigned char)*(s0+2)*(s1+2),"ON_MAP");
	init_pvar(ON_MAP,NULL,'c',0.0);

	BCondFlow = (int *) nalloc(sizeof(int)*(s0+2)*(s1+2),"BCondFlow");
	init_pvar(BCondFlow,NULL,'d',0.0);
	HAB = (unsigned char *) nalloc(sizeof(unsigned char)*(s0+2)*(s1+2),"HAB");
	init_pvar(HAB,NULL,'c',0.0);
        IC_MULT_MAP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"IC_MULT_MAP"); 
	init_pvar(IC_MULT_MAP,NULL,'f',0.0); 
	basn = (int *) nalloc(sizeof(int)*(s0+2)*(s1+2),"basn"); /* basin map */
        init_pvar(basn,NULL,'d',0.0);

	ALG_INCID_LIGHT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ALG_INCID_LIGHT");
	init_pvar(ALG_INCID_LIGHT,NULL,'f',0.0);
	ALG_LIGHT_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ALG_LIGHT_CF");
	init_pvar(ALG_LIGHT_CF,NULL,'f',0.0);
	ALG_LIGHT_EXTINCT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ALG_LIGHT_EXTINCT");
	init_pvar(ALG_LIGHT_EXTINCT,NULL,'f',0.0);
	ALG_WAT_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ALG_WAT_CF");
	init_pvar(ALG_WAT_CF,NULL,'f',0.0);
	ALG_TEMP_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ALG_TEMP_CF");
	init_pvar(ALG_TEMP_CF,NULL,'f',0.0);
	ALG_AVAIL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ALG_AVAIL");
	init_pvar(ALG_AVAIL,NULL,'f',0.0);
	ALG_REFUGE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ALG_REFUGE");
	init_pvar(ALG_REFUGE,NULL,'f',0.0);
	ALG_SAT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ALG_SAT");
	init_pvar(ALG_SAT,NULL,'f',0.0);
	TP_algResp = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_algResp");
	init_pvar(TP_algResp,NULL,'f',0.0);

	NC_ALG_AVAIL_MORT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_AVAIL_MORT");
	init_pvar(NC_ALG_AVAIL_MORT,NULL,'f',0.0);
	NC_ALG_CONS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_CONS");
	init_pvar(NC_ALG_CONS,NULL,'f',0.0);
	NC_ALG_GPP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_GPP");
	init_pvar(NC_ALG_GPP,NULL,'f',0.0);
	NC_ALG_MORT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_MORT");
	init_pvar(NC_ALG_MORT,NULL,'f',0.0);
	NC_ALG_MORT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_MORT_POT");
	init_pvar(NC_ALG_MORT_POT,NULL,'f',0.0);
	NC_ALG_NPP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_NPP");
	init_pvar(NC_ALG_NPP,NULL,'f',0.0);
	NC_ALG_NUT_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_NUT_CF");
	init_pvar(NC_ALG_NUT_CF,NULL,'f',0.0);
	NC_ALG_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_PR1");
	init_pvar(NC_ALG_PR1,NULL,'f',0.0);
	NC_ALG_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_PR2");
	init_pvar(NC_ALG_PR2,NULL,'f',0.0);
	NC_ALG_PROD_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_PROD_CF");
	init_pvar(NC_ALG_PROD_CF,NULL,'f',0.0);
	NC_ALG_RESP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_RESP");
	init_pvar(NC_ALG_RESP,NULL,'f',0.0);
	NC_ALG_RESP_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_RESP_POT");
	init_pvar(NC_ALG_RESP_POT,NULL,'f',0.0);
	NC_ALG = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG");
	init_pvar(NC_ALG,NULL,'f',0.0);

	C_ALG_AVAIL_MORT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_AVAIL_MORT");
	init_pvar(C_ALG_AVAIL_MORT,NULL,'f',0.0);
	C_ALG_CONS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_CONS");
	init_pvar(C_ALG_CONS,NULL,'f',0.0);
	C_ALG_GPP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_GPP");
	init_pvar(C_ALG_GPP,NULL,'f',0.0);
	C_ALG_MORT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_MORT");
	init_pvar(C_ALG_MORT,NULL,'f',0.0);
	C_ALG_MORT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_MORT_POT");
	init_pvar(C_ALG_MORT_POT,NULL,'f',0.0);
	C_ALG_NPP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_NPP");
	init_pvar(C_ALG_NPP,NULL,'f',0.0);
	C_ALG_NUT_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_NUT_CF");
	init_pvar(C_ALG_NUT_CF,NULL,'f',0.0);
	C_ALG_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_PR1");
	init_pvar(C_ALG_PR1,NULL,'f',0.0);
	C_ALG_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_PR2");
	init_pvar(C_ALG_PR2,NULL,'f',0.0);
	C_ALG_PROD_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_PROD_CF");
	init_pvar(C_ALG_PROD_CF,NULL,'f',0.0);
	C_ALG_RESP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_RESP");
	init_pvar(C_ALG_RESP,NULL,'f',0.0);
	C_ALG_RESP_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_RESP_POT");
	init_pvar(C_ALG_RESP_POT,NULL,'f',0.0);
	C_ALG = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG");
	init_pvar(C_ALG,NULL,'f',0.0);

	NC_ALG_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_P");
	init_pvar(NC_ALG_P,NULL,'f',0.0);
	C_ALG_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_P");
	init_pvar(C_ALG_P,NULL,'f',0.0);
	NC_ALG_GPP_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_GPP_P");
	init_pvar(NC_ALG_GPP_P,NULL,'f',0.0);
	C_ALG_GPP_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_GPP_P");
	init_pvar(C_ALG_GPP_P,NULL,'f',0.0);
	NC_ALG_MORT_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_MORT_P");
	init_pvar(NC_ALG_MORT_P,NULL,'f',0.0);
	C_ALG_MORT_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_ALG_MORT_P");
	init_pvar(C_ALG_MORT_P,NULL,'f',0.0);
	NC_ALG_PCrep = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_ALG_PCrep");
	init_pvar(NC_ALG_PCrep,NULL,'f',0.0);
	C_ALG_PCrep = (float *) nalloc(sizeof(double)*(s0+2)*(s1+2),"C_ALG_PCrep");
	init_pvar(C_ALG_PCrep,NULL,'f',0.0);

	NC_ALG_PC = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"NC_ALG_PC");
	init_pvar(NC_ALG_PC,NULL,'l',0.0);
	C_ALG_PC = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"C_ALG_PC");
	init_pvar(C_ALG_PC,NULL,'l',0.0);

        CONS_BIOM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_BIOM");
	init_pvar(CONS_BIOM,NULL,'f',0.0);
	CONS_EGEST = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_EGEST");
	init_pvar(CONS_EGEST,NULL,'f',0.0);
	CONS_INGEST = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_INGEST");
	init_pvar(CONS_INGEST,NULL,'f',0.0);
	CONS_INGEST_ALGAE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_INGEST_ALGAE");
	init_pvar(CONS_INGEST_ALGAE,NULL,'f',0.0);
	CONS_INGEST_DEPOM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_INGEST_DEPOM");
	init_pvar(CONS_INGEST_DEPOM,NULL,'f',0.0);
	CONS_INGEST_NPHBIO = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_INGEST_NPHBIO");
	init_pvar(CONS_INGEST_NPHBIO,NULL,'f',0.0);
	CONS_INGEST_PHBIO = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_INGEST_PHBIO");
	init_pvar(CONS_INGEST_PHBIO,NULL,'f',0.0);
	CONS_INGEST_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_INGEST_POT");
	init_pvar(CONS_INGEST_POT,NULL,'f',0.0);
	CONS_INGEST_FLOC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_INGEST_FLOC");
	init_pvar(CONS_INGEST_FLOC,NULL,'f',0.0);
	CONS_INGEST_STDETR = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_INGEST_STDETR");
	init_pvar(CONS_INGEST_STDETR,NULL,'f',0.0);
	CONS_MORT_BIOM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_MORT_BIOM");
	init_pvar(CONS_MORT_BIOM,NULL,'f',0.0);
	CONS_RESPIRATION = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_RESPIRATION");
	init_pvar(CONS_RESPIRATION,NULL,'f',0.0);
	CONS_SPEC_ING = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CONS_SPEC_ING");
	init_pvar(CONS_SPEC_ING,NULL,'f',0.0);

	DIN_SED_WT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"DIN_SED_WT");
	init_pvar(DIN_SED_WT,NULL,'l',0.0);
	DIN_SF_WT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"DIN_SF_WT");
	init_pvar(DIN_SF_WT,NULL,'l',0.0);

	DIN_DNFLOW = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_DNFLOW");
	init_pvar(DIN_DNFLOW,NULL,'f',0.0);
	DIN_DNFLOW_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_DNFLOW_POT");
	init_pvar(DIN_DNFLOW_POT,NULL,'f',0.0);
	DIN_FR_RAIN = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_FR_RAIN");
	init_pvar(DIN_FR_RAIN,NULL,'f',0.0);
	DIN_SED_CONC_INT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SED_CONC_INT");
	init_pvar(DIN_SED_CONC_INT,NULL,'f',0.0);
	DIN_SEDWT_CONCACT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_CONCACT");
	init_pvar(DIN_SEDWT_CONCACT,NULL,'f',0.0);
	DIN_SEDWT_CONCTOT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_CONCTOT");
	init_pvar(DIN_SEDWT_CONCTOT,NULL,'f',0.0);
	DIN_SEDWT_DENIT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_DENIT");
	init_pvar(DIN_SEDWT_DENIT,NULL,'f',0.0);
	DIN_SEDWT_DENIT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_DENIT_POT");
	init_pvar(DIN_SEDWT_DENIT_POT,NULL,'f',0.0);
	DIN_SEDWT_MINER = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_MINER");
	init_pvar(DIN_SEDWT_MINER,NULL,'f',0.0);
	DIN_SEDWT_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_PR1");
	init_pvar(DIN_SEDWT_PR1,NULL,'f',0.0);
	DIN_SEDWT_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_PR2");
	init_pvar(DIN_SEDWT_PR2,NULL,'f',0.0);
	DIN_SEDWT_UPTAKE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_UPTAKE");
	init_pvar(DIN_SEDWT_UPTAKE,NULL,'f',0.0);
	DIN_SEDWT_UPTAKE_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SEDWT_UPTAKE_POT");
	init_pvar(DIN_SEDWT_UPTAKE_POT,NULL,'f',0.0);
	DIN_SFWT_CONC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SFWT_CONC");
	init_pvar(DIN_SFWT_CONC,NULL,'f',0.0);
	DIN_SFWT_MINER = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SFWT_MINER");
	init_pvar(DIN_SFWT_MINER,NULL,'f',0.0);
	DIN_SFWT_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SFWT_PR1");
	init_pvar(DIN_SFWT_PR1,NULL,'f',0.0);
	DIN_SFWT_UPT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SFWT_UPT_POT");
	init_pvar(DIN_SFWT_UPT_POT,NULL,'f',0.0);
	DIN_SFWT_UPTAKE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_SFWT_UPTAKE");
	init_pvar(DIN_SFWT_UPTAKE,NULL,'f',0.0);
	DIN_UPFLOW = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_UPFLOW");
	init_pvar(DIN_UPFLOW,NULL,'f',0.0);
	DIN_UPFLOW_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIN_UPFLOW_POT");
	init_pvar(DIN_UPFLOW_POT,NULL,'f',0.0);

	DEPOS_ORG_MAT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"DEPOS_ORG_MAT");
	init_pvar(DEPOS_ORG_MAT,NULL,'l',0.0);
        
	DOM_Z = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_Z");
	init_pvar(DOM_Z,NULL,'f',0.0);
	DOM_C = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_C");
	init_pvar(DOM_C,NULL,'f',0.0);
	DOM_CONS_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_CONS_POT");
	init_pvar(DOM_CONS_POT,NULL,'f',0.0);
	DOM_CONSUMED = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_CONSUMED");
	init_pvar(DOM_CONSUMED,NULL,'f',0.0);
	DOM_DECOMP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_DECOMP");
	init_pvar(DOM_DECOMP,NULL,'f',0.0);
	DOM_DECOMP_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_DECOMP_POT");
	init_pvar(DOM_DECOMP_POT,NULL,'f',0.0);
	DOM_FIRE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_FIRE");
	init_pvar(DOM_FIRE,NULL,'f',0.0);
	DOM_FR_CONS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_FR_CONS");
	init_pvar(DOM_FR_CONS,NULL,'f',0.0);
	DOM_FR_MAC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_FR_MAC");
	init_pvar(DOM_FR_MAC,NULL,'f',0.0);
	DOM_fr_nphBio = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_fr_nphBio");
	init_pvar(DOM_fr_nphBio,NULL,'f',0.0);
	Floc_fr_phBio = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Floc_fr_phBio");
	init_pvar(Floc_fr_phBio,NULL,'f',0.0);
	DOM_FR_FLOC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_FR_FLOC");
	init_pvar(DOM_FR_FLOC,NULL,'f',0.0);
	DOM_FR_STDET = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_FR_STDET");
	init_pvar(DOM_FR_STDET,NULL,'f',0.0);
	DOM_MOIST_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_MOIST_CF");
	init_pvar(DOM_MOIST_CF,NULL,'f',0.0);
	DOM_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_PR1");
	init_pvar(DOM_PR1,NULL,'f',0.0);
	DOM_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_PR2");
	init_pvar(DOM_PR2,NULL,'f',0.0);
	DOM_QUALITY_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_QUALITY_CF");
	init_pvar(DOM_QUALITY_CF,NULL,'f',0.0);
	DOM_SED_AEROB_Z = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_SED_AEROB_Z");
	init_pvar(DOM_SED_AEROB_Z,NULL,'f',0.0);
	DOM_SED_ANAEROB_Z = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_SED_ANAEROB_Z");
	init_pvar(DOM_SED_ANAEROB_Z,NULL,'f',0.0);
	DOM_TEMP_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_TEMP_CF");
	init_pvar(DOM_TEMP_CF,NULL,'f',0.0);
	OM_ABV_SED = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"OM_ABV_SED");
	init_pvar(OM_ABV_SED,NULL,'f',0.0);
	OM_TOT_C_AVAIL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"OM_TOT_C_AVAIL");
	init_pvar(OM_TOT_C_AVAIL,NULL,'f',0.0);
	ELEVATION = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ELEVATION");
	init_pvar(ELEVATION,NULL,'f',0.0);
	SED_ELEV = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SED_ELEV");
	init_pvar(SED_ELEV,NULL,'f',0.0);
	SED_ELEV_IC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SED_ELEV_IC");
	init_pvar(SED_ELEV_IC,NULL,'f',0.0);
	SED_INACT_Z = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SED_INACT_Z");
	init_pvar(SED_INACT_Z,NULL,'f',0.0);
	DOP_FLOC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOP_FLOC");
	init_pvar(DOP_FLOC,NULL,'f',0.0);
	DOP_nphBio = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOP_nphBio");
	init_pvar(DOP_nphBio,NULL,'f',0.0);
	DOP_STDET = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOP_STDET");
	init_pvar(DOP_STDET,NULL,'f',0.0);

        DOM_P_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"DOM_P_OM");
	init_pvar(DOM_P_OM,NULL,'l',0.0);

        TPtoSOIL = (float *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TPtoSOIL");
	init_pvar(TPtoSOIL,NULL,'l',0.0);
        TPtoVOL = (float *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TPtoVOL");
	init_pvar(TPtoVOL,NULL,'l',0.0);
        BulkD = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"BulkD");
	init_pvar(BulkD,NULL,'f',0.0);
	DOM_BD = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOM_BD");
	init_pvar(DOM_BD,NULL,'f',0.0);
	Inorg_Z = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Inorg_Z");
	init_pvar(Inorg_Z,NULL,'f',0.0);
	DIM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DIM");
	init_pvar(DIM,NULL,'f',0.0);
	DOP_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOP_CF");
	init_pvar(DOP_CF,NULL,'f',0.0);
	DOP_DECOMP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DOP_DECOMP");
	init_pvar(DOP_DECOMP,NULL,'f',0.0);

	DOP = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"DOP");
	init_pvar(DOP,NULL,'l',0.0);

	FIRE_DIRECT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_DIRECT");
	init_pvar(FIRE_DIRECT,NULL,'f',0.0);
	FIRE_DIST_DUMP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_DIST_DUMP");
	init_pvar(FIRE_DIST_DUMP,NULL,'f',0.0);
	FIRE_DIST_IN = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_DIST_IN");
	init_pvar(FIRE_DIST_IN,NULL,'f',0.0);
	FIRE_FUEL_BULK_DENS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_FUEL_BULK_DENS");
	init_pvar(FIRE_FUEL_BULK_DENS,NULL,'f',0.0);
	FIRE_FUEL_LOADING = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_FUEL_LOADING");
	init_pvar(FIRE_FUEL_LOADING,NULL,'f',0.0);
	FIRE_INIT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_INIT");
	init_pvar(FIRE_INIT,NULL,'f',0.0);
	FIRE_LIGHT_HIT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_LIGHT_HIT");
	init_pvar(FIRE_LIGHT_HIT,NULL,'f',0.0);
	FIRE_LOSS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_LOSS");
	init_pvar(FIRE_LOSS,NULL,'f',0.0);
	FIRE_MOIST_EFFECT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_MOIST_EFFECT");
	init_pvar(FIRE_MOIST_EFFECT,NULL,'f',0.0);
	FIRE_SPECIFIC_RATE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_SPECIFIC_RATE");
	init_pvar(FIRE_SPECIFIC_RATE,NULL,'f',0.0);
	FIRE_SPREAD_RATE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_SPREAD_RATE");
	init_pvar(FIRE_SPREAD_RATE,NULL,'f',0.0);
	FIRE_TOT_DIST = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_TOT_DIST");
	init_pvar(FIRE_TOT_DIST,NULL,'f',0.0);
        
	HYD_CANOPY_TRANSP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_CANOPY_TRANSP");
	init_pvar(HYD_CANOPY_TRANSP,NULL,'f',0.0);
	HYD_DOM_ACTWAT_PRES = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_DOM_ACTWAT_PRES");
	init_pvar(HYD_DOM_ACTWAT_PRES,NULL,'f',0.0);
	HYD_DOM_ACTWAT_VOL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_DOM_ACTWAT_VOL");
	init_pvar(HYD_DOM_ACTWAT_VOL,NULL,'f',0.0);
	HYD_EVAP_CALC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_EVAP_CALC");
	init_pvar(HYD_EVAP_CALC,NULL,'f',0.0);
	HYD_MANNINGS_N = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_MANNINGS_N");
	init_pvar(HYD_MANNINGS_N,NULL,'f',0.0);
	HYD_MAX_SED_WAT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_MAX_SED_WAT");
	init_pvar(HYD_MAX_SED_WAT,NULL,'f',0.0);
	HYD_PRECIP_DAY = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_PRECIP_DAY");
	init_pvar(HYD_PRECIP_DAY,NULL,'f',0.0);
	HYD_RCCONDUCT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_RCCONDUCT");
	init_pvar(HYD_RCCONDUCT,NULL,'f',0.0);
	HYD_SAT_POT_TRANS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_SAT_POT_TRANS");
	init_pvar(HYD_SAT_POT_TRANS,NULL,'f',0.0);
	HYD_SED_WAT_VOL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_SED_WAT_VOL");
	init_pvar(HYD_SED_WAT_VOL,NULL,'f',0.0);
	HYD_TOT_POT_TRANSP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_TOT_POT_TRANSP");
	init_pvar(HYD_TOT_POT_TRANSP,NULL,'f',0.0);
	HydTotHd = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HydTotHd");
	init_pvar(HydTotHd,NULL,'f',0.0);
	HYD_TRANSP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_TRANSP");
	init_pvar(HYD_TRANSP,NULL,'f',0.0);
	HYD_UNSAT_POT_TRANS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_UNSAT_POT_TRANS");
	init_pvar(HYD_UNSAT_POT_TRANS,NULL,'f',0.0);
	HYD_WATER_AVAIL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HYD_WATER_AVAIL");
	init_pvar(HYD_WATER_AVAIL,NULL,'f',0.0);
/* sfwmm rainfall mapped to elm grid */
	wmm_rain = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"wmm_rain");
	init_pvar(wmm_rain,NULL,'f',0.0);
	SAT_ADD_FM_RISE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_ADD_FM_RISE");
	init_pvar(SAT_ADD_FM_RISE,NULL,'f',0.0);
	SAT_SURF_UP_FL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_SURF_UP_FL");
	init_pvar(SAT_SURF_UP_FL,NULL,'f',0.0);
	SAT_TO_UNSAT_FL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_TO_UNSAT_FL");
	init_pvar(SAT_TO_UNSAT_FL,NULL,'f',0.0);
	SAT_TO_UNSAT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_TO_UNSAT_POT");
	init_pvar(SAT_TO_UNSAT_POT,NULL,'f',0.0);
	SAT_VS_UNSAT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_VS_UNSAT");
	init_pvar(SAT_VS_UNSAT,NULL,'f',0.0);
	SAT_WATER = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_WATER");
	init_pvar(SAT_WATER,NULL,'f',0.0);
	SAT_WT_HEAD = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_WT_HEAD");
	init_pvar(SAT_WT_HEAD,NULL,'f',0.0);
	SAT_WT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_WT_POT");
	init_pvar(SAT_WT_POT,NULL,'f',0.0);
	SAT_WT_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_WT_PR1");
	init_pvar(SAT_WT_PR1,NULL,'f',0.0);
	SAT_WT_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_WT_PR2");
	init_pvar(SAT_WT_PR2,NULL,'f',0.0);
	SAT_WT_PR3 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_WT_PR3");
	init_pvar(SAT_WT_PR3,NULL,'f',0.0);
	SAT_WT_RECHG = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_WT_RECHG");
	init_pvar(SAT_WT_RECHG,NULL,'f',0.0);
	SAT_WT_TRANSP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAT_WT_TRANSP");
	init_pvar(SAT_WT_TRANSP,NULL,'f',0.0);
	SF_WT_EVAP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SF_WT_EVAP");
	init_pvar(SF_WT_EVAP,NULL,'f',0.0);
	SF_WT_FROM_RAIN = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SF_WT_FROM_RAIN");
	init_pvar(SF_WT_FROM_RAIN,NULL,'f',0.0);
	SF_WT_HEAD = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SF_WT_HEAD");
	init_pvar(SF_WT_HEAD,NULL,'f',0.0);
	SF_WT_INFILTRATION = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SF_WT_INFILTRATION");
	init_pvar(SF_WT_INFILTRATION,NULL,'f',0.0);
	SF_WT_POT_INF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SF_WT_POT_INF");
	init_pvar(SF_WT_POT_INF,NULL,'f',0.0);
	SF_WT_TO_SAT_DOWNFLOW = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SF_WT_TO_SAT_DOWNFLOW");
	init_pvar(SF_WT_TO_SAT_DOWNFLOW,NULL,'f',0.0);
	SFWAT_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SFWAT_PR1");
	init_pvar(SFWAT_PR1,NULL,'f',0.0);
	SFWT_VOL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SFWT_VOL");
	init_pvar(SFWT_VOL,NULL,'f',0.0);
	SURFACE_WAT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SURFACE_WAT");
	init_pvar(SURFACE_WAT,NULL,'f',0.0);
	UNSAT_AVAIL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_AVAIL");
	init_pvar(UNSAT_AVAIL,NULL,'f',0.0);
	UNSAT_CAP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_CAP");
	init_pvar(UNSAT_CAP,NULL,'f',0.0);
	delaymap01 = (float*)nalloc(sizeof(float)*(s0+2)*(s1+2),"delay map 1");
        UNSAT_CAP_CHANGE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_CAP_CHANGE");
	init_pvar(UNSAT_CAP_CHANGE,NULL,'f',0.0);
	UNSAT_DEPTH = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_DEPTH");
	init_pvar(UNSAT_DEPTH,NULL,'f',0.0);
	UNSAT_HYD_COND_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_HYD_COND_CF");
	init_pvar(UNSAT_HYD_COND_CF,NULL,'f',0.0);
	UNSAT_MOIST_PRP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_MOIST_PRP");
	init_pvar(UNSAT_MOIST_PRP,NULL,'f',0.0);
	UNSAT_PERC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_PERC");
	init_pvar(UNSAT_PERC,NULL,'f',0.0);
	UNSAT_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_PR1");
	init_pvar(UNSAT_PR1,NULL,'f',0.0);
	UNSAT_TO_SAT_FL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_TO_SAT_FL");
	init_pvar(UNSAT_TO_SAT_FL,NULL,'f',0.0);
	UNSAT_TRANSP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_TRANSP");
	init_pvar(UNSAT_TRANSP,NULL,'f',0.0);
	UNSAT_WATER = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_WATER");
	init_pvar(UNSAT_WATER,NULL,'f',0.0);
	UNSAT_WT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UNSAT_WT_POT");
	init_pvar(UNSAT_WT_POT,NULL,'f',0.0);

	MAC_HEIGHT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_HEIGHT");
	init_pvar(MAC_HEIGHT,NULL,'f',0.0);
	MAC_LAI = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_LAI");
	init_pvar(MAC_LAI,NULL,'f',0.0);
	LAI_eff = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"LAI_eff");
	init_pvar(LAI_eff,NULL,'f',0.0);
	MAC_LIGHT_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_LIGHT_CF");
	init_pvar(MAC_LIGHT_CF,NULL,'f',0.0);
	MAC_MAX_BIO = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_MAX_BIO");
	init_pvar(MAC_MAX_BIO,NULL,'f',0.0);
        MAC_NOPH_BIOMAS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_NOPH_BIOMAS");
	init_pvar(MAC_NOPH_BIOMAS,NULL,'f',0.0);
	MAC_NUT_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_NUT_CF");
	init_pvar(MAC_NUT_CF,NULL,'f',0.0);
	MAC_PH_BIOMAS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_PH_BIOMAS");
	init_pvar(MAC_PH_BIOMAS,NULL,'f',0.0);
	MAC_PROD_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_PROD_CF");
	init_pvar(MAC_PROD_CF,NULL,'f',0.0);
	MAC_REL_BIOM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_REL_BIOM");
	init_pvar(MAC_REL_BIOM,NULL,'f',0.0);
	MAC_SALT_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_SALT_CF");
	init_pvar(MAC_SALT_CF,NULL,'f',0.0);
	MAC_TEMP_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_TEMP_CF");
	init_pvar(MAC_TEMP_CF,NULL,'f',0.0);
	MAC_TOT_BIOM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_TOT_BIOM");
	init_pvar(MAC_TOT_BIOM,NULL,'f',0.0);
	MAC_WATER_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_WATER_CF");
	init_pvar(MAC_WATER_CF,NULL,'f',0.0);
	MAC_WATER_AVAIL_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MAC_WATER_AVAIL_CF");
	init_pvar(MAC_WATER_AVAIL_CF,NULL,'f',0.0);
	NPHBIO_AVAIL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_AVAIL");
	init_pvar(NPHBIO_AVAIL,NULL,'f',0.0);
	NPHBIO_CONSUMED = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_CONSUMED");
	init_pvar(NPHBIO_CONSUMED,NULL,'f',0.0);
	NPHBIO_FIRE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_FIRE");
	init_pvar(NPHBIO_FIRE,NULL,'f',0.0);
	NPHBIO_MORT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_MORT");
	init_pvar(NPHBIO_MORT,NULL,'f',0.0);
	NPHBIO_MORT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_MORT_POT");
	init_pvar(NPHBIO_MORT_POT,NULL,'f',0.0);
	NPHBIO_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_PR1");
	init_pvar(NPHBIO_PR1,NULL,'f',0.0);
	NPHBIO_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_PR2");
	init_pvar(NPHBIO_PR2,NULL,'f',0.0);
	NPHBIO_PR3 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_PR3");
	init_pvar(NPHBIO_PR3,NULL,'f',0.0);
	NPHBIO_REFUGE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_REFUGE");
	init_pvar(NPHBIO_REFUGE,NULL,'f',0.0);
	NPHBIO_SAT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_SAT");
	init_pvar(NPHBIO_SAT,NULL,'f',0.0);
	NPHBIO_TRANSLOC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_TRANSLOC");
	init_pvar(NPHBIO_TRANSLOC,NULL,'f',0.0);
	NPHBIO_TRANSLOC_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NPHBIO_TRANSLOC_POT");
	init_pvar(NPHBIO_TRANSLOC_POT,NULL,'f',0.0);
	PHBIO_AVAIL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_AVAIL");
	init_pvar(PHBIO_AVAIL,NULL,'f',0.0);
	PHBIO_CONSUMED = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_CONSUMED");
	init_pvar(PHBIO_CONSUMED,NULL,'f',0.0);
	PHBIO_FIRE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_FIRE");
	init_pvar(PHBIO_FIRE,NULL,'f',0.0);
	PHBIO_GROW_MAX = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_GROW_MAX");
	init_pvar(PHBIO_GROW_MAX,NULL,'f',0.0);
	PHBIO_MORT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_MORT");
	init_pvar(PHBIO_MORT,NULL,'f',0.0);
	PHBIO_MORT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_MORT_POT");
	init_pvar(PHBIO_MORT_POT,NULL,'f',0.0);
	PHBIO_NPP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_NPP");
	init_pvar(PHBIO_NPP,NULL,'f',0.0);
	PHBIO_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_PR1");
	init_pvar(PHBIO_PR1,NULL,'f',0.0);
	PHBIO_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_PR2");
	init_pvar(PHBIO_PR2,NULL,'f',0.0);
	PHBIO_PR3 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_PR3");
	init_pvar(PHBIO_PR3,NULL,'f',0.0);
	PHBIO_REFUGE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_REFUGE");
	init_pvar(PHBIO_REFUGE,NULL,'f',0.0);
	PHBIO_SAT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_SAT");
	init_pvar(PHBIO_SAT,NULL,'f',0.0); 
	PHBIO_TRANSLOC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_TRANSLOC");
	init_pvar(PHBIO_TRANSLOC,NULL,'f',0.0);
	PHBIO_TRANSLOC_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PHBIO_TRANSLOC_POT");
	init_pvar(PHBIO_TRANSLOC_POT,NULL,'f',0.0);

	phbio_npp_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"phbio_npp_P");
	init_pvar(phbio_npp_P,NULL,'l',0.0);
	phbio_npp_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"phbio_npp_OM");
	init_pvar(phbio_npp_OM,NULL,'l',0.0);
	phbio_fire_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"phbio_fire_P");
	init_pvar(phbio_fire_P,NULL,'l',0.0);
	phbio_fire_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"phbio_fire_OM");
	init_pvar(phbio_fire_OM,NULL,'l',0.0);
	phbio_mort_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"phbio_mort_P");
	init_pvar(phbio_mort_P,NULL,'l',0.0);
	phbio_mort_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"phbio_mort_OM");
	init_pvar(phbio_mort_OM,NULL,'l',0.0);
	phbio_transl_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"phbio_transl_P");
	init_pvar(phbio_transl_P,NULL,'l',0.0);
	phbio_transl_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"phbio_transl_OM");
	init_pvar(phbio_transl_OM,NULL,'l',0.0);
	nphbio_transl_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"nphbio_transl_P");
	init_pvar(nphbio_transl_P,NULL,'l',0.0);
	nphbio_transl_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"nphbio_transl_OM");
	init_pvar(nphbio_transl_OM,NULL,'l',0.0);
	nphbio_fire_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"nphbio_fire_P");
	init_pvar(nphbio_fire_P,NULL,'l',0.0);
	nphbio_fire_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"nphbio_fire_OM");
	init_pvar(nphbio_fire_OM,NULL,'l',0.0);
	nphbio_mort_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"nphbio_mort_P");
	init_pvar(nphbio_mort_P,NULL,'l',0.0);
	nphbio_mort_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"nphbio_mort_OM");
	init_pvar(nphbio_mort_OM,NULL,'l',0.0);
	mac_nph_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_nph_P");
	init_pvar(mac_nph_P,NULL,'l',0.0);
	mac_nph_PC = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_nph_PC");
	init_pvar(mac_nph_PC,NULL,'l',0.0);
	mac_nph_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_nph_OM");
	init_pvar(mac_nph_OM,NULL,'l',0.0);
	mac_nph_CtoOM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_nph_CtoOM");
	init_pvar(mac_nph_CtoOM,NULL,'l',0.0);
	mac_ph_P = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_ph_P");
	init_pvar(mac_ph_P,NULL,'l',0.0);
	mac_ph_PC = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_ph_PC");
	init_pvar(mac_ph_PC,NULL,'l',0.0);
	mac_ph_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_ph_OM");
	init_pvar(mac_ph_OM,NULL,'l',0.0);
	mac_ph_CtoOM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_ph_CtoOM");
	init_pvar(mac_ph_CtoOM,NULL,'l',0.0);

	mac_nph_PC_rep = (float *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_nph_PC_rep");
	init_pvar(mac_nph_PC_rep,NULL,'l',0.0);
	mac_ph_PC_rep = (float *) nalloc(sizeof(double)*(s0+2)*(s1+2),"mac_ph_PC_rep");
	init_pvar(mac_ph_PC_rep,NULL,'l',0.0);

	TP_SED_WT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_SED_WT");
	init_pvar(TP_SED_WT,NULL,'l',0.0);
	TP_SED_CONC = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_SED_CONC");
	init_pvar(TP_SED_CONC,NULL,'l',0.0);
	TP_SEDWT_CONCACT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_SEDWT_CONCACT");
	init_pvar(TP_SEDWT_CONCACT,NULL,'l',0.0);
	TP_SF_WT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_SF_WT");
	init_pvar(TP_SF_WT,NULL,'l',0.0);
	TP_SFWT_CONC = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_SFWT_CONC");
	init_pvar(TP_SFWT_CONC,NULL,'l',0.0);
	TP_SORB = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_SORB");
	init_pvar(TP_SORB,NULL,'l',0.0);
	TP_SORBCONC = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_SORBCONC");
	init_pvar(TP_SORBCONC,NULL,'l',0.0);
	TP_SED_WT_AZ = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_SED_WT_AZ");
	init_pvar(TP_SED_WT_AZ,NULL,'l',0.0);
	TP_Act_to_Tot = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"TP_Act_to_Tot");
	init_pvar(TP_Act_to_Tot,NULL,'l',0.0);
	TP_Act_to_TotRep = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_Act_to_TotRep");
	init_pvar(TP_Act_to_TotRep,NULL,'f',0.0);

	TP_DNFLOW = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_DNFLOW");
	init_pvar(TP_DNFLOW,NULL,'f',0.0);
	TP_DNFLOW_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_DNFLOW_POT");
	init_pvar(TP_DNFLOW_POT,NULL,'f',0.0);
	TP_FR_RAIN = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_FR_RAIN");
	init_pvar(TP_FR_RAIN,NULL,'f',0.0);
	TP_K = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_K");
	init_pvar(TP_K,NULL,'f',0.0);
	TP_SED_MINER = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SED_MINER");
	init_pvar(TP_SED_MINER,NULL,'f',0.0);
	TP_SEDWT_CONCACTMG = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SEDWT_CONCACTMG");
	init_pvar(TP_SEDWT_CONCACTMG,NULL,'f',0.0);
	TP_SEDWT_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SEDWT_PR1");
	init_pvar(TP_SEDWT_PR1,NULL,'f',0.0);
	TP_SEDWT_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SEDWT_PR2");
	init_pvar(TP_SEDWT_PR2,NULL,'f',0.0);
	TP_SEDWT_UPT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SEDWT_UPT_POT");
	init_pvar(TP_SEDWT_UPT_POT,NULL,'f',0.0);
	TP_SEDWT_UPTAKE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SEDWT_UPTAKE");
	init_pvar(TP_SEDWT_UPTAKE,NULL,'f',0.0);
	TP_SFWT_CONC_MG = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SFWT_CONC_MG");
	init_pvar(TP_SFWT_CONC_MG,NULL,'f',0.0);
	TP_SFWT_MINER = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SFWT_MINER");
	init_pvar(TP_SFWT_MINER,NULL,'f',0.0);
	TP_SFWT_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SFWT_PR1");
	init_pvar(TP_SFWT_PR1,NULL,'f',0.0);
	TP_SFWT_UPT_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SFWT_UPT_POT");
	init_pvar(TP_SFWT_UPT_POT,NULL,'f',0.0);
	TP_SFWT_UPTAK = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SFWT_UPTAK");
	init_pvar(TP_SFWT_UPTAK,NULL,'f',0.0);
	TP_settl = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_settl");
	init_pvar(TP_settl,NULL,'f',0.0);
	TP_SORB_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SORB_POT");
	init_pvar(TP_SORB_POT,NULL,'f',0.0);
	TP_SORBTION = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_SORBTION");
	init_pvar(TP_SORBTION,NULL,'f',0.0);
	TP_UPFLOW = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_UPFLOW");
	init_pvar(TP_UPFLOW,NULL,'f',0.0);
	TP_UPFLOW_MG = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_UPFLOW_MG");
	init_pvar(TP_UPFLOW_MG,NULL,'f',0.0);
	TP_UPFLOW_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_UPFLOW_POT");
	init_pvar(TP_UPFLOW_POT,NULL,'f',0.0);

	WQMsettlVel = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"WQMsettlVel");
	init_pvar(WQMsettlVel,NULL,'f',0.0);
	TP_settlDays = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_settlDays");
	init_pvar(TP_settlDays,NULL,'f',0.0);

	SAL_SED_WT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"SAL_SED_WT");
	init_pvar(SAL_SED_WT,NULL,'l',0.0);
	SAL_SF_WT_mb = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"SAL_SF_WT_mb");
	init_pvar(SAL_SF_WT_mb,NULL,'l',0.0);
	SALT_SED_WT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"SALT_SED_WT");
	init_pvar(SALT_SED_WT,NULL,'l',0.0);
	SALT_SURF_WT = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"SALT_SURF_WT");
	init_pvar(SALT_SURF_WT,NULL,'l',0.0);
        
	SAL_MEAN = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAL_MEAN");
	init_pvar(SAL_MEAN,NULL,'f',0.0);
	SAL_SF_WT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SAL_SF_WT");
	init_pvar(SAL_SF_WT,NULL,'f',0.0);
	SALT_SED_TO_SF_FLOW = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SALT_SED_TO_SF_FLOW");
	init_pvar(SALT_SED_TO_SF_FLOW,NULL,'f',0.0);
	SALT_SFWAT_DOWNFL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SALT_SFWAT_DOWNFL");
	init_pvar(SALT_SFWAT_DOWNFL,NULL,'f',0.0);
	SALT_SFWAT_DOWNFL_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SALT_SFWAT_DOWNFL_POT");
	init_pvar(SALT_SFWAT_DOWNFL_POT,NULL,'f',0.0);

	FLOC_C_AVAIL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_C_AVAIL");
	init_pvar(FLOC_C_AVAIL,NULL,'f',0.0);
	FLOC_CONS_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_CONS_POT");
	init_pvar(FLOC_CONS_POT,NULL,'f',0.0);
	FLOC_CONSUMED = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_CONSUMED");
	init_pvar(FLOC_CONSUMED,NULL,'f',0.0);
	FLOC_DECOMP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_DECOMP");
	init_pvar(FLOC_DECOMP,NULL,'f',0.0);
	FLOC_DECOMP_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_DECOMP_POT");
	init_pvar(FLOC_DECOMP_POT,NULL,'f',0.0);
	FLOC_DECOMP_QUAL_CF = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_DECOMP_QUAL_CF");
	init_pvar(FLOC_DECOMP_QUAL_CF,NULL,'f',0.0);
	FLOC_Z = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_Z");
	init_pvar(FLOC_Z,NULL,'f',0.0);
	FLOC_DEPO = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_DEPO");
	init_pvar(FLOC_DEPO,NULL,'f',0.0);
	FLOC_DEPO_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_DEPO_POT");
	init_pvar(FLOC_DEPO_POT,NULL,'f',0.0);
	FLOC_FR_ALGAE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_FR_ALGAE");
	init_pvar(FLOC_FR_ALGAE,NULL,'f',0.0);
	FLOC_FR_CONSUM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_FR_CONSUM");
	init_pvar(FLOC_FR_CONSUM,NULL,'f',0.0);
	FLOC_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_PR1");
	init_pvar(FLOC_PR1,NULL,'f',0.0);
	FLOC_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_PR2");
	init_pvar(FLOC_PR2,NULL,'f',0.0);
	FLOC_PR3 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC_PR3");
	init_pvar(FLOC_PR3,NULL,'f',0.0);
	FLOC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FLOC");
	init_pvar(FLOC,NULL,'f',0.0);
	FlocP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FlocP");
	init_pvar(FlocP,NULL,'f',0.0);
	FlocP_DECOMP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FlocP_DECOMP");
	init_pvar(FlocP_DECOMP,NULL,'f',0.0);
	FlocP_FR_ALGAE = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FlocP_FR_ALGAE");
	init_pvar(FlocP_FR_ALGAE,NULL,'f',0.0);
	FlocP_PhBio = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FlocP_PhBio");
	init_pvar(FlocP_PhBio,NULL,'f',0.0);
	FlocP_DEPO = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FlocP_DEPO");
	init_pvar(FlocP_DEPO,NULL,'f',0.0);
	FlocP_OMrep = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FlocP_OMrep");
	init_pvar(FlocP_OMrep,NULL,'f',0.0);
	FlocP_OM = (double *) nalloc(sizeof(double)*(s0+2)*(s1+2),"FlocP_OM");
	init_pvar(FlocP_OM,NULL,'l',0.0);

	STAND_DETRITUS = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STAND_DETRITUS");
	init_pvar(STAND_DETRITUS,NULL,'f',0.0);
	STDET_AVAIL = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_AVAIL");
	init_pvar(STDET_AVAIL,NULL,'f',0.0);
	STDET_BURN = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_BURN");
	init_pvar(STDET_BURN,NULL,'f',0.0);
	STDET_BURN_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_BURN_POT");
	init_pvar(STDET_BURN_POT,NULL,'f',0.0);
	STDET_CONSUMED = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_CONSUMED");
	init_pvar(STDET_CONSUMED,NULL,'f',0.0);
	STDET_PR1 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_PR1");
	init_pvar(STDET_PR1,NULL,'f',0.0);
	STDET_PR2 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_PR2");
	init_pvar(STDET_PR2,NULL,'f',0.0);
	STDET_PROD = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_PROD");
	init_pvar(STDET_PROD,NULL,'f',0.0);
	STDET_REFUG = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_REFUG");
	init_pvar(STDET_REFUG,NULL,'f',0.0);
	STDET_SAT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_SAT");
	init_pvar(STDET_SAT,NULL,'f',0.0);
	STDET_TO_DOM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_TO_DOM");
	init_pvar(STDET_TO_DOM,NULL,'f',0.0);
	STDET_TO_DOM_POT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"STDET_TO_DOM_POT");
	init_pvar(STDET_TO_DOM_POT,NULL,'f',0.0);

	stdet_prod_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_prod_P");
	init_pvar(stdet_prod_P,NULL,'f',0.0);
	stdet_prod_OM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_prod_OM");
	init_pvar(stdet_prod_OM,NULL,'f',0.0);
	stdet_burn_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_burn_P");
	init_pvar(stdet_burn_P,NULL,'f',0.0);
	stdet_burn_OM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_burn_OM");
	init_pvar(stdet_burn_OM,NULL,'f',0.0);
	stdet_toDOM_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_toDOM_P");
	init_pvar(stdet_toDOM_P,NULL,'f',0.0);
	stdet_toDOM_OM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_toDOM_OM");
	init_pvar(stdet_toDOM_OM,NULL,'f',0.0);
	stdet_P = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_P");
	init_pvar(stdet_P,NULL,'f',0.0);
	stdet_PC = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_PC");
	init_pvar(stdet_PC,NULL,'f',0.0);
	stdet_OM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_OM");
	init_pvar(stdet_OM,NULL,'f',0.0);
	stdet_CtoOM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"stdet_CtoOM");
	init_pvar(stdet_CtoOM,NULL,'f',0.0);

	WIND_SPEED = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"WIND_SPEED");
	init_pvar(WIND_SPEED,NULL,'f',0.0);
	WIND_SPEED_CANOPY = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"WIND_SPEED_CANOPY");
	init_pvar(WIND_SPEED_CANOPY,NULL,'f',0.0);
	WIND_STORM = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"WIND_STORM");
	init_pvar(WIND_STORM,NULL,'f',0.0);
	FIRE_WIND_EFFECT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"FIRE_WIND_EFFECT");
	init_pvar(FIRE_WIND_EFFECT,NULL,'f',0.0);
	CLOUDY = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"CLOUDY");
	init_pvar(CLOUDY,NULL,'f',0.0);
	SOLRADGRD = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SOLRADGRD");
	init_pvar(SOLRADGRD,NULL,'f',0.0);
	SOLRAD274 = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SOLRAD274");
	init_pvar(SOLRAD274,NULL,'f',0.0);
	AIR_TEMP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"AIR_TEMP");
	init_pvar(AIR_TEMP,NULL,'f',0.0);
	H2O_TEMP = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"H2O_TEMP");
	init_pvar(H2O_TEMP,NULL,'f',0.0);
	PAN_CT = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PAN_CT");
	init_pvar(PAN_CT,NULL,'f',0.0);
	PAN_CH = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PAN_CH");
	init_pvar(PAN_CH,NULL,'f',0.0);
	PAN_CW = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PAN_CW");
	init_pvar(PAN_CW,NULL,'f',0.0);
	HUMIDITY = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HUMIDITY");
	init_pvar(HUMIDITY,NULL,'f',0.0);
	DewPt = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"DewPt");
	init_pvar(DewPt,NULL,'f',0.0);
        
	ALG_MAX = NULL;

	DIN_CONC_GRAD = NULL;

	DOM_AEROBTHIN = NULL;
	DOM_MAXDEPTH = NULL;
	DOM_NPHBIO_PROP = NULL;
	DOM_PHBIO_PROP = NULL;

	FIRE_FUEL_HEAT = NULL;
	FIRE_RX_VELOC = NULL;

	HYD_POROSITY = NULL;
	HYD_RCINFILT = NULL;
	HYD_SPEC_YIELD = NULL;

	MAC_CANOPDECOUP = NULL;
	MAC_KSN = NULL;
	MAC_KSP = NULL;
	MAC_LIGHTSAT = NULL;
	MAC_MAXCANOPCOND = NULL;
	MAC_MAXHT = NULL;
	MAC_MAXLAI = NULL;
	MAC_MAXROUGH = NULL;
	MAC_MINROUGH = NULL;
	MAC_SALIN_THRESH = NULL;
	MAC_SPR_GROW = NULL;
	MAC_TEMPOPT = NULL;
	MAC_WAT_TOLER = NULL;
	NPHBIO_ABVBEL = NULL;
	NPHBIO_CTOOM = NULL;
	NPHBIO_MAX = NULL;
	NPHBIO_NC = NULL;
	NPHBIO_PC = NULL;
	NPHBIO_ROOTDEPTH = NULL;
	PHBIO_CTOOM = NULL;
	PHBIO_MAX = NULL;
	PHBIO_NC = NULL;
	PHBIO_PC = NULL;
	PHBIO_RCMORT = NULL;
	PHBIO_RCNPP = NULL;
        Mac_transl_rc = NULL;
        
        TP_CONC_GRAD = NULL;
        
	SALT_ICSEDWAT = NULL;
	SALT_ICSFWAT = NULL;
        
	FLOC_CONSPROP = NULL;        
	FLOC_CTOOM = NULL;
	FLOC_IC = NULL;
	FLOC_NC = NULL;
	FLOC_PC = NULL;
        
	STDET_ABVBEL = NULL;
	STDET_CTOOM = NULL;
	STDET_MAX = NULL;
	STDET_NC = NULL;
	STDET_PC = NULL;
	STDET_SHRED = NULL;
	STDET_WIND = NULL;


}

#if NeedFunctionPrototypes
float tempCF(int form, float c1, float topt, float tmin, float tmax, float tempC)
#else
float tempCF(form, c1, topt, tmin, tmax, tempC)
      float c1,topt, tmin, tmax, tempC;
      int form;

#endif
{
    if (form == 1) {
/* Lassiter et al. 1975, where c1 = curvature parm */
        return (Exp(c1*(tempC-topt)) 
                *pow(((tmax-tempC)/(tmax-topt)),
                     (c1*(tmax-topt))) );
    }
    else {
/* Jorgensen 1976 */
        return ( Exp(-2.3 * ABS((tempC-topt)/(topt-tmin))) );
    }
    
        
}



/*******
the remaining functions are time-series graph interpolaters
******/

#if NeedFunctionPrototypes
float graph5(unsigned char y, float x)
#else
float graph5(y,x)
	float x; unsigned char y;
#endif
{
	float s;
	int ig=0, Np=12;

	while(1) {
	if (x <= g5[ig][0]) { if(ig>0) ig=ig-1; else return(g5[0][1+y]);}
	else if (x > g5[ig][0] && x <= g5[ig+1][0]) {
		     s = 	(g5[ig+1][1+y]-g5[ig][1+y])/
				    (g5[ig+1][0]-g5[ig][0]);
		     return(s * (x-g5[ig][0]) + g5[ig][1+y]); }
	  else if (x > g5[ig+1][0]) { if(ig<Np) ig=ig+1; else return(g5[Np][1+y]);}
	}
}
#if NeedFunctionPrototypes
float graph7(unsigned char y, float x)
#else
float graph7(y,x)
	float x; unsigned char y;
#endif
{
	float s;
	int ig=0, Np=10;

	while(1) {
	if (x <= g7[ig][0]) { if(ig>0) ig=ig-1; else return(g7[0][1+y]);}
	else if (x > g7[ig][0] && x <= g7[ig+1][0]) {
		     s = 	(g7[ig+1][1+y]-g7[ig][1+y])/
				    (g7[ig+1][0]-g7[ig][0]);
		     return(s * (x-g7[ig][0]) + g7[ig][1+y]); }
	  else if (x > g7[ig+1][0]) { if(ig<Np) ig=ig+1; else return(g7[Np][1+y]);}
	}
}
#if NeedFunctionPrototypes
float graph8(unsigned char y, float x)
#else
float graph8(y,x)
	float x; unsigned char y;
#endif
{
	float s;
	int ig=0, Np=10;

	while(1) {
	if (x <= g8[ig][0]) { if(ig>0) ig=ig-1; else return(g8[0][1+y]);}
	else if (x > g8[ig][0] && x <= g8[ig+1][0]) {
		     s = 	(g8[ig+1][1+y]-g8[ig][1+y])/
				    (g8[ig+1][0]-g8[ig][0]);
		     return(s * (x-g8[ig][0]) + g8[ig][1+y]); }
	  else if (x > g8[ig+1][0]) { if(ig<Np) ig=ig+1; else return(g8[Np][1+y]);}
	}
}

