static char *SCCSID = "@(#)grid_io.c	1.8 04/14/95 SFWMD Research and Planning Departments";

/*

Copyright, 1995, South Florida Water Management District

DISCLAIMER:

ANY INFORMATION, INCLUDING BUT NOT LIMITED TO SOFTWARE AND DATA,
RECEIVED FROM THE SOUTH FLORIDA WATER MANAGEMENT DISTRICT ("DISTRICT")
IN FULFILLMENT OF A PUBLIC RECORDS REQUEST IS PROVIDED "AS IS" WITHOUT
WARRANTY OF ANY KIND, AND THE DISTRICT EXPRESSLY DISCLAIMS ALL EXPRESS
AND IMPLIED WARRANTIES, INCLUDING BUT NOT LIMITED TO THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
THE DISTRICT DOES NOT WARRANT, GUARANTEE, OR MAKE ANY REPRESENTATIONS
REGARDING THE USE, OR THE RESULTS OF THE USE, OF THE INFORMATION
PROVIDED TO YOU BY THE DISTRICT IN TERMS OF CORRECTNESS, ACCURACY,
RELIABILITY, TIMELINESS OR OTHERWISE.  THE ENTIRE RISK AS TO THE
RESULTS AND PERFORMANCE OF ANY INFORMATION OBTAINED FROM THE DISTRICT
IS ENTIRELY ASSUMED BY THE RECIPIENT.

*/

/* -------------------------------------------------------------
   grid_io
   This module containes routines to read and write the grid
   for the SFWMM.  The following routines are included:

   write_grid_header - writes the grid definition header 
       in binary format to a file
   read_grid_header - reads the grid definition header in
       binary format from a file
   grid_write  - writes the grid values in binary format
   grid_read   - reads the grid values in binary format

   This module should contain all necessary routines for 
   handling binary grid data from the SFWMM.  The functionality
   should include at least the following:

      (1) definition of a general binary format containing at 
      least (a) specification of the geographical grid (b)
      run identification (c) snapshot identification (date) 
      (d) snapshot of data for the entire grid

      (2) reading and writing of a snapshot from and to files

      (3) 
   ------------------------------------------------------------- */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <varargs.h>
#include <stdlib.h>

#include "grid_io.h"

#define TOO_MANY_NODES 10000
#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

/* internal routines */

void grid_io_fatal();
void F77_to_C_string();
void C_to_F77_string();

/* data for fortran interface routines */

static GRID fortran_grid;

/* -------------------------------------------------------------
   grid_write_header
   writes the grid definition header to a file
   ------------------------------------------------------------- */
int grid_write_header(file, grid)
FILE *file;
GRID *grid;
{
  int array_size, errs;
  errs = 0;

  if (fwrite(&grid->header, sizeof(GR_HEADER), 1, file) == 0) 
    ++errs;

  if (grid->header.number_of_rows < MAX_GRID_ROWS)
    array_size = MAX_GRID_ROWS;
  else
    array_size = grid->header.number_of_rows;

  if (fwrite(grid->config.xstart, sizeof(int), array_size, file) == 0)
    ++errs;

  if (fwrite(grid->config.xend, sizeof(int), array_size, file) == 0)
    ++errs;

  if (fwrite(grid->config.cum_node_count, sizeof(int), array_size, file) == 0)
    ++errs;

  if (errs)
    grid_io_fatal("write_grid_header", "Unable to write file header\n");
  return (0);
}

/* -------------------------------------------------------------
   wgridhd_
   Fortran interface to the grid_write_header

	deleted
   ------------------------------------------------------------- */

/* -------------------------------------------------------------
   grid_read_header
   reads grid definition information from a file
   returns 0 on success, -1 on eof, or some positive number on
   error
   ------------------------------------------------------------- */
int grid_read_header(file, grid)
FILE *file;
GRID *grid;
{
  int errs = 0, num_read, array_size;

  if ((num_read = fread(&grid->header, sizeof(GR_HEADER), 1, file)) == 0) {
    if (feof(file))
      errs = -1;
    else 
      grid_io_fatal("read_grid_header", "Unable to read header from file\n");
  }

  if (grid->header.number_of_rows < MAX_GRID_ROWS)
    array_size = MAX_GRID_ROWS;
  else
    array_size = grid->header.number_of_rows;

  /* allocate space for configuration arrays */
  grid->config.xstart = (int *) malloc(array_size * sizeof(int));
  grid->config.xend = (int *) malloc(array_size * sizeof(int));
  grid->config.cum_node_count = (int *) malloc(array_size * sizeof(int));

  if ((num_read = fread(grid->config.xstart, sizeof(int), array_size, file)) == 0) {
    if (feof(file))
      errs = -1;
    else 
      grid_io_fatal("read_grid_header", "Unable to read header from file\n");
  } 

  if ((num_read = fread(grid->config.xend, sizeof(int), array_size, file)) == 0) {
    if (feof(file))
      errs = -1;
    else
      grid_io_fatal("read_grid_header", "Unable to read header from file\n");
  }

  if ((num_read = fread(grid->config.cum_node_count, sizeof(int), array_size, file)) == 0) {
    if (feof(file))
      errs = -1;
    else
      grid_io_fatal("read_grid_header", "Unable to read header from file\n");
  }

  return(errs);
}

/* -------------------------------------------------------------
   gridrhd_
   Fortran interface to the grid_read_header routine.  Note
   that it changes the contents of the fortran_grid.  To get
   the header info a call to getgrid_
   ------------------------------------------------------------- */

/* -------------------------------------------------------------
   grid_write
   writes a ``snapshot'' of areal data (defined by grid) to a 
   binary file 
   ------------------------------------------------------------- */
int grid_write(file, grid, tag, values)
FILE *file;
GRID *grid;
char *tag;
float *values;
{
  char *char_ptr;
  int written, errs = 0;

  if ((written = fwrite(tag, sizeof(char), GRID_TAG_LENGTH, file)) == 0)
    grid_io_fatal("grid_write", "Unable to write grid tag\n");
  if (written < GRID_TAG_LENGTH) errs++;

  if ((written = fwrite(values, sizeof(float), grid->header.number_of_nodes, file)) == 0)
    grid_io_fatal("grid_write", "Unable to write grid data\n");
  if (written < grid->header.number_of_nodes) errs++;
    
  return(errs);
}

/* -------------------------------------------------------------
   gwrite_
   Fortran interface to the grid_write routine
   ------------------------------------------------------------- */
  

/* -------------------------------------------------------------
   grid_read
   reads a ``snapshot'' of areal data
   returns 0 on success, -1 on eof, and some positive number on
   partial read
   ------------------------------------------------------------- */
int grid_read(file, grid, tag, values)
FILE *file;
GRID *grid;
char *tag;
float *values;
{
  int num_read, errs = 0;

  /* read the tag */
  if ((num_read = fread(tag, sizeof(char), GRID_TAG_LENGTH, file)) == 0) {
    if (feof(file))
      return (-1);
    else
      grid_io_fatal("grid_read", "Uable to read grid tag\n");
  }
  if (num_read != GRID_TAG_LENGTH) errs++;

  /* read the values  */
  if ((num_read = fread(values, sizeof(float), grid->header.number_of_nodes, file)) == 0) 
    grid_io_fatal("grid_read", "Unable to read data\n");
  if (num_read != grid->header.number_of_nodes) errs++;
  return (errs);
}


/* -------------------------------------------------------------
   gread_
   Fortran interface to the grid_read routine
   ------------------------------------------------------------- */

/* -------------------------------------------------------------
   grid_skip
   This routine will move the file pointer the specified number
   of records.
   ------------------------------------------------------------- */
int grid_skip(file, grid, count)
FILE *file;
GRID *grid;
int count;
{
  int header_size, array_size;
  int errs = 0;
  long int rec_len =  GRID_TAG_LENGTH*sizeof(char) +
    grid->header.number_of_nodes*sizeof(float);
  long int end, current;

  if (count < 0) {
    if (grid->header.number_of_rows < MAX_GRID_ROWS)
      array_size = MAX_GRID_ROWS;
    else
      array_size = grid->header.number_of_rows;
    header_size = sizeof(GRID) + 3 * sizeof(int) * array_size;
    if (ftell(file) <  header_size + abs(count)*rec_len)
      errs = grid_top(file, grid);
    else
      fseek(file, count*rec_len, 1);
  } else if (count > 0) {
    current = ftell(file);
    fseek(file, 0, 2);
    end = ftell(file);
    if ((end - current)/rec_len < count)
      errs = grid_bottom(file, grid);
    else 
      errs = fseek(file, (long)(current + count*rec_len), 0);
  }
  
  return(errs);
}

/* -------------------------------------------------------------
   gridskp_
   Fortran interface to the grid_skip routine.  Note
   that it changes the contents of the fortran_grid.  To get
   the header info a call to getgrid_ is needed (cjn 1/94)
   ------------------------------------------------------------- */

/* -------------------------------------------------------------
   grid_top
   This routine places the file pointer before the first data
   record in the file
   ------------------------------------------------------------- */
int 
grid_top(file, grid)
FILE *file;
GRID *grid;
{
  GRID junk;

  fseek(file, 0L, 0);
  return(grid_read_header(file, &junk));
}

/* -------------------------------------------------------------
   grid_bottom
   This routine moves the file pointer to just before the final
   data record in the file.
   ------------------------------------------------------------- */
int 
grid_bottom(file, grid)
FILE *file;
GRID *grid;
{
  fseek(file, 0L, 2);
  return(grid_skip(file, grid, -1));
}

/*
char *strsed();
*/
/* -------------------------------------------------------------
   grid_tag_search
   This routine searches the grid_tags from the current position
   in the file for the regular expression passed.  The routine
   returns 1 if a matching tag was found, 0 if not.  If the
   the search was unsuccessfull the file is set to the bottom
   grid data.
   ------------------------------------------------------------- */
/*
int grid_tag_search(file, grid, string)
FILE *file;
GRID *grid;
char *string;
{
  char *char_ptr, tag[GRID_TAG_LENGTH], buffer[GRID_TAG_LENGTH];
  int range[2];
  float *data = (float *)malloc(grid->header.number_of_nodes*sizeof(float));
  int found = FALSE;

  sprintf(buffer, "/%s/", string);

  while (!found) {
    if (grid_read(file, grid, tag, data) != 0) {
      grid_bottom(file, grid);
      break;
    } else {
      if (strsed(tag, buffer, range) == 0)
	grid_io_fatal("grid_tag_search", "Error using strsed routine");
      else if (range[0] != -1 && range[1] != -1) {
         found = TRUE;
	 grid_skip(file, grid, -1);
       }
    }
  }

  char_ptr = (char *) data;
  free(char_ptr);
  return (found);
}

*/


/* -------------------------------------------------------------
   grid_node
   This routine returns the array index of the node cooresponding
   to the row and column number passed.
   ------------------------------------------------------------- */
int 
grid_node(grid, row, column)
GRID *grid;
int row, column;
{
  if (row > 0 && row <= grid->header.number_of_rows) {
    if (column >= grid->config.xstart[row - 1] && column <= grid->config.xend[row - 1]) 
      return ((grid->config.cum_node_count[row - 1] - 1) + column - grid->config.xstart[row - 1] + 1);
  }
  return -1;
}

/* -------------------------------------------------------------
   setgrid_
   Fortran call to set internal grid definition record 
   ------------------------------------------------------------- */
void 
setgrid_(title, nrows, nnodes, xsize, ysize, xstart, xend, cum_count, title_len)
char *title;
int *nrows, *nnodes;
float *xsize, *ysize;
int *xstart, *xend, *cum_count, title_len;
{
  int i, array_size;

  F77_to_C_string(fortran_grid.header.title, title, 
		  ((title_len < GRID_TITLE_LENGTH) ? title_len : GRID_TITLE_LENGTH));

  fortran_grid.header.number_of_rows = *nrows;
  fortran_grid.header.number_of_nodes = *nnodes;
  fortran_grid.header.size.x = *xsize;
  fortran_grid.header.size.y = *ysize;

  /* allocate space for configuration arrays */
  array_size = (*nrows < MAX_GRID_ROWS) ? MAX_GRID_ROWS : *nrows;
  fortran_grid.config.xstart = (int *) malloc(array_size * sizeof(int));
  fortran_grid.config.xend = (int *) malloc(array_size * sizeof(int));
  fortran_grid.config.cum_node_count = (int *) malloc(array_size * sizeof(int));
  
  for (i = 0; i < fortran_grid.header.number_of_rows; i++) {
    fortran_grid.config.xstart[i] = xstart[i];
    fortran_grid.config.xend[i] = xend[i];
    fortran_grid.config.cum_node_count[i] = cum_count[i];
  }
}

/* -------------------------------------------------------------
   getgrid_
   Fortran call to get internal grid definition record 
   by Karen Lythgoe and Cal Neidrauer, since Bill Perkins
   documented a non-existent routine.  Documentation is like sex,
   too much is much better than not enough!
   ------------------------------------------------------------- */
void 
getgrid_(title, nrows, nnodes, xsize, ysize, xstart, xend, cum_count, title_len)
char *title;
int *nrows, *nnodes;
float *xsize, *ysize;
int *xstart, *xend, *cum_count, title_len;
{
  int i;

  C_to_F77_string(title,fortran_grid.header.title,  
		  ((title_len < GRID_TITLE_LENGTH) ? title_len : GRID_TITLE_LENGTH));

  *nrows = fortran_grid.header.number_of_rows;
  *nnodes = fortran_grid.header.number_of_nodes;
  *xsize = fortran_grid.header.size.x;
  *ysize = fortran_grid.header.size.y;
  for (i = 0; i < fortran_grid.header.number_of_rows; i++) {
    xstart[i] = fortran_grid.config.xstart[i];
    xend[i] = fortran_grid.config.xend[i];
    cum_count[i] =fortran_grid.config.cum_node_count[i];
  }
}


/* -------------------------------------------------------------
   grid_io_fatal
   prints fatal error messages
   ------------------------------------------------------------- */
void grid_io_fatal(name, message)
char *name, *message;
{ 
  (void) fprintf(stderr, "ERROR in %s: ", name);
  (void) fprintf(stderr, message);
  (void) abort();
}

/* -------------------------------------------------------------
   F77_to_C_string
   converts fortran style strings to C style strings
   ------------------------------------------------------------- */
void 
F77_to_C_string(dest, src, length)
char *dest, *src;
int length;
{
  int i;
  strncpy(dest, src, length);
  for (i = length - 1; isspace(dest[i]); i--)
    dest[i] = '\0';
}


/* -------------------------------------------------------------
   C_to_F77_string
   converts C style strings to Fortran style
   ------------------------------------------------------------- */
void 
C_to_F77_string(dest, src, length)
char *dest, *src;
int length;
{
  int i;

  strncpy(dest,src,length);
  for (i = strlen(dest); i < length; i++)
    dest[i] = ' ';
}
