
#include "params.h"

#ifdef COUPLED


/**********************************************************************
 Dave Kohr's library, hacked by Ming Xu and hacked some more by ITF.

Public functions for the Unnamed Timing Package (UTP).
**********************************************************************/
#define _ALL_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "UTP.h"
#include "UTP_private.h"


/**********************************************************************
Public UTP functions.
**********************************************************************/
char *UTP_outFilename = NULL;
unsigned UTP_numtimers;
UTP_Timer_t *UTP_timers;

void UTP_init(unsigned numTimers)
{
	int i;

	if (!(UTP_timers = malloc(numTimers * sizeof(UTP_Timer_t)))) {
		UTP_warn("UTP_init: malloc() failed for timers");
		UTP_numtimers = 0;
		return;
	}
	
	UTP_numtimers = numTimers;
	/* This loop not only initializes every
	   field; by "touching" everything, it also
	   makes sure it is paged in. */
	for (i = 0; i < numTimers; i++) {
	    UTP_timers[i].state = UTP_TIMER_STOPPED;
	    UTP_timeZero(&UTP_timers[i].startTime);
	    UTP_timeZero(&UTP_timers[i].accumulatedTime);
	    UTP_timers[i].numEvents = 0;
	}
}

void UTP_start_timer(unsigned timerNumber)
{
	UTP_Timer_t *theTimer;

	theTimer = &UTP_timers[timerNumber];

	theTimer->state = UTP_TIMER_RUNNING;
	UTP_readTime(&theTimer->startTime);
}

void UTP_stop_timer(unsigned timerNumber)
{
	UTP_Timer_t *theTimer;
	UTP_TimeValue_t stopTime, elapsedTime;

	theTimer = &UTP_timers[timerNumber];

#ifndef NDEBUG
	if (theTimer->state != UTP_TIMER_RUNNING) {
		UTP_warnInt("UTP_stop_timer: timer %d not running",
			    (int) timerNumber);
		return;
	}
#endif /* #ifndef NDEBUG */

	theTimer->state = UTP_TIMER_STOPPED;
	UTP_readTime(&stopTime);
	UTP_timeDifference(&elapsedTime, &theTimer->startTime, &stopTime);
	UTP_timeAdd(&theTimer->accumulatedTime, &elapsedTime);
	theTimer->numEvents++;
}

void UTP_reset_timer(unsigned timerNumber)
{
	UTP_Timer_t *theTimer;

	theTimer = &UTP_timers[timerNumber];

	theTimer->state = UTP_TIMER_STOPPED;
	UTP_timeZero(&theTimer->startTime);
	UTP_timeZero(&theTimer->accumulatedTime);
	theTimer->numEvents = 0;
}

void
UTP_get_accum_time(unsigned timerNumber, double *time, int *precision)
{
	UTP_Timer_t *theTimer;
	theTimer = &UTP_timers[timerNumber];

	UTP_timeToDouble(time, precision, &theTimer->accumulatedTime);
}

/**********************************************************************
Private UTP functions.
**********************************************************************/

void
UTP_readTime(UTP_TimeValue_t *tv)
{
#ifdef UNIX
	int flag = gettimeofday(tv, NULL);

#ifdef DEBUG
	if (flag) {
		UTP_warn("UTP_readTime: gettimeofday() failed");
/*		UTP_warnStr("UTP_readTime: system error message is \"%s\"", sys_errlist[errno]);   */
	}
#endif /* #ifdef DEBUG */
#endif /* #ifdef UNIX */
}


void
UTP_timeDifference(UTP_TimeValue_t *diff, const UTP_TimeValue_t *start,
		   const UTP_TimeValue_t *end)
{
#ifdef UNIX
	diff->tv_sec = end->tv_sec - start->tv_sec;

	if (end->tv_usec < start->tv_usec) {
		diff->tv_sec--;		/* "Borrow" from seconds. */
		diff->tv_usec = 1000000 + end->tv_usec - start->tv_usec;
	} else
		diff->tv_usec = end->tv_usec - start->tv_usec;
#endif /* #ifdef UNIX */
}


void
UTP_timeAdd(UTP_TimeValue_t *oldElapsed,
	    const UTP_TimeValue_t *newElapsed)
{
#ifdef UNIX
	oldElapsed->tv_usec += newElapsed->tv_usec;

	if (oldElapsed->tv_usec >= 1000000) {
		oldElapsed->tv_sec++;	/* "Carry" into seconds. */
		oldElapsed->tv_usec -= 1000000;
	}

	oldElapsed->tv_sec += newElapsed->tv_sec;
#endif /* #ifdef UNIX */
}


void
UTP_timeZero(UTP_TimeValue_t *tv)
{
#ifdef UNIX
	tv->tv_sec = 0;
	tv->tv_usec = 0;
#endif /* #ifdef UNIX */
}


void
UTP_timeToDouble(double *time, int *precision, const UTP_TimeValue_t *tv)
{
#ifdef UNIX
	*time = (double) tv->tv_sec + ((double) tv->tv_usec) / 1000000.0;

	*precision = 6;
#endif /* #ifdef UNIX */
}


void
UTP_timeToString(char timeString[], const UTP_TimeValue_t *tv)
{
	double tvAsDouble;
	int precision;

	UTP_timeToDouble(&tvAsDouble, &precision, tv);

	sprintf(timeString, "%.*lf", precision, tvAsDouble);
}


void
UTP_warn(const char *messageStr)
{
#ifdef UNIX
	fprintf(stderr, messageStr);
	fprintf(stderr, "\n");
	fflush(stderr);		/* Just to be safe. */
#endif /* #ifdef UNIX */
}

void
UTP_warnStr(const char *messageStr, const char *argStr)
{
	char buff[256];

	sprintf(buff, messageStr, argStr);
	UTP_warn(buff);
}

void
UTP_warnInt(const char *messageStr, int argInt)
{
	char buff[256];

	sprintf(buff, messageStr, argInt);
	UTP_warn(buff);
}

#endif

