/* 
  TString class implementation
 */

#include "CString.h"
#include "Environ.h"
#include "PixVec.h"
#include "Utility.h"

extern "C" {
#include "sme_types.h"
}

HIDDEN const int MAXSTR = 2048;
CString spark( "", MAXSTR );

#ifdef USE_MPI
#include "mpi.h"
const int kCStringTag = 1097;
#endif

#undef OK

void TString::error(const char* msg) const
{
  gPrintErr( msg );
}

TString::operator const char*() const
{ 
  return (const char*)chars();
}

//  globals

TStrRep  _nilTStrRep( 1 );   // nil TStrings point here
TString _nilTString;               // nil TSubStrings point here




/*
 the following inline fcts are specially designed to work
 in support of TString classes, and are not meant as generic replacements
 for libc "str" functions.

 inline copy fcts -  I like left-to-right from->to arguments.
 all versions assume that `to' argument is non-null

 These are worth doing inline, rather than through calls because,
 via procedural integration, adjacent copy calls can be smushed
 together by the optimizer.
*/

// copy n bytes
inline static void ncopy(const char* from, char* to, int n)
{
  if (from != to) while (--n >= 0) *to++ = *from++;
}

// copy n bytes, null-terminate
inline static void ncopy0(const char* from, char* to, int n)
{
  if (from != to) 
  {
    while (--n >= 0) *to++ = *from++;
    *to = 0;
  }
  else
    to[n] = 0;
}

// copy until null
inline static void scopy(const char* from, char* to)
{
  if (from != 0) while((*to++ = *from++) != 0);
}

// copy right-to-left
inline static void revcopy(const char* from, char* to, short n)
{
  if (from != 0) while (--n >= 0) *to-- = *from--;
}


inline static int slen(const char* t) // inline  strlen
{
  if (t == 0)
    return 0;
  else
  {
    const char* a = t;
    while (*a++ != 0);
    return a - 1 - t;
  }
}

// minimum & maximum representable rep size

#define MAXTStrRep_SIZE   ((1 << (sizeof(short) * CHAR_BIT - 1)) - 1)
#define MINTStrRep_SIZE   16

#ifndef MALLOC_MIN_OVERHEAD
#define MALLOC_MIN_OVERHEAD  4
#endif

// The basic allocation primitive:
// Always round request to something close to a power of two.
// This ensures a bit of padding, which often means that
// concatenations don't have to realloc. Plus it tends to
// be faster when lots of TStrings are created and discarded,
// since just about any version of malloc (op new()) will
// be faster when it can reuse identically-sized chunks

TStrRep::TStrRep( int size,  char* str ) { 
	unsigned int allocsiz = MINTStrRep_SIZE;
	while ( allocsiz < (size+1) ) allocsiz <<= 1;
	if (allocsiz >= MAXTStrRep_SIZE) { gPrintErr("Requested length out of range"); }
	len=0; sz = allocsiz; 
	s = new char[allocsiz];
	if( str ) { ncopy0( str, s, size); len=size; }
	else s[0] = 0;
}

// Do-something-while-allocating routines.

// We live with two ways to signify empty Sreps: either the
// null pointer (0) or a pointer to the nilTStrRep.

// We always signify unknown source lengths (usually when fed a char*)
// via len == -1, in which case it is computed.

// allocate, copying src if nonull

TStrRep* TStrRep::Salloc(TStrRep* old, const char* src, int srclen, int newlen)
{
  if (old == &_nilTStrRep) old = 0;
  if (srclen < 0) srclen = slen(src);
  if (newlen < srclen) newlen = srclen;
  TStrRep* rep;
  if (old == 0 || newlen >= old->sz)
    rep = Snew(newlen);
  else
    rep = old;

  rep->len = newlen;
  ncopy0(src, rep->s, srclen);

  if (old != rep && old != 0) delete old;

  return rep;
}

// reallocate: Given the initial allocation scheme, it will
// generally be faster in the long run to get new space & copy
// than to call realloc

TStrRep* TStrRep::Sresize(TStrRep* old, int newlen)
{
  if (old == &_nilTStrRep) old = 0;
  TStrRep* rep;
  if (old == 0)
    rep = Snew(newlen);
  else if (newlen >= old->sz)
  {
    rep = Snew(newlen);
    ncopy0(old->s, rep->s, old->len);
    delete old;
  }
  else
    rep = old;

  rep->len = newlen;

  return rep;
}

void
TString::alloc (int newsize)
{
  unsigned short old_len = rep->len;
  rep = TStrRep::Sresize(rep, newsize);
  rep->len = old_len;
}

int TString::searchLine(int& start, int sl, char c, char end_char ) const
{
  const char* s = chars(); 
  if (sl > 0)
  {
    if (start >= 0)
    {
      const char* a = &(s[start]);
      const char* lasta = &(s[sl]);
      if( *a == end_char ) return start;
      while( *a == c ) { a++; start++; if( *a == end_char ) return start; }
      while (a < lasta) { 
				if ( *a == c || *a == end_char ) return a - s;
				a++;
			}
    }
    else
    {
      const char* a = &(s[sl + start + 1]);
			if( *a == end_char ) return start;
			while( *a == c ) { a--; start--; if( *a == end_char ) return start; }
      while (a >= s) { if (*a == c || *a == end_char) return a - s; a--; }
    }
  }
  return -1;
}

//  unsigned short    len;         // TString length 
//  unsigned short    sz;          // allocated space
//  char*              s;          // the TString starts here 

void TStrRep::clean_control_chars() {
	 char *sptr = s, *end = s+len;
	 do {
		if( iscntrl(*sptr) ) { *sptr = ' '; }
	} while( ++sptr < end );
}

// like allocate, but we know that src is a TStrRep

TStrRep* TStrRep::Scopy(TStrRep* old, const TStrRep* s)
{
  if (old == &_nilTStrRep) old = 0;
  if (s == &_nilTStrRep) s = 0;
  if (old == s) 
    return (old == 0)? &_nilTStrRep : old;
  else if (s == 0)
  {
    old->s[0] = 0;
    old->len = 0;
    return old;
  }
  else 
  {
    TStrRep* rep;
    int newlen = s->len ;
    if (old == 0 || newlen >= old->sz)
    {
      if (old != 0) delete old;
      rep = Snew(newlen);
    }
    else
      rep = old;
    rep->len = newlen;
    ncopy0(s->s, rep->s, newlen);
    return rep;
  }
}

// allocate & concatenate

TStrRep* TStrRep::Scat(TStrRep* old, const char* s, int srclen, const char* t, int tlen)
{
  if (old == &_nilTStrRep) old = 0;
  if (srclen < 0) srclen = slen(s);
  if (tlen < 0) tlen = slen(t);
  int newlen = srclen + tlen;
  TStrRep* rep;

  if (old == 0 || newlen >= old->sz || 
      (t >= old->s && t < &(old->s[old->len]))) // beware of aliasing
    rep = Snew(newlen);
  else
    rep = old;

  rep->len = newlen;

  ncopy(s, rep->s, srclen);
  ncopy0(t, &(rep->s[srclen]), tlen);

  if (old != rep && old != 0) delete old;

  return rep;
}

// double-concatenate

TStrRep* TStrRep::Scat(TStrRep* old, const char* s, int srclen, const char* t, int tlen,
             const char* u, int ulen)
{
  if (old == &_nilTStrRep) old = 0;
  if (srclen < 0) srclen = slen(s);
  if (tlen < 0) tlen = slen(t);
  if (ulen < 0) ulen = slen(u);
  int newlen = srclen + tlen + ulen;
  TStrRep* rep;
  if (old == 0 || newlen >= old->sz || 
      (t >= old->s && t < &(old->s[old->len])) ||
      (u >= old->s && u < &(old->s[old->len])))
    rep = Snew(newlen);
  else
    rep = old;

  rep->len = newlen;

  ncopy(s, rep->s, srclen);
  ncopy(t, &(rep->s[srclen]), tlen);
  ncopy0(u, &(rep->s[srclen+tlen]), ulen);

  if (old != rep && old != 0) delete old;

  return rep;
}

// like cat, but we know that new stuff goes in the front of existing rep

TStrRep* TStrRep::Sprepend(TStrRep* old, const char* t, int tlen)
{
  char* s;
  int srclen;
  if (old == &_nilTStrRep || old == 0)
  {
    s = 0; old = 0; srclen = 0;
  }
  else
  {
    s = old->s; srclen = old->len;
  }
  if (tlen < 0) tlen = slen(t);
  int newlen = srclen + tlen;
  TStrRep* rep;
  if (old == 0 || newlen >= old->sz || 
      (t >= old->s && t < &(old->s[old->len])))
    rep = Snew(newlen);
  else
    rep = old;

  rep->len = newlen;

  revcopy(&(s[srclen]), &(rep->s[newlen]), srclen+1);
  ncopy(t, rep->s, tlen);

  if (old != rep && old != 0) delete old;

  return rep;
}


// TString compare: first argument is known to be non-null

inline static int scmp(const char* a, const char* b)
{
  if (a == 0) return 1;
  if (b == 0) {
    return *a != 0;
  } else  {
    int diff = 0;
    while ((diff = *a - *b++) == 0 && *a++ != 0);
    return diff;
  }
}


inline static int ncmp(const char* a, int al, const char* b, int bl)
{
  int n = (al <= bl)? al : bl;
  int diff;
  while (n-- > 0) if ((diff = *a++ - *b++) != 0) return diff;
  return al - bl;
}

int fcompare(const TString& x, const TString& y)
{
  const char* a = x.chars();
  const char* b = y.chars();
  int al = x.length();
  int bl = y.length();
  int n = (al <= bl)? al : bl;
  int diff = 0;
  while (n-- > 0)
  {
    char ac = *a++;
    char bc = *b++;
    if ((diff = ac - bc) != 0)
    {
      if (ac >= 'a' && ac <= 'z')
        ac = ac - 'a' + 'A';
      if (bc >= 'a' && bc <= 'z')
        bc = bc - 'a' + 'A';
      if ((diff = ac - bc) != 0)
        return diff;
    }
  }
  return al - bl;
}

// these are not inline, but pull in the above inlines, so are 
// pretty fast

int compare(const TString& x, const char* b)
{
  return scmp(x.chars(), b);
}

int compare(const TString& x, const TString& y)
{
  return scmp(x.chars(), y.chars());
}

int compare(const TString& x, const TSubString& y)
{
  return ncmp(x.chars(), x.length(), y.chars(), y.length());
}

int compare(const TSubString& x, const TString& y)
{
  return ncmp(x.chars(), x.length(), y.chars(), y.length());
}

int compare(const TSubString& x, const TSubString& y)
{
  return ncmp(x.chars(), x.length(), y.chars(), y.length());
}

int compare(const TSubString& x, const char* b)
{
  if (b == 0)
    return x.length();
  else
  {
    const char* a = x.chars();
    int n = x.length();
    int diff;
    while (n-- > 0) if ((diff = *a++ - *b++) != 0) return diff;
    return (*b == 0) ? 0 : -1;
  }
}

/*
 index fcts
*/

int TString::search(int start, int sl, char c) const
{
  const char* s = chars();
  if (sl > 0)
  {
    if (start >= 0)
    {
      const char* a = &(s[start]);
      const char* lasta = &(s[sl]);
      while (a < lasta) if (*a++ == c) return --a - s;
    }
    else
    {
      const char* a = &(s[sl + start + 1]);
      while (--a >= s) if (*a == c) return a - s;
    }
  }
  return -1;
}

TString& TString::strip(const char* k)
{
  int j = 0;
  const char* s = chars();
  for (int i = 0; s[i]; i++)
    {
      char c = s[i];
      if (strchr(k, c) == NULL) *((*this)(j++)) = c;
    }
  cut(j);
  return *this;
}

int TString::search(int start, int sl, const char* t, int tl) const
{
  const char* s = chars();
  if (tl < 0) tl = slen(t);
  if (sl > 0 && tl > 0)
  {
    if (start >= 0)
    {
      const char* lasts = &(s[sl - tl]);
      const char* lastt = &(t[tl]);
      const char* p = &(s[start]);

      while (p <= lasts)
      {
        const char* x = p++;
        const char* y = t;
        while (*x++ == *y++) if (y >= lastt) return --p - s;
      }
    }
    else
    {
      const char* firsts = &(s[tl - 1]);
      const char* lastt =  &(t[tl - 1]);
      const char* p = &(s[sl + start + 1]); 

      while (--p >= firsts)
      {
        const char* x = p;
        const char* y = lastt;
        while (*x-- == *y--) if (y < t) return ++x - s;
      }
    }
  }
  return -1;
}

int TString::match(int start, int sl, int exact, const char* t, int tl) const
{
  if (tl < 0) tl = slen(t);

  if (start < 0)
  {
    start = sl + start - tl + 1;
    if (start < 0 || (exact && start != 0))
      return -1;
  }
  else if (exact && sl - start != tl)
    return -1;

  if (sl == 0 || tl == 0 || sl - start < tl || start >= sl)
    return -1;

  int n = tl;
  const char* s = &(rep->s[start]);
  while (--n >= 0) if (*s++ != *t++) return -1;
  return tl;
}

int TString::starts_with( const char* t, int tl ) const {
  if( t == NULL ) return 0;
  if (tl < 0) tl = slen(t);
  int n = tl;
  const char* s = &(rep->s[0]);
  while (--n >= 0) if (*s++ != *t++) return 0;
  return tl;
}


void TSubString::assign(const TStrRep* ysrc, const char* ys, int ylen)
{
  if (&S == &_nilTString) return;

  if (ylen < 0) ylen = slen(ys);
  TStrRep* targ = S.rep;
  int sl = targ->len - len + ylen;

  if (ysrc == targ || sl >= targ->sz)
  {
    TStrRep* oldtarg = targ;
    targ = TStrRep::Sresize(0, sl);
    ncopy(oldtarg->s, targ->s, pos);
    ncopy(ys, &(targ->s[pos]), ylen);
    scopy(&(oldtarg->s[pos + len]), &(targ->s[pos + ylen]));
    delete oldtarg;
  }
  else if (len == ylen)
    ncopy(ys, &(targ->s[pos]), len);
  else if (ylen < len)
  {
    ncopy(ys, &(targ->s[pos]), ylen);
    scopy(&(targ->s[pos + len]), &(targ->s[pos + ylen]));
  }
  else
  {
    revcopy(&(targ->s[targ->len]), &(targ->s[sl]), targ->len-pos-len +1);
    ncopy(ys, &(targ->s[pos]), ylen);
  }
  targ->len = sl;
  S.rep = targ;
}



/*
 * substitution
 */


int TString::_gsub(const char* pat, int pl, const char* r, int rl)
{
  int nmatches = 0;
  if (pl < 0) pl = slen(pat);
  if (rl < 0) rl = slen(r);
  int sl = length();
  if (sl <= 0 || pl <= 0 || sl < pl)
    return nmatches;
  
  const char* s = rep->s;

  // prepare to make new rep
  TStrRep* nrep = 0;
  int nsz = 0;
  char* x = 0;

  int si = 0;
  int xi = 0;
  int remaining = sl;

  while (remaining >= pl)
  {
    int pos = search(si, sl, pat, pl);
    if (pos < 0)
      break;
    else
    {
      ++nmatches;
      int mustfit = xi + remaining + rl - pl;
      if (mustfit >= nsz)
      {
        if (nrep != 0) nrep->len = xi;
        nrep = TStrRep::Sresize(nrep, mustfit);
        nsz = nrep->sz;
        x = nrep->s;
      }
      pos -= si;
      ncopy(&(s[si]), &(x[xi]), pos);
      ncopy(r, &(x[xi + pos]), rl);
      si += pos + pl;
      remaining -= pos + pl;
      xi += pos + rl;
    }
  }

  if (nrep == 0)
  {
    if (nmatches == 0)
      return nmatches;
    else
      nrep = TStrRep::Sresize(nrep, xi+remaining);
  }

  ncopy0(&(s[si]), &(x[xi]), remaining);
  nrep->len = xi + remaining;

  if (nrep->len <= rep->sz)   // fit back in if possible
  {
    rep->len = nrep->len;
    ncopy0(nrep->s, rep->s, rep->len);
    delete(nrep);
  }
  else
  {
    delete(rep);
    rep = nrep;
  }
  return nmatches;
}

/*
 * deletion
 */

void TString::truncate( int len )
{
  if ( len <= 0 ) return ;
  int nlen = length() - len;
  nlen = (nlen<0) ? 0 : nlen;
  rep->len = nlen;
  rep->s[nlen] = 0;
}

void TString::del(int pos, int len)
{
  if (pos < 0 || len <= 0 || (unsigned)(pos + len) > length()) return ;
  int nlen = length() - len;
  int first = pos + len;
  ncopy0(&(rep->s[first]), &(rep->s[pos]), length() - first);
  rep->len = nlen;
}

void TString::del(const char* t, int startpos)
{
  int tlen = slen(t);
  int p = search(startpos, length(), t, tlen);
  del(p, tlen);
}

void TString::del(const TString& y, int startpos)
{
  del(search(startpos, length(), y.chars(), y.length()), y.length());
}

void TString::del(const TSubString& y, int startpos)
{
  del(search(startpos, length(), y.chars(), y.length()), y.length());
}

void TString::del(char c, int startpos)
{
  del(search(startpos, length(), c), 1);
}

/*
 * TSubString extraction
 */


TSubString TString::at(int first, int len)
{
  return _substr(first, len);
}

TSubString TString::operator() (int first, int len)
{
  return _substr(first, len);
}

TSubString TString::before(int pos)
{
  return _substr(0, pos);
}

TSubString TString::through(int pos)
{
  return _substr(0, pos+1);
}

TSubString TString::after(int pos) 
{
  return _substr(pos + 1, length() - (pos + 1));
}

TSubString TString::from(int pos) 
{
  return _substr(pos, length() - pos);
}

TSubString TString::at(const TString& y, int startpos) 
{
  int first = search(startpos, length(), y.chars(), y.length());
  return _substr(first,  y.length());
}

TSubString TString::at(const TSubString& y, int startpos) 
{
  int first = search(startpos, length(), y.chars(), y.length());
  return _substr(first, y.length());
}


TSubString TString::at(const char* t, int startpos) 
{
  int tlen = slen(t);
  int first = search(startpos, length(), t, tlen);
  return _substr(first, tlen);
}

TSubString TString::at(char c, int startpos) 
{
  int first = search(startpos, length(), c);
  return _substr(first, 1);
}

TSubString TString::before(const TString& y, int startpos) 
{
  int last = search(startpos, length(), y.chars(), y.length());
  if( last == -1 ) last = length();
  return _substr(0, last);
}

TSubString TString::before(const TSubString& y, int startpos) 
{
  int last = search(startpos, length(), y.chars(), y.length());
  if( last == -1 ) last = length();
  return _substr(0, last);
}

TSubString TString::before(char c, int startpos) 
{
  int last = search(startpos, length(), c);
  if( last == -1 ) last = length();
  return _substr(0, last);
}

TSubString TString::before(const char* t, int startpos) 
{
  int tlen = slen(t);
  int last = search(startpos, length(), t, tlen);
  if( last == -1 ) last = length();
  return _substr(0, last);
}

TSubString TString::through(const TString& y, int startpos) 
{
  int last = search(startpos, length(), y.chars(), y.length());
  if (last >= 0) last += y.length();
  return _substr(0, last);
}

TSubString TString::through(const TSubString& y, int startpos) 
{
  int last = search(startpos, length(), y.chars(), y.length());
  if (last >= 0) last += y.length();
  return _substr(0, last);
}


TSubString TString::through(char c, int startpos) 
{
  int last = search(startpos, length(), c);
  if (last >= 0) last += 1;
  return _substr(0, last);
}

TSubString TString::through(const char* t, int startpos) 
{
  int tlen = slen(t);
  int last = search(startpos, length(), t, tlen);
  if (last >= 0) last += tlen;
  return _substr(0, last);
}

TSubString TString::after(const TString& y, int startpos) 
{
  int first = search(startpos, length(), y.chars(), y.length());
  if (first >= 0) first += y.length();
  return _substr(first, length() - first);
}

TSubString TString::after(const TSubString& y, int startpos) 
{
  int first = search(startpos, length(), y.chars(), y.length());
  if (first >= 0) first += y.length();
  return _substr(first, length() - first);
}

TSubString TString::after(char c, int startpos) 
{
  int first = search(startpos, length(), c);
  if (first >= 0) first += 1;
  return _substr(first, length() - first);
}

TSubString TString::getTSubString(char c, int& startpos, char endChar)  // gets TSubString from startpos to next occurrence of char c
{
	int end = searchLine(startpos, length(), c, endChar);
	int oldStart = startpos; startpos = end;
	return _substr(oldStart,end-oldStart);
}

TSubString TString::after(const char* t, int startpos) 
{
  int tlen = slen(t);
  int first = search(startpos, length(), t, tlen);
  if (first >= 0) first += tlen;
  return _substr(first, length() - first);
}

TSubString TString::from(const TString& y, int startpos) 
{
  int first = search(startpos, length(), y.chars(), y.length());
  return _substr(first, length() - first);
}

TSubString TString::from(const TSubString& y, int startpos) 
{
  int first = search(startpos, length(), y.chars(), y.length());
  return _substr(first, length() - first);
}


TSubString TString::from(char c, int startpos) 
{
  int first = search(startpos, length(), c);
  return _substr(first, length() - first);
}

TSubString TString::from(const char* t, int startpos) 
{
  int tlen = slen(t);
  int first = search(startpos, length(), t, tlen);
  return _substr(first, length() - first);
}



/*
 * split/join
 */


int split(const TString& src, TString results[], int n, const TString& sep)
{
  TString x = src;
  const char* s = x.chars();
  int sl = x.length();
  int i = 0;
  int pos = 0;
  while (i < n && pos < sl)
  {
    int p = x.search(pos, sl, sep.chars(), sep.length());
    if (p < 0)
      p = sl;
    results[i].rep = TStrRep::Salloc(results[i].rep, &(s[pos]), p - pos, p - pos);
    i++;
    pos = p + sep.length();
  }
  return i;
}

void join(TString& x, TString src[], int n, const TString& separator) 
{
  TString sep = separator;
  int xlen = 0;
  int i;
  for (i = 0; i < n; ++i)  xlen += src[i].length();
  
  xlen += (n - 1) * sep.length();

  x.rep = TStrRep::Sresize (x.rep, xlen);

  int j = 0;
	char* tstr = x(0);
  
  for (i = 0; i < n - 1; ++i)
  {
    ncopy(src[i].chars(), tstr+j, src[i].length());
    j += src[i].length();
    ncopy(sep.chars(), tstr+j, sep.length());
    j += sep.length();
  }
  ncopy0(src[i].chars(), tstr+j, src[i].length());
}
  
/*
 misc
*/

    
TStrRep* TStrRep::Sreverse(const TStrRep* src, TStrRep* dest)
{
  int n = src->len;
  if (src != dest)
    dest = Salloc(dest, src->s, n, n);
  if (n > 0)
  {
    char* a = dest->s;
    char* b = &(a[n - 1]);
    while (a < b)
    {
      char t = *a;
      *a++ = *b;
      *b-- = t;
    }
  }
  return dest;
}


TStrRep* TStrRep::Supcase(const TStrRep* src, TStrRep* dest)
{
  int n = src->len;
  if (src != dest) dest = Salloc(dest, src->s, n, n);
  char* p = dest->s;
  char* e = &(p[n]);
  for (; p < e; ++p) if (islower(*p)) *p = toupper(*p);
  return dest;
}

TStrRep* TStrRep::Sdowncase(const TStrRep* src, TStrRep* dest)
{
  int n = src->len;
  if (src != dest) dest = Salloc(dest, src->s, n, n);
  char* p = dest->s;
  char* e = &(p[n]);
  for (; p < e; ++p) if (isupper(*p)) *p = tolower(*p);
  return dest;
}

TStrRep* TStrRep::Scapitalize(const TStrRep* src, TStrRep* dest)
{
  int n = src->len;
  if (src != dest) dest = Salloc(dest, src->s, n, n);

  char* p = dest->s;
  char* e = &(p[n]);
  for (; p < e; ++p)
  {
    int at_word;
    if (at_word = islower(*p))
      *p = toupper(*p);
    else 
      at_word = isupper(*p) || isdigit(*p);

    if (at_word)
    {
      while (++p < e)
      {
        if (isupper(*p))
          *p = tolower(*p);
	/* A '\'' does not break a word, so that "Nathan's" stays
	   "Nathan's" rather than turning into "Nathan'S". */
        else if (!islower(*p) && !isdigit(*p) && (*p != '\''))
          break;
      }
    }
  }
  return dest;
}


void replicate(TString& w, char c, int n)
{
  w.rep =  TStrRep::Sresize(w.rep, n);
  char* p = w(0);
  while (n-- > 0) *p++ = c;
  *p = 0;
}

void replicate(TString& w, const TString& y, int n)
{
  int len = y.length();
  w.rep =  TStrRep::Sresize(w.rep, n * len);
  char* p = w(0);
  while (n-- > 0)
  {
    ncopy(y.chars(), p, len);
    p += len;
  }
  *p = 0;
}

void common_prefix(TString& r, const TString& x, const TString& y, int startpos)
{
  const char* xchars = x.chars();
  const char* ychars = y.chars();
  const char* xs = &(xchars[startpos]);
  const char* ss = xs;
  const char* topx = &(xchars[x.length()]);
  const char* ys = &(ychars[startpos]);
  const char* topy = &(ychars[y.length()]);
  int l;
  for (l = 0; xs < topx && ys < topy && *xs++ == *ys++; ++l);
  r.rep =  TStrRep::Salloc(r.rep, ss, l, l);
}

void common_suffix(TString& r, const TString& x, const TString& y, int startpos) 
{
  const char* xchars = x.chars();
  const char* ychars = y.chars();
  const char* xs = &(xchars[x.length() + startpos]);
  const char* botx = xchars;
  const char* ys = &(ychars[y.length() + startpos]);
  const char* boty = ychars;
  int l;
  for (l = 0; xs >= botx && ys >= boty && *xs == *ys ; --xs, --ys, ++l);
  r.rep =  TStrRep::Salloc(r.rep, ++xs, l, l);
}

int TString::freq(const TSubString& y) const
{
  int found = 0;
  for (unsigned int i = 0; i < length(); i++) 
    if (match(i,length(),0,y.chars(), y.length())>= 0) found++;
  return(found);
}

int TString::freq(const TString& y) const
{
  int found = 0;
  for (unsigned int i = 0; i < length(); i++) 
    if (match(i,length(),0,y.chars(),y.length()) >= 0) found++;
  return(found);
}

int TString::freq(const char* t) const
{
  int found = 0;
  for (unsigned int i = 0; i < length(); i++) 
    if (match(i,length(),0,t) >= 0) found++;
  return(found);
}

int TString::freq(char c) const
{
  int found = 0;
  for (unsigned int i = 0; i < length(); i++) 
    if (match(i,length(),0,&c,1) >= 0) found++;
  return(found);
}


int TString::OK() const
{
  if (rep == 0             // don't have a rep
    || rep->len > rep->sz     // TString oustide bounds
    || rep->s[rep->len] != 0)   // not null-terminated
      error("invariant failure");
  return 1;
}

int TSubString::OK() const
{
  int v = S != (const char*)0; // have a TString;
  v &= S.OK();                 // that is legal
  v &= pos + len >= S.rep->len;// pos and len within bounds
  if (!v) S.error("TSubString invariant failure");
  return v;
}

void CString::ReAlloc( int newlen ) {
  if (rep == &_nilTStrRep) rep = 0;
  TStrRep* new_rep =  TStrRep::Snew(newlen);
  if(rep) {
    ncopy0( rep->s, new_rep->s, rep->len );
    new_rep->len = rep->len;
    delete rep; 
  } else new_rep->len = 0;
  rep = new_rep;
  memset(rep->s+rep->len,newlen-rep->len,0);
}

CString& CString::trim( char schar, char echar ){
  int ip = length(); char tch;
  while( ((tch=rep->s[ip-1]) == ' ') || ( echar && (tch == echar )) ) { rep->s[--ip] = '\0'; }
  del(ip,length()-ip); ip = 0;
  while( ((tch=rep->s[ip]) == ' ') || ( schar && (tch == schar)) ) { ip++; }
  del(0,ip);
  return *this;
}

void CString::resize() { 	rep->len = slen(rep->s); }

int CString::containsName( const char* name, int start, Bool checkBeg, Bool CheckEnd  ) const {
  int p = index(name,start);
  if( p >= 0 ) {
    int test0=1, test1=1;
    const char* s = rep->s;
    if( p > 0 && checkBeg) {
      char c0 = s[p-1];
      if( Util::isalnumf( c0 ) ) test0=0;
    }
    if( p+slen(name) < length() && CheckEnd) {
      char c1 = s[p+slen(name)];
      if( Util::isalnumf( c1 ) && (c1 != '@') ) test1=0;
    }
    if( test0 && test1 ) return p; 
    else return containsName( name, p+slen(name) );
  }
  return -1;
}

TStrRep* TStrRep::Sinsert( TStrRep* old, const char* t, int tlen, int iloc )
{
  char* s;
  int srclen;
  if (old == &_nilTStrRep || old == 0)
  {
    s = 0; old = 0; srclen = 0;
  }
  else
  {
    s = old->s; srclen = old->len;
  }
  if (tlen < 0) tlen = slen(t);
  int newlen = srclen + tlen;
  TStrRep* rep;
  if (old == 0 || newlen > old->sz || 
      (t >= old->s && t < &(old->s[old->len])))
    rep = Snew(newlen);
  else
    rep = old;

  rep->len = newlen;

  if ( old != rep ) ncopy(s, rep->s, iloc);
  revcopy( &(s[srclen]), &(rep->s[newlen]), srclen+1-iloc );
  ncopy(t, &(rep->s[iloc]), tlen);
  

  if (old != rep && old != 0) delete old;

  return rep;
}


int CString::replaceName( CString& name, CString& repl, Bool checkBeg, Bool checkEnd  ) {
  int ioff =0;
  while( (ioff = containsName(name,ioff,checkBeg,checkEnd)) >= 0 ) {
    int ioff1 = ioff+name.length();
    ioff1 = scan_relcode(ioff1);
    del(ioff,ioff1-ioff);
    insert(repl,ioff);
    ioff += repl.length();
  }
  return ioff;
}

int CString::ProcSend( int from_proc, int to_proc ){
	if( from_proc == to_proc ) return 0;
	int size = 0;
#ifdef USE_MPI
  if( gIProc == to_proc ) {
    MPI_Status status; 
    MPI_Probe(from_proc,kCStringTag,MPI_COMM_WORLD,&status);
    MPI_Get_count(&status,MPI_CHAR,&size);
		if( size > 0 ) {
			alloc(size+1);
			MPI_Recv(rep->s,size,MPI_CHAR,from_proc,kCStringTag,MPI_COMM_WORLD,&status);
			rep->len = size;	
		}
  } else if( gIProc == from_proc ) {
    size = length();
    if(size>0) {
			MPI_Send(rep->s,size,MPI_CHAR,to_proc,kCStringTag,MPI_COMM_WORLD);
		}
  }
#endif
	return 0;
}

int CString::size() const { 
	if( length() == 0 ) return 0;
	return slen( rep->s ); 
}

int CString::scan_relcode(int ioff1) const {
  if( rep->s[ioff1] == '@' ) { 
    int inpar = 0;
    while ( ioff1 < rep->len ) {
      char ch = rep->s[++ioff1];
      if( ch == '(' ) inpar = 1;
      else if( inpar && ch == ')' ) inpar = 0;
      else if( inpar==0 && ( ch == 'N' || ch == 'S' || ch == 'E' || ch == 'W' )  ) {;}
      else if( inpar==1 && ( isdigit(ch) || ch == ',') ) {;}
      else break;
    }
  }
  return ioff1;
}

enum ECharState { kCol, kQuote, kSpace };
 
int CString::splitColumns( CString res[], int nres )
{
	char *s = rep->s;
	int ch, len = rep->len;
	ECharState cl = kSpace;
	CString* sres = NULL;
	int col_index=-1;
	for( int i=0; i<len; i++ ) {
		ch = s[i];
		if( (ch=='\n') || (ch=='\r') || (ch=='\f') || (ch ==EOF) || (ch == 0) ) { return  col_index+1; }
		switch(cl) {
			case kCol:
				if( (ch==' ') || (ch=='\t') || (ch==',') ) {  
					cl = kSpace; 
				} else { 
					(*sres) += (char)ch; 
				}
				break;
			case kQuote:
				if( ch=='"' ) {  
					cl = kSpace; 
				} else { 
					(*sres) += (char)ch; 
				}
				break;
			case kSpace:
				if( (ch==' ') || (ch=='\t') ) { ; }
				else if( ch == '"' ) { 
					cl = kQuote;
					if( ++col_index == nres ) { 
						gPrintErr(" Truncated data in splitColumns " );
						return nres;
					}
					sres = res+col_index;					
					(*sres) = (char)ch;
				} else {
					cl = kCol;
					if( ++col_index == nres ) { 
						gPrintErr(" Truncated data in splitColumns " );
						return nres;
					}
					sres = res+col_index;					
					(*sres) = (char)ch;
				}
				break;
		}
	}
	return  col_index+1;		
}



VObject* TString::dup() const { 
	TString* s = new TString(); 
	(*s) = (*this);
	return s; 
}

VObject* CString::dup() const { 
	CString* s = new CString(); 
	(*s) = (*this);
	return s; 
}

 ///////////////////////////////////////////////////////////
 // Filename
 ///////////////////////////////////////////////////////////

 inline bool is_not_slash(char s)
 {  return s != '\\' && s != '/'; }

 // Certified 90%
 const char* TFilename::ext() const
 {
   const char* d = strrchr(name(), '.');
   if (d && is_not_slash(*(++d))) return d;
   return "";
 }

 // Certified 90%
 void TFilename::ext(const char* e)
 {
   char* d = strrchr(name(), '.');
   if (d && is_not_slash(*(d+1))) *d = '\0';

   if (*e && *e != '.') *this += ".";
   *this += e;
 }


 // Certified 90%
 const char* TFilename::name() const
 {
   const char* s = chars(); 
   const char* d = strrchr(s, '/');
   if (d == NULL) d = strrchr(s, '\\');
   if (d == NULL) d = strchr(s, ':');
   if (d == NULL) d = s-1;
   return d+1;
 }

 // Certified 90%
 const char* TFilename::path() const
 {
   const char* s = chars(); 
   const char* d = strrchr( s, '/');
   if (d == NULL) d = strrchr( s, '\\');
   if (d == NULL)
   {
     d = strchr( s, ':');
     if (d != NULL) d++;
   }
   if (d == NULL) spark.cut(0);
   else {
	 spark = chars();
	 spark.truncate( d - s );
   }
   return spark;
 }

/*

 const TFilename& TFilename::tempdir()
 {
  const char* s = chars(); 
   const char* dirpref = getenv("TEMP");
   if (dirpref == NULL) dirpref = getenv("TMP");
   if (dirpref == NULL) dirpref = "/tmp";
   set(dirpref);

   const int last = size()-1;
   if (!is_not_slash(s[last])) 
     cut(last);

   int res = 0;

   if (!fexist(chars())) {  
     res =  mkdir(chars(), 0777);
   }

   if (res != 0)
     fatal_box("Impossibile creare la directory '%s' per i file temporanei", chars());

   return *this;
 }



 const TFilename& TFilename::temp(const char* prefix)
 {
   const TFilename dirpref(tempdir());
   char* t = NULL;

   if (prefix)
   {
     set(prefix);      // Copia prefisso e ...
     strip("$#");      // ... toglie caratteri jolly

     const CString f(prefix);
     if (f.find("$$") != -1) appendIndex((long)getpid());
     if (f.find("##") != -1) appendIndex((long)getuid());

     t = tempnam((char*)(const char*)dirpref, (char*)chars());
   }
   else
     t = tempnam((char*)(const char*)dirpref, NULL);

   set(t);

 #ifdef DBG
   if (fexist(chars())) 
     fatal_box("Il file '%s' esiste gia'", chars());
 #endif  
   if (t) free(t);

   return *this;
 }
*/

 ///////////////////////////////////////////////////////////
 // Token string
 ///////////////////////////////////////////////////////////

 // Certified 100%
 TToken_string::TToken_string(const char* s, char separator)
 : CString(s), _separator(separator)
 {
   restart();
 }

 // Certified 100%
 TToken_string::TToken_string(int n, char separator)
 : CString(n), _separator(separator), _last(0)
 {}

 // Certified 100%
 TToken_string::TToken_string(const TToken_string& s)
 : CString(s), _separator(s._separator), _last(s._last)
 {}


 // Certified 100%
 TObject* TToken_string::dup() const
 {
   return new TToken_string(chars(), _separator);
 }

 // Certified 90%
 const char* TToken_string::get()
 {
   const char* s = chars(); 
   if (_last < 0) return NULL;

   const int start = _last;

   if ( s[start] == '\0')
   {
     _last = -1;
     return NULL;
   }
   else
     if (s[start] == _separator)
     {
       _last = start+1;
     }
     else
     {
       const int k = find(_separator, start);
       _last = (k >= 0) ? k+1 : -1;
       spark = sub(start, k);
       return spark.chars();
     }
   return "";
 }

 // Certified 50%
 const char* TToken_string::get(int n)
 {
   const char* ts = chars(); 
   if (n < 0)
   {
     if (n == -2)
     {
       const char* sep = strrchr(ts, _separator);
       _last = -1;
       return sep ? sep+1 : ts;
     }
     else return get();
   }
   int sep = 0; const char* s;
   for (s = ts; sep < n && *s; s++)
     if (*s == _separator) sep++;

   if (sep >= n)
   {
     char* p = strchr(s, _separator);
     if (p == NULL)
     {
       spark = s;
       _last = -1;
     }
     else
     {
       *p = '\0';
       spark = s;
       *p = _separator;
       _last = (int)((const char*)p - ts) + 1;
     }
   }
   else
   {
     _last = -1;
     return NULL;
   }

   return spark;
 }


 // Certified 99%
 char TToken_string::get_char(int n)
 {
   const char* const car = get(n);
   return car ? *car : '\0';
 }

 // Certified 99%
 int TToken_string::get_int(int n)
 {
   const char* const num = get(n);
   return num ? atoi(num) : 0;
 }

 // Certified 99%
 float TToken_string::get_float(int n)
 {
   const char* const num = get(n);
   return num ? (float)atof(num) : 0.0;
 }

 // Certified 99%
 long TToken_string::get_long(int n)
 {
   const char* const num = get(n);
   return num ? atol(num) : 0L;
 }


 // Certified 70%
 bool TToken_string::set_item(const char* v, int n)
 {
   const char* s = chars(); 
   int sep = 0; int i;
   for (i = 0; sep < n && s[i]; i++)
     if (s[i] == _separator) sep++;

   if (sep < n)  // Aggiunge items mancanti prima della posizione n
   {
     for (;sep < n; sep++) *this += _separator;
     *this += v;
     return FALSE;
   }

   int e = find(_separator, i);
   if (e < 0) e = size();

   spark = &s[e];           // Salva items seguenti
   cut(i);                     // Considera solo items precedenti
   (*this += v) += spark;        // Aggiunge item desiderato e seguenti
   return TRUE;
 }


 // Certified 80%
 int TToken_string::get_pos(const char* s)
 {
   const char* item;

   restart();
   for (int i = 0; (item = get()) != NULL; i++)
     if (strcmp(item, s) == 0) return i;

   return -1;
 }


 // Certified 90%
 bool TToken_string::empty_items() const
 {
   for (const char* c = chars(); *c; c++)
     if (!isspace(*c) && *c != _separator) return FALSE;
   return TRUE;
 }


 // Certified 80%
 int TToken_string::items() const
 {
   int t = 0;
   if (!empty())
   {
     t++;
     for (const char* s = chars(); *s; s++) 
       if (*s == _separator) t++;
   }  
   return t;
 }

 // Adds an item to the token string
 // Certified 99%
 void TToken_string::add(const char* s, int pos)
 {
   if (s == NULL || *s == '\0') 
     s = " ";
   if (pos < 0)
   {
     if (!empty()) *this += _separator;
     *this += s;
   }
   else
     set_item(s, pos);
   if (_last < 0) _last = 0;
 }

 // Certified 0%
 void TToken_string::add(char c, int pos)
 {
   const char s[2] = { c, '\0' };
   add(s, pos);
 }

 // Adds an integer value to the token string
 // Certified 100%
 void TToken_string::add(long n, int pos)
 {
   char s[16];
   sprintf(s, "%ld", n);
   add(s, pos);
 }

 void TToken_string::add(int n, int pos)
 {
   char s[16];
   sprintf(s, "%d", n);
   add(s, pos);
 }

 // Certified 50%
 void TToken_string::destroy(int n)
 {
   if (_last == -2) return ;

   if (n < 0)
   {
     char* s = strrchr(chars(), _separator);
     if (s != NULL) *s = '\0';
   }
   else
   {
     int sep = 0;
     char* s;
     for (s = data(); sep < n && *s; s++)
       if (*s == _separator) sep++;

     if (sep >= n)
     {
       const char* p = strchr(s, _separator);
       *s = '\0';
       if (p != NULL) strcat(s,  p+1);
     }
   }
   restart();  
 }

 TToken_string& TToken_string::tokenize(const char* sp, const char* sep)
 {
   set("");
   const char* s = sp;

   for(const char* tk = strpbrk(s, sep); *s; tk = strpbrk(s, sep))
     {      
       if (tk == NULL)
	 {
	   add(s);
	   break;
	 }
       else 
	 {
	   add( CString( CString(s).mid(0, tk-s)) );
	   s = ++tk;
	   // go to start of next token
	   while (*s && strchr(sep, *s) != NULL)
	     s++;
	 }
     }

  restart();
  return *this;
}

///////////////////////////////////////////////////////////
// Paragraph string
///////////////////////////////////////////////////////////

TParagraph_string::TParagraph_string(const char* s, int width)
: TToken_string(s, '|'), _width(width)
{ tokenize(); }

const TString& TParagraph_string::operator =(const char* s)
{
  TToken_string::operator=(s);
  tokenize();
  return *this;
}

void TParagraph_string::tokenize()
{
  const char* s = chars(); 
  int start = 0;
  for (int end = start+_width; end < size(); end = start+_width)
  {
    int i;
    for (i = end; i >= start; i--)
      if (isspace(s[i])) break;

    if (i < start)
    {
      insert("|", end);
      start = end+1;
    }
    else
    {
      (*this)(i) = '|';
      start = i+1;
    }
  }
}


//----------------------------------------------------------------------------------------
//						TNamedObject
//----------------------------------------------------------------------------------------

int TNamedObject::Register(Pix p, int index) {
  if( index >= 0 ) {
		if( fP == NULL ) { fP = new PixVec(10); }
    int s = fP->capacity();
    if( index >= s ) { fP->resize( index + 10 ); fP->fill(0,s); }
    (*fP)[index] = p;
  }
  return index;
}

Pix TNamedObject::GetListPix(int listIndex) { return ( fP == NULL ) ? NULL : fP->elem(listIndex); }

void TNamedObject::name_break(const char* name) const { 
	if( SName() == name ) { 
		sprintf(gMsgStr," %s: Name Stop", Name()); gPrintScreen(); 
		Env::debugger_stop_here(); 
	} 
} 

