#ifdef HAS_GRASS_4_2
#include "gis.h"
#endif
#include <stdio.h>

/* grass_init()
 * this rouinte initializes the GRASS library
 * this routine should be called at least once. It can
 * be called more than once (but only the first call does anything)
 */

void
grass_init()
{
#ifdef HAS_GRASS_4_2
    static first = 1;
    if(first)
    {
	G_gisinit ("");
	first = 0;
    }
#endif
}

/* grass_get_region (struct Cell_head *region)
 *  gets the current (in memory) GRASS region
 *  if no 'in memory' region has been set,
 *  then this routine * reads the GRASS region from
 *  the GRASS database and sets the 'in memory' region
 *  region.rows, region.cols
 *     are the number of rows and columns in the region
 *  region.north, region.east, region.west, region.south
 *     are the spatial extent of the region (ie coordinates)
 */
 
#ifdef HAS_GRASS_4_2
 
void
grass_get_region(region)
    struct Cell_head *region;
{
    grass_init();
    G_get_set_window (region);
}

void
grass_set_region(region)
    struct Cell_head *region;
{
    grass_init();
    G_set_window (region);
}

#endif

void grass_set_region_rows_and_cols (nrows, ncols)
    int nrows, ncols;
{
#ifdef HAS_GRASS_4_2
    struct Cell_head region;

    grass_get_region (&region);
    region.rows = nrows;
    region.cols = ncols;
    G_adjust_Cell_head (&region, 1, 1);
    G_set_window (&region);
#endif
}


/* grass_read_float_raster_map (char *name, float *data, float *nv)
 *
 *  reads a GRASS map into the data array. The data array must be
 *  the size of the grass region (number of rows and cols) and
 *  must be allocated by the caller data = calloc (nrows *ncols, sizeof(float))
 *  nv is a point to a bit pattern that represents null values. GRASS
 *  has the concept of null values and these null values will be replaced
 *  by the nv bit pattern
 */

grass_read_float_raster_map (name, data, nv)
    char *name;
    float *data;
    float *nv;
{
    int nrows, ncols;
    char *mapset;
    int fd, row, col, rv=0;
#ifdef HAS_GRASS_4_2

		struct Cell_head region;
    grass_get_region (&region);
		nrows = region.rows;
    ncols = region.cols;

    mapset = G_find_cell (name,"");
    if (!mapset)
    {
			fprintf (stderr, "\nwarning: cant find mapset for GRASS map <%s>", name);
			return -1;
    } else {
			fprintf (stderr, "\ninfo: found GRASS mapset < %s : %s > ", mapset, name );    
    }

    fd = G_open_cell_old (name, mapset);
    if (fd < 0) {
			fprintf (stderr, "\nwarning: can't open GRASS map %s:%s", mapset, name );
			return -1;
		}

    for (row = 0; row < nrows; row++, data += ncols)
    {
			if (G_get_f_raster_row (fd, data, row) < 0) break;
			for (col = 0; col < ncols; col++)
			{
				 if (G_is_f_null_value(&data[col]))
				 memcpy (&data[col], nv, sizeof(float));
			}
    }
    G_close_cell(fd);
    if (row < nrows) { rv = -1; }
#endif
		fprintf (stderr, "\ninfo: Read %d rows from GRASS map <%s>", row, name );
    return rv;
}


grass_read_int_raster_map (name, data, nv)
    char *name;
    int *data;
    int *nv;
{
    int nrows, ncols;
    char *mapset;
    int fd, row, col, rv = 0;
#ifdef HAS_GRASS_4_2
    struct Cell_head region;
    grass_get_region(&region);
    nrows = region.rows;
    ncols = region.cols;

    mapset = G_find_cell (name,"");
    if (!mapset)
    {
			fprintf (stderr, "\nwarning: cant find mapset for GRASS map <%s>", name);
			return -1;
    } else {
			fprintf (stderr, "\ninfo: found GRASS mapset < %s : %s > ", mapset, name );    
    }

    fd = G_open_cell_old (name, mapset);
    if (fd < 0) {
			fprintf (stderr, "\nwarning: can't open GRASS map %s:%s", mapset, name );
			return -1;
		} 

    for (row = 0; row < nrows; row++, data += ncols)
    {
			if (G_get_c_raster_row (fd, data, row) < 0) break;
			for (col = 0; col < ncols; col++) 	{
					if (G_is_c_null_value(&data[col]))
				memcpy (&data[col], nv, sizeof(int));
			}
    }
    G_close_cell(fd);
    if (row < nrows) rv = -1;
#endif
		fprintf (stderr, "\ninfo: Read %d rows from GRASS map <%s>", row, name );
    return rv;
}

/* grass_write_float_raster_map (char *name, float *data, float *nv)
 *
 *  writes a GRASS map from the data array. The data array must be
 *  the size of the grass region (number of rows and cols) and
 *  must be allocated by the caller data = calloc (nrows *ncols, sizeof(float))
 *  nv is a point to a bit pattern that represents null values. GRASS
 *  has the concept of null values and these null values will be replaced
 *  by the nv bit pattern. If the nv point is non null, then this bit pattern
 *  in the data array will be written as GRASS null values.
 */

grass_write_float_raster_map( name, data, nv)
    char* name;
    float *data;
    float *nv;
{
    int fd, row, col;
    int nrows, ncols;
#ifdef HAS_GRASS_4_2
    struct Cell_head region;
    grass_get_region(&region);
    nrows = region.rows;
    ncols = region.cols;

    fd = G_open_fp_cell_new (name);
    if (fd < 0)
	return -1;

    for (row = 0; row < nrows; row++, data += ncols)
    {
	for (col = 0; col < ncols; col++)
	{
	    if (nv && memcmp (&data[col], nv, sizeof(float)) == 0)
		G_set_f_null_value (&data[col],1);
	}

	if (G_put_f_raster_row (fd, data) < 0)
	    break;

	for (col = 0; col < ncols; col++)
	{
	    if (nv && G_is_f_null_value(&data[col]))
		memcpy (&data[col], nv, sizeof(float));
	}
    }

    if (row < nrows)
    {
	G_unopen_cell(fd); 
	return -1;
    }

    G_close_cell(fd);
#endif
    return 0;
}

grass_write_int_raster_map(name, data, nv)
    char* name;
    int *data;
    int *nv;
{
    int fd, row, col;
    int nrows, ncols; 
      
#ifdef HAS_GRASS_4_2

    struct Cell_head region;
    grass_get_region(&region);
    nrows = region.rows;
    ncols = region.cols;

    fd = G_open_cell_new (name);
    if (fd < 0)
	return -1;

    for (row = 0; row < nrows; row++, data += ncols)
    {
	for (col = 0; col < ncols; col++)
	{
	    if (nv && memcmp (&data[col], nv, sizeof(int)) == 0)
		G_set_c_null_value (&data[col],1);
	}

	if (G_put_c_raster_row (fd, data) < 0)
	    break;

	for (col = 0; col < ncols; col++)
	{
	    if (nv && G_is_c_null_value(&data[col]))
		memcpy (&data[col], nv, sizeof(int));
	}
    }

    if (row < nrows)
    {
	G_unopen_cell(fd); 
	return -1;
    }

    G_close_cell(fd);
#endif
    return 0;
}



