#include "IntPixHashMap.h"


Pix intPixMap::seek(int  item)
{
  Pix i;
  for ( i = first(); i != 0 && ( key(i) != item ); next(i));
  return i;
}

int intPixMap::owns(Pix idx)
{
  if (idx == 0) return 0;
  for (Pix i = first(); i; next(i)) if (i == idx) return 1;
  return 0;
}

void intPixMap::clear()
{
  Pix i = first(); 
  while (i != 0)
  {
    del(key(i));
    i = first();
  }
}

int intPixMap::contains (int  item)
{
  return seek(item) != 0;
}


void intPixMap::error(const char* msg)
{
  gPrintErr( msg );
}


/* codes for status fields */

#define EMPTYCELL   0
#define DELETEDCELL 1
#define VALIDCELL   2
#define MARKEDCELL  3


intPixVHMap::intPixVHMap(Pix  dflt, unsigned int sz)
     :intPixMap(dflt)
{
  tab = new int[size = sz];
  cont = new Pix[size];
  status = new char[size];
  for (unsigned int i = 0; i < size; ++i) status[i] = EMPTYCELL;
}

intPixVHMap::intPixVHMap(intPixVHMap& a) : intPixMap(a.def)
{
  tab = new int[size = a.size];
  cont = new Pix[size];
  status = new char[size];
  for (unsigned int i = 0; i < size; ++i) status[i] = EMPTYCELL;
  count = 0;
  for (Pix p = a.first(); p; a.next(p)) (*this)[a.key(p)] = a.contents(p);
}


/* 
 * hashing method: double hash based on high bits of hash fct,
 * followed by linear probe. Can't do too much better if table
 * sizes not constrained to be prime.
*/


static inline unsigned int doublehashinc(unsigned int h, unsigned int s)
{
  unsigned int dh =  ((h / s) % s);
  return (dh > 1)? dh : 1;
}

static inline int intHASH(int key) { return key; }
static inline int intEQ( int key1, int key2 ) { return (key1 == key2); }
static inline int HASHTABLE_TOO_CROWDED( int count, int size) {
	return ( (((float)count)/size) > 0.9 );	
}

Pix intPixVHMap::seek(int  key)
{
  unsigned int hashval = intHASH(key);
  unsigned int h = hashval % size;
  for (unsigned int i = 0; i <= size; ++i)
  {
    if (status[h] == EMPTYCELL)
      return 0;
    else if ( (status[h] >= VALIDCELL) && ( key == tab[h] ) )
      return Pix(&tab[h]);
    if (i == 0)
      h = (h + doublehashinc(hashval, size)) % size;
    else if (++h >= size)
      h -= size;
  }
  return 0;
}

const char* intPixVHMap::text_representation() {
	tbuffer = "";
  for (unsigned int pos = 0; pos < size; ++pos) {
    if (status[pos] >= VALIDCELL) {
			int key = tab[pos];
			CString* name =  (CString*)(cont[pos]);
			tbuffer.appendIndex(key); 
			tbuffer += ":"; 
			tbuffer += (*name); 
			tbuffer += ",";
		}
 }
 return tbuffer.chars();
}    

Pix& intPixVHMap::operator [](int  item)
{
  if (HASHTABLE_TOO_CROWDED(count, size))  resize();

  unsigned int bestspot = size;
  unsigned int hashval = intHASH(item);
  unsigned int h = hashval % size;
  for (unsigned int i = 0; i <= size; ++i) {
    if (status[h] == EMPTYCELL) {
      ++count;
      if (bestspot >= size) bestspot = h;
      tab[bestspot] = item;
      status[bestspot] = VALIDCELL;
      cont[bestspot] = def;
      return cont[bestspot];
    }
    else if (status[h] == DELETEDCELL) {
      if (bestspot >= size) bestspot = h;
    }
    else if ( tab[h] == item ) return cont[h];

    if (i == 0)
      h = (h + doublehashinc(hashval, size)) % size;
    else if (++h >= size)
      h -= size;
  }

  ++count;
  status[bestspot] = VALIDCELL;
  tab[bestspot] = item;
  cont[bestspot] = def;
  return cont[bestspot];
}


void intPixVHMap::del(int  key)
{
  unsigned int hashval = intHASH(key);
  unsigned int h = hashval % size;
  for (unsigned int i = 0; i <= size; ++i)
  {
    if (status[h] == EMPTYCELL)
      return;
    else if (status[h] >= VALIDCELL && intEQ(key, tab[h]))
    {
      status[h] = DELETEDCELL;
      --count;
      return;
    }
    if (i == 0)
      h = (h + doublehashinc(hashval, size)) % size;
    else if (++h >= size)
      h -= size;
  }
}


void intPixVHMap::clear()
{
  for (unsigned int i = 0; i < size; ++i) status[i] = EMPTYCELL;
  count = 0;
}

void intPixVHMap::resize(unsigned int newsize)
{
  if (newsize <= count)
  {
    newsize = INTVHMAP_DEFAULT_INITIAL_CAPACITY;
    while (HASHTABLE_TOO_CROWDED(count, newsize)) newsize <<= 1;
  }
  int* oldtab = tab;
  Pix* oldcont = cont;
  char* oldstatus = status;
  unsigned int oldsize = size;
  tab = new int[size = newsize];
  cont = new Pix[size];
  status = new char[size];
  unsigned int i;
  for (i = 0; i < size; ++i) status[i] = EMPTYCELL;
  count = 0;
  for (i = 0; i < oldsize; ++i) 
    if (oldstatus[i] >= VALIDCELL) 
      (*this)[oldtab[i]] = oldcont[i];
  delete [] oldtab;
  delete [] oldcont;
  delete [] oldstatus;
}

Pix intPixVHMap::first()
{
  for (unsigned int pos = 0; pos < size; ++pos)
    if (status[pos] >= VALIDCELL) return Pix(&tab[pos]);
  return 0;
}

void intPixVHMap::next(Pix& i)
{
  if (i == 0) return;
  unsigned int pos = (((int*)i) - tab) + 1;
  for ( ; pos < size; ++pos)
    if (status[pos] >= VALIDCELL)
    {
      i = Pix(&tab[pos]);
      return;
    }
  i = 0;
}
  

int intPixVHMap::OK()
{
  int v = tab != 0;
  v &= status != 0;
  int n = 0;
  for (unsigned int i = 0; i < size; ++i) 
  {
    if (status[i] >= VALIDCELL) ++n;
    else if (status[i] != DELETEDCELL && status[i] != EMPTYCELL)
      v = 0;
  }
  v &= n == count;
  if (!v) error("invariant failure");
  return v;
}


void intPixVHMap::mark(Pix i, int index ) {
  if (i == 0) error("null Pix");
  if( (index<0) || ( index > 255-MARKEDCELL ) ) error("illegal index");
  status[((int*)i) - tab] = MARKEDCELL+index;
}

int intPixVHMap::is_marked(Pix i, int index ) {
  if (i == 0) error("null Pix");
  char s = status[((int*)i) - tab];
  if( s < VALIDCELL ) { error("invalid cell"); return -1; }
  return ( s == (MARKEDCELL+index) );
}
