//----------------------------------------------------------------------------------------
  //	MsgBuff.h
  //	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
  //	Change History:
  //----------------------------------------------------------------------------------------
  
#ifndef __MsgBuff__
#define __MsgBuff__

#ifdef USE_MPI

#include "mpi.h"
#include "Globals.h"
#include "Environ.h"
extern "C" {
#ifdef HAS_HDF
#include "mfhdf.h"
#else
#include "sme_types.h"
#endif
}

enum TMsgBufInfo { kMB_NewComm };

class TMsgBuff {
  
 private:
  
  byte* fTemp;  
  int fTempSize;
  int fMemPtr;
  int fDataSize;
  MPI_Comm fComm;
  byte fInfo[4];
  
 public:
  
  inline TMsgBuff( int initBuff = 1000, MPI_Comm comm = 0 ) { 
    fTemp = new byte[initBuff]; 
    fTempSize = initBuff; 
    fMemPtr = fDataSize = 0; 
    memset(fInfo,0,4);
    fComm = comm;
    if ( fComm == 0 ) { 	
			if( GetComm(&fComm,0) ) { 
				gPrintErr("Error in TMsgBuff"); 
			}   
			fInfo[kMB_NewComm] = 1; 
		}
  }

  inline ~TMsgBuff() { 
    delete[] fTemp; fTempSize = 0; fMemPtr = 0; fDataSize = 0;
    if(fInfo[kMB_NewComm]) MPI_Comm_free(&fComm); 
  }

  inline void Clear() { fMemPtr = 0; fDataSize = 0; }	
	inline void ReSet() { fMemPtr = 0; }	
 inline int DataSize() { return fDataSize;}	
  inline void CheckBufferSize(int dataSize, Bool CopyData = True ) { if( dataSize > fTempSize ) ReAllocBuff(dataSize,CopyData); }
  void ReAllocBuff(int dataSize, Bool CopyData );
  inline byte* Data(int offset) { return fTemp+offset; }
  inline int IntData(int index, int offset) { return ((int*)(fTemp+offset))[index]; }

  void UnPack( void* data, int nItems, MPI_Datatype type );
  int Pack( const void* data, int nItems, MPI_Datatype type); 

  inline int PackInt(const int* data, int nItems) { return Pack( data, nItems, MPI_INT ); }
  inline int PackInt(int idata) { return Pack( &idata, 1, MPI_INT ); } 
  inline int PackInt32(int32 idata) { 
		return Pack( &idata, 4, MPI_BYTE ); 
	} 
  
  inline int PackFloat(const float* data, int nItems) { return Pack( data, nItems, MPI_FLOAT ); } 
  inline int PackFloat( const float data ) { return Pack( &data, 1, MPI_FLOAT ); } 
  
  inline int PackBytes(const void* data, int dataSize) { return Pack( data, dataSize, MPI_BYTE ); } 
  inline int PackByte(const byte data ) { return Pack( &data, 1, MPI_BYTE ); } 
  inline int PackString(const char* data) { 
    int dataSize = strlen(data)+1;
    return Pack( data, dataSize, MPI_CHAR );	
  }

  inline void UnPackInt(int* data, int nItems) { UnPack( data, nItems, MPI_INT ); }
  inline int UnPackInt() { int data; UnPack( &data, 1, MPI_INT ); return data; } 
  inline int32 UnPackInt32() { 
		int32 data; UnPack( &data, 4, MPI_BYTE );  return data; 
	} 
  
  inline void UnPackFloat(float* data, int nItems) { UnPack( data, nItems, MPI_FLOAT ); } 
  inline float UnPackFloat() { float data; UnPack( &data, 1, MPI_FLOAT ); return data; } 
  
  inline void UnPackBytes(void* data, int dataSize) { UnPack( data, dataSize, MPI_BYTE ); } 
  inline byte UnPackByte() { byte data; UnPack( &data, 1, MPI_BYTE ); return data; } 
  inline void UnPackString(char* data, int dataSize ) { UnPack( data, dataSize, MPI_CHAR ); } 

  void Send( int type, int dest );
  MPI_Request ISend( int type, int dest );
  void BCast( int rootProc=0 );
  MPI_Status Recv( int type, int src = MPI_ANY_SOURCE );
  MPI_Status RecvAppend( int type, int src = MPI_ANY_SOURCE );
  
  inline int GetComm(MPI_Comm* commP, int newC = 1) {
		int flag, err;
		err = MPI_Initialized(&flag);
		if( flag == 0 ) { sprintf(gMsgStr, "MPI Not Initialized! error= %d\n",err); gPrintErr(); return 1; }
		if( newC || fComm==0 ) { MPI_Comm_dup(MPI_COMM_WORLD,commP); }
		else *commP = fComm; 
		return 0;	
	}

};

#endif
#endif
