#ifdef __GNUG__
#pragma implementation
#endif
#include "PixPlex.h"

const int kInitialChunkSize = 100;


// IChunk support

void PixIChunk::error(const char* msg) const
{
  gPrintErr( msg );
}

void PixIChunk::index_error() const
{
  error("attempt to use invalid index");
}

void PixIChunk::empty_error() const
{
  error("invalid use of empty chunk");
}

void PixIChunk::full_error() const
{
  error("attempt to extend chunk beyond bounds");
}

PixIChunk:: ~PixIChunk() {}

PixIChunk::PixIChunk(Pix*     d,    
                     int      baseidx,
                     int      lowidx,
                     int      fenceidx,
                     int      topidx)
{
  if (d == 0 || baseidx > lowidx || lowidx > fenceidx || fenceidx > topidx)
    error("inconsistent specification");
  data = d;
  base = baseidx;
  low = lowidx;
  fence = fenceidx;
  top = topidx;
  nxt = prv = this;
}

void PixIChunk:: re_index(int lo)
{
  int delta = lo - low;
  base += delta;
  low += delta;
  fence += delta;
  top += delta;
}


void PixIChunk::clear(int lo)
{
  int s = top - base;
  low = base = fence = lo;
  top = base + s;
}

void PixIChunk::cleardown(int hi)
{
  int s = top - base;
  low = top = fence = hi;
  base = top - s;
}

int PixIChunk:: OK() const
{
  int v = data != 0;             // have some data
  v &= base <= low;              // ok, index-wise
  v &= low <= fence;
  v &= fence <= top;

  v &=  nxt->prv == this;      // and links are OK
  v &=  prv->nxt == this;
  if (!v) error("invariant failure");
  return(v);
}


// error handling


void PixPlex::error(const char* msg) const
{
  gPrintErr(msg);
}

void PixPlex::index_error() const
{
  error("attempt to access invalid index");
}

void PixPlex::empty_error() const
{
  error("attempted operation on empty plex");
}

void PixPlex::full_error() const
{
  error("attempt to increase size of plex past limit");
}

// generic plex ops

PixPlex:: ~PixPlex()
{
  invalidate();
}  


void PixPlex::append (const PixPlex& a)
{
  for (int i = a.low(); i < a.fence(); a.next(i)) add_high(a[i]);
}

void PixPlex::prepend (const PixPlex& a)
{
  for (int i = a.high(); i > a.ecnef(); a.prev(i)) add_low(a[i]);
}

void PixPlex::reverse()
{
  Pix tmp;
  int l = low();
  int h = high();
  while (l < h)
  {
    tmp = (*this)[l];
    elem(l) = (*this)[h];
    elem(h) = tmp;
    next(l);
    prev(h);
  }
}


void PixPlex::fill(const Pix  x)
{
  for (int i = lo; i < fnc; ++i) elem(i) = x;
}

void PixPlex::fill(const Pix  x, int low, int hi)
{
  for (int i = low; i <= hi; ++i)  elem(i) = x;
}


void PixPlex::del_chunk(PixIChunk* x)
{
  if (x != 0)
  {
    x->unlink();
    Pix* data = (Pix*)(x->invalidate());
    delete [] data;
    delete x;
  }
}


void PixPlex::invalidate()
{
  PixIChunk* t = hd;
  if (t != 0)
  {
    PixIChunk* tail = tl();
    while (t != tail)
    {
      PixIChunk* nxt = t->next();
      del_chunk(t);
      t = nxt;
    } 
    del_chunk(t);
    hd = 0;
  }
}

int PixPlex::reset_low(int l)
{
  int old = lo;
  int diff = l - lo;
  if (diff != 0)
  {
    lo += diff;
    fnc += diff;
    PixIChunk* t = hd;
    do
    {
      t->re_index(t->low_index() + diff);
      t = t->next();
    } while (t != hd);
  }
  return old;
}





PixXPlex:: PixXPlex()
{
  lo = fnc = 0;
  csize = kInitialChunkSize;
  Pix* data = new Pix[csize];
  set_cache(new PixIChunk(data,  lo, lo, fnc, lo+csize));
  hd = ch;
}

PixXPlex:: PixXPlex(int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = 0;
  if (chunksize > 0)
  {
    csize = chunksize;
    Pix* data = new Pix[csize];
    set_cache(new PixIChunk(data,  lo, lo, fnc, csize));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    Pix* data = new Pix[csize];
    set_cache(new PixIChunk(data,  chunksize, lo, fnc, fnc));
    hd = ch;
  }
}


PixXPlex:: PixXPlex(int l, int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = l;
  if (chunksize > 0)
  {
    csize = chunksize;
    Pix* data = new Pix[csize];
    set_cache(new PixIChunk(data,  lo, lo, fnc, csize+lo));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    Pix* data = new Pix[csize];
    set_cache(new PixIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
}

void PixXPlex::make_initial_chunks(int up)
{
  int need = fnc - lo;
  hd = 0;
  if (up)
  {
    int l = lo;
    do
    {
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      Pix* data = new Pix [csize];
      PixIChunk* h = new PixIChunk(data,  l, l, l+sz, l+csize);
      if (hd != 0)
        h->link_to_next(hd);
      else
        hd = h;
      l += sz;
      need -= sz;
    } while (need > 0);
  }
  else
  {
    int hi = fnc;
    do
    {
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      Pix* data = new Pix [csize];
      PixIChunk* h = new PixIChunk(data,  hi-csize, hi-sz, hi, hi);
      if (hd != 0)
        h->link_to_next(hd);
      hd = h;
      hi -= sz;
      need -= sz;
    } while (need > 0);
  }
  set_cache(hd);
}

PixXPlex:: PixXPlex(int l, int hi, const Pix  initval, int chunksize)
{
  lo = l;
  fnc = hi + 1;
  if (chunksize == 0)
  {
    csize = fnc - l;
    make_initial_chunks(1);
  }
  else if (chunksize < 0)
  {
    csize = -chunksize;
    make_initial_chunks(0);
  }
  else
  {
    csize = chunksize;
    make_initial_chunks(1);
  }
  fill(initval);
}

PixXPlex::PixXPlex(const PixXPlex& a)
{
  lo = a.lo;
  fnc = a.fnc;
  csize = a.csize;
  make_initial_chunks();
  for (int i = a.low(); i < a.fence(); a.next(i))  elem(i)  = a[i];
}

void PixXPlex::operator= (const PixXPlex& a)
{
  if (&a != this)
  {
    invalidate();
    lo = a.lo;
    fnc = a.fnc;
    csize = a.csize;
    make_initial_chunks();
    for (int i = a.low(); i < a.fence(); a.next(i))  elem(i)  = a[i];
  }
}


void PixXPlex::cache(int idx) const
{
  const PixIChunk* tail = tl();
  const PixIChunk* t = ch;
  while (idx >= t->fence_index())
  {
    if (t == tail) index_error();
    t = (t->next());
  }
  while (idx < t->low_index())
  {
    if (t == hd) index_error();
    t = (t->prev());
  }
  set_cache(t);
}


void PixXPlex::cache(const Pix* p) const
{
  const PixIChunk* old = ch;
  const PixIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) index_error();
  }
  set_cache(t);
}

int PixXPlex::owns(Pix px) const
{
  Pix* p = (Pix*)px;
  const PixIChunk* old = ch;
  const PixIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) { set_cache(t); return 0; }
  }
  set_cache(t);
  return 1;
}


Pix* PixXPlex::dosucc(const Pix* p) const
{
  if (p == 0) return 0;
  const PixIChunk* old = ch;
  const PixIChunk* t = ch;
 
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old)  return 0;
  }
  int i = t->index_of(p) + 1;
  if (i >= fnc) return 0;
  if (i >= t->fence_index()) t = (t->next());
  set_cache(t);
  return (t->pointer_to(i));
}

Pix* PixXPlex::dopred(const Pix* p) const
{
  if (p == 0) return 0;
  const PixIChunk* old = ch;
  const PixIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->prev());
    if (t == old) return 0;
  }
  int i = t->index_of(p) - 1;
  if (i < lo) return 0;
  if (i < t->low_index()) t = (t->prev());
  set_cache(t);
  return (t->pointer_to(i));
}


int PixXPlex::add_high(const Pix  elem)
{
  PixIChunk* t = tl();
  if (!t->can_grow_high())
  {
    if (t->PixIChunk::empty() && one_chunk())
      t->clear(fnc);
    else
    {
      Pix* data = new Pix [csize];
      t = (new PixIChunk(data,  fnc, fnc, fnc,fnc+csize));
      t->link_to_prev(tl());
    }
  }
  *((t->PixIChunk::grow_high())) = elem;
  set_cache(t);
  return fnc++;
}

int PixXPlex::del_high ()
{
  if (empty()) empty_error();
  PixIChunk* t = tl();
  t->PixIChunk::shrink_high();
  if (t->PixIChunk::empty() && !one_chunk())
  {
    PixIChunk* pred = t->prev();
    del_chunk(t);
    t = pred;
  }
  set_cache(t);
  return --fnc - 1;
}

int PixXPlex::add_low (const Pix  elem)
{
  PixIChunk* t = hd;
  if (!t->can_grow_low())
  {
    if (t->PixIChunk::empty() && one_chunk())
      t->cleardown(lo);
    else
    {
      Pix* data = new Pix [csize];
      hd = new PixIChunk(data,  lo-csize, lo, lo, lo);
      hd->link_to_next(t);
      t = hd;
    }
  }
  *((t->PixIChunk::grow_low())) = elem;
  set_cache(t);
  return --lo;
}


int PixXPlex::del_low ()
{
  if (empty()) empty_error();
  PixIChunk* t = hd;
  t->PixIChunk::shrink_low();
  if (t->PixIChunk::empty() && !one_chunk())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
  }
  set_cache(t);
  return ++lo;
}

void PixXPlex::reverse()
{
  Pix tmp;
  int l = lo;
  int h = fnc - 1;
  PixIChunk* loch = hd;
  PixIChunk* hich = tl();
  while (l < h)
  {
    Pix* lptr = loch->pointer_to(l);
    Pix* hptr = hich->pointer_to(h);
    tmp = *lptr;
    *lptr = *hptr;
    *hptr = tmp;
    if (++l >= loch->fence_index()) loch = loch->next();
    if (--h < hich->low_index()) hich = hich->prev();
  }
}

void PixXPlex::fill(const Pix  x)
{
  for (int i = lo; i < fnc; ++i)  elem(i)  = x;
}

void PixXPlex::fill(const Pix  x, int l, int hi)
{
  for (int i = l; i <= hi; ++i)  elem(i) = x;
}


void PixXPlex::clear()
{
  if (fnc != lo)
  {
    PixIChunk* t = tl();
    while (t != hd)
    {
      PixIChunk* prv = t->prev();
      del_chunk(t);
      t = prv;
    }
    t->PixIChunk::clear(lo);
    set_cache(t);
    fnc = lo;
  }
}


int PixXPlex::OK () const
{
  int v = hd != 0 && ch != 0;     // at least one chunk

  v &= fnc == tl()->fence_index();// last chunk fence == plex fence
  v &= lo == ((hd))->PixIChunk::low_index();    // first lo == plex lo

// loop for others:
  int found_ch = 0;                   // to make sure ch is in list;
  const PixIChunk* t = (hd);
  for (;;)
  {
    if (t == ch) ++found_ch;
    v &= t->PixIChunk::OK();              // each chunk is OK
    if (t == tl())
      break;
    else                              // and has indices contiguous to succ
    {
      v &= t->top_index() == t->next()->base_index();
      if (t != hd)                  // internal chunks full
      {
        v &= !t->empty();
        v &= !t->can_grow_low();
        v &= !t->can_grow_high();
      }
      t = t->next();
    }
  }
  v &= found_ch == 1;
  if (!v) error("invariant failure");
  return v;
}

