#ifndef _PixXPlex_h
#define _PixXPlex_h 1

#include "Environ.h"

// Plexes are made out of PixIChunks

#include <stddef.h>

class PixIChunk
{
//public: // kludge until C++ `protected' policies settled
protected:      

  Pix*           data;           // data, from client

  int            base;           // lowest possible index
  int            low;            // lowest valid index
  int            fence;          // highest valid index + 1
  int            top;            // highest possible index + 1

  PixIChunk*     nxt;            // circular links
  PixIChunk*     prv;

public:

// constructors

                 PixIChunk(Pix*     d,       // ptr to array of elements
                        int      base_idx, // initial indices
                        int      low_idx,  
                        int      fence_idx,
                        int      top_idx);

  virtual       ~PixIChunk();

// status reports

  int            size() const;    // number of slots

  inline virtual int    empty() const ;
  inline virtual int    full() const ; 

  int            can_grow_high () const ;  // there is space to add data
  int            can_grow_low () const;        
  
  int            base_index() const;   // lowest possible index;
  int            low_index() const;    // lowest actual index;
  inline virtual int    first_index() const;  // lowest valid index or fence if none
  inline virtual int    last_index() const;   // highest valid index or low-1 if none
  int            fence_index() const;  // highest actual index + 1
  int            top_index() const;    // highest possible index + 1

// indexing conversion

  int            possible_index(int i) const; // i between base and top
  int            actual_index(int i) const;   // i between low and fence
  inline virtual int    valid_index(int i) const;    // i not deleted (mainly for mchunks)

  int            possible_pointer(const Pix* p) const; // same for ptr
  int            actual_pointer(const Pix* p) const; 
  inline virtual int    valid_pointer(const Pix* p) const; 

  Pix*           pointer_to(int i) const ;   // pointer to data indexed by i
                                      // caution: i is not checked for validity
  int            index_of(const Pix* p) const; // index of data pointed to by p
                                      // caution: p is not checked for validity

  inline virtual int    succ(int idx) const;     // next valid index or fence if none
  inline virtual int    pred(int idx) const;     // previous index or low - 1 if none

  inline virtual Pix*   first_pointer() const;   // pointer to first valid pos or 0
  inline virtual Pix*   last_pointer() const;    // pointer to first valid pos or 0
  inline virtual Pix*   succ(Pix*  p) const;     // next pointer or 0
  inline virtual Pix*   pred(Pix* p) const;     // previous pointer or 0

// modification

  inline virtual Pix*   grow_high ();      // return spot to add an element
  inline virtual Pix*   grow_low ();  

  inline virtual void   shrink_high ();    // logically delete top index
  inline virtual void   shrink_low ();     

  virtual void   clear(int lo);     // reset to empty ch with base = lo
  virtual void   cleardown(int hi); // reset to empty ch with top = hi
  void           re_index(int lo);  // re-index so lo is new low

// chunk traversal

  PixIChunk*     next() const;
  PixIChunk*     prev() const;

  void           link_to_prev(PixIChunk* prev);
  void           link_to_next(PixIChunk* next);
  void           unlink();

// state checks

  Pix*           invalidate();     // mark self as invalid; return data
                                   // for possible deletion

  virtual int    OK() const;             // representation invariant

  void   error(const char*) const;
  void   empty_error() const;
  void   full_error() const;
  void   index_error() const;
};

// PixPlex is a partly `abstract' class: few of the virtuals
// are implemented at the Plex level, only in the subclasses

class PixPlex
{
protected:      

  PixIChunk*       hd;          // a chunk holding the data
  int              lo;          // lowest  index
  int              fnc;         // highest index + 1
  int              csize;       // size of the chunk

  void             invalidate();              // mark so OK() is false
  void             del_chunk(PixIChunk*);        // delete a chunk

  PixIChunk*       tl() const;                // last chunk;
  int              one_chunk() const;         // true if hd == tl()

public:

// constructors, etc.

                    PixPlex();                  // no-op

  virtual           ~PixPlex();

  
// Access functions 
  
  virtual Pix&      elem(int idx) = 0; // access by index;
  virtual Pix&      elem (Pix p) = 0;   // access by Pix;

  virtual Pix&      high_elem () = 0;      // access high element
  virtual Pix&      low_elem () = 0;       // access low element

// read-only versions for const Plexes

  virtual Pix operator [] (int idx) const = 0; // access by index;
  virtual const Pix& operator () (Pix p) const = 0;   // access by Pix;

  virtual const Pix& high_element () const = 0;      // access high element
  virtual const Pix& low_element () const = 0;       // access low element


// Index functions

  virtual int       valid (int idx) const = 0;      // idx is an OK index

  virtual int       low() const = 0;         // lowest index or fence if none
  virtual int       high() const = 0;        // highest index or low-1 if none

  int               ecnef() const;         // low limit index (low-1)
  int               fence() const;         // high limit index (high+1)

  virtual void      prev(int& idx) const= 0; // set idx to preceding index
                                          // caution: pred may be out of bounds

  virtual void      next(int& idx) const = 0;       // set to next index
                                          // caution: succ may be out of bounds

  virtual Pix       first() const = 0;        // Pix to low element or 0
  virtual Pix       last() const = 0;         // Pix to high element or 0
  virtual void      prev(Pix& pix) const = 0;  // preceding pix or 0
  virtual void      next(Pix& pix) const = 0;  // next pix or 0
  virtual int       owns(Pix p) const = 0;     // p is an OK Pix

// index<->Pix 

  virtual int       Pix_to_index(Pix p) const = 0;   // get index via Pix
  virtual Pix       index_to_Pix(int idx) const = 0; // Pix via index

// Growth

  virtual int       add_high(const Pix  elem) =0;// add new element at high end
                                                // return new high

  virtual int       add_low(const Pix  elem) = 0;   // add new low element,
                                                // return new low

// Shrinkage

  virtual int       del_high() = 0;           // remove the element at high end
                                          // return new high
  virtual int       del_low() = 0;        // delete low element, return new lo

                                          // caution: del_low/high
                                          // does not necessarily 
                                          // immediately call Pix::~Pix


// operations on multiple elements

  virtual void      fill(const Pix  x);          // set all elements = x
  virtual void      fill(const Pix  x, int from, int to); // fill from to to
  virtual void      clear() = 0;                // reset to zero-sized Plex
  virtual int       reset_low(int newlow); // change low index,return old
  virtual void      reverse();                   // reverse in-place
  virtual void      append(const PixPlex& a);    // concatenate a copy
  virtual void      prepend(const PixPlex& a);   // prepend a copy

// status

  virtual int       can_add_high() const = 0;
  virtual int       can_add_low() const = 0;
  
  int               length () const;       // number of slots

  int               empty () const;        // is the plex empty?
  virtual int       full() const = 0;      // it it full?

  int               chunk_size() const;    // report chunk size;

  virtual int       OK() const = 0;        // representation invariant

  void		    error(const char* msg) const;
  void		    index_error() const;
  void		    empty_error() const;
  void		    full_error() const;
};


// PixIChunk ops

inline int PixIChunk:: size() const
{
  return top - base;
}


inline int PixIChunk:: base_index() const
{
  return base;
}

inline  int PixIChunk:: low_index() const
{
  return low;
}

inline  int  PixIChunk:: fence_index() const
{
  return fence;
}

inline  int  PixIChunk:: top_index() const
{
  return top;
}

inline  Pix* PixIChunk:: pointer_to(int i) const
{
  return &(data[i-base]);
}

inline  int  PixIChunk:: index_of(const Pix* p) const
{
  return (p - data) / sizeof(Pix) + base;
}

inline  int  PixIChunk:: possible_index(int i) const
{
  return i >= base && i < top;
}

inline  int  PixIChunk:: possible_pointer(const Pix* p) const
{
  return p >= data && p < &(data[top-base]);
}

inline  int  PixIChunk:: actual_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  PixIChunk:: actual_pointer(const Pix* p) const
{
  return p >= data && p < &(data[fence-base]);
}

inline  int  PixIChunk:: can_grow_high () const
{
  return fence < top;
}

inline  int  PixIChunk:: can_grow_low () const
{
  return base < low;
}

inline  Pix* PixIChunk:: invalidate()
{
  Pix* p = data;
  data = 0;
  return p;
}


inline PixIChunk* PixIChunk::prev() const
{
  return prv;
}

inline PixIChunk* PixIChunk::next() const
{
  return nxt;
}

inline void PixIChunk::link_to_prev(PixIChunk* prev)
{
  nxt = prev->nxt;
  prv = prev;
  nxt->prv = this;
  prv->nxt = this;
}

inline void PixIChunk::link_to_next(PixIChunk* next)
{
  prv = next->prv;
  nxt = next;
  nxt->prv = this;
  prv->nxt = this;
}

inline void PixIChunk::unlink()
{
  PixIChunk* n = nxt;
  PixIChunk* p = prv;
  n->prv = p;
  p->nxt = n;
  prv = nxt = this;
}

inline  int PixIChunk:: empty() const
{
  return low == fence;
}

inline  int  PixIChunk:: full() const
{
  return top - base == fence - low;
}

inline int PixIChunk:: first_index() const
{
  return (low == fence)? fence : low;
}

inline int PixIChunk:: last_index() const
{
  return (low == fence)? low - 1 : fence - 1;
}

inline  int  PixIChunk:: succ(int i) const
{
  return (i < low) ? low : i + 1;
}

inline  int  PixIChunk:: pred(int i) const
{
  return (i > fence) ? (fence - 1) : i - 1;
}

inline  int  PixIChunk:: valid_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  PixIChunk:: valid_pointer(const Pix* p) const
{
  return p >= &(data[low - base]) && p < &(data[fence - base]);
}

inline  Pix* PixIChunk:: grow_high ()
{
  if (!can_grow_high()) full_error();
  return &(data[fence++ - base]);
}

inline  Pix* PixIChunk:: grow_low ()
{
  if (!can_grow_low()) full_error();
  return &(data[--low - base]);
}

inline  void PixIChunk:: shrink_high ()
{
  if (empty()) empty_error();
  --fence;
}

inline  void PixIChunk:: shrink_low ()
{
  if (empty()) empty_error();
  ++low;
}

inline Pix* PixIChunk::first_pointer() const
{
  return (low == fence)? 0 : &(data[low - base]);
}

inline Pix* PixIChunk::last_pointer() const
{
  return (low == fence)? 0 : &(data[fence - base - 1]);
}

inline Pix* PixIChunk::succ(Pix* p) const
{
  return ((p+1) <  &(data[low - base]) || (p+1) >= &(data[fence - base])) ? 
    0 : (p+1);
}

inline Pix* PixIChunk::pred(Pix* p) const
{
  return ((p-1) <  &(data[low - base]) || (p-1) >= &(data[fence - base])) ? 
    0 : (p-1);
}


// generic Plex operations

inline PixPlex::PixPlex() {}

inline int PixPlex::chunk_size() const
{
  return csize;
}

inline  int PixPlex::ecnef () const
{
  return lo - 1;
}


inline  int PixPlex::fence () const
{
  return fnc;
}

inline int PixPlex::length () const
{
  return fnc - lo;
}

inline  int PixPlex::empty () const
{
  return fnc == lo;
}

inline PixIChunk* PixPlex::tl() const
{
  return hd->prev();
}

inline int PixPlex::one_chunk() const
{
  return hd == hd->prev();
}

class PixXPlex: public PixPlex
{
  PixIChunk*       ch;           // cached chunk

  void             make_initial_chunks(int up = 1);

  void             cache(int idx) const;
  void             cache(const Pix* p) const;

  Pix*             dopred(const Pix* p) const;
  Pix*             dosucc(const Pix* p) const;

  inline void             set_cache(const PixIChunk* t) const; // logically, 
                                               // not physically const
public:
                   PixXPlex();                 // set low = 0;
                                               // fence = 0;
                                               // csize = default

                   PixXPlex(int ch_size);      // low = 0; 
                                               // fence = 0;
                                               // csize = ch_size

                   PixXPlex(int lo,            // low = lo; 
                            int ch_size);      // fence=lo
                                               // csize = ch_size

                   PixXPlex(int lo,            // low = lo
                            int hi,            // fence = hi+1
                            const Pix  initval,// fill with initval,
                            int ch_size = 0);  // csize= ch_size
                                               // or fence-lo if 0

                   PixXPlex(const PixXPlex&);
  
  void             operator= (const PixXPlex&);

// virtuals


  inline Pix&             high_elem ();
  inline Pix&             low_elem ();
  
  inline const Pix&       high_element () const;
  inline const Pix&       low_element () const;

  inline Pix              first() const;
  inline Pix              last() const;
  inline void             prev(Pix& ptr) const;
  inline void             next(Pix& ptr) const;
  int              owns(Pix p) const;
  inline Pix&             elem (Pix p);
  inline const Pix&       operator () (Pix p) const;

  inline int              low() const; 
  inline int              high() const;
  inline int              valid(int idx) const;
  inline void             prev(int& idx) const;
  inline void             next(int& x) const;
  inline Pix&             elem (int index);
  inline Pix              operator [] (int index) const;
  inline int              find_elem ( Pix targ ) const; 
    
  inline int              Pix_to_index(Pix p) const;
  inline Pix              index_to_Pix(int idx) const;    

  inline int              can_add_high() const;
  inline int              can_add_low() const;
  inline int              full() const;

	inline void 		 grow_bounds( int index, Pix fill=0 ) { 
															while( fnc <= index ) add_high(fill); 
															while( lo > index ) add_low(fill);   }
	
  int              add_high(const Pix  elem);
  int              del_high ();
  int              add_low (const Pix  elem);
  int              del_low ();

  void             fill(const Pix  x);
  void             fill(const Pix  x, int from, int to);
  void             clear();
  void             reverse();
    
  int              OK () const; 

};


inline void PixXPlex::prev(int& idx) const
{
  --idx;
}

inline void PixXPlex::next(int& idx) const
{
  ++idx;
}

inline  int PixXPlex::full () const
{
  return 0;
}

inline int PixXPlex::can_add_high() const
{
  return 1;
}

inline int PixXPlex::can_add_low() const
{
  return 1;
}

inline  int PixXPlex::valid (int idx) const
{
  return idx >= lo && idx < fnc;
}

inline int PixXPlex::low() const
{
  return lo;
}

inline int PixXPlex::high() const
{
  return fnc - 1;
}

inline Pix& PixXPlex:: elem (int idx)
{
	grow_bounds( idx );
  if (!ch->actual_index(idx)) cache(idx);
  return *(ch->pointer_to(idx));
}

inline Pix PixXPlex:: operator [] (int idx) const
{
  if (!ch->actual_index(idx)) cache(idx);
  return *((Pix*)(ch->pointer_to(idx)));
}

inline int PixXPlex:: find_elem ( Pix targ ) const {
	for( int idx = lo; idx < fnc; idx++ ) {
		if (!ch->actual_index(idx)) cache(idx);
		if( *((Pix*)(ch->pointer_to(idx))) == targ ) return idx;
	}
	return -1;
}

inline  Pix& PixXPlex::low_elem ()
{
  if (empty()) index_error();
  return *(hd->pointer_to(lo));
}

inline  const Pix& PixXPlex::low_element () const
{
  if (empty()) index_error();
  return *((const Pix*)(hd->pointer_to(lo)));
}

inline  Pix& PixXPlex::high_elem ()
{
  if (empty()) index_error();
  return *(tl()->pointer_to(fnc - 1));
}

inline const Pix& PixXPlex::high_element () const
{
  if (empty()) index_error();
  return *((const Pix*)(tl()->pointer_to(fnc - 1)));
}

inline  int PixXPlex::Pix_to_index(Pix px) const
{
  Pix* p = (Pix*)px;
  if (!ch->actual_pointer(p)) cache(p);
  return ch->index_of(p);
}

inline  Pix PixXPlex::index_to_Pix(int idx) const
{
  if (!ch->actual_index(idx)) cache(idx);
  return (Pix)(ch->pointer_to(idx));
}

inline Pix PixXPlex::first() const
{
  return Pix(hd->PixIChunk::first_pointer());
}

inline Pix PixXPlex::last() const
{
  return Pix(tl()->PixIChunk::last_pointer());
}

inline void PixXPlex::prev(Pix& p) const
{
  Pix q = Pix(ch->PixIChunk::pred((Pix*) p));
  p = (q == 0)? Pix(dopred((const Pix*) p)) : q;
}

inline void PixXPlex::next(Pix& p) const
{
  Pix q = Pix(ch->PixIChunk::succ((Pix*) p));
  p = (q == 0)? Pix(dosucc((const Pix*)p)) : q;
}

inline Pix& PixXPlex:: elem (Pix p)
{
  return *((Pix*)p);
}

inline const Pix& PixXPlex:: operator () (Pix p) const
{
  return *((const Pix*)p);
}

inline void PixXPlex::set_cache(const PixIChunk* t) const
{
  ((PixXPlex*)(this))->ch = (PixIChunk*)t;
}


class PixStack
{
public:
                        PixStack() { }
  virtual              ~PixStack() {;}

  virtual void          push(Pix  item) = 0;
  virtual Pix           pop() = 0;
  virtual Pix&          top() = 0;               
  virtual void          del_top() = 0;

  virtual int           empty() = 0;
  virtual int           full() = 0;
  virtual int           length() = 0;

  virtual void          clear() = 0;

  void                  error(const char* msg) { gPrintErr(msg); }
  virtual int           OK() = 0;
};


class PixXPStack : public PixStack
{
  PixXPlex      p;

public:
                PixXPStack( int chunksize = 100 );
                PixXPStack(const PixXPStack& s);
  inline               ~PixXPStack();

  void          operator = (const PixXPStack&);

  inline void          push(Pix  item);
  inline Pix           pop();
  inline Pix&          top();               
  inline void          del_top();

  inline int           empty();
  inline int           full();
  inline int           length();

  inline void          clear();

  inline int           OK();

};


inline PixXPStack::PixXPStack(int chunksize) 
     : p(chunksize) {}
inline PixXPStack::PixXPStack(const PixXPStack& s) : p(s.p) {}

inline PixXPStack::~PixXPStack() {}

inline void PixXPStack::push(Pix item)
{
  p.add_high(item);
}

inline Pix PixXPStack::pop()
{
  Pix res = p.high_element();
  p.del_high();
  return res;
}

inline Pix& PixXPStack::top()
{
  return p.high_elem();
}

inline void PixXPStack::del_top()
{
  p.del_high();
}

inline void PixXPStack::operator =(const PixXPStack& s)
{
  p.operator = (s.p);
}

inline int PixXPStack::empty() 
{
  return p.empty();
}

inline int PixXPStack::full() 
{
  return p.full();
}

inline int PixXPStack::length() 
{
  return p.length();
}

inline int PixXPStack::OK() 
{
  return p.OK();
}

inline void PixXPStack::clear() 
{
  p.clear();
}


#endif
