#ifdef __GNUG__
#pragma implementation
#endif

#include "TimePlex.h"
#include "Environ.h"
// #include <builtin.h>

float TimeSlice::fGarbage = 0.0;
float TimeSlice::fdGarbage = 0.0;

const int kDefaultInitialCapacity = 64;

floatVec TTemporalTimeSliceRPlex::fTmpBuff;
floatVec TSpatialTimeSliceRPlex::fTmpBuff;
TTemporalTimeSlice* TTemporalTimeSliceRPlex::fGarbage = new TTemporalTimeSlice( -1.0, 0.0 );


/*
*************************************************************************
*									*
* TSpatialTimeSliceIChunk:      					*
*									*
*************************************************************************
*/

// IChunk support

void TSpatialTimeSliceIChunk::error(const char* msg) const
{
  gPrintErr(msg);
}

void TSpatialTimeSliceIChunk::index_error() const
{
  error("attempt to use invalid index");
}

void TSpatialTimeSliceIChunk::empty_error() const
{
  error("invalid use of empty chunk");
}

void TSpatialTimeSliceIChunk::full_error() const
{
  error("attempt to extend chunk beyond bounds");
}

TSpatialTimeSliceIChunk:: ~TSpatialTimeSliceIChunk() {}

TSpatialTimeSliceIChunk::TSpatialTimeSliceIChunk(TSpatialTimeSlice*     d,    
                     int      baseidx,
                     int      lowidx,
                     int      fenceidx,
                     int      topidx)
{
  if (d == 0 || baseidx > lowidx || lowidx > fenceidx || fenceidx > topidx)
    error("inconsistent specification");
  data = d;
  base = baseidx;
  low = lowidx;
  fence = fenceidx;
  top = topidx;
  nxt = prv = this;
}

void TSpatialTimeSliceIChunk:: re_index(int lo)
{
  int delta = lo - low;
  base += delta;
  low += delta;
  fence += delta;
  top += delta;
}


void TSpatialTimeSliceIChunk::clear(int lo)
{
  int s = top - base;
  low = base = fence = lo;
  top = base + s;
}

void TSpatialTimeSliceIChunk::cleardown(int hi)
{
  int s = top - base;
  low = top = fence = hi;
  base = top - s;
}

int TSpatialTimeSliceIChunk:: OK() const
{
  int v = data != 0;             // have some data
  v &= base <= low;              // ok, index-wise
  v &= low <= fence;
  v &= fence <= top;

  v &=  nxt->prv == this;      // and links are OK
  v &=  prv->nxt == this;
  if (!v) error("invariant failure");
  return(v);
}

/*
*************************************************************************
*									*
* TSpatialTimeSlicePlex:      						*
*									*
*************************************************************************
*/

// error handling


void TSpatialTimeSlicePlex::error(const char* msg) const
{
  gPrintErr( msg);
}

void TSpatialTimeSlicePlex::index_error() const
{
  error("attempt to access invalid index");
}

void TSpatialTimeSlicePlex::empty_error() const
{
  error("attempted operation on empty plex");
}

void TSpatialTimeSlicePlex::full_error() const
{
  error("attempt to increase size of plex past limit");
}

// generic plex ops

TSpatialTimeSlicePlex:: ~TSpatialTimeSlicePlex()
{
  invalidate();
}  


void TSpatialTimeSlicePlex::append (const TSpatialTimeSlicePlex& a)
{
  for (int i = a.low(); i < a.fence(); a.next(i)) add_high(a[i]);
}

void TSpatialTimeSlicePlex::prepend (const TSpatialTimeSlicePlex& a)
{
  for (int i = a.high(); i > a.ecnef(); a.prev(i)) add_low(a[i]);
}

void TSpatialTimeSlicePlex::reverse()
{
  TSpatialTimeSlice tmp;
  int l = low();
  int h = high();
  while (l < h)
  {
    tmp = (*this)[l];
    elem(l) = (*this)[h];
    elem(h) = tmp;
    next(l);
    prev(h);
  }
}


void TSpatialTimeSlicePlex::fill(const TSpatialTimeSlice& x)
{
  for (int i = lo; i < fnc; ++i) elem(i) = x;
}

void TSpatialTimeSlicePlex::fill(const TSpatialTimeSlice& x, int low, int hi)
{
  for (int i = low; i <= hi; ++i) elem(i) = x;
}


void TSpatialTimeSlicePlex::del_chunk(TSpatialTimeSliceIChunk* x)
{
  if (x != 0)
  {
    x->unlink();
    TSpatialTimeSlice* data = (TSpatialTimeSlice*)(x->invalidate());
    delete [] data;
    delete x;
  }
}


void TSpatialTimeSlicePlex::invalidate()
{
  TSpatialTimeSliceIChunk* t = hd;
  if (t != 0)
  {
    TSpatialTimeSliceIChunk* tail = tl();
    while (t != tail)
    {
      TSpatialTimeSliceIChunk* nxt = t->next();
      del_chunk(t);
      t = nxt;
    } 
    del_chunk(t);
    hd = 0;
  }
}

int TSpatialTimeSlicePlex::reset_low(int l)
{
  int old = lo;
  int diff = l - lo;
  if (diff != 0)
  {
    lo += diff;
    fnc += diff;
    TSpatialTimeSliceIChunk* t = hd;
    do
    {
      t->re_index(t->low_index() + diff);
      t = t->next();
    } while (t != hd);
  }
  return old;
}

/*
*************************************************************************
*									*
* TSpatialTimeSliceRPlex:      						*
*									*
*************************************************************************
*/

typedef TSpatialTimeSliceIChunk* _TSpatialTimeSliceIChunk_ptr;

TSpatialTimeSliceRPlex:: TSpatialTimeSliceRPlex()
{
  lo = fnc = 0;
  csize = kDefaultInitialCapacity;
  TSpatialTimeSlice* data = new TSpatialTimeSlice[csize];
  set_cache(new TSpatialTimeSliceIChunk(data,  lo, lo, fnc, lo+csize));
  hd = ch;
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _TSpatialTimeSliceIChunk_ptr[maxch];
  memset(chunks,0,sizeof(_TSpatialTimeSliceIChunk_ptr)*maxch);
  chunks[lch] = ch;
}

TSpatialTimeSliceRPlex:: TSpatialTimeSliceRPlex(int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = 0;
  if (chunksize > 0)
  {
    csize = chunksize;
    TSpatialTimeSlice* data = new TSpatialTimeSlice[csize];
    set_cache(new TSpatialTimeSliceIChunk(data,  lo, lo, fnc, csize+lo));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    TSpatialTimeSlice* data = new TSpatialTimeSlice[csize];
    set_cache(new TSpatialTimeSliceIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _TSpatialTimeSliceIChunk_ptr[maxch];
  memset(chunks,0,sizeof(_TSpatialTimeSliceIChunk_ptr)*maxch);
  chunks[lch] = ch;
}


TSpatialTimeSliceRPlex:: TSpatialTimeSliceRPlex(int l, int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = l;
  if (chunksize > 0)
  {
    csize = chunksize;
    TSpatialTimeSlice* data = new TSpatialTimeSlice[csize];
    set_cache(new TSpatialTimeSliceIChunk(data,  lo, lo, fnc, lo+csize));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    TSpatialTimeSlice* data = new TSpatialTimeSlice[csize];
    set_cache(new TSpatialTimeSliceIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _TSpatialTimeSliceIChunk_ptr[maxch];
  memset(chunks,0,sizeof(_TSpatialTimeSliceIChunk_ptr)*maxch);
  chunks[lch] = ch;
}

void TSpatialTimeSliceRPlex::make_initial_chunks(int up)
{
  int count = 0;
  int need = fnc - lo;
  hd = 0;
  if (up)
  {
    int l = lo;
    do
    {
      ++count;
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      TSpatialTimeSlice* data = new TSpatialTimeSlice [csize];
      TSpatialTimeSliceIChunk* h = new TSpatialTimeSliceIChunk(data,  l, l, l+sz, l+csize);
      if (hd != 0)
        h->link_to_next(hd);
      else
        hd = h;
      l += sz;
      need -= sz;
    } while (need > 0);
  }
  else
  {
    int hi = fnc;
    do
    {
      ++count;
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      TSpatialTimeSlice* data = new TSpatialTimeSlice [csize];
      TSpatialTimeSliceIChunk* h = new TSpatialTimeSliceIChunk(data,  hi-csize, hi-sz, hi, hi);
      if (hd != 0)
        h->link_to_next(hd);
      hd = h;
      hi -= sz;
      need -= sz;
    } while (need > 0);
  }
  set_cache((TSpatialTimeSliceIChunk*)hd);
  
  maxch = MIN_NCHUNKS;
  if (maxch < count * 2)
    maxch = count * 2;
  chunks = new _TSpatialTimeSliceIChunk_ptr[maxch];
  lch = maxch / 3;
  fch = lch + count;
  base = ch->base_index() - csize * lch;
  int k = lch;
  do
  {
    chunks[k++] = ch;
    set_cache(ch->next());
  } while (ch != hd);
}

TSpatialTimeSliceRPlex:: TSpatialTimeSliceRPlex(int l, int hi, const TSpatialTimeSlice& initval, int chunksize)
{
  lo = l;
  fnc = hi + 1;
  if (chunksize == 0)
  {
    csize = fnc - l;
    make_initial_chunks(1);
  }
  else if (chunksize < 0)
  {
    csize = -chunksize;
    make_initial_chunks(0);
  }
  else
  {
    csize = chunksize;
    make_initial_chunks(1);
  }
  fill(initval);
}

TSpatialTimeSliceRPlex::TSpatialTimeSliceRPlex(const TSpatialTimeSliceRPlex& a)
{
  lo = a.lo;
  fnc = a.fnc;
  csize = a.csize;
  make_initial_chunks();
  for (int i = a.low(); i < a.fence(); a.next(i)) elem(i) = a[i];
}

void TSpatialTimeSliceRPlex::operator= (const TSpatialTimeSliceRPlex& a)
{
  if (&a != this)
  {
    invalidate();
    lo = a.lo;
    fnc = a.fnc;
    csize = a.csize;
    make_initial_chunks();
    for (int i = a.low(); i < a.fence(); a.next(i)) elem(i) = a[i];
  }
}


void TSpatialTimeSliceRPlex::cache(const TSpatialTimeSlice* p) const
{
  const TSpatialTimeSliceIChunk* old = ch;
  const TSpatialTimeSliceIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) index_error();
  }
  set_cache(t);
}

int TSpatialTimeSliceRPlex::owns(Pix px) const
{
  TSpatialTimeSlice* p = (TSpatialTimeSlice*)px;
  const TSpatialTimeSliceIChunk* old = ch;
  const TSpatialTimeSliceIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) return 0;
  }
  set_cache(t);
  return 1;
}


TSpatialTimeSlice* TSpatialTimeSliceRPlex::dosucc(const TSpatialTimeSlice* p) const
{
  if (p == 0) return 0;
  const TSpatialTimeSliceIChunk* old = ch;
  const TSpatialTimeSliceIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) return 0;
  }
  int i = t->index_of(p) + 1;
  if (i >= fnc) return 0;
  if (i >= t->fence_index()) t = (t->next());
  set_cache(t);
  return t->pointer_to(i);
}

TSpatialTimeSlice* TSpatialTimeSliceRPlex::dopred(const TSpatialTimeSlice* p) const
{
  if (p == 0) return 0;
  const TSpatialTimeSliceIChunk* old = ch;
  const TSpatialTimeSliceIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->prev());
    if (t == old) return 0;
  }
  int i = t->index_of(p) - 1;
  if (i < lo) return 0;
  if (i < t->low_index()) t = (t->prev());
  set_cache(t);
  return (t->pointer_to(i));
}

int TSpatialTimeSliceRPlex::add_high(const TSpatialTimeSlice& elem)
{
  TSpatialTimeSliceIChunk* t = tl();
  if (!t->can_grow_high())
  {
    if (t->TSpatialTimeSliceIChunk::empty() && one_chunk())
    {
      t->clear(fnc);
      base = t->base_index() - lch * csize;
    }
    else
    {
      TSpatialTimeSlice* data = new TSpatialTimeSlice [csize];
      t = (new TSpatialTimeSliceIChunk(data,  fnc, fnc, fnc,fnc+csize));
      t->link_to_prev(tl());
      if (fch == maxch)
      {
        maxch *= 2;
        TSpatialTimeSliceIChunk** newch = new _TSpatialTimeSliceIChunk_ptr [maxch];
        memcpy(newch, chunks, fch * sizeof(_TSpatialTimeSliceIChunk_ptr));
        delete[] chunks;
        chunks = newch;
      }
      chunks[fch++] = t;
    }
  }
  *((t->TSpatialTimeSliceIChunk::grow_high())) = elem;
  set_cache(t);
  return fnc++;
}

TSpatialTimeSlice* TSpatialTimeSliceRPlex::grow_high(int& current_index)
{
  TSpatialTimeSliceIChunk* t = tl();
  if (!t->can_grow_high())
  {
    if (t->TSpatialTimeSliceIChunk::empty() && one_chunk())
    {
      t->clear(fnc);
      base = t->base_index() - lch * csize;
    }
    else
    {
      TSpatialTimeSlice* data = new TSpatialTimeSlice [csize];
      t = (new TSpatialTimeSliceIChunk(data,  fnc, fnc, fnc,fnc+csize));
      t->link_to_prev(tl());
      if (fch == maxch)
      {
        maxch *= 2;
        TSpatialTimeSliceIChunk** newch = new _TSpatialTimeSliceIChunk_ptr [maxch];
        memcpy(newch, chunks, fch * sizeof(_TSpatialTimeSliceIChunk_ptr));
        delete[] chunks;
        chunks = newch;
      }
      chunks[fch++] = t;
    }
  }
  set_cache(t); 
  current_index = fnc++;
  return t->TSpatialTimeSliceIChunk::grow_high();
}

int TSpatialTimeSliceRPlex::del_high ()
{
  if (empty()) empty_error();
  TSpatialTimeSliceIChunk* t = tl();
  if (t->TSpatialTimeSliceIChunk::empty()) // kill straggler first
  {
    TSpatialTimeSliceIChunk* pred = t->prev();
    del_chunk(t);
    t = (pred);
    --fch;
  }
  t->TSpatialTimeSliceIChunk::shrink_high();
  if (t->TSpatialTimeSliceIChunk::empty() && !one_chunk())
  {
    TSpatialTimeSliceIChunk* pred = t->prev();
    del_chunk(t);
    t = (pred);
    --fch;
  }
  set_cache(t);
  return --fnc - 1;
}

int TSpatialTimeSliceRPlex::add_low (const TSpatialTimeSlice& elem)
{
  TSpatialTimeSliceIChunk* t = hd;
  if (!t->can_grow_low())
  {
    if (t->TSpatialTimeSliceIChunk::empty() && one_chunk())
    {
      t->cleardown(lo);
      base = t->base_index() - lch * csize;
    }
    else
    {
      TSpatialTimeSlice* data = new TSpatialTimeSlice [csize];
      hd = new TSpatialTimeSliceIChunk(data,  lo-csize, lo, lo, lo);
      hd->link_to_next(t);
      t = ( hd);
      if (lch == 0)
      {
        lch = maxch;
        fch += maxch;
        maxch *= 2;
        TSpatialTimeSliceIChunk** newch = new _TSpatialTimeSliceIChunk_ptr [maxch];
        memcpy(&(newch[lch]), chunks, lch * sizeof(_TSpatialTimeSliceIChunk_ptr));
        delete[] chunks;
        chunks = newch;
        base = t->base_index() - (lch - 1) * csize;
      }
      chunks[--lch] = t;
    }
  }
  *((t->TSpatialTimeSliceIChunk::grow_low())) = elem;
  set_cache(t);
  return --lo;
}


int TSpatialTimeSliceRPlex::del_low ()
{
  if (empty()) empty_error();
  TSpatialTimeSliceIChunk* t = hd;
  if (t->TSpatialTimeSliceIChunk::empty())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
    ++lch;
  }
  t->TSpatialTimeSliceIChunk::shrink_low();
  if (t->TSpatialTimeSliceIChunk::empty() && !one_chunk())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
    ++lch;
  }
  set_cache(t);
  return ++lo;
}

void TSpatialTimeSliceRPlex::reverse()
{
  TSpatialTimeSlice tmp;
  int l = lo;
  int h = fnc - 1;
  TSpatialTimeSliceIChunk* loch = hd;
  TSpatialTimeSliceIChunk* hich = tl();
  while (l < h)
  {
    TSpatialTimeSlice* lptr = loch->pointer_to(l);
    TSpatialTimeSlice* hptr = hich->pointer_to(h);
    tmp = *lptr;
    *lptr = *hptr;
    *hptr = tmp;
    if (++l >= loch->fence_index()) loch = loch->next();
    if (--h < hich->low_index()) hich = hich->prev();
  }
}

void TSpatialTimeSliceRPlex::fill(const TSpatialTimeSlice& x)
{
  for (int i = lo; i < fnc; ++i) elem(i) = x;
}

void TSpatialTimeSliceRPlex::fill(const TSpatialTimeSlice& x, int low, int hi)
{
  for (int i = low; i <= hi; ++i) elem(i) = x;
}


void TSpatialTimeSliceRPlex::clear()
{
  for (int i = lch + 1; i < fch; ++i)
    del_chunk(chunks[i]);
  fch = lch + 1;
  set_cache(chunks[lch]);
  ch->TSpatialTimeSliceIChunk::clear(lo);
  fnc = lo;
}

int TSpatialTimeSliceRPlex::reset_low(int l)
{
  int old = lo;
  int diff = l - lo;
  if (diff != 0)
  {
    lo += diff;
    fnc += diff;
    TSpatialTimeSliceIChunk* t = hd;
    do
    {
      t->re_index(t->low_index() + diff);
      t = t->next();
    } while (t != hd);
  }
  base = hd->base_index() - lch * csize;
  return old;
}


int TSpatialTimeSliceRPlex::OK () const
{
  int v = hd != 0 && ch != 0;         // at least one chunk

  v &= fnc == tl()->fence_index();  // last chunk fnc == plex fnc
  v &= lo == hd->TSpatialTimeSliceIChunk::low_index();    // first lo == plex lo

  v &= base == hd->base_index() - lch * csize; // base is correct;
  v &= lch < fch;
  v &= fch <= maxch;                  // within allocation;

// loop for others:

  int k = lch;                        // to cross-check nch

  int found_ch = 0;                   // to make sure ch is in list;
  const TSpatialTimeSliceIChunk* t = (hd);
  for (;;)
  {
    v &= chunks[k++] == t;             // each chunk is at proper index
    if (t == ch) ++found_ch;
    v &= t->TSpatialTimeSliceIChunk::OK();              // each chunk is OK
    if (t == tl())
      break;
    else                              // and has indices contiguous to succ
    {
      v &= t->top_index() == t->next()->base_index();
      if (t != hd)                  // internal chunks full
      {
        v &= !t->empty();
        v &= !t->can_grow_low();
        v &= !t->can_grow_high();
      }
      t = t->next();
    }
  }
  v &= found_ch == 1;
  v &= fch == k;
  if (!v) error("invariant failure");
  return v;
}

const float* TSpatialTimeSliceRPlex::TimeSeriesData(int array_index) {
  fTmpBuff.clear(); 
  for( int i=lo; i<fnc; i++ ) { 
		TSpatialTimeSlice& ts = elem (i);
    fTmpBuff.add( ts.Value(array_index) );
  }
  return fTmpBuff.data();
}

const float* TSpatialTimeSliceRPlex::TimeStampData() {
  fTmpBuff.clear(); 
  for( int i=lo; i<fnc; i++ ) { 
		TSpatialTimeSlice& ts = elem (i);
    fTmpBuff.add(ts.Time());
  }
  return fTmpBuff.data();
}

void TSpatialTimeSliceRPlex::DumpToFile( FILE* oFile, int size ) {
  float ftmp;
  for( int i=lo; i<fnc; i++ ) { 
	TSpatialTimeSlice& ts = elem (i);
    fprintf(oFile,"\n%f",ts.Time());
	for( int i=0; i<size; i++ ) {
	  ftmp =  ts.Value(i);
	  fprintf(oFile," %f",ftmp);
	}
  }
}

/*
*************************************************************************
*									*
* TemporalTimeSliceIChunk:      						*
*									*
*************************************************************************
*/

// IChunk support

void TTemporalTimeSliceIChunk::error(const char* msg) const
{
  gPrintErr( msg);
}

void TTemporalTimeSliceIChunk::index_error() const
{
  error("attempt to use invalid index");
}

void TTemporalTimeSliceIChunk::empty_error() const
{
  error("invalid use of empty chunk");
}

void TTemporalTimeSliceIChunk::full_error() const
{
  error("attempt to extend chunk beyond bounds");
}

TTemporalTimeSliceIChunk:: ~TTemporalTimeSliceIChunk() {}

TTemporalTimeSliceIChunk::TTemporalTimeSliceIChunk(TTemporalTimeSlice*     d,    
                     int      baseidx,
                     int      lowidx,
                     int      fenceidx,
                     int      topidx)
{
  if (d == 0 || baseidx > lowidx || lowidx > fenceidx || fenceidx > topidx)
    error("inconsistent specification");
  data = d;
  base = baseidx;
  low = lowidx;
  fence = fenceidx;
  top = topidx;
  nxt = prv = this;
}

void TTemporalTimeSliceIChunk:: re_index(int lo)
{
  int delta = lo - low;
  base += delta;
  low += delta;
  fence += delta;
  top += delta;
}


void TTemporalTimeSliceIChunk::clear(int lo)
{
  int s = top - base;
  low = base = fence = lo;
  top = base + s;
}

void TTemporalTimeSliceIChunk::cleardown(int hi)
{
  int s = top - base;
  low = top = fence = hi;
  base = top - s;
}

int TTemporalTimeSliceIChunk:: OK() const
{
  int v = data != 0;             // have some data
  v &= base <= low;              // ok, index-wise
  v &= low <= fence;
  v &= fence <= top;

  v &=  nxt->prv == this;      // and links are OK
  v &=  prv->nxt == this;
  if (!v) error("invariant failure");
  return(v);
}

/*
*************************************************************************
*									*
* TemporalTimeSlicePlex:      						*
*									*
*************************************************************************
*/

// error handling


void TTemporalTimeSlicePlex::error(const char* msg) const
{
  gPrintErr( msg);
}

void TTemporalTimeSlicePlex::index_error() const
{
  error("attempt to access invalid index");
}

void TTemporalTimeSlicePlex::empty_error() const
{
  error("attempted operation on empty plex");
}

void TTemporalTimeSlicePlex::full_error() const
{
  error("attempt to increase size of plex past limit");
}

// generic plex ops

TTemporalTimeSlicePlex:: ~TTemporalTimeSlicePlex()
{
  invalidate();
}  


void TTemporalTimeSlicePlex::append (const TTemporalTimeSlicePlex& a)
{
  for (int i = a.low(); i < a.fence(); a.next(i)) add_high(a[i]);
}

void TTemporalTimeSlicePlex::prepend (const TTemporalTimeSlicePlex& a)
{
  for (int i = a.high(); i > a.ecnef(); a.prev(i)) add_low(a[i]);
}

void TTemporalTimeSlicePlex::reverse()
{
  TTemporalTimeSlice tmp;
  int l = low();
  int h = high();
  while (l < h)
  {
    tmp = (*this)[l];
    elem(l) = (*this)[h];
    elem(h) = tmp;
    next(l);
    prev(h);
  }
}


void TTemporalTimeSlicePlex::fill(const TTemporalTimeSlice& x)
{
  for (int i = lo; i < fnc; ++i) elem(i) = x;
}

void TTemporalTimeSlicePlex::fill(const TTemporalTimeSlice& x, int low, int hi)
{
  for (int i = low; i <= hi; ++i) elem(i) = x;
}


void TTemporalTimeSlicePlex::del_chunk(TTemporalTimeSliceIChunk* x)
{
  if (x != 0)
  {
    x->unlink();
    TTemporalTimeSlice* data = (TTemporalTimeSlice*)(x->invalidate());
    delete [] data;
    delete x;
  }
}


void TTemporalTimeSlicePlex::invalidate()
{
  TTemporalTimeSliceIChunk* t = hd;
  if (t != 0)
  {
    TTemporalTimeSliceIChunk* tail = tl();
    while (t != tail)
    {
      TTemporalTimeSliceIChunk* nxt = t->next();
      del_chunk(t);
      t = nxt;
    } 
    del_chunk(t);
    hd = 0;
  }
}

int TTemporalTimeSlicePlex::reset_low(int l)
{
  int old = lo;
  int diff = l - lo;
  if (diff != 0)
  {
    lo += diff;
    fnc += diff;
    TTemporalTimeSliceIChunk* t = hd;
    do
    {
      t->re_index(t->low_index() + diff);
      t = t->next();
    } while (t != hd);
  }
  return old;
}

/*
*************************************************************************
*									*
* TemporalTimeSliceRPlex:      						*
*									*
*************************************************************************
*/

typedef TTemporalTimeSliceIChunk* _TTemporalTimeSliceIChunk_ptr;

TTemporalTimeSliceRPlex:: TTemporalTimeSliceRPlex()
{
  lo = fnc = 0;
  csize = kDefaultInitialCapacity;
  TTemporalTimeSlice* data = new TTemporalTimeSlice[csize];
  set_cache(new TTemporalTimeSliceIChunk(data,  lo, lo, fnc, lo+csize));
  hd = ch;
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _TTemporalTimeSliceIChunk_ptr[maxch];
  chunks[lch] = ch;
}

TTemporalTimeSliceRPlex:: TTemporalTimeSliceRPlex(int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = 0;
  if (chunksize > 0)
  {
    csize = chunksize;
    TTemporalTimeSlice* data = new TTemporalTimeSlice[csize];
    set_cache(new TTemporalTimeSliceIChunk(data,  lo, lo, fnc, csize+lo));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    TTemporalTimeSlice* data = new TTemporalTimeSlice[csize];
    set_cache(new TTemporalTimeSliceIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _TTemporalTimeSliceIChunk_ptr[maxch];
  chunks[lch] = ch;
}


TTemporalTimeSliceRPlex:: TTemporalTimeSliceRPlex(int l, int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = l;
  if (chunksize > 0)
  {
    csize = chunksize;
    TTemporalTimeSlice* data = new TTemporalTimeSlice[csize];
    set_cache(new TTemporalTimeSliceIChunk(data,  lo, lo, fnc, lo+csize));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    TTemporalTimeSlice* data = new TTemporalTimeSlice[csize];
    set_cache(new TTemporalTimeSliceIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _TTemporalTimeSliceIChunk_ptr[maxch];
  chunks[lch] = ch;
}

void TTemporalTimeSliceRPlex::make_initial_chunks(int up)
{
  int count = 0;
  int need = fnc - lo;
  hd = 0;
  if (up)
  {
    int l = lo;
    do
    {
      ++count;
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      TTemporalTimeSlice* data = new TTemporalTimeSlice [csize];
      TTemporalTimeSliceIChunk* h = new TTemporalTimeSliceIChunk(data,  l, l, l+sz, l+csize);
      if (hd != 0)
        h->link_to_next(hd);
      else
        hd = h;
      l += sz;
      need -= sz;
    } while (need > 0);
  }
  else
  {
    int hi = fnc;
    do
    {
      ++count;
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      TTemporalTimeSlice* data = new TTemporalTimeSlice [csize];
      TTemporalTimeSliceIChunk* h = new TTemporalTimeSliceIChunk(data,  hi-csize, hi-sz, hi, hi);
      if (hd != 0)
        h->link_to_next(hd);
      hd = h;
      hi -= sz;
      need -= sz;
    } while (need > 0);
  }
  set_cache((TTemporalTimeSliceIChunk*)hd);
  
  maxch = MIN_NCHUNKS;
  if (maxch < count * 2)
    maxch = count * 2;
  chunks = new _TTemporalTimeSliceIChunk_ptr[maxch];
  lch = maxch / 3;
  fch = lch + count;
  base = ch->base_index() - csize * lch;
  int k = lch;
  do
  {
    chunks[k++] = ch;
    set_cache(ch->next());
  } while (ch != hd);
}

TTemporalTimeSliceRPlex:: TTemporalTimeSliceRPlex(int l, int hi, const TTemporalTimeSlice& initval, int chunksize)
{
  lo = l;
  fnc = hi + 1;
  if (chunksize == 0)
  {
    csize = fnc - l;
    make_initial_chunks(1);
  }
  else if (chunksize < 0)
  {
    csize = -chunksize;
    make_initial_chunks(0);
  }
  else
  {
    csize = chunksize;
    make_initial_chunks(1);
  }
  fill(initval);
}

TTemporalTimeSliceRPlex::TTemporalTimeSliceRPlex(const TTemporalTimeSliceRPlex& a)
{
  lo = a.lo;
  fnc = a.fnc;
  csize = a.csize;
  make_initial_chunks();
  for (int i = a.low(); i < a.fence(); a.next(i)) elem(i)= a[i];
}

void TTemporalTimeSliceRPlex::operator= (const TTemporalTimeSliceRPlex& a)
{
  if (&a != this)
  {
    invalidate();
    lo = a.lo;
    fnc = a.fnc;
    csize = a.csize;
    make_initial_chunks();
    for (int i = a.low(); i < a.fence(); a.next(i)) elem(i) = a[i];
  }
}


void TTemporalTimeSliceRPlex::cache(const TTemporalTimeSlice* p) const
{
  const TTemporalTimeSliceIChunk* old = ch;
  const TTemporalTimeSliceIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) index_error();
  }
  set_cache(t);
}

int TTemporalTimeSliceRPlex::owns(Pix px) const
{
  TTemporalTimeSlice* p = (TTemporalTimeSlice*)px;
  const TTemporalTimeSliceIChunk* old = ch;
  const TTemporalTimeSliceIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) return 0;
  }
  set_cache(t);
  return 1;
}


TTemporalTimeSlice* TTemporalTimeSliceRPlex::dosucc(const TTemporalTimeSlice* p) const
{
  if (p == 0) return 0;
  const TTemporalTimeSliceIChunk* old = ch;
  const TTemporalTimeSliceIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) return 0;
  }
  int i = t->index_of(p) + 1;
  if (i >= fnc) return 0;
  if (i >= t->fence_index()) t = (t->next());
  set_cache(t);
  return t->pointer_to(i);
}

TTemporalTimeSlice* TTemporalTimeSliceRPlex::dopred(const TTemporalTimeSlice* p) const
{
  if (p == 0) return 0;
  const TTemporalTimeSliceIChunk* old = ch;
  const TTemporalTimeSliceIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->prev());
    if (t == old) return 0;
  }
  int i = t->index_of(p) - 1;
  if (i < lo) return 0;
  if (i < t->low_index()) t = (t->prev());
  set_cache(t);
  return (t->pointer_to(i));
}

int TTemporalTimeSliceRPlex::add_high(const TTemporalTimeSlice& elem)
{
  TTemporalTimeSliceIChunk* t = tl();
  if (!t->can_grow_high())
  {
    if (t->TTemporalTimeSliceIChunk::empty() && one_chunk())
    {
      t->clear(fnc);
      base = t->base_index() - lch * csize;
    }
    else
    {
      TTemporalTimeSlice* data = new TTemporalTimeSlice [csize];
      t = (new TTemporalTimeSliceIChunk(data,  fnc, fnc, fnc,fnc+csize));
      t->link_to_prev(tl());
      if (fch == maxch)
      {
        maxch *= 2;
        TTemporalTimeSliceIChunk** newch = new _TTemporalTimeSliceIChunk_ptr [maxch];
        memcpy(newch, chunks, fch * sizeof(_TTemporalTimeSliceIChunk_ptr));
        delete[] chunks;
        chunks = newch;
      }
      chunks[fch++] = t;
    }
  }
  *((t->TTemporalTimeSliceIChunk::grow_high())) = elem;
  set_cache(t);
  return fnc++;
}

TTemporalTimeSlice* TTemporalTimeSliceRPlex::grow_high(int& current_index)
{
  TTemporalTimeSliceIChunk* t = tl();
  if (!t->can_grow_high())
  {
    if (t->TTemporalTimeSliceIChunk::empty() && one_chunk())
    {
      t->clear(fnc);
      base = t->base_index() - lch * csize;
    }
    else
    {
      TTemporalTimeSlice* data = new TTemporalTimeSlice [csize];
      t = (new TTemporalTimeSliceIChunk(data,  fnc, fnc, fnc,fnc+csize));
      t->link_to_prev(tl());
      if (fch == maxch)
      {
        maxch *= 2;
        TTemporalTimeSliceIChunk** newch = new _TTemporalTimeSliceIChunk_ptr [maxch];
        memcpy(newch, chunks, fch * sizeof(_TTemporalTimeSliceIChunk_ptr));
        delete[] chunks;
        chunks = newch;
      }
      chunks[fch++] = t;
    }
  } 
  set_cache(t); 
  current_index = fnc++;
  return  t->TTemporalTimeSliceIChunk::grow_high();
}

int TTemporalTimeSliceRPlex::del_high ()
{
  if (empty()) empty_error();
  TTemporalTimeSliceIChunk* t = tl();
  if (t->TTemporalTimeSliceIChunk::empty()) // kill straggler first
  {
    TTemporalTimeSliceIChunk* pred = t->prev();
    del_chunk(t);
    t = (pred);
    --fch;
  }
  t->TTemporalTimeSliceIChunk::shrink_high();
  if (t->TTemporalTimeSliceIChunk::empty() && !one_chunk())
  {
    TTemporalTimeSliceIChunk* pred = t->prev();
    del_chunk(t);
    t = (pred);
    --fch;
  }
  set_cache(t);
  return --fnc - 1;
}

int TTemporalTimeSliceRPlex::add_low (const TTemporalTimeSlice& elem)
{
  TTemporalTimeSliceIChunk* t = hd;
  if (!t->can_grow_low())
  {
    if (t->TTemporalTimeSliceIChunk::empty() && one_chunk())
    {
      t->cleardown(lo);
      base = t->base_index() - lch * csize;
    }
    else
    {
      TTemporalTimeSlice* data = new TTemporalTimeSlice [csize];
      hd = new TTemporalTimeSliceIChunk(data,  lo-csize, lo, lo, lo);
      hd->link_to_next(t);
      t = ( hd);
      if (lch == 0)
      {
        lch = maxch;
        fch += maxch;
        maxch *= 2;
        TTemporalTimeSliceIChunk** newch = new _TTemporalTimeSliceIChunk_ptr [maxch];
        memcpy(&(newch[lch]), chunks, lch * sizeof(_TTemporalTimeSliceIChunk_ptr));
        delete[] chunks;
        chunks = newch;
        base = t->base_index() - (lch - 1) * csize;
      }
      chunks[--lch] = t;
    }
  }
  *((t->TTemporalTimeSliceIChunk::grow_low())) = elem;
  set_cache(t);
  return --lo;
}


int TTemporalTimeSliceRPlex::del_low ()
{
  if (empty()) empty_error();
  TTemporalTimeSliceIChunk* t = hd;
  if (t->TTemporalTimeSliceIChunk::empty())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
    ++lch;
  }
  t->TTemporalTimeSliceIChunk::shrink_low();
  if (t->TTemporalTimeSliceIChunk::empty() && !one_chunk())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
    ++lch;
  }
  set_cache(t);
  return ++lo;
}

void TTemporalTimeSliceRPlex::reverse()
{
  TTemporalTimeSlice tmp;
  int l = lo;
  int h = fnc - 1;
  TTemporalTimeSliceIChunk* loch = hd;
  TTemporalTimeSliceIChunk* hich = tl();
  while (l < h)
  {
    TTemporalTimeSlice* lptr = loch->pointer_to(l);
    TTemporalTimeSlice* hptr = hich->pointer_to(h);
    tmp = *lptr;
    *lptr = *hptr;
    *hptr = tmp;
    if (++l >= loch->fence_index()) loch = loch->next();
    if (--h < hich->low_index()) hich = hich->prev();
  }
}

void TTemporalTimeSliceRPlex::fill(const TTemporalTimeSlice& x)
{
  for (int i = lo; i < fnc; ++i) elem(i) = x;
}

void TTemporalTimeSliceRPlex::fill(const TTemporalTimeSlice& x, int low, int hi)
{
  for (int i = lo; i <= hi; ++i) elem(i) = x;
}


void TTemporalTimeSliceRPlex::clear()
{
  for (int i = lch + 1; i < fch; ++i)
    del_chunk(chunks[i]);
  fch = lch + 1;
  set_cache(chunks[lch]);
  ch->TTemporalTimeSliceIChunk::clear(lo);
  fnc = lo;
}

int TTemporalTimeSliceRPlex::reset_low(int l)
{
  int old = lo;
  int diff = l - lo;
  if (diff != 0)
  {
    lo += diff;
    fnc += diff;
    TTemporalTimeSliceIChunk* t = hd;
    do
    {
      t->re_index(t->low_index() + diff);
      t = t->next();
    } while (t != hd);
  }
  base = hd->base_index() - lch * csize;
  return old;
}


int TTemporalTimeSliceRPlex::OK () const
{
  int v = hd != 0 && ch != 0;         // at least one chunk

  v &= fnc == tl()->fence_index();  // last chunk fnc == plex fnc
  v &= lo == hd->TTemporalTimeSliceIChunk::low_index();    // first lo == plex lo

  v &= base == hd->base_index() - lch * csize; // base is correct;
  v &= lch < fch;
  v &= fch <= maxch;                  // within allocation;

// loop for others:

  int k = lch;                        // to cross-check nch

  int found_ch = 0;                   // to make sure ch is in list;
  const TTemporalTimeSliceIChunk* t = (hd);
  for (;;)
  {
    v &= chunks[k++] == t;             // each chunk is at proper index
    if (t == ch) ++found_ch;
    v &= t->TTemporalTimeSliceIChunk::OK();              // each chunk is OK
    if (t == tl())
      break;
    else                              // and has indices contiguous to succ
    {
      v &= t->top_index() == t->next()->base_index();
      if (t != hd)                  // internal chunks full
      {
        v &= !t->empty();
        v &= !t->can_grow_low();
        v &= !t->can_grow_high();
      }
      t = t->next();
    }
  }
  v &= found_ch == 1;
  v &= fch == k;
  if (!v) error("invariant failure");
  return v;
}

const float* TTemporalTimeSliceRPlex::TimeSeriesData() {
  fTmpBuff.clear();
  for( int i=lo; i<fnc; i++ ) { 
    fTmpBuff.add( (float) elem(i).Value() );
  }
  return fTmpBuff.data();
}

float TTemporalTimeSliceRPlex::max() {
  float rv = -FLT_MAX;
  for( int i=lo; i<fnc; i++ ) { 
    float val = (float) elem(i).Value();
		if( val > rv ) rv = val;
  }
  return rv;
}

float TTemporalTimeSliceRPlex::min() {
  float rv = FLT_MAX;
  for( int i=lo; i<fnc; i++ ) { 
    float val = (float) elem(i).Value();
		if( val < rv ) rv = val;
  }
  return rv;
}

void TTemporalTimeSliceRPlex::DumpToFile(FILE* oFile) {
  float ftmp;
  for( int i=lo; i<fnc; i++ ) { 
		TTemporalTimeSlice& ts = elem (i);
    ftmp = ts.Value( ); 
    fprintf(oFile,"\n%f \t%f",ts.Time(),ftmp);
  }
}

const float* TTemporalTimeSliceRPlex::TimeStampData() {
  fTmpBuff.clear(); 
  for( int i=lo; i<fnc; i++ ) { 
		TTemporalTimeSlice& ts = elem (i);
    fTmpBuff.add(ts.Time());
  }
  return fTmpBuff.data();
}
