//---------------------------------------------------------Model------------------------------
  // Model.cc
  // Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
  //----------------------------------------------------------------------------------------
  
#include "Model.h"
#include "Module.h"
#include "Variable.h"
#include "ObjectIterator.h"


//========================================================================================
// CLASS TModel
//========================================================================================

  //========================================================================================
  // 				CLASS TModel 
  //========================================================================================

TModel*  TModel::kNullModel = (TModel*) 0x1;
  
  //----------------------------------------------------------------------------------------
  //				TModel::TModel:
  //----------------------------------------------------------------------------------------

 TModel::TModel() : Model("Model")
{
  memset(Ivar,0,100);
  fNProcDims = 0;
  fDerivativeWeight = 1.0;
  fDerivativeMode = 0;
  fCEqnFile = NULL;
} // TModel::TModel 
 
//----------------------------------------------------------------------------------------
// TModel::Config: 
//----------------------------------------------------------------------------------------

int  TModel::Config(TConfigData& cd) {
  const CString& cmd = cd.Cmd();
  if(gDebug) { 
    sprintf(gMsgStr,"\nReading Application Config for %s: Nargs = %d, CMD = %s",
	    fName.chars(), cd.NArgs(), cmd.chars() );	
    gPrintLog();  
  }
  int iarg; float farg;
	if( cmd == "N" ) {
    cd.IntArg(0,fProcDims[0],CD::kRequired);
    if( cd.IntArg(1,iarg) > 0 ) fProcDims[1] = iarg;
    if( cd.IntArg(2,iarg) > 0 ) fProcDims[2] = iarg;
  }  else  if ( cmd == "DS" ) {
		if( cd.FloatArg( 0, fDerivativeWeight ) ) {
			if( cd.HasArg(1) ) { 
				CString& cmode = *cd.Arg(1); 
				fDerivativeMode = (byte) cmode(0); 
			}
		}
	} else {
    return Model::Config(cd);
  }
  return 1;
} //  TModel::Config

int TModel::ConfigExternal(TConfigData& cd, Module* m) {
  ExternalModuleRec* er = (ExternalModuleRec*) fExternalModulesList.GetObjectByName( cd.Name(1) ); 
  if( er == NULL ) {
    er = new ExternalModuleRec( cd.Name() );
    fExternalModulesList.append(*er);
  }
  CString cmd( cd.Cmd() );
  cmd += "("; cmd += *(cd.Arg(0));
  for( int i=1; i< cd.NArgs(); i++ ) { cmd += ","; cmd += *(cd.Arg(i)); }
  cmd += ")"; 
  er->Cmds().Add(cmd);
  return 1;
}

  //----------------------------------------------------------------------------------------
  //				TModel::WriteDataToConfig:
  //----------------------------------------------------------------------------------------

  void  TModel::WriteDataToConfig( CStream& outStream ) 
{
  if(fNProcDims > 0) outStream << " N(" << fProcDims[0] << "," << fProcDims[1] << ")";
  outStream << " DS(" << fDerivativeWeight << "," << (int)fDerivativeMode << ")";

  for( Pix p =fExternalModulesList.first(); p; fExternalModulesList.next(p) ) {
    ExternalModuleRec& er = (ExternalModuleRec&) fExternalModulesList(p); 
    WriteNameToConfigFile( outStream, '~', er.SName() ); 
    ArgArray& cmds = er.Cmds();
    for( int i=0; i< cmds.NArgs(); i++ ) outStream << cmds[i].chars() << ' ';
  }
  Model::WriteDataToConfig( outStream );

} //	TModel::WriteDataToConfig

int TModel::WriteSetupFile( char* cnfgExtension ) {
	int rv = 1, i;
  CPathString path(Env::ConfigPath());
  path.Add( Env::ModelName() );
  FILE* sf = fopen(path.Path(cnfgExtension),"w");
  if( sf ) {
		fprintf(sf,"config input File names = ( ");
		for( i=0; i<fConfigFileList.NArgs(); i++ ) {
			if( i > 0 ) { fprintf(sf,","); }
			fprintf(sf," %s",fConfigFileList[i].chars());
		}
		fprintf(sf," ); ");
		fprintf(sf,"\nmml input File names = ( ");
		for( i=0; i<fMMLFileList.NArgs(); i++ ) {
			if( i > 0 ) { fprintf(sf,","); }
			fprintf(sf," %s",fMMLFileList[i].chars());
		}
		fprintf(sf," ); ");
		fprintf(sf,"\nconfig output File name = %s;",fConfigOutFile.chars());
		fprintf(sf,"\nmml output File name = %s;",fMMLOutFile.chars());
		fclose(sf);
	} else {
		sprintf(gMsgStr,"Can't generate setup file: %s", path.Path(cnfgExtension).chars());  gPrintErr();
		rv = 0; 
	} 
  return rv;
}
      
//----------------------------------------------------------------------------------------
// TModel::Run: 
//----------------------------------------------------------------------------------------
  
  void TModel::Run() 
{
	ExprBuilder::SetMode(ExprBuilder::kMML);
	ExprBuilder::SetDefaultLibrary("SL");

	int useDefaultConfig = 1;
	int readDBParamValues = 0;
	int useDefaultMML = 1;
	int readSMML = 0;
	fConfigOutFile = Env::ModelName(); ((fConfigOutFile += ".") += Env::ScenarioName()) += ".conf.out";
	fConfigTemplateOutFile = Env::ModelName();  ((fConfigTemplateOutFile += ".") += Env::ScenarioName()) + ".conf.template";
//  fMMLOutFile = Env::ModelName() + ".MML.new";
  int Argc = Env::Argc();
  for(int i=1; i<Argc; i++) {
		if ( Env::Argv(i) == "-mi" ) {
      if ((i+1) < Argc) { 
				fMMLFileList.Add( Env::Argv(++i) ); 
				useDefaultMML = 0; 
			}
    } else if ( Env::Argv(i) == "-co" ) {
      if ((i+1) < Argc) fConfigOutFile = Env::Argv(++i); 
	} else 	if( Env::Argv(i) == "-ignore_STELLA_arrays" ) { 
			readDBParamValues = 0;
    } else if ( Env::Argv(i) == "-ci" ) {
      if ((i+1) < Argc) {
		fConfigFileList.Add( Env::Argv(++i) ); 
		useDefaultConfig = 0;
	  }
	} else if (Env::Argv(i) ==  "-pd") {
      if ((i+1) < Argc) fParserDebug = atoi( Env::Argv(++i) ); 
	} else if (Env::Argv(i) ==  "-smp") {
      if ((i+1) < Argc) fSMP = atoi( Env::Argv(++i) ); 
    } else if (Env::Argv(i) ==  "-smml") {
      if ((i+1) < Argc) readSMML = atoi( Env::Argv(++i) ); 
	  ExprBuilder::SetMode(ExprBuilder::kSMML);
    } else if (Env::Argv(i) == "-help") {
      gPrintScreen("CGP command line Arguments:");
      gPrintScreen(" -mi FILE    Use FILE as MML input file (default: <ProjDir>/Models/<ModelName>.MML)");
      gPrintScreen(" -co FILE	   Use FILE as configuration output file (default: <ProjDir>/Config/<ModelName>.C.config)");
      gPrintScreen(" -ci FILE    Use FILE as configuration input file  (default: <ProjDir>/Config/<ModelName>.C.config)");
      gPrintScreen(" -pd INT     Set parser debug level to INT (default: 0) ");
      exit(0);
    }  
      
  }
  
//  if( getSetup ) { 	
//		if( !ReadSetupFile(".CGP.MML") ) {

	if(useDefaultConfig) { 
	  CString config_file(Env::ModelName());  ((config_file += ".") += Env::ScenarioName()) + ".config";
	  fConfigFileList.Add(config_file); 
	}
	if(useDefaultMML) { fMMLFileList.Add( Env::ModelName() + ".MML" ); }


//			WriteSetupFile( ".CGP.MML" );
//		}
//	}

  CPathString dpath(Env::ModelPath());
  dpath.Add("Driver");
  dpath.Add( Env::ModelName() );
  fCEqnFile = fopen(dpath.Path(".cc"),"w");
  if(fCEqnFile==NULL) { sprintf(gMsgStr,"Error, unable to open C output file: %s",dpath.Path(".cc").chars()); gFatal(); }
  fCHdrFile = fopen(dpath.Path(".h"),"w");
  if(fCHdrFile==NULL) { sprintf(gMsgStr,"Error, unable to open C header file: %s",dpath.Path(".h").chars()); gFatal(); }

#ifdef SME_LIBRARY  
  CPathString path(SME_LIBRARY);
#else 
  CPathString path(Env::DriverPath());
  path += "lib";
#endif
  if (getenv("SME_LIBRARY") != NULL) path += getenv("SME_LIBRARY");
  
  ReadMMLFile( path, "StellaLib.MML" );
  ReadMMLFile( path, "SpaceLib.MML" );

  CPathString path_str(Env::ModelPath());
  path_str.Add("UserCode");
  ReadMMLFile( path_str, "UserFunctions.MML" );

	SetNode( ReadMMLFiles() );
	ProcessNode();
	
	SetFieldsFromMML();

	if( readDBParamValues ) { ReadDBParamValues( "DBaseParameters.out", 1 ); }	
	ReadDBParamValues("DBaseParameters1.out"); 		
	
	if( Env::ReadConf())  { ReadConfigFiles(); }
	
	ProcessBiflowDeclarations();

	RunNodeProcessing(kPCDependencies);

	RunNodeProcessing(kPCDerivatives);

	CreateEventLists();

	RunNodeProcessing(kPCCodeGen);

	GenerateCode();

  if( Env::ReadConf())  { WriteConfigFile(); }
/* 
  CPathString path1(Env::ConfigPath());
  path1.Add(Env::ModelName());
  FILE* cFile = fopen(path1.Path(".run.MML"),"r");
  if(cFile==NULL) {
    FILE* cFile = fopen(path1.Path(".run.MML"),"w");
    if(cFile==NULL) { sprintf(gMsgStr,"Can't open file %s",path1.chars()); gPrintErr(); }
    else {
      fprintf(cFile,"config input list File names = ( %s );\n", fConfigOutFile.chars() );
      fprintf(cFile,"mml input list File names = ( );\n");
      fprintf(cFile,"Debug level = 0;\n");
      fclose(cFile);
    }
  }
*/
	CreateUserCodeMakefile();
	
  if( gDebug ) { Dump( Env::LogFile() ); }

  gPrintScreen("Code Generation Completed!\n\n");
    
} // TModel::Run
  
//----------------------------------------------------------------------------------------
//				TModel::CreateNewModule:
//----------------------------------------------------------------------------------------

Module* TModel::CreateNewModule( const CString& name ) {
  TModule *mod = (TModule*) fModuleList.GetObjectByName( name );
  if( mod == NULL ) {
	mod = new TModule(name);
	fModuleList.Insert(*mod);
  }
  return mod; 
}

//----------------------------------------------------------------------------------------
//				TModel::WriteCCode:
//----------------------------------------------------------------------------------------

void  TModel::WriteCCode( FILE* CHdrFile, FILE* CEqnFile) {

  fprintf(CHdrFile,"#ifndef __ModelEqns__\n#define __ModelEqns__\n\n#include \"SpaceSim.h\"\n");
  fprintf(CEqnFile,"#include \"%s.h\"\n",(const char*)Env::ModelName());
	for( Pix p = fCodeFileList.first(); p; fCodeFileList.next(p) ) {
		const TNamedObject& fileName = fCodeFileList(p);
		fprintf(CHdrFile,"#include \"%s.h\"\n",fileName.Name());
	}	

//  fprintf(CEqnFile,"\n\nvoid gSetupExternals() {");
//  for( Pix p =fExternalModulesList.first(); p; fExternalModulesList.next(p) ) {
//    ExternalModuleRec& er = (ExternalModuleRec&) fExternalModulesList(p);  
//    fprintf(CEqnFile,"\n\t%s::Setup();",er.Name());
//  }
//  fprintf(CEqnFile,"\n}\n");

  Model::WriteCCode( CHdrFile, CEqnFile);

  fprintf(CHdrFile,"\n\n#endif\n");

}
//----------------------------------------------------------------------------------------
//				TModel::SetFieldsFromMML:
//----------------------------------------------------------------------------------------
 
void TModel::SetFieldsFromMML()
{
  sprintf(gMsgStr,"SetFieldsFromMML"); gPrintScreen();
  CObjectIterator iter( fModuleList ); 	   
  for(TModule* aModule = iter.FirstModule(); iter.More(); aModule = iter.NextModule()) { 		
    aModule->SetFieldsFromMML();
  }
}  //  TDocCG::WriteDefaultConfig

 
  //----------------------------------------------------------------------------------------
  //				TModel::Free:
  //----------------------------------------------------------------------------------------
  
  void TModel :: Free()
{
  if(fCEqnFile) fclose(fCEqnFile);
  if(fCHdrFile) fclose(fCHdrFile);
  
} // TModel::Free	

//----------------------------------------------------------------------------------------
//				TModel::GenerateCode:
//----------------------------------------------------------------------------------------
  
void TModel :: GenerateCode() { WriteCCode(fCHdrFile,fCEqnFile); } 

//----------------------------------------------------------------------------------------
//				TModel::CreateUserCodeMakefile:
//----------------------------------------------------------------------------------------

void TModel :: CreateUserCodeMakefile() {

#ifdef SME_LIBRARY  
  CPathString path(SME_LIBRARY);
#else 
  CPathString path(Env::DriverPath());
  path += "lib";
#endif
//  if (getenv("SME_LIBRARY") != NULL) path += getenv("SME_LIBRARY");
  path += "makefile.template";
  FILE* mk_template = fopen(path,"r");
  sprintf(gMsgStr,"Generating UserCode makefile from template %s", path.chars() ); gPrintScreen();
	if( mk_template == NULL ) { sprintf(gMsgStr,"Can't open %s",path.chars()); gPrintErr(); return; }
	  
	CPathString path1(Env::ModelPath());
  path1.Add("UserCode"); path1.Add("makefile");
  FILE* makefile = fopen(path1,"w");
	if( makefile == NULL )  { sprintf(gMsgStr,"Can't open %s",path1.chars()); gPrintErr(); return; }

  int ch, cnt=0; Pix p;
  Bool has_code = (fCodeFileList.length() > 0);
  while ( (ch = fgetc(mk_template) ) != EOF ) {
		fputc(ch,makefile);
		if( ch == '!' ) {
			switch( cnt++ ) {
				case 0:
					if(has_code) {
						fprintf(makefile,"\nTargetLibName = lib%sUser.a",Env::ModelName().chars());
						fprintf(makefile,"\nTargetLibNameTarg = $(TargetLibName)");
					} else { fprintf(makefile,"\nTargetLibName = "); }
				break;
				case 1:
					fprintf(makefile,"\nsrcs = ");
					for( p = fCodeFileList.first(); p; fCodeFileList.next(p) ) {
						const TNamedObject& fileName = fCodeFileList(p);
						fprintf(makefile," %s.cc ",fileName.Name());
					}	
				break;		
				case 2:
					fprintf(makefile,"\nobjs = ");			
					for( p = fCodeFileList.first(); p; fCodeFileList.next(p) ) {
						const TNamedObject& fileName = fCodeFileList(p);
						fprintf(makefile," %s.o ",fileName.Name());
					}
				break;			
				case 3:
				break;
					for( p = fCodeFileList.first(); p; fCodeFileList.next(p) ) {
						const TNamedObject& fileName = fCodeFileList(p);
						fprintf(makefile,"\n%s.o: %s.h $(SS_LIB) $(PPG_LIB) $(MML_LIB) $(BASE_LIB)",fileName.Name(),fileName.Name());
					}
				break;			
			}		
		}  
  }
  fclose(mk_template);
  fclose(makefile);

#ifdef SME_LIBRARY  
  CPathString dpath(SME_LIBRARY);
#else 
  CPathString dpath(Env::DriverPath());
  dpath.Add("lib");
#endif
	char* mpi_suffix = getenv("MPI_SUFFIX");
	CString makefilename("");
	if( mpi_suffix ) { makefilename += mpi_suffix; }
	else { gPrintErr("MPI_SUFFIX undefined"); }
	makefilename += "makefile";
  dpath.Add(makefilename);
  mk_template = fopen(dpath,"r");
  sprintf(gMsgStr,"Generating Driver makefile from template %s", dpath.chars() ); gPrintScreen();
	if( mk_template == NULL ) { sprintf(gMsgStr,"Can't open %s",dpath.chars()); gPrintErr(); return; }
	  
	CPathString dpath1(Env::ModelPath());
  dpath1.Add("Driver"); dpath1.Add(makefilename);
  makefile = fopen(dpath1,"w");
  int first = 1;
	if( makefile == NULL )  { sprintf(gMsgStr,"Can't open %s",dpath1.chars()); gPrintErr(); return; }
  while ( (ch = fgetc(mk_template) ) != EOF ) {
		fputc(ch,makefile);
		if( ch == '!' && first ) {
			first = 0;
			if(has_code) {
				CPathString path2(Env::ModelPath());
				path2 += "UserCode";
				fprintf(makefile,"\n\nUSER_CODE_LIB = -L%s -l%sUser\n",path2.Path().chars(),Env::ModelName().chars());
			} else fprintf(makefile,"\n\nUSER_CODE_LIB = \n");
		}  
  }
  fclose(mk_template);
  fclose(makefile);  
} 

void TModel :: ProcessBiflowDeclarations() {
   const char* biflow_name;
   while( (biflow_name = Env::RetrieveBiFlow()) != NULL ) {
	 CString full_name(biflow_name);
	 CString mod_name(full_name.before(' ')); mod_name.trim( '\n', ' ' );
 	 CString var_name(full_name.after(' '));  var_name.trim( ' ', '\n' );
 	 Module* m = GetModule(mod_name,False);
 	 Variable* v = ( m == NULL ) ?  NULL : m->GetVariable(var_name,False); 
	 if( v != NULL ) {
	   EFluxType ft = v->getFluxTypeIndex();
	   if( ft == kF_Unknown ) { 
		 v->setFluxTypeIndex( kF_BiFlow );
	   }
	 } else {
	   gPrintErr( format("Unknown variable in BiFlow file: %s:%s",mod_name.chars(),var_name.chars()) );
	 }
   }
}
