//----------------------------------------------------------------------------------------
//	UModule.cc
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History:
//----------------------------------------------------------------------------------------
  
#include "ObjectIterator.h"
#include "Model.h"
#include "Module.h"
#include "Variable.h"
#include "MML_Event.h"
#include "MML_Connection.h"


//========================================================================================
// 				CLASS TModule 
//========================================================================================
    
  //----------------------------------------------------------------------------------------
  //				TModule::TModule:
  //----------------------------------------------------------------------------------------

EConfigInfo TModule::kStatus = kCInfo2;
EConfigInfo TModule::kOpLevel = kCInfo3;
  
TModule::TModule ( const CString& name ) : Module( name )
{ 
  fSuper = NULL;
  fCD[0] = fCD[1] = 0;
  
  Variable* v = CreateNewVariable("TIME");
  v->SetCInfo(Variable::kType,Variable::kTime);
//  v->SetF(FisTIME,True,"IModule");
  v->SetF(FisConstant,False,"IModule"); 
   
} // TModule::IModule
  

//----------------------------------------------------------------------------------------
//				TModule::ReadEvent:
//----------------------------------------------------------------------------------------
  
int  TModule::ReadEvent( TMMLData& md, TEvent::EType et ) { return 0; } //				TModule::ReadEvent:

int TModule::ConfigExternal(TConfigData& cd, Module* m) {
  ExternalModuleRec* er = (ExternalModuleRec*) fExternalModulesList.GetObjectByName( cd.Name() ); 
  if( er == NULL ) {
    er = new ExternalModuleRec( cd.Name() );
    fExternalModulesList.append(*er);
  }
  CString cmd( cd.Cmd() );
  cmd += "("; cmd += *(cd.Arg(0));
  for( int i=1; i< cd.NArgs(); i++ ) { cmd += ","; cmd += *(cd.Arg(i)); }
  cmd += ")"; 
  er->Cmds().Add(cmd);
  return 1;
}

//----------------------------------------------------------------------------------------
//				TModule::Free:
//----------------------------------------------------------------------------------------
  
void  TModule::Free ()
{
}  // TModule::Free
  
//----------------------------------------------------------------------------------------
//				TModule::CreateSubModule:
//----------------------------------------------------------------------------------------
  
  
Module* TModule :: AddSubModule( const char* module_name, Bool create )
{
  TModule* thisModule = (TModule*) TModel::I().GetModule(module_name);
  Pix p = 0;

  if( thisModule == NULL ) {
	if( create ) {
	  thisModule = new TModule(module_name);						// Allocate a new module object;
	  TModel::I().ModuleList().Insert(*thisModule);	 
	  p = fModuleList.Insert(*thisModule);						// Add to submodule list
	  thisModule->fSuper = this;
	} 						
  } else {
  
	p = fModuleList.Insert(*thisModule);						// Add to submodule list
	thisModule->fSuper = this;
	
	if( gDebug && (p!=0) ) {
		sprintf(gMsgStr," Adding SubModule: %s (%x) to Module %s (%x)",
						  thisModule->Name(), thisModule, Name(), this );
		gPrintScreen();
	}
  }
  
  return thisModule;
  
} // TModule::CreateSubModule
	
//----------------------------------------------------------------------------------------
//				TModule::CreateNewVariable:
//----------------------------------------------------------------------------------------
  
Variable* TModule :: CreateNewVariable( const CString& variable_name, Variable* template_var ) { 
  TVariable* newVariable = (TVariable*) fVarList.GetObjectByName(variable_name);

  if( newVariable && newVariable->GetF(FisIgnored) ) {
	fVarList.Delete(*newVariable);
	newVariable = NULL;
  }

  if(newVariable == NULL) { 
    newVariable = new TVariable(variable_name);
    newVariable->Init(this);
  }
  return newVariable;

} // TModule::CreateNewVariable  

 
//----------------------------------------------------------------------------------------
//				TModule::MakeConnections:
//----------------------------------------------------------------------------------------
  
void TModule::MakeConnections (void) {
  CObjectIterator iter(fVarList); 
  TVariable* cVar; TModule* aMod;
  Bool gotConnection = False;
  for(TVariable* aVar = iter.FirstVariable(); iter.More(); aVar = iter.NextVariable()) {      
    if( aVar->GetF(FisImport) ) {
      if(fSuper) if( cVar = (TVariable*) fSuper->ConnectionList()->GetConnection( Name(), aVar->Name(), TConnectionList::kVar ) ) {
				gotConnection = True;
				aMod = (TModule*) fSuper->ConnectionList()->GetConnection( Name(), aVar->Name(), TConnectionList::kMod );
      }
      if( !gotConnection && ConnectionList() ) {
				if( cVar = (TVariable*) ConnectionList()->GetConnection( Name(), aVar->Name(), TConnectionList::kVar  ) ) {
					gotConnection = True; 
					aMod = (TModule*) ConnectionList()->GetConnection( Name(), aVar->Name(), TConnectionList::kMod );
				}
      }   	
      if(gotConnection) {
				aVar->MakeConnection(cVar);
				if( cVar && cVar->GetCInfo(Variable::kInitMode) == Variable::kDBase ) {
					DBaseInputPipe* p = (DBaseInputPipe*) cVar->GetPipe(Pipe::kDBaseInput);
					if(p) aVar->AddMapInputs( *p );
				}
      }
      else {
				sprintf(gMsgStr,"Missing connection for input: %s in Module %s", aVar->Name(), (char*)Name() );
				gPrintErr();
			}	    	
    }
  }      
} // TModule::SetupDependencies:


//----------------------------------------------------------------------------------------
//				TModule::SetFieldsFromMML:
//----------------------------------------------------------------------------------------

void TModule ::  SetFieldsFromMML () {

  static int foundRoot = 0;

  if( fSuper == NULL ) { 
    if(foundRoot) { sprintf(gMsgStr,"Warning: Multiple Root Modules: %s (%x)",Name(),this); gPrintErr(); }
    SetCInfo(TModule::kStatus,TModule::kGlobal); 
    foundRoot = 1; 
  }

  CObjectIterator iter(fVarList);
  for(TVariable* aVar = iter.FirstVariable(); iter.More(); aVar = iter.NextVariable()) {
    aVar->SetFieldsFromMML ();
  }

  MakeConnections();
}

//----------------------------------------------------------------------------------------
//				TModule::ParseConfig:
//----------------------------------------------------------------------------------------

int TModule::Config( TConfigData& cd) {
  int iarg; float farg; 
  const CString& cmd = cd.Cmd();

  if( cmd == "cd" ) {
    if( cd.IntArg(0, iarg, CD::kRequired) ) fCD[0] = iarg;
    if( cd.IntArg(1, iarg) )  fCD[0] = iarg;
    return 1;
  } else if( cmd == "i" ) {
    SetCInfo( kStatus, kIgnored ); 
    return 1;
  } else {
    return Module::Config(cd);
  }
} //  TModule::ParseConfig

//----------------------------------------------------------------------------------------
//				TModule::WriteDataToConfig:
//----------------------------------------------------------------------------------------

void TModule::WriteNameToConfigFile(CStream& cnfgStream, char type, const CString& name) {
  int i, nspace, cnt=0;
  nspace = 33-name.length();
  cnfgStream.put('\n');
  cnfgStream.put(type);
  cnfgStream.put(' ');
  cnfgStream << name.chars();
  for(i=0; i<=nspace; i++) cnfgStream.put(' ');
}

void  TModule::WriteDataToConfig( CStream& outStream ) {
  Module::WriteDataToConfig( outStream );
  if( fCD[0] > 0 || fCD[1] > 0 ) outStream << " cd(" << fCD[0] << "," << fCD[1] << ")"; 
  if( GetCInfo( kStatus ) == kIgnored  )  outStream << " i() "; 

  for( Pix p =fExternalModulesList.first(); p; fExternalModulesList.next(p) ) {
    ExternalModuleRec& er = (ExternalModuleRec&) fExternalModulesList(p); 
    WriteNameToConfigFile( outStream, '~', er.SName() ); 
    ArgArray& cmds = er.Cmds();
    for( int i=0; i< cmds.NArgs(); i++ ) outStream << cmds[i].chars() << ' ';
  }

} //	TModule::WriteDataToConfig


void TModule::WriteModuleCode(FILE* CHdrFile, FILE* CEqnFile) const {

  fprintf(CHdrFile,"\n\tint LinkFluxes();"); 
  fprintf(CEqnFile,"\nint %s::LinkFluxes() {",(char*)Name());
  for( Pix p = fVarList.first(); p; fVarList.next(p) ) {
    TVariable& v = (TVariable&) fVarList(p);
    if( v.GetF(FisFlux) ) {
      if( v.Destination() ) fprintf(CEqnFile,"\n\t%s.SetDestination(&%s);", v.Name(), v.Destination()->Name() );
      if( v.Origin() ) fprintf(CEqnFile,"\n\t%s.SetOrigin(&%s);", v.Name(), v.Origin()->Name() );
    }
  }
  fprintf(CEqnFile,"\n}\n");
}


					  
