#include "ByteGrid.h"

//----------------------------------------------------------------------------------------
//						ByteGrid
//----------------------------------------------------------------------------------------

ByteGrid::ByteGrid(int r0, int c0, int r1, int c1, int bytes, int r_inc, int c_inc) 
  : GeoRegion2( r0, c0, r1, c1, r_inc, c_inc ) 
{ 
  fData = NULL; 
  ReAlloc(r0,c0,r1,c1,bytes,r_inc,c_inc);
}

ByteGrid::ByteGrid( const ByteGrid& md )  : GeoRegion2(md) { 
  fNBytes = md.fNBytes; 
  fMaxVal = Util::pow2(fNBytes*8)-2; 
  fData = NULL; 
  ReAlloc(md); 
}

void ByteGrid::CopyDataTo( byte* data, int bytes, const Region2& r) const {   
                                                          // Copies from fData to data in region r.
                                                          // data should be of size r.nelem()*bytes.
  if( bytes == fNBytes && r == (*this) ) {
    memcpy( data, fData, r.nelem()*bytes);      
  } else {
    Region2 ri(r); ri *= (*this);                                   
    int cbytes = ( bytes<fNBytes ) ? bytes : fNBytes;
    for(int ir = ri.lower(0); ir<= ri.upper(0); ir+=ri.increment(0) ) {
      for(int ic = ri.lower(1); ic<= ri.upper(1); ic+=ri.increment(1) ) {
				memcpy( data + r.bindex(ir,ic), fData + bindex(ir,ic), cbytes);      
      }
    }
  }
}

void ByteGrid::CopyDataFrom( const int* data, int bytes, const Region2& r) {
                                                              // Copies from data to fData in region r.
                                                             // data should be of size r.nelem()*bytes.
  Region2 ri(r); ri *= (*this);                                   
  int cbytes = ( bytes<fNBytes ) ? bytes : fNBytes;
  for(int ir = ri.lower(0); ir<= ri.upper(0); ir+=ri.increment(0) ) {
    for(int ic = ri.lower(1); ic<= ri.upper(1); ic+=ri.increment(1) ) {
      memcpy( fData + bindex(ir,ic), data + r.bindex(ir,ic), cbytes);      
    }
  }
}

void ByteGrid::CopyDataFrom( const float* data, int bytes, float s, float o, const Region2& r) {
                                                              // Copies from data to fData in region r.
                                                             // data should be of size r.nelem()*bytes.
  Region2 ri(r); ri *= (*this);                                   
  int cbytes = ( bytes<fNBytes ) ? bytes : fNBytes;
  for(int ir = ri.lower(0); ir<= ri.upper(0); ir+=ri.increment(0) ) {
    for(int ic = ri.lower(1); ic<= ri.upper(1); ic+=ri.increment(1) ) {
      int iv = (int) ( s*data[r.bindex(ir,ic)] - o );
      memcpy( fData + bindex(ir,ic), &iv, cbytes);      
    }
  }
}

void ByteGrid::CopyDataFrom( const byte* data, int bytes, const Region2& r) {   
                                                              // Copies from data to fData in region r.
                                                             // data should be of size r.nelem()*bytes.
  if( bytes == fNBytes && r == (*this) ) {
    memcpy( fData, data, r.nelem()*bytes);      
  } else {
    Region2 ri(r); ri *= (*this);                                   
    int cbytes = ( bytes<fNBytes ) ? bytes : fNBytes;
    for(int ir = ri.lower(0); ir<= ri.upper(0); ir+=ri.increment(0) ) {
      for(int ic = ri.lower(1); ic<= ri.upper(1); ic+=ri.increment(1) ) {
				memcpy( fData + bindex(ir,ic), data + r.bindex(ir,ic), cbytes);      
      }
    }
  }
}

void ByteGrid::SetValue(  int ir, int ic, unsigned long dval ) {   // Sets value at (ir,ic) to dval
  if( !inside(ir,ic) ) return;
  dval = (dval>0) ? dval : 0;
  dval = (dval<fMaxVal) ? dval : fMaxVal;
  int offset = bindex(ir,ic);
  byte* mPtr = fData + offset*fNBytes;
  Util::enc_Nb( &mPtr, dval, fNBytes);  
}

int ByteGrid::SetRainbowValue(  int ir, int ic, byte dval ) {   // Sets value at (ir,ic) to dval
  if( !inside(ir,ic) ) return 1;
  if( fNBytes != 3 ) { sprintf(gMsgStr,"Wrong number of bytes for color map: %d",fNBytes); gPrintErr(); return -1; }
  int offset = bindex(ir,ic);
  byte* mPtr = fData + offset*fNBytes;
  *mPtr++ = dval;  
  *mPtr++ = ( dval < 128 ) ? 2*dval : 510 - 2*dval;  
  *mPtr++ = 255-dval;
  return 0;  
}

void ByteGrid::ReAlloc( int r0, int c0, int r1, int c1, int bytes, int r_inc, int c_inc ) {
  if(r_inc<0) r_inc = inc(0);   if(c_inc<0) c_inc = inc(1);
  Region2 r( r0, c0, r1, c1, r_inc, c_inc );
  if( (r != *this) ||  (bytes != fNBytes) ) {
	  byte* data = new byte[ r.nelem() * bytes ];
	  if(fData) {
		  CopyDataTo(data,bytes,r);
		  Free();
	  }
	  setregion(r);
	  fNBytes = bytes;
	  fData = data;
	  fMaxVal = Util::pow2(fNBytes*8)-2;
  }
}

void ByteGrid::ReAlloc( const Region2& r ) {
  byte* data = new byte[r.nelem()*fNBytes];
  if(fData) {
    CopyDataTo(data,fNBytes,r);
    Free();
  } else { memset(data, 0, r.nelem()*fNBytes ); }
  setregion(r);
  fData = data;
}

