#ifndef __ByteGrid__
#define __ByteGrid__

#include "Point1.h"
#include "Point2.h"
#include "Region1.h"
#include "Region2.h"
#include "Environ.h"
#include "Utilities.h"

typedef class ByteGrid ByteGrid;

//----------------------------------------------------------------------------------------
//						ByteGrid
//----------------------------------------------------------------------------------------

class ByteGrid : public GeoRegion2
{
protected:
  byte* fData;
  int fNBytes;
  unsigned long int fMaxVal;
  
public:
                         // Constructors

  ByteGrid(int r0, int c0, int r1, int c1, int bytes, int r_inc = 1, int c_inc = 1);
  ByteGrid(const ByteGrid& md);

  inline ByteGrid() : GeoRegion2() { fNBytes= 1; fData = NULL; fMaxVal=254; }
  
  inline ByteGrid(const Region2& r) : GeoRegion2(r) { fNBytes= 1; fData = NULL;  fMaxVal=254; ReAlloc(r); }
  inline ByteGrid(const GeoRegion2& r) : GeoRegion2(r) { fNBytes= 1; fData = NULL;  fMaxVal=254; ReAlloc(r); }
  
  inline int BSize() const { return nelem()*fNBytes; }
  inline int NBytes() const { return fNBytes; }
  virtual void SetValue(  int ir, int ic, unsigned long dval );
  int SetRainbowValue(  int ir, int ic, byte dval );

  inline void SetValue( const Point2* point, unsigned long dval ) {
    SetValue((*point)(0),(*point)(1),dval);
  }
  inline ByteGrid* Rescale(int scale) { return this; }

  virtual void Set( byte val ) {
    memset(fData,val,nelem()*fNBytes);  
  }

	virtual inline float FltValue( int ir, int ic ) const { return (float) IntValue(ir,ic);  }
	virtual inline float FltValue( Point2& p ) const  { return (float) IntValue(p);  }
//	virtual inline long IntValue( int ir, int ic ) const  { return IntValue(ir,ic);  }
//	virtual inline long IntValue( Point2& p) const  { return IntValue(p);  }

  inline int NElements() const { return nelem(); }

  inline void SetFields(int rows, int cols, int nbytes, int r_inc = 1, int c_inc = 1) { 
    setupper(rows-1,cols-1); setlower(0,0); setincrement(r_inc,c_inc); 
    fNBytes = nbytes; fMaxVal = Util::pow2(fNBytes*8)-2; 
  }

  inline void SetDimensions( int rows, int cols ) { 
    setupper(rows-1,cols-1); setlower(0,0); 
    fMaxVal = Util::pow2(fNBytes*8)-2; 
  }

  inline void SetFields( const Region2& r, int nbytes, int r_inc = 1, int c_inc = 1) { 
    setregion(r); setincrement(r_inc,c_inc); 
    fNBytes = nbytes; fMaxVal = Util::pow2(fNBytes*8)-2; 
  }
  
  inline int CheckByteSize(int maxValue) const {
    int iexp2;
    frexp(maxValue,&iexp2);
    int nbytes = (iexp2-1)/8 + 1;
    return (nbytes > fNBytes) ? nbytes : -1;
  }
  virtual float Value(Point2& pt) const { return (float) IntValue(pt); }
 
  void CopyDataTo( byte* data, int bytes, const Region2& r) const;
  void CopyDataFrom( const byte* data, int bytes, const Region2& r);
  void CopyDataFrom( const int* data, int bytes, const Region2& r);
  void CopyDataFrom( const float* data, int bytes, float s, float o, const Region2& r);
  
  virtual inline long IntValue( int ir, int ic ) const { 
    if(inside(ir,ic)) {
      byte* bptr = fData + bindex(ir,ic)*fNBytes;
      return Util::dec_Nb( &bptr, fNBytes );
    }
    else return -2;
  }
  
  virtual inline long IValue( int ir, int ic ) const { 
      byte* bptr = fData + bindex(ir,ic)*fNBytes;
      return Util::dec_Nb( &bptr, fNBytes );
  }
  
  virtual inline long IntValue( Point2& p ) const { return IntValue( p(0), p(1) ); }
  virtual inline long IValue( Point2& p ) const { return IValue( p(0), p(1) ); }

  virtual inline byte BValue( int ir, int ic, int ibyte ) const { 
#ifdef DEBUG
    if( !inside(ir,ic) ) { gPrintErr("Illegal Index"); return 0; }
    if( ibyte >= fNBytes ) { gPrintErr("Illegal Index"); return 0; }
#endif
    return fData[ bindex(ir,ic)*fNBytes + ibyte ];
  }
  
  inline byte BValue( int index, int ibyte ) const { 
#ifdef DEBUG
      if( index >= nelem() ) { gPrintErr("Illegal Index"); return 0; }
      if( ibyte >= fNBytes ) { gPrintErr("Illegal Index"); return 0; }
#endif
      return fData[ index*fNBytes + ibyte ];
  }
  
  inline byte BValue( int index ) const { 
#ifdef DEBUG
    if( index >= nelem() ) { gPrintErr("Illegal Index"); return 0; }
    if( fNBytes > 1 ) { gPrintErr("Illegal Index"); return 0; }
#endif
      return fData[ index ];
  }
  
  inline void Free() {
    if(fData) {
	  delete[] fData;
	  fData = NULL;
	}  
  }
  
  inline void Clear() {
    Free();
	fNBytes = 0;
	setempty();
  }
  
  inline virtual void Alloc() {
    Free();
    fData = new byte[BSize()];
  }

  inline virtual void ReAlloc(int bSize) {
	if( (fNBytes != bSize) || (fData == NULL) ) {
	  Free();
	  fNBytes = bSize;
	  fData = new byte[BSize()];
	}
  }
  
  virtual void ReAlloc( int r0, int c0, int r1, int c1, int bytes, int r_inc=-1, int c_inc=-1 );
  virtual void ReAlloc( const Region2& r  );

  virtual inline byte* Data() const { return fData; }	
  
};


#endif

