#if __GNUG__ >= 2
#  pragma implementation
#endif

#include "Externals.h"
#include "Environ.h"
#include "MML_Model.h"

PixXPlex Externals::_EModRecs;

inline static int scmp(const char* a, const char* b) {    // fast string comparison
  int diff = 0;
  while ((diff = *a - *b++) == 0 && *a++ != 0);
  return diff;
}

inline static int scmp_NC2(const char* a, const char* b) {   // fast case-insensitive ( 2nd arg; assumes 1st arg is lower-case ) string comparison
  int diff = 0;
  while ((diff = *a - tolower(*b++) ) == 0 && *a++ != 0);
  return diff;
}

EVarLink::EVarLink( const char* name, Variable* v, CString* modality ) : TNamedObject(name) { 
  _var=v; 
  if( modality == NULL ) {
	_modality = kInOut;
  } else {
	const char* smod = modality->chars();
	if( scmp_NC2("in",smod) == 0 ) {
	  _modality = kIn;
	} else if( scmp_NC2("out",smod) == 0 ) {
	  _modality = kOut;
	}  else if( (scmp_NC2("init-out",smod) == 0) || (scmp_NC2("initout",smod) == 0) ) {
	  _modality = kInitOut;
	}  else if( (scmp_NC2("init-in",smod) == 0) || (scmp_NC2("initin",smod) == 0) ) {
	  _modality = kInitIn;
	}  else if( (scmp_NC2("in-out",smod) == 0) || (scmp_NC2("inout",smod) == 0) ) {
	  _modality = kInOut;
	}  else  {
	  sprintf(gMsgStr, "Unknown External Variable Link Modality: %s for %s",smod,v->Name()); gPrintErr();
	}
  }; 
}

EVarLink::EVarLink( const char* name, Variable* v, EModality modality ) : TNamedObject(name)  { 
  _var=v; 
  _modality = modality; 
}

int EVarLink::is_applicable( EModality modality_context ) {
  switch( modality_context ) {
	case kInitIn:   return  (( _modality == kInitIn ) || ( _modality == kIn ) || ( _modality == kInOut ) );
	case kInitOut:  return  (( _modality == kInitOut) || ( _modality == kOut ) || ( _modality == kInOut ) );
	case kIn:       return  (( _modality == kIn ) || ( _modality == kInOut ));
	case kOut:      return  (( _modality == kOut ) || ( _modality == kInOut )); 
	case kInOut:    return  (( _modality == kIn ) || ( _modality == kOut ) || ( _modality == kInOut )); 
  }	
} 
  
Bool EVarLink::isInput() {
  switch( _modality ) {
	case kInitIn:   return  True;
	case kInitOut:  return  False; 
	case kIn:       return  True;
	case kOut:      return  False; 
	case kInOut:    return  True; 
  }	
}

EVarLink* ALRec::getVarLink( const char* var_name, const char* module_name ) {
  if(  !_external_var_links.empty () ) {
	for( int ei=0; ei < _external_var_links.length(); ei++ ) {
		EVarLink* link = (EVarLink*)_external_var_links.elem(ei);
		Variable* v = link->getVar(); 
		if( scmp( v->Name(), var_name ) == 0 ) {
		  Module* m = v->GetModule();
		  const char* mod_name = ((TNamedObject*)m)->Name();
		  if(( module_name == NULL ) || ( scmp( mod_name, module_name ) == 0 ) ) {
			return link;
		  }
		}
	}
  }
  return NULL;
}

EVarLink* ALRec::getVarLink( Variable* v ) {
  if(  !_external_var_links.empty () ) {
	for( int ei=0; ei < _external_var_links.length(); ei++ ) {
		EVarLink* link = (EVarLink*)_external_var_links.elem(ei);
		if( link->getVar() == v ) {
			return link;
		}
	}
  }
  return NULL;
}

ALRec* EModRec::get_ALRec( int layer_index ) {
  ALRec* alrec = NULL;
  for( Pix p = _activationLayerRecs.first(); p; _activationLayerRecs.next(p) ) {
	ALRec* alr = (ALRec*) _activationLayerRecs.elem(p);
	if( alr->get_AL() == layer_index ) { alrec = alr; break; }
  }
  if( alrec == NULL ) {
	alrec = new ALRec(layer_index);
	_activationLayerRecs.add_high(alrec);
  }
  return alrec;
}

int EModRec::init() {
  if( _activated == 1 ) {
	if( scmp( Name(), "om3" ) == 0 ) {
#ifdef OM3
	  om3_init( Env::DataPath().chars(), Env::ArchivePath().chars(),  _args.NArgs(), _args.Args() );
	  _activated = 2;  
#else 
	  gFatal("OM3 support not built");
#endif
	}
  }
  return _activated;
}

int EModRec::run( double end_time ) {
  if( _activated == 2 ) {
	if( scmp( Name(), "om3" ) == 0 ) {
#ifdef OM3
	  double om3_end_time = translate_time_units( end_time,  ExSchedule::kSecond );
	  if( gDebug ) {
		sprintf(gMsgStr,"Running om3 model to time %f ( %f sec ), dt = %f sec", end_time, om3_end_time, om3_end_time - _time ); 
		gPrintScreen();
	  }
	  om3_run( om3_end_time ); 
	  _time = om3_end_time;
	  return 1;
#endif
	}
  }
  return 0;
}

int EModRec::finalize() {
  int cnt=0;
  if( _activated > 1 ) {
	if( scmp( Name(), "om3" ) == 0 ) {
#ifdef OM3
	  om3_finalize(); 
	  return 1;
#endif
	}
  }
  return 0;
}

double EModRec::translate_time_units( double time,  ExSchedule::ETimeUnit to_unit ) {
  if( _unit == ExSchedule::kUndefined ) {
	ExSchedule* schedule = Model::I0().GetSchedule();
	_unit = (ExSchedule::ETimeUnit) schedule->GetObjInfo(ExSchedule::kUnit);	
	if( _unit == ExSchedule::kUndefined ) { gFatal("No time unit specified in Model OT(DT,Start,Stop,Repeat,Unit) config command" ); }
  }
  if( _unit == to_unit ) {
	return time;
  } else if ( _unit > to_unit ) {
	switch( to_unit ) {
	  case 	ExSchedule::kSecond: return translate_time_units( time*60,  ExSchedule::kMinute ); 
	  case 	ExSchedule::kMinute: return translate_time_units( time*60,  ExSchedule::kHour ); 
	  case 	ExSchedule::kHour:   return translate_time_units( time*24,  ExSchedule::kDay ); 
	  case 	ExSchedule::kDay:    return translate_time_units( time*7,  ExSchedule::kWeek ); 
	  case 	ExSchedule::kWeek:   return translate_time_units( time*52.143,  ExSchedule::kYear ); 
	}; 
  } else {
	switch( to_unit ) {
	  case 	ExSchedule::kMinute: return translate_time_units( time/60,  ExSchedule::kSecond ); 
	  case 	ExSchedule::kHour:   return translate_time_units( time/60,  ExSchedule::kMinute ); 
	  case 	ExSchedule::kDay:    return translate_time_units( time/24,  ExSchedule::kHour );  
	  case 	ExSchedule::kWeek:   return translate_time_units( time/7,   ExSchedule::kDay ); 
	  case 	ExSchedule::kYear:   return translate_time_units( time/52.143,  ExSchedule::kWeek ); 
	}; 
  }
}


void  EModRec::WriteDataToConfig( CStream& outStream, Variable* var ) {
  if( var == NULL ) {
	outStream << " AM(" << Name();
	for( int i=0; i<_args.NArgs(); i++ ) {
	  outStream << "," << _args.elem(i);
	}
	outStream << ")";
  } else {
	for( Pix p = _activationLayerRecs.first(); p; _activationLayerRecs.next(p) ) {
	  ALRec* alrec = (ALRec*) _activationLayerRecs.elem(p);
	  PixXPlex& ext_var_links = alrec->get_var_links();
	  if(  !ext_var_links.empty () ) {
	  for( int ei=0; ei < ext_var_links.length(); ei++ ) {
		  EVarLink* link = (EVarLink*)ext_var_links.elem(ei);
		  Variable* v = link->getVar(); 
		  if(  var == v  ) {
			outStream << " EL(" << Name() << "," << link->Name() << ",";
			switch( link->getModality() ) {
			  case EVarLink::kInitIn:  outStream << "initin";  break;
			  case EVarLink::kIn:      outStream << "in";      break;
			  case EVarLink::kInitOut: outStream << "initout"; break; 
			  case EVarLink::kOut:     outStream << "out";     break;
			  case EVarLink::kInOut:   outStream << "inout";   break;
			}
			outStream << ")";
		  }
		}
	  }
	}  
  }
}

int  EModRec::update_variables( EVarLink::EModality modality, Module* m, Variable* var ) { 
  for( Pix p = _activationLayerRecs.first(); p; _activationLayerRecs.next(p) ) {
	ALRec* alrec = (ALRec*) _activationLayerRecs.elem(p);
	PixXPlex& ext_var_links = alrec->get_var_links();
	if(  !ext_var_links.empty () ) {
	  for( int ei=0; ei < ext_var_links.length(); ei++ ) {
		EVarLink* link = (EVarLink*)ext_var_links.elem(ei);
		if( link->is_applicable( modality ) ) {
		  Variable* v = link->getVar(); 
		  if( ((var==NULL) || ( var == v )) && (( m==NULL) || ( m == v->GetModule())) ) {
			v->UpdateValueExternal(Name(),link->Name(),modality);
		  }
		}
	  }
	}
  }
  return 0;
}

int  EModRec::has_Externals( EVarLink::EModality modality, Module* m, Variable* var  ) { 
  for( Pix p = _activationLayerRecs.first(); p; _activationLayerRecs.next(p) ) {
	ALRec* alrec = (ALRec*) _activationLayerRecs.elem(p);
	PixXPlex& ext_var_links = alrec->get_var_links();
	if(  !ext_var_links.empty () ) {
	  for( int ei=0; ei < ext_var_links.length(); ei++ ) {
		EVarLink* link = (EVarLink*)ext_var_links.elem(ei);
		if( link->is_applicable( modality ) ) {
		  Variable* v = link->getVar(); 
		  if( ((var==NULL) || ( var == v )) && (( m==NULL) || ( m == v->GetModule())) ) {
			return 1;
		  }
		}
	  }
	}
  }
  return 0;
}

EVarLink* Externals::get_VarLink( Variable* v, const char* ext_mod_name) {
  if( v== NULL ) return NULL;
  for( Pix p = _EModRecs.first(); p; _EModRecs.next(p) ) {
	EModRec* emr = (EModRec*) _EModRecs.elem(p);
	if( (ext_mod_name == NULL ) || (scmp(emr->Name(),ext_mod_name) == 0) ) { 
	  ALRec* alrec = emr->get_ALRec(  v->activationLayerIndex() );
	  if ( alrec != NULL ) {
		EVarLink* vL = alrec->getVarLink( v );
		if( vL != NULL ) return vL;
	  }
	}
  }
  return NULL;
}

EModRec* Externals::get_EModRec( const char* ext_mod_name, Bool create, ArgArray* args ) {
  EModRec* emod = NULL;
  for( Pix p = _EModRecs.first(); p; _EModRecs.next(p) ) {
	EModRec* emr = (EModRec*) _EModRecs.elem(p);
	if( scmp(emr->Name(),ext_mod_name) == 0 ) { emod = emr; break; }
  }
  if( emod == NULL && create ) {
	emod = (args==NULL) ? new EModRec(ext_mod_name) : new EModRec(ext_mod_name,*args);
	_EModRecs.add_high(emod);
  }
  return emod;
}


int Externals::initialize_module( const char* name, ArgArray* args, Bool exec_init ) {  
  if (name != NULL) { 
	EModRec* emr = get_EModRec( name, True, args );
	if( gDebug ) {
	  sprintf(gMsgStr,"Initializing external module: %s",name); gPrintScreen();
	}
	if( exec_init ) {
	  return emr->init();
	}
  }
  return 0;
}


void  Externals::run( double end_time ) {
  for( Pix p = _EModRecs.first(); p; _EModRecs.next(p) ) {
	EModRec* emr = (EModRec*) _EModRecs.elem(p);
	emr->run(end_time);
  }
}

void  Externals::finalize() {
  for( Pix p = _EModRecs.first(); p; _EModRecs.next(p) ) {
	EModRec* emr = (EModRec*) _EModRecs.elem(p);
	emr->finalize();
  }
}

void  Externals::WriteDataToConfig( CStream& outStream, Variable* var ) {
  for( Pix p = _EModRecs.first(); p; _EModRecs.next(p) ) {
	EModRec* emr = (EModRec*) _EModRecs.elem(p);
	emr->WriteDataToConfig(outStream,var);
  }
}

EVarLink* Externals::register_variable_link( const char* ext_mod_name, const char* ext_var_name, Variable* var, CString *modality  ) {
  float* data = NULL;
  if (ext_mod_name != NULL) {
	int found_ext_var = 0;
#ifdef OM3 
	if( scmp("om3",ext_mod_name) == 0 ) {
	  found_ext_var = om3_set_current_data_array( ext_var_name, 0 );
	}
#endif
	if( found_ext_var ) {
#ifdef OM3 
	  om3_tracer_name( om3_get_tracer_index(), var->Name() );
#endif
	  EVarLink* link = new EVarLink(ext_var_name,var,modality);
	  ALRec* rec = get_ALRec( ext_mod_name, var->activationLayerIndex() );
	  rec->addLinkRec( link );
	  var->SetF(FLinkEdges,True,"register_variable_link");
	  return link;
	} 
	sprintf(gMsgStr, "Can't link Variable %s to external variable %s:%s",var->Name(),ext_mod_name,ext_var_name); gPrintErr();
  }
  return NULL;
}

int  Externals::has_Externals( EVarLink::EModality modality, Module* m, const char* ext_mod_name, Variable* var  ) {
  for( Pix p = _EModRecs.first(); p; _EModRecs.next(p) ) {
	EModRec* emr = (EModRec*) _EModRecs.elem(p);
	if ( (ext_mod_name == NULL) || (scmp(emr->Name(),ext_mod_name) == 0) ) {
	  if( emr->has_Externals(  modality, m, var ) ) return 1;
	}
  }
  return 0;
}	

int  Externals::update_variables( EVarLink::EModality modality, Module* m, const char* ext_mod_name, Variable* var ) {
  if( (modality == EVarLink::kInitIn) ||  (modality == EVarLink::kInitOut) ||  (modality == EVarLink::kOut) ) { m = NULL; }
  for( Pix p = _EModRecs.first(); p; _EModRecs.next(p) ) {
	EModRec* emr = (EModRec*) _EModRecs.elem(p);
	if ( (ext_mod_name == NULL) || (scmp(emr->Name(),ext_mod_name) == 0) ) { 
	  emr->update_variables(modality,m,var);
	}
  }
  return 0;
}


int Externals::get_partition_type() {
#ifdef OM3
  EModRec* erec;
  if(  (erec = get_EModRec( "om3" )) != NULL ) {
	return 1;
  }
#endif
  return -1;
}

int Externals::get_partition_dim( int dimension ) {
#ifdef OM3
  EModRec* erec;
  if(  (erec = get_EModRec( "om3" )) != NULL ) {
	return om3_get_proc_dim(dimension); 
  }
#endif
  return -1;
}

int Externals::get_grid_dim( int dimension ) {
#ifdef OM3
  EModRec* erec;
  if(  (erec = get_EModRec( "om3" )) != NULL ) {
	return om3_get_grid_dim(dimension); 
  }
#endif
  return -1;
}
		

