//----------------------------------------------------------------------------------------
//      MML.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History
//----------------------------------------------------------------------------------------

#ifndef __MML__
#define __MML__
  
#include "Utilities.h"
#include "SList.h"
#include "StringPlex.h"
#include "floatVec.h"

enum EAttributeValueType { kAVUnknown, kAVFloat, kAVInt, kAVString, kAVString2  };
enum EProcessContext { kPCSetup, kPCDependencies, kPCMMLGen, kPCCodeGen, kPCDerivatives };
enum EDependencyMode { kDMSerial, kDMParFunction, kDSpatialFunction, kArrayFunction, kDMRelCode, kFluxDep };

typedef class TNode TNode;
typedef class  TEquationNode TEquationNode;
typedef class TDeclarationNode TDeclarationNode;
typedef class TCommandNode TCommandNode;
typedef class TAttributeNode TAttributeNode;

class MML_AttributeValue {
	EAttributeValueType _valType;
	long int _intVal;
	float _floatVal;
	const CString* _stringVal;
public:
	inline operator const CString*() {
#ifdef SME3_DBG
		if( _valType != kAVString) { gPrintErr("Illegal string Value Access in MML_AttributeValue"); }
#endif 
		return  _stringVal; 
	}
	inline operator int() {
#ifdef SME3_DBG
		if( _valType != kAVInt ) { gPrintErr("Illegal int Value Access in MML_AttributeValue"); }
#endif 
		return  (int)_intVal; 
	}
	inline operator long() {
#ifdef SME3_DBG
		if( _valType != kAVInt ) { gPrintErr("Illegal int Value Access in MML_AttributeValue"); }
#endif 
		return _intVal; 
	}
	inline operator float() { 
#ifdef SME3_DBG
		if( _valType != kAVFloat ) { gPrintErr("Illegal float Value Access in MML_AttributeValue"); }
#endif
		return  _floatVal; 
	}
	inline void GetStringVal(CString& s) {
		switch(_valType) {
			case kAVFloat: s = _floatVal; break;
			case kAVInt: s = ""; s.appendIndex(_intVal); break;
			case kAVString: s = *_stringVal; break;
		}
	}
	inline void SetInt(long i) { _intVal = i; _valType = kAVInt; }
	inline void SetFloat(float i) { _floatVal = i; _valType = kAVFloat; }
	inline void SetString(const CString* s) { _stringVal = s; _valType = kAVString; }
	inline void Clear() { _valType = kAVUnknown; }
};

//----------------------------------------------------------------------------------------
//						TMMLData
//----------------------------------------------------------------------------------------

#define kMMLMaxPackSize 512

class TMMLData {

protected:

  char fPackBuf[kMMLMaxPackSize];

  ArgArray fCArg;
  ArgArray fName;
  CString fCmd;
  ArgArray fValue;
  byte fInfo[8];

public:

  enum EStatus { kRequired, kOptional };
  enum EInfo { fData0, fData1, fData2, kStatus, kListLevel  };

  TMMLData ()  { memset(fInfo,0,8); }
  int Pack();
  void UnPack();
  inline byte Info(EInfo index) { return fInfo[index]; }
  inline void SetInfo( EInfo index, byte val ) { fInfo[index] = val; }

  ArgArray& Args() { return fCArg; }
  ArgArray& Values() { return fValue; }
  ArgArray& Names() { return fName; }

  const CString* Arg( int index, EStatus status = kOptional ) const ;
  const CString* Value( int index=0, EStatus status = kRequired ) const ;
  int IntArg( int index, int& arg, EStatus status = kOptional );    
  int FloatArg( int index, float& arg, EStatus status = kOptional );

  inline const CString* Name( EStatus status = kOptional ) const { 
    if( fName.NArgs() == 0 ) { if( status == kRequired ) gPrintErr( "Syntax Error, Missing Name."); return NULL; }
    return &fName.Top(); 
  }

  inline const CString* Name( int index, EStatus status = kOptional  ) const { 
    if( fName.NArgs() <= index ) { if( status == kRequired ) gPrintErr( "Syntax Error, Missing Name."); return NULL; }
    return &fName[ index ];
  }

  inline CString& Cmd() { return fCmd; } 

};

//----------------------------------------------------------------------------------------
//						TMMLObject
//----------------------------------------------------------------------------------------

class MMLObject {

protected:

  CString fDoc;
  TNode* fNode;
  static MML_AttributeValue fAttrValue;
	int  GetAttributeStringList( const TNode* n, ArgArray& sl ); 
	int  GetAttributeFloatList( const TNode* n, floatVec& fv, int sublist_index = -1 );
	int  GetIndexedAttrValue( const TNode* n, int index, EAttributeValueType av = kAVUnknown, TMMLData::EStatus status = TMMLData::kOptional );
	int  GetAttrValue( const TNode* n, EAttributeValueType av0 = kAVUnknown, EAttributeValueType av1 = kAVUnknown  );
	int  NodeTypeError( const TNode* n);
	int  MissingValueError( const TNode* n, TMMLData::EStatus status);
  
public:

  enum EWriteMode { kDeclaration, kDefinition };
  enum ELevel { kTop, kMod, kVar };

	MMLObject()  { fNode=NULL; }
  virtual int AddModifier( const CString& name, Bool isExtension = False ) { return 0;}
  virtual int GetModifierString( CString& name ) const { return 0; }
  int ProcessNode( TNode* n = NULL );
  virtual int RunNodeProcessing( EProcessContext pc ) { return 0;}
	inline void SetNode( TNode* n ) { fNode = n; }
	inline TNode* GetNode() { return fNode; }
  void AddQualifiers( const TDeclarationNode* dn, int isExtension );
  void remove_special_characters( char* str ) const; 
	virtual int ProcessCommand(const TCommandNode* n) { return 0;}
	virtual int ProcessAttribute(const TAttributeNode* n) { return 0;}
	virtual int ProcessGeneral(TNode* n) { return 0;}
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) = 0;
  virtual void WriteCCode( FILE* CHdrFile, FILE* CEqnFile) {;}
  MML_AttributeValue& AttrValue() { return fAttrValue; }
	inline void  SetDoc( CString& doc ) { fDoc = doc; }  
	inline CString&  GetDoc() { return fDoc; }

};

#endif
