//----------------------------------------------------------------------------------------
//	Event.h;
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.;
//	Change History:;
//----------------------------------------------------------------------------------------;
  
#ifndef __MML_Event__
#define __MML_Event__

extern "C" {
#include "time.h"
}
  
#include "Globals.h"
#include "MML_Schedule.h"
#include "MML_Function.h"
#include "ConfigObjects.h"
#include "MML.h"
#include "MML_Time.h"

typedef class Variable Variable;
typedef class Module Module;  
typedef class TEvent TEvent;
typedef class TPipe TPipe;
typedef class TNode TNode;
typedef class TDerivative TDerivative;
typedef class TFunctionNode TFunctionNode;
typedef class TGraphNode TGraphNode;
typedef class TStringListNode TStringListNode;

//----------------------------------------------------------------------------------------
//						TDependency
//----------------------------------------------------------------------------------------

class TDependency : public TNamedObject
{  
  Variable* fVar;
  TConstant* fConstant;
  CString fRelCode;
  char 	       		fFunction;     		// Special Function Code;

public:

  const EInfoIndex kDependencyMode;

  TDependency(Variable& var,int offset);
	TDependency(TConstant& var);
	Variable* GetVar() { return fVar; }
  TConstant* Constant() { return fConstant; }
  inline char& Function() { return fFunction; }
  const Variable* Var() const { return fVar; }
  CString* RelCode() { if(GetObjInfo(kDependencyMode) == kDMRelCode ) return &fRelCode; else return NULL; }
//  void SetRelCode(CString& value) { fRelCode = value; SetObjInfo(kDependencyMode,kDMRelCode); }
  void SetRelCode(const char* value) { fRelCode = value; SetObjInfo(kDependencyMode,kDMRelCode); }
};

//----------------------------------------------------------------------------------------
//						TCommand 
//----------------------------------------------------------------------------------------

class TCommand : public MMLObject, public TNamedObject
{
public:

  enum EStatus { kActive, kInActive };
  enum EType { kUndefined, kInit, kUpdate, kIntegrate, kFunction, kDerivative };
  enum EExType { kNoEx, kCode, kScript, kPipe, kImport, kExternalUpdate };
  enum EDepIndex { kNone, kDepSet, kHasMethodCalls, kHasParallelDep, kOptimized, kParameter };
 
  static EInfoIndex kStatus;
  static EInfoIndex kType;
  static EInfoIndex kDepIndex;
  static EInfoIndex kExType;

protected:

  Variable* fVar;
  CString fTranslation;
  TDerivative* fDerivative;
  CString fCode;
  ExSchedule* fSchedule;	
  TNamedObjectList fDependencyList;
  char fFunction;     		// Special Function Code;
  int fExecuteCnt;
  int fCodeGenMode;
  clock_t fTotalClicks;
  TOrderedObjectDLList* fArrayNodes;
  signed char fArrayOrder;
  byte fHasArraysInEqn;

  byte fCommandInfo[8];

  void PrintCloseSpaceLoop(FILE* CFile, int ntabs = 1 ) const;
  void PrintOpenSpaceLoop(FILE* CFile, int ntabs = 1 ) const; 
  void PrintCloseOptSpatialLoop( FILE* CEqnFile, int ntabs  ) const ;
  void PrintOpenOptSpatialLoop( FILE* CEqnFile, int ntabs  ) const ;

public:
 
  static ArgArray kTypeList;  // ( "undefined", "init", "update", "integrate", "function" );
  static ArgArray kExTypeList;  // ( "undefined", "code", "script", "pipe", "import" );

  inline int SetType( const CString& mod, Bool isExtension = False ) {
    int index = kTypeList.Find(mod);
	if( isExtension && index != GetObjInfo(kType) ) gPrintErr(" Error, can't change Variable Type in Extension.");
	else  if(index>0) { SetObjInfo(kType, index ); }
	if( index <= 0 ) {
		int eindex = kExTypeList.Find(mod);
		if( isExtension && eindex != GetObjInfo(kExType) ) gPrintErr(" Error, can't change Variable Type in Extension.");
		else  if(eindex>0) { SetObjInfo(kExType,eindex); }
	} else SetObjInfo(kExType,kCode);
    return index;
  }

  enum EIntegratonMethod { kEuler, kRK2, kRK4 };
  enum EIntegrationInfo { kIntegrationMethod, kTimeMode, kClamp };
  enum ECommandInfo { kBreak=4, kPosts, kFluxType };

  TCommand(Variable* v, const CString& val, EType type = kUndefined  )  
    { fVar = v; SetObjInfo(kType,type); fFunction = 0; fTotalClicks = 0; fArrayOrder = -1; fHasArraysInEqn = 0;
	  fArrayNodes = NULL; memset(fCommandInfo,0,8);	fSchedule = NULL; fExecuteCnt = 0; fCodeGenMode=0; fDerivative = NULL;
    }
  TCommand( const CString& name, Variable* v, EType type = kUndefined  ) : TNamedObject(name)
    { fVar = v; SetObjInfo(kType,type); fFunction = 0; memset(fCommandInfo,0,8);  fArrayOrder = -1;	fHasArraysInEqn = 0;
	 fArrayNodes = NULL; fSchedule = NULL;	fExecuteCnt = 0; fCodeGenMode=0; fDerivative = NULL; fTotalClicks = 0;
		}
  TCommand( const TCommand& c  ) 	
		{ fVar = c.fVar;  SetObjInfo(kType,c.GetObjInfo(kType)); 
		 fArrayNodes = c.fArrayNodes; fFunction = c.fFunction; memcpy(fCommandInfo,c.fCommandInfo,8);  fArrayOrder = -1; fHasArraysInEqn = 0;
			fSchedule = NULL;	fExecuteCnt = 0;  fCodeGenMode=0; fDerivative = NULL; fTotalClicks = 0;
		}

	~TCommand();
	
  inline char& GetFunction() { return fFunction; }
  inline char Function() const { return fFunction; }
  CString& getValueString( int code_index= -1, int array_index=-1  );
  inline CString& Value() { return getValueString( -1 ); }
  void SetValue( CString& value );
  byte  ArrayOrder() { return  fArrayOrder; }

	void AddArrayNode( TEquationNode* n );
	inline TOrderedObjectDLList* getArrayNodes() { return fArrayNodes; }
  
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void WriteXML( FILE* oFile );
   virtual int GetModifierString( CString& name ) const ;
   virtual int GetXMLAttributes( CString& name ) const ;
  virtual void Dump( FILE* oFile );
  int AddModifier( const CString& mod, Bool isExtension = False );
  void SetIntegrationInfo(EIntegrationInfo index, byte value) { fCommandInfo[index] = value; }
  byte GetIntegrationInfo(EIntegrationInfo index) const { return fCommandInfo[index]; }
  TNamedObjectList& DependencyList() const { return (TNamedObjectList&) fDependencyList; }

  const CString& VarName() const;
//  inline CString&  Value() { return fValue[ fCodeGenMode ]; }
//  inline  void SetValue(CString& val) { fNode=NULL; for(int i=0; i<4; i++ ) fValue[ i ] = val; }
  inline Variable* GetVar() { return fVar; }
	virtual const Module* GetModuleC();
	virtual Module* GetModule();
  inline Variable* Var() { return fVar; }
  inline const Variable* VarC() const { return fVar; }
  inline int operator != ( TCommand& c ) { return ( fVar != c.fVar ); }
  inline int operator == ( TCommand& c ) { return ( fVar == c.fVar ); }
  void PrintDependencies(FILE* oFile) const;
  int AddPipeDependencies( Pipe* p );
  inline virtual const Pipe* GetPipeC() const { return NULL; }
	inline virtual Pipe* GetPipe() { return NULL; }
//  virtual void WriteCCode0( FILE* CHdrFile, FILE* CEqnFile) const;
//  virtual void WriteInnerCCode( FILE* CHdrFile, FILE* CEqnFile, TEquationNode* en );
  TEquationNode* getEquationNode(EType etype, TStringListNode* args); 
  int WriteSwitchCCode(  FILE* CEqnFile, int mode, Bool isFunction=False  );
  virtual void WriteCCode( FILE* CHdrFile, FILE* CEqnFile);
  void WriteCCommandCode(FILE* CEqnFile);
  virtual inline int Execute(Module* m = NULL) { return 0;}
  Pix AddDependency( Variable& v, const char* relcode = NULL ) ;
  Pix AddDependency( TTime t );
  Pix AddDependency( TConstant& c ) ;  
  virtual int Config( TConfigData & cd ); 
  
	int ProcessNodes(TNode* n, EProcessContext pc);  
  int ProcessNodeRecur(TNode* n, EProcessContext pc) ;
  
  virtual int ProcessCommand(const TCommandNode* n);
	virtual int ProcessAttribute(const TAttributeNode* n);
	
 	virtual inline int ProcessGeneral(TNode* n) { return ProcessNodeRecur(n,kPCSetup); }
 	virtual int RunNodeProcessing( EProcessContext pc ); 
	 
	int ProcessSetup(TNode* n);
	int ProcessDependencies(TNode* n);
	int ProcessDerivatives(TNode* n);
	int ProcessFunction(TFunctionNode* fn); 
	int ProcessGraph(TGraphNode* n); 
	    
	int ExecuteOK( byte event_type );
  void SetBreak( int breakIndex = 1 );
  inline int Break() { return fCommandInfo[kBreak]; }

	inline void Post() { fCommandInfo[kPosts]++; }
	inline void UnPost() { fCommandInfo[kPosts]--; }
	inline int NPost() { return fCommandInfo[kPosts]; }
	inline const ExSchedule* ScheduleC() const { return fSchedule; }	
	inline ExSchedule* Schedule() { return fSchedule; }	
	int SetTimeSchedule(float dt, float time);
	int SetStepSchedule(int dt, int step );
	int Optimizable() const;
	virtual word class_id() const;
	inline CString& Code() { return fCode; }
	virtual int Finalize() { return 0; }
	inline  int CodeGenMode() { return fCodeGenMode; }
	inline  TDerivative*& Derivative() { return fDerivative; }
	int getCommandInfoTextString( char* info, int len );

};
//----------------------------------------------------------------------------------------
//						TPipeCommand 
//----------------------------------------------------------------------------------------


class TPipeCommand : public TCommand
{

  Pipe* fPipe;
  static int fPipeCmdIndex;

public:

  TPipeCommand( const CString& name, Variable* v, Pipe* p=NULL  )  : TCommand(name,v) { fPipe = p; SetObjInfo(kExType,kPipe); fCode = "Pipe";}
  TPipeCommand( const TCommand& c  )  : TCommand(c) { fPipe = NULL; fCode = "Pipe"; }
  ~TPipeCommand();
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual int GetModifierString( CString& name ) const  { return TCommand::GetModifierString(name); }
  virtual void Dump( FILE* oFile ){;}

  int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  inline void AddPipe(Pipe* p) { fPipe = p; }
  inline virtual const Pipe* GetPipeC() const { return fPipe; }
  inline virtual Pipe* GetPipe() { return fPipe; }
  static TPipeCommand* CreatePipeCommand( TConfigData& cd, Pipe* p, Variable* v ); 
  
  static inline TPipeCommand* CreatePipeCommand( const char* n, Pipe* p, Variable* v ) {
		CString name(n); name += "_c__"; name.appendIndex(fPipeCmdIndex++);
		return new TPipeCommand(name,v,p);
  }
  
	virtual int Config( TConfigData & cd ); 

  virtual int Execute(Module* m = NULL) ;
  
	virtual int Finalize() { return (fPipe) ? fPipe->Finalize() : 0;  }
};

//----------------------------------------------------------------------------------------
//						TExecuteCommand 
//----------------------------------------------------------------------------------------

#ifndef _Execute_typedefs
#define _Execute_typedefs 1
typedef int  (*ExecuteFn)( Variable* obj );
#endif

int IGNORE_exe_g( Variable* obj );

class TExecuteCommand : public TCommand
{

  ExecuteFn fEF;

  
public:

  TExecuteCommand( const CString& name, Variable* v, EType type=kUndefined, ExecuteFn ef=NULL )  
    : TCommand(name,v,type)  { 
    fEF = ef;  
    SetObjInfo(kExType,kCode);
    fFunction = 0;
  }
  TExecuteCommand( const TExecuteCommand& c  )  : TCommand(c) { fEF = c.fEF; SetObjInfo(kExType,kCode);  }

  virtual int Execute( Module* m = NULL ) ;
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) { TCommand::WriteMML( oFile, mode ); }
  virtual int GetModifierString( CString& name ) const { return TCommand::GetModifierString(name); }
  virtual void Dump( FILE* oFile ){;}

  int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  void SetExecutable(ExecuteFn ef) { fEF = ef; }
};

//----------------------------------------------------------------------------------------
//						TImportCommand 
//----------------------------------------------------------------------------------------

int IMPORT_exe_g( Variable* obj );

class TImportCommand : public TCommand
{

public:

  TImportCommand( const CString& name, Variable* v )  
    : TCommand(name,v)  { 
    SetObjInfo(kExType,kImport); 
  }
  TImportCommand( const TImportCommand& c  )  : TCommand(c) { 
    SetObjInfo(kExType,kImport); 
    SetObjInfo(kType,kUndefined); 
  }
  virtual int Execute( Module* m = NULL );
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual int GetModifierString( CString& name ) const  { return TCommand::GetModifierString(name); }
  virtual void Dump( FILE* oFile ){;}

  int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
};


 //----------------------------------------------------------------------------------------
//						TScriptCommand 
//----------------------------------------------------------------------------------------


class TScriptCommand : public TCommand
{
 Module* fModule;
 
public:

  TScriptCommand( const CString& name, Module* m );
  TScriptCommand( const CString& name, Variable* v );
  TScriptCommand( const TScriptCommand& c  );
  int Execute( Module* m = NULL );
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition ) {;}
  virtual int GetModifierString( CString& name ) const { return TCommand::GetModifierString(name); }
  virtual void Dump( FILE* oFile ){;}
  int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
	inline CString& Script() { return getValueString(); }
	inline virtual Module* GetModule() { return fModule; }
};

//----------------------------------------------------------------------------------------
//						TEvent 
//----------------------------------------------------------------------------------------

class TEvent : public TNamedObject, public MMLObject
{

public:

  enum EType { kUnknown, kInit, kFirstUpdate, kFinalUpdate, kIntegrate, kImport, kGlobalUpdate }; 
  enum ECmdPosition { kEnd, kBeginning }; 
  enum ESplitConsequence { kDoNothing, kDelete, kDeActivate };

  static EInfoIndex kType;
  static EInfoIndex kSpatial;
  
  static ArgArray kTypeList;  // ( " ", "init", "update", "integrate", "import" );

  inline int SetType( const CString& mod, Bool isExtension = False ) {
    int index = kTypeList.Find(mod);
    if( index > 0 ) {
      if( isExtension && index != GetObjInfo(kType) ) gPrintErr(" Error, can't change Varaible Type in Extension.");
      else  SetObjInfo(kType,index);
    }
    return index;
  }

private:
  
  ExSchedule* fSchedule;
  Module* fModule;
  TNamedObjectList fCommandList;
  float fTime;
  Pix fEventListPosted;
  Pix fCurrentCommand;
  Pix fLastExecutedCommand;

 public:

  enum EScope { kModuleS, kListS, kModelS };
  
  TEvent( const char* name, Module* aModule  );
  ~TEvent() { ; }
  
  inline int AddModifier(const CString& mod, Bool isExtension = False ) {
    if( SetType( mod, isExtension) ) { return 1; }
    return 0;
  }

  virtual int GetModifierString( CString& mod ) const  {
    mod += kTypeList[ GetObjInfo(kType) ];
    return 1;
  }

  Pix  AddCommand( Variable* v, const CString& val );
  Pix  AddCommand( TCommand &c, ECmdPosition pos = kEnd );
  inline Module* GetModule() { return  fModule; }

  void SetStatus( TCommand::EStatus  s, Variable* var = NULL );
  void SetStatus ( TCommand::EType type = TCommand::kUndefined, TCommand::EStatus status = TCommand::kInActive );

  TNamedObjectList& CommandList() { return fCommandList; }

  TEvent&  operator += ( TEvent& e );
  TEvent&  Merge ( TEvent& e, Bool deepCopy = False, Bool allowDuplNames = True, Bool removeOld = False  );

  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void Dump( FILE* oFile );
  virtual void CreateDependencies( TNamedObjectList& varList );

  inline float Time() { return fTime; }
  inline void SetTime(float t) { fTime = t; }

  void AddDependencies( TEvent& e, Bool inactivate_additions = False );
  void AddFluxDependencies(); 
  int CreateDependency( Pix p0, Pix p1, Bool mark_link = False );              //  p0 is dependent on p1;
  int DeactivateDependency( Pix p0, int link_index );
  void PrintDependencies( FILE* oFile, Variable* var = NULL, int dump=0 ) const;

  Pix FindVarByName( char* name ) const;
  Pix FindVar( Variable* var, Pix offSet=0, int verbose = 0 ) const;

  int Sort() { return fCommandList.SortByLinks(); }
  int IsSortable() { return fCommandList.IsLinkSortable(); }
  void Split( TEvent& e, int splitIndex, enum ESplitConsequence sc = kDelete );
  void SplitSpatially( TEvent& e, enum ESplitConsequence sc = kDelete );
  TCommand*  SplitByDependency( TEvent& ei, TEvent& ed, enum ESplitConsequence sc = kDoNothing, int localVarsOnly = 1 );
  Pix MoveCmd( Pix p0, TEvent& ei, enum ESplitConsequence sc = kDoNothing );

  virtual void WriteCCode( FILE* CHdrFile, FILE* CEqnFile);
  int Execute( int breakIndex=0, int set_context = 0 );
  int  FastExecute();
  int  ExecuteNextCommand( int breakIndex=0, int set_context = 0 );
  int Schedule(TEvent::ECmdPosition cp = TEvent::kEnd );
  int SetSchedule ( ExSchedule* s = NULL );
  
	void Clear();
	void SetCurrentContext(Module*& currentModule, Variable*& currentVariable, TCommand*& currentCommand);
	void RemoveInActiveCommands ( Bool free = False );
	
	virtual int ProcessCommand(const TCommandNode* n);
	virtual int ProcessAttribute(const TAttributeNode* n);

};


#endif
