#include "MML_Model.h"
#include "MML_Module.h"
#include "MML_Variable.h"
#include "MML_Event.h"
#include "MML_Time.h"
#include "ExprBuilder.h"  
#include "sys/stat.h"
#include "Externals.h"
#include "SMMLParser.h"
#include "GOFCalc.h"

#include <signal.h>
int MMLerror (char* s);
int MMLlex(void);
int MMLparse(void);
void MMLinitParse( int debug_level, int context );
		
extern int MMLleng;
extern FILE *MMLin, *MMLout;

#ifdef HAS_SOCKETS

extern "C" {
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <arpa/inet.h>
}

#endif

//========================================================================================
// CLASS TDerivative
//========================================================================================

TDerivative::TDerivative( Variable* indep_var ) : TNamedObject( indep_var->SName() ) {
	fIndepVar = indep_var;
	fWeight = 1.0;
	fMode = 0;
}

Pix TDerivative::AddInitiator( Variable* v0 ) { 
	for( Pix pt = fInitiators.first(); pt; fInitiators.next(pt) ) {
		const Variable* v = (Variable*)&fInitiators(pt); 
		if( v == v0 ) return pt;
	}
	return fInitiators.append(*v0); 
}

//========================================================================================
// CLASS ArrayRec
//========================================================================================

void ArrayRec::WriteDataToConfig( CStream& outStream ) {
  outStream << " Ar( " << fName;
  for( int i=0; i<_args->NArgs(); i++ ) {
	outStream << ", ";
	outStream << _args->elem(i);
  }
  outStream << " ) ";
}

void  ArrayRec::WriteConfigXML( FILE* oFile ) {
	fprintf(oFile,"\n\t\t<config name=\"Ar\">");
	fprintf(oFile,"\n\t\t\t<arg name=\"array_name\"> %s </arg>",fName.chars());
	for( int i=0; i<_args->NArgs(); i++ ) {
	  fprintf(oFile,"\n\t\t\t<arg name=\"element%d\"> %s </arg>",i,_args->elem(i).chars());
	}
	fprintf(oFile,"\n\t\t</config>");
}

//========================================================================================
// CLASS Model
//========================================================================================

Model* Model::fCurrentModel = NULL;
TNamedObjectList  Model::fModelList;
EConfigInfo Model::kDebug = kCInfo0;		
EConfigInfo Model::kOptLevel = kCInfo1;
int Model::fTM = 1;


 //----------------------------------------------------------------------------------------
  //				Model::SignalHandler:
  //----------------------------------------------------------------------------------------

void SME_SignalHandler( int sig );
static int l_interrupt = 0;

void SME_SignalHandler( int sig ) {
	switch( sig ) {
		case SIGINT:  l_interrupt = 1; break;
	}
}

void Model::Interrupt() { 
  l_interrupt = 1; 
}

//----------------------------------------------------------------------------------------
//				  Model
//----------------------------------------------------------------------------------------

Model* Model::fInstance = NULL;

int Model::Init() {
  fNSim = 1;
  SetCInfo(kDebug,kNoConfig);
  fSeed = 4332;
  srand(fSeed); 
  SetCInfo(kOptLevel,0);
//  signal(SIGINT,SME_SignalHandler);
//	signal(SIGINT,SIG_IGN);
  return fNSim;
}

int Model::FrameConfig( TConfigData& cd, Module* m) {
  const CString& cmd = cd.Cmd();
  int iarg=0; 
	if( (iarg =  Frame().Config(cd,m)) > 0 ) {
    LayerConfig* lc = Frame().SetDefaultLayers(fActivationLayer,fLinkLayer);
    if( fLayerConfig == NULL )  { fLayerConfig = lc; }
	}
	return iarg;
}

int Model::Config( TConfigData& cd) {
  const CString& cmd = cd.Cmd();
  CString* s;

  int iarg; float farg; 
	if( (iarg = FrameConfig( cd, NULL )) > 0 ) {
    return iarg;
  } else if(cmd == "Y") {
		 float y, dt;
		 cd.FloatArg(0, y, CD::kRequired);
		 cd.FloatArg(1, dt, CD::kRequired);
		 TTime::SetYear(y,dt);
	} else if(cmd == "UTM") {
		 float c[2];
		 cd.IntArg(0, iarg, CD::kRequired);
		 cd.FloatArg(1, c[0], CD::kRequired);
		 cd.FloatArg(2, c[1], CD::kRequired);
		 if( iarg == 0 ) {
			 fLowerCornerUTM[0] = c[0];
			 fLowerCornerUTM[1] = c[1];
		 } else {
			 fUpperCornerUTM[0] = c[0];
			 fUpperCornerUTM[1] = c[1];
		 } 
	} else if(cmd == "OT") {
	  int rv = fSchedule.Config(cd);
	  if( rv ) {
		TTime::DT( fSchedule.DT() );
		TTime::SetStartTime( fSchedule.StartTime() );
	  }
	  return rv;
  }  else if( cmd == "s" ) {
    cd.IntArg(0, fSeed, CD::kRequired);
    srand(fSeed); 
    return 1;
  } else if( cmd == "ngl" ) {
    cd.IntArg(0, fMaxGhost, CD::kRequired);
    return 1;
  } else if( cmd == "op" ) {
    cd.IntArg(0,iarg,CD::kRequired);
    SetCInfo(kOptLevel,0);
  } else if( cmd == "d" ) {
    if( cd.IntArg(0, iarg, CD::kRequired) > 0 ) SetCInfo(kDebug,iarg);
    Env::SetDebugLevel(iarg);
    return 1;
  } else if( cmd == "ms" ) {
    Env::SetGlobalSort( FALSE ); 
    return 1;
  } else if( cmd == "sp" ) {
	  if( cd.HasArg(0) ) { fSliderParms = *cd.Arg(0);  }      
  } else if( cmd == "comment" ) {
	  if( cd.HasArg(0) ) { fComment = *cd.Arg(0);  }   
   } else if( cmd == "n" ) {
    if( cd.IntArg(0, iarg, CD::kRequired) > 0 ) { fNSim = iarg; }
    return 1;
#ifdef POSTGRESS
  } else if( cmd == "db" ) {
		CString *s0, *s1=NULL, *s2=NULL;
    if( (fDB==NULL) && ((s0 = cd.Arg(0)) != NULL) ) {     // db name
			s1 = cd.Arg(1);                                     // db port
			s2 = cd.Arg(2);                                     // db host
			const char* port = (s1) ? s1->chars() : (const char*)NULL;
			const char* host = (s2) ? s2->chars() : (const char*)NULL;
			fDB = new sme_pg95(*s0, 0, port, host);
			if (fDB==NULL) {
				CString tmp("Can't connect to database: "); tmp += (*s0);
				gFatal(tmp);
			}
			fDBName = *s0;
			if(port) { fDBPort = port; }
			if(host) { fDBHost = host; }
		} else gPrintErr("Must specify dataBase Name in db() command");
#endif
	} else if (cmd == "dc" ) {
    // TBI CME data collection or HDF input datafile
    fDataSource = *cd.Arg(1);
  } else if( cmd == "AM" ) {
    CString *s0, *s1;
    ArgArray args;
	if( (s0 = cd.Arg(0)) != NULL ) {
	  int aindex=1;
	  while( (s1 = cd.Arg(aindex++)) != NULL ) { args.Add( *s1 ); }
	  Externals::initialize_module( s0->chars(), &args, ( Env::GetInfo(Env::kAppType) == Env::kDriver )  ); 
	}
  } else if( cmd == "MPE"  ) {
	  cd.addToString(fGOFCmds); 
	  CString methodName("null");
	  if( (s = cd.Arg(0)) != NULL ) {
		 methodName = *s;
#ifdef HAS_MPE
		GOFCalc::configureGOF( methodName, cd );
#endif
	  }
  } 
  return 0;
}

void  Model::BuildConfigXML( ArgArray& msg, int mode ) {
  msg.clear();
  msg.Add(format("<?xml version=\"1.0\"?>\n"));
  msg.Add(format("\n<!DOCTYPE compound PUBLIC \"-//miiee sme//dtd mml//en\" \"http://iee.umces.edu/SME3/dtd/mml.dtd\">") );
  msg.Add(format("\n<compound id=\"global\" name=\"%s\" >\n",Name()));
  ArgArray tmp;
  for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
	  Module& m = (Module&) fModuleList(p);
	  if( m.GetCInfo(Module::kIgnorable) == 0 ) { 
		  tmp.clear();
		  if( m.BuildConfigXML( tmp, mode ) ) {
			msg += tmp;
		  }
	  }
  }
  msg.Add(format("\n</compound>"));
}

void Model::ReadSliderParms() {
   if( fSliderParms.empty() ) return;
   CPathString data_path = Env::DataPath(); 
   data_path.Add(fSliderParms);
   FILE* infile = fopen(data_path,"r");
   int test; char ctmp; 
   ExprBuilder::SetMode(ExprBuilder::kMML);
   ExprBuilder::ClearNodes();
   CString line_data, args[5], var_name, data_value;
   TListNode* root = NULL;
   if(infile==NULL) { 
	 sprintf(gMsgStr,"No STELLA slider data detected: %s\n", data_path.chars() ); 
	 gPrintScreen(); return; 
   }
   for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
	  Module& m = (Module&) fModuleList(p);
	  m.SetNode(NULL);
   }
   while( test != EOF ) {
		line_data = var_name = data_value = "";
		while( (( test = fgetc(infile) ) != EOF ) && (test != '\n') ) {
		  line_data += (char) test;
		}
		if( line_data != "" ) {
		  if(      line_data[0] == '\225' ) { var_name = "P3_"; }
		  else if( line_data[0] == '\245' ) { var_name = "P3_"; }
		  else if( line_data[0] == '\306' ) { var_name = "P2_"; }
		  else if( line_data[0] == ''    ) { var_name = "P_"; }
		  else { var_name = line_data[0]; }
		  int char_index = 1, in_array=0, nargs = 0;
		  while( (ctmp = line_data[char_index++]) != '\t') {
			if( ctmp == ']' ) { in_array = 0; }
			else if( ctmp == '[' ) { in_array = 1; args[nargs++] = ""; }
			else if( ctmp == ',' ) { args[nargs++] = ""; }
			else if(in_array) { 
			  args[nargs-1] += ctmp;
			} else {
			  var_name += ctmp;
			}
		  }
		  if( var_name != "Years") {
			var_name.upcase(); 
			var_name.gsub(" ","_");
			while( line_data[char_index] == '\t') { char_index++; }
			while( line_data[char_index] != '\t') {
			  data_value += line_data[char_index++];
			}
			if( Util::is_float ( data_value.chars() ) ) {
			  sprintf(gMsgStr,"Setting data from slider value: %s[%s] = %s", var_name.chars(), args[0].chars(),  data_value.chars() ); gPrintScreen(); 
			  TNode* valueNode = ExprBuilder::CreateConstNode( kAVFloat, data_value.chars() );
			  Variable* var = GetVariable(var_name);
			  if( var == NULL ) {
				 sprintf(gMsgStr,"Unrecognized variable %s in slider data file %s\n", var_name.chars(), data_path.chars() ); 
				 gPrintErr(); 
			  } else {
				Module* m = var->GetModule();
				TVarNode* varNode =  (TVarNode*) ExprBuilder::CreateVarNode( var_name ); 
				if( nargs > 0 ) {
				  args[0].upcase();
				  TStringListNode* slNode = (TStringListNode*) ExprBuilder::CreateStringListNode( args[0] );
				  for( int iarg=1; iarg<nargs; iarg++ ) { args[iarg].upcase(); slNode->AddString( args[iarg] ); }
				  varNode->AddMapList( slNode );
				}
				TNode* eNode = ExprBuilder::CreateEquationNode( TEquationNode::kUpdate, varNode, valueNode ); 
				TModuleNode *mnode = (TModuleNode*) m->GetNode();
				if( mnode == NULL ) {
				  TListNode* lnode = (TListNode*) ExprBuilder::CreateListNode( eNode ); 
				  CString mod_name(m->SName()); mod_name.gsub("_module","");
				  mnode = (TModuleNode *) ExprBuilder::CreateModuleNode( mod_name, lnode );
				  m->SetNode(mnode); mnode->SetModule( m  );
				} else {
				  TListNode* lnode = mnode->EqnList();
				  lnode->AddNode( eNode ); 
				}
				if( root == NULL ) {
				  root = (TListNode*) ExprBuilder::CreateListNode( mnode ); 
				} else {
				  root->AddNode( mnode ); 
				}
			  }
			} else {
			   sprintf(gMsgStr,"Bad data for variable %s in slider data file %s: %s\n", var_name.chars(), data_path.chars(), data_value.chars() ); 
			   gPrintErr(); 
			}
		  }
		}
	}
	ProcessNode(root);
}

void  Model::addArrayDeclaration( ArrayRec* array ) {
  _Arrays.append(*array);
}

ArrayRec* Model::getArrayDeclaration( const CString& name, int& index, int& is_dimension_name  ) {
  index = 0; is_dimension_name = 0;
  for( Pix p = _Arrays.first(); p;  _Arrays.next(p) ) {
	ArrayRec& array = (ArrayRec&) _Arrays.elem(p);
	if(array.SName().equals_nocase(name) ) {
	  is_dimension_name = 1;
	  return &array;
	}
	ArgArray* a = array.getArgs(); 
	if( a->Contains( name, False ) ) {
	  return &array;
	}
	index++;
  }
  return NULL;
}

void  Model::WriteDataToConfig( CStream& outStream ) 
{
  outStream << " n(" << fNSim << ")";
  outStream << " s(" << fSeed<< ")";
  outStream << " ngl(" << fMaxGhost << ")";
  outStream << " op(" << GetCInfo(kOptLevel) << ")";
  if( !Env::GlobalSort() ) { outStream << " ms()"; }
  if( TTime::Year() > 0.0 ) { outStream << " Y( " << TTime::Year() << "," << TTime::DTYear() << ")"; }
  if( fSchedule.ScheduleConfigured() ) fSchedule.WriteDataToConfig(outStream);
  else  { outStream << " OT(1,0,20) "; }
  if( GetCInfo(kDebug) ==  kNoConfig ) {  SetCInfo( kDebug, 0 ); }
  outStream << " d( " << (int)GetCInfo(kDebug)<< ")";
  if(fFrame) fFrame->WriteGlobalDataToConfig( outStream );
  if( fDBName != "default" ) {
		CString cmd = fDBName;
		if( fDBPort != "default" ) { 
			cmd += ","; 
			cmd += fDBPort;
			if( fDBHost != "default" ) { cmd += ","; cmd += fDBHost; }
		}
		outStream << " db( " << cmd.chars() << ")";
	}
	outStream << " UTM(0," << fLowerCornerUTM[0] << "," << fLowerCornerUTM[1]<< ")";
	outStream << " UTM(1," << fUpperCornerUTM[0] << "," << fUpperCornerUTM[1]<< ")";
	
	Externals::WriteDataToConfig( outStream );
	
	if( !fGOFCmds.empty() ) { outStream << "  " << fGOFCmds; }
	if( !fComment.empty() ) { outStream << " | " << fComment; }


} //	TModel::WriteDataToConfig

int Model::ParseMMLFile( const CString& path, int warn ) {
  
  /******* return 0; ********/
  MMLin = fopen( path, "r" );
  if( MMLin ) {
		MMLinitParse(fParserDebug,0);
    MMLparse ();
    fclose(MMLin);
    return 1;
  } else {
		if( (warn==2) || (( warn == 1 ) && (gDebug > 1)) ) {
			sprintf(gMsgStr,"Can't find MML file %s",path.chars());
			gPrintScreen();
		}
  }
  return 0;
}

int Model::ReadSetupFile( char* cnfgExtension ) {

  int rv = 1;
  CPathString path(Env::ConfigPath());
  path.Add(Env::ModelName());  
  ExprBuilder::FreeNodes();
  const CString& cpath = path.Path(cnfgExtension);
  if( ParseMMLFile( cpath ) ) {
    sprintf(gMsgStr,"Processing setup file: %s",cpath.chars());
    ProcessNode(ExprBuilder::GetRootNode());
  } else {
		if( gDebug > 2 ) {
			sprintf(gMsgStr,"Can't open setup file: %s: Using defaults.",cpath.chars());
		}
    rv = 0;
  }	
  gPrintScreen();
  return rv;
}

int Model::DBaseObjectFileRW( char mode ) {

  int rv = 1;
  CPathString path(Env::DataPath());
  path.Add("cache.in");  
  path.Add(Env::ModelName());  
	const CString& cpath = path.Path("_DBO.MML");
  if( mode == 'r' ) {
		ExprBuilder::FreeNodes();
		if( ParseMMLFile( cpath, 0 ) ) {
			sprintf(gMsgStr,"Processing setup file: %s",cpath.chars());
			ProcessNode(ExprBuilder::GetRootNode());
		} else {
			if( gDebug > 2 ) {
				sprintf(gMsgStr,"Can't open setup file: %s: Using defaults.",cpath.chars());
			}
			rv = 0;
		}	
		gPrintScreen();
	} else {
		FILE* dbf = fopen(cpath,"w");
		if(dbf == NULL) {
			CString tmp("Can't open DBase Objects File: "); tmp += cpath;
			gPrintErr(tmp); 
			return 0; 
		}
#ifdef POSTGRESS	
		TNamedObjectList&  rl = pgObjRec::ObjRecList();
		for( Pix p = rl.first(); p; rl.next(p) ) {
			pgObjRec& pgr = (pgObjRec&)rl(p);
			pgr.WriteMML(dbf);
		}
#endif	
	}
  return rv;
}

int Model::UpdateDBaseInCache() {
#ifdef POSTGRESS
	if(fDB == NULL) return 0;
  CPathString path(Env::DataPath());
  path.Add("cache.in");
	TNamedObjectList&  rl = pgObjRec::ObjRecList();
	for( Pix p = rl.first(); p; rl.next(p) ) {
		pgObjRec& pgr = (pgObjRec&)rl(p);
		if( pgr.Active() && pgr.Mode() == pgObjRec::kSMEInput ) {
			int ntup = fDB->get_object_data(pgr);
			if( (ntup>0) && ( pgr.DateCmp() <= 0 ) ) {                    // check date of cached file
				fDB->CacheObject(pgr);
			} else {                                 // check to see if file still exists in cache
				struct stat filestat; 
				path.Add(pgr.FileName()); 
				if( (ntup>0) && stat( path, &filestat ) < 0 )  { 
					fDB->CacheObject(pgr); 
				}
			}
		}
	}
	DBaseObjectFileRW( 'w' );
#endif
	return 1;
}

int Model::SaveDBaseOutCache() {
#ifdef POSTGRESS
	if(fDB == NULL) return 0;
  CPathString path(Env::DataPath());
  path.Add("cache.out");
	TNamedObjectList&  rl = pgObjRec::ObjRecList();
	for( Pix p = rl.first(); p; rl.next(p) ) {
		pgObjRec& pgr = (pgObjRec&)rl(p);
		if( pgr.Active() && pgr.Mode() == pgObjRec::kSMEOutput ) {
			fDB->set_object_data(pgr);
			fDB->ImportObject(pgr);
		}
	}
#endif
	return 1;
}

//----------------------------------------------------------------------------------------
//	Model::WriteConfigFile():
//----------------------------------------------------------------------------------------
  
void Model::WriteConfigFile(const char* name)
{
  CStream cnfgStream;
  CPathString path;
  Pix pv;  
  
  if( Env::TmpPath().empty() ) { 
	  path = Env::ConfigPath();
  } else { 
	  path = Env::TmpPath(); 
  }
  if( name == NULL ) {
	if( !fConfigOutFile.empty() ) {
	  path.Add(fConfigOutFile); 
	}
  } else {
	path.Add(name); 
  }
  
  if( cnfgStream.Open( path, "w") ) {
	CString msg("Writing Config-Out File: "); msg += path; 
	gPrintScreen(msg);
	CString name("global");
	WriteNameToConfigFile(cnfgStream, '#', name );	
	WriteDataToConfig(cnfgStream);
	for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
	  Module& m = (Module&) fModuleList(p);
	  WriteNameToConfigFile(cnfgStream, '$', m.SName() );
	  m.WriteDataToConfig(cnfgStream);
	  TNamedObjectList& variableList = m.VariableList();
	  variableList.Alphabetize();
	  for(pv = variableList.first(); pv; variableList.next(pv) ) { 
				Variable& aVar = (Variable&) variableList(pv);
				if( !aVar.GetF(FisImport) ) {
					WriteNameToConfigFile(cnfgStream, '*', aVar.SName() );
					aVar.WriteDataToConfig(cnfgStream);
				}
	  }
	  for(pv = variableList.first(); pv; variableList.next(pv) ) { 
				Variable& aVar = (Variable&) variableList(pv);
				if( aVar.GetF(FisImport) ) {
					WriteNameToConfigFile(cnfgStream, '*', aVar.SName() );
					aVar.WriteDataToConfig(cnfgStream);
				}
	  }
	}
	cnfgStream.Close();
  } else {
	CString msg("Can't open Config File: "); msg += path;
	gPrintErr(msg);	
  }

  if( Env::TmpPath().empty() ) { 
	  path = Env::ConfigPath();
  } else { 
	  path = Env::TmpPath(); 
  }
  if( !fConfigTemplateOutFile.empty() ) {
	path.Add(fConfigTemplateOutFile); 

	if( cnfgStream.Open( path, "w") ) {
	  CString msg("Writing Config-Template File: "); msg += path; 
	  gPrintScreen(msg);
	  CString name("global");
	  WriteNameToConfigFile(cnfgStream, '#', name );	
	  for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
		Module& m = (Module&) fModuleList(p);
		WriteNameToConfigFile(cnfgStream, '$', m.SName() );
		TNamedObjectList& variableList = m.VariableList();
		variableList.Alphabetize();
		for(pv = variableList.first(); pv; variableList.next(pv) ) { 
				  Variable& aVar = (Variable&) variableList(pv);
				  if( !aVar.GetF(FisImport) ) {
					  WriteNameToConfigFile(cnfgStream, '*', aVar.SName() );
				  }
		}
		for(pv = variableList.first(); pv; variableList.next(pv) ) { 
				  Variable& aVar = (Variable&) variableList(pv);
				  if( aVar.GetF(FisImport) ) {
					  WriteNameToConfigFile(cnfgStream, '*', aVar.SName() );
				  }
		}
	  }
	  cnfgStream.Close();
	} else {
	  CString msg("Can't open Template Config File: "); msg += path;
	  gPrintErr(msg);	
	}
  }
}  

//----------------------------------------------------------------------------------------
//	Model::WriteDBParameters():
//----------------------------------------------------------------------------------------
  
void Model::WriteDBParameters()
{
  FILE* file;
  CPathString path;
  Pix pv;
	if( Env::TmpPath().empty() ) { 
		path = Env::DataPath();
	} else { 
		path = Env::TmpPath(); 
	}
	path.Add("DBaseParameters.out"); 
  if( file = fopen( path, "w" ) ) {
    for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
      Module& m = (Module&) fModuleList(p);
      TNamedObjectList& variableList = m.VariableList();
      for(pv = variableList.first(); pv; variableList.next(pv) ) { 
				Variable& aVar = (Variable&) variableList(pv);
				aVar.PrintDBParamValues(file);
      }
    }
    fclose(file);
  } else {
    CString msg("Can't open DBaseParameters File: "); msg += path;
    gPrintErr(msg);	
  }
}  

void Model::WriteNameToConfigFile(CStream& cnfgStream, char type, const CString& name)
{
  int i, nspace, cnt=0;
  nspace = 33-name.length();
  cnfgStream.put('\n');
  cnfgStream.put(type);
  cnfgStream.put(' ');
  cnfgStream << name.chars();
  for(i=0; i<=nspace; i++) cnfgStream.put(' ');
}

TNode* Model::ReadMMLFiles() {    
  MMLout = fopen("parser.out","w");
  ExprBuilder::FreeNodes();
  TNode* rv = NULL;
  CPathString path(Env::ModelPath());
  path.Add("Models");
  if( ExprBuilder::GetMode() == ExprBuilder::kSMML ) { path.Add(Env::ModelName()); }
  for( int i=0; i<fMMLFileList.NArgs(); i++ ) {	
    TNode* tmp_node = ReadMMLFile(path,fMMLFileList[i],False);
    if( tmp_node != NULL ) { rv = tmp_node; }
  }	
  if( rv == NULL ) { rv = ExprBuilder::GetRootNode(); }
  ProcessNode(rv);
  ReadSliderParms();
  return rv;
}


TNode* Model::ReadMMLFile( const CPathString& path, const char* file, int process ) {  
  
  CString spec_file(file); 
  TNode* root_node = NULL; 
  if( spec_file.contains( ".xml", spec_file.size()-4 ) ) {
	ExprBuilder::SetMode(ExprBuilder::kSMML);
	SMMLParser* smml = new SMMLParser( NULL,  file, fParserDebug );
	smml->parseFile( path.Path(), spec_file ); 
	root_node = smml->rootNode(); 
	delete smml;
  } else {
	  ExprBuilder::SetMode(ExprBuilder::kMML);
	  CString infile(path.Path());  (infile += "/") += spec_file;
	  sprintf(gMsgStr,"Parsing input file: %s",infile.chars());
	  gPrintScreen();
	  MMLin = fopen( infile.chars(), "r" );
	  if( MMLin ) {
			ExprBuilder::ClearNodes();	
			MMLinitParse(fParserDebug,0);
		if( MMLparse () ) { ExprBuilder::StackDump(gLogFile); }
		fclose(MMLin);
	  } else gPrintScreen( infile + ": Can't find MML file" );
	  
	  root_node = ExprBuilder::SetRootNode();	
  } 
  if( process && (root_node != NULL) ) { ProcessNode(root_node); }
  return root_node;
}

Model* Model::New(const char* name) { 
  fCurrentModel = new Model(name); 
  if( fModelList.GetObjectByName(name) ) { gPrintErr("Model name clash"); }
  fModelList.append(*fCurrentModel);
  fCurrentModel->Init();
  return fCurrentModel;
}

void Model::ReadConfigFiles() {
  CPathString path;
  for( int i=0; i< fConfigFileList.NArgs(); i++ ) {
		if( Env::TmpPath().empty() ) { 
			path = Env::ConfigPath();
		} else { path = Env::TmpPath(); }
    path.Add(fConfigFileList[i]);  
    if( fCR.Open(path) ) {
      fCR.Read();
      fCR.Close();
    }
  }
  Frame().SetupMaps();
}

Pix Model::AddVariable( Variable* v ) { 
//	return fVariableList.Insert(*v); 
	return fVariableList.append(*v);
}

Variable* Model::GetVariable(const char* name) {  // Return v if it's owned by one of the modules 
  Variable* rv = NULL; 
  Module *loser = NULL;
  int error = 0;
  for( Pix pt = fVariableList.first(); pt; fVariableList.next(pt) ) {
    Variable& aVar = (Variable&) fVariableList(pt); 
    if( (aVar.SName() == name) && !aVar.GetF(FisImport) ) {
	  if( rv != NULL ) {
		Module *m = aVar.GetModule();
		CString m_name(m->SName()); m_name.upcase();
		int m_active = ( m_name.starts_with("XXXIMPORT") == 0 );
		Module* rm = rv->GetModule();    			
		CString rm_name(rm->SName()); rm_name.upcase();
		int rm_active = ( rm_name.starts_with("XXXIMPORT") == 0 );
		if( m_active ) {
			if( rm_active ) {
			  loser = m;   // conflict- rm and m both active
			  error = 2;
			  break;
			} else {
			  rv = &aVar;  // aVar in active mod wins over rv in inactive mod.
			  loser = rm;
			  if( error < 2 ) { error = 0; }		  
			}
		  } else {
			// Do nothing - aVar in inactive module loses to rv in active module.
			loser = m;
			if( !rm_active ) { error = 1; }  // error: both inactive
			else if( error < 2 ) { error = 0; }	
		  }
	  } else {
		rv = &aVar;		  
	  }
    }
  }
  if( rv != NULL ) {
	Module* rm = rv->GetModule();    			
	if( error == 2 ) {
	  sprintf(gMsgStr,"Multiple definitions of %s, using %s:%s (also defined in %s)",
										  name, rm->Name(), name, loser->Name() ); 
	  gPrintErr();
	} 
	if( ( rm != NULL) && ( rm->GetCInfo(Module::kIgnorable) ) ) {
	  rm->SetCInfo( Module::kIgnorable, False );
	}
	CString rm_name(rm->SName()); rm_name.upcase();
	int rm_active = ( rm_name.starts_with("XXXIMPORT") == 0 );
	if( !rm_active ) {
	  sprintf(gMsgStr,"Using out-of-sector default: %s:%s ", name, rm->Name(), name ); gPrintScreen();
	  rv->SetF(FisIgnored,False,"Model::GetVariable");
	}
  }
  return rv;	// Else, it's a special variable, return NULL
}

int Model::ProcessCommand(const TCommandNode* cn) { 
	const TDeclarationNode* dn = cn->Declaration();
	const CString& cmd = dn->Command();
	const CString& name = (dn->Name()) ? dn->Name()->Top() : gNullString;
 
  if( (cmd == "Module") ||  (cmd == "compound") ) {
    int isExtension = dn->IsExtension();
    Module* currentMod = (isExtension) ? GetModule(name,False) : GetModule(name);
    if( currentMod == NULL ) gFatal(name + ": Can't Find Module for Extension");
		currentMod->AddQualifiers(dn,isExtension);
		currentMod->SetNode(cn->StatementList()); 
    return currentMod->ProcessNode();
  }
	else if( cmd == "Function" ) {
		TFunction* f = TFunction::New(name);
		f->SetNode(cn->StatementList()); 
		int rv = f->ProcessNode();
		if( f->Source() == kFS_UserLib ) {
			AddCodeFile(f->Library());
		}
		return rv;	
	}
	else if( cmd == "Constant" ) {
		TConstant* c = TConstant::New(name);
		c->SetNode(cn->StatementList()); 
		return c->ProcessNode();	
	}
#ifdef POSTGRESS
	else if( cmd == "DBaseObject" ) {
		pgObjRec* pgr = pgObjRec::New(name);
		pgr->SetNode(cn->StatementList()); 
		return pgr->ProcessNode();	
	}
	else if( cmd == "DBaseFunction" ) {
		pgSQLRec* pgr = pgSQLRec::New(name);
		pgr->SetNode(cn->StatementList()); 
		return pgr->ProcessNode();	
	}
#endif
	return 0;
}

void Model::AddCodeFile( CString& codeFileName ) {
	Pix p = fCodeFileList.GetObjectByNameP(codeFileName);
	if( p == 0 ) {
		TNamedObject* file = new TNamedObject(codeFileName);
		fCodeFileList.append(*file);
	}
}

int Model::ProcessAttribute(const TAttributeNode* an) {
	const TDeclarationNode* dn = an->Declaration();
	const TNode* vn = an->Value();
	CString& cmd = (CString&) dn->Command();
	const ArgArray&  ql = dn->QualifierList()->List();
	if( vn == NULL ) return 0;
	cmd.downcase();
	
  int iarg; float farg;
  if( cmd == "debug" ) {
		if( GetAttrValue( vn, kAVInt ) ) {
				Env::SetDebugLevel( (int)fAttrValue );
		}
  }
  if( (cmd == "parserdebug") || (cmd == "parsedebug")  ) {
		if( GetAttrValue( vn, kAVInt ) ) {
				fParserDebug = (int)fAttrValue;
		}
  }
	else if( cmd == "file" ) {
    if( ql.Contains("output") ) {
      if( ql.Contains("mml") ) {
				if( GetAttrValue( vn, kAVString ) ) {
					const CString* s = fAttrValue;
					if(s) fMMLOutFile = *s;
				}		
      }
      else if( ql.Contains("config") ) {
				if( GetAttrValue( vn, kAVString ) ) {
					const CString* s = fAttrValue;
					if(s) fConfigOutFile = *s;
				}		
      }      
    } else if( ql.Contains("input") ) {
      if( ql.Contains("mml") ) {
				GetAttributeStringList(vn,fMMLFileList);
      } else if( ql.Contains("config") ) {
				GetAttributeStringList(vn,fConfigFileList);
      } else if( ql.Contains("stella") ) {
				GetAttributeStringList(vn,fInputFileList);
      }
      else gPrintErr( " Unknown input File Type in Model config ( mml or config ? ). ");
    } else gPrintErr( " Must specify input/output File cmd modifier in Model config. ");
  } else return 0;
  return 1;
}

//----------------------------------------------------------------------------------------
//				TModel::PostPipe:
//----------------------------------------------------------------------------------------

Pix Model::PostPipe( TPipeCommand& c ) {

		Module* gm  = GlobalMod();                             // Post at beginning of init event.
		if( gm == NULL ) { gFatal( "No Root Module Specified." ); }
		TEvent* e = gm->GetEvent("StateVarInit",TEvent::kInit);
		Pix rp = e->AddCommand(c,TEvent::kBeginning);
		sprintf(gMsgStr,"\nPosting Pipe Command: %s:%s to event %s",Name(),c.Name(),e->Name()); gPrintLog();
		return rp;
}


//----------------------------------------------------------------------------------------
//				TModel::CreateEventLists:
//----------------------------------------------------------------------------------------
 
void Model::CreateEventLists()
{
  sprintf(gMsgStr,"CreateEventLists"); gPrintScreen();
  Pix p;
  for( p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& aModule = (Module&) fModuleList(p);
    aModule.ProcessConnections();
  }

  sprintf(gMsgStr,"ProcessTemporalDependencies"); gPrintScreen();
  int go, cnt = 0;
  do {
    go = 0;
    for( p = fModuleList.first(); p; fModuleList.next(p) ) {
      Module& aModule = (Module&) fModuleList(p);
      if( aModule.ProcessTemporalDependencies() ) { go = 1; }
      if( cnt++ == 500 ) gProgramBreak("Long Sort in ProcessTemporalDependencies");
    }
  } while (go);

  sprintf(gMsgStr,"ProcessSpatialDependencies"); gPrintScreen(); cnt=0;
  do {
    go = 0;
    for( p = fModuleList.first(); p; fModuleList.next(p) ) {
      Module& aModule = (Module&) fModuleList(p);
      if( aModule.ProcessSpatialDependencies() ) { go = 1; }
      if( cnt++ == 500 ) gProgramBreak("Long Sort in ProcessSpatialDependencies");
    }
  } while (go);

	gPrintScreen("CreateEventLists");
	Module* gm  = Model::I0().GlobalMod();
  if( gm == NULL ) { gFatal( "No Root Module Specified." ); }
	TEvent* initEvent = gm->GetEvent("StateVarInit",TEvent::kInit);

  for( p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& aModule = (Module&) fModuleList(p);
    aModule.CreateEventLists(initEvent);
  }

	gPrintScreen("FillInitializationList");
	FillEventListByDependency (*initEvent);
/* 
  do {
		go = 0;
		for( p = fModuleList.first(); p; fModuleList.next(p) ) {
			Module& aModule = (Module&) fModuleList(p);
			if( aModule.FillInitializationList(initEvent) ) { go = 1; }
		}
	} while (go);
*/	  
  sprintf(gMsgStr,"Split & Sort Lists"); gPrintScreen();

	if( (Model::I0().GetCInfo(Model::kOptLevel) < 5) /* && !Env::GlobalSort() */ ) {    
		for( p = fModuleList.first(); p; fModuleList.next(p) ) {  // Split serial from Spatial events.
			Module& aModule = (Module&) fModuleList(p);
			aModule.SplitListsSpatially();
		}
  }
  
  for( p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& aModule = (Module&) fModuleList(p);
    aModule.CreateEventDependencies();
    aModule.SortLists();		
  }

  startDelete:
  for( p = fEventList.first(); p; fEventList.next(p) ) {
    TEvent& aEvent = (TEvent&) fEventList(p);
    if( aEvent.CommandList().empty() ) { 
      fEventList.del(p);
      goto startDelete;
    }
  }
/*
  for( p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& aModule = (Module&) fModuleList(p);
    aModule.DeleteEmptyEvents();
  }
*/
}  //  TDocCG::WriteDefaultConfig

void Model :: FillEventListByDependency( TEvent& event ) {	
	int skipStateVars = (event.GetObjInfo(TEvent::kType) == TEvent::kInit);
	TNamedObjectList& cL = event.CommandList() ;
	Pix rp, pv, p1, pD; 
repeat:
	for( p1 = cL.first(); p1; cL.next(p1) ) {
		TCommand &ci = (TCommand&) cL(p1);
		if( ci.GetObjInfo(TCommand::kStatus) == TCommand::kActive ) {
			TNamedObjectList& depList = ci.DependencyList();
			for( pv = depList.first(); pv; depList.next(pv) ) {
				TDependency& d = (TDependency&) depList(pv);
				Variable* depVar = d.GetVar();
				if( depVar  ) {
					TNamedObjectList& cLD = depVar->CommandList();
					for( pD = cLD.first(); pD; cLD.next(pD) ) {
						TCommand &cd = (TCommand&) cLD(pD);
						TCommand::EType type =  (TCommand::EType) cd.GetObjInfo(TCommand::kType);
						TCommand::EExType extype =  (TCommand::EExType) cd.GetObjInfo(TCommand::kExType);
						int isIntegrationCommand = (type == TCommand::kIntegrate);
						if( ((extype == TCommand::kCode) ||  (extype == TCommand::kImport)) 
													&& !(skipStateVars && isIntegrationCommand) && !(type == TCommand::kFunction) ) {
							if( gDebug > 1 ) {
								sprintf(gMsgStr,"Checking Dep(skipSV:%d): %s->%s, %s:%s -> %s : ",
									skipStateVars, ci.Name(), cd.Name(), ci.Var()->GetModule()->Name(), ci.Var()->Name(),depVar->Name());
								gPrintLog();
							}
							if( (rp = event.AddCommand(cd)) != NULL ) {
								if( gDebug > 1 ) { sprintf(gMsgStr," >> cmd %s: move ", cd.Name() ); gPrintLog(); 	}
								goto repeat;
							}
						}
					}
				}
			}
		}
	}
}

//----------------------------------------------------------------------------------------
//				Model::WriteMML:
//----------------------------------------------------------------------------------------

void  Model::WriteMML( FILE* oFile, EWriteMode mode )  {
  ExprBuilder::SetMode(ExprBuilder::kMML);
  for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    if( m.GetCInfo(Module::kIgnorable) == 0 ) { 
			m.WriteMML(oFile,kDefinition);
		}
  }
	fputc('\n',oFile);
	fputc('~',oFile);
	fclose(oFile);
}

void  Model::WriteXML( CString xmlPath ) {
	ExprBuilder::SetMode(ExprBuilder::kSMML);
	for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    if( ( m.Super() == NULL ) && ( m.GetCInfo(Module::kIgnorable) == 0 ) ) { 
		m.WriteXML(xmlPath);
	}
  }
}

void  Model::WriteConfigXML( CString xmlPath ) {
	ExprBuilder::SetMode(ExprBuilder::kSMML);
	for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    if( ( m.Super() == NULL ) && ( m.GetCInfo(Module::kIgnorable) == 0 ) ) { 
		m.WriteConfigXML(xmlPath);
	}
  }
}

void Model::Dump( FILE* oFile ){
  if( oFile == NULL ) return;
  for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    m.Dump(oFile);
  }
}

int Model::Finalize() {
	int rv=0;
  for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    rv += m.Finalize();
  }
  return rv;
}

int Model::RunNodeProcessing( EProcessContext pc ) {
	switch(pc) {
		case kPCDerivatives: 
			for( Pix p = fDerivatives.first(); p; fDerivatives.next(p) ) {
				TDerivative* d = (TDerivative*) &fDerivatives(p);
				InitializeGlobalDependencies(d);	
				TNamedObjectList& initiators = d->fInitiators;
				for( Pix pt = initiators.first(); pt; initiators.next(pt) ) {
					Variable* v = (Variable*)&initiators(pt); 
					v->CreateDerivativeVariable( d );
				}
			} 
		break;		
	}
  for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    m.RunNodeProcessing(pc);
  }
  if( pc == kPCDependencies ) {
	for( Pix p = fModuleList.first(); p; fModuleList.next(p) ) {
	  Module& m = (Module&) fModuleList(p);
	  if( m.SName().index( "XXXImports" ) == 0 ) {
		m.RunNodeProcessing(pc);
	  }
	}
  }
  return 0;
}

void Model::InitializeGlobalDependencies(TDerivative* d) {
	Variable* target = d->fIndepVar;
	Pix p, targPix = 0;
	int index = d->Index();
	TNode *node; TCommand* cmd;	
	TNamedObjectList depTree;
	PixArray varsWithDep;
	TNamedObjectList& initiators = d->fInitiators;
	for( Pix pt = initiators.first(); pt; initiators.next(pt) ) {
		Variable& v = (Variable&)initiators(pt); 
		if( (cmd = v.GetDynamicCommand()) != NULL ) {
			if( (node = cmd->GetNode()) != NULL ) {
				if( node->getDepValue( index ) == TNode::kNew ) {
					depTree.append( v );
					node->setDepValue( TNode::kInDepTree, index );
					if( gDebug > 1 ) {
						sprintf( gMsgStr, "DepSweep %s: adding seed var to dependency tree: %s",target->Name(),v.Name());
						gPrintLog();
					}
				}
			}
		}
	}

	for( p = depTree.first(); p; depTree.next(p) ) {			
		Variable *v1, *v = (Variable*)&depTree(p);
		if( (cmd = v->GetDynamicCommand()) != NULL ) {
			if( v == target ) { 
				varsWithDep.add(p); targPix = p; 
				if( (node = cmd->GetNode()) != NULL ) { 
					node->setDepValue( TNode::kHasDep, index  ); 
					if( gDebug > 1 ) { sprintf( gMsgStr, "DepSweep %s: Marking target: %s",target->Name(),v->Name()); gPrintLog(); }
				}
			} else {
				TNamedObjectList& dlist = cmd->DependencyList(); 
				for( Pix pd = dlist.first(); pd; dlist.next(pd) ) {
					TDependency& dep = (TDependency&) dlist(pd);
					Variable* dv = dep.GetVar();
					if( dv->GetF(FisImport) && dv->Connection() ) { dv = dv->Connection(); } 
					if( (cmd = dv->GetDynamicCommand()) != NULL ) {
						if( (node = cmd->GetNode()) != NULL ) {
							TNode::EDependencyStatus ds = node->getDepValue( index );
							switch( ds ) {
								case TNode::kHasDep: {
									Pix pn = depTree.append( *dv );
									varsWithDep.add(pn);
									depTree.slink ( pn, p );
									if( gDebug > 1 ) {
										sprintf( gMsgStr, "DepSweep %s: adding var with dep to dependency tree: %s -> %s",target->Name(),v->Name(),dv->Name());
										gPrintLog();
									}
								} break;
								case TNode::kNoDependency: break;
								case TNode::kInDepTree: {
									v1 = NULL;
									for( Pix p1 = depTree.first(); p1; depTree.next(p1) ) {			
										Variable* vt = (Variable*)&depTree(p1);
										if( vt == dv ) { 
											depTree.slink ( p1, p );
											if( gDebug > 1 ) { sprintf( gMsgStr, "DepSweep %s: BackLink: %s -> %s",target->Name(),vt->Name(),v->Name()); gPrintLog(); } 
											v1=vt; 
											break; 
										}
									} 
								} if( v1 ) break;
								case TNode::kNew: { 
									Pix pn = depTree.append( *dv );
									depTree.slink ( pn, p );
									if( gDebug > 1 ) { sprintf( gMsgStr, "DepSweep %s: BackLink: %s -> %s",target->Name(),dv->Name(),v->Name()); gPrintLog(); } 
									node->setDepValue( TNode::kInDepTree, index  ); 
									if( gDebug > 1 ) {
										sprintf( gMsgStr, "DepSweep %s: adding var to dependency tree: %s -> %s",target->Name(),v->Name(),dv->Name());
										gPrintLog();
									}
								} break;
							}
						}
					}
				}
			}
		}
	}
	
	for( p = depTree.first(); p; depTree.next(p) ) {			
		Variable* v = (Variable*)&depTree(p);
		if( (cmd = v->GetDynamicCommand()) != NULL ) {
			if( (node = cmd->GetNode()) != NULL ) {
				TNode::EDependencyStatus ds = node->getDepValue( index );
				if( ds == TNode::kInDepTree ) { node->setDepValue( TNode::kNoDependency, index  ); }
			}
		}
	}

	int i=0;
	TOrderedObjectDLListNode* n0; 
	while( n0 = (TOrderedObjectDLListNode*) varsWithDep.elem(i++) ) {
		Variable* v0 = (Variable*)&depTree(n0);
		DLLinkIterator iter(n0);
		for ( p =  iter.first(); p; p= iter.next() ) {
			Variable* v = (Variable*)&depTree(p);
			if( (cmd = v->GetDynamicCommand()) != NULL ) {
				if( (node = cmd->GetNode()) != NULL ) {
					TNode::EDependencyStatus ds = node->getDepValue( index );
					switch( ds ) {
						case TNode::kHasDep: break;
						case TNode::kNoDependency: 
							varsWithDep.add(p);
							node->setDepValue( TNode::kHasDep, index  ); 
							if( gDebug > 1 ) {
								sprintf( gMsgStr, "DepSweep %s: Marking dep for (%s->) %s",target->Name(),v0->Name(),v->Name()); gPrintLog();
							}
						break;
						case TNode::kInDepTree: 
						case TNode::kNew: 
							gPrintErr("Error in backward dependency sweep");
						break;
					}
				}
			} 
		}
	}
	if( gDebug > 1 ) {
		sprintf( gMsgStr, "^^^ Dependency dump (backward dependencies)  for %s:\n", target->Name() ); gPrintLog(); i=0; 
		target->PrintDependencies( targPix, 1, depTree );
	}
}	

int Model::ProcessGeneral(TNode* n) {
	switch( n->NodeType() ) {
		case TNode::kModule: {
			TModuleNode* mn = (TModuleNode*)n;
			Module* m = mn->GetModule();
			if( m == NULL ) { m = (Module*) fModuleList.GetObjectByName( mn->Name() ); }
			if( m == NULL ) { m = Model :: I0().GetModule( mn->Name(), True ); }
			if( m ) {
				TNode* n = m->GetNode();
//				if( n && (n !=  mn) ) gPrintErr( m->SName() + ": Module Name Clash!");
				m->SetNode( mn ); mn->SetModule(m); 
			}
			else { gPrintErr(m->SName() +  ": Module creation error! "); }
			return m->ProcessNode(n);
		}
	}
	return 1;
}


  //----------------------------------------------------------------------------------------
  //				TModel::GetModule:
  //----------------------------------------------------------------------------------------
  
  
Module* Model :: GetModule( const CString& name, Bool create )
{
  Module* thisModule = (Module*) fModuleList.GetObjectByName( name );

  if( thisModule == NULL && create ) {
    thisModule = CreateNewModule(name);						
  }
  return fCurrentModule = thisModule;
  
} // TModule::CreateNewModule

  //----------------------------------------------------------------------------------------
  //				Model::CreateNewModule:
  //----------------------------------------------------------------------------------------

Module* Model :: CreateNewModule(const CString& s) { 
	Module* m = new Module( s );
	fModuleList.Insert(*m);	
	if( fGlobalModule != NULL ) { fGlobalModule->AddSubModule(m); }
	m->SetModel(this);
	return m;
}

  //----------------------------------------------------------------------------------------
  //				Model::WriteCCode:
  //----------------------------------------------------------------------------------------

void  Model::WriteCCode( FILE* CHdrFile, FILE* CEqnFile) {
	Pix p;
  fprintf(CEqnFile,"\n\nvoid gInitializeModel() {");
  const Module* gm = GlobalMod();
  if( gm == NULL ) { gPrintErr("No global Module."); return; }
  fprintf(CEqnFile,"\n\tTModel::I().GlobalMod(&%s::I());",gm->Name()); 
  fprintf(CEqnFile,"\n\t%s::I().Initialize();",gm->Name()); 
  fprintf(CEqnFile,"\n\t%s::I().SetMInfo(TModule::kisGlobal,True);",gm->Name()); 
  for( p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    if( &m != gm ) {
      fprintf(CEqnFile,"\n\t%s::I().Initialize();",m.Name()); 
    }
  }
  for( p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    fprintf(CEqnFile,"\n\t%s::I().MakeConnections();",m.Name());
  }
  fprintf(CEqnFile,"\n}\n");

  for( p = fModuleList.first(); p; fModuleList.next(p) ) {
    Module& m = (Module&) fModuleList(p);
    m.WriteCCode(CHdrFile,CEqnFile); 
  }
}
 //----------------------------------------------------------------------------------------
  //				Model::RunModel:
  //----------------------------------------------------------------------------------------

int Model::RunModel ( float stop_time, int breakIndex, int set_context  ) {
	float ctime, ptime = -1;
	if( Model::I0().GetCInfo(Model::kOptLevel) < 5 ) {
		while ( 1 ) {
			TEvent* e = GetNextEvent();
			if( e && ((ctime = e->Time()) <= stop_time) ) {
				TTime::Update(ctime);
				if( ctime > ptime ) {
				  Externals::run( ctime );				
				}
				if( e->Execute(breakIndex,set_context) < 0 ) return 0;
				if( breakIndex < 3 ) breakIndex = 0;
				ptime = ctime;
			}
			else break;
		}
	} else {
		Module* gm  = Model::I0().GlobalMod();
		if( gm == NULL ) { gFatal( "No Root Module Specified." ); }
		TEvent* initEvent = gm->GetEvent("StateVarInit",TEvent::kInit);
		TEvent* updateEvent = gm->GetEvent("GlobalUpdate",TEvent::kGlobalUpdate);
//		initEvent->SetTime(0.0);
		initEvent->FastExecute();
		while ( 1 ) {
			TTime::Update();
			if( gTime() > ptime ) {
			  Externals::run( ctime );				
			}
			if( gTime() <= stop_time ) {
//				updateEvent->SetTime(gTime());
				updateEvent->FastExecute();
			}
			else break;
			ptime = gTime();
		}
	}
  return 1;
}
  //----------------------------------------------------------------------------------------
  //				Model::Step:
  //----------------------------------------------------------------------------------------

float Model::StepModel ( EStepMode sm, int breakIndex, int set_context	) {
  float ctime;
  TEvent* e = GetNextEvent();
  if( e ) {
    ctime = e->Time();
    TTime::Update(ctime);
    switch ( sm ) { 
			 case kEvent: 
				 e->Execute( breakIndex, set_context ); 		
				 break;
			 case kCommand: 
				 e->ExecuteNextCommand( breakIndex, set_context ); 		
				 break;
			case kUpdate:
				 QueryInterface( ); 
				 break;
		}
  }
  return ctime;
}

TDerivative* Model::GetDerivative( Variable* indep_var ) {
	Pix p;
	for( p = fDerivatives.first();  p; fDerivatives.next(p) ) {
		TDerivative* d = (TDerivative*) &fDerivatives(p);
		if( d->fIndepVar == indep_var ) return d;
	}
	TDerivative* d = new TDerivative(indep_var);
	d->fWeight = fDerivativeWeight;
	d->fMode = fDerivativeMode;
	p = fDerivatives.append(*d);
	int index = fDerivatives.index(p);
	d->SetIndex( index );
	return d;
} 

void Model::DumpEventList() {
	sprintf(gMsgStr,"\nDump Event List:"); gPrintLog();
	for( Pix p = fEventList.first(); p; fEventList.next(p) ) {
		TEvent& ev = (TEvent&) fEventList(p);
		Module* m = ev.GetModule();
		sprintf(gMsgStr,"\n\tEvent %s:%s",m->Name(),ev.Name()); gPrintLog();
		TNamedObjectList& cl = ev.CommandList();
		for( Pix pc = cl.first(); pc; cl.next(pc) ) {
			TCommand& c = (TCommand&) cl(pc);
			Variable* v = c.GetVar();
			sprintf(gMsgStr,"\n\t\tCommand %s:%s",v->Name(),cl.Name()); gPrintLog();
		}
	}
}
  //----------------------------------------------------------------------------------------
  //				Model::PostEvent:
  //----------------------------------------------------------------------------------------

void Model::ClearEventList() {
	TEvent* e=NULL;
	while( e = GetNextEvent() ) e->Clear();
	fCurrentEvent = NULL;
}

 //----------------------------------------------------------------------------------------
  //				Model::QueryInterupt:
  //----------------------------------------------------------------------------------------
 
int Model::QueryInterupt() { 
	if( l_interrupt == 1 ) { 
		l_interrupt = 0;
		return 1; 
	}
	return 0;
}

 

  //----------------------------------------------------------------------------------------
  //				Model::PostEvent:
  //----------------------------------------------------------------------------------------

Pix Model::PostEvent( TEvent& e, float next_time, TEvent::ECmdPosition cp ) {
	Pix p = 0;
	switch( cp ) {
		case TEvent::kEnd: 
			for( p = fEventList.last(); p; fEventList.prev(p) ) {
				TEvent& ev = (TEvent&) fEventList(p);
				if( ev.Time() <= next_time ) { p = fEventList.ins_after( p, e); break; }
			}
			if( p == 0 ) { p = fEventList.prepend( e ); }
		break;
		case TEvent::kBeginning:
			for( p = fEventList.first(); p; fEventList.next(p) ) {
				TEvent& ev = (TEvent&) fEventList(p);
				if( ev.Time() >= next_time ) { p = fEventList.ins_before( p, e); break; }
			}
			if( p == 0 ) { p = fEventList.append( e ); }
		break;
	}
	if( gDebug ) {
	  gPrintLog( format("Posting Event %s at time %f: %p", e.Name(), next_time, p ) );
	}
	return p;
}


  //----------------------------------------------------------------------------------------
  //				Model::PostEvent:
  //----------------------------------------------------------------------------------------

static inline int readDataString( FILE* file, CString& string) {
	int test;
	string.clear();
	while( isspace( test = fgetc(file) ) ) {;}
	if( test == EOF ) return 1; 
	else string += (char) test;
	while( !isspace( test = fgetc(file) ) ) {
		if( test == EOF ) break; 
		string += (char) test; 
	}
	return 0;
}

static inline void DBP_format_error(int line ) { sprintf(gMsgStr,"Format error DBParameter data: line %d", line ); gPrintErr(); }

int Model::ReadDBParamValues( const char* fileName, int overrideParameters ) {
  FILE* file;
  CPathString path;
	if( Env::TmpPath().empty() ) { 
		path = Env::DataPath();
	} else { 
		path = Env::TmpPath(); 
	}
	path.Add(fileName); 
  if( file = fopen( path, "r" ) ) {
		if( gDebug > 0 ) {
			sprintf(gMsgStr,"Reading DBParameter file %s", fileName ); gPrintScreen();
		}
		int cnt = -1;
		CString module, variable,  map, parm, value;
		Variable* override = NULL;
		while( ++cnt >= 0 ) {
			if( readDataString( file, module ) ) return cnt;
			if( readDataString( file, variable ) )  {  DBP_format_error(cnt);  return cnt; }
			if( readDataString( file, map ) ) {  DBP_format_error(cnt); return cnt; }
			if( readDataString( file, parm ) ) {  DBP_format_error(cnt); return cnt; }
			if( readDataString( file, value ) ) {  DBP_format_error(cnt); return cnt; }

			Module* m = GetModule( module, False);
			if( m == NULL ) { 
			  static int merror = 1;
			  sprintf(gMsgStr,"Unknown module in DBParameter data: %s", module.chars() ); 
			  gPrintLog(); 
			  if( merror ) { gPrintScreen(); merror = 0; }
			}
			else {
				Variable* v = m->GetVariable( variable, False ); 
				Variable::EInitMode imode = (v == NULL) ? Variable::kDefault : (Variable::EInitMode) v->GetCInfo(Variable::kInitMode);
				if( v == NULL ) { 
					static int verror = 1;
					sprintf(gMsgStr,"Unknown variable in DBParameter data: %s", variable.chars() ); 
					gPrintLog();
					if( verror ) { gPrintScreen(); verror = 0; }
				} else if( v->GetCInfo(Variable::kType) == Variable::kState ) { 	
					sprintf(gMsgStr,"Attempt to declare state variable %s as DBase Parameter ignored ", v->Name() ); gPrintLog();
				} /* else if( (overrideParameters==0) && (imode != Variable::kDefault) && (imode != Variable::kDBase) ) { 
					if( override != v ) {	
						if( gDebug > 0 ) {
							sprintf(gMsgStr,"Overriding Array declaration for %s with configuration type: %x", v->Name(), v->GetCInfo(Variable::kInitMode) );  
							gPrintScreen(); 
						}
						override = v;
					}
				} */ else {
//					v->SetupDBParam(map);
					v->AddDBParamValue(parm,value);
				}
			}		
		}
		fclose(file);	
	} else { sprintf(gMsgStr,"No %s file found.", fileName ); gPrintScreen(); }
}
