//----------------------------------------------------------------------------------------
//      Model.h
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//	Change History
//----------------------------------------------------------------------------------------

#ifndef __MML_Model__
#define __MML_Model__
  
#include "Utilities.h"
#include "SList.h"
#include "StringPlex.h"
#include "CnfgProcessor.h"
#include "CStream.h"
#include "ExprBuilder.h"
#include "MML_Frame.h"
#include "MML_Schedule.h"
#include "MML_Event.h"
#include "sme_postgres.h"

typedef class TEvent TEvent;
typedef class Module Module;
typedef class Variable Variable;
typedef class TCommand TCommand;
typedef class LayerConfig LayerConfig;

//========================================================================================
// CLASS TDerivative
//========================================================================================

class TDerivative : public TNamedObject {
	public:

	enum EStatus { kNew, kProcessed, kUndifferentiable, kInDirectDep, kDirectDep };
	
	TNamedObjectList fInitiators;	
	Variable* fIndepVar;	
	byte fMode;
	float fWeight;
	
	TDerivative( Variable* indep_var );	
	Pix AddInitiator(Variable* v0);
	
	inline void mark_initiator( Pix p, byte index ) { fInitiators.subset_add( p, index ); }	
	inline byte initiator_marked( Pix p, byte index ) { return fInitiators.subset_contains( p, index ); }	
};

class ArrayRec : public TNamedObject {
  ArgArray* _args;
  Variable* _map;
  public:
  ArrayRec( CString& name, ArgArray* args, Variable* map ) : TNamedObject(name) {
	_args = args; _map = map;
  }
  void WriteDataToConfig( CStream& outStream ) ;
  void WriteConfigXML( FILE* oFile );

  ArgArray* getArgs() { return _args; }
  Variable* getMap() { return _map; }
  
};

//----------------------------------------------------------------------------------------
//			       Model
//----------------------------------------------------------------------------------------

class Model : public TConfigObject, public MMLObject { 

protected:

  int fNSim;
  int fSeed;
  int fMaxGhost;
	int fSMP;
	float fYear;
	float fDTperYear;
  int fParserDebug;
  static int fTM;
  float fUpperCornerUTM[2];
  float fLowerCornerUTM[2]; 

 	static Model* fCurrentModel;
  static TNamedObjectList  fModelList;
  TNamedObjectList  fCodeFileList;
  TNamedObjectList fDerivatives;
  float fDerivativeWeight;
  byte fDerivativeMode;
  TNamedObjectList _Arrays;
  CString fGOFCmds;  
  CString fComment;  
  CString fSliderParms;

	Module* fCurrentModule;
	Variable* fCurrentVariable;
  TCommand* fCurrentCommand;
  Module* fGlobalModule;
  TEvent* fCurrentEvent;
#ifdef POSTGRESS
  sme_pg95* fDB;
#endif
  CString fDataSource;

	TNamedObjectList  fModuleList;
  TNamedObjectList  fVariableList;
  TNamedObjectList fParameterList;
  TNamedObjectList fEventList;

  ArgArray fMMLFileList;
  ArgArray fConfigFileList;
  ArgArray fInputFileList;
  CString fMMLOutFile;
  CString fConfigOutFile;
  CString fConfigTemplateOutFile;
  ExSchedule fSchedule;
  CnfgProcessor fCR;
	int fActivationLayer;
	int fLinkLayer;
	LayerConfig* fLayerConfig;
	
  static Model* fInstance;
  MFrame* fFrame;
	int fMemoryOp;

	CString fDBName;
	CString fDBPort;
	CString fDBHost;
	
  inline TEvent* GetNextEvent() {
    Pix p = fEventList.first();
    return fCurrentEvent = (p) ? (TEvent*) &fEventList(p) : (TEvent*) NULL;
  }
  
  void ClearEventList();
  void WriteNameToConfigFile(CStream& cnfgStream, char type, const CString& name);

  inline ExSchedule* CreateDefaultSchedule() {
    float dt = 1.0, start = 0.0, stop = 10.0;
    return new ExSchedule("Global_Default", dt, start, stop );
  }
	inline virtual  MFrame* GetFrame() { return new MFrame(); }

public:

	byte Ivar[100];
  static EConfigInfo kDebug;		
  static EConfigInfo kOptLevel;
  enum EStepMode { kEvent, kCommand, kUpdate };

  Model(const char* name = "Model") : TConfigObject ( name, kModel ), fSchedule("Global"), fDataSource(""), 
		fDBName("default"), fDBPort("default"), fDBHost("default"), fSliderParms("STELLASliders.data")
  { fFrame = NULL; fCurrentEvent = NULL; fGlobalModule = NULL; fMemoryOp = 0; 
#ifdef POSTGRESS
		fDB = NULL; 
#endif
		fParserDebug=0; fDerivativeWeight=1.0; fDerivativeMode=0;	fActivationLayer = -1; fLinkLayer = -1; fLayerConfig = NULL;
		fLowerCornerUTM[0] = 0; fLowerCornerUTM[1] = 0; fUpperCornerUTM[0] = 1;  fUpperCornerUTM[1] = 1; 
#ifdef USE_MPI
		fMaxGhost=1;
#else
		fMaxGhost=0;
#endif
	}

  inline static Model& I0() { 
    if(fInstance == NULL) { gFatal("Error, Model instance must be allocated by child class."); }
    return *fInstance; 
  }
  inline LayerConfig* SetDefaultLayers( int& activationLayer, int& linkLayer ) { 
		activationLayer = fActivationLayer; 
		linkLayer = fLinkLayer; 
		return fLayerConfig;
	}
  int FrameConfig( TConfigData& cd, Module* m);
  
	inline void ModuleListDump(FILE* oFile=NULL) {
		FILE* f = (oFile) ? oFile : gLogFile;
		fprintf(f,"\n Modules:\n");
		fModuleList.Dump(f);
	}
	inline void VariableListDump(FILE* oFile=NULL) {
		FILE* f = (oFile) ? oFile : gLogFile;
		fprintf(f,"\nVariables:\n");
		fVariableList.Dump(f);
	}
	inline void EventListDump(FILE* oFile=NULL) {
		FILE* f = (oFile) ? oFile : gLogFile;
		fprintf(f,"\nEvents:\n");
		fEventList.Dump(f);
	}
	
  inline TNamedObjectList& getArrayList() { return _Arrays; }	
  virtual void QueryInterface( const char* moduleName, const char* variableName = NULL ) { ; }
	virtual void QueryInterface() { ; }
	Module* GetModule( const CString& moduleName, Bool create = True );
  virtual Module* CreateNewModule( const CString& name );
  inline void AddParameter( TNamedObject& v ) {  fParameterList.Insert(v); }

  int Config( TConfigData & cd );
  void ReadSliderParms();
  virtual int ConfigExternal( TConfigData& cd, Module* m ) { return 0; }
  int Init();

  inline Module* CurrentMod(Module* mod = NULL) { return (mod) ? fCurrentModule = mod : fCurrentModule; }
  void  BuildConfigXML( ArgArray& msg, int mode ); 
  inline Module* GlobalMod(Module* mod = NULL) { return (mod) ? fGlobalModule = mod : fGlobalModule; }
  inline const Module* GlobalMod() const { return fGlobalModule; }
  inline Module* CurrentMod(CString name) { return fCurrentModule = GetModule(name); }
  inline Module* GlobalMod(CString name) { return fGlobalModule = GetModule(name); }
  inline TEvent* CurrentEvent() { return fCurrentEvent; } 
  
	inline static Model* Current() { return fCurrentModel; }
	inline static TNamedObjectList&  ModelList() { return fModelList; }
	static Model* New(const char* name) ;
		
	inline Variable* CurrentVariable() { return fCurrentVariable; }
	virtual inline int GetFormat() { return -1; }
 	virtual inline int SetFormat(int f) { return 0; }
  	
  virtual void WriteDataToConfig( CStream& outStream );
  void CreateEventLists(); 
  void ReadConfigFiles() ;
  TNode* ReadMMLFiles();
  void ReadSMMLFiles();
  TNode* ReadMMLFile( const CPathString& path, const char* file, int process = True );
  void WriteConfigFile(const char* name = NULL);
  virtual int ReadSetupFile( char* cnfgExtension ) ;
  int DBaseObjectFileRW( char mode );
  int UpdateDBaseInCache();
  int SaveDBaseOutCache();
  int ParseMMLFile( const CString&  path, int warn=1 );
  Pix PostPipe( TPipeCommand& c );
  void addArrayDeclaration( ArrayRec* array ); 
  ArrayRec* getArrayDeclaration( const CString& name, int& index, int& is_dimension_name  ); 

  virtual int ProcessCommand(const TCommandNode* n);
	virtual int ProcessAttribute(const TAttributeNode* n);
	virtual int ProcessGeneral( TNode* n);
	virtual int RunNodeProcessing( EProcessContext pc );
	void InitializeGlobalDependencies(TDerivative* d);
	void AddCodeFile( CString& codeFileName );
 
  virtual void SetFieldsFromMML() {;}
  MFrame& Frame() { if( fFrame==NULL ) fFrame = GetFrame(); return *fFrame; }
	Pix AddVariable( Variable* v );
	Variable* GetVariable(const char* name);
//	Variable* GetSpecialVariable(const CString& name);

  virtual int AddModifier( const CString& mod, Bool isExtension = False ) { return 0;}
  virtual void WriteMML( FILE* oFile, EWriteMode mode = kDefinition );
  virtual void WriteXML( CString xmlPath );
  void WriteConfigXML( CString xmlPath );
  TNamedObjectList&  ModuleList() { return fModuleList; }
  TNamedObjectList&  ParameterList() { return fParameterList; }
  virtual void Dump( FILE* oFile );
  virtual void  WriteCCode( FILE* CHdrFile, FILE* CEqnFile);
  inline void RemoveEvent(Pix p) { fEventList.del(p); }
  Pix PostEvent( TEvent& e, float next_time, TEvent::ECmdPosition cp = TEvent::kEnd  );
  int ReadDBParamValues( const char* filename, int overrideParameters=0 );
  void DumpEventList();
  int RunModel ( float stop_time = FLT_MAX, int breakIndex =0, int set_context=0 );
  virtual inline int Open() { return 0; }
  virtual inline void Close() { ; }
  float StepModel (EStepMode sm = kEvent, int breakIndex =0, int set_context=1 );
  inline ExSchedule* GetSchedule() {
    if( !fSchedule.ScheduleConfigured() ) fSchedule = *CreateDefaultSchedule();
    return &fSchedule; 
  }
  int QueryInterupt();
  void Interrupt();

  inline void SetContext(TEvent* currentEvent, TCommand* currentCommand, Variable* currentVariable, Module* currentModule) {
    fCurrentModule = currentModule;
    fCurrentVariable = currentVariable;
    fCurrentCommand = currentCommand;
    fCurrentEvent = currentEvent;
  }

  inline TEvent* SetCurrentContext(Module*& currentModule, Variable*& currentVariable, TCommand*& currentCommand) {
    currentModule = fCurrentModule;
    currentVariable = fCurrentVariable;
    currentCommand = fCurrentCommand;
    return fCurrentEvent;
  }

  inline ArgArray& MMLFileList() { return fMMLFileList; }
  inline ArgArray& ConfigFileList() { return  fConfigFileList; }
  inline ArgArray& InputFileList() { return  fInputFileList; }
  inline CString& MMLOutFile() { return  fMMLOutFile; }
  inline CString& ConfigOutFile() { return  fConfigOutFile; }
	inline int MemoryOp() { return fMemoryOp; }
	virtual int Finalize();
#ifdef POSTGRESS
	inline sme_pg95* DataBase() { return fDB; }
#endif
	inline int ParseDebugLevel() { return fParserDebug; }
	void WriteDBParameters();
	void FillEventListByDependency( TEvent& event );
	TDerivative* GetDerivative( Variable* indep_var ); 
	inline TNamedObjectList& Derivatives() { return fDerivatives; }
  inline int SMP() { return fSMP; }
	inline float UTM( int coordType, int dim ) {
		if( coordType == 0 ) { return fLowerCornerUTM[dim]; }
		else { return fUpperCornerUTM[dim]; }
	}
};



#endif
