#if __GNUG__ >= 2
#  pragma implementation
#endif

#include "ActivationLayer.h"
#include "MultiGrid.h"

//---- CellPlex ---------------------------------------------------------

void CellPlex::copyRegion( MultiGrid* g, Region2& r, int layer_index ) {
	TLayer* l = g->getCellLayer( layer_index );
	for( TCell* c = l->firstCell(); c;  l->nextCell(c) ) {
		if( c->GetObjInfo( TCell::kGhostIndex ) == 0 ) {
			Point2& p = l->getCellLocPoint( c );
			if( r.inside(p) ) add( c );
		}
	}
}

//---- ActivationLayer ---------------------------------------------------------


int32 ActivationLayer::finalizeCell( TCell* c ) {
	int32 cell_index = CellPlex::add( c ); 
	int ig = 0;
#ifdef USE_MPI
	if( (ig = c->GetObjInfo( TCell::kGhostIndex )) > 0	 ) {
		int owner = c->GetObjIndex( TCell::kOwner );
		if( ghostCellMap(owner) == NULL ) { _ghostCellMap[owner] = new GhostCellMap( _maxGhost );  _nReceiveRequests++; }
		_ghostCellMap[owner]->AddEntry( cell_index, ig, kL_Input );
		TMsgBuff* mb = 	ghostMapData(owner);
		mb->PackByte( c->GetObjInfo( TCell::kLayerIndex )  );
		mb->PackByte( c->GetObjInfo( TCell::kGhostIndex )  );
		mb->PackInt32( c->Index() );
		_sendmapGenerated = False;	
	} 
#endif
	c->finalizeActivation( Index(), cell_index );
	return cell_index;
}

void ActivationLayer::addCell( TCell* c ) {
	byte index = c->GetObjInfo( TCell::kGhostIndex );
	PixArray* pv; 
	if( index > 0 ) {
		pv = (PixArray*)_ghostBuffer.elem(index);
		if( pv == NULL ) {
			_ghostBuffer.add(index) = pv = new PixArray(200,100,(Pix)0);
		}
	} else {
		index = c->GetObjInfo( TCell::kLayerIndex );
		if( _currentLayerIndex < 0 ) { _currentLayerIndex = index; }
		else if( _currentLayerIndex != index ) { _multiLayers = True; }
		pv = (PixArray*)_cellBuffer.elem(index);
		if( pv == NULL ) {
			_cellBuffer.add(index) = pv = new PixArray(1000,1000,(Pix)0);
		}
	}
	pv->add(c);
}

inline unsigned long  CellRank( Pix p ) { 
#ifdef DEBUG
	if( p==0 ) { gPrintErr(" NULL pix error in sort " ); return 0; }
#endif
	return ((TCell*)p)->Index(); 
}

void ActivationLayer::finalizeCells() {
	 if( !_cellsFinalized ) {
		 PixArray* pv; TCell* c; int i;
		 int32 nCells, mem_index = 0;
		 for( i=0; i<_cellBuffer.nElem(); i++) {
			 pv = (PixArray*)_cellBuffer[i];
			 if( pv != NULL ) {
				 nCells = pv->nElem();
				 if( nCells > 0 )  {
					 pv->qsort(0,nCells-1,&CellRank);
					 for( int j=0; j<nCells; j++) {
						 c = (TCell*) (*pv)[j];
						 if( c != NULL ) {
							 mem_index = finalizeCell( c );
							 if( _LayerOffset(i) < 0 ) { 
								 _LayerOffset.elem(i) = mem_index; 
							 }
							 _nActiveCells++;
						 }
						}
					} else if( _LayerOffset(i) < 0 ) { _LayerOffset.elem(i) = mem_index;   }
				 delete pv;
			 } else if( _LayerOffset(i) < 0 ) { _LayerOffset.elem(i) = mem_index;   }
		 }
		 _LayerOffset.elem( _cellBuffer.nElem() ) = _nActiveCells; 
		 _cellBuffer.free();  
		 for( i=0; i<_ghostBuffer.nElem(); i++) {
			 pv = (PixArray*)_ghostBuffer[i]; 
			 if( pv != NULL ) {
				 nCells = pv->nElem();
				 if( nCells > 0 )  {
					 pv->qsort(0,nCells-1,&CellRank);
					 for( int j=0; j<nCells; j++) {
						 c = (TCell*) (*pv)[j];
						 if( c != NULL ) {
							 mem_index = finalizeCell( c );
							 if( _GhostOffset(i) < 0 ) { _GhostOffset.elem(i) = mem_index; }
							 _nGhostCells++;
						 }
					 }
				 } else if( _GhostOffset(i) < 0 ) { _GhostOffset.elem(i) = mem_index; }
				 delete pv;
			 } else if( _GhostOffset(i) < 0 ) { _GhostOffset.elem(i) = mem_index; }
		 }
		 _GhostOffset.elem( _ghostBuffer.nElem() ) = _nActiveCells + _nGhostCells; 
		 _ghostBuffer.free();
		 _cellsFinalized = TRUE;
	 }
}

float ActivationLayer::mapValue( TCell* c, int activationLevel, const float* origonData, Bool doAverage, float nullValue ) {
	if( c->activated( activationLevel ) ) {
		return origonData[ c->memoryIndex(activationLevel) ];
	}
	if( c->activatedAbove( activationLevel ) ) {
		while( c = c->parent() ) {
#ifdef DEBUG
			if( c == NULL ) { gPrintErr( "Can't find parent cell in ActivationLayer::mapValue."); return nullValue; }
#endif
			if( c->activated( activationLevel ) ) {
				return origonData[ c->memoryIndex(activationLevel) ];
			}
		}
	}
	if( c->activatedBelow( activationLevel ) ) {
		float sum_value = nullValue;
		if( doAverage ) {
			int icount = 0;
			c->mergeDataOverChildren(origonData,activationLevel,sum_value,&icount);
			return (icount>0) ? sum_value/icount : nullValue;
		} else {
			c->mergeDataOverChildren( origonData, activationLevel, sum_value );
			return sum_value;
		}
	}
	return nullValue;	
}

#ifdef USE_MPI

TMsgBuff* ActivationLayer::ghostMapData(int iproc) { 
	TMsgBuff* rv = (TMsgBuff*) _ghostMapData[iproc];
	if( rv == 0 ) {
		rv =  new TMsgBuff(1000,MPI_COMM_WORLD);
		rv->PackInt32(gIProc);
		rv->PackInt32(fIndex);
		_ghostMapData[iproc] = rv;
	}
	return rv;
}

void ActivationLayer::transferSendmapData(int nprocs) {
	byte* dataBuff = new byte[nprocs]; 
	memset( dataBuff, 0, nprocs );
	for( Pix px =  _ghostMapData.first(); px; _ghostMapData.next(px) ) {
		int dest = _ghostMapData.key(px);
		dataBuff[dest] = 1;
	}
	MPI_Alltoall(dataBuff,1,MPI_BYTE,dataBuff,1,MPI_BYTE,MPI_COMM_WORLD);
	for( Pix p =  _ghostMapData.first(); p; _ghostMapData.next(p) ) {
		int dest = _ghostMapData.key(p);
		TMsgBuff* mb = (TMsgBuff*) _ghostMapData[dest];
		mb->ISend(399,dest);
	}
	TMsgBuff inBuff(1000,MPI_COMM_WORLD); _nSendRequests = 0;
	for( int i=0; i<nprocs; i++ ) { if( dataBuff[i] ) _nSendRequests++; }
	int nmsg = _nSendRequests;
	while(nmsg) {
		inBuff.Recv(399);
		createSendMap(inBuff);
		nmsg--;
	}	
	_sendmapGenerated = True;
	delete[] dataBuff;
}

void ActivationLayer::createSendMap( TMsgBuff& inBuff ) {
	int npoints = (inBuff.DataSize()-8)/6;
	int source = inBuff.UnPackInt32();
	int activation_index = inBuff.UnPackInt32();
	if( ghostCellMap(source) == NULL ) { _ghostCellMap[source] = new GhostCellMap( _maxGhost ); }
	GhostCellMap* map = _ghostCellMap[source];
	for( int i=0; i<npoints; i++ ) {
		int layer = inBuff.UnPackByte(); 
		byte ghost_index = inBuff.UnPackByte(); 
		int32 key = inBuff.UnPackInt32(); 
		TLayer* l = _multiGrid->getCellLayer( layer );
		if( l == NULL ) { sprintf(gMsgStr,"Illegal layer %d in createSendMap",layer); gPrintErr(); }
		else {
			TCell* c = l->getCell(key);
			if( c == NULL ) { sprintf(gMsgStr,"Illegal key %d for layer %d in createSendMap",key,layer); gPrintErr(); }
			else {
				int32 mIndex = c->memoryIndex(activation_index);
				if( gDebug>1 ) { sprintf(gMsgStr,"Adding cell to Send Map: index %d, ghost %d",mIndex,ghost_index); gPrintLog(); }
				map->AddEntry( mIndex, ghost_index, kL_Output );
			}
		}
	}
}

GhostCellMap* ActivationLayer::ghostCellMap( int iproc ) {
	if( _ghostCellMap == NULL ) {
		_ghostCellMap = (GhostCellMap**) malloc ( sizeof(GhostCellMap*) * gNProc );
		memset( _ghostCellMap, 0, sizeof(GhostCellMap*) * gNProc );
	}
	return _ghostCellMap[iproc];		
}

MPI_Request* ActivationLayer::setupCommunication( int& nrequest, int nprocs, int nGhost, const float* data, MPI_Comm comm, int tag ) {
	int local_debug = 0;
	_maxGhost = nGhost;
	finalizeCells();
  if( !_sendmapGenerated ) {  transferSendmapData(nprocs); }
	static int MPIBuffersize  = 0;
	int buff_size, ireq=0, ip;
	MPI_Datatype cellMap;
	nrequest = _nSendRequests + _nReceiveRequests;
	MPI_Request* request = new MPI_Request[ nrequest ];
	if( local_debug ) { lprint2("LinkEdges: Created MPI requests, %d send, %d receive.",_nSendRequests,_nReceiveRequests); }
	for(ip=0; ip<nprocs; ip++) if( _ghostCellMap[ip] ) {
		cellMap = _ghostCellMap[ip]->GetGhostIndexMap( nGhost, kL_Input );
		if( cellMap != MPI_DATATYPE_NULL ) {       // post receive request
			MPI_Type_size(cellMap,&buff_size);
			if( buff_size > 0 ) {
				MPI_Recv_init((void*)data,1,cellMap,ip,tag,comm,request+(ireq++));
				if(local_debug) {
					sprintf(gMsgStr,"\nPosted Receive Request, index(%d), Proc(%d), data size = %d, ghostSize = %d",ireq-1,ip,buff_size,nGhost);
					gPrintLog(); 
				}
			}
		}
	}	
	for(ip=0; ip<nprocs; ip++) if( _ghostCellMap[ip] ) {
		cellMap = _ghostCellMap[ip]->GetGhostIndexMap( nGhost, kL_Output );
		if( cellMap != MPI_DATATYPE_NULL ) {     // post send request
			MPI_Type_size(cellMap,&buff_size);
			if( buff_size > 0 ) {
				MPI_Send_init((void*)data,1,cellMap,ip,tag,comm,request+(ireq++));
				if(local_debug) {
					sprintf(gMsgStr,"\nPosted Send Request, index(%d), Proc(%d), data size = %d, ghostSize = %d",ireq-1,ip,buff_size,nGhost);
					gPrintLog();
				}
			}
		}
	}
	return request;
}

void ActivationLayer::dumpTransferData( int iproc, int nGhost, const float* data, EG_LinkMode mode ) {
	if( _ghostCellMap[iproc] ) { _ghostCellMap[iproc]->DumpData( nGhost, data, mode ); }	
}
#endif
