#ifndef __Coverage__
#define __Coverage__

#include "DistributedGrid.h"
#include "MemoryMgr.h"
#include "TMap2.h"

/*
*************************************************************************
*									*
* PointSetCoverage     					        *
*									*
*************************************************************************
*/

enum EReduceOp { kR_Max, kR_Min, kR_Sum, kR_Ave };
#define PSCserial 		1
#define PSCfullGrid 	2
#define PSCsubGrid 		3

class PointSetCoverage: public MemObject, public floatVec {

  int fNReq;
  int fTag;
  byte fInfo[8];
  float fGarbage;
  int fNObjects;

  static MemoryManager* fMemMgr;

#ifdef USE_MPI
  static MPI_Comm fComm;
  MPI_Request* fReq;
  MPI_Status* fStatus;
#endif

  int SetupCommunication(int do_buffer=0);
  void Compress( byte* data, int bsize );
  byte*  MergeData(byte* dest, byte* src, int srcProc, int bsize, int nelem);

protected:

  DistributedGrid* fPSet;
  int fGhostSize;

public:

  enum EInfo { kBaseType };
  enum EUpdateMode { kEulerClamped, kEuler };
  enum EDataLayout { kCompressed, kGrid }; 

  PointSetCoverage();
  PointSetCoverage( DistributedGrid* list, int gSize = 0, int memoryIndex = MemObject::kPermanentMemory );
  PointSetCoverage(  PointSetCoverage& cov ) { Setup( cov.fPSet, cov.fGhostSize, cov.fMemoryIndex ); }

  int Setup( DistributedGrid* list, int gSize, int memoryIndex = MemObject::kPermanentMemory  );
  int Setup( DistributedGrid* pset, PointSetCoverage& cov, int gSize, int memoryIndex = MemObject::kPermanentMemory );
  int Setup( PointSetCoverage& cov, int memoryIndex  = MemObject::kPermanentMemory );
  inline int Allocated() { return ( fPSet == NULL ) ? 0 : 1; } 
	inline TMap2* RegionMap() { return fPSet->RegionMap(); }

	inline void PointSetCoverage::ShallowCopy(FloatStore& v){ shallow_copy(v); }

  int LinkEdges(const char* name);
  inline DistributedGrid& Grid() { 
    if(fPSet == NULL) gFatal( "Attempt to Access Unallocated Grid."); 
    return *fPSet; 
  }
  inline DistributedGrid* GridP() { 
    return fPSet; 
  }
  inline Pix MapPix( Pix p, DistributedGrid& g ) { return fPSet->MapPix(p,g); }
  inline Pix MapPixFast( Pix p, DistributedGrid& g ) { return fPSet->MapPixFast(p,g); }

  inline const OrderedPoint* GetPoint( Point2& point ) const { 
    PointRef pr( point(0), point(1), Region() );
    return fPSet->GetPoint(pr);
  }

  inline Pix GetPix ( Point2& point ) const { 
    const OrderedPoint* pt = GetPoint(point); 
    return (pt) ? fPSet->GetPix(*pt) : (Pix)0; 
  }

  virtual void Dump( FILE* oFile ){ fprintf(oFile,"PointSetCoverage Dump:\n"); TextDump(oFile); }

  int StartLinkEdges();
  int CompleteLinkEdges(const char* name);

  PointSetCoverage& operator=( ByteGrid& map );

	inline void GrabMemory() { ((FloatStore*)fOrigin)->SetCurrentOwner(this); }
	inline int PointSetCoverage::CheckMemory() { return ((FloatStore*)fOrigin)->CheckCurrentOwner(this); }

  inline MemoryManager* Manager() { return fMemMgr; }
  inline DistributedGrid* PointSet() const {  return fPSet; }
  inline const float* Data() { return data(); }
  int Size() { return fNObjects; }

  int CopyData( const float* fdata, int nelem ) { 
    if( nelem != fNObjects ) { gPrintErr("Bad Data size PointSetCoverage::CopyData"); return 0; } 
    deep_copy(fdata,nelem);
    return nelem;
  }

  inline Region2& Region(int iproc=gIProc ) const { 
#ifdef DEBUG
    if( fPSet == NULL ) gFatal( "Attempt to access unallocated Frame.");
#endif 
    return fPSet->Region(iproc); 
  } 

  inline Region2& GlobalRegion( ) const {
#ifdef DEBUG
    if( fPSet == NULL ) gFatal( "Attempt to access unallocated Frame.");
#endif 
    return fPSet->GlobalRegion(); 
  } 

  void Condense( byte*& cData, int targetProc, int bsize, EDataLayout layout, float s = 1.0, float o = 0.0 );
  void CondenseToMap( TMap2& map, int targetProc );
  void Initialize() { reinitialize(); }

  float AggregateValue( const OrderedPoint& point, int scale );

  inline float Value(int index) const {
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage");  } 
#endif
    return operator[] (index);
  }    

  inline float Value( const OrderedPoint& point ) const {
    int index = fPSet->Index( point );
    if( index < 0 ) return 0.0;
#ifdef DEBUG
    if( index >= fNObjects ) { gFatal("Illegal Index in PointSetCoverage");  } 
    return operator[] (index);
#else
    return operator[] (index);
#endif
  }
  
  inline float Value( Pix p ) const {
    const int index = fPSet->Index( p );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage");  } 
#endif
    return operator[] (index);
  }

  
  inline float ValueRel( const OrderedPoint& point, int dr, int dc ) const {
    Pix p0 = fPSet->GetPix(point);
    return  ((p0) ? ValueRel( p0, dr, dc ) : 0.0);
  }

  inline float& operator() ( Pix p0, Point2 ptrel ) {
    Pix rp = fPSet->TranslateByLinks( p0, ptrel );    
    if(rp == 0) { 
#ifdef DEBUG
      if(p0 == 0) sprintf(gMsgStr,"Null Pix in ValueRel\n"); 
      if(gDebug>2){
	const OrderedPoint& point = fPSet->GetPoint(p0);
	sprintf(gMsgStr,">>Bad Point Access: (%d,%d)->(%d,%d)\n",point(0),point(1),ptrel(0),ptrel(1)); 
	gPrintLog(); 
      }
#endif
      return fGarbage = 0;
    }
    const int index = fPSet->Index( rp );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage");  }
#endif
    return elem(index);
  }

  inline float& operator() ( Pix p0, int lIndex, int nLinks ) {
    Pix rp = fPSet->TranslateByLinkIndex( p0, lIndex, nLinks );    
    if(rp == 0) { 
#ifdef DEBUG
      if(p0 == 0) sprintf(gMsgStr,"Null Pix in ValueRel\n"); 
      if(gDebug>2){
	const OrderedPoint& point = fPSet->GetPoint(p0);
	sprintf(gMsgStr,">>Bad Point Access: (%d,%d)->( L:%d, nL:%d )\n",
		point(0),point(1),lIndex,nLinks); 
	gPrintLog(); 
      }
#endif
      return fGarbage = 0;
    }
    const int index = fPSet->Index( rp );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage");  }
#endif
    return elem(index);
  }

  inline float ValueRel( Pix p0, int dr, int dc ) const {
    Pix rp = fPSet->TranslateByLinks( p0, dr, dc );
    if(rp == 0) { 
#ifdef DEBUG
      if(p0 == 0) sprintf(gMsgStr,"Null Pix in ValueRel\n"); 
      if(gDebug>2){
				const OrderedPoint& point = fPSet->GetPoint(p0);
				sprintf(gMsgStr,">>Bad Point Access: (%d,%d)->(%d,%d)\n",point(0),point(1),dr,dc); 
				gPrintLog(); 
      }
#endif
      return 0.0;
    } 
    const int index = fPSet->Index( rp );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage"); } 
#endif
    return  operator[] (index);
  }
 
  inline PointSetCoverage& SetValue(int index,float value) {
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage");  } 
#endif
    elem(index) = value;
    return *this;
  } 
 
  inline PointSetCoverage& SetValue( OrderedPoint& point, float value ) {
    int index = fPSet->Index( point );
    if(index<0) return *this;
#ifdef DEBUG
    if(index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage");  }
#endif
    elem(index) = value;
    return *this;
  }

  inline float& operator() ( const OrderedPoint& point ) {
    int index = fPSet->Index( point );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage");  }
#endif
    return elem(index);
  }

  inline int SetValue( Pix p, float value ) {
    const int index = fPSet->Index( p );
    elem(index) = value;
#ifdef DEBUG
		if( ( gDebug > 1 ) && DistributedGrid::fDebug ) { 
			sprintf(gMsgStr,"\nSet Point (%d) -> (%f)",index,value);
			gPrintLog(); 
		}
#endif
    return index;
  }
  
  inline int AddValue( Pix p, float value ) {
    const int index = fPSet->Index( p );
    elem(index) += value;
    return index;
  }

  inline float& operator()( Pix p ) {
    const int index = fPSet->index( p );
#ifdef DEBUG
    if(index<0 || index >= fNObjects) { gFatal("Illegal Index in PointSetCoverage");  }
#endif
    return elem(index);
  }

  inline void Set(float value) { for(int i=0; i<fNObjects; i++) elem(i) = value; }

  inline float* DataStart() { return  fValue; }
  inline const float* DataEnd() { return  data() + fNObjects; }
  inline const float* GridEnd() { return  data() + fPSet->InteriorSize(); }

  void SubSet(byte index, float value);
  PointSetCoverage& operator = (PointSetCoverage& b);
  float Reduce ( OrderedPointSet& set, EReduceOp op, int rootProc=0 );
  float RemoteValue( Pix p, int recvProc=0 );    // p should be non-zero on only one proc!
  void CopyToMap(  TMap2& map, Bool swap = FALSE, Bool directScale = FALSE );           // Convert to map;
//  void CopyToMapRescaled( TMap2& m, float min, float max, 
//    float umin, float umax, Bool rescale = FALSE);

	void SetDataValues( float* data, int layOut, int size );
};


typedef PointSetCoverage Cov;

#endif

