//----------------------------------------------------------------------------------------
//	DistributedGrid.cc
//	Developed by Tom Maxwell, MIIEE, Chesapeake Biological Lab.
//----------------------------------------------------------------------------------------

// Setting rows:0 cols:1   And scale already factored in.

#include "DistributedGrid.h"
#include "Ghost.h"
#include "FLBase.h"

DistributedGrid* gCurrentPGL = NULL;

/*
*************************************************************************
*									*
* DistributedGrid								*
*									*
*************************************************************************
*/

Pix DistributedGrid::firstLink( Pix p, EPointOrdering po, int& lindex, int link_layer ) {
	if( link_layer < 0 ) link_layer = fGridLinkLayer;
	if( fGridLinkLayer = link_layer ) {
		return fGridLayer->neighborCell( (TCell*)p,  link_layer, lindex = 0 ); 
	} 
	else if( GetInfo(kFrameType) == kTree ) {
		switch( po ) {
		 case kPO_Undefined:
			 gPrintErr("Undefined Point Ordering for firstLink");
			 return NULL;
		 case kGridOrdering:
			 gPrintErr("Wrong Frame Type for this Point Ordering");
			 return NULL;
		 case kUpLinkOrdering:
			 fGridLayer->neighborCell( (TCell*)p, link_layer, lindex = 1  ); 
		 case kDownLinkOrdering:
			 fGridLayer->neighborCell( (TCell*)p, link_layer, lindex = 0  ); 
		}
	} else gPrintErr("Point Ordering not implemented for this frame type.");
	return 0;
}

Pix DistributedGrid::nextLink(Pix p, EPointOrdering po, int& lindex, int link_layer) {  
	if( link_layer < 0 ) link_layer = fGridLinkLayer;
	if( fGridLinkLayer = link_layer ) {
		return fGridLayer->neighborCell( (TCell*)p, link_layer, lindex = 0  ); 
	} 
	else if( GetInfo(kFrameType) == kTree ) {
		switch( po ) {
		 case kPO_Undefined:
			 gPrintErr("Undefined Point Ordering for nextLink");
			 return NULL;
		 case kGridOrdering:
			 gPrintErr("Wrong Frame Type for this Point Ordering");
			 return NULL;
		 case kUpLinkOrdering:
			 fGridLayer->neighborCell( (TCell*)p, link_layer, lindex++ ); 
		 case kDownLinkOrdering:
			return NULL;
		}
	} else gPrintErr("Point Ordering not implemented for this frame type.");
	return 0;
}

int DistributedGrid::CalcPointOrdering( EPointOrdering po, const OrderedPoint* pt ) {	
	int ordering_index = fGridInfo[kOrderingIndex];
	if( ordering_index == 0 ) return 0;
	if( po != kUpLinkOrdering ) { gPrintErr(" This Point Ordering Not yet Implemented." ); return -1; }
	int pv_start=0, pv_end = 0, lindex;
	Region2& local_region = Region();
	Region2 globalRegion; GlobalRegion(globalRegion);
	TMap2 point_map( &globalRegion );
	PixVec& pointVec = fPointOrdering[ordering_index-1];
	Pix p;
	point_map.Set(0);
	if( pt == NULL ) {
		pv_end = pointVec.copy(fRootPointVec);
	} else {
		pointVec.elem(pv_end++) = (void*)pt; 
	}
	if(pv_end==0) { gPrintErr("No start points in UpLink Ordering."); return ordering_index; }
	while(1) {
		int new_points = 0, new_bpoints=0, go=0;						
		while( p = pointVec.elem(pv_start) ) {	
			pv_start++;	
			TCell* c = (TCell*)p;
			int gi =  c->GetObjInfo( TCell::kGhostIndex );

			if( gDebug > 1 ) {
				sprintf(gMsgStr,"\n\nDGrid (%d): (%d,%d:%d)",pv_start,row(c),col(c),gi);
				gPrintLog();
			}
			if( gi == 0 ) {
				for( Pix p1 = firstLink(p,po,lindex); p1; p1 = nextLink(p,po,lindex) ) {
					TCell* c1 = (TCell*)p1;
					int gi1 =	c1->GetObjInfo( TCell::kGhostIndex );
					int ir = row(c1), ic = col(c1);
					if( point_map.Value(ir,ic) == 0 ) {
						point_map.SetValue(ir,ic,1);
						pointVec.elem(pv_end++) = p1; 
						new_points++;
						if(gDebug > 1) {
							sprintf(gMsgStr,"\nAdd Point(%d:%d:%d): (%d,%d)\n",pv_end,lindex,gi1,ir,ic);
							gPrintLog();
						}
#ifdef USE_MPI
						if( gi1 > 0 ) {
							fCommBuff.resize((++new_bpoints)*sizeof(int)*2);
							int* ibuff = (int*)fCommBuff.buff();
							ibuff[new_bpoints*2-2] = ir;
							ibuff[new_bpoints*2-1] = ic;
						}
#endif
					}
				}
			}
		}
#ifdef USE_MPI
		for( int ip=0; ip<gNProc; ip++ ) {
			int size = new_bpoints;
			MPI_Bcast( &size, 1, MPI_INT, ip, MPI_COMM_WORLD );
			if(size>0) {
				go = 1; 
				fCommBuff.resize(size*sizeof(int)*2);
				MPI_Bcast( fCommBuff.buff(), size*2, MPI_INT, ip, MPI_COMM_WORLD );
				for( int ipt=0; ipt<size; ipt++ ) {
					int* ibuff = (int*)fCommBuff.buff();
					unsigned int ir = ibuff[ipt*2], ic = ibuff[ipt*2+1];
					TCell* c0 = fGridLayer->getCell(ir,ic); 
					if( c0 && ( point_map.Value(ir,ic) == 0 ) ) {
						pointVec.elem(pv_end++) = c0; 
						if(gDebug > 1) {
							sprintf(gMsgStr,"\nAdd Boundary Point(%d:%d): (%d,%d)\n",pv_end,ipt,ir,ic);
							gPrintLog();
						}
					}
				}
			}
		}
#endif
		if( !go ) break;
	} 
	if( gDebug>1 ) { 
		CString mapName =  "DGrid_PointMap"; 
		mapName.appendIndex(ordering_index);
		CPathString pathName(Env::ArchivePath());
		point_map.WriteM2File( pathName, mapName, gIProc, NULL, gIProc);
	}
	return ordering_index;
}








