#include "Ghost.h"
#include "sme_types.h"

/****************************************************************
* Region2 GhostIterator::operator()() const			*
*								*
* returns the current ghost Region2				*
*****************************************************************/

Region2 GhostIterator::operator()() const
{
   int axis = state / 2;
   int direction = (state % 2);

   Region2 result = R;

   if (!direction) {
      Point2 upp = R.upper();
      upp.elem(axis) = R.lower(axis) + ghost(axis) - 1;
      result.setupper(upp);
   }
   else {
      Point2 lwp = R.lower();
      lwp.elem(axis) = R.upper(axis) - ghost(axis) + 1;
      result.setlower(lwp);
   }

   return result;
}
/****************************************************************
* various simple operators					*
*****************************************************************/
void GhostIterator::operator ++()
{
   state++;
}

int GhostIterator::operator !()  const
{
   return (state >= 2*2);
}


GhostIterator::operator void *()
{
   return (state >= 2*2) ? 0:this;
}

/*
*************************************************************************
*									*
* GhostIndexSet								*
*									*
*************************************************************************
*/


GhostIndexSet::GhostIndexSet( int nLayers, EG_LinkMode mode ) : fLayerOffset( MAX(5,nLayers+1) ), 
                                             fBOffset(20),  fBlockLen(20) {
    fCurrentOffset = -2; fCurrentGhost = 0; fMaxGhost = nLayers;
    fMode = mode;
		fFinalized = False;
		fLayerOffset.fill(0);
#ifdef USE_MPI
    fGhostCellLayout = new MPI_Datatype[nLayers+1];
    for(int i=0; i<=nLayers; i++) fGhostCellLayout[i] = MPI_DATATYPE_NULL;
#endif
  }

#ifdef USE_MPI
MPI_Datatype GhostIndexSet::GetGhostIndexMap( int ighost ) { 
	if( !fFinalized  ) Finalize();
	if(fCurrentOffset>=0) {
		if( ighost > fMaxGhost ) { sprintf(gMsgStr,"Undefined ghost layer: %d > %d layers",ighost,fMaxGhost); gPrintErr(); return MPI_DATATYPE_NULL; }
		if( fGhostCellLayout[ighost]==MPI_DATATYPE_NULL ) SetupMap(ighost);
		return fGhostCellLayout[ighost];
	} else return MPI_DATATYPE_NULL;
}

#endif

void GhostIndexSet::DumpData( int ighost, const float* data ) {
	int nblocks = Length(ighost);
	sprintf(gMsgStr,"\nGhostIndexSet::DumpData: ghost %d, nblocks: %d", ighost, nblocks ); gPrintLog();
	int* bLength = BlockLength();
	int* bOffset = BlockOffset();
	for(int i=0; i < nblocks; i++ ) {
		sprintf(gMsgStr,"  ** Block %d: length %d, offset %d",i,bLength[i],bOffset[i]); gPrintLog();
		for( int j = 0; j < bLength[i]; j++ ) {
			int offset = bOffset[i] + j;
			float value = data[ offset ];
			sprintf(gMsgStr,"\t%d: %f",offset,value);
			gPrintLog();
		}	
	} 
}

void GhostIndexSet::SetupMap( int ighost ) {
#ifdef USE_MPI
  MPI_Type_indexed( Length(ighost), BlockLength(), BlockOffset(), MPI_FLOAT, fGhostCellLayout+ighost);
  MPI_Type_commit(fGhostCellLayout+ighost);
#ifdef DEBUG
  if(gDebug) {
		int i;
    if( fMode == kL_Output ) {
      sprintf(gMsgStr,"\n(G%d)Gmap Output: Length (%d): ",ighost,fLayerOffset(ighost)); gPrintLog(); 
    } else { 
      sprintf(gMsgStr,"\n(G%d)Gmap Input: Length (%d): ",ighost,fLayerOffset(ighost)); gPrintLog();
    }
    for( i=0; i< fLayerOffset(ighost); i++ ) {
      sprintf(gMsgStr," ( Len:%d, Offset:%d ) ", fBlockLen(i), fBOffset(i) ); gPrintLog();
    }
    for( i=1; i<= ighost; i++ ) {
      sprintf(gMsgStr," ( G%d: LayerOffset: %d ) ", i, fLayerOffset(i) ); gPrintLog();
    }
  }
#endif
#endif
}  

void GhostIndexSet::Finalize() {
	if(fCurrentOffset>=0) {
		fBOffset.movemark(1);
		fBlockLen.movemark(1);
		fBOffset.setvalue(fCurrentOffset+1);
		fBlockLen.setvalue(0);
		while( fCurrentGhost <= fMaxGhost+1 )  { fLayerOffset.setvalue(fBlockLen.mark(),fCurrentGhost++); }
	}
  fFinalized = True;
#ifdef DEBUG
  if(gDebug>1) {
    char cMode[20];
    if(fMode == kL_Output) sprintf(cMode,"Send"); else sprintf(cMode,"Receive"); 
    sprintf(gMsgStr,"\n(P%d)GCM %s Finalize: (%d,%d):: BO(%d,%d,%d) BL(%d,%d,%d) LO(%d,%d,%d)",
	    gIProc,cMode,fCurrentOffset,fCurrentGhost,
	    fBOffset(0),fBOffset(1),fBOffset(2),
	    fBlockLen(0),fBlockLen(1),fBlockLen(2),
	    fLayerOffset(1),fLayerOffset(2),fLayerOffset(3) ); 
    gPrintLog(); 
  }
#endif

}

int GhostIndexSet::AddEntry(int offset, int ighost) {
  if( ighost <= 0 ) return -1;
  if( (offset == fCurrentOffset + 1) && (ighost == fCurrentGhost) ) {
    fBlockLen.incr_value(1);
    fCurrentOffset++;
  } else {
    if(fCurrentOffset>=0) {
      fBOffset.movemark(1);
      fBlockLen.movemark(1);
    }
    fCurrentOffset = fBOffset.setvalue(offset);
    fBlockLen.setvalue(1);
  }
  if(ighost > fCurrentGhost) {
    while( fCurrentGhost < ighost )  { fLayerOffset.setvalue(fBlockLen.mark(),fCurrentGhost++); }
  }
#ifdef DEBUG
  if(gDebug>1) {
    char cMode[20];
    if(fMode == kL_Output) sprintf(cMode,"Send"); else sprintf(cMode,"Receive"); 
    sprintf(gMsgStr,"\n(P%d)GCM %s Add Entry: (%d,%d): (%d,%d):: BO(%d,%d,%d) BL(%d,%d,%d) LO(%d,%d,%d)",
	    gIProc,cMode,offset,ighost,fCurrentOffset,fCurrentGhost,
	    fBOffset(0),fBOffset(1),fBOffset(2),
	    fBlockLen(0),fBlockLen(1),fBlockLen(2),
	    fLayerOffset(1),fLayerOffset(2),fLayerOffset(3) ); 
    gPrintLog(); 
  }
#endif
  return fCurrentOffset;
}


