#if __GNUG__ >= 2
#  pragma implementation
#endif

#include "Layer.h"
#include "Ghost.h"
#include "MultiGrid.h"

Point2 TLayer::_tmpP;
EInfoIndex TLayer::kType =  kO_Info0;

//---- newclass ---------------------------------------------------------

TLayer::TLayer( MultiGrid* g, int index, unsigned int rows, unsigned int cols ) : TOrderedObject(index) {
	_grid = g; 
	_dim[0] = rows;
	_dim[1] = cols;
	_maxKey = _dim[0]*_dim[1];
	_cells = new TLink[_maxKey];
	memset(_cells,0,_maxKey*sizeof(TLink));
	_firstCell = _lastCell = NULL;
	_coords[0].val[0] = 0.0; 
	_coords[0].val[1] = 0.0;
	_coords[1].val[0] = 1.0; 
	_coords[1].val[1] = 1.0;
	_extent.val[1] = 1.0;
	_extent.val[0] = 1.0;
	compute_cellsize();
	_partition = NULL;
}

TLayer::~TLayer() {
	delete[] _cells;
}


void TLayer::setCoords( CoordType ct, float f0, float f1 ) {
		_coords[(int)ct].val[kRows] = f0; 
		_coords[(int)ct].val[kCols] = f1;
		_extent.val[kRows] = _coords[kUpper].val[kRows] - _coords[kLower].val[kRows];
		_extent.val[kCols] = _coords[kUpper].val[kCols] - _coords[kLower].val[kCols];
		compute_cellsize();
		if( gDebug ) { 
		  sprintf(gMsgStr,"Set Coords [%d]: c(%.2f,%.2f), extent(%.2f,%.2f)",ct,f0,f1,_extent.val[kRows],_extent.val[kCols]); 
		  gPrintLog(); 
		}
}

TCell* TLayer::addCell( unsigned int ir, unsigned int ic ) {
	int32 key = getKey( ir, ic );
	if( key < 0 ) return NULL;
	if( _cells[key] == 0 ) {
		TCell* cell = new TCell(key,(byte)fIndex);
		cell->_cellsize = _cellsize;
		_cells[key] = cell;
		if( _firstCell == NULL ) { _firstCell = _lastCell = cell; }
		else { _lastCell->link(cell); _lastCell = cell; }
		return cell;
	}
	return NULL;
}

void TLayer::addActivationLayer(int activation_index) {
	TCell* c = firstCell();
	if( c!= NULL ) {
	  if( c->checkActivationLevel( activation_index ) ) {
		  while( c = c->_next ) {
			  c->createActivationLevel( activation_index );
		  }
	  }
	}
}

void TLayer::addChildLinkLayer(uint16 nlinks) {
	for( TCell* c = firstCell(); c; c = c->_next ) {
		c->getChildLinkLayer( nlinks );
	}
}

void TLayer::addNeighborLinkLayer(byte link_layer, uint16 nlinks, byte max_link_layer) {
	TCell* c = firstCell();
#ifdef DEBUG
	if( gDebug > 1 ) {
	  sprintf(gMsgStr,"TLayer%d::addNeighborLinkLayer: %d %d %d, first cell: ( %d, %d )", Index(), link_layer, nlinks, max_link_layer, c->row(), c->col() ); 
	  gPrintLog();
	}
#endif
	if( c!= NULL ) {
	  byte test = c->checkNeighborLinkLayers( link_layer );
	  if( test || ( c->neighborLinkLayer( link_layer ) == NULL ) ) {
		  for( ; c; c = c->_next ) {
			  c->getNeighborLinkLayer( link_layer, nlinks, max_link_layer );
		  }
	  }
	}
}

TPartition2* TLayer::partition( TPartition2* default_partition ) { 
	if( _partition == NULL ) {
		_partition = (default_partition == NULL ) ? new TPartition2 : default_partition; 
	}
	return _partition; 
}

void TLayer::linkNeighbors( byte link_layer, int activationLayer, TMap2* linkMap, ETopology type, FLBase* flBase, const char* methodName, int maxLayers ) {
  int rr, rc;
  SetObjInfo( kType, type );
  TCell* c = firstCell();
  if( c == NULL ) return;
  if(  link_layer >= c->GetObjInfo(TCell::kNLinkLayers) ) {
		switch( type ) {
			case kGrid:   addNeighborLinkLayer(link_layer, 8, MAX(link_layer+1,2) ); break;
			case kVolume: addNeighborLinkLayer(link_layer, 10, MAX(link_layer+1,2) ); break;
			case kTree:   addNeighborLinkLayer(link_layer, 1, MAX(link_layer+1,2) ); break;
		}
	}
	if( methodName && flBase ) {
		flBase->Run(methodName, linkMap, activationLayer, Index(), link_layer); 
	}
	else if ( methodName && strcmp(methodName,"default") ) { 
		sprintf(gMsgStr,"FrameLink Undefined for %s",methodName); gFatal(); 
	} else {
		int cnt=0;
		for( TCell* c = firstCell(); c; c = c->_next ) {
			Point2& point = getCellLocPoint( c );
			unsigned int ir = point(0);
			unsigned int ic = point(1);
			switch( type ) {
			case kGrid: {
				for( Grid_Direction il = firstGD(); moreGD(il); incrGD(il) ) {
					GMR(rr,rc,il);   
					TCell* cR = getCell(ir+rr, ic+rc);
					if( cR ) { c->addNeighbor( link_layer, cR, il-NE ); cnt++; } 
				}
			} break;
			case kVolume: {
				 TCell* cR;
				for( Grid_Direction il = firstGD(); moreGD(il); incrGD(il) ) {
					GMR(rr,rc,il);   
					cR = getCell(ir+rr, ic+rc);
					if( cR ) { c->addNeighbor( link_layer, cR, il-NE ); cnt++;  }
				}
				int layer_index = Index()-1;
				if( layer_index >= 0 ) {
					TLayer* l = _grid->getCellLayer(layer_index);
					if( l != NULL ) {  
						cR = l->getCell(ir, ic);
						if( cR ) { c->addNeighbor( link_layer, cR, UU-NE ); cnt++;  }
					}
				}				
				layer_index = Index()+1;  
				if( layer_index < maxLayers ) { 
					int layer_index = Index()-1;
					TLayer* l = _grid->getCellLayer(layer_index);
					if( l != NULL ) {  
						cR = l->getCell(ir, ic);
						if( cR ) { c->addNeighbor( link_layer, cR, DD-NE ); cnt++;  }
					}
				}
			} break;
			case kTree: {
				byte mVal = linkMap->BValue(ir,ic,0);
				if( !GMR(rr,rc,(Grid_Direction)mVal) ) { 
					gFatal("Illegal Index in Link Points");  
				}
				TCell* cR = getCell(ir+rr,ic+rc);
				if( cR ) { c->addNeighbor( link_layer, cR ); cnt++;  }
			} break;
			case kNetwork:
			break;
			case kCollection:
				break;
			}
		}
		sprintf(gMsgStr,"Linking cells for cell layer %d, activation layer %d, link layer %d, %d links configured",
															Index(), activationLayer, link_layer, cnt ); 
		gPrintScreen(); 
	}
}

Point2& TLayer::getCellPartitionPoint( TCell*  c ) { 
	_tmpP.elem(0) = c->Index()/_dim[1]; 
	_tmpP.elem(1) = c->Index() % _dim[1]; 
	if( _partition ) { 
			Region2* r = _partition->region(gIProc);
		_tmpP.elem(0) -= r->lower(0); 
		_tmpP.elem(1) -= r->lower(1); 
	}
	return _tmpP; 
}


int TLayer::getCellPartitionIndex( TCell*  c ) {
	if( _partition == NULL ) { return -1; }
	Region2* r = _partition->region(gIProc);
	int ir  = c->Index()/_dim[1] - r->lower(0); 
	int ic  = c->Index() % _dim[1] - r->lower(1);
	return ic + ir*r->extents(1); 
}

void TLayer::addCells( int boundarySize ) {
	if( _partition == NULL ) { gFatal( "Must create a partition before adding cells to grid layer." ); }
  ByteGrid* cellDistribution = _partition->cellDistribution();
  if( cellDistribution == NULL ) { gFatal( "Failed to set Cell Distribution in Partition." ); }
  _dim[0] = cellDistribution->extents(0);
  _dim[1] = cellDistribution->extents(1);
  int isVol = ( GetObjInfo( kType ) ==  kVolume );
	_maxKey = _dim[0]*_dim[1];
	if( boundarySize < 0 ) boundarySize = _partition->maxGhost();
  int nprocs = _partition->nProcs();
	compute_cellsize();
  Region2 globalRegion(0,0,_dim[0]-1,_dim[1]-1);
	Region2* r = _partition->region(gIProc);
	if( (r == NULL) || (cellDistribution == NULL) ) { 
		gPrintErr( " Undefined Partition in TLayer::addCells " ); 
		return; 
	}
	TCell* c = NULL;
	long cd_value, thresh;
	for(int ir = r->lower(0); ir<= r->upper(0); ir++ ) {
		for(int ic = r->lower(1); ic<= r->upper(1); ic++ ) {
			cd_value = cellDistribution->IntValue( ir, ic );
			thresh = ( isVol ) ? Index() : 0;
			if( cd_value > thresh ) {
				c = addCell(ir,ic);
				if( c != NULL ) { c->SetObjIndex( TCell::kOwner, gIProc ); }
			} 
		}	
	} 
	if( boundarySize>0 ) { gPrintLog("Create Ghost Layers."); }
	for( int ig = 1; ig <= boundarySize; ig++) {   // Iterate over Ghost Layers ( ghost(0) = inside region )
		int ir, ic;
		GhostCounter gc(ig,r,ir,ic);
		while( gc.Count(ir,ic) ) {    // Iterate Counter-Clockwise over ghost layer ig
			if( globalRegion.inside(ir,ic) ) {
				cd_value = cellDistribution->IntValue( ir, ic );
				thresh = ( isVol ) ? Index() : 0;
				if( cd_value > thresh ) { 
					c = addCell(ir,ic);
					if( c != NULL ) {
						byte ghost_index = (ig<255) ? (byte)ig : 255;
						c->SetObjInfo( TCell::kGhostIndex, ghost_index );	
						int owner = _partition->getowner( this, c );
						if( owner < 0 ) { 
						  sprintf(gMsgStr,"Can't find owner for cell at (%d,%d) on proc %d ( %d > %d ): global extents (%d,%d) ",ir,ic,gIProc,cd_value,thresh,_dim[0],_dim[1]); gFatal(); 
						}
						c->SetObjIndex( TCell::kOwner, owner );
					}
				}
			}
		}
	}
}




