#include "MultiCoverage.h"

enum ECoverageTags { kCondenseTag, kRemoteValueTag };
const byte kBackGround = 255;

MemoryManager* MultiCoverage::fMemMgr = NULL;
int MultiCoverage::fCoverageCount = 0;

#ifdef USE_MPI
MPI_Comm MultiCoverage::fComm = MPI_COMM_WORLD;
#endif

int MultiCoverage::kLinkEdgesTagBase = 200;

/*
*************************************************************************
*									*
* MultiCoverage     					        *
*									*
*************************************************************************
*/

void MultiCoverage::initialize_fields() {
	fGhostSize=1; 
	fMultiGrid = NULL; 
	memset(fInfo,0,8); 
	fActivationLayer = NULL; 
	fCoverageDataType = kExtensive;
#ifdef USE_MPI
	fDumpTransferData=0;
  fReq = NULL;
	fNReq = 0;
  fStatus = NULL;
#endif
}

MultiCoverage::MultiCoverage() : floatVec(), MemObject() { 
	initialize_fields();
}


MultiCoverage::MultiCoverage( MultiGrid* mgrid, int activationLevel, int ghostSize, int memoryIndex ) : floatVec(), MemObject() { 
	initialize_fields();
  Setup(mgrid,activationLevel,ghostSize,memoryIndex); 
}

int MultiCoverage::Setup( MultiGrid* mgrid, int activationIndex, int ghostSize, int memoryIndex ) {
  if( Allocated() ) return 0;
  fMemoryIndex = memoryIndex;
  if(fMemMgr == NULL) fMemMgr = new MemoryManager();
  fMultiGrid = mgrid;
  fActivationLayer	= fMultiGrid->activationLayer(activationIndex);
	fActivationIndex = activationIndex;
  fGhostSize = (ghostSize>=0) ? ghostSize : fActivationLayer->maxGhost();
  fNCells = fActivationLayer->nCells();
	fTag = kLinkEdgesTagBase + fCoverageCount++;
  ShallowCopy( fMemMgr->GetVector( fNCells+1, *this ) );
  elem(fNCells) = 0.0;   // create buffer to make purify happy in spatial loops.
  return 1;
}

int MultiCoverage::Setup( MultiCoverage& cov, int gSize, int memoryIndex ) {
  if( Allocated() ) return 0;
  fMemoryIndex = memoryIndex;
  fGhostSize = (gSize<0) ? cov.fGhostSize : gSize;
  fMultiGrid = cov.fMultiGrid;
  fActivationIndex = cov.fActivationIndex;
  fActivationLayer	= fMultiGrid->activationLayer(fActivationIndex);
  fNCells = fActivationLayer->nCells();
	fTag = kLinkEdgesTagBase + fCoverageCount++;	
  ShallowCopy( fMemMgr->GetVector( fNCells+1, *this, &cov ) );
  elem(fNCells) = 0.0;   // create buffer to make purify happy in spatial loops.
  return 1;
}

int MultiCoverage::Setup( DistributedGrid* pset, int gSize, int memoryIndex ) {
  if( Allocated() ) return 0;
  fMemoryIndex = memoryIndex;
  fGhostSize = gSize;
  fGridPlex.elem(0) = pset;
  fTag  = kLinkEdgesTagBase + fCoverageCount++;	
  fMultiGrid = pset->getMultiGrid();
  fActivationIndex = pset->activationLayerIndex();
  fActivationLayer	= fMultiGrid->activationLayer(fActivationIndex);
  fNCells = fActivationLayer->nCells();
  ShallowCopy( fMemMgr->GetVector( fNCells+1, *this ) );
  elem(fNCells) = 0.0;   // create buffer to make purify happy in spatial loops.
  return 1;
}

DistributedGrid* MultiCoverage::Grid( int gridLayer, int linkLayer ) { 
  if(fMultiGrid == NULL) gFatal( "Attempt to Access Unallocated Grid.");
  if( gridLayer < 0 ) { gridLayer = fMultiGrid->defaultLayerIndex(); } 
  DistributedGrid* g = NULL;
  int ngrids = fGridPlex.length();
  for( int i=0; i < ngrids; i++ ) {
	  DistributedGrid* g = (DistributedGrid*)fGridPlex[i];
	  if( (g->getGridLayer() == gridLayer) && (g->getLinkLayer() == linkLayer)) { return g; }
  }
  if( g == NULL ) {
		  g = new DistributedGrid( fMultiGrid, NULL, fActivationIndex, gridLayer, linkLayer ); 
		  fGridPlex.elem(ngrids) = g;
  }
  return g;
}
	
int MultiCoverage::StartLinkEdges() {
  int rv = 0;
#ifdef USE_MPI
  if( fReq == NULL ) {
		ActivationLayer* al = activationLayer();
		fReq = al->setupCommunication( fNReq, gNProc, fGhostSize, data(), fComm, fTag );
		fStatus = new MPI_Status[fNReq];
		lprint2( "\nLinkEdges(%f): Creating %d comm requests.", gTime(), fNReq ); 
	}
	if( fDumpTransferData ) { dumpTransferData(kL_Output); }
  rv = MPI_Startall(fNReq,fReq);
#endif
  return rv;
}

int MultiCoverage::LinkEdges( const char* name, int dumpTransfer ) {
	int rv = 0;
#ifdef USE_MPI
	if( gNProc == 1 ) return rv;
	fDumpTransferData = dumpTransfer;
	Env::StartCommTimer();
  StartLinkEdges();
  rv = CompleteLinkEdges( name );
	Env::StopCommTimer();
#endif
	return rv;
}

int MultiCoverage::CompleteLinkEdges( const char* name ) {
  int rv = 0;
#ifdef USE_MPI
  if( fReq  ) {
		if(gDebug>2) {
			sprintf(gMsgStr,"Link Edges %s: Waiting for completion of %d MPI requests.", name, fNReq );
			gPrintScreen(gMsgStr,TRUE);
		}
    rv = MPI_Waitall(fNReq,fReq,fStatus);      
    if(gDebug>2) {
      int count;
			sprintf(gMsgStr,"Link Edges %s Done", name );
			gPrintScreen(gMsgStr,TRUE); gPrintLog();
      for(int i=0; i<fNReq; i++) {
				MPI_Get_count( fStatus+i, MPI_FLOAT, &count );
				sprintf(gMsgStr,"\nLinkMsg Completed: source(%d), tag(%d), nElem(%d): ",
					fStatus[i].MPI_SOURCE, fStatus[i].MPI_TAG, count );
				gPrintLog();
      }
    }
  }
	if( fDumpTransferData ) { dumpTransferData(kL_Input); }
#endif
  return rv;  
}

MultiCoverage&  MultiCoverage::operator = (MultiCoverage& cov) {
  if (&cov != this) {
		if( fMultiGrid != cov.fMultiGrid ) { gPrintErr("Multiple MultiGrids not allowed!"); return *this; }
		ActivationLayer* al = activationLayer();
		if( fActivationIndex == cov.fActivationIndex ) {
			deep_copy( cov.data(), al->nActiveCells() );
		} else {
			TCell* c; float value;
#pragma parallel 
			{ 
#pragma pfor 
			for( int32 i = 0; i < al->nActiveCells(); i++ ) {
				c = al->cell(i);
				value = al->mapValue( c, cov.fActivationIndex, (float*)data(), (fCoverageDataType == kIntensive), 0.0 );
				SetValue( i, value ); 
			}
			}
		}
  }
  return *this;
}

MultiCoverage&  MultiCoverage::operator += (MultiCoverage& cov) {
  if (&cov != this) {
		if( fMultiGrid != cov.fMultiGrid ) { gPrintErr("Multiple MultiGrids not allowed!"); return *this; }
		ActivationLayer* al = activationLayer();
		if( fActivationIndex == cov.fActivationIndex ) {
			float* data0 = fValue;
			const float* data1 = cov.data();
			float* dataEnd = data0 + al->nActiveCells(); 
#pragma parallel 
			{ 
#pragma pfor
			while( data0 < dataEnd ) {
				*data0++ += *data1++;
			}
			}
		} else {
			TCell* c; float value;
#pragma parallel 
			{ 
#pragma pfor 
			for( int32 i = 0; i < al->nActiveCells(); i++ ) {
				c = al->cell(i);
				value = al->mapValue( c, cov.fActivationIndex, (float*)data(), (fCoverageDataType == kIntensive), 0.0 );
				AddValue( i, value ); 
			}
			}
		}
  }
  return *this;
}

float MultiCoverage::Reduce ( EReduceOp op, int rootProc, CellPlex* cp, int ignoreINF ) {
  float ftmp = 0.0; 
  register int fp=1, fastmap=0; register float value; register int32 mi;
  if( cp == NULL ) { cp = activationLayer(); fastmap=1; }
	register int ncells = cp->nActiveCells();
	for( int32 i = 0; i < ncells; i++ ) {
		mi = ( fastmap ) ? i : cp->cell(i)->memoryIndex( fActivationIndex );
    value = Value(mi);
    if( (ignoreINF==0) || ( ( value < FLT_MAX ) && ( value > -FLT_MAX ) )  ) {
			switch(op) {
			case kR_Max:
				if(fp) { ftmp = value; fp = 0; }
				else { ftmp = ( ftmp < value ) ? value : ftmp; }
				break;
			case kR_Min:
				if(fp) { ftmp = value; fp = 0; }
				else { ftmp = ( ftmp > value ) ? value : ftmp; }
				break;
			case kR_Sum:
				ftmp+= value;
				break;
			case kR_Ave:
				ftmp+= value;
			}
		}
  }
#ifdef USE_MPI
  float rv; int do_cast = 0;
  if( rootProc < 0 ) { rootProc = 0; do_cast = 1; }
  switch(op) {
  case kR_Max:
    MPI_Reduce(&ftmp,&rv,1,MPI_FLOAT,MPI_MAX,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Min:
    MPI_Reduce(&ftmp,&rv,1,MPI_FLOAT,MPI_MIN,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Sum:
    MPI_Reduce(&ftmp,&rv,1,MPI_FLOAT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Ave:
    int ctmp, count = ncells;
    MPI_Reduce(&ftmp,&rv,1,MPI_FLOAT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    MPI_Reduce(&count,&ctmp,1,MPI_INT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    if(gIProc == rootProc) { rv /= ctmp; }
  }
  if( do_cast ) {
		MPI_Bcast( &rv, 1, MPI_FLOAT, 0, MPI_COMM_WORLD );
  }
  return rv;
#else
	if( op == kR_Ave ) { ftmp /= ncells; }
  return ftmp;
#endif
}

float MultiCoverage::Reduce ( EReduceOp op, int rootProc, MultiCoverage& mask, int ignoreINF ) {
  register int fp = 1; 
  register float value, ftmp = 0.0; 
  if( activationIndex() != mask.activationIndex() ) { gPrintErr("Activation Layer mismatch in Reduce mask."); return 0.0; }
	register int ncells = activationLayer()->nActiveCells();
	for( int32 i = 0; i < ncells; i++ ) {
    value = Value(i) * mask.Value(i);
	  if( (ignoreINF==0) || ( ( value < FLT_MAX ) && ( value > -FLT_MAX ) )  ) {
	  switch(op) {
	  case kR_Max:
		if(fp) { ftmp = value; fp = 0; }
		else { ftmp = ( ftmp < value ) ? value : ftmp; }
		break;
	  case kR_Min:
		if(fp) { ftmp = value; fp = 0; }
		else { ftmp = ( ftmp > value ) ? value : ftmp; }
		break;
	  case kR_Sum:
		ftmp+= value;
		break;
	  case kR_Ave:
		ftmp+= value;
	  }
	}
  }
#ifdef USE_MPI
  float rv; int do_cast = 0;
  if( rootProc < 0 ) { rootProc = 0; do_cast = 1; }
  switch(op) {
  case kR_Max:
    MPI_Reduce(&ftmp,&rv,1,MPI_FLOAT,MPI_MAX,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Min:
    MPI_Reduce(&ftmp,&rv,1,MPI_FLOAT,MPI_MIN,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Sum:
    MPI_Reduce(&ftmp,&rv,1,MPI_FLOAT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Ave:
    int ctmp, count = ncells;
    MPI_Reduce(&ftmp,&rv,1,MPI_FLOAT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    MPI_Reduce(&count,&ctmp,1,MPI_INT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    if(gIProc == rootProc) { rv /= ctmp; }
  }
  if( do_cast ) {
		MPI_Bcast( &rv, 1, MPI_FLOAT, 0, MPI_COMM_WORLD );
  }
  return rv;
#else
	if( op == kR_Ave ) { ftmp /= ncells; }
  return ftmp;
#endif
}

inline int round( float fval ) { 
  register int rv = (int)fval;
  return ( (fval-rv) > 0.5 ) ? ++rv : rv;
}

void MultiCoverage::SpreadValues( float* values, MultiCoverage& mask ) {
  if( activationIndex() != mask.activationIndex() ) { gPrintErr("Activation Layer mismatch in Reduce mask."); return; }
  register int ncells = activationLayer()->nActiveCells();
  for( int32 i = 0; i < ncells; i++ ) {
	int index =  round( mask.Value(i) );
	float fval = values[index];
	SetValue( i, fval);
  }
}
float* MultiCoverage::MultiReduce ( EReduceOp op, int rootProc, MultiCoverage& mask, int ignoreINF ) {
  int max_cat = 256, debug = 1, *fp = new int[max_cat];
  float reduce_value; 
  float *reduce_in = new float[max_cat], *reduce_out = new float[max_cat];
  int *cell_count_in = new int[max_cat], *cell_count_out = new int[max_cat];
  for( int i=0; i<max_cat; i++) { cell_count_in[i] = cell_count_out[i] = 0; fp[i] = 1; reduce_in[i] = reduce_out[i] = 0.0; } 
  if( activationIndex() != mask.activationIndex() ) { gPrintErr("Activation Layer mismatch in Reduce mask."); return NULL; }
  register int ncells = activationLayer()->nActiveCells();
  for( int32 i = 0; i < ncells; i++ ) {
	float data_value = Value(i);
	int index =  round( mask.Value(i) );
	if( index > max_cat ) { gFatal( format(" Maximum category (%d) exceeded: %d", max_cat, index ) ); }
	if( (ignoreINF==0) || ( ( data_value < FLT_MAX ) && ( data_value > -FLT_MAX ) )  ) {
	  cell_count_in[index]++;
	  if(fp[index]) { 
		fp[index] = 0; 
		reduce_value = data_value;
	  } else {
		reduce_value = reduce_in[ index ];
		switch(op) {
		case kR_Max:
		  reduce_value = ( data_value > reduce_value ) ? data_value : reduce_value; 
		  break;
		case kR_Min:
		  reduce_value = ( data_value < reduce_value ) ? data_value : reduce_value; 
		  break;
		case kR_Sum:
		  reduce_value += data_value; 
		  break;
		case kR_Ave:
		  reduce_value += data_value;
		}
	  }
	  reduce_in[index] = reduce_value;
	}
  }
#ifdef USE_MPI
  if( debug ) { 
	printf("\n (P%d): Doing MultiReduce: ", gIProc );
	for( int i=0; i<10; i++ ) { 
	  printf(" %7.1f ", reduce_in[i] ); 
	}
  }
  int do_cast = 0;
  if( rootProc < 0 ) { rootProc = 0; do_cast = 1; }
  MPI_Reduce(cell_count_in,cell_count_out,max_cat,MPI_INT,MPI_SUM,rootProc,MPI_COMM_WORLD);
  switch(op) {
  case kR_Max:
    MPI_Reduce(reduce_in,reduce_out,max_cat,MPI_FLOAT,MPI_MAX,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Min:
    MPI_Reduce(reduce_in,reduce_out,max_cat,MPI_FLOAT,MPI_MIN,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Sum:
    MPI_Reduce(reduce_in,reduce_out,max_cat,MPI_FLOAT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Ave:
    MPI_Reduce(reduce_in,reduce_out,max_cat,MPI_FLOAT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    break;
  }
  if(do_cast) {
        MPI_Bcast( reduce_out, max_cat, MPI_FLOAT, 0, MPI_COMM_WORLD );
        MPI_Bcast( cell_count_out, max_cat, MPI_FLOAT, 0, MPI_COMM_WORLD );
  }
#endif
  for( int i=0; i<max_cat; i++) { 
	if( cell_count_out[i] == 0 ) {
	  reduce_out[i] = FLT_MAX;
	} else if(op == kR_Ave) {
	  reduce_out[i] = reduce_out[i]/cell_count_out[i];
	}
  }

  if( debug ) { 
	printf("\n (P%d): Did MultiReduce: ", gIProc );
	for( int i=0; i<10; i++ ) { 
	  float rval = reduce_out[i];
	  if( rval == FLT_MAX ) { printf("    ND   " ); }
	  else { printf(" %7.1f ", rval ); }
	}
  }
  delete[] cell_count_out;
  delete[] cell_count_in;
  delete[] reduce_in;   
  delete[] fp;
  return reduce_out;
}

/*
float* MultiCoverage::MultiReduce ( EReduceOp op, int rootProc, MultiCoverage& mask, int ignoreINF ) {
  int max_cat = 256, fp = 1;
  float reduce_value, *rval = new float[max_cat];
  int* cell_count_in = new int[max_cat], cell_count_out = new int[max_cat];
  for( int i=0; i<max_cat; i++) { cell_count[i] = 0; } 
  if( activationIndex() != mask.activationIndex() ) { gPrintErr("Activation Layer mismatch in Reduce mask."); return NULL; }
  register int ncells = activationLayer()->nActiveCells();
  for( int32 i = 0; i < ncells; i++ ) {
	float data_value = Value(i);
	int index =  round( mask.Value(i) );
	if( index > max_cat ) { gFatal( format(" Maximum category (%d) exceeded: %d", max_cat, index ) ); }
	if( (ignoreINF==0) || ( ( data_value < FLT_MAX ) && ( data_value > -FLT_MAX ) )  ) {
	  cell_count[index]++;
	  if(fp) { reduce_value = data_value; fp = 0; }
	  else {
		reduce_value = rval[ index ];
		switch(op) {
		case kR_Max:
		  reduce_value = ( data_value > reduce_value ) ? data_value : reduce_value; 
		  break;
		case kR_Min:
		  reduce_value = ( data_value < reduce_value ) ? data_value : reduce_value; 
		  break;
		case kR_Sum:
		  reduce_value += data_value; 
		  break;
		case kR_Ave:
		  reduce_value += data_value;
		}
	  }
	  rval[index] = reduce_value;
	}
  }
#ifdef USE_MPI
  if( rootProc < 0 ) { rootProc = 0; }
  MPI_Reduce(cell_count_in,cell_count_out,256,MPI_INT,MPI_SUM,rootProc,MPI_COMM_WORLD);
  switch(op) {
  case kR_Max:
    MPI_Reduce(rval_in,rval_out,256,MPI_FLOAT,MPI_MAX,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Min:
    MPI_Reduce(rval_in,rval_out,256,MPI_FLOAT,MPI_MIN,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Sum:
    MPI_Reduce(rval_in,rval_out,256,MPI_FLOAT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    break;
  case kR_Ave:
    MPI_Reduce(rval_in,rval_out,256,MPI_FLOAT,MPI_SUM,rootProc,MPI_COMM_WORLD);
    break;
  }
  MPI_Bcast( rval, 256, MPI_FLOAT, 0, MPI_COMM_WORLD );
  
#endif
  for( int i=0; i<max_cat; i++) { 
	if( cell_count[i] == 0 ) {
	  rval[i] = FLT_MAX;
	} else if(op == kR_Ave) {
	  rval[i] = rval[i]/cell_count[i];
	}
  }
  delete[] cell_count;
  return rval;
}
*/
int MultiCoverage::LocalGridToMap( TMap& map, unsigned long background, EScaleType st, int layer_index, int targetProc ) {
  int bsize = map.NBytesClamped();
  int local_debug = 1;
  int isFloatMap = ( map.GetObjInfo(TMap::kDataType) == TMap::kFloats  );
  int isIntMap = ( map.GetObjInfo(TMap::kDataType) == TMap::kInts  );
  float nullValue = FLT_MAX;
	ActivationLayer* al = activationLayer();
	TLayer* l = fMultiGrid->getCellLayer( layer_index );
	TPartition2* p = l->partition();
	int32 ncells = al->nActiveCells();
	Region2 r;
	if( p && ( gIProc != targetProc ) ) { r = *(p->region( gIProc )); }
	else { l->globalRegion(r); }
	if( r != ((Region2&)map) ) { map.ReAlloc( r ); }
	int y = r.extents(0) - 1;
	float value, maxmin[2]; unsigned long ival;
	if( isFloatMap ) {
	  map.Set( FLT_MAX ); 
	} else if( isIntMap ) {
	  map.Set( (float) 0 ); 
	} else {
	  if(  (st == kAutoScaling) || (st == kRainbowScaling) ) { 
		  float max = Reduce ( kR_Max, -1, NULL ); 
		  float min = Reduce ( kR_Min, -1, NULL ); 
		  if(st == kRainbowScaling) map.setDataCeiling( 255 ); 
		  map.SetBounds( min, max );
	  }
	  map.Set( (byte) 255 );
	} 
	for( TCell* c = l->firstCell(); c;  l->nextCell(c) ) {
		if(  c->GetObjInfo( TCell::kGhostIndex ) == 0 ) {
			value = al->mapValue( c, fActivationIndex, (float*)data(), (fCoverageDataType == kIntensive), nullValue );
			Point2& p = l->getCellLocPoint( c );
			if( isFloatMap || isIntMap ) {
			   map.SetValue( p(0), p(1), value );
			} else {
			  if( value == FLT_MAX ) { ival = background; }
			  else {
				  if(st == kDirectScaling ) { ival = map.DirectScale(value); }
				  else { ival = map.Scale( value ); }
			  }
			  if( st == kRainbowScaling ) { map.SetRainbowValue( p(0), p(1), ival ); }
			  else { map.SetValue( p(0), p(1), ival ); }
			}
			if( local_debug > 1 ) { 
				int32 cellindex = c->Index();
				int32 memindex = c->memoryIndex( al->Index() );
				sprintf(gMsgStr,"cell (%d:%d), setting value (%d,%d) -> %d (%f)",cellindex,memindex,p(0),p(1),ival,value); gPrintLog(); 
			}
		}
	}
	if( local_debug ) {
		gPrintLog("MultiCoverage::LocalGridToMap");
		map.Dump(Env::LogFile(),background);
	}
	return bsize;	
}


void MultiCoverage::CondenseToMap( TMap& map, unsigned long background, EScaleType st, int layer_index, int targetProc ) {
	LocalGridToMap( map, background, st, layer_index, targetProc );
	if( gNProc <= 1 ) return;
	int local_debug = 0;
#ifdef USE_MPI
	int cTag = map.getTag();
	int isFloatMap = ( map.GetObjInfo(TMap::kDataType) == TMap::kFloats  );
	int isIntMap = ( map.GetObjInfo(TMap::kDataType) == TMap::kInts  );
	void* dTemp = NULL;
	int dTempSize = 0;

  CString mname; 
  if( local_debug ) {
	mname = map.VariableName(); (mname += ":") += map.DSetName(); 
	sprintf(gMsgStr,"Runnning CondenseToMap: map=%s, tag=%d",mname.chars(),cTag); 
	gPrintLog(gMsgStr,True); 
  }
  if( gIProc == targetProc ) {
	TLayer* l = fMultiGrid->getCellLayer( layer_index );
	TPartition2* p = l->partition();
	Region2 r;
    MPI_Status status; int flag, source, count, msgCnt=1;
    while(1) {
      MPI_Iprobe(MPI_ANY_SOURCE,cTag,MPI_COMM_WORLD,&flag,&status);
      if( flag == True ) {
		  source = status.MPI_SOURCE;
		  if( isFloatMap ) {
			 MPI_Get_count(&status,MPI_FLOAT,&count);
			 if( count > dTempSize ) {
				if( dTemp ) { delete[] dTemp; dTempSize = 0; }
				dTemp = new float[ dTempSize = count ];
				if( local_debug ) { 
					sprintf(gMsgStr,"CondenseToMap(%s): proc %d, tag %d: Allocating float data: %d float elements",mname.chars(),gIProc,cTag,dTempSize); 
					gPrintLog(gMsgStr,True); 
				}
			 }
		  } else if( isIntMap ) {
			 MPI_Get_count(&status,MPI_INT,&count);
			 if( count > dTempSize ) {
				if( dTemp ) { delete[] dTemp; dTempSize = 0; }
				dTemp = new int[ dTempSize = count ];
				if( local_debug ) { 
					sprintf(gMsgStr,"CondenseToMap(%s): proc %d, tag %d: Allocating int data: %d int elements",mname.chars(),gIProc,cTag,dTempSize); 
					gPrintLog(gMsgStr,True); 
				}
			 }
		  } else {
			 MPI_Get_count(&status,MPI_BYTE,&count);
			 if( count > dTempSize ) {
				if( dTemp ) { delete[] dTemp; dTempSize = 0; }
				dTemp = new byte[ dTempSize = count ];
				if( local_debug ) { 
					sprintf(gMsgStr,"CondenseToMap(%s): proc %d, tag %d: Allocating byte data: %d byte elements",mname.chars(),gIProc,cTag,dTempSize); 
					gPrintLog(gMsgStr,True); 
				}
			 }
		  }
		  if( p ) { r = *(p->region( source )); }
		  else { l->globalRegion(r); }
		  if( isFloatMap ) {
			MPI_Recv(dTemp,count,MPI_FLOAT,source,cTag,MPI_COMM_WORLD,&status);
			if( local_debug ) { 
				sprintf(gMsgStr,"CondenseToMap(%s): proc %d, tag %d: Receiving float data from proc %d: %d float elements",mname.chars(),gIProc,cTag,source, count); 
				gPrintLog(gMsgStr,True); 
			}
		  } else if( isIntMap ) {
			MPI_Recv(dTemp,count,MPI_INT,source,cTag,MPI_COMM_WORLD,&status);
			if( local_debug ) { 
				sprintf(gMsgStr,"CondenseToMap(%s): proc %d, tag %d: Receiving int data from proc %d: %d int elements",mname.chars(),gIProc,cTag,source,count); 
				gPrintLog(gMsgStr,True); 
			}
		  } else {
			MPI_Recv(dTemp,count,MPI_BYTE,source,cTag,MPI_COMM_WORLD,&status);
			if( local_debug ) { 
				sprintf(gMsgStr,"CondenseToMap(%s): proc %d, tag %d: Receiving byte data from proc %d: %d bytes",mname.chars(),gIProc,cTag,source,count); 
				gPrintLog(gMsgStr,True); 
			}
		  }
		  MergeData(map,dTemp,r,count); 
		  if(++msgCnt == gNProc) break;
	  }
    }
  } else {
	  if( isFloatMap ) {
		MPI_Send(map.FloatData(),map.nelem(),MPI_FLOAT,targetProc,cTag,MPI_COMM_WORLD);
		if( local_debug ) { 
			sprintf(gMsgStr,"CondenseToMap: Sending data from proc %d to proc %d: %d float elements",gIProc,targetProc,map.nelem()); 
			gPrintLog(gMsgStr,True); 
		}
	  } else if( isIntMap ) {
		MPI_Send(map.IntData(),map.nelem(),MPI_INT,targetProc,cTag,MPI_COMM_WORLD);
		if( local_debug ) { 
			sprintf(gMsgStr,"CondenseToMap: Sending data from proc %d to proc %d: %d int elements",gIProc,targetProc,map.nelem()); 
			gPrintLog(gMsgStr,True); 
		}
	  } else {
		MPI_Send(map.Data(),map.BSize(),MPI_BYTE,targetProc,cTag,MPI_COMM_WORLD);
		if( local_debug ) { 
			sprintf(gMsgStr,"CondenseToMap: Sending data from proc %d to proc %d: %d bytes",gIProc,targetProc,map.BSize()); 
			gPrintLog(gMsgStr,True); 
		}
	  }
  }
  if( dTemp != NULL ) {
	delete[] dTemp; dTempSize = 0;
  }
#endif  
}

int  MultiCoverage::CopyToMap( TMap& map, Bool swap, EScaleType st, int layer_index, int targetProc )
{
	if( fMultiGrid == NULL ) return 0;
	unsigned long background = map.defaultBackground();
	CondenseToMap( map, background, st, layer_index, targetProc );
	if( swap ) {  map.Reflect(0); }
	return 1;
}


void MultiCoverage::MergeData( TMap& map, void* src, Region2& r, int nbytes ) {
  int local_debug = 1;
  if( local_debug ) {
	  sprintf(gMsgStr,"MergeData: l(%d %d) u(%d %d) ",r.lower(0),r.lower(1),r.upper(0),r.upper(1)); 
	  gPrintLog();
  }
  if( map.GetObjInfo(TMap::kDataType) == TMap::kFloats ) {
	float *fdata = map.FloatData(), *sdata = (float*)src;
 	for(int ir=r.lower(0); ir<=r.upper(0); ir++ ) {
	  for(int ic=r.lower(1); ic<=r.upper(1); ic++) { 
		  *(fdata + map.bindex(ir,ic)) = *sdata++; 
	  }
	}
  } else {
	int bsize = map.NBytesClamped();
	byte* bsrc = (byte*)src;
	int mybytes = r.nelem()*bsize;
	unsigned long dval = 0; byte* mPtr;
	if( nbytes != mybytes ) { sprintf(gMsgStr,"Data Size Conflict in MultiCoverage::MergeData: %d vs %d",nbytes,mybytes); gPrintErr(); return; }
	for(int ir=r.lower(0); ir<=r.upper(0); ir++ ) {
	  for(int ic=r.lower(1); ic<=r.upper(1); ic++) {  
		  mPtr = map.Data() + map.bindex(ir,ic)*bsize;
		  memcpy(mPtr,bsrc,bsize);
		  bsrc += bsize;
	  }
	}
  }
}

MultiCoverage&  MultiCoverage::CopyFromMap( TMap& map, int layer_index, int srcProc, const char* name ) {
	if( (srcProc >= 0) && (gNProc > 1) ) {
	  map.BroadcastData(srcProc,"");
	}
	ActivationLayer* al = activationLayer();
	Region2 r;
	TLayer* l = fMultiGrid->getCellLayer( layer_index );
	l->globalRegion(r);
	Region2& r1 = ((Region2&)map);
	if( r != r1 ) { 
		sprintf(gMsgStr,"Map size discrepancy in  MultiCoverage::CopyFromMap(%s:%s): layer(%d,%d) vs map(%d,%d)" , 
		((name==NULL) ? "null" : name ), map.MapName().chars(), r.extents(0), r.extents(1), r1.extents(0), r1.extents(1) ); gPrintErr();
	  return *this; 
	}
	for( TCell* c = l->firstCell(); c;  l->nextCell(c) ) {
		if( c->GetObjInfo( TCell::kGhostIndex ) == 0 ) {
			Point2& p = l->getCellLocPoint( c );
			SetValue( c, map.Value(p) ); 
		}
	}
	return *this;
}

void MultiCoverage::SetDataValues( float* data, int layOut, int size ) {
	switch(layOut) {
		case PSCserial: {  
			int my_size = ( size > fSize ) ?  fSize : size;
			float *valp = fValue, *val_end = fValue+my_size;
			while( valp <  val_end ) {
				*valp++ = *data++;
			}
		} break;
		case PSCfullGrid:  break;
		case PSCsubGrid:  break;
	}

}

float& MultiCoverage::Value( unsigned int row, unsigned int col, int layer_index ) {
	TCell* c = fMultiGrid->getCellLayer(layer_index)->getCell(row,col); 
	return Value(c);
}


/*
void MultiCoverage::SubSet( byte iset, float value ) {
  for( Pix p = fPSet->first(); p; fPSet->next(p) ) {
    byte test = fPSet->subset_contains( p, iset );
    if( test ) SetValue(p,value);
  }
}



float MultiCoverage::RemoteValue( Pix p, int recvProc ) {    // p should be non-zero on only one proc!
#ifdef USE_MPI
  MPI_Request request;
  MPI_Status status;
  float value;
  if( p ) {
    value = Value(p);
    if( gIProc == recvProc ) return value;
    else MPI_Isend( &value, 1, MPI_FLOAT, recvProc, kRemoteValueTag, fComm, &request );  
  }
  else if( gIProc == recvProc) {    
    MPI_Irecv( &value, 1, MPI_FLOAT, MPI_ANY_SOURCE, kRemoteValueTag, fComm, &request ); 
    MPI_Wait(&request,&status); 
  }
  return value;
#else
  return Value(p);
#endif
}



float  MultiCoverage::AggregateValue( const OrderedPoint& point, int scale ) {
  return Value(point);
}

void MultiCoverage::Condense( byte*& cData, int targetProc, int bsize, EDataLayout layout, float s, float o )
{
#ifdef USE_MPI
  bsize = Util::iclamp(bsize,0,sizeof(float));
  int dLen = fNObjects*bsize;
  byte* cDataTmp = new byte[dLen];
  if( dLen > 0 ) {
    if( bsize < sizeof(float) ) {	
      unsigned long itmp, imax = Util::pow2(bsize*8) - 1; 
      float ftmp;
      byte* dPtr = cDataTmp;
      for(int i=0; i<fNObjects; i++)  {
				itmp = (unsigned long) Util::fclamp(((fValue[i]-o)/s),0.0,imax);
				Util::enc_Nb(&dPtr,itmp,bsize);
      }
    } else {
      for(int i=0; i<fNObjects; i++)  memcpy(cDataTmp+i*bsize,fValue+i,bsize);
    }
  }
  if( gIProc == targetProc ) {
    MPI_Status status; int flag, source, count, msgCnt=1;
    cData = MergeData(cData,cDataTmp,targetProc,bsize,fNObjects);
    while(1) {
      MPI_Iprobe(MPI_ANY_SOURCE,cTag,fComm,&flag,&status);
      if( flag == true ) {
				MPI_Get_count(&status,MPI_CHAR,&count);
				source = status.MPI_SOURCE;
				byte dTemp[count];
				MPI_Recv(dTemp,count,MPI_CHAR,source,cTag,fComm,&status);
				cData = MergeData(cData,dTemp,source,bsize,count/bsize);
				if(++msgCnt == gNProc) break;
			}
    }
  } else {
      MPI_Send(cDataTmp,dLen,MPI_CHAR,targetProc,cTag,fComm);
  }
  delete[] cDataTmp; 
  if( layout == kCompressed ) Compress(cData,bsize);
#endif  
}




void MultiCoverage::Compress( byte* data, int bsize ) {
  Region2& globalRegion = fPSet->GlobalRegion();
  ByteGrid* regionMap = fPSet->RegionMap();
  byte mVal; int pCnt = 0, sIndex, dIndex;   
  for(int ir=globalRegion.lower(0); ir<=globalRegion.upper(0); ir+=globalRegion.increment(0) ) {
    for(int ic=globalRegion.lower(1); ic<=globalRegion.upper(1); ic+=globalRegion.increment(1) ) { 
      mVal = regionMap->BValue(ir,ic,0);
      if( fPSet->ActivePoint(mVal) ) {
				sIndex = bsize*pCnt++;
				dIndex = globalRegion.bindex(ir,ic)*bsize;
				memcpy(data+sIndex,data+dIndex,bsize);
      }
    }
  }
}

// inline void my_memset(byte* dest, byte fillVal, int nLoc) {
//   for(int i=0; i<nLoc; i++) dest[i] = fillVal;
// }




MultiCoverage&  MultiCoverage::operator=( ByteGrid& map )
{
  for (Pix p = fPSet->first(); p; fPSet->next(p) ) { 
    OrderedPoint& pt = (OrderedPoint&) (*fPSet)(p);
    SetValue( p, map.Value(pt) );
  }
  return *this;
}




	
	*/


